Extension { #name : 'ObsoleteTimeZone' }

{ #category : 'Instance Creation' }
ObsoleteTimeZone class >> _timeDifferenceSec: secs dstSec: dstSec atTimeSec: startTimeSec
fromDayNum: startDay toDayNum: endDay on: nameOfDay beginning: startYear
stdPrintString: stdString dstPrintString: dstString [

| aTimeZone |

aTimeZone := super new.
aTimeZone secondsFromGmt: secs ;
          secondsForDst: dstSec ;
          timeStartDst: startTimeSec ;
	  weekDayStartDst: nameOfDay;
	  dayStartDst: startDay;
	  dayEndDst: endDay;
	  yearStartDst: startYear;
	  standardPrintString: stdString;
	  dstPrintString: dstString.

aTimeZone dstStartTimeList: (IntegerKeyValueDictionary new);
          dstEndTimeList: (IntegerKeyValueDictionary new).

^ aTimeZone.

]

{ #category : 'cache' }
ObsoleteTimeZone class >> cache [
        ^TimeZones.

]

{ #category : 'Session Control' }
ObsoleteTimeZone class >> current [

"Returns the current session's current ObsoleteTimeZone. E.g. ObsoleteTimeZone current."

^ System __sessionStateAt: 17.

]

{ #category : 'Updating' }
ObsoleteTimeZone class >> default [

"Returns the system wide default ObsoleteTimeZone."

^ Default

]

{ #category : 'Accessing' }
ObsoleteTimeZone class >> default: aTimeZone [

"Makes the specified time zone the default time zone. Returns aTimeZone.
 Must be SystemUser to do so."

aTimeZone _validateClass: ObsoleteTimeZone.
System myUserProfile userId = 'SystemUser' ifFalse:[
  self error:'instance only modifiable by SystemUser'.
  self _uncontinuableError .
].
Default := aTimeZone.
^ aTimeZone.

]

{ #category : 'Accessing' }
ObsoleteTimeZone class >> for: aPlace [

"Returns a ObsoleteTimeZone object for the specified place if it has been defined
 and stored in the class. Returns nil if it is not defined.
 E.g. ObsoleteTimeZone for: #Montreal."

^ TimeZones at: aPlace otherwise: nil.

]

{ #category : 'Updating' }
ObsoleteTimeZone class >> for: aPlace put: aTimeZone [

"Stores aTimeZone as the ObsoleteTimeZone object identified with a particular
 place. A single ObsoleteTimeZone can be associated with any number of places.
 E.g. ObsoleteTimeZone for: #Berlin put: aTimeZone; ObsoleteTimeZone for: #Germany
 put: aTimeZone. Returns aTimeZone."

^ TimeZones at: aPlace put: aTimeZone.

]

{ #category : 'Storing and Loading' }
ObsoleteTimeZone class >> loadFrom: passiveObj [

"Creates and returns an active instance of the receiver from the passive
 form of the object."

| inst secFromGmt secForDst timeStartDst weekDayStart dayStart
  dayEnd yearStart stdPrint dstPrint marker |

marker := passiveObj objectPositionMarker.

passiveObj readNamedIV.
secFromGmt := passiveObj ivValue.
passiveObj readNamedIV.
secForDst := passiveObj ivValue.
passiveObj readNamedIV.
timeStartDst := passiveObj ivValue.
passiveObj readNamedIV.
weekDayStart := passiveObj ivValue.
passiveObj readNamedIV.
dayStart := passiveObj ivValue.
passiveObj readNamedIV.
dayEnd := passiveObj ivValue.
passiveObj readNamedIV.
yearStart := passiveObj ivValue.
passiveObj readNamedIV.
stdPrint := passiveObj ivValue.
passiveObj readNamedIV.
dstPrint := passiveObj ivValue.

passiveObj skipNamedInstVars.

inst := self _timeDifferenceSec: secFromGmt
    dstSec: secForDst
    atTimeSec: timeStartDst
    fromDayNum: dayStart toDayNum: dayEnd
    on: weekDayStart beginning: yearStart
    stdPrintString: stdPrint dstPrintString: dstPrint .

passiveObj hasRead: inst marker: marker .
^inst.

]

{ #category : 'Instance Creation' }
ObsoleteTimeZone class >> migrateNew [

"Override default migrateNew behavior with #_basicNew."

^ self _basicNew

]

{ #category : 'Instance Creation' }
ObsoleteTimeZone class >> new [

"Disallowed.  To create a new ObsoleteTimeZone, use another instance creation method."

self shouldNotImplement: #new

]

{ #category : 'Instance Creation' }
ObsoleteTimeZone class >> new: anInteger [

"Disallowed.  To create a new ObsoleteTimeZone, use another instance creation method."

self shouldNotImplement: #new:

]

{ #category : 'Instance Creation' }
ObsoleteTimeZone class >> timeDifferenceHrs: hours dstHrs: dstHrs atTimeHrs: startTimeHrs
fromDayNum: startDay toDayNum: endDay on: nameOfDay beginning: startYear
stdPrintString: stdString dstPrintString: dstString [

"Returns a new instance of ObsoleteTimeZone with the given parameters. The parameters
 are as follows:

	timeDifferenceHrs: time difference from GMT in hours. Regions east of
	Greenwich meridian have positive time differences values, those west
	of the meridian  have negative values.

	dstHrs: daylight savings adjustment in hours.

	atTimeHrs: number of hours after midnight when daylight savings starts.
	E.g. 2 implies 2.00 AM.

        on: name of the day on which daylight savings starts. E.g. #Sunday.

	fromDayNum: daylight savings program should have started by this
        day. It actually starts on the preceding nameOfDay (usually Sunday).

	toDayNum: daylight savings should have ended by this day. It actually
	ends on the preceding nameOfDay (usually Sunday).

	beginning: The year in which daylight savings program started.

	stdPrintString: The string to be used for printing during standard
	time.

	dstString: The string to be used for printing during daylight savings
	time."

^ self _timeDifferenceSec: hours * 3600
    dstSec: dstHrs * 3600
    atTimeSec: startTimeHrs * 3600
    fromDayNum: startDay toDayNum: endDay
    on: nameOfDay beginning: startYear
    stdPrintString: stdString dstPrintString: dstString

]

{ #category : 'Instance Creation' }
ObsoleteTimeZone class >> timeDifferenceMin: minutes dstMin: dstMins atTimeMin: startTimeMins
fromDayNum: startDay toDayNum: endDay on: nameOfDay beginning: startYear
stdPrintString: stdString dstPrintString: dstString [

"Returns a new instance of ObsoleteTimeZone with the given parameters. The parameters
 are as follows:

        timeDifferenceMin: time difference from GMT in minutes. Regions east of
        Greenwich meridian have positive time differences values, those west
        of the meridian  have negative values.

        dstMin: daylight savings adjustment in minutes.

        atTimeMin: number of hours after midnight when daylight savings starts.
        E.g. 120 implies 2.00 AM.

        on: name of the day on which daylight savings starts. E.g. #Sunday.

	fromDayNum: daylight savings program should have started by this
        day. It actually starts on the preceding nameOfDay (usually Sunday).

	toDayNum: daylight savings should have ended by this day. It actually
	ends on the preceding nameOfDay (usually Sunday).

        beginning: The year in which daylight savings program started.

        stdPrintString: The string to be used for printing during standard
        time.

        dstString: The string to be used for printing during daylight savings
        time."

^ self _timeDifferenceSec: minutes * 60
    dstSec: dstMins * 60
    atTimeSec: startTimeMins * 60
    fromDayNum: startDay toDayNum: endDay
    on: nameOfDay beginning: startYear
    stdPrintString: stdString dstPrintString: dstString

]

{ #category : 'Inquiries' }
ObsoleteTimeZone >> dateTimeClass [

"Returns the class of DateTime objects that are to be created by the
 various methods in this class."

^ DateTime

]

{ #category : 'Accessing' }
ObsoleteTimeZone >> dayEndDst [

"Returns the dayEndDst instance variable."

^ dayEndDst

]

{ #category : 'Updating' }
ObsoleteTimeZone >> dayEndDst: aSmallInteger [

"Sets the dayEndDst instance variable. Returns the receiver."

dayEndDst := aSmallInteger.
^ self

]

{ #category : 'Accessing' }
ObsoleteTimeZone >> dayStartDst [

"Returns the dayStartDst instance variable."

^ dayStartDst

]

{ #category : 'Updating' }
ObsoleteTimeZone >> dayStartDst: aSmallInteger [

"Sets the dayStartDst instance variable. Returns the receiver."

dayStartDst := aSmallInteger.
^ self

]

{ #category : 'Accessing' }
ObsoleteTimeZone >> dstEndTimeList [

"Returns the dstEndTimeList instance variable."

^ dstEndTimeList

]

{ #category : 'Updating' }
ObsoleteTimeZone >> dstEndTimeList: aDictionary [

"Sets the dstEndTimeList instance variable. Returns the receiver."

dstEndTimeList := aDictionary.
^ self

]

{ #category : 'Accessing' }
ObsoleteTimeZone >> dstPrintString [

"Returns the dstPrintString instance variable."

^ dstPrintString

]

{ #category : 'Updating' }
ObsoleteTimeZone >> dstPrintString: aString [

"Sets the dstPrintString instance variable. Returns the receiver."

dstPrintString := aString.
^ self

]

{ #category : 'Accessing' }
ObsoleteTimeZone >> dstStartTimeList [

"Returns the dstStartTimeList instance variable."

^ dstStartTimeList

]

{ #category : 'Updating' }
ObsoleteTimeZone >> dstStartTimeList: aDictionary [

"Sets the dstStartTimeList instance variable. Returns the receiver."

dstStartTimeList := aDictionary.
^ self

]

{ #category : 'Private' }
ObsoleteTimeZone >> endOfDstFor: aYear [

| aDateTime theDay aDate |

"Check if the datetime object for the start of dst already exists."
aDateTime := dstEndTimeList at: aYear otherwise: nil.
(aDateTime ~~ nil)
  ifTrue: [ ^ aDateTime ].

"Check for year does not have end (e.g., starts this year in October,
ends next year in March; bug #43664)"
(dayEndDst < dayStartDst and: [aYear == yearStartDst]) ifTrue: [^nil].

"Adjust for leap year if necessary"
((dayStartDst > 60) and: [ Date isLeap: aYear ])
  ifTrue: [ theDay := dayEndDst + 1 ]
  ifFalse: [ theDay := dayEndDst ].

"If this falls on dayOfWeek, then the day has been found. Otherwise, find
 the previous dayOfWeek."
aDate := (Date newDay: theDay + 1 year: aYear) previous: weekDayStartDst.

"create a dateTime and store it the receiver for future accesses."
aDateTime := self dateTimeClass newGmtWithDate: aDate
                  seconds: (timeStartDst - secondsForDst - secondsFromGmt)
                  timeZone: self.
dstEndTimeList at: aYear put: aDateTime .

^ aDateTime.

]

{ #category : 'Session Control' }
ObsoleteTimeZone >> installAsCurrentTimeZone [

"Sets the receiver as the current session's current Timezone. Returns the
 receiver."

System __sessionStateAt: 17 put: self.
^ self.

]

{ #category : 'Accessing' }
ObsoleteTimeZone >> secondsForDst [

"Returns the secondsForDst instance variable."

^ secondsForDst

]

{ #category : 'Updating' }
ObsoleteTimeZone >> secondsForDst: aSmallInteger [

"Sets the secondsForDst instance variable. Returns the receiver."

secondsForDst := aSmallInteger.
^ self

]

{ #category : 'Accessing' }
ObsoleteTimeZone >> secondsFromGmt [

"Returns the secondsFromGmt instance variable."

^ secondsFromGmt

]

{ #category : 'Updating' }
ObsoleteTimeZone >> secondsFromGmt: aSmallInteger [

"Sets the secondsFromGmt instance variable. Returns the receiver."

secondsFromGmt := aSmallInteger.
^ self

]

{ #category : 'Storing and Loading' }
ObsoleteTimeZone >> shouldWriteInstVar: instVarName [

"Returns whether the given instance variable should be written out."

"exclude the ditionaries"

instVarName == #dstStartTimeList ifTrue:[ ^ false ].
instVarName == #dstEndTimeList ifTrue:[ ^ false ].
^ true

]

{ #category : 'Accessing' }
ObsoleteTimeZone >> standardPrintString [

"Returns the standardPrintString instance variable."

^ standardPrintString

]

{ #category : 'Updating' }
ObsoleteTimeZone >> standardPrintString: aString [

"Sets the standardPrintString instance variable. Returns the receiver."

standardPrintString := aString.
^ self

]

{ #category : 'Private' }
ObsoleteTimeZone >> startOfDstFor: aYear [

| aDateTime theDay aDate |

"Check if the datetime object for the start of dst already exists."
aDateTime := dstStartTimeList at: aYear otherwise: nil.
(aDateTime ~~ nil)
  ifTrue: [ ^ aDateTime ].

"Adjust for leap year if necessary"
((dayStartDst > 60) and: [ Date isLeap: aYear ])
  ifTrue: [ theDay := dayStartDst + 1 ]
  ifFalse: [ theDay := dayStartDst ].

"If this falls on dayOfWeek, then the day has been found. Otherwise, find
 the previous dayOfWeek."
aDate := (Date newDay: theDay + 1 year: aYear) previous: weekDayStartDst.

"create a dateTime and store it the receiver for future accesses."
aDateTime := self dateTimeClass newGmtWithDate: aDate
                  seconds: (timeStartDst - secondsFromGmt)
                  timeZone: self.
dstStartTimeList at: aYear put: aDateTime .

^ aDateTime.

]

{ #category : 'Accessing' }
ObsoleteTimeZone >> timeStartDst [

"Returns the timeStartDst instance variable."

^ timeStartDst

]

{ #category : 'Updating' }
ObsoleteTimeZone >> timeStartDst: aSmallInteger [

"Sets the timeStartDst instance variable. Returns the receiver."

timeStartDst := aSmallInteger.
^ self

]

{ #category : 'Accessing' }
ObsoleteTimeZone >> weekDayStartDst [

"Returns the weekDayStartDst instance variable."

^ weekDayStartDst

]

{ #category : 'Updating' }
ObsoleteTimeZone >> weekDayStartDst: aSymbol [

"Sets the weekDayStartDst instance variable. Returns the receiver."

weekDayStartDst := aSymbol.
^ self

]

{ #category : 'Accessing' }
ObsoleteTimeZone >> yearStartDst [

"Returns the yearStartDst instance variable."

^ yearStartDst

]

{ #category : 'Updating' }
ObsoleteTimeZone >> yearStartDst: aSmallInteger [

"Sets the yearStartDst instance variable. Returns the receiver."

yearStartDst := aSmallInteger.
^ self

]
