"
A ClassOrganizer can answer queries about classes in the image, provides
 tools and reports on information on classes and methods and can perform
 cross-referencing and fileout.

 Instance variables:

 categories - A Dictionary of category->classes associations.
 classes - A ClassSet of all the classes found by an instance.
 classNames - Class name information for auto completion.
 hierarchy - An IdentityDictionary of class->subclasses associations.
 rootClass - The root class of the instance. The instance includes
    all classes in the specified symbolList which are subclasses of 
    rootClass, plus all superClasses of rootClass up to Object.
    Instance creation methods take a SymbolList or UserProfile as
    an argument.
 user - holds the SymbolList used by the instance
 
 dynamic instance variables  
    #envId     
    #optimizedSelectors
    #methodOops
    #restrictedSymbolList
		#traits
"
Class {
	#name : 'ClassOrganizer',
	#superclass : 'Object',
	#instVars : [
		'classes',
		'classNames',
		'user',
		'hierarchy',
		'categories',
		'rootClass'
	],
	#gs_reservedoop : '93185',
	#category : nil
}

{ #category : 'Private' }
ClassOrganizer class >> _newWithRoot: aClass restrictedSymbolList: symList env: envId [
  | inst |
  inst := super new .
  inst _symbolList: symList .
  inst dynamicInstVarAt: #envId put: envId .
  inst rootClass: aClass.
  inst classes: (ClassSet new add: aClass ; yourself).
  inst dynamicInstVarAt: #restrictedSymbolList put: true .
  ^ inst _build 
]

{ #category : 'Private' }
ClassOrganizer class >> _newWithRoot: aClass symbolList: symList env: envId [
  | inst |
  inst := super new .
  inst _symbolList: symList .
  inst dynamicInstVarAt: #envId put: envId .
  inst rootClass: aClass.
  inst classes: (ClassSet new add: aClass ; yourself).
  ^ inst _build .
]

{ #category : 'Updating' }
ClassOrganizer class >> clearCachedOrganizer [
  "do nothing in base image"
  ^ self

]

{ #category : 'Instance Creation' }
ClassOrganizer class >> new [

"Creates and returns a new instance of ClassOrganizer with a root of Object,
 and using the symbolList of the current UserProfile."

^ self _newWithRoot: Object symbolList: GsCurrentSession currentSession symbolList
	env: 0

]

{ #category : 'Instance Creation' }
ClassOrganizer class >> newExcludingGlobals [
  "Example
     topaz 1> send ClassOrganizer newExcludingGlobals
     topaz 1> define CurrentClassOrganizer **
   Then subsequent organizer commands in topaz (such as  senders, implementors) 
   will exclude methods in classes in Globals .
  " 
  ^ self _newWithRoot: Object 
    restrictedSymbolList: (SymbolList withAll:
          (GsCurrentSession currentSession symbolList reject:[:x |x == Globals]))
     env: 0
]

{ #category : 'Instance Creation' }
ClassOrganizer class >> newForEnvironment: envId [

"Creates and returns a new instance of ClassOrganizer with a root of Object,
 and using the symbolList of the current UserProfile.
 Uses the environment envId ( a SmallInteger >= 0) for superclass and method
 dictionary lookups."

^ self _newWithRoot: Object symbolList: GsCurrentSession currentSession symbolList
	env: envId

]

{ #category : 'Instance Creation' }
ClassOrganizer class >> newWithRoot: aClass [

^ self _newWithRoot: aClass symbolList: GsCurrentSession currentSession symbolList
      env: 0

]

{ #category : 'Instance Creation' }
ClassOrganizer class >> newWithRoot: aClass forEnvironment: envId [

" Uses the environment envId ( a SmallInteger >= 0) for superclass and method
 dictionary lookups."

^ self _newWithRoot: aClass symbolList: GsCurrentSession currentSession symbolList
      env: envId

]

{ #category : 'Instance Creation' }
ClassOrganizer class >> newWithRoot: aClass forUserId: aUserId [

"Creates a new ClassOrganizer that is limited to the given subtree of objects using
 the SymbolList from aUserId.  Caller must have read access to given user's SymbolList."

^ self _newWithRoot: aClass symbolList: (AllUsers userWithId: aUserId) symbolList
	env: 0

]

{ #category : 'Instance Creation' }
ClassOrganizer class >> newWithRoot: aClass forUserId: aUserId forEnvironment: envId [

"Creates a new ClassOrganizer that is limited to the given subtree of objects using
 the SymbolList from aUserId.  Caller must have read access to given user's SymbolList.
 Uses the environment envId ( a SmallInteger >= 0) for superclass and method
 dictionary lookups."

^ self _newWithRoot: aClass symbolList: (AllUsers userWithId: aUserId) symbolList
	env: envId

]

{ #category : 'Instance Creation' }
ClassOrganizer class >> newWithRoot: aClass from: anotherOrganizer [

"Creates a new ClassOrganizer that uses the specified rootClass."

| inst |
inst := super new.
inst dynamicInstVarAt: #envId put: anotherOrganizer environmentId .
inst rootClass: aClass.
inst classes: ClassSet new.
inst classes add: aClass.
inst classes addAll: (anotherOrganizer allSubclassesOf: aClass).
inst traits: anotherOrganizer traits.
inst rebuildHierarchy .
^inst
]

{ #category : 'Private' }
ClassOrganizer >> _asCategoriesReportString: anArray [
"Returns a String. The result is sorted by method categories"
| lf dict cats rpt |
lf := Character lf .
dict := Dictionary new .
1 to: anArray size do:[:j | | aMeth str cls sel cat arr |
  aMeth := anArray at: j .
  cls := aMeth inClass .
  sel := aMeth selector .
  str := String withAll: cls name .
  str addAll: ' >> ' ; addAll: sel .
  cat := cls categoryOfSelector: sel environmentId: aMeth environmentId .
  arr := (dict at: cat otherwise: nil ) ifNil:[ dict at: cat put: { } ].
  arr add: str .
].
cats := SortedCollection sortBlock:[ :x :y | x key <= y key ].
dict associationsDo:[:assoc | cats add: assoc ].
rpt := String new .
cats do:[:assoc | | arr |
  rpt add: assoc key ; add: lf .
  arr := SortedCollection withAll: assoc value .
  1 to: arr size do:[:j |
    rpt add: '   '; add:(arr at:j) ; add: lf
  ].
].
^ rpt

]

{ #category : 'Private' }
ClassOrganizer >> _asReportString: anArray [
  ^ self _asReportString: anArray indent: ''

]

{ #category : 'Private' }
ClassOrganizer >> _asReportString: anArray indent: indentString [
"Returns a String, one line per method in anArray."
| result LF arr deprecSet numDeprecated includeOops |
numDeprecated := 0 .
self includeDeprecatedMethodsInReports ifFalse:[
  deprecSet := Object _selectorsInBaseCategory:#'Deprecated Notification' .
].
arr := SortedCollection new .
LF := Character lf .
includeOops := self includeMethodOops .
1 to: anArray size do:[:j | | aMeth str |
  aMeth := anArray at: j .
  (deprecSet ~~ nil and:[ (aMeth _selectorPool * deprecSet) size ~~ 0]) ifTrue:[
    "the method sends a variant of #deprecated... "
    numDeprecated := numDeprecated + 1 .
  ] ifFalse:[ | cls |
    str := String withAll: (cls := aMeth inClass) name .
    includeOops ifTrue:[
      cls isMeta ifTrue:[ cls := cls thisClass ].
      str add:'  '; add: cls asOop asString
    ].
    str addAll: ' >> ' ; addAll: aMeth selector .
    includeOops ifTrue:[ str add:'   '; add: aMeth asOop asString ].
    arr add: str .
  ].
].
result := String new .
1 to: arr size do:[:j |
  result add: indentString; add:(arr at:j) ; add: LF
].
numDeprecated > 0 ifTrue:[
  result add:'(Omitted ' , numDeprecated asString, ' deprecated methods)'; lf
].
^ result

]

{ #category : 'Private' }
ClassOrganizer >> _build [
  self collectClassesFromSymbolList: user"the symbolList" .
  self rebuildHierarchy .
  "self collectClassNames . " "AutoCompleter is built only on demand"
  self rebuildCategories .
]

{ #category : 'Private' }
ClassOrganizer >> _hierarchyIvReport: aClass [
| str |
str := String new .
self _hierarchyIvReportForClass: aClass indent: '' report: str .
^ str

]

{ #category : 'Private' }
ClassOrganizer >> _hierarchyIvReportForClass: aClass indent: indent report: report [
| subClsArray nextIndent ivNames |
report addAll: indent; add: aClass name .
ivNames := aClass _instVarNamesWithSeparator:  ''   .
ivNames size ~~ 0 ifTrue: [ report add: $(; add: ivNames ; add: $) ].
report lf .
subClsArray := SortedCollection withAll:
  ((hierarchy at: aClass otherwise: nil) ifNil:[ #( ) ]) .
nextIndent := indent , '  ' .
subClsArray do:[ :aSubCls |
  self _hierarchyIvReportForClass: aSubCls indent: nextIndent report: report
  ].

]

{ #category : 'Private' }
ClassOrganizer >> _hierarchyReportForClass: aClass indent: indent report: report [
  "used by GBS"

^ self _hierarchyReportForClass: aClass indent: indent report: report withOops: false
	withInstvars: false


]

{ #category : 'Private' }
ClassOrganizer >> _hierarchyReportForClass: aClass indent: indent report: report withOops: aBool [

^ self _hierarchyReportForClass: aClass indent: indent report: report withOops: aBool
        withInstvars: false

]

{ #category : 'Private' }
ClassOrganizer >> _hierarchyReportForClass: aClass indent: indent report: report
withOops: oopsBool withInstvars: ivsBool [

| subClsArray nextIndent |
report addAll: indent; add: (aClass name ifNil:[ '(nil name)' ]).
oopsBool ifTrue:[ report add: '   '; add: aClass asOop asString ].
ivsBool ifTrue:[ | ivNames |
  ivNames := aClass _instVarNamesWithSeparator:  ''.
  ivNames size ~~ 0 ifTrue:[ report add: ' ('; add: ivNames; add: $) ].
].
report add: Character lf .
subClsArray := self sortClasses:
  ((hierarchy at: aClass otherwise: nil) ifNil:[ #( ) ]) .
nextIndent := indent , '  ' .
subClsArray do:[ :aSubCls |
  self _hierarchyReportForClass: aSubCls indent: nextIndent
        report: report withOops: oopsBool withInstvars: ivsBool
  ].

]

{ #category : 'Private' }
ClassOrganizer >> _methodCategories [
 "Returns an IdentitySet of all method categories"
| set |
set := IdentitySet new .
1 to: classes size do: [ :i |
  1 to: 2 do: [ :which | | cls |
    cls := classes _at: i.
    which == 2 ifTrue: [ cls := cls class ].
    cls categorysDo:[ :aCateg :selectors |  set add: aCateg ].
  ].
].
^ set

]

{ #category : 'Private' }
ClassOrganizer >> _methodsInCategory: aBlock [
  "aBlock is a one argument block taking a category as an argument"
| sortedClasses methsSet methsArr cls |
methsSet := IdentitySet new .
methsArr := { } .
sortedClasses := self sortClasses: classes  .
1 to: sortedClasses size do: [ :i |
  1 to: 2 do: [ :which |
    cls := sortedClasses _at: i.
    which == 2 ifTrue: [ cls := cls class ].
    cls  env: 0 categorysDo:[ :categName :sels |
      (aBlock value: categName) ifTrue:[  | mDict |
        mDict := cls _fullMethodDictEnv: self environmentId .
        sels do:[ :aSel | | meth |
          (meth := mDict at: aSel otherwise: nil) ifNotNil:[
             (methsSet _addIfAbsent: meth) ifTrue:[ methsArr add: meth ]
          ].
        ].
      ].
    ].
  ].
].
^ methsArr

]

{ #category : 'Queries' }
ClassOrganizer >> _referencesToLiteral: aLiteral in: aclassSet withOffsets: withOfsBool [
 "Returns an Array of two Arrays.  The first subarray contains GsNMethods
 that reference given literal.

 If aLiteral is an invariant SymbolAssociation,
 references to either aLiteral or (aLiteral value) will be returned.

 If withOfsBool==true, The second subarray contains indexes where the
 first reference to the literal occurs within the sourceString of the method."

  | result resulti cset cls methsSet env litString |
  methsSet := IdentitySet new.
  result := {}.
  resulti := {}.
  cset := self sortClasses: aclassSet.
  env := self environmentId .
  litString := aLiteral _isSymbol ifTrue:[ aLiteral ]
           ifFalse:[ (aLiteral isKindOf: Association) ifTrue:[ aLiteral key ]
                             ifFalse:[ aLiteral asString ]].
  1 to: cset size do: [ :i |
    cls := cset at: i.
    2 timesRepeat: [ | mDict |
      mDict := cls _fullMethodDictEnv: env .
      mDict valuesDo: [ :method |
	| found |
	found := false.
	(aLiteral _isSymbol and:[ method _literalsIncludesSymbol: aLiteral value: nil])
	  ifTrue: [ found := true ]
	  ifFalse: [
	    (aLiteral _isSymbol not and:[ method literals _refersToNonSymbolLiteral: aLiteral])
	      ifTrue: [ found := true ]
	      ifFalse: [
		(method pragmas _refersToLiteral: aLiteral)
		  ifTrue: [ found := true ] ] ].
	found
	  ifTrue:[
	    (methsSet _addIfAbsent: method)
	      ifTrue: [
		result add: method.
		withOfsBool ifTrue:[ resulti add:
		    (method sourceString findString: litString startingAt: 1) ] ] ]].
      cls := cls class ] ].
  ^ {result .  resulti}

]

{ #category : 'Queries' }
ClassOrganizer >> _sendersOf: aSelector in: aclassSet includeOptimized: aBoolean [

"Returns an Array of two Arrays.  The first subarray contains GsNMethods
 that send the given selector.

 If aBoolean == false, inlined sends of optimized selectors are excluded.

 For non-optimized selectors, the second subarray contains indexes where
 the first use of the selector occurs within the sourceString of the method.

 Results are limited to classes in aclassSet. The elements in aclassSet are
 expected to be classes, not metaclasses ."

| result resulti cset cls aSymbol methsSet optimSels env |

methsSet := IdentitySet new .
result := { } .
resulti := { } .
aSymbol := Symbol _existingWithAll: aSelector .
aSymbol ifNil:[ ^{ result . resulti } ].

(aclassSet isKindOf: ClassSet) ifFalse: [
  cset := ClassSet withAll: aclassSet
] ifTrue: [
  cset := aclassSet
].
aBoolean ifTrue:[
  (optimSels := self dynamicInstVarAt: #optimizedSelectors) ifNil:[
    optimSels := IdentitySet withAll: GsNMethod optimizedSelectors .
    self dynamicInstVarAt: #optimizedSelectors put: optimSels .
  ].
  (optimSels includes: aSymbol) ifTrue:[
    ^ self _referencesToLiteral: aSymbol in: classes withOffsets:false
  ].
].
cset := self sortClasses: cset .
env := self environmentId .
1 to: cset size do: [ :i |
  cls := cset at: i.
  2 timesRepeat:[ | mDict |
    mDict := cls _fullMethodDictEnv: env .
    mDict valuesDo:[ :gsMethod | | srcOffset |
      srcOffset := gsMethod _sourceOffsetOfFirstSendOf: aSymbol .
      srcOffset ifNotNil:[
        (methsSet _addIfAbsent: gsMethod) ifTrue:[
          result add: gsMethod .
          resulti add: srcOffset.
        ].
      ].
    ].
    cls := cls class .
  ].
].
^{ result . resulti }

]

{ #category : 'Private' }
ClassOrganizer >> _substringSearch: aString in: aclassSet ignoreCase: icBoolean [
"Returns an Array of three Arrays.  The first subarray contains GsNMethods
 whose sources include the given substring.  The second subarray contains
 indexes where the first occurrence of the substring was found.
 The third subarray contains classes for which the result of   aClass comment 
 contains the specified string "

| meths offsets comments list cls methsSet env |
methsSet := IdentitySet new .
meths := { } .
offsets := { } .
comments := { } .
list := self sortClasses: aclassSet .
env := self environmentId .
1 to: list size do: [ :i |
  cls := list at: i.
  2 timesRepeat:[ | mDict |
    mDict := cls _fullMethodDictEnv: env .
    mDict valuesDo:[ :gsMethod | | index |
        index := gsMethod sourceString _findString: aString startingAt: 1
				      ignoreCase: icBoolean .
        index > 0 ifTrue: [
          (methsSet _addIfAbsent: gsMethod) ifTrue:[
            meths add: gsMethod .
            offsets add: index
          ].
             ].
    ].
    cls := cls class .
  ].
  cls := list at: i .
  (cls _extraDictAt: #comment ) ifNotNil:[:str | | ofs |
     ofs := str _findString: aString startingAt: 1 ignoreCase: icBoolean .
     ofs > 0 ifTrue:[ comments add: cls ].
  ].
].
^{ meths . offsets . comments }

]

{ #category : 'Private' }
ClassOrganizer >> _superClass: aClass [

  ^ (aClass superclassForEnv: self environmentId) ifNil:[ #nil ]

]

{ #category : 'Private' }
ClassOrganizer >> _symbolList: aSymbolList [
  user := aSymbolList

]

{ #category : 'Private' }
ClassOrganizer >> _traitSubstringSearch: aString in: atraitSet ignoreCase: icBoolean [
"Returns an Array of three Arrays.  The first subarray contains method signatures
 whose sources include the given substring.  The second subarray contains
 indexes where the first occurrence of the substring was found.
 The third subarray contains traits for which the result of   aTrait comment 
 contains the specified string "

| methSigs offsets comments list trait methSigsSet qualifier |
methSigsSet := IdentitySet new .
methSigs := { } .
offsets := { } .
comments := { } .
list := self sortClasses: atraitSet .
1 to: list size do: [ :i |
  trait := list at: i.
  qualifier := ''.
  2 timesRepeat: [
     trait localSelectors do:[ :selector | | index |
        index := (trait sourceCodeAt: selector) _findString: aString startingAt: 1
				      ignoreCase: icBoolean .
        index > 0 ifTrue: [
          | methodSignature |
          methodSignature  := String withAll: trait name , qualifier, ' >> ', selector.
          (methSigsSet _addIfAbsent: methodSignature) ifTrue:[
            methSigs add: methodSignature .
            offsets add: index
          ].
             ].
    ].
    trait := trait classTrait .
    qualifier := ' classTrait'.
  ].
  trait := list at: i .
  (trait _extraDictAt: #comment ) ifNotNil:[:str | | ofs |
     ofs := str _findString: aString startingAt: 1 ignoreCase: icBoolean .
     ofs > 0 ifTrue:[ comments add: trait ].
  ].
].
^{ methSigs . offsets . comments }
]

{ #category : 'Queries' }
ClassOrganizer >> accessorsOf: ivName inClass: aClass [
"Return an array of GsNMethods that directly access the instance
 variable with the given name in the given class"

  | allSubs arr nam |
  arr := { } .
  (nam := Symbol _existingWithAll: ivName) ifNil:[ ^ arr ].
  allSubs := self allSubclassesOf: aClass .
  allSubs add: aClass .
  allSubs do:[:cls |
    cls selectors do:[:sel | | meth |
      meth := cls compiledMethodAt: sel .
      (meth instVarsAccessed includes: nam) ifTrue:[ arr add: meth ].
    ].
  ].
  ^ arr

]

{ #category : 'Reports' }
ClassOrganizer >> accessorsReport: ivName inClass: aClass [
"Return a String describing the methods that directly access the instance
 variable with the given name in the given class"

  ^ self _asReportString:( self accessorsOf: ivName inClass: aClass)

]

{ #category : 'Private' }
ClassOrganizer >> addCachedClassNames [

"Adds new class names to the auto-complete set.  New class names are cached
 until the auto-completer is needed, then they are all merged in to the
 completer in one shot.  Users that don't make use of auto-completion features
 never have to pay the price of updating the completer's structures."

| completer cache |
completer := classNames at: 1.
cache := classNames at: 2.
1 to: cache size do: [:i |
  completer addString: (cache at: i).
].
classNames at: 2 put: nil

]

{ #category : 'Updating' }
ClassOrganizer >> addClass: cls [

"Adds the class cls, replacing any existing class with the same superclass."

| nm superCls cat old matching cset |

hierarchy ifNil: [^self].
nm := cls name.
superCls := self _superClass: cls .

cset := hierarchy at: superCls otherwise: nil .
cset ifNotNil:[  | newCset |
  hierarchy at: superCls put: ( newCset := cset select: [:e | e name ~= nm]).
  newCset add: cls.
] ifNil: [
  (cset := ClassSet new) add: cls .
  hierarchy at: superCls put: cset .
].

matching := classes select:[ :each | each name = nm ].
matching size > 0 ifTrue: [
  classes := classes - matching
].
classes add: cls.

cat := categories at: cls category otherwise: nil.
cat ifNil: [
  categories at: cls category put: { cls }.
] ifNotNil: [
  nm := cls name.
  [old := cat findFirst: [:x | x name = nm].
    old > 0] whileTrue: [
    cat removeFrom: old to: old
  ].
  cat add: cls
].

self addedClassName: nm

]

{ #category : 'Private' }
ClassOrganizer >> addedClassName: name [

"Adds the new class name to the auto-complete set."

classNames ifNotNil:[ :cn |
  (cn at: 2) ifNil: [
    cn at: 2 put: { name } .
  ] ifNotNil:[ :arr |
    arr add: name.
  ]
].

]

{ #category : 'Queries' }
ClassOrganizer >> allReferencesTo: selector [

"Returns an Array of two Arrays.  The first contains GsNMethods that
 implement, send, or refer to the given selector.  The second contains the
 indexes into sourceStrings where the first reference takes place."

^self allReferencesTo: selector in: classes

]

{ #category : 'Queries' }
ClassOrganizer >> allReferencesTo: aSelector in: classSet [

"Returns an Array of two Arrays.  The first contains GsNMethods that
 implement, send, or refer to the given selector.  The second contains the
 indexes into sourceStrings where the first reference takes place."

| methods indices cset selectorSym methsSet |

methsSet := IdentitySet new .
methods := { } .
indices := { } .
selectorSym := Symbol _existingWithAll: aSelector .
selectorSym ifNotNil:[
  "Sort the class set by name, then search"
  cset := self sortClasses: classSet  .
  1 to: cset size do: [ :i |
    1 to: 2 do: [ :n | | cls mDict |
      cls := (n == 1 ifTrue: [cset at: i] ifFalse: [cls class]).
      mDict := cls _fullMethodDictEnv: self environmentId .
      mDict valuesDo: [ :method | | srcOffset |
	  "Check for implementors first"
	  method selector == selectorSym ifTrue: [
	    (methsSet _addIfAbsent: method) ifTrue:[
	      methods add: method.
	      indices add: 1.
	    ].
	  ] ifFalse: [
	    "Check for senders"
	    srcOffset := method _sourceOffsetOfFirstSendOf: selectorSym .
	    srcOffset ifNotNil:[
	      (methsSet _addIfAbsent: method) ifTrue:[
		methods add: method .
		indices add: srcOffset.
	      ].
	    ] ifNil: [
	      (method _literalsIncludesSymbol: selectorSym value: nil) ifTrue:[
		(methsSet _addIfAbsent: method) ifTrue:[
		  methods add: method.
		  indices add: (method sourceString findString: selectorSym startingAt: 1).
		].
	      ].
	    ].
	  ].
      ].
    ].
  ].
].
^{ methods . indices }

]

{ #category : 'Queries' }
ClassOrganizer >> allSubclassesOf: aClass [

"Returns a collection of all the subclasses of the given class: an Array that
 holds a depth-first traversal of the class hierarchy subtree rooted at
 aClass."

| result subs |

classes ifNil: [
  self updateClassInfo
].
(classes includesIdentical: aClass) ifFalse: [
  self addClass: aClass.
  ^self allSubclassesOf: aClass
].

result := { } .
subs := hierarchy at: aClass otherwise: nil.
subs size > 0 ifTrue: [
  subs := self sortClasses: subs  .
  subs do: [:each |
    result add: each; addAll: (self allSubclassesOf: each)
  ].
].
^result

]

{ #category : 'Queries' }
ClassOrganizer >> allSuperclassesOf: aClass [
	"Returns a collection of all the superclasses of the given class: an Array that
	 holds the inheritence path of aClass."

	| result current |
	classes ifNil: [self updateClassInfo].
	(classes includesIdentical: aClass) ifFalse:[
    self addClass: aClass.
		^self allSuperclassesOf: aClass
  ].
	result := {}.
	current := aClass.
	[ current := self _superClass: current .
    current ~~ nil and:[ current ~~ #nil] ] whileTrue: [result add: current].
	^result reverse

]

{ #category : 'Accessing' }
ClassOrganizer >> categories [

"Returns the value of the instance variable 'categories', which are the
 class categories."

^categories

]

{ #category : 'Queries' }
ClassOrganizer >> categoryCrossReference [

"Returns a dictionary of all method categories and the classes with methods
 in each category."

| resultDict cls |

resultDict := SymbolDictionary new.
1 to: classes size do: [ :i |
  1 to: 2 do: [ :which |
    cls := classes _at: i.
    which == 2 ifTrue: [ cls := cls class ].
    cls categorysDo:[ :aCateg :selectorSet | | clsset |
        clsset := resultDict at: aCateg otherwise: nil.
        clsset ifNil: [
	  clsset := { }.
	  resultDict at: aCateg put: clsset.
        ].
        clsset add: cls.
    ].
  ].
].
^resultDict

]

{ #category : 'Reports' }
ClassOrganizer >> categoryCrossReferenceByName [

"Returns a String containing a report from a cross-reference of method
 categories. For each method category, the report contains a list of
 names of classes which have methods in that category."

^self categoryCrossReferenceReportByName: nil

]

{ #category : 'Reports' }
ClassOrganizer >> categoryCrossReferenceReportByName: catsDict [

"Returns a String containing a report from a cross-reference of categories.
 For each method category, the report contains a list of
 names of classes which have methods in that category.

 The argument may be nil or a dictionary similar to the one that
 categoryCrossReference returns."

  | dict report cats |
(dict := catsDict) ifNil: [
  dict := self categoryCrossReference
].
report := String new.
cats := SortedCollection withAll: dict keys.
1 to: cats size do: [ :i | | cat clsset |
  cat := cats at: i.
  report add: cat; add: Character tab.
  clsset := dict at: cat.
  1 to: clsset size do: [ :j |
    report add: (clsset at: j) name.
    j < clsset size ifTrue: [ report add: $, ].
  ].
  report lf .
].
^report .

]

{ #category : 'Accessing' }
ClassOrganizer >> classCompletion [

"Returns the AutoComplete holding the class names."

| cn |
cn := classNames ifNil:[ self collectClassNames ].
(cn at: 2) ifNotNil: [
  self addCachedClassNames
].
^(cn at: 1).

]

{ #category : 'Accessing' }
ClassOrganizer >> classes [

"Returns the ClassSet of classes held by the receiver."

^classes

]

{ #category : 'Updating' }
ClassOrganizer >> classes: aClassSet [

"Updates the set of classes held by the receiver.  The receiver's
 hierarchy should be rebuilt after this (see rebuildHierarchy)."

classes := aClassSet

]

{ #category : 'Accessing' }
ClassOrganizer >> classNames [

"Returns the Array of class names held by the receiver."

| cn |
cn := classNames ifNil:[ self collectClassNames ].
(cn at: 2) ifNotNil: [
  self addCachedClassNames
].
^(cn at: 1) strings.

]

{ #category : 'Private' }
ClassOrganizer >> collectClasses [

"Rebuilds the class hierarchy structure.  This message should be sent whenever
 new classes have been created or imported from other users."

^ self collectClassesFromSymbolList: self symbolList

]

{ #category : 'Accessing' }
ClassOrganizer >> collectClassesFromSymbolList: aSymbolList [
  "Rebuilds the class hierarchy structure from the given SymbolList."

  | allClasses symlist dict rootIsObj done rootHist allTraits |
  allClasses := ClassSet new.   " make a list of all the named classes and trait instances "
  allTraits := IdentitySet new.
  symlist := Array withAll: aSymbolList.
  rootIsObj := rootClass superclass isNil.
  rootIsObj
    ifTrue: [ rootHist := IdentitySet new ]
    ifFalse: [
      rootHist := IdentitySet withAll: rootClass classHistory.
      rootHist remove: rootClass ].
  done := IdentitySet new.
  [ symlist size > 0 ]
    whileTrue: [
      dict := symlist at: 1.
      (done includesIdentical: dict)
        ifFalse: [
          dict
            valuesDo: [ :aValue |
              | anObj |
              anObj := aValue.
              anObj isBehavior
                ifTrue: [
                  anObj isMeta
                    ifTrue: [ anObj := anObj thisClass ].
                  (rootIsObj
                    or: [ (anObj _subclassOf: rootClass) or: [ rootHist includes: anObj ] ])
                    ifTrue: [ allClasses add: anObj ] ]
                ifFalse: [
                  anObj isTrait
                    ifTrue: [ allTraits add: anObj ] ] ].
            done add: dict ].
      symlist removeFrom: 1 to: 1 ].
  rootIsObj
    ifFalse: [
      | cls |
      "now add superclasses up to object"
      cls := rootClass superclass.
      [ cls ~~ nil ]
        whileTrue: [
          allClasses add: cls.
          cls := cls superclass ] ].
  classes := allClasses.
  self traits: allTraits
]

{ #category : 'Private' }
ClassOrganizer >> collectClassNames [

"Causes the receiver to collect all of the names of classes from the
 current collection of classes and form an auto-completer for the names."

| arr sz |

sz := classes size .
arr := Array new: sz .
1 to: classes size do: [:i | arr at: i put:((classes _at: i) name)  ].

classNames := { AutoComplete new strings: arr . nil }.
^ classNames

]

{ #category : 'Fileout Aids' }
ClassOrganizer >> determineClassFileoutOrder: classdict [

"Returns an ordered collection of the values that are classes in classdict,
 specifying the order of fileout.  The argument should be a SymbolDictionary."

| org result classSet block |

( classdict isEmpty or:
[ nil == (classdict associationsDetect: [ :assoc | assoc value isBehavior ] ifNone: [ nil ]) ])
  ifTrue: [ ^ { } ].

"form a hierarchy for the set of classes"
org := Dictionary new.
org at: #nil put: ClassSet new.

classSet := ClassSet new.
classdict associationsDo: [:assn | | sub |
  sub := assn value.
  sub isBehavior ifTrue: [ | superCls |
    classSet add: sub.
    [ superCls := self _superClass: sub .
      superCls ~~ nil and:[ superCls ~~ #nil] ] whileTrue: [ | assoc |
      assoc := org associationAt: superCls otherwise: nil.
      assoc ifNil: [
        assoc := Association newWithKey: superCls value: ClassSet new.
          org add: assoc
      ].
      assoc value add: sub.
      sub := superCls
    ].
    (org at: #nil) add: sub.
  ].
].

"make a recursive block to order the subclass sets and weed out unwanted
 classes"
result := { } .
block := [:order: subs |
  1 to: subs size do: [:i | | assoc class |
    class := subs at: i.
    (classSet includesIdentical: class) ifTrue: [
      order add: class
    ].
    assoc := org associationAt: class otherwise: nil.
    (assoc ~~ nil and: [assoc value size > 0]) ifTrue: [
      block value: order value: (self sortClasses: assoc value )
    ]
  ]
].

block value: result value: (self sortClasses:(org at: #nil) ).
^ result

]

{ #category : 'Fileout Aids' }
ClassOrganizer >> determineTraitFileoutOrder: traitdict [

"Returns an ordered collection of the values that are traits in traitdict,
 specifying the order of fileout.  The argument should be a SymbolDictionary."

| traitSet |

traitSet :=  traitdict select: [ :aValue | aValue isTrait ].
traitSet isEmpty
  ifTrue: [ ^ { } ].

^ traitSet sort: [:a :b | a name < b name ]
]

{ #category : 'EnvironmentId' }
ClassOrganizer >> environmentId [
  "Return the environmentId"

  ^ (self dynamicInstVarAt: #envId) ifNil:[ 0 ]

]

{ #category : 'EnvironmentId' }
ClassOrganizer >> environmentId: envId [
  "Set the environmentId for subsequent loading of classes
   and reporting results.
   Normally done only just after instance creation."

  self dynamicInstVarAt: #envId put: envId

]

{ #category : 'Fileout Aids' }
ClassOrganizer >> fileOutClasses: order on: stream [

"Writes out code in topaz filein format on the given stream, that creates the
 given classes.  order arg is an array of classes."

	1 to: order size
		do:
			[:j | | cls |
			cls := order at: j.
			cls fileOutClassDefinitionOn: stream.
            ].

]

{ #category : 'Fileout Aids' }
ClassOrganizer >> fileOutClasses: order on: stream inDictionary: dict named: dictName [
	"Writes out code on the given stream that creates the given classes in the
	 dictionary with the given name.  The dict argument should be a SymbolDictionary
	 of classes."

	| class lf head term nm |

self deprecated: 'ClassOrganizer >> fileOutClasses:on:inDictionary:named: deprecated v3.2.  Use fileOutClasses:on: instead'.

	order size == 0 ifTrue: [^self].
	head := 'doit
'.
	term := '
' , '%
'.
	lf := Character lf.

	"Determine the dictionary name to use"
	nm := dictName.
	1 to: order size
		do:
			[:j |
			class := order at: j.
			class fileOutPreClassOn: stream.
			stream
				nextPutAll: head;
				nextPutAll: (class _modifiableDefinitionInDictionary: dict named: nm);
				nextPut: $.;
				nextPutAll: term.
			class fileOutCommentOn: stream].

	"now make non-modifiable classes non-modifiable "
	1 to: order size
		do:
			[:k |
			class := order at: k.
			class isModifiable
				ifFalse:
					[stream
						nextPutAll: head;
						nextPutAll: (dict keyAtValue: class);
						nextPutAll: ' immediateInvariant.';
						nextPutAll: term]].

]

{ #category : 'Fileout Aids' }
ClassOrganizer >> fileOutClassesAndMethodsInDictionary: aSymbolDictionary on: aStream [

"Files out all source code for classes in aSymbolDictionary in Topaz filein
 format on aStream."

| order |
order := self determineClassFileoutOrder: aSymbolDictionary.
self fileOutClasses: order on: aStream.
self fileOutMethods: aSymbolDictionary order: order on: aStream.
self fileOutTraits: aSymbolDictionary order: order on: aStream
]

{ #category : 'Fileout Aids' }
ClassOrganizer >> fileOutMethods: classdict order: order on: stream [

"File out each class's code and embedded classes."

| class |
classdict size == 0 ifTrue: [ ^self ].
"put the class dictionary in the temporary symbol list so it will be used for
  name resolution during method fileout"
1 to: order size do: [:l |
  class := order at: l.
  class fileOutCategoriesOn: stream.
].

]

{ #category : 'Fileout Aids' }
ClassOrganizer >> fileOutOtherMethods: methodInfo on: stream [

"Files out a set of methods on the given stream/file.  methodInfo must be an
 Array of pairs: #(class selector)."

| pair sz |
(sz := methodInfo size) > 0 ifTrue:[
  stream nextPutAll:'set compile_env: 0'; lf .
  1 to: sz do: [:i |
    pair := methodInfo at: i.
    "use the fileOutMethod: version here so the category will be included"
    (pair at: 1) fileOutMethod: (pair at: 2) on: stream .
  ].
].

]

{ #category : 'Fileout Aids' }
ClassOrganizer >> fileOutTraitDefinitions: order on: stream [

"Writes out code in topaz filein format on the given stream, that creates the
 given traits.  order arg is an array of traits."

	1 to: order size
		do:
			[:j | | trait |
			trait := order at: j.
			trait fileOutTraitOn: stream.
            ].
]

{ #category : 'Fileout Aids' }
ClassOrganizer >> fileOutTraits: classdict order: order on: stream [

"File out each class's trait registrations."

| class |
classdict size == 0 ifTrue: [ ^self ].
"put the class dictionary in the temporary symbol list so it will be used for
  name resolution during method fileout"
1 to: order size do: [:l |
  class := order at: l.
  class fileOutTraitsOn: stream.
].
]

{ #category : 'Fileout Aids' }
ClassOrganizer >> fileOutTraitsClassesAndMethodsInDictionary: aSymbolDictionary on: aStream [

"Files out all source code for classes in aSymbolDictionary in Topaz filein
 format on aStream."

| traitsOrder order |
traitsOrder := self determineTraitFileoutOrder: aSymbolDictionary.
self fileOutTraitDefinitions: traitsOrder on: aStream.
order := self determineClassFileoutOrder: aSymbolDictionary.
self fileOutClasses: order on: aStream.
self fileOutMethods: aSymbolDictionary order: order on: aStream.
self fileOutTraits: aSymbolDictionary order: order on: aStream
]

{ #category : 'Accessing' }
ClassOrganizer >> hierarchy [

"Returns an IdentityDictionary which is the value of the instance variable 'hierarchy'.
 In this dictionary the value for key #nil is the list of classes whose superclass is nil.
 Otherwise all keys in the dictionary are classes. "

^hierarchy

]

{ #category : 'Reports' }
ClassOrganizer >> hierarchyReport [

"Returns a String that is a class hierarchy report for all classes
 known to the receiver."

| report |
report := String new .
self _hierarchyReportForClass: Object indent: '' report: report withOops: false.
rootClass == Object ifTrue:[ | cset |
  cset := self subclassesOf: #nil .
  cset remove: Object .
  cset do:[: aCls |
    self _hierarchyReportForClass: aCls indent: '' report: report withOops: false .
  ].
].
^ report

]

{ #category : 'Queries' }
ClassOrganizer >> implementorsOf: aSelector [

"Returns a collection of GsNMethods that implement the given selector."

^self implementorsOf: aSelector in: classes

]

{ #category : 'Queries' }
ClassOrganizer >> implementorsOf: aSelector in: aclassSet [

"Returns a collection of GsNMethods that implement the given selector, limited to
 classes in aclassSet. Does not distinguish between class and instance methods; the
 elements in aclassSet are expected to be classes, not metaclasses ."

| result cset methsSet env |

methsSet := IdentitySet new .
result := { } .
cset := self sortClasses: aclassSet .
env := self environmentId .
1 to: cset size do: [ :j | | cls mDict meth |
  2 timesRepeat:[
    cls ifNil:[ cls := cset at: j ] ifNotNil:[ cls := cls class].
    mDict := cls _fullMethodDictEnv: env . "only env 0 includes GsPackagePolicy"
    meth := mDict at: aSelector otherwise: nil .
    meth ifNotNil: [ (methsSet _addIfAbsent: meth) ifTrue:[ result add: meth ]].
  ].
].
^result

]

{ #category : 'Reports' }
ClassOrganizer >> implementorsOfByCategoryReport: aSelector [
 "Returns a String describing the methods that are implementors of the specified
  selector, sorted into method categories"

  ^ self _asCategoriesReportString: (self implementorsOf: aSelector)

]

{ #category : 'Reports' }
ClassOrganizer >> implementorsOfReport: aSelector [

"Returns a String describing the methods that are implementors of the specified
 selector."

^ self _asReportString: (self implementorsOf: aSelector) indent: ''

]

{ #category : 'Report Shortcuts' }
ClassOrganizer >> impls: aSelector [
  "a variant of implementorsOfReport: easier to type when using topaz ."
  ^ self implementorsOfReport: aSelector

]

{ #category : 'Report Shortcuts' }
ClassOrganizer >> implsC: aSelector [
  "shortcut method for implementorsOfByCategoryReport:"
  ^ self _asCategoriesReportString: (self implementorsOf: aSelector)

]

{ #category : 'Accessing' }
ClassOrganizer >> includeDeprecatedMethodsInReports [

  ^ (self dynamicInstVarAt: #includeDeprecated) ~~ false

]

{ #category : 'Updating' }
ClassOrganizer >> includeDeprecatedMethodsInReports: aBoolean [

  "Controls output from the various Report methods which use
   _asReportString:  ,  such as sendersOfReport: .
  An argument of false will cause the reports to omit deprecated methods."

  self dynamicInstVarAt: #includeDeprecated put: aBoolean .

]

{ #category : 'Accessing' }
ClassOrganizer >> includeMethodOops [
  ^ (self dynamicInstVarAt: #methodOops) == true

]

{ #category : 'Updating' }
ClassOrganizer >> includeMethodOops: aBoolean [
  ^ self dynamicInstVarAt: #methodOops put: aBoolean .

]

{ #category : 'Reports' }
ClassOrganizer >> literalsReport: aLiteral [

"Returns a String describing the methods whose source contains
 a literal reference to the specified literal.
 The argument may be any object that is legal as a literal,
 including a String, Symbol, Number, Array, Boolean, nil.
 If aLiteral is a SymbolAssociation,
 references to either aLiteral or (aLiteral value) will be returned.
 "

^ self _asReportString:
      ((self _referencesToLiteral: aLiteral in: classes withOffsets:false ) at: 1)

]

{ #category : 'Modifying Classes' }
ClassOrganizer >> makeInstancesNonPersistent: aClass [

"Recursively make aClass and all subclasses non-persistent.

 If an error occurs, the session will be unable to commit and
 must logout/login before commits are allowed again."

<primitive: 2001>
| prot |
prot := System _protectedMode .
[
  aClass instancesNonPersistent ifFalse:[ | success |
    [ | subCs |
      success := false .
      aClass _makeInstancesNonPersistent .
      subCs := self subclassesOf: aClass .
      subCs do:[ :aSubcls |
	self makeInstancesNonPersistent: aSubcls .
      ].
      success := true .
    ] ensure:[
      success ifFalse:[ System _disallowCommitClassModFailure ].
    ]
  ].
] ensure:[
  prot _leaveProtectedMode
]

]

{ #category : 'Modifying Classes' }
ClassOrganizer >> makeInstancesPersistent: aClass [

"Recursively make aClass and all subclasses persistable.

 If an error occurs, the session will be unable to commit and
 must logout/login before commits are allowed again."

<primitive: 2001>
| prot |
prot := System _protectedMode .
[
  aClass instancesNonPersistent ifTrue:[ | success |
    [ | subCs |
      success := false .
      aClass _makeInstancesPersistent .
      subCs := self subclassesOf: aClass .
      subCs do:[ :aSubcls |
	self makeInstancesPersistent: aSubcls .
      ].
      success := true .
    ] ensure:[
      success ifFalse:[ System _disallowCommitClassModFailure ].
    ]
  ].
] ensure:[
  prot _leaveProtectedMode
]

]

{ #category : 'Reports' }
ClassOrganizer >> methodCategories [
  "Returns a String containing a report of all the method categories"
| report cats |
cats := SortedCollection withAll: self _methodCategories .
report := String new.
cats do:[ :aCateg| report add: aCateg ; lf ].
^ report

]

{ #category : 'Reports' }
ClassOrganizer >> methodsInCategory: aString [
  "Return a String containing a report of all methods in the specified category"
| sym |
sym := Symbol _existingWithAll: aString .
sym ifNil:[ ^ '' ].
^ self _asReportString:( self _methodsInCategory: [:categ| sym == categ])

]

{ #category : 'Private' }
ClassOrganizer >> methodsInCategoryMatching: aSubString [
  "Return a String containing a report of all methods whose category
   includes aSubString (case-insenitive)"
  | ucStr |
  ucStr := aSubString asUppercase .
^ self _asReportString:(
    self _methodsInCategory: [:categ | categ asUppercase includesString: ucStr ])

]

{ #category : 'Report Shortcuts' }
ClassOrganizer >> methsInCat: aString [
  "Convenience method to access methodsInCategory:, returning a String containing
  a report of all methods in the specified category"

^ self methodsInCategory: aString

]

{ #category : 'Queries' }
ClassOrganizer >> nonOptimizedSendersOf: aSelector [

^ self _sendersOf: aSelector in: classes includeOptimized: false

]

{ #category : 'Reports' }
ClassOrganizer >> nonOptimizedSendersOfReport: aSelector [

"Returns a String describing the methods that are senders of the specified
 selector."

^ self _asReportString: ((self nonOptimizedSendersOf: aSelector) at:1 )

]

{ #category : 'Reports' }
ClassOrganizer >> packageSendersOfReport: aSelector [
  "Returns a String describing the methods that are senders of the
   specified selector, and for which the methods' category begins with '*' "

^ self _asReportString:(
  ((self sendersOf: aSelector) at:1 ) select:[ :meth |
       ((meth inClass categoryOfSelector: meth selector ) at: 1) == $*  ])

]

{ #category : 'Report Shortcuts' }
ClassOrganizer >> pkgsends: aSelector [
"Convenience shortcut to access packageSendersOfReport:"

  ^ self packageSendersOfReport: aSelector

]

{ #category : 'Private' }
ClassOrganizer >> rebuildCategories [

| cls arr cat catd theClasses |

   "gemstone64,  changed categories from a SymbolDictionary
    to a StringKeyValueDictionary to avoid unnecessary symbol creation"

catd := StringKeyValueDictionary new.
categories := catd .
theClasses := classes .
1 to: theClasses size do: [:i |
  cls := theClasses _at: i.
  cat := cls category.
  arr := catd at: cat otherwise: nil.
  arr ifNil: [
    arr := { }.
    catd at: cat put: arr
  ].
  arr add: cls
].

]

{ #category : 'Private' }
ClassOrganizer >> rebuildHierarchy [

"Builds an IdentityDictionary containing all classes as keys and ClassSets of
 their subclasses as values.  As a side-effect, the classes collection is
 expanded to include any superclasses that aren't in the user's name space."

| hier supers each sup c symList |

hier := IdentityDictionary new.
hier at: #nil put: ClassSet new .
supers := ClassSet new.
symList := (self dynamicInstVarAt: #restrictedSymbolList) ifNotNil:[ user ].

"add each class to its superclass's subclass set"
1 to: classes size do: [:i |
  each := classes _at: i.
  sup := self _superClass: each .
	"if not using a restricted symbolList, keep track of superclasses that aren't in the class set"
  (sup ~~ #nil and:[ symList ~~ nil ]) ifTrue:[ 
    (symList resolveSymbol: sup name) ifNil:[ sup := #nil ].
  ].
  sup ~~ #nil ifTrue:[ | aCls |
    aCls := sup .
    [aCls == #nil or:[ classes includesIdentical: aCls]] whileFalse:[
      supers add: aCls .
      aCls := self _superClass: aCls .
      aCls ifNil:[ aCls := #nil ].
    ].
    (c := hier at: sup otherwise: nil) ifNil: [
      c := ClassSet new.
      hier at: sup put: c.
    ].
    c add: each
  ] ifFalse:[
    (hier at: #nil) add: each
  ].
].

"get rid of superclasses that have already been processed"
supers := supers - classes.

"complete the class hierarchies for superclasses that weren't in the
  initial set of classes"
1 to: supers size do: [:i |
  each := supers _at: i.
  sup := self _superClass: each .
  (c := hier at: sup otherwise: nil) ifNil: [
    c := ClassSet new.
    hier at: sup put: c.
  ].
  c add: each
].

classes := classes + supers.
hierarchy := hier

]

{ #category : 'Updating' }
ClassOrganizer >> recategorize: class to: newCategory [

"Move the class from its present category to the given category."

| oldcat oldlist newlist idx |
oldcat := class category.
class category: newCategory.
oldlist := categories at: oldcat otherwise: nil.
oldlist ifNotNil: [
  idx := oldlist indexOf: class.
  idx > 0 ifTrue: [
    oldlist removeFrom: idx to: idx
  ]
].
newlist := categories at: newCategory otherwise: nil.
newlist ifNil: [
  categories at: newCategory put: { class }
] ifNotNil: [
  newlist add: class
]

]

{ #category : 'Queries' }
ClassOrganizer >> referencesTo: aSymbol [

"Returns an Array of two sequenceable collections.  The first contains
 GsNMethods that refer to the given symbol, and the second contains
 corresponding indexes into sourceStrings where the first reference takes
 place."

^self referencesTo: aSymbol in: classes

]

{ #category : 'Queries' }
ClassOrganizer >> referencesTo: aSymbol in: aclassSet [

"Returns an Array of two sequenceable collections.  The first contains
 GsNMethods that reference the given symbol as a literal,
 and the second contains corresponding indexes into sourceStrings where
 the first such reference takes place."

| result resulti cset cls methsSet sym val env |

methsSet := IdentitySet new .
result := { } .
resulti := { } .
sym := Symbol _existingWithAll: aSymbol .
sym ifNotNil:[
  env := self environmentId .
  (GsCurrentSession currentSession resolveSymbol: sym) ifNotNil:[:assoc|
    assoc isInvariant ifTrue:[
      val := assoc _value "pick up optimized literals when searching"
    ] .
  ].
  cset := self sortClasses: aclassSet .
  1 to: cset size do: [ :i |
    cls := cset at: i.
    2 timesRepeat:[ | mDict |
      mDict := cls _fullMethodDictEnv: env .
      mDict valuesDo:[ :method |
	(method _literalsIncludesSymbol: sym value: val ) ifTrue:[
	   (methsSet _addIfAbsent: method) ifTrue:[
	     result add: method.
	     resulti add: (method sourceString findString: sym  startingAt: 1)
	   ].
	 ].
      ].
      cls := cls class .
    ].
  ].
].
^{ result . resulti }

]

{ #category : 'Queries' }
ClassOrganizer >> referencesToLiteral: aLiteral [
  "Returns an Array of two Arrays.  The first subarray contains GsNMethods
 that reference given literal.  The second subarray contains indexes where the
 first reference to the literal occurs within the sourceString of the method."

  ^ self referencesToLiteral: aLiteral in: classes

]

{ #category : 'Queries' }
ClassOrganizer >> referencesToLiteral: aLiteral in: aclassSet [
  "Returns an Array of two Arrays.  The first subarray contains GsNMethods
 that reference given literal.  The second subarray contains indexes where the
 first reference to the literal occurs within the sourceString of the method.
 Results are limited to classes in aclassSet. The elements in aclassSet are
 expected to be classes, not metaclasses .
 If aLiteral is an invariant SymbolAssociation,
 references to either aLiteral or (aLiteral value) will be returned."

 ^ self _referencesToLiteral: aLiteral in: aclassSet withOffsets: true

]

{ #category : 'Queries' }
ClassOrganizer >> referencesToObject: anObject [

"Returns an Array of GsNMethods that reference the given object
 (through a variable binding)"

^self referencesToObject: anObject in: classes

]

{ #category : 'Queries' }
ClassOrganizer >> referencesToObject: anObject in: aclassSet [

"Returns an Array of GsNMethods that reference the given object
 (through a variable binding). Results are limited to classes in aclassSet.
 The elements in aclassSet are expected to be classes, not metaclasses."

| result cset cls env |

result := { } .
env := self environmentId .
cset := self sortClasses: aclassSet .
1 to: cset size do: [ :i |
  cls := cset at: i.
  2 timesRepeat:[ | mDict |
    mDict := cls _fullMethodDictEnv: env .
    mDict valuesDo:[ :method |
        (method _literalsIncludesValue: anObject) ifTrue:[
           result add: method
        ].
    ].
    cls := cls class .
  ].
].
^result

]

{ #category : 'Accessing' }
ClassOrganizer >> rootClass [

"Returns the root class for this organizer."

^rootClass

]

{ #category : 'Updating' }
ClassOrganizer >> rootClass: aClass [

"Sets the root class of the receiver's hierarchy.  Not generally a useful
 thing to do."

rootClass := aClass

]

{ #category : 'Queries' }
ClassOrganizer >> searchForCategory: catname in: classSet [

  "Returns a collection of GsNMethods in the given category."
| result cset catSym envId |
result := { } .
envId := 0 .
catname _validateByteClass: CharacterCollection.
catSym := Symbol _existingWithAll: catname .
catSym ifNil:[ ^ result ].
cset := self sortClasses: classSet .
1 to: cset size do: [ :j |
  { cset at: j . (cset at: j) class } do: [ :cls | | mdict |
    cls categorysDo:[ :cat :selectors |
      cat == catSym ifTrue:[
	mdict := cls methodDictForEnv: envId .
	selectors do: [ :selector | | method |
	  (method := mdict at: selector otherwise: nil) ifNil:[ |emsg|
            emsg := 'Missing method ' , selector quoted ,
			 ' in category ' , catSym , ' in class ' , cls name.
            [
	      self notify: emsg   "notify: only available with Seaside/Ruby"
            ] onSynchronous: MessageNotUnderstood do:[:ex |
              self error: emsg
            ].
	  ] ifNotNil: [
	    result add: method
	  ].
	].
      ].
    ].
  ].
].
^result

]

{ #category : 'Queries' }
ClassOrganizer >> sendersOf: aSelector [

"Returns an Array of two Arrays.  The first subarray contains GsNMethods
 that send the given selector.  For non-optimized selectors,
 the second subarray contains indexes where the first use of the
 selector occurs within the sourceString of the method."

^self sendersOf: aSelector in: classes

]

{ #category : 'Report Shortcuts' }
ClassOrganizer >> sendersOf: aSelector in: aclassSet [

"Returns an Array of two Arrays.  The first subarray contains GsNMethods
 that send the given selector.

 For non-optimized selectors, the second subarray contains indexes where
 the first use of the selector occurs within the sourceString of the method.

 Results are limited to classes in aclassSet. The elements in aclassSet are
 expected to be classes, not metaclasses ."

^ self _sendersOf: aSelector in: aclassSet includeOptimized: true

]

{ #category : 'Reports' }
ClassOrganizer >> sendersOfByCategoryReport: aSelector [
  "Return a String describing the methods that send the given selector,
   sorted into method categories."

  ^ self _asCategoriesReportString: ((self sendersOf: aSelector) at:1 )

]

{ #category : 'Reports' }
ClassOrganizer >> sendersOfReport: aSelector [

"Returns a String describing the methods that are senders of the specified
 selector."

^ self _asReportString: ((self sendersOf: aSelector) at:1 )

]

{ #category : 'Report Shortcuts' }
ClassOrganizer >> sends: aSelector [
  "a variant of sendersOfReport: easier to type when using topaz ."
  ^ self sendersOfReport: aSelector

]

{ #category : 'Report Shortcuts' }
ClassOrganizer >> sendsC: aSelector [
  "convenience method for sendersOfByCategoryReport, returning a string
   with methods that send the argument, sorted by method categories"
  ^ self sendersOfByCategoryReport: aSelector

]

{ #category : 'Report Shortcuts' }
ClassOrganizer >> sendsNotImpl [
"A convenience shortcut for sentButNotImplementedReport "

 ^ self sentButNotImplementedReport

]

{ #category : 'Queries' }
ClassOrganizer >> sentButNotImplemented [
 "Returns an Array of selectors which are sent but not implemented"

  | sent implemented env |
  sent := IdentitySet new.
  implemented := IdentitySet new .
  env := self environmentId .
  classes do:[ :aClass | | cls |
    cls := aClass .
    2 timesRepeat:[ | mDict |
      mDict := cls _fullMethodDictEnv: env .
      mDict valuesDo:[ :method |
        implemented add:  method selector .
        sent addAll:  method _selectorPool
      ].
      cls := cls class .
    ].
  ].
  ^ Array withAll:( SortedCollection withAll: (sent - implemented))

]

{ #category : 'Reports' }
ClassOrganizer >> sentButNotImplementedReport [
  "Returns a report of selectors sent but not implemented."
  | report LF |
  report := String new .
  LF := Character lf .
  self sentButNotImplemented do:[:sel | | methsRpt |
    report addAll: sel ; add: LF  .
    methsRpt := self _asReportString:((self sendersOf: sel) at:1) indent: '   '.
    methsRpt size > 0
      ifTrue:[  report addAll: methsRpt ]
      ifFalse:[ report addAll: '   <no sender methods found>'; add: LF ].
  ].
  ^ report

]

{ #category : 'Private' }
ClassOrganizer >> sortClasses: aCollection [

 ^ SortedCollection withAll: aCollection
    sortBlock:[:x :y | x name <= y name ]

]

{ #category : 'Report Shortcuts' }
ClassOrganizer >> strings: aString [

"A variant of substringReport:  . Used by topaz <= v3.5 " 

^ self substringReport: aString

]

{ #category : 'Report Shortcuts' }
ClassOrganizer >> stringsC: aString [

"Shortcut method for substringByCategoryReport:"

^self substringByCategoryReport: aString

]

{ #category : 'Report Shortcuts' }
ClassOrganizer >> stringsIc: aString [

"Used by topaz <= v3.5"

^ self substringIgnoreCaseReport: aString

]

{ #category : 'Report Shortcuts' }
ClassOrganizer >> stringsIcC: aString [

"Shortcut method for substringIgnoreCaseByCategoryReport:"

^ self substringIgnoreCaseByCategoryReport: aString

]

{ #category : 'Reports' }
ClassOrganizer >> stringsReport: aString ignoreCase: icBool includeClassComments: commentsBool [
 "used by topaz"

  | arr rpt |
  arr := self _substringSearch: aString in: classes ignoreCase: icBool . 
  rpt := self _asReportString: (arr at: 1 ).
  commentsBool ifTrue:[
    (arr at: 3) do:[:cls |  rpt add: cls name ; add:' comment' ; lf ].
  ].
  ^ rpt 
]

{ #category : 'Queries' }
ClassOrganizer >> subclassesOf: aClass [

"Returns a copy of the set of subclasses for the given class.
 Generates an error if the receiver does not hold the given class."

| s |
classes ifNil: [
  self updateClassInfo
].
aClass ifNil:[
  ^ (hierarchy at: #nil otherwise: nil)
      ifNotNil:[ :aSet | aSet copy ] ifNil:[ ClassSet new ]
].
(s := hierarchy at: aClass otherwise: nil) ifNil:[
  (rootClass == Object or: [aClass isSubclassOf: rootClass]) ifTrue: [
    self addClass: aClass .
    (classes includesIdentical: aClass) ifFalse:[
      Error signal:'add class failed'.
    ].
    s := hierarchy at: aClass otherwise: nil .
  ].
].
^ s ifNotNil:[ s copy ] ifNil:[ ClassSet new ]

]

{ #category : 'Reports' }
ClassOrganizer >> subclassesReport: aClass [

 "Return a String listing all subclasses of aClass, sorted by name, each class
  on a line."

  ^ self subclassesReport: aClass includeOops: false

]

{ #category : 'Reports' }
ClassOrganizer >> subclassesReport: aClass includeOops: withOopsBool [

 "Return a String listing all subclasses of aClass, sorted by name, each class
  on a line.  If withOopsBool is true, include the oop of the class"

 | sorted str |
 sorted := self sortClasses: (self subclassesOf: aClass) .
 str := String new .
 sorted do:[ :cls |
   str add: cls name .
   withOopsBool ifTrue:[ str add: '   '; add: cls asOop asString ].
   str lf .
 ].
 ^ str

]

{ #category : 'Reports' }
ClassOrganizer >> subhierarchyReport: aClass includeOops: withOopsBool [
| r |
r := String new .
self _hierarchyReportForClass: aClass indent: '' report: r withOops: withOopsBool
	withInstvars: true .
^ r

]

{ #category : 'Reports' }
ClassOrganizer >> substringByCategoryReport: aString [

"Return a string containing the methods that include the given string, case
  senstitive, sorted into method categories."

^ self _asCategoriesReportString:
      ((self _substringSearch: aString in: classes ignoreCase: false) at: 1)

]

{ #category : 'Reports' }
ClassOrganizer >> substringIgnoreCaseByCategoryReport: aString [

"Returns a String describing the methods whose source contains
 the specified string, ignoring case. The methods are sorted into
 method categories"

^ self _asCategoriesReportString:
      ((self _substringSearch: aString in: classes ignoreCase: true) at: 1)

]

{ #category : 'Reports' }
ClassOrganizer >> substringIgnoreCaseReport: aString [

"Returns a String describing the methods whose source contains
 the specified string, ignoring case."

^ self _asReportString:
      ((self _substringSearch: aString in: classes ignoreCase: true) at: 1)

]

{ #category : 'Reports' }
ClassOrganizer >> substringReport: aString [

"Returns a String describing the methods whose source contains
 the specified string ."

^ self _asReportString:
      ((self _substringSearch: aString in: classes ignoreCase: false) at: 1)

]

{ #category : 'Queries' }
ClassOrganizer >> substringSearch: aString [

 "Search for methods and class comments that include the given substring. 
  Search is case senstive.  

  Returns an Array of three Arrays.  The first subarray contains GsNMethods
  whose sources include the given substring.  The second subarray contains
  indexes where the first occurrence of the substring was found. The third
  array includes classes, for which the result of executing aClass comment
  contains the specified string."

^self substringSearch: aString in: classes

]

{ #category : 'Queries' }
ClassOrganizer >> substringSearch: aString ignoreCase: ignoreCase [

 "Search for methods and class comments that include the given substring.  
  Search is case insensitive or case sensitive, depending on ignoreCase. 

  Returns an Array of three Arrays.  The first subarray contains GsNMethods
  whose sources include the given substring.  The second subarray contains
  indexes where the first occurrence of the substring was found. The third
  array includes classes, for which the result of executing aClass comment 
  contains the specified string."

^ self _substringSearch: aString in: classes ignoreCase: ignoreCase

]

{ #category : 'Queries' }
ClassOrganizer >> substringSearch: aString in: aclassSet [

 "Search for methods and class comments within the given set of classes that 
  include the given substring. Search is case senstive.  

  Returns an Array of three Arrays.  The first subarray contains GsNMethods
  whose sources include the given substring.  The second subarray contains
  indexes where the first occurrence of the substring was found. The third
  array includes classes, for which the result of executing aClass comment
  contains the specified string.

  Results are limited to classes in <aclassSet>. The elements in <aclassSet>
  are expected to be classes, not metaclasses."

^ self substringSearch: aString in: aclassSet ignoreCase: false

]

{ #category : 'Queries' }
ClassOrganizer >> substringSearch: aString in: aclassSet ignoreCase: ignoreCase [

 "Search for methods and class Comments within the given set of classes that 
  include the given substring. Search is case insensitive or case sensitive, 
  depending on ignoreCase. 

  Returns an Array of three Arrays.  The first subarray contains GsNMethods
  whose sources include the given substring.  The second subarray contains
  indexes where the first occurrence of the substring was found. The third
  array includes classes, for which the result of executing aClass comment
  contains the specified string.

  Results are limited to classes in <aclassSet>. The elements in <aclassSet> 
  are expected to be classes, not metaclasses."

^ self _substringSearch: aString in: aclassSet ignoreCase: ignoreCase

]

{ #category : 'Accessing' }
ClassOrganizer >> symbolList [
  "Returns the symbolList used by the receiver"
  ^ user ifNil: [ GsCurrentSession currentSession symbolList ]

]

{ #category : 'Reports' }
ClassOrganizer >> traitImplementorsOfReport: aSelector [
	"Returns a String describing the methods that are implementors of the specified
 selector in Traits."

	| result LF arr selector |
	selector := aSelector asSymbol.
	LF := Character lf.
	arr := SortedCollection new.
	self traits
		do: [ :aTrait | 
			| str name |
			name := aTrait name.
			(aTrait localSelectors includes: selector)
				ifTrue: [ 
					str := String withAll: name.
					str
						addAll: ' >> ';
						addAll: selector.
					arr add: str ].
			(aTrait classTrait localSelectors includes: selector)
				ifTrue: [ 
					str := String withAll: name , ' classTrait'.
					str
						addAll: ' >> ';
						addAll: selector.
					arr add: str ] ].
	result := String new.
	1 to: arr size do: [ :j | 
		result
			add: (arr at: j);
			add: LF ].
	^ result
]

{ #category : 'Accessing' }
ClassOrganizer >> traits [
  "Returns the list of Traits found by the receiver during the class scan "
  ^ self dynamicInstVarAt: #traits

]

{ #category : 'Accessing' }
ClassOrganizer >> traits: aSet [
  "Sets the list of Traits found by the receiver during the class scan "
  ^ self dynamicInstVarAt: #traits put: aSet

]

{ #category : 'Reports' }
ClassOrganizer >> traitStringsReport: aString ignoreCase: icBool includeTraitComments: commentsBool [
	"used by topaz"

	| arr rpt sorted |
	arr := self _traitSubstringSearch: aString in: self traits ignoreCase: icBool.
	sorted := SortedCollection withAll: (arr at: 1).
	rpt := String new.
	1 to: sorted size do: [ :j | 
		rpt
			add: (sorted at: j);
			lf ].
	commentsBool
		ifTrue: [ 
			(arr at: 3)
				do: [ :cls | 
					rpt
						add: cls name;
						add: ' comment';
						lf ] ].
	^ rpt
]

{ #category : 'Class Collection' }
ClassOrganizer >> update [

"Causes the receiver to rescan for classes and rebuild internal
 structures.  Synonymous with updateClassInfo."

self updateClassInfo

]

{ #category : 'Class Collection' }
ClassOrganizer >> updateClassInfo [

"Causes the receiver to rescan for classes and rebuild internal structures."

self collectClasses; "find all visible classes"
    rebuildHierarchy; "build the hierarchy, fleshing out class set too"
    collectClassNames; "pull out the names of the classes and form auto-complete set"
    rebuildCategories "build the class categories"

]
