"
A client forwarder is an instance of the new class ClientForwarder in GemStone.
 When a client forwarder is sent a message during GemStone Smalltalk execution,
 GemBuilder for Smalltalk directs that message to the client object that
 corresponds to the ClientForwarder.  Arguments to the message are replicated as
 necessary, and when client execution completes, the result is replicated back
 into GemStone as the result of the message send to the ClientForwarder.

 This class is considered private and for use by GemStone implementors only.
 Design of an error-handling mechanism that integrates the exception mechanisms
 of the client and GemStone can is deferred until a later release.

 If the execution stack containing the forwarded send includes client
 stack frames down the stack from the GemStone method that invoked the send
 to the client, the exception handler environment in the context of the
 forwarded method execution will include exceptions in the previous client
 stack, but will not include exceptions in GemStone.  That is, exception
 handlers installed in the GemStone part of the stack will be ignored, and
 an exception handler in a previous client invocation may be invoked.

 If an exception handler executing in a previous client stack section above
 the GemStone Activation that sent the forwarded message proceeds in a
 manner which discards stack above the frame where the exception is
 handled, the entire section of GemStone stack will be discarded.

Constraints:
	clientObject: Integer
"
Class {
	#name : 'ClientForwarder',
	#superclass : 'nil',
	#instVars : [
		'clientObject'
	],
	#gs_reservedoop : '109569',
	#category : nil
}

{ #category : 'Class Membership' }
ClientForwarder >> _class [
"Returns the object that is the receiver's Smalltalk class."

<primitive: 610 >
^ self _primitiveFailed: #_class .

]

{ #category : 'Accessing' }
ClientForwarder >> clientObject [

"Returns the OOP (in the form of some kind of Integer) in the client object
 space of the object to which messages are to be forwarded."

^ clientObject

]

{ #category : 'Updating' }
ClientForwarder >> clientObject: anInteger [

""

clientObject := anInteger

]

{ #category : 'Compatibility' }
ClientForwarder >> doesNotUnderstand: aMessageDescriptor [

"Sends the message described by aMessageDescriptor to the client object
 represented by the value of the receivers' instance variable clientObject.
 Returns the result from executing the forwarded message in the client."

  | res |
  res := ClientForwarderSend new
      receiver: self
           clientObj: clientObject
           selector: (aMessageDescriptor at: 1)
           args: (aMessageDescriptor at: 2) "arguments to selector" ;
      defaultAction  "return error direct to GCI" .
  1 timesRepeat:[ self _class ]. "loop to detect/handle termination interrupt"
  ^ res

]

{ #category : 'Forwarding' }
ClientForwarder >> doesNotUnderstand: aSymbol args: anArray envId: envId [

"Sends the message described by aSymbol and anArray to the client object
 represented by the value of the receivers' instance variable clientObject.
 Returns the result from executing the forwarded message in the client."

"Implemented by raising the error #clientForwarderSend to the client.
 This error will be returned to the client as the result of the GemBuilder for C
 call which caused the GemStone Smalltalk execution containing the forwarded
 message send.

 Upon completion of the method invoked by the forwarded message, the
 client should call GciContinueWith(anOop), where anOop is the GemStone
 replicate of the result of the client message send."
| res |
envId == 0 ifFalse:[
  ImproperOperation new details:
       'ClientForwarder not implemented for environmentId > 0' ;
    signalNotTrappable . "avoid infinite DNU"
].
res := ClientForwarderSend new
      receiver: self
           clientObj: clientObject
           selector: aSymbol
           args: anArray ;
    defaultAction  "return error direct to GCI" .
1 timesRepeat:[ self _class ]. "loop to detect/handle termination interrupt"
^ res

]

{ #category : 'Testing' }
ClientForwarder >> isBehavior [

"Although a ClientForwarder may indeed represent a client behavior
 object, the GemStone system should not consider it as a GemStone
 behavior."
^false

]
