Extension { #name : 'FixedPoint' }

{ #category : 'Private' }
FixedPoint class >> _fromString: aString decimalPoint: dp [

"Private.  Given aString such as '34.23', returns a non-reduced FixedPoint.
 Using the specified decimal point character .
 If dp == nil , the session's locale state is used.
 Returns nil if there is a format error in aString"

<primitive: 465>
aString _validateClass: String.
dp ifNotNil:[ dp _validateClass: Character ].
self _errIncorrectFormat: aString.
self _primitiveFailed: #_fromString:decimalPoint: args: { aString . dp } .

]

{ #category : 'Instance Creation' }
FixedPoint class >> fromString: aString [

"Given aString such as '34.23', returns an instance of FixedPoint with
 appropriate numerator and denominator, and with scale equal to the number
 of digits to the right of the decimal point.
 If a String includes the literal indicator $p, then an sequence of digits
 immediately following $p are used for the scale. Characters other than this
 after the $p are ignored.
 The session's locale state determines the expected decimal point character.
"

^ ( self _fromString: aString decimalPoint: nil ) _reduce immediateInvariant

]

{ #category : 'Instance Creation' }
FixedPoint class >> fromStringLocaleC: aString [

"Given aString such as '34.23', returns an instance of FixedPoint with
 appropriate numerator and denominator, and with scale equal to the number
 of digits to the right of the decimal point.
 If a String includes the literal indicator $p, then an sequence of digits
 immediately following $p are used for the scale. Characters other than this
 after the $p are ignored.
 The expected decimal point character is $.  "

^ ( self _fromString: aString decimalPoint: $. ) _reduce immediateInvariant

]

{ #category : 'Storing and Loading' }
FixedPoint class >> loadFrom: passiveObj [

"Reads from passiveObj the passive form of an object.  Converts the object to
 its active form by loading the information into a new instance of the receiver.
 Returns the new instance."

| inst num den scale |
inst := self basicNew .
passiveObj hasRead: inst .
passiveObj readNamedIV.
num := passiveObj ivValue.
passiveObj readNamedIV.
den := passiveObj ivValue.
passiveObj readNamedIV.
scale := passiveObj ivValue.

passiveObj skipNamedInstVars.

(den = 0) ifTrue: [ ^ num _errorDivideByZero ].
inst _numerator: num denominator: den scale: scale .
^inst.

]

{ #category : 'Instance Creation' }
FixedPoint class >> numerator: numerator denominator: denominator scale: scale [

"Returns an instance of FixedPoint with the given numerator and denominator.

 The arguments numerator and denominator must be Integers.
 The argument  scale  must be a SmallInteger . "

(denominator = 0) ifTrue: [ ^ numerator _errorDivideByZero ].
^ self basicNew _numerator: numerator denominator: denominator scale: scale

]

{ #category : 'Formatting' }
FixedPoint >> _asString: dpChar [

| x num numer denom aString wholePart fraction |

aString := String new.
scale ifNil:[ ^ '(uninitialized FixedPoint)' "fix bug 13190"].
x := 10 raisedToInteger: scale .
numer := numerator .
denom := denominator .
num := (( numer * x) + (denom quo: 2)) // denom .
(numer < 0) ifTrue:[
    aString add: $- .
    num := num negated .
].
wholePart := num // x .
fraction := num \\ x .

aString add: wholePart asString .
aString add: dpChar .
scale timesRepeat: [
  fraction := fraction * 10 .
  aString add: (fraction // x) asString .
  fraction := fraction \\ x .
].
^ aString .

]

{ #category : 'Converting' }
FixedPoint >> _coerce: aNumber [

"Reimplemented from Number."

^ aNumber asFixedPoint: scale

]

{ #category : 'Private' }
FixedPoint >> _generality [

"Returns an Integer representing the ordering of the receiver in
 the generality hierarchy."

^ 65

]

{ #category : 'Private' }
FixedPoint >> _getKind [
	| num denom |
	(num := self numerator) ifNil: [ ^ 5	"nan" ].
	(denom := self denominator) ifNil: [ ^ 5 ].
	denom == 0
		ifTrue: [ 
			num == 0
				ifTrue: [ ^ 5	"NaN" ]
				ifFalse: [ ^ 3	"infinity" ] ].
	num == 0
		ifTrue: [ ^ 4	"zero" ].
	^ 1	"normal"
]

{ #category : 'Private' }
FixedPoint >> _numerator: num denominator: den scale: sc [

"Private.  Assigns the receiver's instance variables, reduces it, and makes it
 invariant."

num _isInteger ifTrue:[   "gemstone64, explicit constraint enforcement"
  den _isInteger ifTrue:[
    sc _isSmallInteger ifTrue:[
      numerator := num.
      denominator := den.
      sc < 0 ifTrue:[ scale := 0 ]
      	    ifFalse:[ scale := sc ].
      ^ self _reduce immediateInvariant
    ] ifFalse:[
      ArgumentTypeError new constrainedIv: 'FixedPoint.scale'
	     expectedClass: SmallInteger actualArg: sc ;
	 signal
    ].
  ] ifFalse: [
    ArgumentTypeError new constrainedIv: 'FixedPoint.denominator'
	expectedClass: Integer actualArg: den ;
      signal
  ].
] ifFalse:[
  ArgumentTypeError new constrainedIv: 'FixedPoint.numerator'
	expectedClass: Integer actualArg: num ;
      signal
].
self _uncontinuableError .

]

{ #category : 'Private' }
FixedPoint >> _reduce [

"Private.  Reduces the receiver."

| gcd |
"now reduce it"
numerator = 0 ifTrue:[
  denominator := 1.
  ^ self
  ].
denominator < 0 ifTrue:[  "denominator is always positive "
  numerator := numerator negated .
  denominator := denominator negated
  ].
gcd := numerator gcd: denominator.
numerator := numerator // gcd.
denominator := denominator // gcd.
^ self

]

{ #category : 'Private' }
FixedPoint >> _scale: aScale [

"Private."

scale := aScale

]

{ #category : 'Arithmetic' }
FixedPoint >> - aFixedPoint [

"Returns the difference between the receiver and aFixedPoint."

(aFixedPoint class == FixedPoint) ifTrue:[
  ^ self + aFixedPoint negated
].
^ self _retry: #- coercing: aFixedPoint

]

{ #category : 'Arithmetic' }
FixedPoint >> * aFixedPoint [

"Returns the result of multiplying the receiver by aFixedPoint."

aFixedPoint class == FixedPoint ifTrue:[
  ^ FixedPoint numerator: numerator * aFixedPoint numerator
           denominator: denominator * aFixedPoint denominator
           scale: scale
].
^ self _retry: #* coercing: aFixedPoint

]

{ #category : 'Arithmetic' }
FixedPoint >> / aFixedPoint [

"Returns the result of dividing the receiver by aFixedPoint ."

aFixedPoint class == FixedPoint ifTrue:[
  ^ self * aFixedPoint  reciprocal
].
^ self _retry: #/ coercing: aFixedPoint

]

{ #category : 'Arithmetic' }
FixedPoint >> + aFixedPoint [

"Returns the sum of the receiver and aFixedPoint."

aFixedPoint class == FixedPoint ifTrue:[
   | denom argDenom commonDenominator newNumerator |
   (denom := denominator) = (argDenom := aFixedPoint denominator) ifTrue:[
     ^ FixedPoint numerator: numerator + aFixedPoint numerator
                  denominator: denom
                  scale: scale
   ].
   commonDenominator := denom lcm: argDenom .
   newNumerator := numerator
                     * (commonDenominator quo: denom)
                     + (aFixedPoint numerator * (commonDenominator quo: argDenom )).
   ^ FixedPoint numerator: newNumerator
         denominator: commonDenominator scale: scale
].
^ self _retry: #+ coercing: aFixedPoint

]

{ #category : 'Comparing' }
FixedPoint >> < aNumber [
	"Returns true if the receiver is less than aNumber; returns false
 otherwise."

	| sk ak |
	(aNumber _isNumber or: [ aNumber isNumber ])
		ifFalse: [ ^ ArgumentTypeError signal: 'Expected a Number' ].
	sk := self _getKind.
	sk > 4
		ifTrue: [ ^ false ].
	sk == 3
		ifTrue: [ ^ self _compareInfinityFor: #'<' with: aNumber ].
	ak := aNumber _getKind.
	ak > 4
		ifTrue: [ ^ false ].
	ak == 3
		ifTrue: [ ^ aNumber sign == 1 ].	"If we get here, self and arg are both finite."
	aNumber class == FixedPoint
		ifTrue: [ 
			aNumber numerator = 0
				ifTrue: [ ^ numerator < 0 ]
				ifFalse: [ ^ self - aNumber < 0 ] ].
	^ (AbstractFraction _coerce: self) < aNumber
]

{ #category : 'Comparing' }
FixedPoint >> <= aNumber [
	"Returns true if the receiver is less than or equal to aNumber;
	returns false otherwise.
	Cannot use '> not' or NaN comparisons would answer true."

	^ self < aNumber or: [ self = aNumber ]
]

{ #category : 'Comparing' }
FixedPoint >> = aNumber [
	| sk ak |
	(aNumber _isNumber or: [ aNumber isNumber ])
		ifFalse: [ ^ false ].
	sk := self _getKind.
	sk > 4
		ifTrue: [ ^ false ].
	sk == 3
		ifTrue: [ ^ self _compareInfinityFor: #'=' with: aNumber ].
	ak := aNumber _getKind.
	ak > 4
		ifTrue: [ ^ false ].
	ak == 3
		ifTrue: [ ^ false ].	"If we get here, self and arg are both finite."
	aNumber class == FixedPoint
		ifTrue: [ 
			aNumber numerator = 0
				ifTrue: [ ^ numerator = 0 ]
				ifFalse: [ ^ aNumber numerator = numerator and: [ aNumber denominator = denominator ] ] ].
	^ (AbstractFraction _coerce: self) = aNumber
]

{ #category : 'Comparing' }
FixedPoint >> >= aNumber [

^ aNumber <= self

]

{ #category : 'Converting' }
FixedPoint >> asDecimalFloat [

"Returns an instance of DecimalFloat that has the value of the receiver."

^ numerator asDecimalFloat / denominator asDecimalFloat

]

{ #category : 'Converting' }
FixedPoint >> asFloat [

"Returns an instance of SmallDouble or Float that has the value of the receiver."

^ numerator asFloat / denominator asFloat

]

{ #category : 'Converting' }
FixedPoint >> asFraction [

"Returns a Fraction that represents the receiver."

^ Fraction numerator: numerator denominator: denominator.

]

{ #category : 'Formatting' }
FixedPoint >> asString [

 "Returns a String of the form '123.56 for a number with scale = 2 ,
  where the decimal point character in the result is per the current Locale."

  ^ self _asString: Locale decimalPoint . "fix 36666"

]

{ #category : 'Formatting' }
FixedPoint >> asStringLocaleC [

 "Returns a String of the form '123.56 for a number with scale = 2.
  Does not use Locale , decimal point character is always $.  "

  ^ self _asString: $.

]

{ #category : 'Accessing' }
FixedPoint >> denominator [

"Returns the denominator of the receiver."

^denominator

]

{ #category : 'Testing' }
FixedPoint >> even [

"Returns true if the receiver is an even integer, false otherwise."

 denominator = 1 ifFalse: [ ^ false ].
 ^ numerator even

]

{ #category : 'Comparing' }
FixedPoint >> hash [

^ self asFloat hash

]

{ #category : 'Accessing' }
FixedPoint >> instVarAt: anIndex put: aValue [

"Disallowed.  You may not change the value of a FixedPoint."

self shouldNotImplement: #instVarAt:put:

]

{ #category : 'Testing' }
FixedPoint >> isZero [

"Returns true if the receiver is zero."

^ numerator = 0 .

]

{ #category : 'Arithmetic' }
FixedPoint >> negated [

"Returns a Number that is the negation of the receiver."

^ FixedPoint numerator: numerator negated denominator: denominator
                scale: scale

]

{ #category : 'Accessing' }
FixedPoint >> numerator [

"Returns the numerator of the receiver."

^ numerator

]

{ #category : 'Testing' }
FixedPoint >> odd [

"Returns true if the receiver is an odd integer, false otherwise."

 denominator = 1 ifFalse: [ ^ false ].
 ^ numerator odd

]

{ #category : 'Copying' }
FixedPoint >> postCopy [
  ^ self immediateInvariant

]

{ #category : 'Arithmetic' }
FixedPoint >> reciprocal [

(numerator = 0) ifTrue: [ ^ self _errorDivideByZero].
^ FixedPoint numerator: denominator denominator: numerator scale: scale

]

{ #category : 'Updating' }
FixedPoint >> reduced [

"Returns a FixedPoint determined by finding the greatest common
 divisor of the numerator and denominator of the receiver."

"Reduce a fraction to its smallest terms."

| gcd numer denom |

(numerator = 0) ifTrue:[
  denominator == 1 ifFalse:[
    denominator := 1 .
  ].
  ^ self
].
gcd := numerator gcd: denominator.
numer := numerator // gcd.
denom := denominator // gcd.
(numer = numerator and:[ denom = denominator]) ifTrue:[ ^ self ].

^ FixedPoint numerator: numer denominator: denom scale: scale.

]

{ #category : 'Accessing' }
FixedPoint >> scale [

"Returns the scale of the receiver."

^scale

]

{ #category : 'Accessing' }
FixedPoint >> size: anInteger [

"Disallowed.  You may not change the size of a FixedPoint."

self shouldNotImplement: #size:

]

{ #category : 'Truncation and Rounding' }
FixedPoint >> truncated [

"Returns the integer that is closest to the receiver, on the same side
 of the receiver as zero is located."

^ numerator quo: denominator

]

{ #category : 'Formatting' }
FixedPoint >> withScale: newScale [

"Returns the receiver with the new scale."

scale == newScale ifTrue:[ ^ self ].
^ (self shallowCopy _scale: newScale) immediateInvariant

]

{ #category : 'Storing and Loading' }
FixedPoint >> writeTo: passiveObj [

"Converts the receiver to its passive form and writes that information on
 passiveObj."

"Reimplemented from Number since the receiver has a non-literal representation."

^super basicWriteTo: passiveObj

]
