"
GsBitmap is a class that implements an in memory bitmap, i.e. a transient
 bitmap that cannot be made persistent. A bitmap is a sparse data structure
 that logically implements a bit array. Each bit in the bitmap represents
 a committed non-special object.  

 Objects that are added to a GsBitmap are stored as a bit number which is a 
 numeric conversion of the oop of the object. This is done transparently;
 GsBitmap methods add and retrieve objects without manually converting to oops. 

 When working with the raw GsBitmap data, conversions between objects and oops 
 is via: 
   GsBitmap class >> bitToOop:
   GsBitmap class >> oopToBit:
   Object >> asOop
   Object class >> objectForOop:

 Access to the hiddenSets in the system is implemented by creating 
 a GsBitmap that references a particular hiddenSet (see newForHiddenSet:).  

 Most of the hiddenSets in the system are read only and generate an error
 if the operation is not allowed.  See the code for the hiddenSetSpecifiers 
 method and note the last mutable hiddenSet.  HiddenSets after this point in 
 the  array generate errors if an attempt is made to modify them and the user 
 is not SystemUser.  The NotifySet and PureExportSet
 have interfaces defined in System class (categories are NotifySet and
 GcISets) and these interfaces should be used to guarantee correct behavior.
 Updates to these 3 sets generate errors when attempted from GsBitmap.

 An instance of GsBitmap uses C Heap memory to store the bit array. 
 The C Heap memory use is minium of 16KBytes per instance of GsBitmap,
 For an instance with more than 2K elements, worst case memory usage is 
 approximately 8 bytes per element up to 1/64th of the objects in the repository. 
 An instance containing all of the objects in the
 repository will use approximately (System _oopHighWaterMark // 8) bytes of 
 C heap memory.
 The C Heap memory associated with an instance of GsBitmap is automatically 
 freed when the instance is garbage collected.
 Extensive use of large GsBitmaps in a session may require use
 of  ( System _vmMarkSweep )  to force garbage collection of the
 instances sooner than memory pressure on temporary object memory 
 would cause them to be garbage collected, to prevent running out of C heap
 memory.

 Do not retain an instance of GsBitmap from one session to another.
 Instances of GsBitmap are intended to exist only within a given GemStone
 session. 

 Instances may not be committed.

 The objects referenced in a GsBitmap are not protected from garbage collection.

"
Class {
	#name : 'GsBitmap',
	#superclass : 'Object',
	#instVars : [
		'hiddenSetId'
	],
	#gs_options : [
		'instancesNonPersistent'
	],
	#gs_reservedoop : '249345',
	#category : nil
}

{ #category : 'Enumerating' }
GsBitmap class >> allObjectsExcept: aCollection [

"Returns an instance of GsBitmap that contains all objects that exist which are not contained in the objects transitively referenced from <aCollection>. The objects in the result may or may not be live objects (reachable from the repository roots)."

| found |

found := GsBitmap withAll: aCollection.
found := GsBitmap allTransitiveReferences: found.
^ (GsBitmap allValidOops) - found

]

{ #category : 'Enumerating' }
GsBitmap class >> allTransitiveReferences: aCollection [

"Similar to transitiveReferences:, however the result includes all of the normally hidden interior nodes for large objects. "

| newFound allFound diff |

newFound := GsBitmap withAll: aCollection.
allFound := newFound copy.

[true] whileTrue: [ 
  newFound := newFound allReferencedObjects.
  diff := newFound - allFound.
  allFound addAll: diff.
  newFound := diff.
  diff isEmpty ifTrue: [ ^ allFound ]
  ] 

]

{ #category : 'Enumerating' }
GsBitmap class >> allValidOops [

"Returns an instance of GsBitmap containing the oops of all valid objects in the repository.  This bitmap only reflects currently committed objects that are not in the process of being garbage collected, i.e, not in the possible dead or deadNotReclaimed sets."

^(GsBitmap new) _1ArgPrim: 11

]

{ #category : 'Private' }
GsBitmap class >> _deadNotReclaimed [
 "Answer a new GsBitmap that contains the objects identified as dead
  by the last epochGc or markForCollection, and which have not been reclaimed.
  You must disable the reclaim gem before executing this. 
  If no error is signalled may include  System class >> continueTransaction. "

  System reclaimGemSessionCount > 0 ifTrue:[
    Error signal:'reclaimGem must be stopped before using #_deadNotReclaimed'.
  ].
  ^ (GsBitmap new) _1ArgPrim: 14 
]

{ #category : 'Instance Creation' }
GsBitmap class >> basicNew [

"disallowed"

self shouldNotImplement: #basicNew

]

{ #category : 'File Operations' }
GsBitmap class >> fileInfo: bitmapFilePath [

" Returns an array which contains the following values:
  1.  The number of oops in the file
  2.  True if the file was written in page order.
  3.  The number of valid oops.
  4.  The number of invalid oops (oops that are not allocated or in the
      process of being garbage collected).

  Raises an error if the file could not be opened for reading, the file
  is corrupt, or bitmapFilePath is not an instance of String.
"

^(GsBitmap new) _2ArgPrim: 20 with: bitmapFilePath

]

{ #category : 'Hidden Set Support' }
GsBitmap class >> hiddenSetIdAsSymbol: oldHiddenSetId [

"Returns the symbol corresponding to the old hiddenSet specifier" 

^ GsBitmap new _2ArgPrim: 19 with: oldHiddenSetId

]

{ #category : 'Hidden Set Support' }
GsBitmap class >> hiddenSetSpecifiers [

"Returns a list of the hidden set specifiers.

**************************************************************************************************
***** DO NOT CHANGE THE ORDER OF THE ELEMENTS IN THE LIST WITHOUT EDITING CODE IN gsbitmap.c *****
**************************************************************************************************
"

^#(
 #ListInstances
 #SaveNewPomObjs "See method #_enableTraceNewPomObjs"
 #ObjectsRead    "See method #_enableTraceObjectsRead"
 #SaveWriteSetUnion
 #CommitReleaseLocksSet
 #CommitOrAbortReleaseLocksSet
 #StrongReadSet
 #GcCandidates
 #Indexing
 #Conversion
 #WeakReferences
 #ObjInventory
 #Customer1
 #Customer2
 #Customer3
 #Customer4
 #Customer5 "last mutable hiddenSet"

 #RcReadSet
 #DepMapWriteSet
 #PomWriteSet    "Empty except after flush for commit, so only useful
		       after a transaction conflict."
 #RemovedDepMapEntries
 #SaveDepMapChangedUnion
 #ReadWriteConflicts  "StrongRead-Write conflicts "
 #WriteWriteConflicts
 #WriteDependencyConflicts
 #WriteReadLockConflicts
 #WriteWriteLockConflicts
 #AllocatedGciOops
 #ExportedDirtyObjs
 #TrackedDirtyObjs_NotImplemented
 #ReferencedSet
 #WriteLockWriteSubset
 #NewDataPages
 #PreviousWsUnion
 #PureExportSet
 #GciTrackedObjs_NotImplemented
 #NotifySet
)

]

{ #category : 'Instance Creation' }
GsBitmap class >> new [

"Returns a new empty GsBitmap"

^ self _basicNew _1ArgPrim: 1

]

{ #category : 'Instance Creation' }
GsBitmap class >> _newForHiddenSetId: hiddenSetId [
 "Returns a GsBitmap that references the specified hiddenSet."
  ^ GsBitmap new _hiddenSetId: hiddenSetId; yourself
]

{ #category : 'Instance Creation' }
GsBitmap class >> newForHiddenSet: hiddenSetSpecifier [

"Returns a GsBitmap that references the specified hiddenSet."

| gsBitmap idx |
gsBitmap := GsBitmap new.
idx := GsBitmap hiddenSetSpecifiers indexOfIdentical: hiddenSetSpecifier.
idx < 1 ifTrue: [ ^ hiddenSetSpecifier _error: #rtErrInvalidArgument
                              args:{ 'not a valid hiddenSetSpecifier' }] .
gsBitmap _hiddenSetId: idx.
^gsBitmap

]

{ #category : 'Enumerating' }
GsBitmap class >> transitiveReferences: aCollection [

"Returns an instance of GsBitmap that contains all the objects which are transitively referenced from the objects in <aCollection>."

| newFound allFound diff |

newFound := GsBitmap withAll: aCollection.
allFound := newFound copy.

[true] whileTrue: [ 
  newFound := newFound referencedObjects.
  diff := newFound - allFound.
  allFound addAll: diff.
  newFound := diff.
  diff isEmpty ifTrue: [ ^ allFound ]
  ] 

]

{ #category : 'Instance Creation' }
GsBitmap class >> with: aValue [

"Returns an instance of the receiver containing the argument"

| inst |
inst := self new.
inst add: aValue.
^inst

]

{ #category : 'Instance Creation' }
GsBitmap class >> with: aValue with: val2 [

"Returns an instance of the receiver containing the arguments"

| inst |
inst := self new.
inst add: aValue; add: val2.
^inst

]

{ #category : 'Instance Creation' }
GsBitmap class >> with: aValue with: val2 with: val3 [

"Returns an instance of the receiver containing the arguments"
| inst |
inst := self new.
inst add: aValue; add: val2; add: val3.
^inst

]

{ #category : 'Instance Creation' }
GsBitmap class >> with: aValue with: val2 with: val3 with: val4 [

"Returns an instance of the receiver containing the arguments"
| inst |
inst := self new.
inst add: aValue; add: val2; add: val3; add: val4.
^inst

]

{ #category : 'Instance Creation' }
GsBitmap class >> withAll: aCollection [

"Returns an instance of the receiver containing the elements of the argument."

| inst |
inst := self new.
aCollection _isArray 
  ifTrue:[ inst addAll: aCollection ]
  ifFalse:[ aCollection do: [ :each | inst add: each ]].
^ inst
]

{ #category : 'Private' }
GsBitmap class >> _2ArgPrim: opCode with: arg2 [
  "opcode  action
   26      clearAllBits  arg2 is hiddenSetId 
  "
  <primitive: 1027>
  opCode _validateClass: SmallInteger.
  self _primitiveFailed: #_2ArgPrim:with: args: { opCode . arg2 }
]

{ #category : 'Private' }
GsBitmap class >> _clearHiddenSetId: hiddenSetId [
  self _2ArgPrim: 26 with: hiddenSetId
]

{ #category : 'Private' }
GsBitmap class >> _3ArgPrim: opCode with: arg2 with: arg3 [
 " Functions with the receiver readOnly should be first
  opCode     function
    3         arg2 is hiddenSetId arg3 is oop to add
  "
 <primitive: 1028>
 opCode _validateClass: SmallInteger.
 self _primitiveFailed: #_3ArgPrim:with:with: args: { opCode . arg2 . arg3}
]

{ #category : 'Private' }
GsBitmap class >> _hiddenSetId: hiddenSetId add: anObject [
  "Returns a Boolean, previous state of bit for anObject.
   Does nothing and returns false if anObject is special."
  ^ self _3ArgPrim: 3 with: hiddenSetId with: anObject 
]

{ #category : 'Private' }
GsBitmap >> _1ArgPrim: opCode [

"opCode     function
   1        _basicNew
   2        removeAll
   3        size
   4        asArray
   5        asArrayOfOops
   6        copy
   7        sizeInPages
   8        peekLast
   9        referencedObjects
   10       allReferencedObjects
   11       allValidOops
   12       primReferencedObjects
   13       _asArrayOfBits
   14       _deadNotReclaimed
"

<primitive: 1026>
opCode _validateClass: SmallInteger.
self _primitiveFailed: #_1ArgPrim: args: { opCode }

]

{ #category : 'Private' }
GsBitmap >> _2ArgPrim: opCode with: arg2 [

" Functions with the receiver readOnly should be first
 opCode     function
   1        includes:
   2        writeToFile:
   3        writeToFileInPageOrder:
   4        equals:
   5        firstObjectThatReferences:
   6        _includesBit:

     First mutable opcode
  10        add:         primitive returns the previous state
  11        addAll:
  12        remove:      primitive returns the previous state
  13        removeAll:
  14        union:
  15        difference:
  16        intersect:
  17        truncateToSize:
  18        _removeFirst:
  19        hiddenSetIdAsSymbol:

  20        fileInfo:
  21        readFromFile:
  22        (was auditFile)
  23        _setBits:
  24        _clearBits:
  25        primFirstObjectThatReferences:
"

<primitive: 1027>
opCode _validateClass: SmallInteger.
self _primitiveFailed: #_2ArgPrim:with: args: { opCode . arg2 }

]

{ #category : 'Private' }
GsBitmap >> _3ArgPrim: opCode with: arg2 with: arg3 [

" Functions with the receiver readOnly should be first
 opCode     function
   1        enumerateWithLimit:startingAfter:
   2        enumerateAsOopsWithLimit:startingAfter:
 "

<primitive: 1028>
opCode _validateClass: SmallInteger.
self _primitiveFailed: #_3ArgPrim:with:with: args: { opCode . arg2 . arg3}

]

{ #category : 'Private' }
GsBitmap >> _asArrayOfBits [

"Returns the bits that are set in the GsBitmap as an Array of Integers
 which correspoind to the raw bit numbers in the receiver.
 See also GsBitmap class >> bitToOop: "

^ self _1ArgPrim: 13

]

{ #category : 'Private' }
GsBitmap >> _asArrayOfOops [

"Returns the bits that are set in the GsBitmap as an Array of SmallIntegers
 which correspoind to the oops of the persistent objects.
 This method can be used to return the results of a large GsBitmap in 
 an array without faulting all of the objects into memory like asArray does.
 The oops can be used as arguments to   Object class >> objectForOop: ."

^self _1ArgPrim: 5

]

{ #category : 'File Operations' }
GsBitmap class >> auditFile: fileName [
"Reads the file and counts the number of objects in the file which 
 are no longer valid.  Objects are considered invalid if they no 
 longer exist (i.e., are not present in the shared object table) 
 or present in the dead objects set maintained by stone.

 The session must be in a transaction when this method is invoked.  If the
 session is not in a transaction, an #rtErrPrimOutsideTrans error is raised.

 Returns an Array containing 2 elements 
    1 - a SmallInteger, total number of oops in the file.
    2 - a SmallInteger, number of invalid oops in the file.

 Raises an error if the file could not be opened for reading, the file
 is corrupt, fileName is not an instance of String.
"

 | arr fileInfo |
 fileInfo := GsBitmap fileInfo: fileName.
 arr := Array new.
 arr add: (fileInfo at: 1).
 arr add: (fileInfo at: 4).
 ^ arr

]

{ #category : 'Private' }
GsBitmap class >> _basicNew [

"creates an instance registered with VM for finalization of cData"

<primitive: 674>
self _primitiveFailed: #_basicNew

]

{ #category : 'Debugging' }
GsBitmap class >> oopToBit: aSmallInteger [
  "For argument that is the oop of a committed non-special object, return
   the bit number."
  (aSmallInteger bitAnd: 16rFF) == 1 ifFalse:[ 
     ArgumentError signal:'argument is oop of a special object'].
  ^ aSmallInteger bitShift: - 8 .
]

{ #category : 'Debugging' }
GsBitmap class >> bitToOop: aSmallInteger [
  "For argument that is bit number for a committed non-special object, return
   the oop as a SmallInteger ."
  ^ (aSmallInteger bitShift:  8 ) bitOr: 1 
]

{ #category : 'Private' }
GsBitmap >> _clearBits: anArray [

"Clears the bits specified by the Integers in anArray from the receiver.
 Returns the number of bits cleared."

^self _2ArgPrim: 24 with: anArray

]

{ #category : 'Private' }
GsBitmap >> _doAsOops: aBlock [

"Executes the one argument block aBlock for each oop in
 in the receiver.  Does a non destructive enumerate of the GsBitmap
 with the argument to the block being anOop.
 Returns the number of objects enumerated."

 | count arr |
 count := 0.
 arr := Array new.
 arr add: 0.
 [ | n |
   arr := self enumerateAsOopsWithLimit: 2000 startingAfter: (arr last).
   n := arr size .
   count := count + n .
   1 to: n do:[:j | aBlock value: (arr at: j) ] .
   n ~~ 0 .
 ] whileTrue.
 ^ count

]

{ #category : 'Private' }
GsBitmap >> _hiddenSetId [

^hiddenSetId

]

{ #category : 'Private' }
GsBitmap >> _hiddenSetId: aSmallInt [

hiddenSetId := aSmallInt

]

{ #category : 'Private' }
GsBitmap >> _removeFirst: count [

"Remove the first count objects from the receiver.  Objects are removed from
 the beginning, going from lowest to highest object ID.

 Returns the number of objects removed from the hidden set."

^self _2ArgPrim: 18 with: count

]

{ #category : 'Private' }
GsBitmap >> _setBits: anArray [

"Add the bits specified by the Integers in anArray to the receiver.
 Returns the number of bits added."

^self _2ArgPrim: 23 with: anArray

]

{ #category: 'Private' }
GsBitmap >> _includesBit: aBit [

"Returns true if the bit is in the GsBitmap."

^self _2ArgPrim: 6 with: aBit

]

{ #category : 'Private' }
GsBitmap >> _truncateToSize: newSize [

"Truncate receiver by removing objects from the end of the set until it reaches a size
 of newSize.  Objects are removed in order, going from highest to lowest object ID.

 Returns the number of objects removed."

^self _2ArgPrim: 17 with: newSize

]

{ #category : 'Private' }
GsBitmap class >> _initializeSystemHiddenSetIds [
  | ary cvName |
  ary := self hiddenSetSpecifiers .
  1 to: ary size  do:[:j | | sym |
    sym := ary at: j .
    cvName := (sym , '_id' ) asSymbol .
    System _addInvariantClassVar: cvName value: j .
    GsFile gciLogServer:'_initializeSystemHiddenSetIds: created ', cvName, ' value ', j asString .
  ].
] 

{ #category : 'Set Arithmetic' }
GsBitmap >> - aGsBitmap [

"Returns a new GsBitmap with the elements in the receiver that are not in aGsBitmap.
 Same as difference:."

^self _2ArgPrim: 15 with: aGsBitmap

]

{ #category : 'Set Arithmetic' }
GsBitmap >> * aGsBitmap [

"Returns a new GsBitmap with the elements that are in both the receiver and in aGsBitmap.
 Same as inersect:."

^self _2ArgPrim: 16 with: aGsBitmap

]

{ #category : 'Set Arithmetic' }
GsBitmap >> + aGsBitmap [

"Returns a new GsBitmap with the elements that are in the receiver or aGsBitmap.
 Same as union:."

^self _2ArgPrim: 14 with: aGsBitmap

]

{ #category : 'Adding' }
GsBitmap >> add: anObject [

"Adds a non special object to the GsBitmap.
 Returns anObject."

self _2ArgPrim: 10 with: anObject.
^anObject

]

{ #category : 'Adding' }
GsBitmap >> addAll: anArrayOrGsBitmap [

"Adds each element of the Array of non special objects or the contents of a GsBitmap
 to the receiver. Returns an Integer, the number of bits added to the receiver."

(anArrayOrGsBitmap == nil) ifTrue: [ ^anArrayOrGsBitmap ].
^self _2ArgPrim: 11 with: anArrayOrGsBitmap

]

{ #category : 'Enumerating' }
GsBitmap >> allReferencedObjects [

"Similar to referencedObjects, returning a new instance of GsBitmap containing the objects directly referenced by the objects in the receiver; but in addition, the result includes all of the normally hidden interior nodes for large objects."

^self _1ArgPrim: 10

]

{ #category : 'Converting' }
GsBitmap >> asArray [

"Returns the bits that are set in the GsBitmap as an Array of objects.
 The results is an object in temporary object memory."

^self _1ArgPrim: 4

]

{ #category : 'Converting' }
GsBitmap >> asGsBitmap [

^ self

]

{ #category : 'Accessing' }
GsBitmap >> copy [

"Returns a new GsBitmap that is a copy of the receiver."

^self _1ArgPrim: 6

]

{ #category : 'Set Arithmetic' }
GsBitmap >> difference: aGsBitmap [

"Returns a new GsBitmap with the elements in the receiver that are not in aGsBitmap.
 Same as -."

^self _2ArgPrim: 15 with: aGsBitmap

]

{ #category : 'Enumerating' }
GsBitmap >> do: aBlock [

"Executes the one argument block aBlock for each object in
 in the receiver.  Does a non destructive enumerate of the GsBitmap.
 Returns the number of objects enumerated."

 | count arr |
 count := 0.
 arr := Array new.
 arr add: 0.
 [ | n |
   arr := self enumerateWithLimit: 2000 startingAfter: (arr last).
   n := arr size .
   count := count + n .
   1 to: n do:[:j | aBlock value: (arr at: j) ] .
   n ~~ 0 .
 ] whileTrue.
 ^ count

]

{ #category : 'Enumerating' }
GsBitmap >> enumerateAsOopsWithLimit: maxResultSize startingAfter: anOop [

"Returns an array containing the first maxResultSize bits as oops
 starting with the first one after the specified oop.
 To start the enumeration use the SmallInteger zero as anOop.
 To get the next batch use the last element of the array returned.

 This enumeration is non destructive, so no values are cleared.

 If the GsBitmap contains fewer than maxResultSize elements after the
 specified object or the maxResultSize has the value 0, then it returns
 all of the elements after the specified object.
"

^self _3ArgPrim: 2 with: maxResultSize with: anOop

]

{ #category : 'Enumerating' }
GsBitmap >> enumerateWithLimit: maxResultSize startingAfter: anObject [

"Returns an array containing the first maxResultSize bits as objects
 starting with the first one after the specified object.
 To start the enumeration use the SmallInteger zero as anObject.
 To get the next batch use the last element of the array returned.

 This enumeration is non destructive, so no values are cleared.

 If the GsBitmap contains fewer than maxResultSize elements after the
 specified object or the maxResultSize has the value 0, then it returns
 all of the elements after the specified object.
"

^self _3ArgPrim: 1 with: maxResultSize with: anObject

]

{ #category : 'Testing' }
GsBitmap >> equals: aGsBitmap [

"Returns true if the receiver and argument contain exactly the same elements."

^self _2ArgPrim: 4 with: aGsBitmap

]

{ #category : 'Searching' }
GsBitmap >> firstObjectThatReferences: anObject [
"Search the receiver to find and return the first object which references anObject.
 Returns nil if no such object was found."

^self _2ArgPrim: 5 with: anObject

]

{ #category : 'Accessing' }
GsBitmap >> hiddenSetName [

"Returns the name as a symbol that corresponds to this instance.
 If there is no hidden set reference then it returns the symbol #NotAHiddenSet."

 hiddenSetId == 0 ifTrue: [ ^ #NotAHiddenSet ]
                  ifFalse: [ ^ (GsBitmap hiddenSetSpecifiers) at: hiddenSetId ]

]

{ #category : 'Testing' }
GsBitmap >> includes: anObject [

"Returns true if anObject is in the GsBitmap."

^self _2ArgPrim: 1 with: anObject

]

{ #category : 'Set Arithmetic' }
GsBitmap >> intersect: aGsBitmap [

"Returns a new GsBitmap with the elements that are in both the receiver and in aGsBitmap.
 Same as *."

^self _2ArgPrim: 16 with: aGsBitmap

]

{ #category : 'Set Arithmetic' }
GsBitmap >> intersectSize: aGsBitmap [

"Returns a SmallInteger which is the number of elements present in both the receiver and aGsBitmap."

| result newBm |
newBm := self * aGsBitmap .
result := newBm size.
newBm removeAll .
^ result

]

{ #category : 'Testing' }
GsBitmap >> isEmpty [

"Returns true there are no entries in the GsBitmap."

^self size == 0

]

{ #category : 'Peeking' }
GsBitmap >> peek [

"Answer the first element in the receiver without removing it, or nil if the receiver is empty."
^ self isEmpty ifTrue:[nil] ifFalse:[ (self peekCount: 1) at: 1]

]

{ #category : 'Peeking' }
GsBitmap >> peekCount: aCount [

"Fetches elements from the receiver without removing them.  Returns an Array that contains
 the minimum of aCount or the receiver's size entries."

^self enumerateWithLimit: aCount startingAfter: 0

]

{ #category : 'Peeking' }
GsBitmap >> peekLast [

"Answer the last in the receiver without removing it, or nil if the receiver is empty."
^ self isEmpty ifTrue:[nil] ifFalse:[ self _1ArgPrim: 8 ]

]

{ #category : 'Private' }
GsBitmap >> primFirstObjectThatReferences: anObject [

"Search the receiver to find and return the first object which references anObject. 
 Returns nil if no such object was found. Similar to the firstObjectThatReferences:
 method except this method may return an internal object as the result"

^self _2ArgPrim: 25 with: anObject

]

{ #category : 'Private' }
GsBitmap >> primReferencedObjects [

"Answer a new GsBitmap that contains the committed, non-special objects
referenced by the objects contained in the receiver. Similar to the 
referencedObjects method except this method includes internal objects
in the result, and does not include the entire collection contents
when traversing large objects or NSCs."

^self _1ArgPrim: 12

]

{ #category : 'Formatting' }
GsBitmap >> printOn: aStream [

"Puts a displayable representation of the receiver on aStream."

aStream nextPutAll: (self printStringWithMaxSize: 8).

]

{ #category : 'Formatting' }
GsBitmap >> printString [

"Returns a String with a displayable representation of the receiver."

^ self printStringWithMaxSize: 1000

]

{ #category : 'Formatting' }
GsBitmap >> printStringWithMaxSize: n [

"Returns the contents of the GsBitmap as a string with oops of the
 objects in the bitmap arranged 8 per line.
"
| count result bmSize |

count := 0.
result := String new.
bmSize := self size.
result add: ('GsBitmap contains ' , bmSize asString , ' objects. ' , Character lf).
(bmSize == 0) ifTrue: [ ^ result ].
result add: ('1. ').
self _doAsOops: [ :oop |
  count := count + 1.
  result add: (' ' , oop asString , ' ').
  ((count rem: 8) = 0 and: [count < bmSize]) ifTrue: [
     result add: (' ' , Character lf , (count + 1) asString , '. ').
     (result size > n and: [n ~= 0]) ifTrue: [
       result add: (' ........' , Character lf).
       result add: ( bmSize asString , '. ' , self peekLast asString, Character lf).
       ^ result
     ]
  ]
].
^result

]

{ #category : 'File Operations' }
GsBitmap >> readFromFile: fileName [

"Reads the contents of the file and adds the oops in the file to the
 receiver.  If the file doesn't exist or is not in the GsBitmap format,
 then this method returns an error.

 Subsets of the file can be read into a GsBitmap with
         readFromFile:withLimit:startingAt:.

 ObjectIds that are in the freelist or in the deadObjs or possibleDead sets
 are not included.

 Returns the number of objects that were not included.
"

^self _2ArgPrim: 21 with: fileName

]

{ #category : 'File Operations' }
GsBitmap >> readFromFile: aString withLimit: maxResultSize startingAt: startIndex [

"Adds the contents of the specified file indexed by the subrange:
 (startingAt, startingAt + maxResultSize).  The resulting GsBitmap
 may contain fewer than maxResultSize elements if the value of
 startingAt + maxResultSize is an index that is larger than the number of
 oops in the file or if some of the ObjectIds represented are no longer
 live objects, i.e., they were found to be in the freeList or the
 deadNotReclaimed or possible dead sets.

 If the file doesn't exist or is not in the GsBitmap format,
 then this method returns an error.

 The first object in the file has an index of 1.  It is an error if
 startIndex is less than 1.

 Returns the number of objects read.
"

<primitive: 1029>
aString _validateClass: String.
maxResultSize _validateClass: SmallInteger.
startIndex _validateClass: SmallInteger.
self _primitiveFailed: #readFromFile:withLimit:startingAt: args: { aString . maxResultSize . startIndex }

]

{ #category : 'Enumerating' }
GsBitmap >> referencedObjects [

"Answer a new GsBitmap that contains the committed, non-special objects
referenced by the objects contained in the receiver.  For each object for
which the current user has read authorization, the objects that are directly
referenced structurally by that object (from named instance variables, indexed
or NSC contents, and dynamic instance variables) are included."

^self _1ArgPrim: 9

]

{ #category : 'Removing' }
GsBitmap >> remove: anObject [

"Removes anObject from the GsBitmap.
 If the object is not present in the GsBitmap it issues #objErrNotInColl,
 otherwise it returns anObject.
"
(self _2ArgPrim: 12 with: anObject)
    ifFalse: [ self _error: #objErrNotInColl args: { anObject } ].
^anObject

]

{ #category : 'Removing' }
GsBitmap >> _remove: anObject [
 "Removes anObject from the GsBitmap.
  Returns true if anObject was present in the GsBitmap"

  ^ self _2ArgPrim: 12 with: anObject
]

{ #category : 'Removing' }
GsBitmap >> remove: anObject ifAbsent: aBlock [

"Removes anObject from the receiver.  If anObject is not in the
 receiver, this method evaluates aBlock and returns its value.
 The argument aBlock must be a zero-argument block.
"

(self _2ArgPrim: 12 with: anObject)
    ifFalse: [ ^ aBlock value ].
^anObject

]

{ #category : 'Removing' }
GsBitmap >> removeAll [

"Clears all of the bits in the receiver."

^self _1ArgPrim: 2

]

{ #category : 'Removing' }
GsBitmap >> removeAll: anArrayOrGsBitmap [

"Removes each element of the Array of non special objects or the contents of a GsBitmap
 from the receiver.  Returns anArrayOrGsBitmap."

^self _2ArgPrim: 13 with: anArrayOrGsBitmap

]

{ #category : 'Removing' }
GsBitmap >> removeCount: maxToRemove [
"Removes entries from the receiver, and returns an Array that contains the
 minimum of maxToRemove or the receiver's size entries."

|result|
result := self peekCount: maxToRemove .
self removeAll: result .
^result

]

{ #category : 'Removing' }
GsBitmap >> removeFirst: count [

"Remove the first count objects from the receiver and return them in a new GsBitmap.
 Objects are removed from the beginning, going from lowest to highest object ID.
"

| copy |
copy := self copy.
self _removeFirst: count.
^copy removeAll: self

]

{ #category : 'Removing' }
GsBitmap >> removeIfPresent: anObject [

"Remove anObject from the receiver.  Returns nil if anObject is
 missing from the receiver, otherwise returns the argument.
"

(self _2ArgPrim: 12 with: anObject)
    ifFalse: [ ^ nil ].
^anObject

]

{ #category : 'Removing' }
GsBitmap >> removeLast: count [

"Remove the last count objects from the receiver and return them in a new GsBitmap.
 Objects are removed from the end, going from higest to the lowest object ID.
"

| copy |
copy := self copy.
self _truncateToSize: (self size - count).
^copy removeAll: self

]

{ #category : 'Removing' }
GsBitmap >> removeNext [
"Removes the first element from the receiver and returns it.  Returns nil if the receiver is empty."

^ self isEmpty ifTrue:[nil] ifFalse:[ (self removeCount: 1) at: 1]

]

{ #category : 'Testing' }
GsBitmap >> size [

"Returns the number of bits that are set in the GsBitmap."

^self _1ArgPrim: 3

]

{ #category : 'Accessing' }
GsBitmap >> sizeInPages [

"Returns the number of 16 K pages used to implement the receiver."

^self _1ArgPrim: 7

]

{ #category : 'Set Arithmetic' }
GsBitmap >> union: aGsBitmap [

"Returns a new GsBitmap with the elements that are in the receiver or aGsBitmap.
 Same as +."

^self _2ArgPrim: 14 with: aGsBitmap

]

{ #category : 'File Operations' }
GsBitmap >> writeToFile: aString [

"Writes the contents of the receiver to the specified file.
 The file created is approximately 5 bytes for each object plus 24 bytes for
 the file header.

 If the file exists an error is returned.

 Returns the number of objectIds written to the file.
"

^self _2ArgPrim: 2 with: aString

]

{ #category : 'File Operations' }
GsBitmap >> writeToFileInPageOrder: aString [

"Writes the contents of the receiver to the specified file in pageOrder.

 Operations like instance migration which process a large volume of objects
 often run significantly faster if the objects are processed in batches that
 are in page order rather than in object ID order.

 The file created is approximately 5 bytes for each object plus 24 bytes for
 the file header.

 If the file exists an error is returned.

 Returns the number of objectIds written to the file.
"

^self _2ArgPrim: 3 with: aString

]

{ #category : 'File Operations' }
GsBitmap class >> numberOfObjectsInPageOrderFile: fileName [

"Answer the number of objects in a page order oop file.

 The session must be in a transaction when this method is invoked.  If the
 session is not in a transaction, an #rtErrPrimOutsideTrans error is raised.

 Raises an error if the file could not be opened for reading, the file
 is corrupt, fileName is not an instance of String, or the file is
 not written in page order.
"

| fileInfo |

fileInfo := GsBitmap fileInfo: fileName.
(fileInfo at: 2) ifFalse: [
   ArgumentTypeError signal: 'The bitmap file is not written in page order'
   ].
^ fileInfo at: 1

]

{ #category : 'File Operations' }
GsBitmap class >> auditPageOrderOopFile: fileName [

"Reads the page order oop file and counts the number of
 objects in the file which are no longer valid.  Objects are
 considered invalid if they no longer exist (i.e., are not
 present in the shared object table) or present in the dead objects
 set maintained by stone.

 The session must be in a transaction when this method is invoked.  If the
 session is not in a transaction, an #rtErrPrimOutsideTrans error is raised.

 Returns an Array containing 2 elements 
    1 - a SmallInteger, total number of oops in the file.
    2 - a SmallInteger, number of invalid oops in the file.

 Raises an error if the file could not be opened for reading, the file
 is corrupt, fileName is not an instance of String, or the file is
 not written in page order.
"

 | arr fileInfo |
 fileInfo := GsBitmap fileInfo: fileName.
 (fileInfo at: 2) ifFalse: [
   ArgumentTypeError signal: 'The bitmap file is not written in page order'
   ].
 arr := Array new.
 arr add: (fileInfo at: 1).
 arr add: (fileInfo at: 4).
 ^ arr
 
]

{ #category : 'File Operations' }
GsBitmap class >> readObjectsFromPageOrderFile: fileName startingAt: startIdx upTo: endIdx [

"Reads, validates and returns objects from a page-ordered file.

 startIndex is the index of the first object in the file to be returned.  The first
 object in the file has an index of 1.  It is an error if startIndex is less than 1
 or greater than endIndex.

 endIndex is the index of the last object to return from the file.  endIndex
 must be greater than or equal to startIndex.  endIndex may exceed the index of
 the last object in the file.  In that case, all valid objects from startIndex to the
 end of the file are returned.

 Returns an array containing the valid oops in page order.
 
 Raises an error if the file could not be opened for reading, the file
 is corrupt, fileName is not an instance of String, or the file is
 not written in page order.

 It is possible that one or more object identifiers contained in the file
 are no longer valid due to garbage collection.  Objects which are no longer
 valid have nil stored in their place in the returned array.  Objects that have been
 garbage collected and are in the free oop list or the dead not reclaimed set
 are considered to be invalid.

 The session must be in a transaction when this method is invoked.  If the
 session is not in a transaction, an #rtErrPrimOutsideTrans error is raised.
"

^(GsBitmap new) _readObjectsFromPageOrderFilePrim: fileName startingAt: startIdx upto: endIdx

]

{ #category : 'Private' }
GsBitmap >> _readObjectsFromPageOrderFilePrim: fileName startingAt: startIdx upto: endIdx [

<primitive: 170>
fileName _validateClass: String.
startIdx _validateClass: SmallInteger.
endIdx _validateClass: SmallInteger.
self _primitiveFailed: #_readObjectsFromPageOrderFile:startingAt:upTo: args: { fileName . startIdx . endIdx }

]

