Extension { #name : 'GsPackagePolicy' }

{ #category : 'Private' }
GsPackagePolicy class >> _originVersion [

"For use in repository upgrade code, before upgradeimage completes, returns a 2 digit
 SmallInteger."
(Globals at:#DbfHistory otherwise: nil) ifNotNil:[:hist |
  self _supportedVersionList do:[:ver | | num vStr |
    num := ScaledDecimal numerator: ver denominator: 10 scale: 1 .
    vStr := 'v', num asStringLocaleC, '.' .
    (hist matchPattern: { $* . vStr .  $? . ' kernel classes filein' . $* }) ifTrue:[
      ^ ver 
    ]
  ].
].
^ 0
]

{ #category : 'Private' }
GsPackagePolicy class >> _previousVersion [
"For use in repository upgrade code, before upgradeimage completes, returns a 2 digit
 SmallInteger."
| prevVer hist |
prevVer := 0 .
(Globals at:#DbfHistory otherwise: nil) ifNotNil:[:h | | ofs |
  hist := h .
  ofs := hist _findLastString: 'upgrade to GemStone' startingAt: hist size
               ignoreCase: true .
  ofs == 0 ifTrue:[ 
     (ImageVersion at: #gsVersion otherwise: nil ) ifNotNil:[:iVer |
        ofs := 1 .
        hist := '  v' , iVer, '  ' .
     ].
  ].
  ofs ~~ 0 ifTrue:[ 
    self _supportedVersionList do:[:ver | | num vStr |
      num := ScaledDecimal numerator: ver denominator: 10 scale: 1 .
      vStr := 'v', num asStringLocaleC, '.' .
      (hist matchPattern: { $* . vStr . $* }) ifTrue:[ ^ ver ].
      vStr := ' ', num asStringLocaleC, '.' .
      (hist matchPattern: { $* . vStr . $* }) ifTrue:[ ^ ver ].
    ].
  ].
].
prevVer == 0 ifTrue:[ prevVer := self _originVersion ].
^ prevVer
]

{ #category : 'Private' }
GsPackagePolicy class >> _supportedVersionList [
	"answer list of version prefixes for which upgrades are supported - should include current release"

	^ #(37 36 35 34 33 32)
]

{ #category : 'Private' }
GsPackagePolicy class >> _upgradedFrom [
"Similar to GsPackagePolicy class >> _previousVersion. For use in repository AFTER upgradeimage 
 completes, returns a 2 digit SmallInteger."
| prevVer ofs1 ofs match |
prevVer := 0.
(Globals at: #'DbfHistory' otherwise: nil)
	ifNotNil: [ :hist | 
		match := 'upgrade to GemStone'.
		ofs1 := hist _findLastString: match startingAt: hist size ignoreCase: true.
		ofs1 ~~ 0
			ifTrue: [ 
				ofs := hist
					_findLastString: match
					startingAt: ofs1 - match size
					ignoreCase: true.
				ofs ~~ 0
					ifTrue: [ 
						| pat |
						pat := hist copyFrom: ofs to: ofs1.
						GsPackagePolicy _supportedVersionList
							do: [ :ver | 
								| num vStr |
								num := ScaledDecimal numerator: ver denominator: 10 scale: 1.
								vStr := 'v' , num asStringLocaleC , '.'.
								(pat matchPattern: {$*.  vStr.  $*})
									ifTrue: [ ^ ver ].
								vStr := ' ' , num asStringLocaleC , '.'.
								(pat matchPattern: {$*.  vStr.  $*})
									ifTrue: [ ^ ver ] ] ] ] ].
prevVer == 0 ifTrue:[ prevVer := self _originVersion ].
^ prevVer
]

{ #category : 'Accessing' }
GsPackagePolicy class >> authorInitials [
    "authorInitials stored in SessionTemps _not_ the IV. Don't want 
     authorInitials persisted"

  | ai key tmps |
  ai := (tmps := SessionTemps current) at: (key := self authorInitialsGlobalName) otherwise:nil.
  ai == nil 
    ifTrue:[
      ai := GsSession currentSession userProfile userId asString.
      tmps at: key put: ai.
    ].
  ^ai

]

{ #category : 'Accessing' }
GsPackagePolicy class >> authorInitials: aString [
    "authorInitials stored in SessionTemps _not_ the IV. Don't want 
     authorInitials persisted"

  SessionTemps current at: self authorInitialsGlobalName put: aString.

]

{ #category : 'Private' }
GsPackagePolicy class >> authorInitialsGlobalName [

  ^#GsPackagePolicy_AuthorInitials "In SessionTemps"

]

{ #category : 'Accessing' }
GsPackagePolicy class >> current [
  "note that with the fix for bug 41433, the logic for initializing the sessionMethodPolicy 
   has been moved to bom.gs and userpro.gs ... lazy initialization left to handle upgraded repos."

  | userGlobals sessionMethodPolicy |
  userGlobals := GsSession currentSession objectNamed: #UserGlobals.
  sessionMethodPolicy := userGlobals at: self globalName otherwise: nil.
  sessionMethodPolicy == nil
    ifTrue: [
      "self pause ."
      sessionMethodPolicy := self new.
      userGlobals at: self globalName put: sessionMethodPolicy
    ].
  ^sessionMethodPolicy

]

{ #category : 'Accessing' }
GsPackagePolicy class >> currentOrNil [

 "Returns nil or the previously installed and enabled GsPackagePolicy."
  | pp |
  pp := (( GsSession currentSession objectNamed: #UserGlobals ) ifNil: [^nil]) at: self globalName otherwise: nil.
  pp ifNotNil:[ pp enabled ifTrue:[ ^ pp ]].
  ^ nil 

]

{ #category : 'Initialize' }
GsPackagePolicy class >> deinitialize [

  (GsSession currentSession objectNamed: #UserGlobals) removeKey: self globalName 
       ifAbsent: [ nil ].

]

{ #category : 'Accessing' }
GsPackagePolicy class >> enabled [
  "As an extended fix to bug 41433, it is necessary to avoid lazy initialization
   of GsPackagePolicy class>>current, until after the initial login, thus the
   necessity of in-lining GsPackagePolicy class>>current"

  | userGlobals sessionMethodPolicy |
  userGlobals := GsSession currentSession objectNamed: #'UserGlobals'.
  userGlobals ifNil: [^false].
  sessionMethodPolicy := userGlobals at: self globalName otherwise: nil.
  sessionMethodPolicy == nil
    ifTrue: [ ^ false ].
  ^ sessionMethodPolicy enabled

]

{ #category : 'Accessing' }
GsPackagePolicy class >> globalName [

  ^#GsPackagePolicy_Current

]

{ #category : 'Initialize' }
GsPackagePolicy class >> initialize [
  ^ self deinitialize

]

{ #category : 'Method lookup control' }
GsPackagePolicy class >> loadSessionMethodDictionary [
  "Install a SessionMethodDictionary,should only be called at session login"

  | statusArray policy |
  (statusArray := Globals at: #ConversionStatus otherwise: nil ) 
    ifNotNil: [
      statusArray size > 3
        ifTrue: [ 
          (statusArray at: 4) ifTrue: [
            "in sessionMethod conversion don't install sessionMethods" 
            ^self ]]].
  policy := (GsSession currentSession objectNamed: #UserGlobals) at: self globalName otherwise: nil.
  "Avoid installing anything if the policy is not enabled or does not exist"
  (policy ~~ nil and: [ policy enabled ]) ifTrue: [ policy refreshSessionMethodDictionary ].

    "install a policy for handling TransactionBoundary notification"
    TransactionBoundaryDefaultPolicy isActive
        ifTrue: [ TransactionBoundaryDefaultPolicy installCurrent ].

    "login notification"
    SystemLoginNotification sessionStart.

]

{ #category : 'Instance Creation' }
GsPackagePolicy class >> new [

  ^self basicNew initialize

]

{ #category : 'Accessing' }
GsPackagePolicy class >> restrictedClasses [
  " restrictedClasses is an IdentitySet of class names"

  ^ restrictedClasses ifNil:[
      restrictedClasses := IdentitySet withAll: #( BasicSortNode
        BtreeBasicInteriorNode
        BtreeBasicLeafNode
        BtreeComparisonForCompare
        BtreeComparisonForSort
        BtreeComparisonQuerySpec
        BtreeInteriorNode
        BtreeLeafNode
        BtreeNode
        BtreeReadStream
        BucketValueBag
        DependencyList
        DepListBucket
        DepListTable
        GciInterface
        GsCurrentSession
        GsNMethod
        GsMethodDictionary
        GsSession
        GsSessionMethodDictionary
        GsPackagePolicy
        GsPackage
        IdentityIndex
        IndexList
        IndexManager
        MappingInfo
        NscBuilder
        RangeEqualityIndex
        RcBtreeBasicInteriorNode
        RcBtreeBasicLeafNode
        RcBtreeInteriorNode
        RcBtreeLeafNode
        RcCollisionBucket
        RcIndexBucket
        RcIndexBucketWithCache
        RcIndexDictionary
        RcRangeEqualityIndex
        Repository
        ObjectSecurityPolicy
        GsObjectSecurityPolicySet
        SymbolAssociation
        SymbolDictionary
        SymbolKeyValueDictionary
        SymbolList
        SymbolSet
        System
        UserProfile
        UserProfileSet
        UserSecurityData
      ).
      restrictedClasses
    ].

]

{ #category : 'Private' }
GsPackagePolicy >> _disableNoRefresh [
  "For use only by SystemUser during image upgrade."
  enabled := false.

]

{ #category : 'Private' }
GsPackagePolicy >> _packageReverse_Do: aBlock [
  | symList |
  symList := self symbolList . 
  symList size _downTo: 1 do:[ :j| | symDict package |
     symDict := symList at: j .
     package := symDict at: GsPackage globalName otherwise: nil.
     package ifNotNil:[  aBlock value: package ].
  ].

]

{ #category : 'Reporting' }
GsPackagePolicy >> _report: includeMethsBool [
 "Reports on the methods without regard to whether enabled is true"
 | str symList |
  str := String new .
  symList := self symbolList .
  1 to: symList size do:[ :j| | symDict package |
     symDict := symList at: j .
     package := symDict at: GsPackage globalName otherwise: nil.
     package ifNotNil:[  
       includeMethsBool ifTrue:[ str add:'===== ' ].
       str add: 'GsPackage oop:' ; add: package asOop asString ;
         add: ' name:' ; add: package name asString ;
         add: ' enabled:' ; add: package enabled asString ; lf .
       includeMethsBool ifTrue:[ str add: package methodsReport ].
     ].
  ].
 ^ str

]

{ #category : 'Private' }
GsPackagePolicy >> _sessionMethodsSet: aValue [
    "aValue is an IdentitySet of classes having session methods installed, or nil
     returns previous value"
  <protected>
  | tmps old key |
  aValue ifNotNil:[ aValue _validateClass: IdentitySet ].
  tmps := SessionTemps current .
  key := self sessionMethodDictionaryGlobalName .
  old := tmps at: key otherwise: nil .
  tmps at: key put: aValue .
  ^ old

]

{ #category : 'Categories' }
GsPackagePolicy >> addCategory: aSymbol for: aBehavior [

  ^ self homeSessionMethods addCategory: aSymbol for: aBehavior

]

{ #category : 'Categories' }
GsPackagePolicy >> addSelector: aSelector toCategory: categoryName for: aBehavior [
   self homeSessionMethods 
      addSelector: aSelector toCategory: categoryName for: aBehavior .

]

{ #category : 'Accessing' }
GsPackagePolicy >> authorInitials [
  ^ self class authorInitials

]

{ #category : 'Accessing' }
GsPackagePolicy >> authorInitials: aString [
  self class authorInitials: aString

]

{ #category : 'Private' }
GsPackagePolicy >> buildSessionMethodDictionary [
 "returns receiver"
 <primitive: 2001>  "enter protected mode"
 | prot |
 prot := System _protectedMode .
 [ | smSet clsDict oldSet oldList envId reenableAlmostOfMemoryThreshold |
    (AlmostOutOfMemory enabled and: [ TransactionBoundaryDefaultPolicy isActive ])
      ifTrue: [
        "If SessionMethodTransactionBoundaryPolicy, then we are running in GLASS 
          environment and that means that AlmostOutMemory handling needs to be
          suspended for the duration of this method. AlmostOfMemory handling 
          always involves a commit and a commit while session methods are in flux
          can lead to MNU or other errors in code that expects the session methods 
          to be installed correctly"
		(GsCurrentSession currentSession objectNamed: 'SessionMethodTransactionBoundaryPolicy')
          ifNotNil: [ reenableAlmostOfMemoryThreshold := AlmostOutOfMemory threshold ] ].
    envId := 0 .
    (smSet := IdentitySet _basicNew) _setNoStubbing .
    "transientMethodDictForEnv:put: will  keep classes in memory"
    clsDict := IdentityDictionary new .
    (Unicode16 _unicodeCompareEnabled) ifTrue:[
      | mapping |
      mapping := Unicode16 _unicodeCompareMapping .
      1 to: mapping size by: 2 do:[:index | 
        | cls |
        cls := mapping at: index .
        clsDict at: cls put: (Unicode16 _unicodeCompareTmdForClass: cls selectors: (mapping at: index + 1)) .
        smSet add: cls .
      ].
    ].
    self enabled ifTrue:[ | rejected |
      rejected := { } .
      self _packageReverse_Do: [:package |
        package behaviorAndMethodDictDo: [:behavior :methodDict | | tmd |
	  (tmd := clsDict at: behavior otherwise: nil ) ifNil:[
             tmd := GsSessionMethodDictionary new .
             clsDict at: behavior put: tmd .
             smSet add: behavior.
          ].
	  methodDict keysAndValuesDo: [:k :v | 
            v class == GsNMethod ifTrue:[ tmd at: k put: v ]
                    ifFalse:[ rejected add: { package . behavior . k . v } ].
          ].
        ].
      ].
      rejected size ~~ 0 ifTrue:[ 
        "rejected is of the form { { package . behavior . key . value } .  ... }"
        ImproperOperation new object: rejected ; reason: 'buildSessionMethodDictionaryFail';
          signal: 'one or more values in package method dictionaries is not a GsNMethod'.
      ].
      oldSet := self _sessionMethodsSet: smSet .  
      "following 2 loops must not use any methods implemented in session methods"
      oldSet 
        ifNotNil:[ 
          "remove or replace transient method dictionaries for each class in list"
          oldList := oldSet asArray .
          1 to: oldList size do:[:n |  | cls |
            cls := oldList at: n .
            (clsDict includesKey: cls)
              ifTrue: [ 
                "replace the method dict with the new one"
                cls transientMethodDictForEnv: envId put: (clsDict removeKey: cls)  ]
              ifFalse: [ 
                "remove the method dictionary"
                cls transientMethodDictForEnv: envId put: nil ] ] ].
       "add the remaining tranisent method dictionaries"
       clsDict keysAndValuesDo:[ :cls :dict |
         cls transientMethodDictForEnv: envId put: dict ]. 
      GsCurrentSession currentSession enableSessionMethods: true env: envId. "clears lookup caches"
    ] ifFalse:[
      oldSet := self _sessionMethodsSet: nil .
      oldSet ifNotNil:[ 
        oldList := oldSet asArray .
        "following loop must not use any methods implemented in session methods"
        1 to: oldList size do:[:n | | cls tmd |
          cls := oldList at: n .
          tmd := (smSet includes: cls) ifTrue:[ clsDict at: cls ]
                                      ifFalse:[ nil ].
          cls transientMethodDictForEnv: envId put: tmd .
        ].
      ].
      GsCurrentSession currentSession enableSessionMethods: false env: envId . "clears lookup caches"
    ].
    Unicode16 _cacheUsingUnicodeCompares .
    reenableAlmostOfMemoryThreshold
		ifNotNil: [:threshold | AlmostOutOfMemory enableAtThreshold: threshold ] .
 ] ensure:[
   prot _leaveProtectedMode
 ]
]

{ #category : 'Categories' }
GsPackagePolicy >> categoryNamesFor: aBehavior into: anArray [

  self packages_Do: [ :package |
    package categoryNamesFor: aBehavior into: anArray.
  ].

]

{ #category : 'Methods' }
GsPackagePolicy >> categoryOfSelector: aSymbol for: aBehavior [
  self packages_Do: [ :package |  | aKey |
    aKey := package categoryOfSelector: aSymbol for: aBehavior.
    aKey ifNotNil: [ ^aKey ].
  ].
  ^ nil

]

{ #category : 'Categories' }
GsPackagePolicy >> categorysDo: aBlock for: aBehavior [
  "evaluates aBlock for each method category of receiver
   in specified environment. Returns the receiver.

   aBlock should be a two argument block
   expecting the args  categoryNameSymbol ,   selectorsSet.
   aBlock may be invoked more than once for each category name.
   The iteration is done directly over the categories in each
   of the receiver's packages."

  self packages_Do:[ :package | package categorysDo: aBlock for: aBehavior ].

]

{ #category : 'Methods' }
GsPackagePolicy >> compiledMethodAt: aSymbol for: aBehavior [
  self packages_Do: [ :package | |meth|
    meth := package compiledMethodAt: aSymbol for: aBehavior.
    meth ifNotNil: [ ^meth ].
  ].
  ^ nil

]

{ #category : 'Methods' }
GsPackagePolicy >> copyCategoryDictFor: aBehavior into: aGsMethodDictionary [

  self packages_Do: [ :package |
    package copyCategoryDictFor: aBehavior into: aGsMethodDictionary.
  ].

]

{ #category : 'Methods' }
GsPackagePolicy >> copyMethodDictFor: aBehavior into: aGsMethodDictionary [

  self packageReverse_Do: [ :package |
    package copyMethodDictFor: aBehavior into: aGsMethodDictionary.
  ].

]

{ #category : 'Accessing' }
GsPackagePolicy >> disable [

  enabled := false.
  self refreshSessionMethodDictionary .


]

{ #category : 'Accessing' }
GsPackagePolicy >> enable [

  enabled := true.
  self refreshSessionMethodDictionary .

]

{ #category : 'Accessing' }
GsPackagePolicy >> enabled [

  ^ enabled

]

{ #category : 'Accessing' }
GsPackagePolicy >> externalSymbolList [

  ^externalSymbolList

]

{ #category : 'Accessing' }
GsPackagePolicy >> externalSymbolList: anArray [

  externalSymbolList := anArray

]

{ #category : 'Private' }
GsPackagePolicy >> extractSelectorFor: aBehavior 
source: sourceString 
dictionaries: aSymbolList 
category: categorySymbol [ 

  ^ self extractSelectorFor: aBehavior source: sourceString

]

{ #category : 'Private' }
GsPackagePolicy >> extractSelectorFor: aBehavior source: sourceString  [

  "Returns a selector or signals a CompileError"
  ^ aBehavior extractSelector: sourceString 

]

{ #category : 'Methods' }
GsPackagePolicy >> findMethods: aSymbol for: aBehavior [
  "Return array of package description , package, methods triples"
  | arr |
  arr := { } .
  self packages_Do: [ :package | | meth |
    meth := package compiledMethodAt: aSymbol for: aBehavior.
    meth ifNotNil: [ arr add: package printString; add: package; add: meth ].
  ].
  ^ arr

]

{ #category : 'Accessing' }
GsPackagePolicy >> homeSessionMethods [

  | package |
  package := self homeSymbolDict  at: GsPackage globalName otherwise: nil.
  package ifNil: [ package := GsPackage installIn: self homeSymbolDict ].
  ^package

]

{ #category : 'Accessing' }
GsPackagePolicy >> homeSymbolDict [

  ^homeSymbolDict

]

{ #category : 'Accessing' }
GsPackagePolicy >> homeSymbolDict: aSymDict [

  homeSymbolDict := aSymDict

]

{ #category : 'Methods' }
GsPackagePolicy >> includesSelector: aSymbol for: aBehavior [

  <primitive: 2001>  "enter protected mode"
  | ans prot |
  prot := System _protectedMode .
  ans := false .
  [
    | mDict |
    mDict := aBehavior transientMethodDictForEnv: 0 .
    mDict ifNotNil:[
      ans := mDict includesKey: aSymbol.
    ].
  ] ensure:[
    prot _leaveProtectedMode
  ].
  ^ans

]

{ #category : 'Initialize-Release' }
GsPackagePolicy >> initialize [

      | currentSession |
  enabled := false.
  currentSession := GsSession currentSession.
  homeSymbolDict := currentSession objectNamed: #UserGlobals.
  externalSymbolList := { } .

]

{ #category : 'Compiling' }
GsPackagePolicy >> methodAndCategoryDictionaryFor: aBehavior 
source: sourceString 
dictionaries: aSymbolList 
category: categorySymbol [ 
  "Returns a 2 element Array, or signals a CompileError"

  self enabled ifTrue: [ | selector |
      selector := self
        extractSelectorFor: aBehavior source: sourceString  .
      (self permitSessionMethodFor: aBehavior selector: selector)
        ifTrue: [ 
           ^self homeSessionMethods methodAndCategoryDictionaryFor: aBehavior 
        ].
    ].
  ^ { nil . nil }

]

{ #category : 'Reporting' }
GsPackagePolicy >> methodsReport [
 "Reports without regard to whether enabled is true"
 ^ self _report: true

]

{ #category : 'Categories' }
GsPackagePolicy >> moveSelector: aSelector toCategory: categoryName for: aBehavior [

   self packages_Do: [ :package |
      package removeSelector: aSelector fromCategoriesFor: aBehavior ].
   self homeSessionMethods 
      moveSelector: aSelector toCategory: categoryName for: aBehavior.

]

{ #category : 'Enumerating' }
GsPackagePolicy >> packageReverse_Do: aBlock [
  "aBlock must be a one argument block which returns true if iteration
   should be terminated early and false if iteration is to continue .
   This is to avoid return-from-home which in Gs64 v3.0 would
   reset the protected mode to zero."
   
  self enabled ifFalse: [ ^self ].
  self _packageReverse_Do: aBlock .

]

{ #category : 'Enumerating' }
GsPackagePolicy >> packages_Do: aBlock [
  "aBlock must be a one argument block which returns true if iteration
   should be terminated early and false if iteration is to continue .
   This is to avoid return-from-home which in Gs64 v3.0 would
   reset the protected mode to zero."
   
  | symList |
  self enabled ifFalse: [ ^self ].
  symList := self symbolList .
  1 to: symList size do:[ :j| | symDict package |
     symDict := symList at: j .
     package := symDict at: GsPackage globalName otherwise: nil.
     package ifNotNil:[  aBlock value: package  ].
  ].

]

{ #category : 'Reporting' }
GsPackagePolicy >> packagesReport [
 "Reports without regard to whether enabled is true"
 ^ self _report: false

]

{ #category : 'Private' }
GsPackagePolicy >> permitSessionMethodFor: aBehavior selector: selector [

  | cl thisName |
  cl := aBehavior whichClassIncludesSelector: selector.
  cl ifNotNil: [ (cl compiledMethodAt: selector) _isProtected ifTrue: [ ^false ].  ].
  thisName := aBehavior thisClass name asSymbol .
  (self class restrictedClasses includes: thisName ) ifTrue: [ ^false ].
  externalSymbolList do: [:symDict | 
		| possible |
		possible := symDict at: thisName otherwise: nil.
		(possible isBehavior and: [aBehavior theNonMetaClass isVersionOf: possible theNonMetaClass]) 
			ifTrue: [ ^true ].
  ].
  ^ (aBehavior canWriteMethodsEnv: 0) not

]

{ #category : 'Compiling' }
GsPackagePolicy >> pragmasForMethod: selector in: aBehavior [

  | pragmas |
  self packages_Do: [ :package | | pragmaDict |
    pragmaDict := package methodPragmaDictFor: aBehavior.
    pragmaDict ifNotNil: [ 
      (Symbol _existingWithAll: selector) ifNotNil:[ :sel |
        pragmas := pragmaDict at: sel otherwise: nil .
      ].
      pragmas ifNotNil: [ ^pragmas ].
    ].
  ].
  ^ nil

]

{ #category : 'Method lookup control' }
GsPackagePolicy >> refreshSessionMethodDictionary [

  self buildSessionMethodDictionary 

]

{ #category : 'Methods' }
GsPackagePolicy >> removeAllMethodsFor: aBehavior [
  "self pause ."
  self packages_Do: [ :package |
    package removeAllMethodsFor: aBehavior.
  ].
  "fix up the session method dictionary"
  self sessionMethodRemoveAllMethodsFor: aBehavior.
  ^nil

]

{ #category : 'Methods' }
GsPackagePolicy >> removeAllSubclassCodeFor: aBehavior [

"Dereference the code objects of all GsMethods for aBehavior,
 to force recompilation of those methods."
  "self pause."
  self packages_Do: [:package |
    package removeAllSubclassCodeFor: aBehavior.
  ].
  self sessionMethodRemoveAllSubclassCodeFor: aBehavior.

]

{ #category : 'Categories' }
GsPackagePolicy >> removeCategory: aSymbol for: aBehavior [
  self packages_Do: [ :package |
    package removeCategory: aSymbol for: aBehavior.
  ].

]

{ #category : 'Methods' }
GsPackagePolicy >> removeMethodAt: aSymbol for: aBehavior [
  | meth |
  self packages_Do: [ :package |
    meth := package removeMethodAt: aSymbol for: aBehavior.
    meth ifNotNil:[
      self sessionMethodRemoveMethodAt: aSymbol for: aBehavior.
      ^ meth 
    ].
  ].
  self sessionMethodRemoveMethodAt: aSymbol for: aBehavior.
  ^ nil

]

{ #category : 'Categories' }
GsPackagePolicy >> removeSelector: aSelector fromCategoriesFor: aBehavior [
    self packages_Do: [ :package |
      package removeSelector: aSelector fromCategoriesFor: aBehavior.
    ].

]

{ #category : 'Methods' }
GsPackagePolicy >> selectorsFor: aBehavior into: anArray [

  <primitive: 2001>  "enter protected mode"
  | prot |
  prot := System _protectedMode .
  [
    | mDict  |
    mDict := aBehavior transientMethodDictForEnv: 0 .
    mDict ifNotNil:[
      anArray addAll: mDict keys.
    ].
  ] ensure:[
    prot _leaveProtectedMode
  ].
  ^self

]

{ #category : 'Categories' }
GsPackagePolicy >> selectorsIn: categoryName for: aBehavior into: anArray [

  self packages_Do: [ :package |
    package selectorsIn: categoryName for: aBehavior into: anArray.
  ].

]

{ #category : 'Transaction Boundaries' }
GsPackagePolicy >> sessionMethodChanged [

    TransactionBoundaryDefaultPolicy isActive
        ifTrue: [ TransactionBoundaryDefaultPolicy current sessionMethodChanged].

]

{ #category : 'Private' }
GsPackagePolicy >> sessionMethodDictionaryGlobalName [

  ^#GsPackagePolicy_SessionMethodDictionary "In SessionTemps"

]

{ #category : 'Private' }
GsPackagePolicy >> sessionMethodRemoveAllMethodsFor: aBehavior [

  <primitive: 2001>  "enter protected mode"
  | prot |
  prot := System _protectedMode .
  [
    | oldDict smSet envId |
    envId := 0 .
    oldDict := aBehavior transientMethodDictForEnv: envId .
    oldDict ifNotNil:[
      aBehavior transientMethodDictForEnv: envId put: nil .
      aBehavior _clearLookupCaches: envId .
      self sessionMethodChanged .
      (smSet := self sessionMethodsSet) ifNotNil:[ smSet remove: aBehavior] .
    ].
  ] ensure:[
    prot _leaveProtectedMode
  ]

]

{ #category : 'Private' }
GsPackagePolicy >> sessionMethodRemoveAllSubclassCodeFor: aBehavior [

  <primitive: 2001>  "enter protected mode"
  | aDict prot |
  prot := System _protectedMode .
  [
    aDict := aBehavior transientMethodDictForEnv: 0 .
    aDict ifNotNil:[
      "All methods in aBehavior have been marked for recompilation, 
       and we have entries in the sessionMethodDictionary for Behavior, 
       so refresh sessionMethodDictionary."
  
      self refreshSessionMethodDictionary .
    ].
  ] ensure:[
    prot _leaveProtectedMode
  ].

]

{ #category : 'Private' }
GsPackagePolicy >> sessionMethodRemoveMethodAt: aSymbol for: aBehavior [

  <primitive: 2001>  "enter protected mode"
  | prot |
  prot := System _protectedMode .
  [
    | mDict oldMeth |
    mDict := aBehavior transientMethodDictForEnv: 0 .
    mDict ifNotNil: [
      oldMeth := mDict removeKey: aSymbol otherwise:  nil .
      oldMeth ifNotNil:[
	aBehavior _refreshLookupCache: aSymbol oldMethod: oldMeth env: 0 .
	self sessionMethodChanged
      ].
    ].
  ] ensure:[
    prot _leaveProtectedMode
  ]

]

{ #category : 'Private' }
GsPackagePolicy >> sessionMethodsSet [
  ^ SessionTemps current at: self sessionMethodDictionaryGlobalName otherwise: nil 

]

{ #category : 'Compiling' }
GsPackagePolicy >> setPragmas: pragmaArrayOrNil
forBehavior: aBehavior
forMethod: selector [

  | aSym |
  aSym := selector asSymbol.
  self packages_Do: [ :package | 
    (package
      setPragmas: pragmaArrayOrNil
          forBehavior: aBehavior
          forMethod: aSym ) ifNotNil:[ ^ self ].
  ].
  ^ nil


]

{ #category : 'Compiling' }
GsPackagePolicy >> setStamp: aStampOrNil
forBehavior: aBehavior
forMethod: selector [
  | aSym |
  aSym := selector asSymbol.
  self packages_Do: [ :package |
    (package
      setStamp: aStampOrNil
            forBehavior: aBehavior
            forMethod: aSym) ifNotNil: [ ^ self ].
  ].
  ^ nil

]

{ #category : 'Compiling' }
GsPackagePolicy >> stampForMethod: selector in: aBehavior [

  self packages_Do: [ :package | |stampDict |
    stampDict := package methodStampDictFor: aBehavior.
    stampDict ifNotNil: [  | stamp |
      (Symbol _existingWithAll: selector) ifNotNil:[ :sel |
        stamp := stampDict at: sel otherwise: nil 
      ].
      stamp ifNotNil: [ ^stamp ].
    ].
  ].
  ^ nil

]

{ #category : 'Accessing' }
GsPackagePolicy >> symbolList [

  ^ GsSession currentSession symbolList      "fix 49328"

]

{ #category : 'Compiling' }
GsPackagePolicy >> updateMethodLookupCacheFor: aGsMethod in: aBehavior [

  self updateMethodLookupCacheForSelector: aGsMethod selector 
				method: aGsMethod in: aBehavior

]

{ #category : 'Compiling' }
GsPackagePolicy >> updateMethodLookupCacheForSelector: selector method: aGsMethod in: aBehavior [

  <primitive: 2001>  "enter protected mode"
  | prot |
  prot := System _protectedMode .
  [
    | mDict oldMeth |
    mDict := aBehavior transientMethodDictForEnv: 0 .
    mDict ifNil: [
      mDict := GsSessionMethodDictionary new.
      self sessionMethodsSet add: aBehavior .
      aBehavior transientMethodDictForEnv:0 put: mDict .
    ] ifNotNil:[
       oldMeth := mDict at: selector otherwise: nil .
    ].
    oldMeth ifNil:[
       "need to find oldMeth if possible to be sure breakpoints are cleared"
       oldMeth := aBehavior compiledMethodAt: selector environmentId: 0 otherwise: nil
    ].
    mDict at: selector put: aGsMethod.
    aBehavior _refreshLookupCache: selector oldMethod: oldMeth env: 0.
    self sessionMethodChanged .
  ] ensure:[
    prot _leaveProtectedMode
  ].

]

{ #category : 'Methods' }
GsPackagePolicy >> recompileFor: aBehavior [
  self packages_Do:[:aPackage |
    aPackage recompileFor: aBehavior .
    false "continue iteration"
  ].
]

