Extension { #name : 'Random' }

{ #category : 'instance creation' }
Random class >> new [
	"Answer the host random singleton"

	^HostRandom new

]

{ #category : 'instance creation' }
Random class >> seed: aSmallInteger [
	"Answer a seeded random number generator seeded from the given SmallInteger"

	^Lag1MwcRandom seed: aSmallInteger

]

{ #category : 'public' }
Random >> float [
	"Answer a random Float in the range [0,1); or equivalently [0, 1[
	There are 2**53 possible results, equally spaced in the interval and
	generated with equal probability.

	To compute a float in this range, we need 53 random bits. Since bits come in chunks of 32,
	we grab 64 bits from the raw generator and discard the excess. More than 53 bits would result
	in floating point rounding which could improperly push the result to 1.0."

	^ (self integer bitAnd: 16rfffff800) * 2.3283064365386963E-10 + self integer * 2.3283064365386963E-10

	"2.3283064365386963E-10, expressed as a binary float, is exactly 2.0 raisedToInteger: -32, so
	multiplying by it is equivalent to a 32-bit right shift."

]

{ #category : 'public' }
Random >> floats: n [
	"Answer a collection of n random floats. N must be a positive integer."

	| result |
	result := Array new: n.
	1 to: n do: [:i | result at: i put: self float].
	^result

]

{ #category : 'public' }
Random >> integer [
	"Answer a random nonnegative 32-bit integer."

	^self subclassResponsibility: #integer

]

{ #category : 'public' }
Random >> integerBetween: l and: h [
	"Answer a random integer in the given interval. Both l and h must be numbers,
	h must be greater than or equal to l, and there must be at least one integer
  in the range from l to h.
	Examples:
		| random x y z |
		random := Random new.
		x := random integerBetween: 1 and: 20.
		y := random integerBetween: -5 and: 5.
		z := random integerBetween: 2.4 and: 257/3
  "
	|  low high modulus bitValues randomBits maxBits |
	low := l ceiling.
	high := h floor.
	modulus := high - low + 1.
	modulus > 0 ifFalse: [ArgumentError signal: 'range includes no integers'].
	bitValues := 16r100000000. "How many possible values of randomBits"
	modulus > bitValues ifTrue: [ArgumentError signal: 'Intervals > 2**32 not yet supported'].
	maxBits := bitValues - (bitValues \\ modulus).
	[randomBits := self integer. "[0, 16rFFFFFFFF]"
	randomBits > maxBits] whileTrue.
	^randomBits \\ modulus + low.

]

{ #category : 'public' }
Random >> integers: n [
	"Answer a collection of n random nonnegative 32-bit integers. N must be a positive integer."

	| result |
	result := Array new: n.
	1 to: n do: [:i | result at: i put: self integer].
	^result

]

{ #category : 'public' }
Random >> integers: n between: l and: h [
	"Answer a collection of n random integers between l and h.  N must be a positive integer."

	| result |
	result := Array new: n.
	1 to: n do: [:i | result at: i put: (self integerBetween: l and: h)].
	^result

]

{ #category : 'compatibility' }
Random >> next [
	"A synonym for #float, provided for compatibility with other Smalltalk implementations."

	^ self float.

]

{ #category : 'public' }
Random >> smallInteger [
	"Answer a random SmallInteger, [-2**60..2**60).
	This requires 61 bits including the sign bit.  We get 64 bits from the underlying generator,
  use 60 bits to compute a full-range positive SmallInteger, then conditionally negate
  it based on one of the extra bits."

	| highOrder lowOrder result |
	highOrder := self integer.
	lowOrder := self integer.
	result := ((highOrder bitAnd: 16rFFFFFFF) bitShift: 32) + lowOrder.
	^(highOrder bitAnd: 16r80000000) = 0
		ifTrue: [result]
		ifFalse: [result negated - 1]

]

{ #category : 'public' }
Random >> smallIntegers: n [
	"Answer a collection of n random SmallIntegers. N must be a positive integer."

	| result |
	result := Array new: n.
	1 to: n do: [:i | result at: i put: self smallInteger].
	^result

]
