Extension { #name : 'TestCase' }

{ #category : 'Accessing' }
TestCase class >> allTestSelectors [
	| answer pivotClass lookupRoot |
	answer := Set withAll: self testSelectors.
	self shouldInheritSelectors
		ifTrue:
			[pivotClass := self.
			lookupRoot := self lookupHierarchyRoot.
			[pivotClass == lookupRoot]
				whileFalse:
					[pivotClass := pivotClass superclass.
					answer addAll: pivotClass testSelectors]].
	^answer asSortedCollection asOrderedCollection

]

{ #category : 'Building Suites' }
TestCase class >> buildSuite [
	| suite |
	^self isAbstract
		ifTrue:
			[suite := self suiteClass named: self name asString.
			self allSubclasses
				do: [:each | each isAbstract ifFalse: [suite addTest: each buildSuiteFromSelectors]].
			suite]
		ifFalse: [self buildSuiteFromSelectors]

]

{ #category : 'Building Suites' }
TestCase class >> buildSuiteFromMethods: testMethods [

	^testMethods
		inject: (self suiteClass named: self name asString)
		into: [:suite :selector |
			suite
				addTest: (self selector: selector);
				yourself]

]

{ #category : 'Building Suites' }
TestCase class >> buildSuiteFromSelectors [
	^self buildSuiteFromMethods: self allTestSelectors

]

{ #category : 'Running' }
TestCase class >> debug [

	^self suite debug

]

{ #category : 'Instance Creation' }
TestCase class >> debug: aSymbol [

	^(self selector: aSymbol) debug

]

{ #category : 'Testing' }
TestCase class >> isAbstract [
	"Override to true if a TestCase subclass is Abstract and should not have
	TestCase instances built from it"

	^self sunitName = #TestCase

]

{ #category : 'Accessing' }
TestCase class >> lookupHierarchyRoot [
	^TestCase

]

{ #category : 'Accessing' }
TestCase class >> resources [

	^#()

]

{ #category : 'Running' }
TestCase class >> run [

	^self suite run

]

{ #category : 'Instance Creation' }
TestCase class >> run: aSymbol [

	^(self selector: aSymbol) run

]

{ #category : 'Instance Creation' }
TestCase class >> selector: aSymbol [

	^self new setTestSelector: aSymbol

]

{ #category : 'Testing' }
TestCase class >> shouldInheritSelectors [
	"I should inherit from an Abstract superclass but not from a concrete one by default, unless I have no testSelectors in which case I must be expecting to inherit them from my superclass.  If a test case with selectors wants to inherit selectors from a concrete superclass, override this to true in that subclass."

	^self ~~ self lookupHierarchyRoot
		and: [self superclass isAbstract
			or: [self testSelectors isEmpty]]

]

{ #category : 'Instance Creation' }
TestCase class >> suite [

	^self buildSuite

]

{ #category : 'Building Suites' }
TestCase class >> suiteClass [
	^TestSuite

]

{ #category : 'Accessing' }
TestCase class >> sunitVersion [
	^'4.0'

]

{ #category : 'Accessing' }
TestCase class >> testSelectors [

	^self sunitSelectors select: [:each | 'test*' sunitMatch: each]

]

{ #category : 'Dependencies' }
TestCase >> addDependentToHierachy: anObject [
	"an empty method. for Composite compability with TestSuite"

]

{ #category : 'Running' }
TestCase >> debug [
	[(self class selector: testSelector) runCase]
		ensure: [TestResource resetResources: self resources]

]

{ #category : 'Running' }
TestCase >> debugAsFailure [
	| semaphore |
	semaphore := Semaphore new.
	[semaphore wait.
	TestResource resetResources: self resources] fork.
	(self class selector: testSelector) runCaseAsFailure: semaphore

]

{ #category : 'Running' }
TestCase >> openDebuggerOnFailingTestMethod [
	"SUnit has halted one step in front of the failing test method. Step over the 'self halt' and
	 send into 'self perform: testSelector' to see the failure from the beginning"

	self
		halt;
		performTest

]

{ #category : 'Private' }
TestCase >> performTest [
	self perform: testSelector asSymbol

]

{ #category : 'Printing' }
TestCase >> printOn: aStream [

	aStream
		nextPutAll: self class printString;
		nextPutAll: ' debug: #';
		nextPutAll: (testSelector isNil ifTrue: ['??'] ifFalse: [testSelector])

]

{ #category : 'Dependencies' }
TestCase >> removeDependentFromHierachy: anObject [
	"an empty method. for Composite compability with TestSuite"

]

{ #category : 'Accessing' }
TestCase >> resources [
	"We give TestCase this instance-side method so that methods polymorphic with TestSuite can be code-identical.  Having this instance-side method also helps when writing tests of resource behaviour. Except for such tests, it is rare to override this method and should not be done without thought.  If there were a good reason why a single test case needed to share tests requiring different resources, it might be legitimate."

	^self class resources

]

{ #category : 'Running' }
TestCase >> run [
	| result |
	result := TestResult new.
	[self run: result]
		ensure: [TestResource resetResources: self resources].
	^result

]

{ #category : 'Running' }
TestCase >> run: aResult [
	aResult runCase: self

]

{ #category : 'Running' }
TestCase >> runCase [
	| tornDown |
	self resources do: [:each | each availableFor: self].
	[ self setUp.
	  self performTest
        ] ensure: [
	  tornDown ifNil:[
		tornDown := true .
		self tearDown
	  ]
        ]

]

{ #category : 'Running' }
TestCase >> runCaseAsFailure: aSemaphore [
	[self resources do: [:each | each availableFor: self].
	[self setUp.
	self openDebuggerOnFailingTestMethod] ensure: [self tearDown]]
		ensure: [aSemaphore signal]

]

{ #category : 'Accessing' }
TestCase >> selector [
	^testSelector

]

{ #category : 'Private' }
TestCase >> setTestSelector: aSymbol [
	testSelector := aSymbol

]

{ #category : 'Private' }
TestCase >> setUp [

]

{ #category : 'Testing' }
TestCase >> should: aBlock [
	self assert: aBlock value

]

{ #category : 'Testing' }
TestCase >> should: aBlock description: aString [
	self assert: aBlock value description: aString

]

{ #category : 'Testing' }
TestCase >> shouldnt: aBlock [
	self deny: aBlock value

]

{ #category : 'Testing' }
TestCase >> shouldnt: aBlock description: aString [
	self deny: aBlock value description: aString

]

{ #category : 'Testing' }
TestCase >> signalFailure: aString [
	TestResult failure sunitSignalWith: aString

]

{ #category : 'Running' }
TestCase >> tearDown [

]
