Extension { #name : 'TestResource' }

{ #category : 'Private' }
TestResource class >> _current [
		"Because we can run multiple sessions, we can get conflicts on a class instance variable.
		Use SessionTemps instead"

	^SessionTemps current
		at: self currentName
		otherwise: nil.

]

{ #category : 'Private' }
TestResource class >> _current: aTestResource [

	SessionTemps current at: self currentName put: aTestResource

]

{ #category : 'Running' }
TestResource class >> availableFor: aTestAsserter [
	aTestAsserter
		assert: self isAvailable
		description:
			'Unavailable resource ' , self name , ' requested by '
				, aTestAsserter printString

]

{ #category : 'Accessing' }
TestResource class >> current [
	"This is a lazy accessor:  the assert of self isAvailable does no work unless current isNil.  However this method should normally be sent only to a resource that should already have been made available, e.g. in a test whose test case class has the resource class in its #resources, so should never be able to fail the assert.
	If the intent is indeed to access a possibly-unprepared or reset-in-earlier-test resource lazily, then preface the call of 'MyResource current' with 'MyResource availableFor: self'."

	self
		assert: self isAvailable
		description:
			'Sent #current to unavailable resource ' , self name ,
					'.  Add it to test case'' class-side #resources (recommended) or send #availableFor: beforehand'.
	^self _current

]

{ #category : 'Accessing' }
TestResource class >> current: aTestResource [

	self _current: aTestResource.

]

{ #category : 'Private' }
TestResource class >> currentName [

	^(self thisClass name , '_current') asSymbol "In SessionTemps"

]

{ #category : 'Testing' }
TestResource class >> isAbstract [
	"Override to true if a TestResource subclass is Abstract and should not have
	TestCase instances built from it"

	^self name = #TestResource

]

{ #category : 'Testing' }
TestResource class >> isAlreadyAvailable [
	^self _current class == self

]

{ #category : 'Testing' }
TestResource class >> isAvailable [
	"This is (and must be) a lazy method.  If my current has a value, an attempt to make me available has already been made:  trust its result.  If not, try to make me available."

	self _current == nil  ifTrue: [self makeAvailable].
	^self isAlreadyAvailable

]

{ #category : 'Creation' }
TestResource class >> makeAvailable [
	"This method must be the _only_ way to set a notNil value for the unique instance (current).  First, obtain a candidate instance and set current to a notNil placeholder (any notNil object not an instance of me would do;  this version uses false).  Next, check any subordinate resources needed by this resource.  Lastly, setUp the candidate and put it in current if it is available, ensuring that it is torn down otherwise."

	| candidate |
	self _current: false.
	candidate := self new.
	self resources do: [:each | each availableFor: candidate].
	[candidate setUp.
	candidate isAvailable ifTrue: [self _current: candidate]]
		ensure: [self _current == candidate ifFalse: [candidate tearDown]]

]

{ #category : 'Creation' }
TestResource class >> new [
	"Use #current to get the valid current instance.  Use of #new to get an instance (that should never be the current one) could be done in bizarre circumstances, so is not blocked, but will usually be inappropriate."

	^super new initialize.

]

{ #category : 'Creation' }
TestResource class >> reset [
	[self isAlreadyAvailable ifTrue: [self _current tearDown]]
		ensure: [self _current: nil]

]

{ #category : 'Private' }
TestResource class >> resetOrAddResourcesTo: aCollection [
	"Add correctly set-up resources to the collection unless already there. Reset any imperfectly-set-up resources, so current isNil will return true if they are re-encountered via an indirectly self-prerequing resource;  circular references cannot be set up so will never reply true to isAlreadyAvailable, but may have correctly-set-up prereqs to add and/or imperfectly-set-up ones to reset, so do not abort the loop first time round."

	self _current == nil  ifTrue: [^self].
	self isAlreadyAvailable
		ifFalse:
			[self reset.
			self resources do: [:each | each resetOrAddResourcesTo: aCollection]]
		ifTrue:
			[(aCollection includes: self)
				ifFalse:
					[self resources do: [:each | each resetOrAddResourcesTo: aCollection].
					aCollection add: self]]

"The cloned 'self resources do: ...' line in both blocks is, I think, the best way to write this method so that its logic is clear.  The first loop resets this resource immediately, before traversing its resources;  the second traverses before adding"

]

{ #category : 'Running' }
TestResource class >> resetResources: topLevelResources [
	"Reset all imperfectly-set-up resources while gathering the rest for ordered resetting."

	| availableResources |
	availableResources := OrderedCollection new: topLevelResources size.
	topLevelResources
		do: [:each | each resetOrAddResourcesTo: availableResources].
	availableResources reverseDo: [:each | each reset]

]

{ #category : 'Accessing' }
TestResource class >> resources [
	^#()

]

{ #category : 'Accessing' }
TestResource >> description [

	description == nil
		ifTrue: [^''].

	^description

]

{ #category : 'Accessing' }
TestResource >> description: aString [

	description := aString

]

{ #category : 'Init / Release' }
TestResource >> initialize [
	"This method used to call setUp but now does nothing;  setUp is called by the framework at the appropriate point.  Subclasses may override to set the object to its default state."

]

{ #category : 'Testing' }
TestResource >> isAvailable [
	"Override to provide information on the readiness of the resource.  Put state-changing behaviour in setUp and keep this a state-preserving check as far as possible.  Where setUp is guaranteed to provide a valid resource if it completes, there is no need to override this."

	^true

]

{ #category : 'Accessing' }
TestResource >> name [

	name == nil
		ifTrue: [^self printString].

	^name

]

{ #category : 'Accessing' }
TestResource >> name: aString [

	name := aString

]

{ #category : 'Printing' }
TestResource >> printOn: aStream [

	aStream nextPutAll: self class printString

]

{ #category : 'Accessing' }
TestResource >> resources [
	^self class resources

]

{ #category : 'Running' }
TestResource >> setUp [
	"Does nothing. Subclasses should override this to initialize their resource"

]

{ #category : 'Running' }
TestResource >> tearDown [
	"Does nothing. Subclasses should override this to tear down their resource"

]
