Extension { #name : 'WriteStreamPortable' }

{ #category : 'Instance Creation' }
WriteStreamPortable class >> new [

"Disallowed.  To create a new WriteStream, use the class method on: instead."

self shouldNotImplement: #new

]

{ #category : 'Instance Creation' }
WriteStreamPortable class >> on: aCollection from: firstIndex to: lastIndex [
"Answer an instance of me on a copy of the argument, aCollection,
determined by the indices firstIndex and lastIndex. Position the instance
at the beginning of the collection."

^self basicNew
	on: aCollection
	from: firstIndex
	to: lastIndex

]

{ #category : 'Instance Creation' }
WriteStreamPortable class >> with: aCollection [
"Answer an instance of me on the argument, aCollection, positioned to
store objects at the end of aCollection."

^self basicNew with: aCollection

]

{ #category : 'Instance Creation' }
WriteStreamPortable class >> with: aCollection from: firstIndex to: lastIndex [
"Answer an instance of me on the subcollection of the argument,
aCollection, determined by the indices firstIndex and lastIndex. Position
the instance to store at the end of the subcollection."

^self basicNew with: (aCollection copyFrom: firstIndex to: lastIndex)

]

{ #category : 'Adding' }
WriteStreamPortable >> _nextPut: anObject [
"see PrintStream>>_nextPut:"

self nextPut: anObject

]

{ #category : 'Accessing' }
WriteStreamPortable >> contents [
"WriteStreams return the portion of their collection that has been written:
 the collection up to the next write-position."

readLimit := readLimit max: position.
^collection copyFrom: 1 to: position

]

{ #category : 'Character writing' }
WriteStreamPortable >> cr [
"Append a return character to the receiver."

self nextPut:  Cr   .

]

{ #category : 'Character writing' }
WriteStreamPortable >> crlf [
"Append a carriage return character followed by a line feed character to the receiver."

self nextPutAll: CrLf .

]

{ #category : 'Character writing' }
WriteStreamPortable >> crtab [
"Append a return character, followed by a single tab character, to the
receiver."

self nextPutAll: CrTab .

]

{ #category : 'Character writing' }
WriteStreamPortable >> crtab: anInteger [
"Append a return character, followed by anInteger tab characters, to the
receiver."

self nextPut: Cr .
anInteger timesRepeat: [self nextPut: Tab  ]

]

{ #category : 'ANSI' }
WriteStreamPortable >> flush [
	"Update a stream's backing store.
	Upon return, if the receiver is a write-back stream, the state of the
	stream backing store must be consistent with the current state of the
	receiver.
	If the receiver is not a write-back stream, the effect of this
	message is unspecified."

	"We do nothing; this method is provided for ANSI compatibility"

]

{ #category : 'Accessing' }
WriteStreamPortable >> next [

"Disallowed.  You cannot read a WriteStream."

self shouldNotImplement: #next

]

{ #category : 'Adding' }
WriteStreamPortable >> next: anInteger putAll: aCollection startingAt: startIndex [
"Store the next anInteger elements from the given collection."

| newEnd numPut |
collection class == aCollection class ifFalse:
	[^ super next: anInteger putAll: aCollection startingAt: startIndex ].

numPut := anInteger min: (aCollection size - startIndex + 1).
newEnd := position + numPut.
newEnd > writeLimit ifTrue:
	[^ super next: anInteger putAll: aCollection startingAt: startIndex "Trigger normal pastEndPut: logic"].

collection replaceFrom: position+1 to: newEnd with: aCollection startingAt: startIndex.
position := newEnd.

]

{ #category : 'Adding' }
WriteStreamPortable >> nextPut: anObject [

"Inserts anObject as the next element that the receiver can access for writing.
 Returns anObject."

 | pos coll |
 (pos := position) == (coll := collection) size ifTrue: [
   coll add: anObject .
   writeLimit := position :=  pos + 1 .
 ] ifFalse:[
   collection at: position + 1 put: anObject.
   position := position + 1.
 ].
 ^ anObject
]

{ #category : 'Adding' }
WriteStreamPortable >> nextPutAll: aCollection [

"Inserts the elements of aCollection as the next elements that the receiver can
 access.  Returns aCollection."
 | coll pos |
 (pos := position) == (coll := collection) size ifTrue:[
   coll addAll: aCollection .
   position := writeLimit := coll size .
 ] ifFalse:[ | newEnd |
   newEnd := pos + aCollection size. 
   collection replaceFrom: pos+1 to: newEnd  with: aCollection startingAt: 1.
   newEnd > writeLimit ifTrue: [
      collection size: newEnd.
      writeLimit := newEnd
   ].
   position := newEnd.
 ].
 ^ aCollection
]

{ #category : 'Adding' }
WriteStreamPortable >> nextPutAllBytes: aCharacterCollection [
  "Inserts the byte contents of aCharacterCollection as the next elements
 that the receiver can access.  Returns aCollection.  The receiver's collection
 must be a type of String."

  "Used in the implementation of PassiveObject."

  position == collection size
    ifFalse: [ ^ super nextPutAllBytes: aCharacterCollection ].
  collection addAllBytes: aCharacterCollection.
  position := position + aCharacterCollection _basicSize.
  ^ aCharacterCollection

]

{ #category : 'Private' }
WriteStreamPortable >> on: aCollection [

super on: aCollection.
readLimit := 0.
writeLimit := aCollection size

]

{ #category : 'Private' }
WriteStreamPortable >> on: aCollection from: firstIndex to: lastIndex [

| len |
collection := aCollection.
readLimit := writeLimit := lastIndex > (len := collection size)
	ifTrue: [len]
	ifFalse: [lastIndex].
position := firstIndex <= 1
	ifTrue: [0]
	ifFalse: [firstIndex - 1]

]

{ #category : 'Private' }
WriteStreamPortable >> pastEndPut: anObject [
"Grow the collection by adding anObject one past end.
Then put <anObject> at the current write position."

collection at: position + 1 put: anObject.
position := position + 1.
writeLimit := collection size.
^ anObject

]

{ #category : 'Positioning' }
WriteStreamPortable >> position: anInteger [
"Refer to the comment in PositionableStream|position:."

readLimit := readLimit max: position.
super position: anInteger

]

{ #category : 'Adding' }
WriteStreamPortable >> print: anObject [

	anObject printOn: self.

]

{ #category : 'Positioning' }
WriteStreamPortable >> reset [
"Refer to the comment in PositionableStream|reset."

readLimit := readLimit max: position.
position := 0

]

{ #category : 'Positioning' }
WriteStreamPortable >> setToEnd [
"Refer to the comment in PositionableStream|setToEnd."

readLimit := readLimit max: position.
super setToEnd.

]

{ #category : 'Positioning' }
WriteStreamPortable >> size [

^readLimit := readLimit max: position

]

{ #category : 'Accessing' }
WriteStreamPortable >> skip: anInteger [
"Set the receiver's position to be the current position+anInteger. Do not
 throw error if skipAmount would exceed collection bounds - ANSI compliance. "

self position: ((position + anInteger max: 0) min: writeLimit)

]

{ #category : 'Character writing' }
WriteStreamPortable >> space: anInteger [
"Append anInteger space characters to the receiver."

anInteger timesRepeat: [self nextPut: $    ]

]

{ #category : 'Character writing' }
WriteStreamPortable >> tab: anInteger [
"Append anInteger tab characters to the receiver."

anInteger timesRepeat: [ self nextPut: Tab ]

]

{ #category : 'Private' }
WriteStreamPortable >> with: aCollection [

super on: aCollection.
position := readLimit := writeLimit := aCollection size

]
