"
Traits provide a way to share sets of behavior (instance and class methods) between 
classes, other than using inheritance from a shared superclass. Traits allow you to abstract
a set of shared behavior that can be applies to unrelated classes. 

Create a Trait using Trait class >> name:instVars:classVars:classInstVars:inDictionary:, and 
add it to a class using Class >> addTrait: to add the instance method, Class >> addClassTrait: 
to add class methods, or addClassAndInstanceTrait: to add both.  Only one instance side and 
one class side trait can be added to a Class, but they do not need to be from the same Trait.

If the class directly implements a method for the given selector, this overrides that method 
in the Trait.

Adding, modifying, or removing a method on a Trait automatically recompiles the method for 
each class using that Trait.
"
Class {
	#name : 'Trait',
	#superclass : 'AbstractTrait',
	#instVars : [
		'classTrait'
	],
	#category : nil
}

{ #category : 'instance creation' }
Trait class >> name: aString instVars: ivNamesArray classVars: cvNamesArray classInstVars: civNamesArray [
	| tr |
	tr := self new.
	tr
		name: aString
		instVars: ivNamesArray
		classVars: cvNamesArray
		classInstVars: civNamesArray.
	^ tr
]

{ #category : 'instance creation' }
Trait class >> name: aString instVars: ivNamesArray classVars: cvNamesArray classInstVars: civNamesArray inDictionary: aSymbolDictionary [
	"if there is an existing trait with equivalent attributes, preserve the identity of the trait, otherwise "

	| newTrait sym |
	sym := aString asSymbol.
	newTrait := self
		name: aString
		instVars: ivNamesArray
		classVars: cvNamesArray
		classInstVars: civNamesArray.
	(aSymbolDictionary at: sym ifAbsent: [  ])
		ifNil: [ 
			"brand new instance"
			aSymbolDictionary at: sym asSymbol put: newTrait.
			^ newTrait ]
		ifNotNil: [ :oldTrait | 
			"if oldTrait equivalent to newTrait leave things as the are"
			(newTrait
				_equivalentToTrait: oldTrait
				name: aString
				newInstVars: ivNamesArray
				newClassInstVars: civNamesArray
				newClassVars: cvNamesArray)
				ifTrue: [ ^ oldTrait ]
				ifFalse: [ 
					"Note that the user is responsible to manage the classes that are dependent upon the old traits 
             ... that is best done after the new trait has been added to system."
 " more work needed, bug 51234"
          aSymbolDictionary  name == #Globals ifTrue:[ Error signal:'attempt to create new version of a Trait'].
					aSymbolDictionary at: sym asSymbol put: newTrait.
					^ newTrait ] ]
]

{ #category : 'browser methods' }
Trait >> _categoryOfSelector: selector [
	"Returns the category of the given selector, or 'unknown' if it isn't found."

	^ self traitImpl _categoryOfSelectorForInstanceTrait: selector
]

{ #category : 'private' }
Trait >> _classVarsEqual: anArrayOfClassvarNames [
	"Return true if the argument matches the classVarNames
   defined by the receiver , false otherwise."

	^ (SortedCollection withAll: self classVarNames)
		= (SortedCollection withAll: (anArrayOfClassvarNames collect: [ :n | n asSymbol ]))
]

{ #category : 'private' }
Trait >> _comment [
	"answer comment or nil if no comment defined"

	^ self traitImpl _comment
]

{ #category : 'private' }
Trait >> _equivalentToTrait: oldTrait name: aString newInstVars: ivNamesArray newClassInstVars: civNamesArray newClassVars: cvNamesArray [
	"answer true if the attributes of the oldTrait match those being passed in as arguments ... traits don't have 
		versions, but equivalent creation methods should not create new instances of a trait .. required for upgradeImage"

	| ivs civs cvars ok |
	ivs := oldTrait _instVarsEqual: ivNamesArray.
	civs := oldTrait classTrait _instVarsEqual: civNamesArray.
	cvars := oldTrait _classVarsEqual: cvNamesArray.
	ok := ivs and: [ civs and: [ cvars ] ] .
  "  ok ifFalse:[ self pause ]. "
  ^ ok .
]

{ #category : 'browser methods' }
Trait >> _selectorsReport: envId matching: aString primitivesOnly: primsBoolean includeDeprecated: inclDeprecBool [
	"Used by topaz (and GBS?).
  Result is a sorted SequenceableCollection  of Symbols, plus an optional string 'Omitted .. deprecated methods'.
  aString if not nil restricts the result to only include those selectors containing
  the case-insensitive substring aString .
  primsBoolean if true further restricts the result to only include only
  selectors of methods which are primitives."

	^ self traitImpl
		_selectorsReportForClassSide: false
		selectors: self localSelectors
		matching: aString
		primitivesOnly: primsBoolean
		includeDeprecated: inclDeprecBool
]

{ #category : 'browser methods' }
Trait >> _topazFileoutClass: headerStr asUtf8: utf8Bool env: envId [
	"method sent from topaz common code ..."

	^ self traitImpl _topazFileoutTrait: headerStr asUtf8: utf8Bool
]

{ #category : 'browser methods' }
Trait >> _topazMethodAt: aString [
	"Returns a GsNMethod, or signals an Error."

	^ self traitImpl _instanceTopazMethodAt: aString
]

{ #category : 'accessing' }
Trait >> addDependent: aClass [
	self traitImpl addInstanceDependent: aClass
]

{ #category : 'browser methods' }
Trait >> categoriesDo: aBlock [
	self traitImpl _instanceCategoriesDo: aBlock
]

{ #category : 'accessing' }
Trait >> category [
	^ self traitImpl _category
]

{ #category : 'accessing' }
Trait >> category: newCategory [
	"Sets the traitCategory variable of the receiver.
 The argument should be a kind of CharacterCollection or nil."

	"category/packageName used in Trait tonel file"

	^ self traitImpl _category: newCategory
]

{ #category : 'browser methods' }
Trait >> categoryOfSelector: selector [
	"Returns the category of the given selector, or nil if it isn't found."

	^ self traitImpl categoryOfSelectorForInstanceTrait: selector
]

{ #category : 'accessing' }
Trait >> classInstVarNames [
	^ self traitImpl classInstVarNames
]

{ #category : 'accessing' }
Trait >> classTrait [
	^classTrait
]

{ #category : 'accessing' }
Trait >> classTrait: object [
	classTrait := object
]

{ #category : 'accessing' }
Trait >> classVarNames [
	^ self traitImpl classVarNames
]

{ #category : 'accessing' }
Trait >> comment [
	^ self traitImpl comment
]

{ #category : 'accessing' }
Trait >> comment: aString [
	^ self traitImpl comment: aString
]

{ #category : 'compiling' }
Trait >> compile: sourceString [
	^self traitImpl compileMethod: sourceString
]

{ #category : 'compiling' }
Trait >> compile: sourceString category: aCategoryString [
	^ self traitImpl compileMethod: sourceString category: aCategoryString
]

{ #category : 'fileout' }
Trait >> fileOutTraitDefinitionOn: stream [

	self traitImpl fileOutTraitDefinitionOn: stream
]

{ #category : 'browser methods' }
Trait >> includesSelector: aString [
	^ self traitImpl _includesSelectorForInstanceTrait: aString
]

{ #category : 'accessing' }
Trait >> instanceTrait [
	^ self
]

{ #category : 'accessing' }
Trait >> instVarNames [
	^ self traitImpl instVarNames
]

{ #category : 'updating' }
Trait >> instVars: ivNamesArray classVars: cvNamesArray classInstVars: civNamesArray [
	self traitImpl
		instVars: ivNamesArray
		classVars: cvNamesArray
		classInstVars: civNamesArray
]

{ #category : 'testing' }
Trait >> isClassTrait [
	"Answer true if the receiver is a Class Trait instance"
	^ false
]

{ #category : 'testing' }
Trait >> isInstanceTrait [
	"Answer true if the receiver is an instance side Trait"
	^ true
]

{ #category : 'accessing' }
Trait >> localSelectors [
	^ self traitImpl instanceSelectors
]

{ #category : 'accessing' }
Trait >> name [
	^ self traitImpl name
]

{ #category : 'initialization' }
Trait >> name: aSymbol instVars: ivNamesArray classVars: cvNamesArray classInstVars: civNamesArray [
	traitImpl := GsTraitImpl
		name: aSymbol
		instVars: ivNamesArray
		classVars: cvNamesArray
		classInstVars: civNamesArray.
	classTrait := ClassTrait trait: traitImpl instance: self.
	traitImpl
		instanceTrait: self;
		classTrait: classTrait
]

{ #category : 'updating' }
Trait >> objectSecurityPolicy: anObjectSecurityPolicy [
	"Assigns the receiver and subcomponents to the given security policy."

	super objectSecurityPolicy: anObjectSecurityPolicy.
]

{ #category : 'updating' }
Trait >> removeAllMethods [
	self traitImpl removeAllInstanceSelectors
]

{ #category : 'updating' }
Trait >> removeDependent: aClass [

	self traitImpl removeInstanceDependent: aClass
]

{ #category : 'updating' }
Trait >> removeDependentOnly: aClass [
	"remove the dependent, but leave the trait methods in aClass"

	self traitImpl removeInstanceDependentOnly: aClass
]

{ #category : 'updating' }
Trait >> removeFromSystem [
	"When a trait is removed from the system it should:

	- Remove it self from its users.
	- Remove its classTrait from its users.
	- Remove itself from it's Symbol Dictionary"

	| arr |
	self traitImpl removeFromSystem.
	arr := GsCurrentSession currentSession symbolList
		dictionariesAndSymbolsOf: self.
	arr
		do: [ :ar | 
			| symDict traitKey |
			symDict := ar at: 1.
			traitKey := ar at: 2.
			symDict removeKey: traitKey ]
]

{ #category : 'updating' }
Trait >> removeSelector: aString [
	self traitImpl removeSelector: aString
]

{ #category : 'accessing' }
Trait >> sourceCodeAt: selectorSymbol [
	^ self traitImpl instanceSourceCodeAt: selectorSymbol
]

{ #category : 'accessing' }
Trait >> traitImpl [
	^traitImpl
]

{ #category : 'accessing' }
Trait >> traitImpl: object [
	traitImpl := object
]
