"
This class is part of the External Sessions implementation and 
represents an execution Error from the GCI interface to the external session.

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
	gciErrSType: Object
	externalSession: Object
	originalNumber: Object

"
Class {
	#name : 'GciError',
	#superclass : 'Error',
	#instVars : [
		'gciErrSType',
		'externalSession',
		'originalNumber'
	],
	#category : 'External Sessions'
}

{ #category : 'signalling' }
GciError class >> signal: aGciErrSType in: anExternalSession [
	self new
		signal: aGciErrSType
		in: anExternalSession
    details: nil

]

{ #category : 'signalling' }
GciError class >> signal: aGciErrSType in: anExternalSession details: aString [
	self new
		signal: aGciErrSType
		in: anExternalSession
    details: aString

]

{ #category : 'signalling' }
GciError >> _error: aGciErrSType in: anExternalSession [
	gciErrSType := aGciErrSType.
	externalSession := anExternalSession.
  aGciErrSType ifNil:[
    originalNumber := 4100 "GCI_ERR_BAD_SESSION_ID".
    messageText := 'invalid session' copy .
  ] ifNotNil:[
         (originalNumber := gciErrSType number) == 1001 ifTrue: [^self signalCompileError].
         messageText := gciErrSType message.
  ].
	"avoid killing current session with fatal errors"
	(originalNumber between: 4000 and: 4999) ifTrue: [
		messageText add: ' original number: ' , originalNumber asString,
		' for session ' , externalSession _describe.
		"and self number is inherited value 2710"
	] ifFalse: [
		self _number: originalNumber.
	].

]

{ #category : 'accessing' }
GciError >> actualExceptionClassOr: aBlock [
	"Answer the class of the actual exception if there is one,
	 or the result of evaluating aBlock if not."

	| actualCls |
	actualCls := (LegacyErrNumMap atOrNil: self number ) ifNotNil: [ :a | a atOrNil: 1].
	^actualCls ifNil: aBlock

]

{ #category : 'accessing' }
GciError >> category [

	^gciErrSType category.

]

{ #category : 'updating' }
GciError >> clearStack [
  (originalNumber ~~ nil and:[ originalNumber < 4000]) ifTrue:[
         externalSession clearStackFor: gciErrSType.
  ].

]

{ #category : 'accessing' }
GciError >> context [

	^gciErrSType context.

]

{ #category : 'updating' }
GciError >> continue [
	"Continue code execution in GemStone after an error.
	See GciContinue() in the GemBuilder for C manual for details."

	^externalSession continue: gciErrSType context.

]

{ #category : 'updating' }
GciError >> continueWith: anObject [
	"This function is a variant of the continue method, except
	that it allows you to modify the call stack before attempting
	to continue the suspended Smalltalk execution.
	See GciContinueWith() in the GemBuilder for C manual for details."

	^externalSession
		continue: gciErrSType context
		with: anObject.

]

{ #category : 'accessing' }
GciError >> externalErrorNumber [

	^gciErrSType number.

]

{ #category : 'accessing' }
GciError >> gciErrSType [

	^gciErrSType

]

{ #category : 'testing' }
GciError >> matchesClasses: expClass [
	"expClass is either a Class or an Array of Classes"

	| actualCls |
	actualCls := self actualExceptionClassOr: [^false].
	expClass _isArray
		ifTrue: [^expClass anySatisfy: [:aCls | actualCls isSubclassOf: aCls]]
		ifFalse: [^actualCls isSubclassOf: expClass]

]

{ #category : 'accessing' }
GciError >> originalNumber [
  ^ originalNumber

]

{ #category : 'signalling' }
GciError >> signal: aGciErrSType in: anExternalSession [
  ^ self signal: aGciErrSType in: anExternalSession details: nil

]

{ #category : 'signalling' }
GciError >> error: aGciErrSType in: anExternalSession details: aString [
  self _error: aGciErrSType in: anExternalSession .
  aGciErrSType ifNotNil:[ | num |
    num := aGciErrSType number .
    (num ~~ 2203"GCI_ERR_OP_IN_PROGRESS" and:[ num ~~ 2336"RT_ERR_CLIENT_FWD_SEND"]) ifTrue:[
	    self stack ifNotNil: [:stack | messageText lf ; add: stack].
    ].
  ].
  aString ifNotNil:[ messageText add:', '; add: aString asString].
  self details: messageText.

]

{ #category : 'signalling' }
GciError >> signal: aGciErrSType in: anExternalSession details: aString [
  self error: aGciErrSType in: anExternalSession details: aString ;
     signal .
]

{ #category : 'signalling' }
GciError >> signalCompileError [

	| delimiter |
	self details: gciErrSType message.
	gsArgs := gciErrSType args.
	gsArgs size: gciErrSType argCount.
	gsArgs := gsArgs collect: [:each |
		externalSession resolveResult: each toLevel: 4.
	].
	messageText := String withAll: 'Compile error(s): '.
	delimiter := ''.
	gsArgs first do: [:each |
		messageText
			addAll: delimiter;
			addAll: (each at: 3);
			addAll: ' (error #';
			addAll: (each at: 1) printString;
			addAll: ') at offset ';
			addAll: (each at: 2) printString.
		delimiter := '; '.
	].
	self signal.

]

{ #category : 'accessing' }
GciError >> stack [
  gciErrSType ifNil:[ ^ nil ].
	gciErrSType context == nil asOop ifTrue: [^nil].	"NO CALL IN PROGRESS"
	self externalErrorNumber >= 4000 ifTrue: [^nil].	"NOT A RUNTIME ERROR"
  ^ externalSession _getStackForOop: gciErrSType context .

]
