"
GsTsExternalSession is an implementation of AbstractExternalSession that
uses the threads-safe GCI library, $GEMSTONE/lib/libgcits*.so, to
communicate to the Gem which is servicing an instance of GsTsExternalSession .

A loaded libgcits library is represented by an instance of GciTsLibrary, which
is a parameter to GsTsExternalSession.

While the target repository is often the same version of GemStone as the 
executing session, you may also login to repository running a different version
of GemStone. GciTsLibrary supports loading a libgcits library from a different 
version of GemStone; this requires cpp; you should have gcc/g++ installed.

GsTsExternalSession is not supported on AIX.

instVar notes
  nbCallInProgress values 
    0 no call ,
    1 nbLogin in progress
    2 execution in progress 
    3 execution in progress for getting error details

Example:
| sess |
sess := GsTsExternalSession newDefault .
sess username: 'DataCurator'; password: 'swordfish' ; login .
sess executeString: 'System stoneName'.
sess executeBlock: [2 + 5].
sess logout.

Example loading a different GciTsLibrary version:
| param sess |
param := GemStoneParameters new
  gemStoneName: 'gemstone353';
  username: 'DataCurator';
  password: 'swordfish';
  gemService: '!#netldi:ldi_353!gemnetobject';
  yourself.
sess := GsTsExternalSession parameters: param
  library: (GciTsLibrary
     newForVersion: '3.5.3'
     product: '/lark/users/gsadmin/GemStone64_353').
sess login.
sess executeString: 'System stoneName'.
sess logout.

Example using a non-blocking login and execute:
| sess result |
sess := GsTsExternalSession newDefault .
sess username: 'DataCurator'; password: 'swordfish' ; nbLogin .
sess waitForReadReady.
sess nbLoginFinished. 
sess nbExecute: '2 + 5'.
sess nbResult.
sess logout.

"
Class {
	#name : 'GsTsExternalSession',
	#superclass : 'AbstractExternalSession',
	#instVars : [
		'tsSession',
		'socket',
		'gciErr',
		'callouts',
		'lib',
		'objInfoBuffers',
		'stoneSessionId',
		'gemProcessId',
		'parameters',
		'performArgs',
		'lastResult',
		'nbCallInProgress',
		'nbLoginSession',
    'stoneSessionSerial',
    'logger'
	],
	#gs_options : [
		'instancesNonPersistent'
	],
	#category : nil
}

{ #category : 'Instance creation' }
GsTsExternalSession class >> newDefault: aGciTsLibrary [
  "This creates an external session that is set to the user, host,
   and stone of the current gem with the password 'swordfish', and the
   argument library version. You may update any of these parameters 
   before login for more complex environments."

 |  params |
 (params :=  GemStoneParameters new)
    gemService: GsNetworkResourceString defaultGemNRSFromCurrent asString ;
    gemStoneName: GsNetworkResourceString defaultStoneNRSFromCurrent asString ;
    username: System myUserProfile userId;
    password: 'swordfish' .
  ^ self parameters: params library: aGciTsLibrary
]

{ #category : 'Instance creation' }
GsTsExternalSession class >> newDefault [
  "This creates an external session that is set to the user, host,
   and stone of the current gem with the password 'swordfish', and the
   same library version as the current image. You may update any
   of these parameters before login for more complex environments."

  ^ self newDefault: GciTsLibrary newDefault 
]

{ #category : 'Instance creation' }
GsTsExternalSession class >> gemNRS: gemNRS stoneNRS: stoneNRS username: aUsername password: aPassword[
 |  params |
 (params :=  GemStoneParameters new)
    gemService: gemNRS asString ;
    gemStoneName: stoneNRS asString ;
    username: aUsername ;
    password: aPassword .
  ^ self parameters: params library: GciTsLibrary newDefault
]

{ #category : 'Instance creation' }
GsTsExternalSession class >> gemNRS: gemNRS stoneNRS: stoneNRS username: aUsername 
        password: aPassword hostUsername: hostUsername hostPassword: hostPassword [
 |  params |
 (params :=  GemStoneParameters new)
    gemService: gemNRS asString ;
    gemStoneName: stoneNRS asString ;
    username: aUsername ;
    password: aPassword ;
    hostUsername: hostUsername ;
    hostPassword:  hostPassword .
  ^ self parameters: params library: GciTsLibrary newDefault
]

{ #category : 'Instance creation' }
GsTsExternalSession class >> parameters: aGemStoneParameters library: aGciTsLibrary [
 "Basic instance creation.  The GciTsLibrary may be for the same version of GemStone
  or for a different version; see GciTsLibrary instance creation methods"

 ^ self _basicNew  
    _parameters: aGemStoneParameters library: aGciTsLibrary
]

{ #category : 'Instance creation' }
GsTsExternalSession class >> new [
  "disallowed"
  self shouldNotImplement: #new
]

{ #category : 'Private' }
GsTsExternalSession >> _calloutAt: ofs name: aSymbol [
 | c n |
 c := callouts at: ofs .
 (n := c name) == aSymbol ifFalse:[ Error signal:'callout mismatch'].
 ^ c

]

{ #category : 'Private' }
GsTsExternalSession >> _clearConnection [
  tsSession := nil .
  gciErr := nil .
  socket := nil .
  gemProcessId := nil .
  stoneSessionId := nil .
  lastResult := nil .
  nbCallInProgress := 0 .
  nbLoginSession  := nil .
  stoneSessionSerial := nil .
]

{ #category : 'Private' }
GsTsExternalSession >> _closeConnection [
  "do not close the socket here, let GciTsNbLogout do that"
  tsSession ifNotNil:[ self nbLogout ].

]

{ #category : 'Private' }
GsTsExternalSession >> _getObjInfo: anOop buffer: aCByteArray size: aSize [ 
 | c res info oId |
 c := self _calloutAt: 9 name: #'GciTsFetchObjInfo'.
 info := objInfoBuffers at: 1 .
 res := c callWith: { tsSession . anOop . 0"BoolType addToExportSet".
                       info .
                       aCByteArray . aSize .
                       gciErr }.
 res == -1 ifTrue:[
    self _signalError:'GciTsFetchObjInfo failed'.
 ].
 (oId := info uint64At: 0 ) == anOop ifFalse:[
   Error signal:'Inconsistent objId'.
 ].
 ^ info

]

{ #category : 'Private' }
GsTsExternalSession >> _getSocketFd [
  | c res |
  c := self _calloutAt: 13 name: #'GciTsSocket' . 
  res := c callWith: { tsSession . gciErr }.
  res > 0 ifFalse:[ 
    self _signalError:'GciTsSocket failed'.
  ].
  ^ res

]

{ #category : 'Private' }
GsTsExternalSession >> _getStackForOop: gcierrContextOop [
  self isCallInProgress ifTrue:[ 
    ^ 'call in progress, stack not available' ]. 
  ^ [ | str |
      str := 'GsTsExternalSession _stackReport: ' , gcierrContextOop asString .
      self _nbExecute: str forError: true detach: false .
      self waitForReadReadyTimeOut: 20000 .
      self nbResult .
    ] on: Error do:[:ex | ex return: 'error getting stack'].

]

{ #category : 'Private' }
GsTsExternalSession >> _nbResult [
 | c res inProg |
 c := self _calloutAt: 12 name: #'GciTsNbResult' .  
 res := c callWith: { tsSession . gciErr }.
 inProg := nbCallInProgress .
 nbCallInProgress := 0 .
 res == 1"OOP_ILLEGAL" ifTrue:[
   inProg > 2 ifTrue:[ Error signal:'GciTsNbResult failed'].
   self _signalError:'GciTsNbResult failed'.
 ] .
 ^ res .

]

{ #category : 'Private' }
GsTsExternalSession >> _parameters: aGemStoneParameters library: aGciTsLibrary [
  parameters := aGemStoneParameters .
  parameters ifNil:[
    (parameters := GemStoneParameters new)
      gemService: GsNetworkResourceString defaultGemNRSFromCurrent asString ;
      gemStoneName: GsNetworkResourceString defaultStoneNRSFromCurrent asString ;
      username: System myUserProfile userId ;
      password: 'swordfish' copy  .
  ].
  callouts := aGciTsLibrary callouts .
  lib := aGciTsLibrary .  "save for debugging"
  nbCallInProgress := 0

]

{ #category : 'Private' }
GsTsExternalSession >> _signalError: aString [
  | err |
  (err := GciError new) error: gciErr in: self details: nil .
  (err originalNumber between: 4000 and: 4999) ifTrue:[
    self _closeConnection . "ensure connection is closed after fatal error"
  ] ifFalse:[
    gciErr := GciErrSType new .
  ].
  err signal .

]

{ #category : 'Private' }
GsTsExternalSession >> _stringArg: aString [
 ^ aString ifNil:[ '' ] ifNotNil:[:s | s ].

]

{ #category : 'Calls' }
GsTsExternalSession >> _waitForLogout [
  "not implemented"
  ^ self
]

{ #category : 'Calls' }
GsTsExternalSession >> abort [ 
  | c res |
  c := self _calloutAt: 3 name: #'GciTsAbort' .
  res := c callWith: { tsSession . gciErr }.
  res == 1 ifFalse:[
    self _signalError:'GciTsAbort failed'.
  ].
]

{ #category : 'Calls' }
GsTsExternalSession >> begin [ 
  | c res |
  c := self _calloutAt: 18 name: #'GciTsBegin' .
  res := c callWith: { tsSession . gciErr }.
  res == 1 ifFalse:[
    self _signalError:'GciTsBegin failed'.
  ].
]

{ #category : 'Accessing' }
GsTsExternalSession >> clearError [
  gciErr ifNotNil:[
    gciErr memset: 0 from: 0 to: gciErr size - 1
  ].

]

{ #category : 'Error handling' }
GsTsExternalSession >> clearStackFor: aGciError [
 | contextOop c res |
 (contextOop := aGciError context) == 20"nil asOop" ifTrue: [^self].
 c := self _calloutAt: 8 name: #'GciTsClearStack' .
 res := c callWith: { tsSession . contextOop . gciErr }.
 res == 1 ifFalse:[
   self _signalError:'GciTsClearStack failed'. 
 ]

]

{ #category : 'Calls' }
GsTsExternalSession >> commitOrError [ 
  "Returns true or signals an error."
  | c res |
  c := self _calloutAt: 4 name: #'GciTsCommit' .
  res := c callWith: { tsSession . gciErr }.
  res == 1 ifFalse:[ | err | 
    (err := GciError new) error: gciErr in: self details: nil .
    err originalNumber >= 4000 ifTrue:[
      self _closeConnection .
      ^ err signal
    ].
    TransactionError new reason: 'GciTsCommit failed' ; 
         arg: err ;
         signal:'commit conflicts, ', err asString
  ].
  ^ true
]

{ #category : 'Calls' }
GsTsExternalSession >> commit [ 
  ^ self commitOrError 
]


{ #category : 'Error handling' }
GsTsExternalSession >> continue: contextOop replacingTopOfStackWithOop: tosOop [
  | c res flags |
  contextOop == 20"nil asOop" ifTrue: [^self].
  c := self _calloutAt: 7 name: #'GciTsContinueWith' .
  flags := 1 "GCI_PERFORM_FLAG_ENABLE_DEBUG" .
  res := c callWith: { tsSession . contextOop . tosOop . 
                       nil "GciErrSType *continueWithError = NULL" .
                       flags . gciErr }.
  res == 1"OOP_ILLEGAL" ifTrue:[
    self _signalError:'GciTsContinueWith failed'.
  ].
  ^ ( lastResult := self resolveResult: res ).

]

{ #category : 'Accessing' }
GsTsExternalSession >> errorString [
  gciErr ifNotNil:[
    gciErr number ~~ 0 ifTrue:[
      ^ 'ERROR ', gciErr number asString, ' ', gciErr message asString
    ].
  ].
  ^ 'no error'.
]

{ #category : 'Calls' }
GsTsExternalSession >> executeString: aString [
  "Execute the string expression in the external Gem and answer the result.
   The best values to return are specials.
   Strings and other byte objects are copied to the local Gem.
   All other responses are returned as a 3-element Array containing the OOP of the 
   result, the OOP of the class of the result, and the size of the result."

  | c res |
  c := self _calloutAt: 5 name: #'GciTsExecute' .
  aString _isOneByteString ifFalse:[
    ArgumentError signal:'argument to executeString is not a String'.
  ].
  res := c callWith: { tsSession . 
                       aString . 74753"String asOop" .  "sourcStr, sourceOop"
                       1"OOP_ILLEGAL,  context" .
                       20"nil asOop , symbolList" .  
                       1"GCI_PERFORM_FLAG_ENABLE_DEBUG, flags" .
                        0"envId" . 
                       gciErr }.
  res == 1"OOP_ILLEGAL" ifTrue:[
    self _signalError:'GciTsExecute failed'.
  ] .
  ^ ( lastResult := self resolveResult: res ).

]

{ #category : 'Calls' }
GsTsExternalSession >> forceLogout [
 ^ self nbLogout

]

{ #category : 'Calls' }
GsTsExternalSession >> forkString: aString [

 "Must be followed by GsTsExternalSession >> nbResult ."

^ self nbExecute: aString

]

{ #category : 'Calls' }
GsTsExternalSession >> forkAndDetachString: aString [
  | str |
 "Wraps aString with a default error handling and printing .
  If result returns, must be followed by GsTsExternalSession >> nbResult.
  The execution will stay running after nbLogout of this session,
  but will still respond to softBreak or hardBreak from this instance prior to logout."

 str := ' [ System gemConfigurationAt:#GemExceptionSignalCapturesStack put: true .
   ', aString ,'
 ] on: Exception do:[:ex___ |
   GsFile gciLogServer: ex___ stackReport .
   System removeGemLogOnExit: false .
   "System waitForDebug"  "uncomment if you want to be able to debug this gem"
   ex___ pass
 ]' .
 ^ self _forkAndDetachString: str .

]

{ #category : 'Calls' }
GsTsExternalSession >> _forkAndDetachString: aString [

 "If result returns, must be followed by GsTsExternalSession >> nbResult.
  The execution will stay running after nbLogout of this session,
  but will still respond to softBreak or hardBreak from this instance prior to logout.

  It is strongly recommended that aString include error handling and printing so that
  there is enough information in the gem log to debug problems. 
  "

  ^ self _nbExecute: aString forError: false  detach: true .

]

{ #category : 'Calls' }
GsTsExternalSession >> forkAndDetachBlock: aBlock [
  "Same detach semantics and default error handling as forkAndDetachString:. "

  ^ self forkAndDetachBlock: aBlock withArguments: { } .

]

{ #category : 'Calls' }
GsTsExternalSession >> forkAndDetachBlock: aBlock withArguments: anArray [
  "Same detach semantics and default error handling as forkAndDetachString:. "
  | str |
  str := self _stringForBlock: aBlock withArguments: anArray .
  ^ self forkAndDetachString: str . 

]

{ #category : 'Parameters' }
GsTsExternalSession >> gemNRS: anNRS [
       "Set the GemService parameters for the logon to the value
        of anNRS, which may be a String or a GsNetworkResourceString instance."
  parameters gemService: anNRS asString .
  (anNRS isKindOf: GsNetworkResourceString) ifTrue:[
    self dynamicInstVarAt: #gemHost put: anNRS node .
  ].
]

{ #category : 'Parameters' }
GsTsExternalSession >> gemNRS [
   ^ parameters ifNotNil:[:p | p gemService ].
]

{ #category : 'Accessing' }
GsTsExternalSession >> gemProcessId [
  ^ gemProcessId ifNil:[ 
    gemProcessId := self executeString: 'System gemProcessId '
  ].
]

{ #category : 'Calls' }
GsTsExternalSession >> hardBreak [
  | c res |
  c := self _calloutAt: 17 name: #'GciTsBreak' .
  res := c callWith: { tsSession . 1 . gciErr }.
  res == 1 ifFalse:[
    self _signalError:'GciTsBegin failed'.
  ].
  ^ self
]

{ #category : 'Parameters' } 
GsTsExternalSession >> hostPassword: aString [
  parameters hostPassword: aString copy.
]

{ #category : 'Parameters' } 
GsTsExternalSession >> hostUsername: aString [
  parameters hostUsername: aString copy.
]

{ #category : 'Parameters' }
GsTsExternalSession >> stoneNRS: anNRS [
  "Set the Stone parameters for the logon to the value
   of anNRS, which may be a String or a GsNetworkResourceString instance."
  parameters gemStoneName: anNRS asString
]

{ #category : 'Accessing' }
GsTsExternalSession >> isCallInProgress [
  "for use after login or nbLogin has completed."
  ^ tsSession ~~ nil and:[ nbCallInProgress ~~ 0 ]
]

{ #category : 'Accessing' }
GsTsExternalSession >> isRemoteServerBigEndian [
  ^ self executeString: 'System gemIsBigEndian'
]

{ #category : 'Accessing' }
GsTsExternalSession >> isLoggedIn [
  | c res |
  tsSession ifNil:[ ^ false ].
  c := self _calloutAt: 13 name: #'GciTsSocket' . 
  res := c callWith: { tsSession . gciErr }.
  ^ res > 0

]

{ #category : 'Non Blocking Calls' }
GsTsExternalSession >> isResultAvailable [
  "Return true if result ready or if the socket shows an error"
  | status |
  self isCallInProgress ifFalse:[ 
     nbLoginSession ifNotNil:[ ^ false ].
     Error signal:'call not in progress'. 
  ].
  status := socket readWillNotBlockWithin: 0 .
  status == true ifTrue:[ self nbResult .  ^ true ].
  status ifNil:[ ^ self _signalError:'socket error' ].
  ^ false 
]

{ #category : 'Accessing' }
GsTsExternalSession >> lastResult [
  "See GsTsExternalSession>> resolveResult: for documentation of lastResult."
  ^ lastResult

]

{ #category : 'Login' }
GsTsExternalSession >> login [
 | c ptr args |
 self isLoggedIn ifTrue:[ ^ ImproperOperation signal:'already logged in'].
 c := self _calloutAt: 1 name: #'GciTsLogin' .
 gciErr := GciErrSType new .
 args := {    
   self _stringArg: parameters gemStoneName .
   self _stringArg: parameters hostUsername .
   self _stringArg: parameters hostPassword .
   parameters passwordIsEncryptedAsIntegerBoolean.  "BoolType  hostPwIsEncrypted"
   self _stringArg: parameters gemService .
   self _stringArg: parameters username .
   self _stringArg: parameters password .
   parameters loginFlags .  "unsigned int loginFlags per GCI_LOGIN* "
	  -1  "haltOnErrNum default (use config file)"
 }.
 c numFixedArgs == 11 ifTrue:[  "GciTs v3.5.x + "
   args add:( CByteArray gcMalloc: 4 "BoolType *executedSessionInit").
 ].
 args add: gciErr .
 ptr := c callWith: args .
 ptr memoryAddress == 0 ifTrue:[
   self _signalError:'GciTsLogin failed'.
 ].
 tsSession := ptr .
 (socket := GsSocket fromFileHandle: self _getSocketFd) 
    setCloseOnGc: true "fix 50624" . 
 self _allocateBuffers .
 gciErr number > 0 ifTrue:[
    self _signalError:'Warning from GciTsLogin'.
 ]
]

{ #category : 'Private' }
GsTsExternalSession >> _debugConnect: processId [
  | c args ptr |
  self isLoggedIn ifTrue:[ ^ ImproperOperation signal:'already logged in'].
  c := self _calloutAt: 22 name: #GciTsDebugConnectToGem .
  gciErr := GciErrSType new .
  args := { processId . gciErr } .
  ptr := c callWith: args .
  ptr memoryAddress == 0 ifTrue:[
    self _signalError:'GciTsDebugConnectToGem failed'.
  ].
  tsSession := ptr .
  (socket := GsSocket fromFileHandle: self _getSocketFd )
     setCloseOnGc: true .
  self _allocateBuffers .
  gciErr number > 0 ifTrue:[ "No warnings expected."
     self _signalError:'error from GciTsDebugConnectToGem'.
  ].
  
]

{ #category : 'Private' }
GsTsExternalSession >> _startDebugService: integerToken [
  "for use only immediately after _debugConnect: "
  | c args res |
  c := self _calloutAt: 23 name: #GciTsDebugStartDebugService .
  args := { tsSession . integerToken . gciErr }.
  res := c callWith: args .
  res == 1"C true" ifFalse:[ self _signalError:'GciTsDebugStartDebugService failed'].

]

{ #category : 'Login' }
GsTsExternalSession >> debugGem: processId token: anInteger [
  "Arguments should be copied from the DEBUGGEM printout in a topaz .out file or log file.
   All normal login arguments are ignored ."
  self _debugConnect: processId .
  self _startDebugService: anInteger
]

{ #category : 'Private' }
GsTsExternalSession >> _allocateBuffers [
 objInfoBuffers ifNil:[
   objInfoBuffers := { CByteArray gcMalloc: 40 "sizeof(GciTsObjInfo)" .
      CByteArray gcMalloc: 1024 .
      {} "array of oops to release" } .
 ]

]

{ #category : 'Login' }
GsTsExternalSession >> loginSolo [
   parameters setSoloLogin .
   ^ self login
]

{ #category : 'Logging' }
GsTsExternalSession >> logger: oneArgBlock [
  logger := oneArgBlock
]

{ #category : 'Logging' }
GsTsExternalSession >> log: aString [
  logger ifNotNil:[:blk | blk value: aString ]
]

{ #category : 'Logging' }
GsTsExternalSession >> loggingToServer [
  self logger: [:message | GsFile gciLogServer: message]
]

{ #category : 'Logging' }
GsTsExternalSession >> suppressLogging [
  logger := nil 
]

{ #category : 'Logging' }
GsTsExternalSession >> loggingToClient [
  "GsTsExternalSession does not support client user actions, so it cannot log
   to the GCI client."
  self shouldNotImplement: #loggingToClient
]

{ #category : 'Calls' }
GsTsExternalSession >> logout [
  | c res |
  socket ifNotNil:[:s | s setCloseOnGc: false ].  "prevent duplicate close after libgcits closes it"
  c := self _calloutAt: 2 name: #'GciTsLogout' .
  res := c callWith: { tsSession . gciErr } .
  self _clearConnection .
]

{ #category : 'Logging' }
GsTsExternalSession >> quiet [
  ^ self 
]

{ #category : 'Logging' }
GsTsExternalSession >> quietLogout [
  ^ self 
]

{ #category : 'Non Blocking Calls' }
GsTsExternalSession >> nbCommittedResult [
 ^ Object objectForOop: self _nbResult  "assumes result is committed in this stone"

]

{ #category : 'Non Blocking Calls' }
GsTsExternalSession >> nbExecute: aString [
 "Must be followed by GsTsExternalSession >> nbResult ."
  ^ self _nbExecute: aString forError: false  detach: false .

]

{ #category : 'Private' }
GsTsExternalSession >> _nbExecute: aString forError: errBoolean detach: detachBool [
 "Must be followed by GsTsExternalSession >> nbResult ."
  | c res flags |
  c := self _calloutAt: 14 name: #'GciTsNbExecute' . 
  aString _isOneByteString ifFalse:[
    ArgumentError signal:'argument to nbExecute: is not a String'.
  ].
  flags := 1 "GCI_PERFORM_FLAG_ENABLE_DEBUG".
  detachBool ifTrue:[ flags := flags bitOr: 16r40"GCI_PERFORM_DETACH" ].
  res := c callWith: { tsSession .
                       aString . 74753"String asOop" .  "sourcStr, sourceOop"
                       1"OOP_ILLEGAL,  context" .
                       20"nil asOop , symbolList" .
                       flags .
                        0"envId" .
                       gciErr }.
  res == 1 ifFalse:[
    errBoolean ifTrue:[ Error signal:'GciTsNbExecute failed' ].
    self _signalError:'GciTsNbExecute failed'.
  ].
  nbCallInProgress := (errBoolean ifTrue:[ 3 ] ifFalse:[ 2 ]) . "fix 50623"

]

{ #category : 'Login' }
GsTsExternalSession >> nbLogin [
 "Start a login via GciTsNbLogin .
  socket  instVar is set to a socket that can be polled to detect completion
  of nbLogin.  
  Use  waitForReadReady or waitForReadReadyTimeOut: to wait for the login to finish .
  #nbLoginFinished must be called once after  
  waitForReadReady or waitForReadReadyTimeOut:  returns true."
 | c ptr args sockHolder |
 self isLoggedIn ifTrue:[ ^ ImproperOperation signal:'already logged in'].
 nbCallInProgress := 1 .
 gciErr := GciErrSType new .
 c := self _calloutAt: 15 name: #'GciTsNbLogin' .
 args := {    
   self _stringArg: parameters gemStoneName .
   self _stringArg: parameters hostUsername .
   self _stringArg: parameters hostPassword .
   parameters passwordIsEncryptedAsIntegerBoolean.  "BoolType  hostPwIsEncrypted" 
   self _stringArg: parameters gemService .
   self _stringArg: parameters username .
   self _stringArg: parameters password .
   parameters loginFlags .  "unsigned int loginFlags per GCI_LOGIN* "
	 -1 . "haltOnErrNum default (use config file)"
   (sockHolder := CByteArray gcMalloc: 4 "int loginPollSocket")
 }.
 ptr := c callWith: args .
 ptr memoryAddress == 0 ifTrue:[
   nbCallInProgress := 0 .
   Error signal:'GciTsNbLogin failed, malloc or socket failure' .
 ].
 nbLoginSession := ptr .
 socket := GsSocket fromFileHandle: (sockHolder int32At: 0) . "rcv side of socket pair"
]

{ #category : 'Login' }
GsTsExternalSession >> nbLoginFinished [
  "Must be called once after  waitForReadReady or waitForReadReadyTimeOut: indicates  
   that the preceeding nbLogin has completed ."
  | c res args |
  nbLoginSession ifNil:[
    tsSession ifNil:[
      Error signal:'nbLogin not in progress'.
    ].
    nbCallInProgress := 0 .
    ^ true
  ].
  c := self _calloutAt: 16 name: #'GciTsNbLoginFinished' .
  args := { nbLoginSession . 
            CByteArray gcMalloc: 4 "BoolType *executedSessionInit".
            gciErr } .
  res := c callWith: args . 
  res == 0 ifTrue:[   ^ false "login still in progress"].
  nbCallInProgress := 0 .
  res < 0 ifTrue:[    
    nbLoginSession := nil .
    self _signalError:'NbLogin failed'
  ].
  tsSession := nbLoginSession .
  nbLoginSession := nil .
  (socket := GsSocket fromFileHandle: self _getSocketFd)
     setCloseOnGc: true . "close it if this session GCed before logout"
  self _allocateBuffers .
  gciErr number > 0 ifTrue:[
     self _signalError:'Warning from GciTsNbLogin'.
  ].
  ^ true  "login finished"
]

{ #category : 'Calls' }
GsTsExternalSession >> nbLogout [
  "Does not wait for a result from the gem."
  | c res |
  socket ifNotNil:[:s | s setCloseOnGc: false ].  "prevent duplicate close after libgcits closes it"
  c := self _calloutAt: 10 name: #'GciTsNbLogout' .
  res := c callWith: { tsSession . gciErr } .
  self _clearConnection .

]

{ #category : 'Non Blocking Calls' }
GsTsExternalSession >> nbResult [
  "See GsTsExternalSession>> resolveResult: for documentation of lastResult."
 ^ (lastResult := self resolveResult: self _nbResult)

]

{ #category : 'Non Blocking Calls' }
GsTsExternalSession >> nbSend: aSelector to: rcvrOop withArguments: anArray [ 
 "Must be followed by GsTsExternalSession >> nbResult .
  The receiver should represent a session on the same stone as the current session.
 "
 | c res args nArgs ofs |
 c := self _calloutAt: 11 name: #'GciTsNbPerform' .  
 (args := performArgs) size < (nArgs := anArray size) ifTrue:[
   args := performArgs := CByteArray gcMalloc: 8 * nArgs .
 ].
 ofs := 0 .
 1 to: nArgs do:[:n |
   args uint64At: ofs put: (anArray at: n) asOop . 
   ofs := ofs + 8 .
 ].
 aSelector _isSymbol ifTrue:[
   res := c callWith: { tsSession . rcvrOop .
                        aSelector asOop .  nil . args . nArgs . 
                        1"GCI_PERFORM_FLAG_ENABLE_DEBUG, flags" .
                        0 "envId" . gciErr }
 ] ifFalse:[
   aSelector _isOneByteString ifFalse:[ Error signal:'aSelector neither a Symbol nor String'].
   res := c callWith: { tsSession .  rcvrOop .
                        1"OOP_ILLEGAL".  aSelector . args . nArgs  .
                        1"GCI_PERFORM_FLAG_ENABLE_DEBUG, flags" .
                        0 "envId" . gciErr }
 ].
 res == 1 ifFalse:[
   self _signalError:'GciTsNbPerform failed'.
 ].
 nbCallInProgress := 2 .

]

{ #category : 'Accessing' }
GsTsExternalSession >> parameters [
  ^ parameters
]

{ #category : 'Accessing' }
GsTsExternalSession >> parameters: aGemStoneParameters [
  parameters := aGemStoneParameters
]

{ #category : 'Public' }
GsTsExternalSession >> resolveResult: anOop [ 
  "anOop is the oop of a result from the external session .
  If anOop specifies a special object, return the special object.
  If anOop is the oop of a String or Symbol, return a String or Symbol 
   ( MultiByteStrings not handled smartly yet).
  If anOop is the oop of some other byte format class, return a ByteArray with the 
  contents of the remote object.  
  Otherwise return an Array  { anOop . classOop . size } containing 
  anOop and the class and size information for anOop."

  | implem info buf classOop objSize |
  (anOop bitAnd: 6) ~~ 0 ifTrue:[ 
    ^ Object _objectForOop: anOop   "a special"
  ]. 
  
  buf := objInfoBuffers at: 2 .
  info := self _getObjInfo: anOop buffer: buf size: 1024 .
  implem := info  uint16At: 34 ."_bits"
  implem := implem bitAnd: 3 .
  implem == 3 ifTrue:[ 
    ^ Object _objectForOop: anOop "a special" 
  ].
  classOop := info uint64At: 8 .
  classOop == 0 ifTrue:[
    Error signal:'Read access denied for object ', anOop asString.
  ].
  objSize := info uint64At: 16 .
  implem == 1 ifTrue:[  "byte format" 
    objSize > buf size  ifTrue:[
      buf := CByteArray gcMalloc: objSize .
      objInfoBuffers at: 2 put: buf .
      self _getObjInfo: anOop buffer: buf size: objSize . 
    ]. 
    self releaseOop: anOop .
    classOop == 74753"String asOop" ifTrue: [
      objSize == 0 ifTrue:[ ^ String new ]. "fix 49669"
      ^ buf stringFrom: 0 to: objSize - 1.
    ].
    classOop == 154369"Unicode7 asOop" ifTrue:[  "fix 51160"
      objSize == 0 ifTrue:[ ^ Unicode7 new ]. 
      ^ buf _copyFrom: 0 to: objSize - 1 resKind: Unicode7 .
    ].
    classOop == 154113"Utf8 asOop" ifTrue:[ | unicodeMode |
      unicodeMode := Unicode16 usingUnicodeCompares .
      objSize == 0 ifTrue:[ 
        ^ unicodeMode ifTrue:[ Unicode7 new ] ifFalse:[ String new ]. 
      ].
      ^ buf decodeUTF8from: 0 to: objSize - 1 unicode: unicodeMode .
    ].
    classOop == 110849"Symbol asOop" ifTrue: [
      objSize == 0 ifTrue:[ ^ #'' ].
     ^ Symbol withAll: (buf stringFrom: 0 to: objSize - 1).
    ].
    objSize == 0 ifTrue:[ ^ ByteArray new ].
    ^ buf byteArrayFrom: 0 to: objSize - 1.
  ].
  ^ { anOop . classOop . objSize }.

]

{ #category : 'Public' }
GsTsExternalSession >> resolveResult: anOop toLevel: anInteger [
  "Used by GciError >> signalCompileError.
   Not fully generalized. 
   Returns a special or byte object, or an Array of resolved objects,
   or a CByteArray containing oop of some other object. 
   See also GsTsExternalSession>>resolveResult:."

  | arr classOop buf res |
  arr := self resolveResult: anOop .
  (arr _isArray) ifFalse:[ ^ arr  ].  "a special or a byte object"
  classOop := arr at: 2 .
  (classOop == 66817"Array asOop" and: [0 < anInteger]) ifTrue: [
    | size array |
    buf := objInfoBuffers at: 2 . "already fetched first 1024 bytes"
    size := (arr at: 3) min:( buf size // 8 ) .
    array := Array new: size .
    1 to: size do:[:n | | oop |
      oop := buf uint64At: (n - 1) * 8 .
      array at: n put: (self resolveResult: oop toLevel: anInteger - 1).
    ].
    ^ array.
  ].
  "Not a recognized object"
  res := CByteArray gcMalloc: 8.
  res uint64At: 0 put: anOop.
  ^ res .

]

{ #category : 'Calls' }
GsTsExternalSession >> send: aSelector to: rcvrOop withArguments: anArray [
 | c res args nArgs ofs |
 c := self _calloutAt: 6 name: #'GciTsPerform' .
 nArgs := anArray size .
 (args := performArgs) size < (8 * nArgs) ifTrue:[
   args := performArgs := CByteArray gcMalloc: 8 * nArgs .
 ].
 ofs := 0 .
 1 to: nArgs do:[:n |
   args uint64At: ofs put: (anArray at: n) asOop . 
   ofs := ofs + 8 .
 ].
 aSelector _isSymbol ifTrue:[
   "The receiver must represent a session on the same stone as the current session."
   res := c callWith: { tsSession . rcvrOop .
                        aSelector asOop .  nil . args . nArgs . 
                        1"GCI_PERFORM_FLAG_ENABLE_DEBUG, flags" .
                        0 "envId" . gciErr }
 ] ifFalse:[
   aSelector _isOneByteString ifFalse:[ Error signal:'aSelector neither a Symbol nor String'].
   res := c callWith: { tsSession .  rcvrOop .
                        1"OOP_ILLEGAL".  aSelector . args . nArgs  .
                        1"GCI_PERFORM_FLAG_ENABLE_DEBUG, flags" .
                        0 "envId" . gciErr }
 ].
 res == 1"OOP_ILLEGAL" ifTrue:[
   self _signalError:'GciTsPerform failed'.
 ] .
 ^ ( lastResult := self resolveResult: res ).
]

{ #category : 'Calls' }
GsTsExternalSession >> send: aSelector to: rcvrOop withOops: anArray [
 | c res args nArgs ofs |
 c := self _calloutAt: 6 name: #'GciTsPerform' .
 nArgs := anArray size .
 (args := performArgs) size < (8 * nArgs) ifTrue:[
   args := performArgs := CByteArray gcMalloc: 8 * nArgs .
 ].
 ofs := 0 .
 1 to: nArgs do:[:n |
   args uint64At: ofs put: (anArray at: n) . 
   ofs := ofs + 8 .
 ].
 aSelector _isSymbol ifTrue:[
   "The receiver must represent a session on the same stone as the current session."
   res := c callWith: { tsSession . rcvrOop .
                        aSelector asOop .  nil . args . nArgs . 
                        1"GCI_PERFORM_FLAG_ENABLE_DEBUG, flags" .
                        0 "envId" . gciErr }
 ] ifFalse:[
   aSelector _isOneByteString ifFalse:[ Error signal:'aSelector neither a Symbol nor String'].
   res := c callWith: { tsSession .  rcvrOop .
                        1"OOP_ILLEGAL".  aSelector . args . nArgs  .
                        1"GCI_PERFORM_FLAG_ENABLE_DEBUG, flags" .
                        0 "envId" . gciErr }
 ].
 res == 1"OOP_ILLEGAL" ifTrue:[
   self _signalError:'GciTsPerform failed'.
 ] .
 ^ ( lastResult := self resolveResult: res ).
]

{ #category : 'Calls' }
GsTsExternalSession >> newUtf8String: aUtf8 toUnicode: aBoolean [
  "Returns an oop of the object created in the external session."
  | c res |
  c := self _calloutAt: 20 name: #'GciTsNewUtf8String' .
  res := c callWith: { tsSession . aUtf8 . (aBoolean ifTrue:[ 1 ] ifFalse:[0]) . gciErr }.
  res == 1"OOP_ILLEGAL" ifTrue:[
    self _signalError:'GciTsNewUtf8String failed'.
  ].
  ^ res "anOop" 
]

{ #category : 'Calls' }
GsTsExternalSession >> releaseOops: anArray [
  "result is void , arg is an Array of oops in the remote session"
  | c res args nArgs ofs |
  c := self _calloutAt: 21 name: #'GciTsReleaseObjs' .
  nArgs := anArray size .
  (args := performArgs) size < (8 * nArgs) ifTrue:[
    args := performArgs := CByteArray gcMalloc: 8 * nArgs .
  ].
  ofs := 0 .
  1 to: nArgs do:[:n |
    args uint64At: ofs put: (anArray at: n) . 
    ofs := ofs + 8 .
  ].
  res := c callWith: { tsSession . args . nArgs . gciErr } .
  res == 1"C true" ifFalse:[
    self _signalError:'GciTsReleaseObjs failed'.
  ]
]

{ #category : 'Calls' }
GsTsExternalSession >> releaseOop: anOop [
  | oopsToRelease |
  oopsToRelease := objInfoBuffers at: 3 .
  oopsToRelease add: anOop .
  oopsToRelease size > 100 ifTrue:[
    self releaseOops: oopsToRelease .
    oopsToRelease size: 0 .
  ].
]

{ #category : 'Calls' }
GsTsExternalSession >> softBreak [
  | c res |
  c := self _calloutAt: 17 name: #'GciTsBreak' .
  res := c callWith: { tsSession . 0 . gciErr }.
  res == 1 ifFalse:[
    self _signalError:'GciTsBegin failed'.
  ].
  ^ self
]

{ #category : 'Accessing' }
GsTsExternalSession >> stoneSessionId [
  ^ stoneSessionId ifNil:[ 
    stoneSessionId := self executeString: 'System session' 
  ].

]

{ #category : 'Accessing' }
GsTsExternalSession >> stoneSessionSerial [
  ^ stoneSessionSerial ifNil:[
    GsSession isSolo ifFalse:[ | stoneStartupStr onMyStn |
      stoneStartupStr := self executeString:'System stoneStartupId asString'.
      onMyStn :=  stoneStartupStr = System stoneStartupId asString .
      stoneSessionSerial := onMyStn ifTrue:[ GsSession serialOfSession: stoneSessionId ]
                              ifFalse:[ self executeString:'GsSession currentSession serialNumber'].
    ].
    stoneSessionSerial
  ]
]

{ #category : 'Parameters' }
GsTsExternalSession >> username: aString [
  parameters username: aString
]

{ #category : 'Parameters' }
GsTsExternalSession >> username [
  ^ parameters username
]

{ #category : 'Parameters' }
GsTsExternalSession >> password: aString [
  parameters password: aString copy
]

{ #category : 'Non Blocking Calls' }
GsTsExternalSession >> waitForReadReady [
  "Use the ProcessorScheduler to wait for this session's socket to 
   be ready to read, allowing other GsProcess to run while we are waiting."

  nbCallInProgress <= 1 ifTrue:[ 
    nbLoginSession ifNotNil:[ self _waitForNbLogin . ^ self ].
    Error signal:'call not in progress'.
  ].
  tsSession ifNil:[ Error signal:'lost session state'].
  socket _waitForReadReady
]

{ #category : 'Non Blocking Calls' }
GsTsExternalSession >> isReadReady  [
  "Return true if the session is ready to read a result or error"

  self isCallInProgress ifFalse:[ 
    nbLoginSession ifNotNil:[ ^ false ].
    Error signal:'call not in progress'.
  ].
  ^ socket readWillNotBlock ~~ false .  "return true if socket error pending"
]

{ #category : 'Non Blocking Calls' }
GsTsExternalSession >> waitForReadReadyTimeOut: msToWait [
  "Use the ProcessorScheduler to wait for this session's socket to 
   be ready to read, allowing other GsProcess to run while we are waiting.
 
   The argument msToWait is a SmallInteger  >= 0 , units of milliseconds.
   Returns true if underling socket is ready to read, false if timed out .
   Signals an error if an error other than timeout occurs.
  "
  | status |
  nbCallInProgress <= 1 ifTrue:[ 
     nbLoginSession ifNotNil:[ 
        self _waitForNbLogin: msToWait otherwise: nil . 
       ^ true
     ] .
     ^ Error signal:'call not in progress'. 
  ].
  tsSession ifNil:[ Error signal:'lost session state'].
  status := socket readWillNotBlockWithin: msToWait .
  status ifNil:[
    nbCallInProgress ~~ 2 ifTrue:[ ^ Error signal:'socket error' ].
    self _signalError:'socket error'.
  ].
  ^ status "fixed 49963"
]

{ #category : 'Non Blocking Calls' }
GsTsExternalSession >> waitForResultForSeconds: anInteger otherwise: aBlock [
  | status timeoutMs |
  timeoutMs := anInteger asInteger * 1000 .
  nbCallInProgress <= 1  ifTrue:[ 
    nbLoginSession ifNotNil:[ ^ self _waitForNbLogin: timeoutMs otherwise: aBlock ].
    Error signal:'call not in progress'. 
  ].
  tsSession ifNil:[ Error signal:'lost session state'].
  status := socket readWillNotBlockWithin: timeoutMs .
  status ~~ true ifTrue:[ ^ aBlock value ]. 
  self nbResult
]

{ #category : 'Non Blocking Calls' }
GsTsExternalSession >> waitForResult [
  "See also GsTsExternalSession>> resolveResult: "
  nbLoginSession ifNotNil:[
    ^ self waitForResultForSeconds: 30 otherwise:[ Error signal:'waitForResult timed out'].
  ].
  socket _waitForReadReady .
  self nbResult. 
]

{ #category : 'Non Blocking Calls' }
GsTsExternalSession >> waitForResultForSeconds: anInteger [
  ^ self waitForResultForSeconds: anInteger 
      otherwise:[ | tNow | tNow := DateAndTime now "for debugging" .  
                   Error signal: 'waitForResult Exceeded wait time of ', anInteger printString ].
]

{ #category : 'Private' }
GsTsExternalSession >> _waitForNbLogin [
  ^ self _waitForNbLogin: 30000 otherwise: nil .
]

{ #category : 'Private' }
GsTsExternalSession >> _waitForNbLogin: timeoutMs otherwise: aBlock [
  | timeLeft |            "fix 50203"
  timeLeft := timeoutMs .
  self nbLoginFinished ifFalse:[
     [ true ] whileTrue:[
       Delay waitForMilliseconds: 20 .
       timeLeft := timeLeft - 20 .
       self nbLoginFinished ifTrue:[ ^ self ].
       timeLeft <= 0 ifTrue:[ 
         nbCallInProgress := 0 .
         nbLoginSession := nil .
         ^ aBlock ifNotNil:[ aBlock value ] 
                     ifNil:[ Error signal:'nbLogin timed out']].
     ]
   ] .
   ^ self 
]

{ #category : 'Public' }
GsTsExternalSession >> printOn: aStream [
  aStream
    nextPut: $a ;
    nextPutAll: self class name;
    nextPut: $(  .
  tsSession ifNil:[ aStream nextPutAll: 'nil' ]
     ifNotNil:[ aStream nextPutAll: self stoneSessionId asString .
        tsSession ifNotNil:[ aStream nextPutAll: ' gem process ';
                               nextPutAll: self gemProcessId asString ]].
  aStream nextPut: $)  .
]

{ #category : 'Parameters' }
GsTsExternalSession >> onetimePassword: aString [

   parameters onetimePassword: aString copy
]

{ #category : 'Private' }
GsTsExternalSession class >> _stackReport: contextOop [
 "executed in the external session .
  contextOop is from   aGciErrSType context"
 | aGsProcess rpt |
 aGsProcess := (Object _objectForOop: contextOop) ifNil:[ ^ ' < NO PROCESS FOUND > '].
 rpt := [ aGsProcess stackReportToLevel: 300 withArgsAndTemps: true andMethods: false
        ] on: Error do:[:ex | 'ERROR during stack report ', ex asString ].
 rpt charSize == 1 ifTrue:[ ^ rpt ].
 ^ rpt encodeAsUTF8

]

{ #category : 'Private' }
GsTsExternalSession >> _describe [
  | str host |
  GsSession isSolo ifFalse:[
    [ host := (System descriptionOfSession: stoneSessionId) at: 3.
      host ifNotNil:[ | ofs |
        ofs := host indexOf: $.  .
        ofs ~~ 0 ifTrue:[ host size: ofs - 1 ].  "trim the .gemtalksystems.com"
      ]
    ] on: Error do:[:ex | "ignore"].
  ].
  (str := 'stone session ' copy)
    add: stoneSessionId asString;
    add: ' gem processId '; add: gemProcessId asString ;
    add: ' '; add: parameters printString .
  host ifNotNil:[ str add: ' on host ' ; add: host ].
  ^ str
] 

{ #category : 'Private' }
GsTsExternalSession >> gciErrorClass [
  ^ GciError
]
