"
AwsDataKey is a class that holds a local encryption key. In order
to use the key, it must be decrypted by accessing a Customer Master Key
(CMK) stored in the Amazon cloud.

Instance variables:

cmsKeyId (String) the Amazon Resource Name (ARN) of the CMK.
encryptedDataKey -(String)  the local data key in encrypted form, stored
                   as base 64 text.
keySizeBytes (SmallInteger) - the size of the key in bytes (either 16 or 32).

AwsDataKeys may be in a locked or unlocked state. Locked means the
key has not yet been decrypted by the current session. Unlocked means the
data key has been decrypted and is stored in session memory. AwsDataKeys
may only be used to encrypt or decrypt data in the unlocked state.

If an AwsDataKey is committed to GemStone, it is always stored on disk 
in its locked state. Once an AwsDataKey is unlocked, it remains unlocked
until it is sent the #lock message or the session logs out.

AWS support is only available on platforms that support both the 
AWS software developer toolkit for C++ and GemStone server VMs. 
Currently those platforms include MacOS and Linux. See 
https://github.com/aws/aws-sdk-cpp#aws-sdk-cpp for more information
on the AWS SDK for C++.

AWS request signing uses the system clock to include a timestamp in the
signature. For this reason, AWS methods may fail if the system clock time is
incorrect.
"

Class {
	#name : 'AwsDataKey',
	#superclass : 'AbstractCloudKey',
	#instVars : [
		'cmsKeyId',
		'encryptedDataKey',
		'keySizeBytes'
	],
	#gs_reservedoop : '250881',
	#category : nil
}

{ #category : 'Private' }
AwsDataKey class >> _threeArgClassPrim: opCode with: obj1 with: obj2 with: obj3 [

"
opCode		class method
0			AwsDataKey createKeyUsingAwsCredentials:cmsKeyId:keySizeBytes:
1			AzureDataKey createKeyUsingAzureCredentials:keyVaultUrl:keyName: 
2                       AzureDataKey _changeKeyNameTo:inKeyVault:usingNewCredentials:
"

<primitive: 1128>
^ self _primitiveFailed: #_threeArgClassPrim:with:with: args: { opCode . obj1 . obj2 . obj3 }

]

{ #category : 'Instance Creation' }
AwsDataKey class >> createKeyUsingAwsCredentials: awsCreds cmsKeyId: keyId keySizeBytes: keySize [

"Creates a new instance of the receiver using awsCreds and keyId.
 The new instance is unlocked.  awsCreds must be valid for keyId 
 and keyId must be the ARN of an AWS Customer Master Key (CMK) or a 
 CMK alias. keySize is the size of the new key in bytes and must 
 be either 16 or 32.

 Raises an exception on error."

^ self _threeArgClassPrim: 0 with: awsCreds with: keyId with: keySize

]

{ #category : 'Private' }
AwsDataKey >> _changeCmsKeyIdTo: newCmsKeyId usingNewCredentials: newCreds [

"Updates the receiver to use newCmsKeyId which is accessed with newCreds.
The receiver must first be unlocked using the #unlockWithAwsCredentials: method before this method
can be successfully invoked.

Private. Do not call this method directly. Use the public method #changeCmsKeyIdTo:usingNewCredentials:
to change the data key."

^self _twoArgInstPrim: 0 with: newCmsKeyId with: newCreds

]

{ #category : 'Private' }
AwsDataKey >> _oneArgInstPrim: opCode with: obj [

"
OpCode		Method
-------------------------
0			AwsDataKey unlockWithAwsCredentials:
1			AwsDataKey _copyKeyTo:
"

<primitive: 1126>
^ self _primitiveFailed: #_oneArgInstPrim: args: { opCode . obj }

]

{ #category : 'Private' }
AwsDataKey >> _twoArgInstPrim: opCode with: obj1 with: obj2 [

"
OpCode		Method
-------------------------
0			AwsDataKey _changeCmsKeyIdTo:usingNewCredentials:
1			AwsDataKey encrypt:into:
2			AwsDataKey decrypt:into:

"

<primitive: 1127>
^ self _primitiveFailed: #_twoArgInstPrim: args: { opCode . obj1 . obj2 }

]

{ #category : 'Private' }
AwsDataKey >> _zeroArgInstPrim: opCode [

"
OpCode		Method
-------------------------
0			isLocked
1			lock
"

<primitive: 1125>
^ self _primitiveFailed: #_zeroArgInstPrim: args: { opCode }

]

{ #category : 'Key Rotation' }
AwsDataKey >> changeCmsKeyIdTo: newCmsKeyId usingNewCredentials: newCreds [

"Atomically updates the receiver to use newCmsKeyId which is accessed with newCreds in a single operation.
The receiver must first be unlocked using the #unlockWithAwsCredentials: method before this method
can be successfully invoked.
Fails with an exception if the session has uncommitted changes or if the receiver is locked.
Fails with an exception if the session cannot obtain a write lock on the receiver.
Returns true on success and leaves the receiver in an unlocked state."

| sys |
self assertNotSolo; assertUnlocked ; assertNoUncommittedChanges .
sys := System .
sys beginTransaction.
^ [ sys writeLock: self ;
	addToCommitOrAbortReleaseLocksSet: self .
self _changeCmsKeyIdTo: newCmsKeyId usingNewCredentials: newCreds .
sys commit
] on: Exception do:[:ex| sys abortTransaction. ex pass ]

]

{ #category : 'Accessing' }
AwsDataKey >> cmsKeyId [
	^cmsKeyId

]

{ #category : 'Updating' }
AwsDataKey >> cmsKeyId: newValue [
	cmsKeyId := newValue

]

{ #category : 'Encrypting' }
AwsDataKey >> decrypt: srcByteObj into: destByteObj [

"Uses the receiver to decrypt srcByteObj and stores the resulting decrypted bytes into destByteObj.
The receiver must be unlocked. srcByteObj must be a non-empty byte object containg Base64 text
obtained by calling one of the encrypt methods in this class.
destByteObj must be a mutable byte object. The contents of destByteObj, if any, are
overwritten by this method.

Encryption is performed using the AES-OCB authenticated encryption, which ensures data that has been 
successfully decrypted has not been modified in any way.

Returns the destByteObj on success or raises an exception on error.
"

^self _twoArgInstPrim: 2 with: srcByteObj with: destByteObj

]

{ #category : 'Encrypting' }
AwsDataKey >> encrypt: srcByteObj into: destByteObj [

"Uses the receiver to encrypt srcByteArg and stores the resulting encrypted bytes into destByteObj 
as a Base64 string. The receiver must be unlocked. srcByteArg must be a non-empty byte object.
destByteArg must be a mutable byte object. Any data present in destByteArg will be overwritten.

Encryption is performed using the AES-OCB authenticated encryption, which ensures data that has been 
successfully decrypted has not been modified in any way.

Returns destByteObj on success and raises an exception on error.
"

| result |
result := self _twoArgInstPrim: 1 with: srcByteObj with: destByteObj .
^ result

]

{ #category : 'Encrypting' }
AwsDataKey >> encryptAndErase: srcByteArg into: destByteObj [

"Uses the receiver to encrypt and erase srcByteArg and store the resulting encrypted bytes into 
destByteObj as a Base64 string. The receiver must be unlocked. srcByteArg must be a non-empty byte 
object. destByteArg must be a mutable byte object. Any data present in destByteArg will be overwritten.

Encryption is performed using the AES-OCB authenticated encryption, which ensures data that has been 
successfully decrypted has not been modified in any way.

Returns destByteObj and sets srcByteArg to empty (zero size) upon success.
Raises an exception on error.
"

| result |
result := self _twoArgInstPrim: 1 with: srcByteArg with: destByteObj .
srcByteArg size: 0 .
^ result

]

{ #category : 'Accessing' }
AwsDataKey >> encryptedDataKey [
	^encryptedDataKey

]

{ #category : 'Updating' }
AwsDataKey >> encryptedDataKey: newValue [
	encryptedDataKey := newValue

]

{ #category : 'Testing' }
AwsDataKey >> isLocked [

"Returns a boolean indicating if the receiver is locked.
Unlocked keys may be used to encrypt and decrypt data.
Locked keys must be unlocked using the #unlockWithAwsCredentials: method
before usage."

^ self _zeroArgInstPrim: 0

]

{ #category : 'Testing' }
AwsDataKey >> isUnlocked [

"Returns a boolean indicating if the receiver is unlocked."

^ self isLocked not

]

{ #category : 'Accessing' }
AwsDataKey >> keySizeBits [
	^keySizeBytes * 8

]

{ #category : 'Accessing' }
AwsDataKey >> keySizeBytes [
	^keySizeBytes

]

{ #category : 'Updating' }
AwsDataKey >> keySizeBytes: newValue [
	keySizeBytes := newValue

]

{ #category : 'Locking' }
AwsDataKey >> lock [

"Locks the receiver and securly removes the encryption key from memory.
Returns the receiver."
^ self _zeroArgInstPrim: 1

]

{ #category : 'Locking' }
AwsDataKey >> unlockWithAwsCredentials: awsCreds [

"Attempts to unlock the receiver using awsCreds.
Returns the receiver on success or if the receiver is already unlocked.
Raises an exception on error."

^self _oneArgInstPrim: 0 with: awsCreds

]

{ #category :'Private' }
AwsDataKey >> _copyKeyTo: newAwsDataKey [

"Private. Do not call this method directly unless you know what you are doing.
If the receiver is unlocked, copy the decrypted data key to newObject.
Copies nothing if the receiver is locked. It is an error if newAwsDataKey
already has data key and and is unlocked. 

Returns newAwsDataKey."

^ self _oneArgInstPrim: 1 with: newAwsDataKey

]

{ #category :'Copying' }
AwsDataKey >> copy [

"Answer a new object which is a deep copy of the receiver. The lock state of the receiver is preserved
when copied, i.e.: if the receiver is unlocked the resulting copy will also
be unlocked."

| result |
result := self class basicNew.
result cmsKeyId: self cmsKeyId copy ;
	encryptedDataKey: self encryptedDataKey copy ;
	keySizeBytes: self keySizeBytes.
^self _copyKeyTo: result

]

{ #category :'Comparing' }
AwsDataKey >> = anotherKey [

(self == anotherKey) ifTrue:[ ^ true ].
(self class == anotherKey class) ifFalse:[ ^ false ].
(self cmsKeyId = anotherKey cmsKeyId) ifFalse: [^ false].
(self encryptedDataKey = anotherKey encryptedDataKey) ifFalse: [^false].
^ true

]

{ #category :'Hashing' }
AwsDataKey >> hash [

"Returns an Integer hash code for the receiver."

^ self cmsKeyId hash bitXor: self encryptedDataKey hash

]

{ #category : 'Exceptions' }
AwsDataKey class >> errorClass [
  ^ AwsError
]
