Extension { #name : 'Behavior' }

{ #category : 'Updating the Method Dictionary' }
Behavior >> __basicRemoveSelector: aSymbol environmentId: envId [
"returns  true if a method found to remove"
| md pmd removed tmeth pmeth |
self _validatePrivilege ifFalse:[ ^ nil ].
md := self transientMethodDictForEnv: envId .
removed := false .
md ifNotNil:[
  tmeth := md removeKey: aSymbol otherwise: nil .
].
pmd := self persistentMethodDictForEnv: envId .
pmd ifNotNil:[
  pmeth := pmd removeKey: aSymbol otherwise: nil .
].
"Now refresh method lookup caches to account for removal of the selector
 and delete any breakpoints in the removed method(s) ."
tmeth ifNotNil:[
  self _refreshLookupCache: aSymbol oldMethod: tmeth env: envId .
  removed := true
].
pmeth ifNotNil:[
  removed := true .
  self _codeChangedForEnv: envId .
  self _refreshLookupCache: aSymbol oldMethod: pmeth env: envId
].
^ removed

]

{ #category : 'Modifying Classes' }
Behavior >> _addClassVar: aSymbol value: aVal [

"Adds the class variable with name aSymbol to the receiver's
 class variables dictionary."
  | dict |
  self _validatePrivilege ifTrue:[
    (dict := classVars) ifNil:[ dict := self _createClassVarsDict ].
    (dict associationAt: aSymbol otherwise: nil) ifNotNil:[ :assoc |
       assoc _value: aVal
    ] ifNil:[ | assocClass |
      assocClass := SessionTemps current at:#ClassVariablesAssociationClass otherwise: SymbolAssociation.

      dict addAssociation:( assocClass newWithKey: aSymbol value: aVal)  .
    ]
  ]

]

{ #category : 'Updating Categories' }
Behavior >> _announceReorganized [
  ^ self

]

{ #category : 'Updating the Method Dictionary' }
Behavior >> _basicRemoveSelector: aSymbol environmentId: envId [

"Private."
| oldMeth removedFromSelf |
self _validatePrivilege ifFalse:[ ^ nil ].
removedFromSelf := false.
envId == 0 ifTrue:[
  oldMeth := GsPackagePolicy currentOrNil ifNotNil:[:pp| pp removeMethodAt: aSymbol for: self].
  oldMeth ifNil: [
    (self persistentMethodDictForEnv: 0) ifNotNil:[:dict |
       oldMeth := dict removeKey: aSymbol . "runtime error here if key not found"
    ].
    self setStamp: nil forMethod: aSymbol.
    removedFromSelf := true.
  ].
  self _codeChangedForEnv: envId .
  self _refreshLookupCache: aSymbol oldMethod: oldMeth env: envId .
] ifFalse:[
  removedFromSelf := self __basicRemoveSelector: aSymbol environmentId: envId .
].
^removedFromSelf

]

{ #category : 'Modifying Classes' }
Behavior >> _compileMethodsFromPreviousClassVersionWithFailedDictionary: aSymbolKeyValueDictionary [

 "Assumes the receiver is the most recent (current) version of a class and 
 that a previous version exists in the receiver's classHistory.
 For each method in the previous version, 
 compile the method in the receiver using the current users symbol list.
 If the method fails to compile, add an entry to aSymbolKeyValueDictionary 
 where the key is the class name and the value is an Array of Associations
 The Associations have key of method selector and value of source string of the failed method.
 Returns the number of methods which failed to compile."
 
 | failed classHis oldClass sz sl |
 classHis := self classHistory.
 (self isMeta not and:[ classHis last ~~ self])
   ifTrue:[ self error: 'receiver is not the most recent version of this class' ].
 sz := classHis size.
 sz < 2 ifTrue:[ self error: 'no previous versions of this class were found' ].
 oldClass := classHis at: (sz - 1).
 self isMeta ifTrue:[ oldClass := oldClass class ].
 failed := Array new.
 sl := System myUserProfile symbolList.
 oldClass env: 0 unifiedCategoriesDo:[ :categName :selectorList |
   selectorList do: [ :aSel| | oldMeth |
     [ oldMeth := oldClass compiledMethodAt: aSel environmentId: 0 .
       self compileMethod: oldMeth sourceString dictionaries: sl
            category: categName environmentId: 0
     ] onException: CompileError do:[:ex | 
        failed add: (Association newWithKey: aSel value: oldMeth sourceString) 
    ].
  ].
 ].
 failed size > 0 ifTrue:[ aSymbolKeyValueDictionary at: self name asSymbol put: failed ].
 ^ failed size
]

{ #category : 'Updating the Method Dictionary' }
Behavior >> _compileTransientMethod: sourceString dictionaries: aSymbolList environmentId: environmentId [
  "This method compiles some source code for the receiver and installs the method into
   the transient method dictionary.
   The first argument, sourceString, is the string of source code to be compiled.

   The second argument is a SymbolList to be used in parsing, along with the list of all
   class variables and pool dictionaries for the receiver and all of its
   superclasses.

   sourceString must be a kind of String or MultiByteString.  Instances of
   JapaneseString are not supported as source strings.  String literals
   ('abc') are generated as instances of the class of sourceString,
   unless sourceString is a Symbol, in which case 'abc' produces a String.
   If sourceString is a DoubleByteSymbol, 'abc' produces a DoubleByteString.

   anEnvironmentId must be a SmallInteger >= 0 and <= 16rFFFF.
   0 denotes the base Smalltalk image.  1 is reserved for use by Ruby.

   If a method with the same selector does not already exist an error is signalled.

   If there are no errors, this adds the resulting compiled method to the
   receiver's method dictionary and returns that method,
   otherwise signals a CompileError .
   A CompileWarning may be signaled, after adding the new method
   to a receiver's method dictionary."

  <protected>
  | meth tmd categ preCompiledMethod |
  self objectSecurityPolicy
    ifNotNil: [:policy |
      policy currentUserCanWrite
        ifFalse: [ self error: 'You must be able to write to the receiver to add transient session methods' ] ].
   preCompiledMethod := self
        _primitiveCompileMethod: sourceString
        symbolList: aSymbolList
        category: #'xxyzzy'
        oldLitVars: nil
        intoMethodDict: GsMethodDictionary new
        intoCategories: GsMethodDictionary new
        environmentId: environmentId. "pre-compile the method source to extract selector"
  self
    _checkCompileResult: preCompiledMethod
    source: sourceString
    suppressCompileWarning: true.
  preCompiledMethod class == GsNMethod
    ifTrue: [
      (self categoryOfSelector: preCompiledMethod selector)
        ifNil: [
          self
            error:
              'Transient method: ' , preCompiledMethod selector asString
                , ' must override an existing method.' ]
        ifNotNil: [ :existingCategory |
          "Use the category of the existing persistent method"
          categ := existingCategory ] ].
  tmd := self transientMethodDictForEnv: environmentId.
  tmd ifNil: [
    tmd := GsMethodDictionary new.
    self _transientSessionMethodBehaviorsCache add: self.
    self transientMethodDictForEnv: environmentId put: tmd.
  ].
  meth := self
    compileMethod: sourceString
    dictionaries: aSymbolList
    category: categ
    intoMethodDict: tmd
    intoCategories: nil
    intoPragmas: nil
    environmentId: environmentId.
  self _clearLookupCaches: environmentId.
  ^ meth

]

{ #category : 'Fileout' }
Behavior >> _fileOutEnvSuperclass: envId on: stream [
  | sup |
  sup := self superclassForEnv: envId .
  sup ~~ self superClass ifTrue:[
    stream nextPutAll: 'doit'; lf ;
       _fileOutAll: self thisClass name ;
       nextPutAll: ' superclassForEnv: '; nextPutAll: envId asString ;
       nextPutAll: ' put: ' ;  _fileOutAll: sup name ; nextPutAll:' . true ' ; lf ;
       nextPut: $% ; lf .
  ].

]

{ #category : 'Fileout' }
Behavior >> _fileOutEnvSuperclassesOn: stream [
  | max |
  max := self _maxSuperclassEnv .
  max downTo: 1 do:[:n |
    self _fileOutEnvSuperclass: n on: stream
  ]

]

{ #category : 'Fileout' }
Behavior >> _fileoutHeaderOn: stream [ 
  ^ self _fileoutHeaderOn: stream environmentId: 0
]

{ #category : 'Fileout' }
Behavior >> _fileoutHeaderOn: stream environmentId: envId [
  | prevEnv |
  stream isEmpty ifTrue:[
    (stream isKindOf: GsFile) ifTrue:[
      stream nextPutAll: 'fileformat utf8' ; lf .
    ].
    "Gs64 v3.3, no SET SOURCESTRINGCLASS directives in fileouts."
  ].
  prevEnv := stream dynamicInstVarAt: #environmentId .
  prevEnv ~~ envId ifTrue:[ 
    stream nextPutAll:'set compile_env: ' ; nextPutAll: envId asString ; lf .
    stream dynamicInstVarAt: #environmentId put: envId .
  ]
]

{ #category : 'Private' }
Behavior >> _fileOutMethod: selector environmentId: envId [

"Returns a UTF8 encoded string with the given method's category and source in Topaz
 Filein format."

| strm |
strm := self _streamForFileout .
self fileOutMethod: selector environmentId: envId on: strm .
^ strm contents .

]

{ #category : 'Updating Categories' }
Behavior >> _needsAnnouncement [
 "Result controls sends of  _announceMethodMoved:oldCategory: .
  This implementation replaced when Seaside or other GUI tools installed. "

  ^ false

]

{ #category : 'Pragmas' }
Behavior >> _pragmasForMethod: aGsNMethod [
  "Get the legacy Pragmas for a method compiled before 3.5.
   This code shared by Classes and metaclasses,
   parameterized by pragmaDictName."

  | pragmas methodPragmaDict eDict selSym |
  selSym := aGsNMethod selector .
  pragmas := GsPackagePolicy currentOrNil ifNotNil:[:pp | pp  pragmasForMethod: selSym in: self].
  pragmas ifNotNil: [ ^pragmas ].

  (eDict := self extraDict) ifNil:[ ^#() ].
  (methodPragmaDict := eDict at: self pragmaDictName otherwise: nil) ifNil: [ ^#() ].
  ^ methodPragmaDict at: selSym otherwise: #()

]

{ #category : 'Modifying Classes' }
Behavior >> _removeAllSubclassCode [

"Dereference the code objects of all environment 0 GsMethods of the receiver,
 to force recompilation of those methods."

| theSubclasses aDict envId |
self _validatePrivilege ifFalse:[ ^ nil ].
envId := 0 .
theSubclasses := self _subclasses .
theSubclasses ifNotNil:[
  theSubclasses do:[:x| x _removeAllSubclassCode ] .
  ].
GsPackagePolicy currentOrNil ifNotNil:[:pp | pp removeAllSubclassCodeFor: self "envId 0 only"].
aDict := self persistentMethodDictForEnv: envId .
aDict ifNotNil:[
  aDict keysAndValuesDo:[ :aKey :aMethod |
    aDict at: aKey put: ((aDict at: aKey) _copyToForceRecompilation).
  ].
  self _codeChangedForEnv: envId  .
].
self _refreshClassCache: true .
^ self

]

{ #category : 'Private' }
Behavior >> _streamForFileout [
  | cls |
  cls := StringConfiguration == String ifTrue:[ String ] ifFalse:[ Unicode7 ].
  ^ AppendStream on: cls new .
]

{ #category : 'Fileout' }
Behavior >> _topazFileOutCategory: catName header: headerStr asUtf8: utf8Bool [

^ self _topazFileOutCategory: catName header: headerStr asUtf8: utf8Bool env: 0

]

{ #category : 'Fileout' }
Behavior >> _topazFileOutCategory: catName header: headerStr asUtf8: utf8Bool env: envId [
| strm |
strm := AppendStream on: String new  .
strm dynamicInstVarAt: #utf8Bool put: utf8Bool .
strm nextPutAll: headerStr .
self fileOutCategory: catName on: strm environmentId: envId .
^ utf8Bool ifTrue:[ strm contents encodeAsUTF8 ] ifFalse:[ strm contents ].

]

{ #category : 'Fileout' }
Behavior >> _topazFileoutClass: headerStr asUtf8: utf8Bool [

^ self _topazFileoutClass: headerStr asUtf8: utf8Bool env: 0

]

{ #category : 'Fileout' }
Behavior >> _topazFileoutClass: headerStr asUtf8: utf8Bool env: envId [
| strm |
strm := AppendStream on: String new .
strm dynamicInstVarAt: #utf8Bool put: utf8Bool .
strm nextPutAll: headerStr .
self fileOutClassOn: strm environmentId: envId .
^ utf8Bool ifTrue:[ strm contents encodeAsUTF8 ] ifFalse:[ strm contents ].

]

{ #category : 'Fileout' }
Behavior >> _topazFileoutMethod: selector header: headerStr asUtf8: utf8Bool [

^ self _topazFileoutMethod: selector header: headerStr asUtf8: utf8Bool env: 0

]

{ #category : 'Fileout' }
Behavior >> _topazFileoutMethod: selector header: headerStr asUtf8: utf8Bool env: envId [

| strm |
strm := AppendStream on: String new .
strm nextPutAll: headerStr .
strm dynamicInstVarAt: #utf8Bool put: utf8Bool .
self fileOutMethod: selector environmentId: envId on: strm .
^ utf8Bool ifTrue:[ strm contents encodeAsUTF8 ] ifFalse:[ strm contents ].

]

{ #category : 'Updating the Method Dictionary' }
Behavior >> _transientSessionMethodBehaviorsCache [
  | key tmps |
  tmps := SessionTemps current .
  (tmps at: (key := #TransientSessionMethod_Behaviors) otherwise: nil)
     ifNotNil:[:behaviors | ^ behaviors ]
     ifNil:[ 
       ^ tmps at: key put: IdentitySet new _setNoStubbing .
     ].

]

{ #category : 'Updating Categories' }
Behavior >> addCategory: aString [
  ^ self addCategory: aString environmentId: 0

]

{ #category : 'Updating Categories' }
Behavior >> addCategory: aString environmentId: envId [

"Adds aString as a method category for the receiver.  If aString is already a
 method category, generates an error.
 Returns the newly added category (a SymbolSet)."

| aSymbol canWrite res policy |
self _validatePrivilege ifFalse:[ ^ nil ].
aSymbol := aString asSymbol .
(self includesCategory: aSymbol environmentId: envId ) ifTrue:[
  ^ self _error: #classErrMethCatExists args: { aString . self }
].
canWrite := self canWriteMethods .
policy := GsPackagePolicy currentOrNil .
(canWrite or:[ envId ~~ 0 or: [ policy == nil ]]) ifTrue:[ | aSet catDict |
  catDict :=  self _baseCategorysForStore: envId .
  catDict at: aSymbol put: ( aSet := SymbolSet new ) .
  res := aSet
] ifFalse: [
  res := policy addCategory: aSymbol for: self
].
self _announceReorganized .
^ res

]

{ #category : 'Method Timestamps' }
Behavior >> authorInitials [
    ^ GsPackagePolicy authorInitials

]

{ #category : 'Testing' }
Behavior >> canWriteMethods [

  ^ self canWriteMethodsEnv: 0

]

{ #category : 'Testing' }
Behavior >> canWriteMethodsEnv: envId [
  | dict ok |
  self canBeWritten ifFalse:[ ^ false ]. "fix 49807"
  ok := true . 
  dict := self persistentMethodDictForEnv: envId .
  dict ifNil:[ dict := methDicts ].
  dict ifNotNil:[  ok := dict canBeWritten ] .
  ok ifTrue:[ | cats |
    cats := self _baseCategorys: envId .
    cats ifNil:[ cats := categorys ] .
    cats ifNotNil:[ ok := cats canBeWritten ].
  ].
  ^ ok
]

{ #category : 'Method Timestamps' }
Behavior >> changeStamp [
  "Answer a string to be pasted into source code to mark who changed it and when."
  ^ self authorInitials , ' ' , (Date today asStringUsingFormat: #(2 1 3 $/ 1 1 $: false )), ' ',
    (Time now asString copyFrom: 1 to: 5)

]

{ #category : 'Class Timestamps' }
Behavior >> commentStamp [

  "Return the comment timeStamp.
   Both a Class and its metaclass share a single comment timeStamp"
  | eDict |
  (eDict := self extraDict) ifNil:[ ^ nil ].
  ^ eDict at: #GSClassCommentStamp otherwise: nil

]

{ #category : 'Class Timestamps' }
Behavior >> commentStamp: aStamp [
  "Set the receiver's comment timeStamp to aStamp.
   Both a Class and its metaclass share a single comment stamp"

  | eDict |
  (eDict := self extraDictForStore) ifNotNil:[
    eDict at: #GSClassCommentStamp put: aStamp
  ].

]

{ #category : 'Updating the Method Dictionary' }
Behavior >> compileTransientMethod: sourceString dictionaries: aSymbolList environmentId: environmentId [
  "This method compiles some source code for the receiver and installs the method into
   the transient method dictionary."

  <primitive: 2001>  "enter protected mode"
  | prot |
  self _validatePrivilege
    ifFalse: [ ^nil].
  [
    prot := System _protectedMode.
    ^ self _compileTransientMethod: sourceString dictionaries: aSymbolList environmentId: environmentId  ]
    ensure: [ prot _leaveProtectedMode ]

]

{ #category : 'Pragmas' }
Behavior >> createPragmaFrom: aPragmaArray for: aGsMethod [
  | pragma args pragmas keyword |
  aPragmaArray isEmpty
    ifTrue: [ ^ nil ].
  pragmas := Array new.
  1 to: aPragmaArray size by: 2 do: [ :i |
    keyword := aPragmaArray at: i.
    args := aPragmaArray at: i + 1.
    args == nil
      ifTrue: [ args := #() ].
    pragma := Pragma keyword: keyword arguments: args.
    pragma setMethod: aGsMethod.
    pragmas add: pragma ].
  ^ pragmas

]

{ #category : 'Fileout' }
Behavior >> fileOutCategories [

"Returns a String or MultiByteString with all the receiver's methods in
 Topaz Filein format."

| strm |
strm := self _streamForFileout .
self fileOutCategoriesOn: strm .
^ strm contents .

]

{ #category : 'Fileout' }
Behavior >> fileOutCategoriesOn: stream [

"Writes the receiver's categories and methods onto the given stream
 in Topaz filein format."

| lf cls nm |

cls := self thisClass.
lf := Character lf.
stream nextPut: lf.
nm := cls name.
self fileOutMethodRemovalOn: stream name: nm.
self fileOutPreMethodsOn: stream.

stream nextPutAll: '! ------------------- Class methods for '; _fileOutAll: nm; nextPut: lf.
cls class sortedCategoryNames do: [:category |
  cls class fileOutCategory: category on: stream
].

stream nextPutAll: '! ------------------- Instance methods for '; _fileOutAll: nm; nextPut: lf.
cls sortedCategoryNames do: [:category |
  cls fileOutCategory: category on: stream
].

self fileOutPostMethodsOn: stream.

^stream

]

{ #category : 'Fileout' }
Behavior >> fileOutCategory: catName [

"Returns a String or MultiByteString containing the methods of the given category in
 Topaz Filein format."

| strm |
strm := self _streamForFileout .
self fileOutCategory: catName on: strm .
^ strm contents .

]

{ #category : 'Fileout' }
Behavior >> fileOutCategory: catName on: stream [

^ self fileOutCategory: catName on: stream environmentId: 0

]

{ #category : 'Fileout' }
Behavior >> fileOutCategory: catName on: stream environmentId: envId [

"Files out the given category on the given stream."

| sels |

self _fileoutHeaderOn: stream environmentId: envId .
sels := self sortedSelectorsIn: catName environmentId: envId .
sels do: [:selector |
  self fileOutMethod: selector environmentId: envId on: stream
]

]

{ #category : 'Fileout' }
Behavior >> fileOutClass [

"Returns a String or MultiByteString with the receiver's class definition
 and all the receiver's methods in Topaz Filein format."

| strm |
strm := self _streamForFileout .
self fileOutClassOn: strm environmentId: 0 .
^ strm contents .

]

{ #category : 'Fileout' }
Behavior >> fileOutClassByCategoryOn: stream [

"Writes the receiver's definition and methods onto the given stream in
 filein format."

self _fileoutHeaderOn: stream  .
self fileOutClassDefinitionOn: stream .
self fileOutCategoriesOn: stream.

]

{ #category : 'Fileout' }
Behavior >> fileOutClassCategoryOn: stream [

"Writes out class category, if there is one for this class."

self _fileoutHeaderOn: stream  .
(self _classCategory) ifNotNil:[ :cat |
  stream nextPutAll: 'expectvalue /Class'; lf ;
         nextPutAll: 'doit'; lf ;
         _fileOutAll: (self name asString);
         nextPutAll: ' category: ';
         _fileOutAll: cat quoted ; lf ;
         nextPut: $% ; lf .
].

]

{ #category : 'Fileout' }
Behavior >> fileOutClassDefinitionOn: stream [

^ self fileOutClassDefinitionOn: stream environmentId: 0

]

{ #category : 'Fileout' }
Behavior >> fileOutClassDefinitionOn: stream environmentId: envId [

"Writes the receiver's preclass, class definition, and comment onto
 the given stream in filein format."

self _fileoutHeaderOn: stream environmentId: envId .
envId == 0 ifTrue:[
  stream    nextPutAll: '! ------------------- Class definition for ' ;
       _fileOutAll: self thisClass name; lf .
  self fileOutPreClassOn: stream .
  stream nextPutAll: 'expectvalue /Class'; lf ;
    nextPutAll: 'doit'; lf ;
    _fileOutAll: self definition; lf ;
    nextPut: $% ; lf .
  self fileOutCommentOn: stream .
  self fileOutClassCategoryOn: stream .
  self _fileOutEnvSuperclassesOn: stream .
] ifFalse:[
  self _fileOutEnvSuperclass: envId on: stream .
].
]

{ #category : 'Fileout' }
Behavior >> fileOutClassOn: stream [

"Writes the receiver's definition and methods onto the given stream in
 filein format."

self _fileoutHeaderOn: stream  .
self fileOutClassDefinitionOn: stream .
self fileOutMethodsOn: stream .
self fileOutTraitsOn: stream .  "Traits are filed out after methods, to allow for overrides of Trait methods"
]

{ #category : 'Fileout' }
Behavior >> fileOutClassOn: stream environmentId: envId [

"Writes the receiver's definition and methods onto the given stream in
 filein format."

self _fileoutHeaderOn: stream environmentId: envId .
self fileOutClassDefinitionOn: stream environmentId: envId .
self fileOutMethodsOn: stream environmentId: envId .
envId = 0
  ifTrue: [
    "Traits are filed out after methods, to allow for overrides of Trait methods ... in env 0" 
    self fileOutTraitsOn: stream ] .
]

{ #category : 'Fileout' }
Behavior >> fileOutCommentOn: stream [

"Writes code to create class comment onto the given stream in
 filein format."

self _fileoutHeaderOn: stream  .
self commentForFileout ifNotNil:[ :cmt |
  stream nextPutAll: 'expectvalue /Class'; lf ;
    nextPutAll: 'doit'; lf ;
    _fileOutAll: self thisClass name; nextPutAll: ' comment: ' ; lf ;
    _fileOutAll: cmt printString; lf ;
    nextPut:    $%;  lf
].

]

{ #category : 'Fileout' }
Behavior >> fileOutMethod: selector [
  ^ self fileOutMethod: selector environmentId: 0

]

{ #category : 'Fileout' }
Behavior >> fileOutMethod: selector environmentId: envId [

"Returns a String or MultiByteString with the given method's category and
 source in Topaz Filein format."

| strm |
strm := self _streamForFileout .
self fileOutMethod: selector environmentId: envId on: strm .
^ strm contents .

]

{ #category : 'Fileout' }
Behavior >> fileOutMethod: selector environmentId: envId on: stream [
  "Writes the given method's source to the given stream in Topaz Filein format. 
     Methods defined by a Trait are not included in the fileout."
  | lf cat src meth |
  envId < 0 ifTrue:[ ArgumentError signal:'envId must be >= 0' ].
  self _fileoutHeaderOn: stream environmentId: envId .
  meth := self compiledMethodAt: selector environmentId: envId.
  meth isFromTrait 
    ifTrue: [ 
      "do not fileout trait methods ... they should be filed out with trait" 
       ^ self ].
  src := meth sourceString .
  (src class ~~ Unicode7) ifTrue:[
    (stream dynamicInstVarAt: #utf8Bool) == false ifTrue:[
       (Unicode7 _withAll: src) ifNotNil:[ :s | "source ok" ]
		   ifNil:[ Error signal: 'source for ' , selector printString,
	     ' contains codePoint > 127  when non-Utf8 fileout requested' ]
    ].
  ].
  lf := Character lf.
  cat := self categoryOfSelector: selector environmentId: envId.
  cat ifNil:[
    ^ self _error: #rtErrKeyNotFound args: { selector }
  ].
  stream nextPutAll: 'category: '''; _fileOutAll: cat; nextPut: $'; nextPut: lf .
  stream
    nextPutAll: (self isMeta ifTrue: [ 'classmethod: ' ]
		ifFalse: [ 'method: ' ]) ;
    _fileOutAll: self thisClass name; nextPut: lf .
  stream _fileOutAll: src.
  src last == lf ifFalse: [
    stream nextPut: lf
  ].
  stream nextPut:  $% ; nextPut: lf .
]

{ #category : 'Fileout' }
Behavior >> fileOutMethod: selector on: stream [
 ^ self fileOutMethod: selector environmentId: 0 on: stream

]

{ #category : 'Fileout' }
Behavior >> fileOutMethodRemovalOn: stream name: clsname [

"Writes code to remove all the receiver's methods onto the given stream
 in filein format."

^ self fileOutMethodRemovalOn: stream name: clsname environmentId: 0 .

]

{ #category : 'Fileout' }
Behavior >> fileOutMethodRemovalOn: stream name: clsname environmentId: envId [

 "Writes code to remove all the receiver's methods onto the given stream
  in filein format."

 self _fileoutHeaderOn: stream  environmentId: envId .
 stream nextPutAll: '! ------------------- Remove existing behavior from '; _fileOutAll: clsname; lf ;
   nextPutAll: 'removeallmethods '; _fileOutAll: clsname; lf ;
   nextPutAll: 'removeallclassmethods '; _fileOutAll: clsname; lf .
]

{ #category : 'Fileout' }
Behavior >> fileOutMethods [

"Returns a String or MultiByteString with all the receiver's methods
 in Topaz Filein format. If Traits are used, you should also use fileOutTraits
 to be filed in after methods filed in."

| strm |
strm := self _streamForFileout .
self fileOutMethodsOn: strm .
^ strm contents .
]

{ #category : 'Fileout' }
Behavior >> fileOutMethodsOn: stream [
  ^ self fileOutMethodsOn: stream environmentId: 0 .

]

{ #category : 'Fileout' }
Behavior >> fileOutMethodsOn: stream environmentId: envId [

"File out this class's methods, but sort the selectors alphabetically."

| cls sels nm |
cls := self thisClass.
nm := cls name.
self fileOutMethodRemovalOn: stream name: nm environmentId: envId .
self fileOutPreMethodsOn: stream environmentId: envId .

stream nextPutAll: '! ------------------- Class methods for '; _fileOutAll: nm; lf.
sels := SortedCollection withAll: (cls class selectorsForEnvironment: envId ).
1 to: sels size do: [:i |
  cls class fileOutMethod: (sels at: i)  environmentId: 0 - envId on: stream .
].
stream nextPutAll: '! ------------------- Instance methods for '; _fileOutAll: nm; lf.
sels := SortedCollection withAll: (cls selectorsForEnvironment: envId ).
1 to: sels size do: [:i |
  cls fileOutMethod: (sels at: i) environmentId: 0 - envId on: stream .
].
self fileOutPostMethodsOn: stream.
^stream

]

{ #category : 'Fileout' }
Behavior >> fileOutPostMethodsOn: stream [

 ^ self fileOutPostMethodsOn: stream environmentId: 0

]

{ #category : 'Fileout' }
Behavior >> fileOutPostMethodsOn: stream environmentId: envId [

"This method gives classes an opportunity to file out information not
 normally emitted by the predefined fileout methods.  Classes may override
 this method (as a class method, of course) to add extra code to the
 output stream.  Emitted code should be in Topaz filein format.  It will
 be placed after method creation."


]

{ #category : 'Fileout' }
Behavior >> fileOutPreClassOn: stream [

"This method gives classes an opportunity to file out information not
 normally emitted by the predefined fileout methods.  Classes may override
 this method (as a class method, of course) to add extra code to the
 output stream.  Emitted code should be in Topaz filein format.  It will
 be placed before any other fileout information for the class."


]

{ #category : 'Fileout' }
Behavior >> fileOutPreMethodsOn: stream [

^ self fileOutPreMethodsOn: stream environmentId: 0

]

{ #category : 'Fileout' }
Behavior >> fileOutPreMethodsOn: stream environmentId: envId [

"This method gives classes an opportunity to file out information not
 normally emitted by the predefined fileout methods.  Classes may override
 this method (as a class method, of course) to add extra code to the
 output stream.  Emitted code should be in Topaz filein format.  It will
 be placed after existing method removal and before method creation."


]

{ #category : 'Fileout' }
Behavior >> fileOutTraits [

"Returns a String or MultiByteString with all the receiver's trait specifications (traits and classTraits)
 in Topaz Filein format. Result should be filed in after filing in methods."

| strm |
strm := self _streamForFileout .
self fileOutTraitsOn: strm .
^ strm contents .
]

{ #category : 'Fileout' }
Behavior >> fileOutTraitsOn: stream [
	"Writes code to create trait registrations onto the given stream in filein format."

	self _fileoutHeaderOn: stream.
	(self traits isEmpty and: [ self classTraits isEmpty ])
		ifTrue: [ ^ self ].
	stream
		nextPutAll: '! ------------------- Traits for ';
		_fileOutAll: self thisClass name;
		lf.
	stream
		nextPutAll: 'doit';
		lf.
	self traits isEmpty
		ifFalse: [ 
			stream
				_fileOutAll: self thisClass name;
				nextPutAll: ' addTrait: ' , self traits any name , '.';
				lf ].
	self classTraits isEmpty
		ifFalse: [ 
			stream
				_fileOutAll: self thisClass name;
				nextPutAll: ' addClassTrait: ' , self classTraits any name , ' classTrait.';
				lf ].
	stream
		nextPutAll: 'true.';
		lf;
		nextPut: $%;
		lf
]

{ #category : 'Accessing' }
Behavior >> indexManager [

^IndexManager current

]

{ #category : 'Method Timestamps' }
Behavior >> methodStampDictName [
  ^ #GSMethodStampDict

]

{ #category : 'Updating Categories' }
Behavior >> moveMethod: aSelector toCategory: categoryName [
  ^ self _rwMoveMethod: aSelector toCategory: categoryName

]

{ #category : 'Updating Categories' }
Behavior >> moveMethod: aSelector toCategory: categoryName environmentId: envId [
	"Moves the method aSelector (a String) from its current category to the
 specified category (also a String).  If either aSelector or categoryName is
 not in the receiver's method dictionary, or if aSelector is already in
 categoryName, generates an error..
 This method does not account for selectors inherited from superclasses."

	| selSym catSym oldCat policy method |
	self _validatePrivilege ifFalse: [^nil].
	(self includesCategory: categoryName environmentId: envId)
		ifFalse: [^self _categoryNotFound: categoryName].
	selSym := aSelector asSymbol.
	policy := GsPackagePolicy currentOrNil.
	method := self
				compiledMethodAt: selSym
				environmentId: envId
				otherwise: nil
				usePackages: oldCat notNil.
	method ifNil: [^self _error: #classErrSelectorNotFound args: {aSelector}].
	oldCat := self categoryOfSelector: aSelector environmentId: envId.
	catSym := categoryName asSymbol.
	(envId ~~ 0 or: [self canWriteMethods or: [policy == nil]])
		ifTrue:
			[| catDict |
			catDict := self _baseCategorysForStore: envId.
			oldCat ifNotNil: [(catDict at: oldCat) remove: selSym].
			(catDict at: catSym) add: selSym]
		ifFalse: [policy moveSelector: selSym toCategory: catSym for: self].
	self _needsAnnouncement
		ifTrue: [self _announceMethodMoved: method oldCategory: oldCat]

]

{ #category : 'Fileout' }
Behavior >> nameForFileout [

"Deprecated. Returns the name to be used for this class for fileout."

| arr sz |
self deprecated: 'Behavior>>nameForFileout deprecated v3.2.'.

arr := GsCurrentSession currentSession symbolList dictionariesAndSymbolsOf: self.
1 to: (sz := arr size) do:[:j | | nm |
  (nm := (arr at: j) at: 2)  == self name ifTrue:[ ^ nm ].
].
sz == 0 ifTrue:[ ^ self thisClass name ].
Error signal:'during fileout, class name does not match name in dictionary(s)'

]

{ #category : 'Pragmas' }
Behavior >> pragmaDictName [

  ^ #GSMethodPragmaDict

]

{ #category : 'Pragmas' }
Behavior >> pragmasForMethod: selector [
  "Get the Pragmas for the method specified by  selector."

  | pragmas selSym meth |
  selSym := Symbol _existingWithAll: selector  .
  selSym ifNil:[ ^ #() ] .
  meth := self compiledMethodAt: selector otherwise: nil .
  meth ifNotNil:[
    pragmas := meth pragmas.
    pragmas ifNotNil:[ ^ pragmas ].
  ].
  ^ self _pragmasForMethod: meth .

]

{ #category : 'Pragmas' }
Behavior >> pragmasForMethod: selector env: envId [
  "Get the Pragmas for the method specified by selector and envId."

  | meth |
  envId == 0 ifTrue:[ ^ self pragmasForMethod: selector ].
  meth := self compiledMethodAt: selector environmentId: envId otherwise: nil.
  meth ifNil:[ ^ #() ].
  meth pragmas ifNotNil:[ :p | ^ p ].
  ^ #()

]

{ #category : 'Modifying Classes' }
Behavior >> recompileAllMethodsInContext: aSymbolList [

"Recompiles all methods for the receiver, using the specified symbol list.

 The environmentId of each method is preserved by the recompilation.

 This method is designed to allow a user interface to issue GciContinue after
 fixing the source code for a method in error.  GciContinue will reattempt the
 compilation of the method which contained an error, then proceed to the next
 method."

| envId |
self _validatePrivilege ifFalse:[ ^ nil ].
envId := 0 .
self env: envId unifiedCategoriesDo:[ :theCategory :selectorList |
    selectorList do: [ :aSelector| | aMeth err theSource |
      [ "This Block supports GciContinue"
        "Make sure the method is there before trying to recompile it.
         An exception handler may have removed the method!"
        err := nil .
        aMeth := self compiledMethodAt: aSelector environmentId: envId otherwise: nil .
        aMeth ifNotNil:[ | mEnvId |
          theSource := aMeth sourceString .
          (mEnvId := aMeth environmentId) == envId ifFalse:[
             self error:'mismatched environmentId'
          ].
          [
             self compileMethod: theSource dictionaries: aSymbolList
                          category: theCategory environmentId: envId .
           ] onException: CompileError do:[:ex |
            err := ex .
            ex outer .
          ]
        ].
        err == nil
      ] untilTrue .
    ] .
].
^ self
]

{ #category : 'Modifying Classes' }
Behavior >> recompileAllSubclassMethodsInContext: aSymbolList [

"Recompiles all methods for the receiver and its subclasses, using
 the specified symbol list.  If the receiver is not modifiable,
 then methods in subclasses will not be recompiled, since only
 modifiable classes should have the Subclasses class variable present."

| theSubclasses |
self _validatePrivilege ifFalse:[ ^ nil ].
self _removeAllSubclassCode .
self recompileAllMethodsInContext: aSymbolList .
theSubclasses := self _subclasses .
theSubclasses ifNotNil:[
  theSubclasses do:[:x | x recompileAllSubclassMethodsInContext: aSymbolList ]
  ].
^ self

]

{ #category : 'Updating the Method Dictionary' }
Behavior >> removeAllMethods [
  ^ self removeAllMethods: 0
]

{ #category : 'Updating Categories' }
Behavior >> removeCategory: categoryName environmentId: envId [

"Removes the specified category and all its methods from the receiver's
 method dictionary.  If categoryName is not in the receiver's categories,
 generates an error.
 Any breakpoints in removed methods are cleared."

| theSelectors catSym allCats |
self _validatePrivilege ifFalse:[ ^ nil ].
(self includesCategory: categoryName environmentId: envId ) ifFalse:[
  ^ self _categoryNotFound: categoryName .
].
catSym := categoryName asSymbol .
allCats := self _unifiedCategorys: envId .
theSelectors := (allCats at: catSym ) copy.
1 to: theSelectors size do: [ :j |
  self _basicRemoveSelector: (theSelectors at: j) environmentId: envId
].
envId == 0 ifTrue:[ | policy |
  policy := GsPackagePolicy currentOrNil .
  policy ifNotNil:[ :pp | pp removeCategory: catSym for: self ].
  (self canBeWritten or: [ policy == nil ]) ifTrue:[
    (self _baseCategorys: envId) removeKey: catSym otherwise: nil .
  ].
] ifFalse: [
  (self _baseCategorys: envId) removeKey: catSym
].
self _announceReorganized .

]

{ #category : 'Updating the Method Dictionary' }
Behavior >> removeSelector: aString environmentId: envId [

"Removes the method whose selector is aString from the receiver's method
 dictionary.  If the selector is not in the method dictionary, generates an
 error."

| selector removedFromBaseDict |

self _validatePrivilege ifFalse:[ ^ nil ].
selector := Symbol _existingWithAll: aString .
selector ifNotNil:[
  removedFromBaseDict := self _basicRemoveSelector: selector environmentId: envId.
  removedFromBaseDict ifTrue:[ | cDict |
    cDict := self _baseCategorys: envId .
    cDict keysAndValuesDo: [:aKey :setOfSelectors |
        (setOfSelectors remove: selector otherwise: nil ) ifNotNil:[
           ^ self "done"
        ]
     ].
    ^ self _error: #classErrSelectorNotFound args: { aString } .
  ].
] ifNil:[
  ^ self _error: #classErrSelectorNotFound args: { aString } .
]

]

{ #category : 'Updating the Method Dictionary' }
Behavior >> removeTransientSelector: aString environmentId: envId [
  "Removes the method whose selector is aString from the receiver's transient
   method dictionary.  If the selector is not in the transient method dictionary,
   generates an error."

  | selector |
  self _validatePrivilege
    ifFalse: [ ^nil].
  selector := Symbol _existingWithAll: aString.
  selector
    ifNotNil: [
      | md tmeth |
      md := self transientMethodDictForEnv: envId.
      md ifNotNil: [ tmeth := md removeKey: selector otherwise: nil ].
      tmeth ifNotNil: [ ^ self _refreshLookupCache: selector oldMethod: tmeth env: envId ] ].
  ^ self _error: #'classErrSelectorNotFound' args: {aString}

]

{ #category : 'Updating the Method Dictionary' }
Behavior >> removeTransientSelector: aString environmentId: envId ifAbsent: aBlock [
  "Removes the method whose selector is aString from the receiver's
   transient method dictionary.  If the selector is not in the
   transient method dictionary, returns the result of evaluating the
   zero-argument block aBlock.  Otherwise, returns the receiver."

  | aKey tmd meth |
  self _validatePrivilege
    ifFalse: [ ^ nil ].
  aKey := Symbol _existingWithAll: aString.
  aKey
    ifNotNil: [
      tmd := self transientMethodDictForEnv: envId.
      tmd ifNotNil: [ meth := tmd at: aKey otherwise: nil ] ].
  meth ifNil: [ ^ aBlock value ].
  self removeTransientSelector: aKey environmentId: envId

]

{ #category : 'Updating Categories' }
Behavior >> renameCategory: categoryName to: newCategoryName [
  ^ self renameCategory: categoryName to: newCategoryName environmentId: 0

]

{ #category : 'Updating Categories' }
Behavior >> renameCategory: categoryName to: newCategoryName environmentId: envId [

"Changes the name of the specified category to newCategoryName (a
 String), and returns the receiver.  If categoryName is not in the
 receiver's method dictionary, or if newCategoryName is already in the
 receiver's method dictionary, generates an error."

| catSym newCatSym oldCatSet policy |
self _validatePrivilege ifFalse:[ ^ nil ].
(self includesCategory: newCategoryName environmentId: envId ) ifTrue:[
  ^ self _error: #classErrMethCatExists args: { newCategoryName . self }
].
(self includesCategory: categoryName environmentId: envId ) ifFalse:[
   ^ self _error: #classErrMethCatNotFound args: { categoryName }
].
newCatSym := newCategoryName asSymbol .
catSym := categoryName asSymbol.
policy := GsPackagePolicy currentOrNil .
(envId ~~ 0 or:[ self canWriteMethods or: [ policy == nil ]]) ifTrue:[ | catDict|
   catDict :=  self _baseCategorysForStore: envId .
   catDict at: newCatSym put: (oldCatSet := catDict at: catSym ).
   catDict removeKey: catSym .
] ifFalse: [   | pkgCats catSet |
   pkgCats := GsMethodDictionary new  .
   policy copyCategoryDictFor: self into: pkgCats .
   oldCatSet := pkgCats at: catSym otherwise: { } .
   catSet := self addCategory: newCatSym environmentId: envId .
   catSet addAll: oldCatSet.
   policy removeCategory: catSym for: self.
].
self _needsAnnouncement ifTrue:[
  oldCatSet do: [:sel |
     self _announceMethodMoved: (self compiledMethodAt: sel) oldCategory: catSym
  ].
].

]

{ #category : 'Updating Categories' }
Behavior >> renameOrMergeCategory: oldName to: newName [
  ^ self renameOrMergeCategory: oldName to: newName environmentId: 0

]

{ #category : 'Updating Categories' }
Behavior >> renameOrMergeCategory: oldName to: newName environmentId: envId [

"Changes the name of the specified category to newName (a String), and
 returns the receiver.  If oldName is not an existing category in the receiver
 generates an error.  If newName is already in the receiver's
 category list, moves all the methods from the old category to the new
 category, and removes the old category."

| oldsym newsym oldCatSet policy |
self _validatePrivilege ifFalse:[ ^ nil ].
(self includesCategory: oldName environmentId: envId) ifFalse:[
  ^ self _categoryNotFound: oldName
].
(self includesCategory: newName environmentId: envId) ifFalse:[
  ^ self renameCategory: oldName to: newName environmentId: envId
].
oldsym := oldName asSymbol.
newsym := newName asSymbol.
policy := GsPackagePolicy currentOrNil .
(envId ~~ 0 or:[ self canWriteMethods or: [ policy == nil ]])  ifTrue:[ | catDict |
  catDict := self _baseCategorysForStore: envId .
  oldCatSet := catDict at: oldsym .
  (catDict at: newsym ) addAll: oldCatSet .
  catDict removeKey: oldsym .
] ifFalse:[ |  pkgCats |
  pkgCats := GsMethodDictionary new  .
  policy copyCategoryDictFor: self into: pkgCats .
  oldCatSet := (pkgCats at: oldsym ) copy .
  oldCatSet do:[ :aSelector |
     policy moveSelector: aSelector toCategory: newsym for: self .
  ].
  policy removeCategory: oldsym for: self.
].
self _needsAnnouncement ifTrue:[
  oldCatSet do: [:sel |
     self _announceMethodMoved: (self compiledMethodAt: sel) oldCategory: oldsym
  ].
].

]

{ #category : 'Pragmas' }
Behavior >> setPragmas: pragmaArray forMethod: selector [

  "Deprecated, pragmas are now generated into GsNMethod's debugInfo during compile.

   Set the pragmas for the given method .
   This code shared by Classes and metaclasses,
   parameterized by pragmaDictName."

  | methodPragmaDict eDict selSym |
  pragmaArray ifNotNil:[
    self deprecated: 'Behavior >> setPragmas:forMethod: deprecated in 3.5' .
  ].
  selSym := Symbol _existingWithAll: selector  .
  selSym ifNil:[ self error:'invalid selector'].
  GsPackagePolicy currentOrNil ifNotNil:[:pp |
     (pp setPragmas: pragmaArray forBehavior: self forMethod: selSym ) ifNotNil:[ ^self ].
  ].
  (eDict := self extraDictForStore) ifNotNil:[ | dictName |
     dictName := self pragmaDictName .
     methodPragmaDict := eDict at: dictName otherwise: nil .
     methodPragmaDict ifNil: [
       GsObjectSecurityPolicy setCurrent: self objectSecurityPolicy while:[
         methodPragmaDict := IdentityKeyValueDictionary new.
       ].
       eDict at: dictName put: methodPragmaDict.
     ].
     pragmaArray ifNil:[ methodPragmaDict removeKey: selSym otherwise: nil  ]
         ifNotNil: [ methodPragmaDict at: selSym put: pragmaArray ]
  ].

]

{ #category : 'Method Timestamps' }
Behavior >> setStamp: aStamp forMethod: selector [
	"Set the timestamp for the given method, when 
		session methods are enabled.
		This code shared by Classes and metaclasses,
		parameterized by methodStampDictName."

	| methodStampDict eDict selSym |
	selSym := Symbol _existingWithAll: selector.
	selSym ifNil: [ self error: 'invalid selector' ].
	GsPackagePolicy currentOrNil
		ifNotNil: [ :pp | 
			(pp setStamp: aStamp forBehavior: self forMethod: selSym) ifNotNil: [ ^ self ].
			(eDict := self extraDictForStore)
				ifNotNil: [ 
					| dictName |
					dictName := self methodStampDictName.
					methodStampDict := eDict at: dictName otherwise: nil.
					methodStampDict
						ifNil: [ 
							aStamp
								ifNil: [ 
									"no need to add a new methodStampDict, 
										when we intend to remove a key from
										it (bug 49169)"
									^ self ].
							GsObjectSecurityPolicy
								setCurrent: self objectSecurityPolicy
								while: [ methodStampDict := IdentityKeyValueDictionary new ].
							eDict at: dictName put: methodStampDict ].
					aStamp
						ifNil: [ methodStampDict removeKey: selSym otherwise: nil ]
						ifNotNil: [ methodStampDict at: selSym put: aStamp ] ] ]
]

{ #category : 'Method Timestamps' }
Behavior >> stampForMethod: selector [
  "Get the timestamp for the given method"

  | stamp methodStampDict eDict selSym |
  selSym := Symbol _existingWithAll: selector  .
  selSym ifNil:[ ^ '' ].
  stamp := GsPackagePolicy currentOrNil ifNotNil:[:pp| pp stampForMethod: selSym in: self].
  stamp ifNotNil: [ ^stamp ].

  (eDict := self extraDictForStore) ifNil:[ ^'' ].
  (methodStampDict := eDict at: self methodStampDictName otherwise: nil) ifNil: [ ^'' ].
  ^ methodStampDict at: selSym otherwise: ''

]

{ #category : 'Class Timestamps' }
Behavior >> theNonMetaClass [
  "Sent to a class or metaclass, always return the class.
   Used by Monticello"

  ^self

]

{ #category : 'Modifying Classes' }
Behavior >> upgradeComments [

  " Upgrade comment representation to GS/64 3.1 standard.
    Prints out info on changes made to log file.
    Prints out WARNING info if any unusual conditions noted.
    Returns an array of two Boolean elements:
      1:  true if any warnings, otherwise false
      2:  true if a change was made to class, otherwise false
  "

  | prefix meta precomment comment descript descriptStr warning changed lf |

  System currentObjectSecurityPolicy: self objectSecurityPolicy.
  prefix := '    Class ' , (self name asString) , ': '.
  comment := nil.
  precomment := self _extraDictAt: #comment.
  meta := self class.
  warning := false.
  changed := false.
  lf := Character lf.
  " Check for comment method "
  (meta includesSelector: #comment)
  ifTrue: [
    comment := [ self comment ] on: Error do:[:ex | 'old comment not available'].
    (comment isKindOf: CharacterCollection)
    ifFalse: [
      GsFile gciLogServer: prefix , lf ,
      '          WARNING: Non-standard comment method.' , lf ,
      '          No changes made.'.
      warning := true.
      ^ Array with: warning with: changed ]
    ifTrue: [
      self _extraDictAt: #comment put: comment.
      meta removeSelector: #comment.
      changed := true.
      GsFile gciLogServer: prefix , 'Using comment method.'.
      precomment isNil ifFalse: [
        (precomment = comment) ifFalse: [
          GsFile gciLogServer:
          '          WARNING: Pre-existing #comment field will be lost!'.
          warning := true.
      "fall-through deliberate"
      ]]]].
  " Now check description field"
  descript := self _extraDictAt: #description.
  descript ifNil: [
    comment ifNil: [
      GsFile gciLogServer: prefix , 'No comment method or #description field.'.
      precomment ifNotNil:[ GsFile gciLogServer:
        '          Pre-existing #comment field will be used.' ]
      ifNil: [ GsFile gciLogServer: '          No changes made.']].
    ^ Array with: warning with: changed ].

  (descript isKindOf: GsClassDocumentation)
  ifFalse: [
    comment ifNil: [
      GsFile gciLogServer: prefix , lf ,
'          WARNING: No comment method and non-standard #description field.'.
      precomment ifNotNil: [
        GsFile gciLogServer:
        '          Pre-existing #comment field will be used.' ].
      GsFile gciLogServer: '          No changes made.'.
      warning := true.
      ^ Array with: warning with: changed ]
    ifNotNil: [
      GsFile gciLogServer:
      '          WARNING: Non-standard #description field will be deleted!'.
      self _extraDictRemoveKey: #description.
      changed := true.
      warning := true.
      ^ Array with: warning with: changed ]].

  descriptStr := descript asString.
  comment ifNil: [
    GsFile gciLogServer: prefix , 'Using #description field.'.
      precomment ifNotNil: [
        (precomment = descriptStr) ifFalse: [
          GsFile gciLogServer:
      '          WARNING: Pre-existing #comment field will be overridden!'.
          warning := true ]].
    self _extraDictAt: #comment put: descriptStr.
    self _extraDictRemoveKey: #description.
    changed := true.
    ^ Array with: warning with: changed ]
  ifNotNil: [
    self _extraDictRemoveKey: #description.
    changed := true.
    (comment = descriptStr)
    ifTrue: [  ^ Array with: warning with: changed ]
    ifFalse: [
      GsFile gciLogServer:
      '          WARNING: Comment method and #description field mismatch.'.
      GsFile gciLogServer:
      '          WARNING: description field will be deleted!'.
      warning := true.
      ^ Array with: warning with: changed ]].
^ Array with: warning with: changed

]
