Extension { #name : 'BtreePlusInteriorNode' }

{ #category : 'Rc Updating' }
BtreePlusInteriorNode >> _at: aKey put: aValue root: rootObject selectiveAbortSet: selectiveAbortSetOrNil [
  "Adds the key/value/root pair to the node.  If the node is full, performs a 'split'
 on the parent.  Returns the new sibling if a split is performed, otherwise
 returns the receiver."

  "self has already been selectively aborted"

  | index node returnNode |
  index := self binarySearchInteriorNodeCoveringKey: aKey put: aValue root: rootObject  selectiveAbortSet: selectiveAbortSetOrNil.
  node := self at: index.
  self _selectiveAbort: node ifNotIn: selectiveAbortSetOrNil.
  returnNode := node
    at: aKey
    put: aValue
    root: rootObject
    selectiveAbortSet: selectiveAbortSetOrNil. "node is selectively aborted and added to rcReadSet"
  ^ self updateAfterAddAt: index child: node returnNode: returnNode selectiveAbortSet: selectiveAbortSetOrNil

]

{ #category : 'Removing' }
BtreePlusInteriorNode >> _attemptTakingFromNeighborForNodeAt: nodeIndex [

"The node at the given index has too few entries, so try to take entries from
 one of its neighboring siblings.  First try the next (right) sibling, then
 the previous (left) sibling."

| node sibling siblingEntrySize siblingLoss numSiblingElements copyStart copyEnd
fromNextSibling fromPreviousSibling targetStart lastChildKey  numToMove keyIndexOffset entrySize lastChildRoot |

node := self at: nodeIndex.
fromNextSibling := false.  " is true if taken from next sibling "
fromPreviousSibling := false.  " is true if taken from previous sibling "

siblingEntrySize := node entrySize.
entrySize := self entrySize.
keyIndexOffset := self keyIndexOffset.
" check if the node is the last entry "
nodeIndex == (self _lastKeyIndex - keyIndexOffset)
    ifFalse: [ " this is not the last entry, check the next sibling "
        sibling := self at: (nodeIndex + self entrySize).
        numSiblingElements := sibling numElements.
        numSiblingElements < (node mergeThreshold + 1)
            ifFalse: [ " can take entries from next sibling "
                fromNextSibling := true.
                siblingLoss := (numSiblingElements - node mergeThreshold) quo: 2.
                siblingLoss == 0 ifTrue: [ siblingLoss := 1 ].
                copyStart := 1.
                copyEnd := siblingLoss * siblingEntrySize.
                numToMove := 0 .
                targetStart := node _lastIndex + 1
            ]
    ].

" if could not get entries from next sibling and this is not the first entry "
(fromNextSibling not and: [ nodeIndex > 1 ] )
    ifTrue: [ " try previous entry "
        sibling := self at: (nodeIndex - self entrySize).
        numSiblingElements := sibling numElements.
        numSiblingElements < (node mergeThreshold + 1)
            ifFalse: [ " can take entries from previous sibling "
                fromPreviousSibling := true.
                siblingLoss := (numSiblingElements - node mergeThreshold) quo: 2.
                siblingLoss == 0 ifTrue: [ siblingLoss := 1 ].
                copyStart := (numSiblingElements - siblingLoss) * siblingEntrySize + 1.
                copyEnd := numSiblingElements * siblingEntrySize.
                numToMove := (node numElements) * siblingEntrySize .
                targetStart := 1
            ]
    ].

" if cannot take from previous or next sibling "
(fromPreviousSibling or:[ fromNextSibling ])
    ifFalse: [ ^ false ].

" copy from sibling and insert into node "
node _insertAt: targetStart
     from: sibling fromStart: copyStart fromEnd: copyEnd
     numToMoveDown: numToMove .

node numElements: (node numElements + siblingLoss).

fromNextSibling
    ifTrue: [
        "entries added to end of node from beginning of sibling"
        sibling _deleteNoShrinkFrom: copyStart to: copyEnd .

        sibling numElements: (numSiblingElements - siblingLoss).

        " update the receiver's key for this entry "
        lastChildKey := node at: (node _lastKeyIndex).
        self primitiveAt: (nodeIndex + keyIndexOffset) put: lastChildKey.
        " see if last root in child node has changed "
        lastChildRoot := node at: node _lastRootIndex.
        self updateLastChildRoot: lastChildRoot at: nodeIndex.
        node _updateLastValue
    ].

fromPreviousSibling
    ifTrue: [
        "entries added to beginning of node from end of sibling"
        sibling numElements: (numSiblingElements - siblingLoss).

        " update the receiver's key for previous entry "
        lastChildKey := sibling at: sibling _lastKeyIndex .
        self  primitiveAt: (nodeIndex + keyIndexOffset - entrySize) put: lastChildKey.
        " see if last root in child node has changed "
        lastChildRoot := sibling at: sibling _lastRootIndex.
        self updateLastChildRoot: lastChildRoot at: nodeIndex - entrySize.
        sibling _updateLastValue .

       " nil out sibling entries at end "
        sibling _deleteNoShrinkFrom: copyStart to: copyEnd .
    ].

^ true

]

{ #category : 'Rc Removing' }
BtreePlusInteriorNode >> _attemptTakingFromNeighborForNodeAt: nodeIndex selectiveAbortSet: selectiveAbortSetOrNil [

"The node at the given index has too few entries, so try to take entries from
 one of its neighboring siblings.  First try the next (right) sibling, then
 the previous (left) sibling."

| node sibling siblingEntrySize siblingLoss numSiblingElements copyStart copyEnd
fromNextSibling fromPreviousSibling targetStart lastChildKey numToMove keyIndexOffset entrySize lastChildRoot |

node := self at: nodeIndex.
self _selectiveAbort: node ifNotIn: selectiveAbortSetOrNil.
fromNextSibling := false.  " is true if taken from next sibling "
fromPreviousSibling := false.  " is true if taken from previous sibling "

siblingEntrySize := node entrySize.
entrySize := self entrySize.
keyIndexOffset := self keyIndexOffset.
" check if the node is the last entry "
nodeIndex == (self _lastKeyIndex - keyIndexOffset)
    ifFalse: [ " this is not the last entry, check the next sibling "
        sibling := self at: (nodeIndex + self entrySize).
        self _selectiveAbort: sibling ifNotIn: selectiveAbortSetOrNil.
        numSiblingElements := sibling numElements.
        numSiblingElements < (node mergeThreshold + 1)
            ifFalse: [ " can take entries from next sibling "
                fromNextSibling := true.
                siblingLoss := (numSiblingElements - node mergeThreshold) quo: 2.
                siblingLoss == 0 ifTrue: [ siblingLoss := 1 ].
                copyStart := 1.
                copyEnd := siblingLoss * siblingEntrySize.
                numToMove := 0 .
                targetStart := node _lastIndex + 1
            ]
    ].

" if could not get entries from next sibling and this is not the first entry "
(fromNextSibling not and: [ nodeIndex > 1 ] )
    ifTrue: [ " try previous entry "
        sibling := self at: (nodeIndex - self entrySize).
        self _selectiveAbort: sibling ifNotIn: selectiveAbortSetOrNil.
        numSiblingElements := sibling numElements.
        numSiblingElements < (node mergeThreshold + 1)
            ifFalse: [ " can take entries from previous sibling "
                fromPreviousSibling := true.
                siblingLoss := (numSiblingElements - node mergeThreshold) quo: 2.
                siblingLoss == 0 ifTrue: [ siblingLoss := 1 ].
                copyStart := (numSiblingElements - siblingLoss) * siblingEntrySize + 1.
                copyEnd := numSiblingElements * siblingEntrySize.
                numToMove := (node numElements) * siblingEntrySize .
                targetStart := 1
            ]
    ].

" if cannot take from previous or next sibling "
(fromPreviousSibling or:[ fromNextSibling ])
    ifFalse: [ ^ false ].

" copy from sibling and insert into node "
node _insertAt: targetStart
     from: sibling fromStart: copyStart fromEnd: copyEnd
     numToMoveDown: numToMove .

node numElements: (node numElements + siblingLoss).

fromNextSibling
    ifTrue: [
        "entries added to end of node from beginning of sibling"
        sibling _deleteNoShrinkFrom: copyStart to: copyEnd .

        sibling numElements: (numSiblingElements - siblingLoss).

        " update the receiver's key for this entry "
        lastChildKey := node at: (node _lastKeyIndex).
        self primitiveAt: (nodeIndex + keyIndexOffset) put: lastChildKey.
        " see if last root in child node has changed "
        lastChildRoot := node at: node _lastRootIndex.
        self updateLastChildRoot: lastChildRoot at: nodeIndex.
        node _updateLastValue: selectiveAbortSetOrNil
    ].

fromPreviousSibling
    ifTrue: [
        "entries added to beginning of node from end of sibling"
        sibling numElements: (numSiblingElements - siblingLoss).

        " update the receiver's key for previous entry "
        lastChildKey := sibling at: sibling _lastKeyIndex .
        self  primitiveAt: (nodeIndex + keyIndexOffset - entrySize) put: lastChildKey.
        " see if last root in child node has changed "
        lastChildRoot := sibling at: sibling _lastRootIndex.
        self updateLastChildRoot: lastChildRoot at: nodeIndex - entrySize.
        sibling _updateLastValue: selectiveAbortSetOrNil .

       " nil out sibling entries at end "
        sibling _deleteNoShrinkFrom: copyStart to: copyEnd .
    ].

self rootNode
  _addBtreePlusNodeToRcReadSet: node;
  _addBtreePlusNodeToRcReadSet: sibling.
^ true

]

{ #category : 'Searching' }
BtreePlusInteriorNode >> _findAllValuesForIdenticalKey: aKey into: aCollection using: aComparison [

"Finds all the values for the given identical key, placing them in the
 collection.  For interior nodes, must first find the appropriate child
 node.  Return whether the last key was equal or not (so the caller
 knows it must check the next node)."

| index eSize maxIndex |
" find first child node that might contain the key "
index := self _findCoveringIndexForKey: aKey totalOrder: false.
index == 0 ifTrue: [ ^ false ].

((self at: index) _findAllValuesForIdenticalKey: aKey into: aCollection using: aComparison)
  ifFalse: [ ^ false ].

eSize := self entrySize.
maxIndex := self _lastIndex.
index := index + eSize.

" now scan child nodes until no values are added to the collection "
[ (index < maxIndex) and:
[ (self at: index) _findAllValuesForIdenticalKey: aKey into: aCollection using: aComparison ] ]
  whileTrue: [ index := index + eSize ].
^ index >= maxIndex

]

{ #category : 'Searching Support' }
BtreePlusInteriorNode >> _findAllValuesGreaterThanKey: aKey into: anArray using: aComparison [
  "Puts into anArray the index and the leaf node of the entry that is greater
 than the given key, plus the index and interior node of the path taken to
 reach the leaf.  Returns true if a value was found, false otherwise.  Do this
 by invoking the same operation on the appropriate child node."

  | index eSize lastKeyIndex keyIndexOffset comparisonForSort |
  lastKeyIndex := self _lastKeyIndex.
  lastKeyIndex == 0
    ifTrue: [ ^ false ].
  " find index where key would be inserted "
  comparisonForSort := self comparisonForSort.
  index := self _findCoveringIndexForKey: aKey totalOrder: false  using: comparisonForSort.
  index == 0
    ifTrue: [ ^ false ].
  "scan across all entries that sort ="
  eSize := self entrySize.
  keyIndexOffset := self keyIndexOffset.
  [ comparisonForSort compareKey: aKey equalTo: (self at: index + keyIndexOffset) ]
    whileTrue: [ index := index + eSize.
      index > lastKeyIndex
        ifTrue: [ ^ false ] ].
  " see if this child has any greater than the key "
  ((self at: index)
    _findAllValuesGreaterThanKey: aKey
    into: anArray
    using: aComparison)
    ifFalse: [ ^ false ].
  ^ true

]

{ #category : 'Searching Support' }
BtreePlusInteriorNode >> _findAllValuesLessThanKey: aKey andEquals: aBoolean into: anArray using: aComparison [
  "Puts into anArray the index and the leaf node of the entry that is less than
 (or less than and equal to if aBoolean is true) the given key, plus the index
 and interior node of the path taken to reach the leaf.  Returns true if a
 value was found, false otherwise.  Does this by invoking the same operation
 on the appropriate child node."

  | index eSize lastIndex lastKeyIndex comparisonForSort keyIndexOffset |
  lastIndex := self _lastIndex.
  lastIndex == 0
    ifTrue: [ ^ false ].
  aBoolean
    ifFalse: [ ^ self _findAllValuesLessThanKey: aKey into: anArray using: aComparison ].
  " first see if all entries are > aKey "
  lastKeyIndex := self _lastKeyIndex.
  comparisonForSort := self comparisonForSort.
  (comparisonForSort
    compareKey: aKey
    lessThanOrEqualTo: (self at: lastKeyIndex))
    ifFalse: [ "bugfix 33805 - verify that aKey is > last entry"
      (aKey == nil
        or: [ comparisonForSort compareKey: aKey greaterThan: (self at: lastKeyIndex) ])
        ifTrue: [ self _putLastIndexOfLastChildInto: anArray.
          ^ true ] ].
  "find index of first entry >= aKey"
  index := self _findCoveringIndexForKey: aKey totalOrder: false using: comparisonForSort.
  index == 0
    ifTrue: [ ^ false ].
  "scan across all of the values that sort ="
  eSize := self entrySize.
  keyIndexOffset := self keyIndexOffset.
  (comparisonForSort compareKey: aKey equalTo: (self at: index + keyIndexOffset))
    ifTrue: [
      [ index < lastIndex
        and: [ comparisonForSort compareKey: aKey equalTo: (self at: index + keyIndexOffset) ] ]
        whileTrue: [ index := index + eSize ].
      " if we scanned past the end, subtract an entry "
      index > lastIndex
        ifTrue: [ index := index - eSize ] ].
  " see if child has any entries less than or equal to aKey "
  ((self at: index)
    _findAllValuesLessThanKey: aKey
    andEquals: true
    into: anArray
    using: comparisonForSort)
    ifFalse: [ " this child did not find any "
      " if it is not the first child "
      index > 1
        ifTrue: [ " then the entry is the rightmost leaf entry of the previous child "
          index := index - eSize.
          (self at: index) _putLastIndexOfLastChildInto: anArray ]
        ifFalse: [ ^ false ] ].
  ^ true

]

{ #category : 'Searching Support' }
BtreePlusInteriorNode >> _findAllValuesLessThanKey: aKey into: anArray using: aComparison [
  "Puts into anArray the index and the leaf node of the entry that is less than
 (or less than and equal to if aBoolean is true) the given key, plus the index
 and interior node of the path taken to reach the leaf.  Returns true if a
 value was found, false otherwise.  Does this by invoking the same operation
 on the appropriate child node."

  | index eSize lastIndex lastKeyIndex comparisonForSort |
  lastIndex := self _lastIndex.
  lastIndex == 0
    ifTrue: [ ^ false ].
  " first see if all entries are > aKey "
  lastKeyIndex := self _lastKeyIndex.
  comparisonForSort := self comparisonForSort.
  (comparisonForSort
    compareKey: aKey
    lessThanOrEqualTo: (self at: lastKeyIndex))
    ifFalse: [ "bugfix 33805 - verify that aKey is > last entry"
      (aKey == nil
        or: [ aComparison compareKey: aKey greaterThan: (self at: lastKeyIndex) ])
        ifTrue: [ self _putLastIndexOfLastChildInto: anArray.
          ^ true ] ].
  " find index where key would be inserted "
  index := self _findCoveringIndexForKey: aKey totalOrder: false using: comparisonForSort.
  index == 0
    ifTrue: [ ^ false ].
  eSize := self entrySize.
  " see if this child has any less than the key "
  ((self at: index)
    _findAllValuesLessThanKey: aKey
    into: anArray
    using: aComparison)
    ifFalse: [ " this child did not find any "
      " if it is not the first child "
      index > 1
        ifTrue: [ " then the entry is the rightmost leaf entry of the previous child "
          index := index - eSize.
          (super at: index) _putLastIndexOfLastChildInto: anArray ]
        ifFalse: [ ^ false ] ].
  ^ true

]

{ #category : 'Searching' }
BtreePlusInteriorNode >> _findAndCountKey: aKey value: aValue root: rootObj into: col [
"return whether the last key was equal or not (so the caller knows to check next node"

| index eSize maxIndex |
index := self _findCoveringIndexForKey: aKey totalOrder: false.
index == 0 ifTrue: [ ^ false ].

((self at: index) _findAndCountKey: aKey value: aValue root: rootObj into: col)
  ifFalse: [ ^ false ].

eSize := self entrySize.
maxIndex := self _lastIndex.
index := index + eSize.

[ (index < maxIndex) and:
[ (self at: index) _findAndCountKey: aKey value: aValue root: rootObj into: col ] ]
  whileTrue: [ index := index + eSize ].
^ index >= maxIndex

]

{ #category : 'Searching' }
BtreePlusInteriorNode >> _findFirstRootForKey: aKey using: aComparison [

"Returns the first root object associated with the given key.  For interior nodes,
 find the appropriate child node and ask it to find the first root."

| index |
" find first child node that might contain the key "
index := self _findCoveringIndexForKey: aKey totalOrder: false.
index == 0 ifTrue: [ ^ #_incompletePathTraversal ].

^ (self at: index) _findFirstRootForKey: aKey using: aComparison

]

{ #category : 'Searching Support' }
BtreePlusInteriorNode >> _lastRootObjectsNotIdenticalTo: aKey into: array [
  "Send this message to all the receiver's children."

  1 to: numElements * self entrySize by: self entrySize do: [ :i | (self at: i) _lastRootObjectsNotIdenticalTo: aKey into: array ].

]

{ #category : 'Enumerating' }
BtreePlusInteriorNode >> _leafKeysValuesAndRootsDo: aBlock [

"Execute the two-argument block for each child node."

1 to: (numElements * self entrySize) by: self entrySize do: [ :i |
    (self at: i) _leafKeysValuesAndRootsDo: aBlock
]

]

{ #category : 'Removing' }
BtreePlusInteriorNode >> _mergeFromNeighborForNodeAt: nodeIndex [

"The node at the given index has too few entries and neighboring siblings had
 too few entries to take from, so merge with one of its neighboring siblings.
 First tries the next (right) sibling, then the previous (left) sibling."

| node sibling eSize myESize numSiblingElements copyStart lastChildKey myLastIndex keyIndexOffset lastChildRoot |
node := self at: nodeIndex.
eSize := node entrySize.
myESize := self entrySize.
keyIndexOffset := self keyIndexOffset.

" check if the node is the last entry "
nodeIndex == self _lastEntryIndex
    ifTrue: [ " this is the last entry, try the previous sibling "
        nodeIndex > 1
            ifTrue: [ " keep sibling, throw away node "
                sibling := self at: (nodeIndex - myESize).
                numSiblingElements := sibling numElements.

                " move node's entries onto end of sibling "
                sibling _insertAt: (sibling _lastIndex + keyIndexOffset)
                   from: node fromStart: 1 fromEnd: (node numElements * eSize)
                   numToMoveDown: 0 .

                sibling numElements: (node numElements + numSiblingElements).

                " update the key for the sibling's entry in the receiver "
                lastChildKey := sibling at: (sibling _lastKeyIndex).
                self primitiveAt: (nodeIndex - myESize + keyIndexOffset) put: lastChildKey.
                " see if last root in child node has changed "
                lastChildRoot := sibling at: sibling _lastRootIndex.
                self updateLastChildRoot: lastChildRoot at: nodeIndex - myESize.
                sibling _updateLastValue.
                "update nextLeaf of sibling and update previousLeaf of node nextLeaf"
                sibling nextLeaf: node nextLeaf.
                node nextLeaf ifNotNil: [:nLeaf | nLeaf previousLeaf: sibling ] ]
            ifFalse:[ self _error: #rtErrBtreeInvalidMerge.
                      self _uncontinuableError ].
        numElements := numElements - 1. "update here -- when losing node (next clause), need to update numElements mid-stream"
    ]
    " this is not the last entry, get the next sibling "
    ifFalse: [ " keep node, throw away sibling "
        sibling := self at: (nodeIndex + myESize).
        numSiblingElements := sibling numElements.

        "copy from sibling onto end of node "
        node _insertAt: (node _lastIndex + keyIndexOffset)
             from: sibling fromStart: 1 fromEnd: (numSiblingElements * eSize)
             numToMoveDown: 0 .

        copyStart := nodeIndex + myESize + myESize.
        " if sibling was not the last entry in the receiver "
        myLastIndex := self _lastIndex .
        copyStart < myLastIndex
            ifTrue: [
                " move entries in receiver to the left "
                self _deleteNoShrinkFrom: (nodeIndex + myESize) to: copyStart - 1
            ].

        numElements := numElements - 1. "subsequent _last*Index calculation needs to be correct"
        node numElements: (node numElements + numSiblingElements).

        " update the key for the node's entry in the receiver "
        lastChildKey := node at: (node _lastKeyIndex).
        self primitiveAt: (nodeIndex + keyIndexOffset) put: lastChildKey.
        " see if last root in child node has changed "
        lastChildRoot := node at: node _lastRootIndex.
        self updateLastChildRoot: lastChildRoot at: nodeIndex.
        node _updateLastValue.
        "update nextLeaf of node using the next element in the receiver"
        nodeIndex == self _lastEntryIndex
          ifTrue: [ "update nextLeaf of node and update previousLeaf of sibling nextLeaf"
            node nextLeaf: sibling nextLeaf.
            sibling nextLeaf ifNotNil: [:nLeaf | nLeaf previousLeaf: node ] ]
          ifFalse: [ "sibling is not the last entry so splice in sibling's sibling"
            | newNextLeaf |
            newNextLeaf := self at: (nodeIndex + myESize).
            node nextLeaf: newNextLeaf.
            newNextLeaf previousLeaf: node ] ].

" nil out the last entry of the receiver, we've either explicitly dropped last entry, or
  we slid the entries down in receiver and need to delete duplicate last entry"
myLastIndex := self _lastIndex .
self _deleteNoShrinkFrom: myLastIndex + 1 to: myLastIndex + myESize.
self _updateLastValue.

]

{ #category : 'Rc Removing' }
BtreePlusInteriorNode >> _mergeFromNeighborForNodeAt: nodeIndex selectiveAbortSet: selectiveAbortSetOrNil [

"The node at the given index has too few entries and neighboring siblings had
 too few entries to take from, so merge with one of its neighboring siblings.
 First tries the next (right) sibling, then the previous (left) sibling."

| node sibling eSize myESize numSiblingElements copyStart lastChildKey myLastIndex keyIndexOffset lastChildRoot |

node := self at: nodeIndex.
self _selectiveAbort: node ifNotIn: selectiveAbortSetOrNil.
eSize := node entrySize.
myESize := self entrySize.
keyIndexOffset := self keyIndexOffset.
" check if the node is the last entry "
nodeIndex == self _lastEntryIndex
    ifTrue: [ " this is the last entry, try the previous sibling "
        nodeIndex > 1
            ifTrue: [ " keep sibling, throw away node "
                sibling := self at: (nodeIndex - myESize).
                self _selectiveAbort: sibling ifNotIn: selectiveAbortSetOrNil.
                numSiblingElements := sibling numElements.

                " move node's entries onto end of sibling "
                sibling _insertAt: (sibling _lastIndex + keyIndexOffset)
                   from: node fromStart: 1 fromEnd: (node numElements * eSize)
                   numToMoveDown: 0 .

                sibling numElements: (node numElements + numSiblingElements).

                " update the key for the sibling's entry in the receiver "
                lastChildKey := sibling at: (sibling _lastKeyIndex).
                self primitiveAt: (nodeIndex - myESize + keyIndexOffset) put: lastChildKey.
                " see if last root in child node has changed "
                lastChildRoot := sibling at: sibling _lastRootIndex.
                self updateLastChildRoot: lastChildRoot at: nodeIndex - myESize.
                sibling _updateLastValue: selectiveAbortSetOrNil.
                "update nextLeaf of sibling and update previousLeaf of node nextLeaf"
                sibling nextLeaf: node nextLeaf.
                node nextLeaf ifNotNil: [:nLeaf |
                  self _selectiveAbort: nLeaf ifNotIn: selectiveAbortSetOrNil.
                  nLeaf previousLeaf: sibling.
                  self rootNode _addBtreePlusNodeToRcReadSet: nLeaf ].
                self rootNode _addBtreePlusNodeToRcReadSet: sibling.
                "write dropped node to force a commit conflict and force replay, in case
                 another session makes mods to the node"
                node numElements: node numElements.
                self rootNode _addBtreePlusNodeToRcReadSet: node ]
            ifFalse:[ self _error: #rtErrBtreeInvalidMerge.
                      self _uncontinuableError ].
        numElements := numElements - 1. "update here -- when losing node (next clause), need to update numElements mid-stream"
    ]
    " this is not the last entry, get the next sibling "
    ifFalse: [ " keep node, throw away sibling "
        sibling := self at: (nodeIndex + myESize).
        self _selectiveAbort: sibling ifNotIn: selectiveAbortSetOrNil.
        numSiblingElements := sibling numElements.

        "copy from sibling onto end of node "
        node _insertAt: (node _lastIndex + keyIndexOffset)
             from: sibling fromStart: 1 fromEnd: (numSiblingElements * eSize)
             numToMoveDown: 0 .

        copyStart := nodeIndex + myESize + myESize.
        " if sibling was not the last entry in the receiver "
        myLastIndex := self _lastIndex .
        copyStart < myLastIndex
            ifTrue: [
                " move entries in receiver to the left "
                self _deleteNoShrinkFrom: (nodeIndex + myESize) to: copyStart - 1
            ].

        numElements := numElements - 1. "subsequent _last*Index calculation needs to be correct"
        node numElements: (node numElements + numSiblingElements).

        " update the key for the node's entry in the receiver "
        lastChildKey := node at: (node _lastKeyIndex).
        self primitiveAt: (nodeIndex + keyIndexOffset) put: lastChildKey.
        " see if last root in child node has changed "
        lastChildRoot := node at: node _lastRootIndex.
        self updateLastChildRoot: lastChildRoot at: nodeIndex.
        node _updateLastValue: selectiveAbortSetOrNil.
        "update nextLeaf of node using the next element in the receiver"
        nodeIndex == self _lastEntryIndex
          ifTrue: [ "update nextLeaf of node and update previousLeaf of sibling nextLeaf"
            node nextLeaf: sibling nextLeaf.
            sibling nextLeaf ifNotNil: [:nLeaf |
              self _selectiveAbort: nLeaf ifNotIn: selectiveAbortSetOrNil.
              nLeaf previousLeaf: node.
              self rootNode _addBtreePlusNodeToRcReadSet: nLeaf ] ]
          ifFalse: [ "sibling is not the last entry so splice in sibling's sibling"
            | newNextLeaf |
            newNextLeaf := self at: (nodeIndex + myESize).
            self _selectiveAbort: newNextLeaf ifNotIn: selectiveAbortSetOrNil.
            node nextLeaf: newNextLeaf.
            newNextLeaf previousLeaf: node.
            self rootNode _addBtreePlusNodeToRcReadSet: newNextLeaf ].
        self rootNode _addBtreePlusNodeToRcReadSet: node.
        "write dropped sibling to force a commit conflict and force replay, in case
         another session makes mods to the node"
        sibling numElements: sibling numElements.
        self rootNode _addBtreePlusNodeToRcReadSet: sibling  ].

" nil out the last entry of the receiver, we've either explicitly dropped last entry, or
  we slid the entries down in receiver and need to delete duplicate last entry"
myLastIndex := self _lastIndex .
self _deleteNoShrinkFrom: myLastIndex + 1 to: myLastIndex + myESize.
self _updateLastValue: selectiveAbortSetOrNil.

]

{ #category : 'Enumerating' }
BtreePlusInteriorNode >> _preOrderDo: aBlock [
  aBlock value: self.
  1 to: numElements * self entrySize by: self entrySize do: [ :i | (self at: i) _preOrderDo: aBlock ]

]

{ #category : 'Sorting Support' }
BtreePlusInteriorNode >> _putAscendingRootObjectsInto: array startingAt: offset [

"Returns the number of elements."

| j |
j := offset.
1 to: (numElements * self entrySize) by: self entrySize do: [ :i |
  j := j + ((self at: i) _putAscendingRootObjectsInto: array startingAt: j)
].
^ j - offset

]

{ #category : 'Sorting Support' }
BtreePlusInteriorNode >> _putDescendingRootObjectsInto: array startingAt: offset [

"Returns the number of elements."

| j |
j := offset.
1 to: (numElements * self entrySize) by: self entrySize do: [ :i |
  j := j - ((self at: i) _putDescendingRootObjectsInto: array startingAt: j)
].
^ offset - j

]

{ #category : 'Searching Support' }
BtreePlusInteriorNode >> _putFirstIndexOfFirstChildInto: anArray [
  "Ask the first child in the receiver to
 putFirstIndexOfFirstChildInto: anArray."

  (self at: 1) _putFirstIndexOfFirstChildInto: anArray

]

{ #category : 'Searching Support' }
BtreePlusInteriorNode >> _putFirstIndexOfFirstChildInto: anArray ifGreaterThanOrEqualTo: aKey using: aComparison [
  "Ask the first child in the receiver to putFirstIndexOfFirstChildInto:ifGreaterThanOrEqualTo:.

 Because of NANs we need to validate the assumption of the caller
 (i.e., that all entries are greater than aKey). See bug33805."

  "make sure that key is less than or equal to first entry"

  (aKey == nil
    or: [ aComparison compareKey: aKey lessThanOrEqualTo: (self at: 1 + self keyIndexOffset) ])
    ifTrue: [ (self at: 1)
        _putFirstIndexOfFirstChildInto: anArray
        ifGreaterThanOrEqualTo: aKey
        using: aComparison ]
    ifFalse: [ anArray addLast: 0 ]

]

{ #category : 'Searching Support' }
BtreePlusInteriorNode >> _putLastIndexOfLastChildInto: anArray [
  "Asks the last child in the receiver to
 putLastIndexOfLastChildInto: anArray ."

  | index |
  index := self _lastKeyIndex - self keyIndexOffset.
  (self at: index)
    _putLastIndexOfLastChildInto: anArray.

]

{ #category : 'Updating' }
BtreePlusInteriorNode >> _splitUsingKey: aChildKey node: aChildNode newSibling: newChildSibling [
  "Called when aChileNode itself was split and there wasn't room in the receiver for the
   newChildSibling. Creates a new sibling node of the receiver; moving the second half of the
   receiver's entries into the newSibling; finds the sibling where aChildNode ended up and inserts
   newChildSibling immediately after aChildNode. "

  | insertionIndex newSibling lastKeyIndex lastKey lastVal lastRootObject comparisonForSort |
  newSibling := self _createSibling.
  " get the last key in the receiver "
  lastKeyIndex := self _lastKeyIndex.
  lastKey := self at: lastKeyIndex.
  " get the last value and last root in the new entry's node (this is what insertion is based upon) "
  lastVal := aChildNode lastValue.
  lastRootObject := aChildNode lastRoot.
  " determine if childNode end up in the first half (self) or second half (newSibling)"
  comparisonForSort := self comparisonForSort.
  (self
    compare: comparisonForSort
    key: aChildKey
    value: lastVal
    lessThanOrEqualToEntryAt: lastKeyIndex
    root: lastRootObject)
    ifTrue: [
      "aChildNode ended up in first half"
      insertionIndex := self binarySearchCoveringKey: aChildKey value: lastVal root: lastRootObject.
      "insert newChildSibling after aChildNode"
      self
        _insertKey: (newChildSibling at: newChildSibling _lastKeyIndex)
        value: newChildSibling
        root: newChildSibling lastRoot
        atIndex: insertionIndex + self entrySize ]
    ifFalse: [
      "aChildNode ended up in second half"
       insertionIndex := newSibling binarySearchCoveringKey: aChildKey value: lastVal root: lastRootObject .
      "insert newChildSibling after aChildNode"
      newSibling
        _insertKey: (newChildSibling at: newChildSibling _lastKeyIndex)
        value: newChildSibling
        root: newChildSibling lastRoot
        atIndex: insertionIndex + self entrySize ].
  ^ newSibling

]

{ #category : 'Updating' }
BtreePlusInteriorNode >> _splitUsingKey: aChildKey node: aChildNode newSibling: newChildSibling selectiveAbortSet: selectiveAbortSetOrNil [
  "Called when aChileNode itself was split and there wasn't room in the receiver for the
   newChildSibling. Creates a new sibling node of the receiver; moving the second half of the
   receiver's entries into the newSibling; finds the sibling where aChildNode ended up and inserts
   newChildSibling immediately after aChildNode. "

  | insertionIndex newSibling lastKeyIndex lastKey lastVal lastRootObject comparisonForSort |
  newSibling := self _createSibling: selectiveAbortSetOrNil.
  " get the last key in the receiver "
  lastKeyIndex := self _lastKeyIndex.
  lastKey := self at: lastKeyIndex.
  " get the last value and last root in the new entry's node (this is what insertion is based upon) "
  lastVal := aChildNode lastValue.
  lastRootObject := aChildNode lastRoot.
  " determine if childNode end up in the first half (self) or second half (newSibling)"
  comparisonForSort := self comparisonForSort.
  (self
    compare: comparisonForSort
    key: aChildKey
    value: lastVal
    lessThanOrEqualToEntryAt: lastKeyIndex
    root: lastRootObject
    selectiveAbortSet: selectiveAbortSetOrNil)
    ifTrue: [
      "aChildNode ended up in first half"
      insertionIndex := self binarySearchCoveringKey: aChildKey value: lastVal root: lastRootObject selectiveAbortSet: selectiveAbortSetOrNil.
      "insert newChildSibling after aChildNode"
      self
        _insertKey: (newChildSibling at: newChildSibling _lastKeyIndex)
        value: newChildSibling
        root: newChildSibling lastRoot
        atIndex: insertionIndex + self entrySize
        selectiveAbortSet: selectiveAbortSetOrNil ]
    ifFalse: [
      "aChildNode ended up in second half"
       insertionIndex := newSibling binarySearchCoveringKey: aChildKey value: lastVal root: lastRootObject selectiveAbortSet: selectiveAbortSetOrNil.
      "insert newChildSibling after aChildNode"
      newSibling
        _insertKey: (newChildSibling at: newChildSibling _lastKeyIndex)
        value: newChildSibling
        root: newChildSibling lastRoot
        atIndex: insertionIndex + self entrySize
        selectiveAbortSet: selectiveAbortSetOrNil ].
  ^ newSibling

]

{ #category : 'Updating' }
BtreePlusInteriorNode >> _updateLastValue [
  "Update the last value of the interior node by asking the last entry what its
 last value is.  Do not change the value unless there has been a change."

  | nodeLastValue nodeLastRoot   lastChildNode |
  lastChildNode := self at: self _lastEntryIndex.
  nodeLastValue := lastChildNode lastValue.
  nodeLastValue == self lastValue
    ifFalse: [ self lastValue: nodeLastValue ].
  nodeLastRoot := lastChildNode lastRoot.
  nodeLastRoot == self lastRoot
    ifFalse: [ self lastRoot: nodeLastRoot ]

]

{ #category : 'Updating' }
BtreePlusInteriorNode >> _updateLastValue: selectiveAbortSetOrNil [
  "Update the last value of the interior node by asking the last entry what its
 last value is.  Do not change the value unless there has been a change."

  | nodeLastValue nodeLastRoot   lastChildNode |
  lastChildNode := self at: self _lastEntryIndex.
  self _selectiveAbort: lastChildNode ifNotIn: selectiveAbortSetOrNil.
  nodeLastValue := lastChildNode lastValue.
  nodeLastValue == self lastValue
    ifFalse: [ self lastValue: nodeLastValue ].
  nodeLastRoot := lastChildNode lastRoot.
  nodeLastRoot == self lastRoot
    ifFalse: [ self lastRoot: nodeLastRoot ]

]

{ #category : 'Updating' }
BtreePlusInteriorNode >> _updateLeafNodeLinksForNewPreviousLeaf: newPreviousLeaf newNextLeaf: newNextLeaf [
  " newPreviousNode is being inserted in before of receiver, update the previous/next leaf values for both nodes"

  "currently a noop"

]

{ #category : 'Updating' }
BtreePlusInteriorNode >> _updateLeafNodeLinksForNewPreviousLeaf: newPreviousLeaf newNextLeaf: newNextLeaf  selectiveAbortSet: selectiveAbortSetOrNil [
  " newPreviousNode is being inserted in before of receiver, update the previous/next leaf values for both nodes"

  "currently a noop"

]

{ #category : 'Updating' }
BtreePlusInteriorNode >> at: aKey put: aValue root: rootObject [

  "Adds the key/value/root tuple to one of the leaf nodes.  If a split occurs, returns
   the new sibling, otherwise returns the receiver."

  | index node returnNode |
  index := self binarySearchInteriorNodeCoveringKey: aKey put: aValue root: rootObject.
  node := self at: index.
  returnNode := node at: aKey put: aValue root: rootObject.
  ^ self updateAfterAddAt: index child: node returnNode: returnNode

]

{ #category : 'Rc Updating' }
BtreePlusInteriorNode >> at: aKey put: aValue root: rootObject selectiveAbortSet: selectiveAbortSetOrNil [
  "Adds the key/value/root pair to the node.  If the node is full, performs a 'split'
 on the parent.  Returns the new sibling if a split is performed, otherwise
 returns the receiver."

  | returnNode |
  self _selectiveAbort: self ifNotIn: selectiveAbortSetOrNil.
  returnNode := self _at: aKey put: aValue root: rootObject selectiveAbortSet: selectiveAbortSetOrNil.
  returnNode ~~ self
    ifTrue: [
      self rootNode _addBtreePlusNodeToRcReadSet: returnNode ].
  self rootNode _addBtreePlusNodeToRcReadSet: self.
  ^ returnNode

]

{ #category : 'Audit' }
BtreePlusInteriorNode >> auditForPathTerm: pathTerm using: auditor parentNode: parentNode previousChild: previousSibling nextChild: nextSibling do: aBlock [
  "Audit the receiver's entries and each child node"

  | sentinel previousKey comparisonForSort keyIndexOffset rootIndexOffset lastEntryIndex lastChildNode previousChild nextChild eSize child |
  sentinel := Object new.
  previousKey := sentinel.
  child := previousChild := nextChild := nil.
  previousSibling ifNotNil: [ child := previousSibling at: previousSibling _lastEntryIndex ].
  comparisonForSort := self comparisonForSort.
  keyIndexOffset := self keyIndexOffset.
  rootIndexOffset := self rootIndexOffset.
  lastEntryIndex := self _lastEntryIndex.
  lastChildNode := self at: lastEntryIndex.
  eSize := self entrySize.
  lastChildNode lastValue == self lastValue
    ifFalse: [ auditor
        btreeInteriorNode: self
        childConsistencyError: pathTerm
        lastValue: self lastValue
        lastChildValue: lastChildNode lastValue
        childNode: lastChildNode ].
  lastChildNode lastRoot == self lastRoot
    ifFalse: [ auditor
        btreeInteriorNode: self
        childConsistencyError: pathTerm
        lastRoot: self lastRoot
        lastChildRoot: lastChildNode lastRoot
        childNode: lastChildNode].
  1 to: numElements * self entrySize by: self entrySize do: [ :i | | k r lastChildKey lastChildRoot |
    k := self at: i + keyIndexOffset.
    r := self at: i + rootIndexOffset.
    previousKey == sentinel
      ifFalse: [ (comparisonForSort compareKey: previousKey lessThanOrEqualTo: k)
          ifFalse: [ auditor
              btreeInteriorNode: self
              keyOrderError: pathTerm
              key: k
              previousKey: previousKey
              index: i ] ].
    previousKey := k.
    previousChild := child.
    (i + eSize) <= lastEntryIndex
      ifTrue: [ nextChild := self at: i + eSize ]
      ifFalse: [
        nextChild := nil.
        nextSibling ifNotNil: [ nextChild := nextSibling at: 1 ] ].
    child := self at: i.
    lastChildKey := child at: child _lastKeyIndex.
    lastChildKey == k
      ifFalse: [ auditor
          btreeInteriorNode: self
          childConsistencyError: pathTerm
          key: k
          lastChildKey: lastChildKey
          childNode: child
          index: i ].
    lastChildRoot := child at: child _lastRootIndex.
    lastChildRoot == r
      ifFalse: [ auditor
          btreeInteriorNode: self
          childConsistencyError: pathTerm
          key: k
          root: r
          lastChildRoot: lastChildRoot
          childNode: child
          index: i ].
    child auditForPathTerm: pathTerm using: auditor parentNode: self previousChild: previousChild nextChild: nextChild do: aBlock ]

]

{ #category : 'Searching' }
BtreePlusInteriorNode >> binarySearchCoveringKey: aKey value: aValue root: rootObject selectiveAbortSet: selectiveAbortSetOrNil [
  "Returns the index for the entry in which aKey/aValue would be inserted.  This
 is the first entry which has its key >= aKey.  If multiple entries have the
 same key (key = aKey), then aValue is used (since entries with duplicate keys
 are inserted by the value's OOP ordering).  If multiple entries have the same value
 (value = aValue), then rootObject is used (since entries with duplicate values are
 inserted by the rootObject's OOP ordering). If the receiver is empty, returns
 1.  Uses a binary search."

  | lowPt midPt highPt entrySize index entryGreaterThan entryEqualTo comparisonForSort keyIndexOffset |
  entrySize := self entrySize.
  lowPt := 1.
  highPt := numElements.
  entryGreaterThan := false.
  comparisonForSort := self comparisonForSort.
  keyIndexOffset := self keyIndexOffset.
  [ lowPt <= highPt ]
    whileTrue: [
      | objAtIndex |
      entryGreaterThan := false.
      entryEqualTo := false.
      midPt := lowPt + highPt quo: 2.
      index := (midPt - 1) * entrySize + 1 + keyIndexOffset.
      objAtIndex := self at: index.
      (comparisonForSort compareKey: aKey lessThan: objAtIndex)
        ifTrue: [ highPt := midPt - 1 ]
        ifFalse: [
          (comparisonForSort compareKey: aKey greaterThan: objAtIndex)
            ifTrue: [
              lowPt := midPt + 1.
              entryGreaterThan := true ]
            ifFalse: [
              "If keys are equal, then secondary sort key is oop value, tertiary sort key os oop of rootObject"
              (comparisonForSort compareKey: aKey equalTo: objAtIndex)
                ifTrue: [
                  (self
                    compareValueOop: aValue
                    lessThanOrEqualToEntryValueOopAt: index
                    root: rootObject
                    selectiveAbortSet: selectiveAbortSetOrNil)
                    ifTrue: [ highPt := midPt - 1 ]
                    ifFalse: [ lowPt := midPt + 1 ].
                  entryEqualTo := true ]
                ifFalse: [ lowPt := midPt + 1 ] ] ] ].
  entryGreaterThan
    ifTrue: [ ^ index + entrySize - keyIndexOffset ].
  (entryEqualTo
    and: [
      "use secondary sort key (oop of value), tertiary sort key os oop of rootObject"
      self
        compareValueOop: aValue
        greaterThanEntryValueOopAt: index
        root: rootObject
        selectiveAbortSet: selectiveAbortSetOrNil ])
    ifTrue: [ ^ index + entrySize - keyIndexOffset ].
  ^ index - keyIndexOffset

]

{ #category : 'Searching' }
BtreePlusInteriorNode >> binarySearchCoveringKey: aKey value: aValue selectiveAbortSet: selectiveAbortSetOrNil [
  "Returns the index for the entry in which aKey/aValue would be inserted.  This
 is the first entry which has its key >= aKey.  If multiple entries have the
 same key (key = aKey), then aValue is used (since entries with duplicate keys
 are inserted by the value's OOP ordering). If the receiver is empty, returns
 1.  Uses a binary search."

  | lowPt midPt highPt entrySize index entryGreaterThan entryEqualTo comparisonForSort keyIndexOffset |
  entrySize := self entrySize.
  lowPt := 1.
  highPt := numElements.
  entryGreaterThan := false.
  comparisonForSort := self comparisonForSort.
  keyIndexOffset := self keyIndexOffset.
  [ lowPt <= highPt ] whileTrue: [ | objAtIndex |
      entryGreaterThan := false.
      entryEqualTo := false.
      midPt := lowPt + highPt quo: 2.
      index := (midPt - 1) * entrySize + 1 + keyIndexOffset.
      objAtIndex := self at: index.
      (comparisonForSort compareKey: aKey lessThan: objAtIndex)
        ifTrue: [ highPt := midPt - 1 ]
        ifFalse: [ (comparisonForSort compareKey: aKey greaterThan: objAtIndex)
            ifTrue: [ lowPt := midPt + 1.
              entryGreaterThan := true ]
            ifFalse: [ "If keys are equal, then secondary sort key is oop value, tertiary sort key os oop of rootObject"
              (comparisonForSort compareKey: aKey equalTo: objAtIndex)
                ifTrue: [ (self
                    compareValueOop: aValue
                    lessThanOrEqualToEntryValueOopAt: index
                    selectiveAbortSet: selectiveAbortSetOrNil)
                    ifTrue: [ highPt := midPt - 1 ]
                    ifFalse: [ lowPt := midPt + 1 ].
                  entryEqualTo := true ]
                ifFalse: [ lowPt := midPt + 1 ] ] ] ].
  entryGreaterThan
    ifTrue: [ ^ index + entrySize - keyIndexOffset ].
  (entryEqualTo
    and: [ "use secondary sort key (oop of value), tertiary sort key os oop of rootObject"
      self
        compareValueOop: aValue
        greaterThanEntryValueOopAt: index
        selectiveAbortSet: selectiveAbortSetOrNil ])
    ifTrue: [ ^ index + entrySize - keyIndexOffset ].
  ^ index - keyIndexOffset

]

{ #category : 'Updating' }
BtreePlusInteriorNode >> binarySearchInteriorNodeCoveringKey: aKey put: aValue root: rootObject [

  "Find the index of the first child node in which the insertion would occur"

  | index lastKeyIndex |
  index := self binarySearchCoveringKey: aKey value: aValue root: rootObject.
  lastKeyIndex := self _lastKeyIndex.
  " if insertion would occur past the last entry, use the last entry "
  index > lastKeyIndex
    ifTrue: [ index := lastKeyIndex - self keyIndexOffset ].
  ^ index.

]

{ #category : 'Updating' }
BtreePlusInteriorNode >> binarySearchInteriorNodeCoveringKey: aKey put: aValue root: rootObject selectiveAbortSet: selectiveAbortSetOrNil [

  "Find the index of the first child node in which the insertion would occur"

  | index lastKeyIndex |
  index := self binarySearchCoveringKey: aKey value: aValue root: rootObject  selectiveAbortSet: selectiveAbortSetOrNil.
  lastKeyIndex := self _lastKeyIndex.
  " if insertion would occur past the last entry, use the last entry "
  index > lastKeyIndex
    ifTrue: [ index := lastKeyIndex - self keyIndexOffset ].
  ^ index.

]

{ #category : 'Comparison Operators' }
BtreePlusInteriorNode >> compare: comparisonForSort key: aKey value: aValue equalToEntryAt: keyIndex [
  "Performs a = comparison between aKey and the entry whose key is at the given
 index.  If the keys are equal, use the OOP of the value as the next basis for
 comparison."

  | entryIndex |
  entryIndex := keyIndex - self keyIndexOffset.
  ^ (comparisonForSort compareKey: aKey equalTo: (self at: keyIndex))
    and: [
      "aValue asOop = (self at: entryIndex) lastValue asOop"
      aValue == (self at: entryIndex) lastValue ]

]

{ #category : 'Comparison Operators' }
BtreePlusInteriorNode >> compare: comparisonForSort key: aKey value: aValue equalToEntryAt: keyIndex root: rootObject [
  "Performs a = comparison between aKey and the entry whose key is at the given
 index.  If the keys are equal, use the OOP of the value as the next basis for
 comparison. If the value oops are equeal, use the oop of the rootObject as the
 next basis for comparison."

  ^ (comparisonForSort compareKey: aKey equalTo: (self at: keyIndex))
    and: [ | entryIndex |
      entryIndex := keyIndex - self keyIndexOffset.
      "aValue asOop = (self at: entryIndex) lastValue asOop"
      aValue == (self at: entryIndex) lastValue
        and: [
          "rootObject asOop = (self at: entryIndex) lastRoot asOop"
          rootObject == (self at: entryIndex) lastRoot ] ]

]

{ #category : 'Comparison Operators' }
BtreePlusInteriorNode >> compare: comparisonForSort key: aKey value: aValue lessThanOrEqualToEntryAt: keyIndex [
  "Performs a <= comparison between aKey and the entry whose key is at the given
 index. If the keys are equal, then use the OOP of the value for the
 comparison."

  | keyIndexEntry |
  keyIndexEntry := self at: keyIndex.
  (comparisonForSort compareKey: aKey lessThan: keyIndexEntry)
    ifTrue: [ ^ true ].
  (comparisonForSort compareKey: aKey equalTo: keyIndexEntry)
    ifTrue: [ | entryIndex |
      entryIndex := keyIndex - self keyIndexOffset.
      ^ aValue asOop <= (self at: entryIndex) lastValue asOop ].
  ^ false

]

{ #category : 'Comparison Operators' }
BtreePlusInteriorNode >> compare: comparisonForSort key: aKey value: aValue lessThanOrEqualToEntryAt: keyIndex root: rootObject [
  "Performs a <= comparison between aKey and the entry whose key is at the given
 index.  If the keys are equal, then use the OOP of the value for the
 comparison. If the value oops are equal, use the OOP of the rootObject for the
 comparison."

  | keyIndexEntry |
  keyIndexEntry := self at: keyIndex.
  (comparisonForSort compareKey: aKey lessThan: keyIndexEntry)
    ifTrue: [ ^ true ].
  (comparisonForSort compareKey: aKey equalTo: keyIndexEntry)
    ifTrue: [ | entryIndexEntry entryIndexEntryLastEntry |
      entryIndexEntry := self at: keyIndex - self keyIndexOffset.
      "aValue asOop = entryIndexEntry lastValue asOop"
      entryIndexEntryLastEntry := entryIndexEntry lastValue.
      aValue == entryIndexEntryLastEntry
        ifTrue: [ ^ rootObject asOop <= entryIndexEntry lastRoot asOop ].
      ^ aValue asOop < entryIndexEntryLastEntry asOop ].
  ^ false

]

{ #category : 'Comparison Operators' }
BtreePlusInteriorNode >> compare: comparisonForSort key: aKey value: aValue lessThanOrEqualToEntryAt: keyIndex root: rootObject selectiveAbortSet: selectiveAbortSetOrNil [
  "Performs a <= comparison between aKey and the entry whose key is at the given
 index.  If the keys are equal, then use the OOP of the value for the
 comparison. If the value oops are equal, use the OOP of the rootObject for the
 comparison."

  | keyIndexEntry |
  keyIndexEntry := self at: keyIndex.
  (comparisonForSort compareKey: aKey lessThan: keyIndexEntry)
    ifTrue: [ ^ true ].
  (comparisonForSort compareKey: aKey equalTo: keyIndexEntry)
    ifTrue: [ | entryIndexEntry entryIndexEntryLastEntry |
      entryIndexEntry := self at: keyIndex - self keyIndexOffset.
      self _selectiveAbort: entryIndexEntry ifNotIn: selectiveAbortSetOrNil.
      "aValue asOop = entryIndexEntry lastValue asOop"
      entryIndexEntryLastEntry := entryIndexEntry lastValue.
      aValue == entryIndexEntryLastEntry
        ifTrue: [ ^ rootObject asOop <= entryIndexEntry lastRoot asOop ].
      ^ aValue asOop < entryIndexEntryLastEntry asOop ].
  ^ false

]

{ #category : 'Comparison Operators' }
BtreePlusInteriorNode >> compare: comparisonForSort key: aKey value: aValue lessThanOrEqualToEntryAt: keyIndex root: rootObject selectiveAbortSetOrNil: selectiveAbortSetOrNil [
  "Performs a <= comparison between aKey and the entry whose key is at the given
 index.  If the keys are equal, then use the OOP of the value for the
 comparison. If the value oops are equal, use the OOP of the rootObject for the
 comparison."

  | keyIndexEntry |
  keyIndexEntry := self at: keyIndex.
  (comparisonForSort compareKey: aKey lessThan: keyIndexEntry)
    ifTrue: [ ^ true ].
  (comparisonForSort compareKey: aKey equalTo: keyIndexEntry)
    ifTrue: [ | entryIndexEntry entryIndexEntryLastEntry |
      entryIndexEntry := self at: keyIndex - self keyIndexOffset.
      self _selectiveAbort: entryIndexEntry ifNotIn: selectiveAbortSetOrNil.
      "aValue asOop = entryIndexEntry lastValue asOop"
      entryIndexEntryLastEntry := entryIndexEntry lastValue.
      aValue == entryIndexEntryLastEntry
        ifTrue: [ ^ rootObject asOop <= entryIndexEntry lastRoot asOop ].
      ^ aValue asOop < entryIndexEntryLastEntry asOop ].
  ^ false

]

{ #category : 'Comparison Operators' }
BtreePlusInteriorNode >> compareLastKeyLessThanOrEqualToKey: aKey value: aValue root: rootObject [
  "Returns whether the receivers last key/value pair is greater than or equal to
the given key/value pair."

  numElements == 0
    ifTrue: [ ^ false ].
  " return aKey <= last key "
  ^ self
    compare: self comparisonForSort
    key: aKey
    value: aValue
    lessThanOrEqualToEntryAt: self _lastKeyIndex
    root: rootObject

]

{ #category : 'Comparison Operators' }
BtreePlusInteriorNode >> compareLastKeyLessThanOrEqualToKey: aKey value: aValue root: rootObject selectiveAbortSet: selectiveAbortSetOrNil [
  "Returns whether the receivers last key/value pair is greater than or equal to
the given key/value pair."

  numElements == 0
    ifTrue: [ ^ false ].
  " return aKey <= last key "
  ^ self
    compare: self comparisonForSort
    key: aKey
    value: aValue
    lessThanOrEqualToEntryAt: self _lastKeyIndex
    root: rootObject
    selectiveAbortSet: selectiveAbortSetOrNil

]

{ #category : 'Comparison Operators' }
BtreePlusInteriorNode >> compareValueOop: aValue greaterThanEntryValueOopAt: keyIndex [
  "Perform a > comparison between the oop of aValue and oop of the entry whose value is
   at the given index."

  ^ aValue asOop > (self at: keyIndex - self keyIndexOffset) lastValue asOop

]

{ #category : 'Comparison Operators' }
BtreePlusInteriorNode >> compareValueOop: aValue greaterThanEntryValueOopAt: keyIndex selectiveAbortSet: selectiveAbortSetOrNil [
  "Perform a > comparison between the oop of aValue and oop of the entry whose value is
   at the given index."

  | entryNode |
  entryNode := self at: keyIndex - self keyIndexOffset.
  self _selectiveAbort: entryNode ifNotIn: selectiveAbortSetOrNil.
  ^ aValue asOop > entryNode lastValue asOop

]

{ #category : 'Comparison Operators' }
BtreePlusInteriorNode >> compareValueOop: aValue lessThanOrEqualToEntryValueOopAt: keyIndex [
  "Perform a <= comparison between the oop of aValue and oop of the entry whose value is
   at the given index."

  ^ aValue asOop <= (self at: keyIndex - self keyIndexOffset) lastValue asOop

]

{ #category : 'Comparison Operators' }
BtreePlusInteriorNode >> compareValueOop: aValue lessThanOrEqualToEntryValueOopAt: keyIndex selectiveAbortSet: selectiveAbortSetOrNil [
  "Perform a <= comparison between the oop of aValue and oop of the entry whose value is
   at the given index."

  | entryNode |
  entryNode := self at: keyIndex - self keyIndexOffset.
  self _selectiveAbort: entryNode ifNotIn: selectiveAbortSetOrNil.
  ^ aValue asOop <= entryNode lastValue asOop

]

{ #category : 'Accessing' }
BtreePlusInteriorNode >> lastRoot [
  ^ lastRoot

]

{ #category : 'Updating' }
BtreePlusInteriorNode >> lastRoot: anObject [
  lastRoot := anObject

]

{ #category : 'Accessing' }
BtreePlusInteriorNode >> lastValue [

"Returns the value of the instance variable 'lastValue'."

^ lastValue

]

{ #category : 'Updating' }
BtreePlusInteriorNode >> lastValue: newValue [

"Modifies the value of the instance variable 'lastValue'."

lastValue := newValue

]

{ #category : 'Accessing' }
BtreePlusInteriorNode >> nextLeaf [
  "BtreePlusLeafNode compat"

  ^ nil

]

{ #category : 'Accessing' }
BtreePlusInteriorNode >> nextLeaf: ignored [
  "BtreePlusLeafNode compat - noop"

]

{ #category : 'Constants' }
BtreePlusInteriorNode >> parentNodeClass [
  "Returns the class of node to be created as the parent when a split occurs."

  ^ self class

]

{ #category : 'Accessing' }
BtreePlusInteriorNode >> previousLeaf [
  "BtreePlusLeafNode compat"

  ^ nil

]

{ #category : 'Accessing' }
BtreePlusInteriorNode >> previousLeaf:  ignored [
  "BtreePlusLeafNode compat - noop"

]

{ #category : 'Updating' }
BtreePlusInteriorNode >> primitiveAt: anIndex put: aValue [

"Stores the argument aValue in the indexed variable of the
 receiver indicated by anIndex.  The argument anIndex must not be
 larger than 1 + the size of the receiver, and must not be less than 1.

 Generates an error if anIndex is not a SmallInteger or is out of
 bounds, if the receiver is not indexable, or if the
 receiver is not of the right class to store the given value.

 The primitive is equivalent to GciStoreIdxOop or GciStoreByte,
 depending on implementation of the receiver."

<primitive: 268>

(anIndex _isInteger)
  ifFalse: [ ^ self _errorNonIntegerIndex: anIndex].
(self class isIndexable) "not an indexable object"
  ifFalse: [ ^ self _errorNotIndexable].
(anIndex < 1 or:[ anIndex > (self size + 1)]) "out of bounds"
  ifTrue: [ ^ self _errorIndexOutOfRange: anIndex].
(self class isBytes)
ifTrue: [ ((aValue class ~~ SmallInteger) or: [ (aValue < 0) | (aValue > 255) ])
    ifTrue: [^ aValue _error: #rtErrExpectedByteValue].
  ].

self _primitiveFailed: #at:put: args: { anIndex . aValue }.
self _uncontinuableError

]

{ #category : 'Printing' }
BtreePlusInteriorNode >> printOn: aStream withIndentationLevel: anInteger [

"Puts a displayable representation of the keys of the receiver on the given
 stream, indenting the given number of spaces."

anInteger timesRepeat: [ aStream nextPut: Character space ].
aStream nextPutAll: '[ ';
    nextPutAll: self lastValue asString;
    nextPutAll: ' ';
    nextPutAll: self lastRoot asString;
    nextPutAll: ' ]';

    nextPut: Character lf.

1 to: (numElements * self entrySize) by: self entrySize do: [ :i |
    anInteger timesRepeat: [ aStream nextPut: Character space ].
    aStream nextPutAll: (self at: (i + self keyIndexOffset)) asString; nextPut: Character lf.
    (self at: i) printOn: aStream withIndentationLevel: (anInteger + 4)
]

]

{ #category : 'Printing' }
BtreePlusInteriorNode >> printOn: aStream withIndentationLevel: anInteger includingValues: aBoolean [

"Puts a displayable representation of the keys of the receiver on the given
 stream, indenting the given number of spaces."

aBoolean
    ifTrue: [
        anInteger timesRepeat: [ aStream nextPut: Character space ].
        aStream nextPutAll: '[ ';
            nextPutAll: self lastValue asString;
            nextPutAll: ' ';
            nextPutAll: self lastRoot asString;
            nextPutAll: ' ]';
            "nextPutAll: '  { '; nextPutAll: self _alias asString; nextPutAll: ' }';"
            nextPut: Character lf.
    ].

1 to: (numElements * self entrySize) by: self entrySize do: [ :i |
    anInteger timesRepeat: [ aStream nextPut: Character space ].
    aStream nextPutAll: (self at: (i + 1)) asString; nextPut: Character lf.
    (self at: i) printOn: aStream withIndentationLevel: (anInteger + 4) includingValues: aBoolean
]

]

{ #category : 'Removing' }
BtreePlusInteriorNode >> removedEntryFromNode: node atIndex: nodeIndex [
  "Remove occurred, check if a merge or coalesce is necessary."

  | keyIndexOffset lastChildKeyIndex lastChildKey lastChildRoot |
  keyIndexOffset := self keyIndexOffset.
  " see if last key in child node has changed "
  lastChildKeyIndex := node _lastKeyIndex.
  lastChildKey := node at: lastChildKeyIndex.
  lastChildKey == (self at: nodeIndex + keyIndexOffset)
    ifFalse: [ " update key and encryption in the receiver "
      self primitiveAt: nodeIndex + keyIndexOffset put: lastChildKey ].
  " see if the node's number of entries is below the threshold "
  node numElements < node mergeThreshold
    ifTrue: [ " try to steal entries from a neighboring sibling "
      (self _attemptTakingFromNeighborForNodeAt: nodeIndex)
        ifFalse: [ " could not take from neighbor, merge with a neighbor instead. lastChildRoot and lastValue updated"
          self _mergeFromNeighborForNodeAt: nodeIndex.
          ^ self  ] ].
  " see if last root in child node has changed "
  lastChildRoot := node at: node _lastRootIndex.
  self updateLastChildRoot: lastChildRoot at: nodeIndex.
  self _updateLastValue.

]

{ #category : 'Rc Removing' }
BtreePlusInteriorNode >> removedEntryFromNode: node atIndex: nodeIndex selectiveAbortSet: selectiveAbortSetOrNil [
  "Remove occurred, check if a merge or coalesce is necessary."

  | keyIndexOffset lastChildKeyIndex lastChildKey lastChildRoot |
  keyIndexOffset := self keyIndexOffset.
  " see if last key in child node has changed "
  lastChildKeyIndex := node _lastKeyIndex.
  lastChildKey := node at: lastChildKeyIndex.
  lastChildKey == (self at: nodeIndex + keyIndexOffset)
    ifFalse: [ " update key and encryption in the receiver "
      self primitiveAt: nodeIndex + keyIndexOffset put: lastChildKey ].
  " see if the node's number of entries is below the threshold "
  node numElements < node mergeThreshold
    ifTrue: [ " try to steal entries from a neighboring sibling "
      (self _attemptTakingFromNeighborForNodeAt: nodeIndex selectiveAbortSet: selectiveAbortSetOrNil)
        ifFalse: [ " could not take from neighbor, merge with a neighbor instead. lastChildRoot and lastValue updated"
          self _mergeFromNeighborForNodeAt: nodeIndex selectiveAbortSet: selectiveAbortSetOrNil.
          ^ self  ] ].
  " see if last root in child node has changed "
  lastChildRoot := node at: node _lastRootIndex.
  self updateLastChildRoot: lastChildRoot at: nodeIndex.
  self _updateLastValue: selectiveAbortSetOrNil.

]

{ #category : 'Removing' }
BtreePlusInteriorNode >> removeKey: aKey value: aValue root: rootObject [
  "Removes the key/value pair.  Find the appropriate child node and invoke the
 remove operation on it.  If the remove occurred, check if a merge or coalesce
 is necessary."

  | nodeIndex node |
  (self compareLastKeyLessThanOrEqualToKey: aKey value: aValue root: rootObject)
    ifFalse: [ ^ false ].
  nodeIndex := self
    binarySearchCoveringKey: aKey
    value: aValue
    root: rootObject.
  node := self at: nodeIndex.
  ^ (node removeKey: aKey value: aValue root: rootObject)
    ifTrue: [ " remove occurred - merge or coalesce if necessary"
      self removedEntryFromNode: node atIndex: nodeIndex.
      true ]
    ifFalse: [ false ]

]

{ #category : 'Rc Removing' }
BtreePlusInteriorNode >> removeKey: aKey value: aValue root: rootObject selectiveAbortSet: selectiveAbortSetOrNil [
  "Removes the key/value pair.  Find the appropriate child node and invoke the
 remove operation on it.  If the remove occurred, check if a merge or coalesce
 is necessary."

  | res nodeIndex node |
  self _selectiveAbort: self ifNotIn: selectiveAbortSetOrNil.
   (self compareLastKeyLessThanOrEqualToKey: aKey value: aValue root: rootObject selectiveAbortSet: selectiveAbortSetOrNil)
    ifFalse: [ ^ false ].
  nodeIndex := self
    binarySearchCoveringKey: aKey
    value: aValue
    root: rootObject
    selectiveAbortSet: selectiveAbortSetOrNil.
  node := self at: nodeIndex.
  self _selectiveAbort: node ifNotIn: selectiveAbortSetOrNil.
  res := (node removeKey: aKey value: aValue root: rootObject selectiveAbortSet: selectiveAbortSetOrNil)
    ifTrue: [ " remove occurred - merge or coalesce if necessary"
      self removedEntryFromNode: node atIndex: nodeIndex selectiveAbortSet: selectiveAbortSetOrNil.
      true ]
    ifFalse: [ false ].
  self rootNode _addBtreePlusNodeToRcReadSet: self.
  ^ res

]

{ #category : 'Accessing' }
BtreePlusInteriorNode >> totalElements [

"Return the total number of elements in the leaf nodes."

| sum |

sum := 0.
1 to: (numElements * self entrySize) by: self entrySize do: [ :i |
    sum := sum + (self at: i) totalElements
].
^ sum

]

{ #category : 'Updating' }
BtreePlusInteriorNode >> updateAfterAddAt: index child: node returnNode: returnNode [

  "Adds the key/value/root tuple to one of the leaf nodes.  If a split occurs, returns
   the new sibling, otherwise returns the receiver."

  | lastChildKey keyIndexOffset lastChildRoot |

  " see if last key in child node has changed "
  lastChildKey := node at: node _lastKeyIndex.
  keyIndexOffset := self keyIndexOffset.
  lastChildKey == (self at: index + keyIndexOffset)
    ifFalse: [ " update key and encryption in the receiver "
      self primitiveAt: index + keyIndexOffset put: lastChildKey ].
  " see if last root in child node has changed "
  lastChildRoot := node at: node _lastRootIndex.
  self updateLastChildRoot: lastChildRoot at: index.


  " see if a split occurred "
  returnNode == node
    ifTrue: [
      " if inserted into the last node, update the last value "
      index == self _lastEntryIndex
        ifTrue: [ self _updateLastValue ] ]
    ifFalse: [ " returnNode is the second half of the split "
      " determine if a split is needed in the receiver "
      numElements == self maxNumberOfElements
        ifTrue: [ | newSibling |
          newSibling := self
            _splitUsingKey: (node at: node _lastKeyIndex)
            node: node
            newSibling: returnNode.

          " update the last value and last root of the new sibling and the receiver "
          newSibling _updateLastValue.
          self _updateLastValue.
          ^ newSibling ]
        ifFalse: [ " entry will fit in the receiver "

          " if child node was the last node "
          index == self _lastEntryIndex
            ifTrue: [
              self lastValue: returnNode lastValue.
              self lastRoot: returnNode lastRoot ].
          self
            _insertKey: (returnNode at: returnNode _lastKeyIndex)
            value: returnNode
            root: (returnNode at: returnNode _lastRootIndex)
            atIndex: index + self entrySize.
          " see if last root in child node has changed "
          lastChildRoot := returnNode at: returnNode _lastRootIndex.
          self updateLastChildRoot: lastChildRoot at: index + self entrySize ] ]

]

{ #category : 'Updating' }
BtreePlusInteriorNode >> updateAfterAddAt: index child: node returnNode: returnNode selectiveAbortSet: selectiveAbortSetOrNil [

  "Adds the key/value/root tuple to one of the leaf nodes.  If a split occurs, returns
   the new sibling, otherwise returns the receiver."

  "node already selectively aborted"

  | lastChildKey keyIndexOffset lastChildRoot |

  " see if last key in child node has changed "
  lastChildKey := node at: node _lastKeyIndex.
  keyIndexOffset := self keyIndexOffset.
  lastChildKey == (self at: index + keyIndexOffset)
    ifFalse: [ " update key and encryption in the receiver "
      self primitiveAt: index + keyIndexOffset put: lastChildKey ].
  " see if last root in child node has changed "
  lastChildRoot := node at: node _lastRootIndex.
  self updateLastChildRoot: lastChildRoot at: index.


  " see if a split occurred "
  returnNode == node
    ifTrue: [
      " if inserted into the last node, update the last value "
      index == self _lastEntryIndex
        ifTrue: [ self _updateLastValue: selectiveAbortSetOrNil ] ]
    ifFalse: [ " returnNode is the second half of the split "
      " determine if a split is needed in the receiver "
      numElements == self maxNumberOfElements
        ifTrue: [ | newSibling |
          newSibling := self
            _splitUsingKey: (node at: node _lastKeyIndex)
            node: node
            newSibling: returnNode
            selectiveAbortSet: selectiveAbortSetOrNil.

          " update the last value and last root of the new sibling and the receiver "
          newSibling _updateLastValue: selectiveAbortSetOrNil.
          self _updateLastValue: selectiveAbortSetOrNil.
          self rootNode _addBtreePlusNodeToRcReadSet: newSibling.
          ^ newSibling ]
        ifFalse: [ " entry will fit in the receiver "

          " if child node was the last node "
          index == self _lastEntryIndex
            ifTrue: [
              self lastValue: returnNode lastValue.
              self lastRoot: returnNode lastRoot ].
          self
            _insertKey: (returnNode at: returnNode _lastKeyIndex)
            value: returnNode
            root: (returnNode at: returnNode _lastRootIndex)
            atIndex: index + self entrySize
            selectiveAbortSet: selectiveAbortSetOrNil.
          " see if last root in child node has changed "
          lastChildRoot := returnNode at: returnNode _lastRootIndex.
          self updateLastChildRoot: lastChildRoot at: index + self entrySize ] ]

]

{ #category : 'Updating' }
BtreePlusInteriorNode >> updateLastChildRoot: lastChildRoot at: index [

  " Update the root object for receiver, if last child root has changed "

  | rootIndexOffset |
  rootIndexOffset := self rootIndexOffset.
  lastChildRoot == (self at: index + rootIndexOffset)
    ifFalse: [  self primitiveAt: index + rootIndexOffset put: lastChildRoot ].

]
