Extension { #name : 'BtreePlusNode' }

{ #category : 'Constants' }
BtreePlusNode class >> entrySize [
  "Returns the size of an entry."

  "in-lined on instance-side for performance"

  self subclassResponsibility: #'entrySize'

]

{ #category : 'Instance Creation' }
BtreePlusNode class >> new [

"Returns a new initialized instance with the correct size."

^ (super new: self entrySize * self maxNumberOfElements) numElements: 0.

]

{ #category : 'Query Support' }
BtreePlusNode >> _addRootObjects: aQuerySpec
start: startIndex
end: endIndex
into: collection [

^aQuerySpec
    _addRootObjectsFrom: self
    entrySize: self entrySize
    start: startIndex
    end: endIndex
    into: collection

]

{ #category : 'Testing' }
BtreePlusNode >> _basicCanCompare: aKey withClass: aClass [
  "Returns true if the receiver can make comparisons with the given key, and
false otherwise.  Used for basic nodes."

  ^ aKey _idxBasicPlusCanCompareWithClass: aClass

]

{ #category : 'Testing' }
BtreePlusNode >> _canCompare: aKey withClass: aClass [
  "Returns whether the receiver can make comparisons with the given key."

  ^ self indexObject optimizingComparison
    ifTrue: [ self _canCompareOptimized: aKey withClass: aClass ]
    ifFalse: [ aKey == nil
        ifTrue: [ ^ true ].
      self _canCompareStandard: aKey withClass: aClass ]

]

{ #category : 'Testing' }
BtreePlusNode >> _canCompareOptimized: aKey withClass: aClass [
  "Returns whether the receiver can make comparisons with the given key.
If optimized compare okay, apply standard rules."

  (aKey _idxOptimizedCompareWithClass: aClass) ifFalse: [ ^ false ].
  ^ self _canCompareStandard: aKey withClass: aClass

]

{ #category : 'Testing' }
BtreePlusNode >> _canCompareStandard: aKey withClass: aClass [
  "Returns whether the receiver can make comparisons with the given key."

  (GsRangeEqualityIndex isBasicClass: aClass)
    ifTrue: [
      "preserve legacy index constraints for basic index classes"
      ^ self _basicCanCompare: aKey withClass: aClass ].
  ^ true

]

{ #category : 'Updating' }
BtreePlusNode >> _createSibling [
  "Creates a new sibling node and move the second half of the receiver's entries into the
sibling.  Returns the new sibling."

  | newSibling sz splitIndex halfNumElements |
  newSibling := self class new.
  newSibling
    objectSecurityPolicy: self objectSecurityPolicy;
    rootNode: self rootNode.
  sz := numElements * self entrySize.
  halfNumElements := numElements quo: 2.
  splitIndex := halfNumElements * self entrySize + 1.	" splitIndex is where to split the receiver "
  newSibling
    _insertAt: 1
    from: self
    fromStart: splitIndex
    fromEnd: sz
    numToMoveDown: 0.	" copy the last half of entries into the new sibling "
  self _deleteNoShrinkFrom: splitIndex to: sz.	" nil out last half of receiver "
  newSibling numElements: numElements - halfNumElements.	" update the element counts in the receiver..."
  numElements := halfNumElements.	"... and the sibling "
  ^ newSibling

]

{ #category : 'Updating' }
BtreePlusNode >> _createSibling: selectiveAbortSetOrNil [
  "Creates a new sibling node and move the second half half of the receiver's entries into the
sibling.  Returns the new sibling."

  ^ self _createSibling

]

{ #category : 'Searching' }
BtreePlusNode >> _findAllValuesForIdenticalKey: aKey into: aCollection [
  "Finds all the values for the given identical key, placing them in the
 collection.  Return whether the last key was equal or not (so the caller
 knows it must check the next leaf node)."

  ^ self
    _findAllValuesForIdenticalKey: aKey
    into: aCollection
    using: self comparisonForSort

]

{ #category : 'Searching' }
BtreePlusNode >> _findCoveringIndexForKey: aKey totalOrder: bool1 [
  "Returns the index for the first entry in which aKey is found.  This is the
first entry whose key >= aKey.  If the receiver's last key is greater than
aKey, or if the receiver is empty, returns 0.  Uses a binary search."

  ^ self _findCoveringIndexForKey: aKey totalOrder: bool1 using: self comparisonForSort

]

{ #category : 'Searching' }
BtreePlusNode >> _findCoveringIndexForKey: aKey totalOrder: bool1 using: aComparison [
  "Returns the index for the first entry in which aKey is found.  This is the
first entry whose key >= aKey.  If the receiver's last key is greater than
aKey, or if the receiver is empty, returns 0.  Uses a binary search."

  | index |
  numElements == 0
    ifTrue: [ ^ 0 ].
  index := self binarySearchCoveringKey: aKey totalOrder: bool1  using: aComparison.	" check if not found "
  ^ index > self _lastKeyIndex
    ifTrue: [ 0 ]
    ifFalse: [ index ]

]

{ #category : 'Accessing' }
BtreePlusNode >> _lastEntryIndex [
  "Returns the index of the last entry in the node, zero if the node is empty."

  numElements == 0
    ifTrue: [ ^ 0 ].
  ^ (numElements - 1) * self entrySize + 1

]

{ #category : 'Accessing' }
BtreePlusNode >> _lastIndex [

"Returns the index of the last used slot in the node."

^ numElements * self entrySize

]

{ #category : 'Accessing' }
BtreePlusNode >> _lastKeyIndex [
  "Returns the index of the key of the last entry, or zero if the node is empty."

  ^ numElements == 0
    ifTrue: [ 0 ]
    ifFalse: [ self _lastEntryIndex + self keyIndexOffset ]

]

{ #category : 'Accessing' }
BtreePlusNode >> _lastRootIndex [
  "Returns the index of the root object of the last entry, or zero if the node is empty."

  ^ numElements == 0
    ifTrue: [ 0 ]
    ifFalse: [ self _lastEntryIndex + self rootIndexOffset ]

]

{ #category : 'Audit' }
BtreePlusNode >> _primEncodingFor: aKey [

"Get the encryption for the given value and return it
 in a 2 slot Array."

<primitive: 871>
self _primitiveFailed: #'_encodingFor:' args: { aKey }

]

{ #category : 'Rc Updating' }
BtreePlusNode >> _selectiveAbort: aNode ifNotIn: selectiveAbortSetOrNil [

  (selectiveAbortSetOrNil ~~ nil
    and: [ (selectiveAbortSetOrNil includes: aNode) not ])
    ifTrue: [
      selectiveAbortSetOrNil add: aNode.
      aNode _selectiveAbort ].

]

{ #category : 'Updating' }
BtreePlusNode >> add: anObject [
  self shouldNotImplement: #add:

]

{ #category : 'Updating' }
BtreePlusNode >> addAll: aCollection [
  self shouldNotImplement: #addAll:

]

{ #category : 'Updating' }
BtreePlusNode >> addLast: anObject [
  self shouldNotImplement: #addLast:

]

{ #category : 'Updating' }
BtreePlusNode >> at: aKey put: aValue [
  self shouldNotImplement: #at:put:

]

{ #category : 'Updating' }
BtreePlusNode >> at: aKey put: aValue root: rootObject [
  "Adds the key/value/root tuple to the node.  If the node is full, performs a 'split'
 on the parent.  Returns the new sibling if a split is performed, otherwise
 returns the receiver."

  self subclassResponsibility: #'at:put:root:'

]

{ #category : 'Rc Updating' }
BtreePlusNode >> at: aKey put: aValue root: rootObject selectiveAbortSet: selectiveAbortSetOrNil [
  "Adds the key/value/root tuple to the node.  If the node is full, performs a 'split'
 on the parent.  Returns the new sibling if a split is performed, otherwise
 returns the receiver."

  self subclassResponsibility: #'at:put:root:selectiveAbortSet:'

]

{ #category : 'Audit' }
BtreePlusNode >> auditDepListForLastElements: setOfKeys bagOfKeys: bagOfKeys for: pathTerm offset: offset using: auditor [
  self rootNode isIdentityIndex
    ifTrue: [ "Identity indexes do not use dependency lists"
      ^ self ].
  1 to: setOfKeys size do: [ :n | | aKey |
    aKey := setOfKeys _at: n.
    (DependencyList for: aKey) ~~ nil
      ifTrue: [ pathTerm
          auditDepListForLastElementObject: aKey
          occurrences: (bagOfKeys occurrencesOf: aKey)
          index: (pathTerm at: offset)
          using: auditor ] ]

]

{ #category : 'Audit' }
BtreePlusNode >> auditForPathTerm: pathTerm using: auditor do: aBlock [
  "Audit the receiver's entries and each child node"

  self auditForPathTerm: pathTerm using: auditor parentNode: nil previousChild: nil nextChild: nil do: aBlock

]

{ #category : 'Audit' }
BtreePlusNode >> auditForPathTerm: pathTerm using: auditor parentNode: parentNode previousChild: previousChild nextChild: nextChild do: aBlock [
  self subclassResponsibility: #'auditForPathTerm:using:parentNode:previousChild:nextChild:do:'

]

{ #category : 'Audit' }
BtreePlusNode >> auditKey: key value: value root: rootObject bag: bag nsc: nsc for: pathTerm offset: offset bagOfRoots: bagOfRoots enumBag: enumBag  setValBag: setValBag using: auditor [
  | num bagSize |
  num := bagOfRoots occurrencesOf: rootObject.
  pathTerm isEnumeratedTerm
    ifTrue: [ (num := enumBag occurrencesOf: value) * num ].
  pathTerm isSetValuedTerm
    ifTrue: [ (num := setValBag occurrencesOf: value) ].
  bagSize := bag size.
  bagSize == num
    ifFalse: [ auditor
        btree: self
        incorrectNumberOfEntries: pathTerm
        key: key
        value: value
        bag: bag
        numExpected: num ].
  bag removeAll: bag.
  enumBag removeAll: enumBag.
  setValBag removeAll: setValBag.
  bagOfRoots removeAll: bagOfRoots.

]

{ #category : 'Audit' }
BtreePlusNode >> auditNsc: nsc for: pathTerm offset: offset using: auditor [
  "Verifies that the btree structure consistent relative to the given pathTerm."

  | indicatesIndexOnNscElements indicatesMultiValue count key value rootObject bag ivOffset bagOfKeys setOfKeys comparisonForSort previousKey previousValue sentinel enumBag previousRoot enumValue setValBag setValValue lastChildNode  bagOfRoots |
  indicatesIndexOnNscElements := pathTerm indicatesIndexOnNscElements.
  indicatesMultiValue := pathTerm indicatesMultiValue.
  count := 0.
  bag := IdentityBag new.
  enumBag := IdentityBag new.
  setValBag := IdentityBag new.
  bagOfKeys := IdentityBag new.
  bagOfRoots := IdentityBag new.
  setOfKeys := IdentitySet new.
  comparisonForSort := self comparisonForSort.
  sentinel := Object new.
  previousRoot := previousKey := sentinel.
  lastChildNode := sentinel.
  self
    auditForPathTerm: pathTerm
    using: auditor
    do: [ :childNode :k :v :r | setValValue := enumValue := false.
        lastChildNode := childNode.
        previousKey == sentinel
        ifFalse: [ (comparisonForSort compareKey: previousKey lessThanOrEqualTo: k)
            ifTrue: [ (comparisonForSort compareKey: previousKey equalTo: k)
                ifTrue: [ previousValue asOop <= v asOop
                    ifFalse: [ auditor
                        btree: childNode
                        outOfOrderKeyValuePairs: pathTerm
                        key: k
                        previous: previousKey
                        value: v
                        previous: previousValue ].
                  previousValue asOop = v asOop
                    ifTrue: [ previousRoot asOop <= r asOop
                        ifFalse: [ auditor
                            btree: childNode
                            outOfOrderKeyValueRootPairs: pathTerm
                            key: k
                            previous: previousKey
                            value: v
                            previous: previousValue
                            root: r
                            previous: previousRoot ] ] ] ]
            ifFalse: [ auditor btree: childNode outOfOrderKeys: pathTerm key: k previous: previousKey ] ].
      previousKey := k.
      previousValue := v.
      previousRoot := r.
      count := count + 1.
      indicatesIndexOnNscElements
        ifTrue: [ pathTerm isSetValuedTerm ifFalse: [v == k
            ifFalse: [ auditor
                btree: childNode
                mismatchedKeyAndValueForIndexOnNscElements: pathTerm
                key: k
                value: v ] ].
          indicatesMultiValue
            ifFalse: [
              v == r
                ifFalse: [
                  auditor
                    btree: childNode
                    mismatchedRootAndValueForIndexOnNscElements: pathTerm
                    key: k
                    value: v
                    root: r ] ] .
pathTerm isSetValuedTerm ifTrue: [ setValValue := true ]]
        ifFalse: [ pathTerm isEnumeratedTerm
            ifTrue: [ | ivOffsets res  |
              ivOffsets := pathTerm _ivOffsetsFor: v.
              res := {}.
              ivOffsets
                do: [ :off | | expected |
                 off ifNotNil: [
                  (expected := v instVarAt: off) == k
                    ifTrue: [ res add: {true} ]
                    ifFalse: [ res add: {false. expected. off} ] ] ].
              res
                detect: [:ar | ar at: 1 ]
                ifNone: [
                  res do: [:ar |
                    auditor
                            btree: childNode
                            incorrectKey: pathTerm
                            key: k
                            value: v
                            expectedKey: (ar at: 2)
                            ivOffset: (ar at: 3) ] ].
              enumValue := true ]
            ifFalse: [ | expected |
              pathTerm isSelectorTerm
                ifTrue: [ expected := v perform: pathTerm termSelector.
                  (expected _idxForSortEqualTo: k)
                    ifFalse: [ auditor
                        btree: childNode
                        incorrectKeyForEnumeratedPathTerm: pathTerm
                        key: k
                        value: v
                        expectedKey: expected
                        selector: pathTerm termSelector ] ]
                ifFalse: [
                  pathTerm isSetValuedTerm
                    ifTrue: [ setValValue := true ]
                    ifFalse: [
                      ivOffset := pathTerm _ivOffsetFor: v.
                      (expected := v instVarAt: ivOffset) == k
                        ifFalse: [ auditor
                          btree: childNode
                          incorrectKey: pathTerm
                          key: k
                          value: v
                          expectedKey: expected
                          ivOffset: ivOffset ] ] ] ] ].
      count > 1
        ifTrue: [
          (key == k and: [ value == v and: [ rootObject == r ] ])
            ifFalse: [ childNode
                auditKey: key
                value: value
                root: rootObject
                bag: bag
                nsc: nsc
                for: pathTerm
                offset: offset
                bagOfRoots: bagOfRoots
                enumBag: enumBag
                setValBag: setValBag
                using: auditor ].
          (comparisonForSort compareKey: key equalTo: k)
            ifTrue: [ bagOfKeys add: k.
              setOfKeys add: k ]
            ifFalse: [
              self
                auditDepListForLastElements: setOfKeys
                bagOfKeys: bagOfKeys
                for: pathTerm
                offset: offset
                using: auditor.
              bagOfKeys removeAll: bagOfKeys.
              setOfKeys removeAll: setOfKeys.
              bagOfKeys add: k.
              setOfKeys add: k ] ]
        ifFalse: [ bagOfKeys add: k.
          setOfKeys add: k ].
      bagOfRoots add: r.
      value := v.
      key := k.
      rootObject := r.
      bag add: v.
      enumValue ifTrue: [ enumBag add: v ].
      setValValue ifTrue: [ setValBag add: v ] ].
  (count > 0 and: [ lastChildNode ~~ sentinel ])
    ifTrue: [ lastChildNode
        auditKey: key
        value: value
        root: rootObject
        bag: bag
        nsc: nsc
        for: pathTerm
        offset: offset
        bagOfRoots: bagOfRoots
        enumBag: enumBag
        setValBag: setValBag
        using: auditor ].
  self
    auditDepListForLastElements: setOfKeys
    bagOfKeys: bagOfKeys
    for: pathTerm
    offset: offset
    using: auditor.
  ^ count

]

{ #category : 'Searching' }
BtreePlusNode >> binarySearchCompareLessThanOrEqualTo: aKey forEntryAt: index using: aComparisonForSort [
  | entry |
  entry := self at: index.
  (aComparisonForSort compareKey: aKey lessThan: entry)
    ifTrue: [ ^ true ]
    ifFalse: [ (aComparisonForSort compareKey: aKey greaterThan: entry)
        ifTrue: [ ^ false ]
        ifFalse: [ ^ true ] ]

]

{ #category : 'Searching' }
BtreePlusNode >> binarySearchCoveringKey: aKey [
  "Returns the index for the entry in which aKey would be inserted.  This
   is the first entry which has its key >= aKey. If the receiver is empty, returns
   1.  Uses a binary search."

  | lowPt midPt highPt entrySize index entryGreaterThan entryEqualTo comparisonForSort keyIndexOffset |
  entrySize := self entrySize.
  lowPt := 1.
  highPt := numElements.
  entryGreaterThan := false.
  comparisonForSort := self comparisonForSort.
  keyIndexOffset := self keyIndexOffset.
  [ lowPt <= highPt ] whileTrue: [ | objAtIndex |
      entryGreaterThan := false.
      entryEqualTo := false.
      midPt := lowPt + highPt quo: 2.
      index := (midPt - 1) * entrySize + 1 + keyIndexOffset.
      objAtIndex := self at: index.
      (comparisonForSort compareKey: aKey lessThan: objAtIndex)
        ifTrue: [ highPt := midPt - 1 ]
        ifFalse: [ (comparisonForSort compareKey: aKey greaterThan: objAtIndex)
            ifTrue: [ lowPt := midPt + 1.
              entryGreaterThan := true ]
            ifFalse: [ "If keys are equal, then secondary sort key is oop value, tertiary sort key os oop of rootObject"
              (comparisonForSort compareKey: aKey equalTo: objAtIndex)
                ifTrue: [ (self
                    compareValueOop: aKey
                    lessThanOrEqualToEntryValueOopAt: index)
                    ifTrue: [ highPt := midPt - 1 ]
                    ifFalse: [ lowPt := midPt + 1 ].
                  entryEqualTo := true ]
                ifFalse: [ lowPt := midPt + 1 ] ] ] ].
  entryGreaterThan
    ifTrue: [ ^ index + entrySize - keyIndexOffset ].
  (entryEqualTo
    and: [ "use secondary sort key (oop of value), tertiary sort key os oop of rootObject"
      self
        compareValueOop: aKey
        greaterThanEntryValueOopAt: index ])
    ifTrue: [ ^ index + entrySize - keyIndexOffset ].
  ^ index - keyIndexOffset

]

{ #category : 'Searching' }
BtreePlusNode >> binarySearchCoveringKey: aKey totalOrder: aBoolean [
  "Returns the index for the first entry in which aKey is found utilizing a
binary search.  This is the first entry whose key >= aKey."

  ^ self binarySearchCoveringKey: aKey totalOrder: aBoolean using: self comparisonForSort

]

{ #category : 'Searching' }
BtreePlusNode >> binarySearchCoveringKey: aKey totalOrder: aBoolean using: aComparisonForSort [
  "Returns the index for the first entry in which aKey is found utilizing a
binary search.  This is the first entry whose key >= aKey."

  | lowPt midPt highPt entrySize index entryGreaterThan keyIndexOffset |
  entrySize := self entrySize.
  lowPt := 1.
  highPt := numElements.
  entryGreaterThan := false.
  keyIndexOffset := self keyIndexOffset.
  [ lowPt <= highPt ] whileTrue: [ midPt := lowPt + highPt quo: 2.
      index := (midPt - 1) * entrySize + 1 + keyIndexOffset.
      entryGreaterThan := false.
      (self
        binarySearchCompareLessThanOrEqualTo: aKey
        forEntryAt: index
        using: aComparisonForSort)
        ifTrue: [ highPt := midPt - 1 ]
        ifFalse: [ lowPt := midPt + 1.
          entryGreaterThan := true ] ].
  entryGreaterThan
    ifTrue: [ ^ index + entrySize - keyIndexOffset ].
  ^ index - keyIndexOffset

]

{ #category : 'Searching' }
BtreePlusNode >> binarySearchCoveringKey: aKey value: aValue [
  "Returns the index for the entry in which aKey/aValue would be inserted.  This
 is the first entry which has its key >= aKey.  If multiple entries have the
 same key (key = aKey), then aValue is used (since entries with duplicate keys
 are inserted by the value's OOP ordering). If the receiver is empty, returns
 1.  Uses a binary search."

  | lowPt midPt highPt entrySize index entryGreaterThan entryEqualTo comparisonForSort keyIndexOffset |
  entrySize := self entrySize.
  lowPt := 1.
  highPt := numElements.
  entryGreaterThan := false.
  comparisonForSort := self comparisonForSort.
  keyIndexOffset := self keyIndexOffset.
  [ lowPt <= highPt ] whileTrue: [ | objAtIndex |
      entryGreaterThan := false.
      entryEqualTo := false.
      midPt := lowPt + highPt quo: 2.
      index := (midPt - 1) * entrySize + 1 + keyIndexOffset.
      objAtIndex := self at: index.
      (comparisonForSort compareKey: aKey lessThan: objAtIndex)
        ifTrue: [ highPt := midPt - 1 ]
        ifFalse: [ (comparisonForSort compareKey: aKey greaterThan: objAtIndex)
            ifTrue: [ lowPt := midPt + 1.
              entryGreaterThan := true ]
            ifFalse: [ "If keys are equal, then secondary sort key is oop value, tertiary sort key os oop of rootObject"
              (comparisonForSort compareKey: aKey equalTo: objAtIndex)
                ifTrue: [ (self
                    compareValueOop: aValue
                    lessThanOrEqualToEntryValueOopAt: index)
                    ifTrue: [ highPt := midPt - 1 ]
                    ifFalse: [ lowPt := midPt + 1 ].
                  entryEqualTo := true ]
                ifFalse: [ lowPt := midPt + 1 ] ] ] ].
  entryGreaterThan
    ifTrue: [ ^ index + entrySize - keyIndexOffset ].
  (entryEqualTo
    and: [ "use secondary sort key (oop of value), tertiary sort key os oop of rootObject"
      self
        compareValueOop: aValue
        greaterThanEntryValueOopAt: index ])
    ifTrue: [ ^ index + entrySize - keyIndexOffset ].
  ^ index - keyIndexOffset

]

{ #category : 'Searching' }
BtreePlusNode >> binarySearchCoveringKey: aKey value: aValue root: rootObject [
  "Returns the index for the entry in which aKey/aValue would be inserted.  This
 is the first entry which has its key >= aKey.  If multiple entries have the
 same key (key = aKey), then aValue is used (since entries with duplicate keys
 are inserted by the value's OOP ordering).  If multiple entries have the same value
 (value = aValue), then rootObject is used (since entries with duplicate values are
 inserted by the rootObject's OOP ordering). If the receiver is empty, returns
 1.  Uses a binary search."

  | lowPt midPt highPt entrySize index entryGreaterThan entryEqualTo comparisonForSort keyIndexOffset |
  entrySize := self entrySize.
  lowPt := 1.
  highPt := numElements.
  entryGreaterThan := false.
  comparisonForSort := self comparisonForSort.
  keyIndexOffset := self keyIndexOffset.
  [ lowPt <= highPt ]
    whileTrue: [
      | objAtIndex |
      entryGreaterThan := false.
      entryEqualTo := false.
      midPt := lowPt + highPt quo: 2.
      index := (midPt - 1) * entrySize + 1 + keyIndexOffset.
      objAtIndex := self at: index.
      (comparisonForSort compareKey: aKey lessThan: objAtIndex)
        ifTrue: [ highPt := midPt - 1 ]
        ifFalse: [
          (comparisonForSort compareKey: aKey greaterThan: objAtIndex)
            ifTrue: [
              lowPt := midPt + 1.
              entryGreaterThan := true ]
            ifFalse: [
              "If keys are equal, then secondary sort key is oop value, tertiary sort key os oop of rootObject"
              (comparisonForSort compareKey: aKey equalTo: objAtIndex)
                ifTrue: [
                  (self
                    compareValueOop: aValue
                    lessThanOrEqualToEntryValueOopAt: index
                    root: rootObject)
                    ifTrue: [ highPt := midPt - 1 ]
                    ifFalse: [ lowPt := midPt + 1 ].
                  entryEqualTo := true ]
                ifFalse: [ lowPt := midPt + 1 ] ] ] ].
  entryGreaterThan
    ifTrue: [ ^ index + entrySize - keyIndexOffset ].
  (entryEqualTo
    and: [
      "use secondary sort key (oop of value), tertiary sort key os oop of rootObject"
      self
        compareValueOop: aValue
        greaterThanEntryValueOopAt: index
        root: rootObject ])
    ifTrue: [ ^ index + entrySize - keyIndexOffset ].
  ^ index - keyIndexOffset

]

{ #category : 'Searching' }
BtreePlusNode >> binarySearchForKey: aKey value: aValue root: rootObject [
  "Returns the 'value' index whose key equality matches aKey and whose root matches rootObject.
   If no match is found, returns zero.  Uses a binary search."

  | lowPt midPt highPt entrySize index comparisonForSort keyIndexOffset |
  numElements == 0
    ifTrue: [ ^ 0 ].
  entrySize := self entrySize.
  lowPt := 1.
  highPt := numElements.
  comparisonForSort := self comparisonForSort.
  keyIndexOffset := self keyIndexOffset.
  [ lowPt <= highPt ] whileTrue: [ midPt := lowPt + highPt quo: 2.
      index := (midPt - 1) * entrySize + 1 + keyIndexOffset.
      (self
        compare: comparisonForSort
        key: aKey
        value: aValue
        equalToEntryAt: index
        root: rootObject)
        ifTrue: [ ^ index - keyIndexOffset ].
      (self
        compare: comparisonForSort
        key: aKey
        value: aValue
        lessThanOrEqualToEntryAt: index
        root: rootObject)
        ifTrue: [ highPt := midPt - 1 ]
        ifFalse: [ lowPt := midPt + 1 ] ].
  ^ 0

]

{ #category : 'Accessing' }
BtreePlusNode >> collator [
  "Answer the collator for the receiver"

  ^ rootNode collator

]

{ #category : 'Comparison Operators' }
BtreePlusNode >> compareValueOop: aValue greaterThanEntryValueOopAt: keyIndex [
  "Perform a > comparison between the oop of aValue and oop of the entry whose value is
   at the given index."

  self subclassResponsibility: #'compareValueOop:greaterThanEntryValueAt:'

]

{ #category : 'Comparison Operators' }
BtreePlusNode >> compareValueOop: aValue greaterThanEntryValueOopAt: keyIndex root: aRootObjectOrNil [
  "Perform a > comparison between the oop of aValue and oop of the entry whose value is
   at the given index. If the value oops are equal, compare the oop of aRootObjectOrNil and the
   oop of the root object at the given index."

  self subclassResponsibility: #'compareValueOop:greaterThanEntryValueAt:root:'

]

{ #category : 'Comparison Operators' }
BtreePlusNode >> compareValueOop: aValue lessThanOrEqualToEntryValueOopAt: keyIndex [
  "Perform a <= comparison between the oop of aValue and oop of the entry whose value is
   at the given index."

  self
    subclassResponsibility: #'compareValueOop:lessThanOrEqualToEntryValueOopAt:'

]

{ #category : 'Comparison Operators' }
BtreePlusNode >> compareValueOop: aValue lessThanOrEqualToEntryValueOopAt: keyIndex root: aRootObjectOrNil [
  "Perform a <= comparison between the oop of aValue and oop of the entry whose value is
   at the given index."

  self
    subclassResponsibility: #'compareValueOop:lessThanOrEqualToEntryValueOopAt:root:'

]

{ #category : 'Comparison Operators' }
BtreePlusNode >> comparisonForSort [
  "Answer the comparison object to be used for this index"

  ^ rootNode comparisonForSort

]

{ #category : 'Constants' }
BtreePlusNode >> entrySize [
  "Returns the size of an entry."

  "in-line of `self class entrySize` for performance"

  self subclassResponsibility: #'entrySize'

]

{ #category : 'Accessing' }
BtreePlusNode >> indexObject [

  ^ self rootNode indexObject

]

{ #category : 'Testing' }
BtreePlusNode >> isBtreePlusNode [
  ^ true

]

{ #category : 'Testing' }
BtreePlusNode >> isLeaf [

"Returns whether the B-tree node is a leaf node."

^ false

]

{ #category : 'Accessing' }
BtreePlusNode >> keyIndexOffset [
  "Answer the offset from entry index for the key. Typical order is value/key/root."

  ^ 1

]

{ #category : 'Accessing' }
BtreePlusNode >> maxNumberOfElements [
  ^ self class maxNumberOfElements

]

{ #category : 'Constants' }
BtreePlusNode >> mergeThreshold [

"Returns the threshold such that if an entry has less than this number, it must
 have elements added from its sibling or be merged with its sibling."

^ self maxNumberOfElements quo: 2.

]

{ #category : 'Accessing' }
BtreePlusNode >> numElements [

"Returns the value of the instance variable 'numElements'."

^ numElements

]

{ #category : 'Updating' }
BtreePlusNode >> numElements: newValue [

"Modify the value of the instance variable 'numElements'."

numElements := newValue

]

{ #category : 'Constants' }
BtreePlusNode >> parentNodeClass [
  "Returns the class of node to be created as the parent when a split occurs."

  ^ self subclassResponsibility: #'parentNodeClass'

]

{ #category : 'Printing' }
BtreePlusNode >> prettyPrint [

""

| aStream |
aStream := PrintStream printingOn: String new.
self printOn: aStream withIndentationLevel: 0 includingValues: false.
^ aStream contents

]

{ #category : 'Accessing' }
BtreePlusNode >> rootIndexOffset [
  "Answer the offset from entry index for the root object. Typical order is value/key/root."

  ^ 2

]

{ #category : 'Accessing' }
BtreePlusNode >> rootNode [
  "Answer the BtreePlusRoot of the receiver"

  ^ rootNode

]

{ #category : 'Accessing' }
BtreePlusNode >> rootNode: aBtreePlusRoot [
  "set the BtreePlusRoot of the receiver"

  rootNode := aBtreePlusRoot

]

{ #category : 'Accessing' }
BtreePlusNode >> valueIndexOffset [
  "Answer the offset from entry index for the value. Typical order is value/key/root."

  ^ 0

]
