Extension { #name : 'BtreePlusReadStream' }

{ #category : 'Instance Creation' }
BtreePlusReadStream class >> on: aBtreeNode [

"Create a stream that can access the entire contents of the B-tree whose
 root node is aBtreeNode."

| arr1 arr2 newStream |
arr1 := { } .
arr2 := { } .

aBtreeNode _putFirstIndexOfFirstChildInto: arr1.
aBtreeNode _putLastIndexOfLastChildInto: arr2.

newStream := self new.
newStream currentIndex: (arr1 at: 1).
newStream currentNode: (arr1 at: 2).
newStream endIndex: (arr2 at: 1).
newStream endNode: (arr2 at: 2).
^ newStream

]

{ #category : 'Testing' }
BtreePlusReadStream >> _atEnd [
  "Returns true if the receiver is positioned at the end of the stream, and
 false otherwise."

  ^ currentIndex == 0

]

{ #category : 'Testing' }
BtreePlusReadStream >> _btreeAtEnd [

""

^ currentIndex == 0

]

{ #category : 'Accessing' }
BtreePlusReadStream >> _btreeNext [

"Returns the next value on a stream of B-tree values and root objects.  Updates the current
 stack for a subsequent 'next'."

^ self _btreeNextValue

]

{ #category : 'Accessing' }
BtreePlusReadStream >> _btreeNextNoValue [

"Returns the next value on a stream of B-tree values and root objects.  Updates the current
 stack for a subsequent 'next'."

" get the index into the leaf node and see if it has reached the end "
currentIndex == 0
    ifTrue: [ ^ self _errorEndOfStream ].

" if this is the end of the stream, sets the top index to zero and returns "
( currentNode == endNode and: [ endIndex == currentIndex ] )
    ifTrue: [
        currentIndex := 0.
        ^ self
    ].

" see if index refers to last entry in this leaf "
currentIndex == currentLastValIndex
    ifTrue: [
        " must look down the stack for the next leaf node "
        self _nextLeaf
    ]
    ifFalse: [ currentIndex := (currentIndex + currentEntrySize) ].

]

{ #category : 'Accessing' }
BtreePlusReadStream >> _btreeNextRootObject [
  "Returns the next root object on a stream of B-tree values and root objects.  Updates the current
 stack for a subsequent 'next'."

  | root |
  " get the root value within the leaf "
  root := currentNode rootObjectAt: currentIndex.
  " if this is the end of the stream, sets the top currentIndex to zero and returns "
  (currentNode == endNode and: [ endIndex == currentIndex ])
    ifTrue: [ currentIndex := 0.
      ^ root ].
  " see if currentIndex refers to last entry in this leaf "
  currentIndex == currentLastValIndex
    ifTrue: [ self _nextLeaf ]
    ifFalse: [ currentIndex := currentIndex + currentEntrySize ].
  ^ root

]

{ #category : 'Accessing' }
BtreePlusReadStream >> _btreeNextValue [

"Returns the next value on a stream of B-tree values and root objects.  Updates the current
 stack for a subsequent 'next'."

| val    |
" get the index into the leaf node and see if it has reached the end "
currentIndex == 0
    ifTrue: [ ^ self _errorEndOfStream ].

" get the leaf and the value within the leaf "
val := currentNode at: currentIndex.

" if this is the end of the stream, sets the top index to zero and returns "
( currentNode == endNode and: [ endIndex == currentIndex ] )
    ifTrue: [
        currentIndex := 0.
        ^ val
    ].

" see if index refers to last entry in this leaf "
currentIndex == currentLastValIndex
    ifTrue: [
        " must look down the stack for the next leaf node "
        self _nextLeaf
    ]
    ifFalse: [ currentIndex := (currentIndex + currentEntrySize) ].

^ val

]

{ #category : 'Error Handling' }
BtreePlusReadStream >> _errorEndOfStream [

"Raises an error due to an attempt to read beyond the end of the stream."

^ self _error: #rtErrBtreeReadStreamEndOfStream .

]

{ #category : 'Query Support' }
BtreePlusReadStream >> _hasSameEndAs: otherStream [

"Returns true if the receiver has the same end index and end node as the
 otherStream."

^ endIndex == otherStream endIndex and: [
   endNode == otherStream endNode ]

]

{ #category : 'Accessing' }
BtreePlusReadStream >> _nextLeaf [
  "Get the next leaf node and make it the current node."
  | nxt |
  (nxt := currentNode nextLeaf) ifNil:[ ^ self _errorEndOfStream].
  self currentNode: nxt .
  currentIndex := 1

]

{ #category : 'Private' }
BtreePlusReadStream >> _peekKey [
  "Returns the key of the current entry in the stream."

  currentIndex == 0
    ifTrue: [ ^ self _errorEndOfStream ].
  ^ currentNode keyAt: currentIndex

]

{ #category : 'Private' }
BtreePlusReadStream >> _peekRoot [
  "Returns the root of the current entry in the stream."


  currentIndex == 0
    ifTrue: [ ^ self _errorEndOfStream ].
  ^ currentNode rootObjectAt: currentIndex

]

{ #category : 'Private' }
BtreePlusReadStream >> _peekValue [
  "Returns the value of the current entry in the stream."

  currentIndex == 0
    ifTrue: [ ^ self _errorEndOfStream ].
  ^ currentNode at: currentIndex

]

{ #category : 'Query Support' }
BtreePlusReadStream >> _positionToEnd [

"Sets the current stack to the end index and end node.  This
 has the effect of advancing the stream to the last object."

currentIndex := endIndex.
self currentNode: endNode

]

{ #category : 'Testing' }
BtreePlusReadStream >> atEnd [
  "Returns true if the receiver is positioned at the end of the stream, and
 false otherwise."

  ^ self _atEnd

]

{ #category : 'Accessing' }
BtreePlusReadStream >> currentIndex [

"Returns the value of the instance variable 'currentIndex'."

^currentIndex

]

{ #category : 'Updating' }
BtreePlusReadStream >> currentIndex: newValue [

"Modify the value of the instance variable 'currentIndex'."

currentIndex := newValue

]

{ #category : 'Accessing' }
BtreePlusReadStream >> currentNode [

"Returns the value of the instance variable 'currentNode'."

^currentNode

]

{ #category : 'Updating' }
BtreePlusReadStream >> currentNode: newValue [
  "Modify the value of the instance variable 'currentNode'."

  currentNode := newValue.
  currentLastValIndex := currentNode _lastEntryIndex.
  currentEntrySize := currentNode entrySize.

]

{ #category : 'Accessing' }
BtreePlusReadStream >> currentStack: array1 [

"Set the currentIndex and currentNode from array1"

currentIndex := array1 at: 1.
currentIndex ~~ 0
  ifTrue: [ self currentNode: ( array1 at: 2) ]

]

{ #category : 'Accessing' }
BtreePlusReadStream >> endIndex [

"Returns the value of the instance variable 'endIndex'."

^endIndex

]

{ #category : 'Updating' }
BtreePlusReadStream >> endIndex: newValue [

"Modify the value of the instance variable 'endIndex'."

endIndex := newValue

]

{ #category : 'Accessing' }
BtreePlusReadStream >> endNode [

"Returns the value of the instance variable 'endNode'."

^endNode

]

{ #category : 'Updating' }
BtreePlusReadStream >> endNode: newValue [

"Modify the value of the instance variable 'endNode'."

endNode := newValue

]

{ #category : 'Accessing' }
BtreePlusReadStream >> next [
  "Returns the next root object on a stream of B-tree values and root objects.  Updates the current
 stack for a subsequent 'next'."

  | root |
  currentIndex == 0
    ifTrue: [ ^ self _errorEndOfStream ].	" get the root value within the leaf "
  root := currentNode rootObjectAt: currentIndex.	" if this is the end of the stream, sets the top currentIndex to zero and returns "
  (currentNode == endNode and: [ endIndex == currentIndex ])
    ifTrue: [ currentIndex := 0.
      ^ root ].	" see if currentIndex refers to last entry in this leaf "
  currentIndex == currentLastValIndex
    ifTrue: [ self _nextLeaf ]
    ifFalse: [ currentIndex := currentIndex + currentEntrySize ].
  ^ root

]

{ #category : 'Adding' }
BtreePlusReadStream >> nextPut: anObject [

"Disallowed.  You cannot write to a BtreeReadStream."

self shouldNotImplement: #nextPut:

]

{ #category : 'Accessing' }
BtreePlusReadStream >> nextRoot [
  "Returns the next root object on a stream of B-tree values and root objects.  Update the current
 stack for a subsequent 'next'."

  currentIndex == 0
    ifTrue: [ ^ self _errorEndOfStream ].
  ^ self _btreeNextRootObject

]

{ #category : 'Accessing' }
BtreePlusReadStream >> size [

"Returns the number of elements contained in the receiver (that is, how many
successful 'next' operations can be performed)."

| total theNode theIndex lastNode |
self atEnd ifTrue: [
  ^ 0
  ].
total := 0.
theNode := currentNode.
theIndex := currentIndex.
lastNode := false.
[ lastNode ] whileFalse: [
  lastNode := theNode == endNode.
  total := total + (theNode
    _totalElementsFrom: theIndex
    toEndNode: endNode
    endIndex: endIndex).
  theNode := theNode nextLeaf ].
^ total

]
