Extension { #name : 'CByteArray' }

{ #category : 'Instance Creation' }
CByteArray class >> _newFrom: aCPointer offset: anOffset numBytes: aSize gcFree: gcKind [
"aCPointer may also be aCByteArray, or anInteger address.
 If aCPointer is a CByteArray with memory size 0 , it is assumed to
 be something like a  pointer field of a Struct which references another
 struct, and no checks are done for aSize <= anOffset + aCPointer.size "

<primitive: 711>
anOffset _validateClass: SmallInteger .
anOffset < 0 ifTrue:[ anOffset _error: #errArgTooSmall args:{ 0 } ].
aSize _validateClass: SmallInteger .
aSize < 0 ifTrue:[ aSize _error: #rtErrArgOutOfRange args:{ 0 } ].
gcKind _validateClass: SmallInteger .
aCPointer ifNotNil:[ | argCls |
  ((argCls:=aCPointer class) == CByteArray or:[ argCls == CPointer
        or:[ aCPointer _isInteger ]]) ifFalse:[
     ArgumentTypeError new name: 'aCPointer' expectedClass: { CPointer. CByteArray. Integer }
		actualArg: aCPointer ; signal
  ].
  (gcKind == 0 or:[ gcKind == 1]) ifFalse:[
     gcKind _error: #rtErrArgOutOfRange args:{ 0 . 1 } ].
  argCls == CByteArray ifTrue:[ | csz |
    (anOffset + aSize) >= (csz := aCPointer size) ifTrue:[
      (anOffset + aSize) _error: #errArgTooLarge args:{ csz } .
    ].
  ].
  aCPointer _isInteger ifTrue:[
    (aSize ~~ 0 or:[ gcKind ~~ 0]) ifTrue:[
      ArgumentError signal: 'invalid args for fromAddress:' .
    ].
  ].
] ifNil:[
  (gcKind == 0 or:[ gcKind == 1]) ifFalse:[ gcKind _error: #rtErrArgOutOfRange].
].
"if we get here malloc probably failed with no more C heap"
^ self _primitiveFailed: #_newFrom:offset:numBytes:gcFree:
       args: { aCPointer . anOffset . aSize . gcKind }

]

{ #category : 'Private' }
CByteArray class >> basicNew [
 "disallowed"
  self shouldNotImplement: #basicNew

]

{ #category : 'Instance Creation' }
CByteArray class >> fromCharStar: aCPointer [
 "no fence, not auto freed, primitive does strlen() to determine size"

^ self _newFrom: aCPointer offset: 0 numBytes: -1 gcFree: 0

]

{ #category : 'Instance Creation' }
CByteArray class >> fromCharStarGcFree: aCPointer [
 "no fence, will auto-free with free(), primitive does strlen() to determine size"

^ self _newFrom: aCPointer offset: 0 numBytes: -1 gcFree: AutoFree

]

{ #category : 'Instance Creation' }
CByteArray class >> fromCPointer: aCPointer numBytes: anInt [
 "no fence, not auto freed. aCPointer may also be aCByteArray"

^ self _newFrom: aCPointer offset: 0 numBytes: anInt gcFree: 0

]

{ #category : 'Instance Creation' }
CByteArray class >> fromCPointerGcFree: aCPointer numBytes: anInt [
 "no fence, will auto-free with free() "

^ self _newFrom: aCPointer offset: 0 numBytes: anInt gcFree: AutoFree

]

{ #category : 'Instance Creation' }
CByteArray class >> fromRegionOf: aCByteArray offset: zeroBasedOffset numBytes: aSize [
  ^ self _newFrom: aCByteArray offset: zeroBasedOffset numBytes: aSize gcFree: 0

]

{ #category : 'Instance Creation' }
CByteArray class >> gcMalloc: numBytes [
 "(freed by in-memory GC, memory is zeroed"

^ self _newFrom: nil offset: 0 numBytes: numBytes gcFree: AutoFree

]

{ #category : 'Instance Creation' }
CByteArray class >> malloc: numBytes [
  "no fence words, never freed , or freed by a C function to be called,
   memory is zeroed."

^ self _newFrom: nil offset: 0 numBytes: numBytes gcFree: 0

]

{ #category : 'Private' }
CByteArray class >> new [
 "disallowed"
  self shouldNotImplement: #new

]

{ #category : 'Instance Creation' }
CByteArray class >> newNull [
  "Returns a CPointer representing a NULL pointer."

  ^ CPointer newNull
]

{ #category : 'Instance Creation' }
CByteArray class >> withAll: anObject [

"anObject may be a String, ByteArray or CByteArray.
 Returns a new instance of the receiver containing a copy of
 the bytes of the argument.   If anObject is a String, the
 resulting ByteArray contains the bytes of the String plus
 a terminator byte with value zero.

 Use CByteArray class >> withAll:nullTerminate: for more control
 over terminator.

 Warning, ByteArray usually stores integers in big-endian order,
 not CPU native byte order."

^ self withAll: anObject nullTerminate: anObject _isOneByteString 

]

{ #category : 'Instance Creation' }
CByteArray class >> withAll: anObject nullTerminate: aBoolean [

"anObject may be a String, ByteArray, Utf8, Utf16 or CByteArray.
 Returns a new instance of the receiver containing a copy of
 the bytes of the argument.   If aBoolean==true the 
 resulting CByteArray contains the bytes of the argument plus
 a terminator byte (or Utf16 codepoint) with value zero.

 If argument is a Utf16, the result has code points in CPU byte order.

 Warning, ByteArray usually stores integers in big-endian order,
 not CPU native byte order."

  | res argSiz allocSiz isUtf16 termSize |
  isUtf16 := false .
  (anObject stringCharSize == 1 
   or:[ (anObject isKindOfClass: CByteArray)
   or:[ (anObject isKindOfClass: Utf8) 
   or:[ (isUtf16:= anObject isKindOfClass: Utf16) ]]]) ifFalse:[
       anObject _validateKindOfClasses: 
          { String . ByteArray . CByteArray . Utf8 . Utf16 }.
  ].
  argSiz := anObject size .
  termSize := aBoolean ifTrue:[ 1 ] ifFalse:[ 0 ] .
  isUtf16 ifTrue:[ 
     argSiz := anObject basicSize  . termSize := termSize + termSize .
  ].
  allocSiz := argSiz + termSize .
  res := self gcMalloc: allocSiz .
  argSiz ~~ 0 ifTrue:[
    res copyBytesFrom: anObject from: 1 to: argSiz into: 0 allowCodePointZero: true .
  ].
  ^ res

]

{ #category : 'Private' }
CByteArray >> _copyFrom: zeroBasedStart to: zeroBasedEnd resKind: aClass [
  "zeroBasedStart, zeroBasedEnd must be SmallInteger in range
    of the receiver's C memory.
   if zeroBasedEnd == -1,  a search for a NUL byte is done to determine the length
   aClass must be CByteArray, String, ByteArray, Utf8, Utf16, or nil.
   if aClass == nil, result is number of bytes from zeroBasedStart to the
   first NUL byte,  i.e. strlen() "

<primitive: 714>
zeroBasedStart _validateClass: SmallInteger .
zeroBasedEnd _validateClass: SmallInteger .
"at this point byteSize or zeroBasedOffset are probably out of range,
  or aClass is invalid"
^self _primitiveFailed: #_copyFrom:to:resKind:
      args: { zeroBasedStart . zeroBasedEnd . aClass }
]

{ #category : 'Formatting' }
CByteArray >> _inspect [
  | res |
  (res := String new)
    add: ' size='; add: self size asString ;
    add: ' gcFree=0x'; add: self gcFreeKind asHexString ;
    add: ' dead='; add: self isDead asString ;
    add: ' address=0x' ; add: self memoryAddress asHexString .
  ^ res

]

{ #category : 'Updating' }
CByteArray >> _signed: byteSize at: zeroBasedOffset put: aValue [

"byteSize describes bytes of the receiver into which
 aValue is to be stored:
    1, 2, 4, or 8 - unsigned integer of specified size
   -1, -2, -4, -8 - signed integer of specified size
    9 -  8 byte C double
    10 - 4 byte C float
 Stores into the specified bytes of the receiver in CPU native byte order
 the specified least sigificant bytes of aValue .
 There is no required alignment for zeroBasedOffset.
 If aValue is an Integer and would loose bits by truncating
 to the specified size, an error is signaled.
 If aValue is a Float, and truncation would loose exponent bits
 or convert a non-zero to a zero, an error is signalled.
"
 <primitive: 246>
 byteSize _validateClass: SmallInteger .
 zeroBasedOffset _validateClass: SmallInteger .
 (byteSize == 9 or:[ byteSize == 10]) ifTrue:[
   aValue _validateClasses: { Float . SmallFloat } .
 ] ifFalse:[
   (aValue isKindOf: Integer) ifTrue:[
     ArgumentError signal:'argument out of range for destination'
   ].
   aValue _validateClass: Integer.
 ].
 "at this point byteSize is invalid"
 self _primitiveFailed: #_signed:at:put:
     args: { byteSize . zeroBasedOffset . aValue }
]

{ #category : 'Updating' }
CByteArray >> _signed: byteSize at: zeroBasedOffset truncatePut: aValue [

"byteSize describes bytes of the receiver into which
 aValue will be stored:
    -1 - 4 byte C float
    0 -  8 byte C double
    1, 2, 4, or 8 - integer of specified size
    9  -  aValue is a CByteArray( or nil)
          address of Cdata of CByteArray (or NULL)
         is to be stored for pointerAt:put:  ,
          or aValue is a CPointer, the encapsulated C pointer
         is to be stored for pointerAt:put:  ,
   10 - value is a CCallin or nil ,
        The native code address to be used for a call back , or NULL
         is to be stored for ccallinAt:put: .
   11 - value is a CCallout or nil ,
        The native code address to be used for a call out , or NULL
         is to be stored for ccalloutAt:put: .
 Stores into the specified bytes of the receiver in CPU native byte order
 the specified least sigificant bytes of aValue .
 There is no required alignment for zeroBasedOffset.
 If aValue is a Float and truncation to 4 byte C float
 would loose exponent bits and produce an Infinity
 an error is signaled, otherwise aValue is silently
 truncated as needed to fit in the destination bytes.
"

<primitive: 717>
byteSize _validateClass: SmallInteger .
zeroBasedOffset _validateClass: SmallInteger .
aValue ifNotNil:[ aValue _validateClass: CByteArray ].
(byteSize < -1 or:[ byteSize > 11]) ifTrue:[
   ArgumentError new name:'byteSize'; signal:'invalid byteSize arg'
].
byteSize <= 0 ifTrue:[ aValue _validateClasses: { Float . SmallFloat } ].
byteSize == 9 ifTrue:[
 aValue ifNotNil:[ aValue _validateClasses:{ CByteArray . CPointer }]
].
byteSize == 10 ifTrue:[
 aValue ifNotNil:[ aValue _validateClass: CCallin ].
].
byteSize == 11 ifTrue:[
 aValue ifNotNil:[ aValue _validateClass: CCallout ].
].
aValue _validateClass: Integer.
self _primitiveFailed: #_signed:at:truncatePut:
     args: { byteSize . zeroBasedOffset . aValue }

]

{ #category : 'Accessing elements' }
CByteArray >> _signed: byteSize at: zeroBasedOffset [

"byteSize must be 1, 2, 4, 8, or 9 .

 If receiver is of size 8, zeroBasedOffset == -1 , and byteSize 1 2 or 4,
 accesses the value  (self int64At:0) in native byte order and 
 returns the specified number of least signifigant bytes of that value,
 as would have been stored by a function arg
 of type  char* , short*, or int*   
 There is no required alignment for zeroBasedOffset.

 If zeroBasedOffset == -2 , and byteSize == 8,
 returns the starting address of the C memory, as a signed 64bit Integer.

 if byteSize == 9 , treats the 8 bytes at zeroBasedOffset
   as a  char*  pointer and returns nil if the pointer is NULL,
   or a new String based on strlen() of that pointer
"
<primitive: 715>
self _primitiveFailed: #_signed:at:
     args: { byteSize . zeroBasedOffset }
]

{ #category : 'Accessing elements' }
CByteArray >> _pointer: opcode at: zeroBasedOffset with: thirdArg with: fourthArg [
" if opcode == 0,  thirdArg is a subclass of CByteArray or CPointer,
   create an instance of thirdArg which encapsulates a C pointer fetched from
   the specified zeroBasedOffset in receiver's C memory .

 if opcode == 1 , treats the 8 bytes at zeroBasedOffset
   as a char* pointer. thirdArg is number of bytes .
   returns nil if the pointer is NULL,
   or a new String containing specified number of bytes without regard to NUL bytes.

 if opcode == 2, treats the 8 bytes at zeroBasedOffset
   as pointer to UTF8 data. thirdArg is number of bytes .
   returns nil if the pointer is NULL,
   or a new Utf8 containing specified number of bytes without regard to NUL bytes.

 if opcode == 3, treats the 8 bytes at zeroBasedOffset
   as pointer to UTF16 data. thirdArg is number of bytes .
   returns nil if the pointer is NULL,
   or a new Utf16 containing specified number of bytes without regard to NUL bytes.
   number of bytes must be even.

 if opcode == 4, treats the 8 bytes at zeroBasedOffset
   as pointer to ByteArray data. thirdArg is number of bytes .
   returns nil if the pointer is NULL,
   or a new ByteArray containing specified number of bytes without regard to NUL bytes.

 if opcode == 5, thirdArg is a subclass of CByteArray,
  create an instance of thirdArg which encapsulates a C pointer fetched from
   the specified zeroBasedOffset in receiver's C memory ,
   with logical size specified by fourthArg .
"
<primitive: 78>
self _primitiveFailed: #_pointer:at:with:with:
     args: { opcode . zeroBasedOffset . thirdArg . fourthArg}
]

{ #category : 'Accessing elements' }
CByteArray >> _unsigned: byteSize at: zeroBasedOffset [

"For byteSize one of 0(double) 1, 2, 4, 8, 9(float)
   For zeroBasedOffset >= 0 , returns the integer
   of specified byte size , or the 8 byte double,
   starting at specified offset.

 If receiver is of size 8, zeroBasedOffset == -1 , and byteSize 1 2 or 4,
 accesses the value  (self int64At:0) in native byte order and 
 returns the specified number of least signifigant bytes of that value.

 If byteSize == -1, then receiver is searched for a byte value of 0
 starting at zeroBasedOffset, and the zero-based offset of the
 first such byte is returned, or -1 returned of no such byte found.

 There is no required alignment for zeroBasedOffset.
"

<primitive: 716>
self _primitiveFailed: #_unsigned:at:
     args: { byteSize . zeroBasedOffset }

]

{ #category : 'Private' }
CByteArray >> _referencesTo: anArray [
  referencesTo := anArray .
]

{ #category : 'Private' }
CByteArray >> _updateReferences: zeroBasedOffset value: anObject [
  | childs idx |
  anObject ifNotNil:[
    (childs := referencesTo) ifNil:[
      childs  := Array new .
      referencesTo := childs .
    ].
    childs size < (idx := zeroBasedOffset+1) ifTrue:[ childs size: idx ].
    childs at: idx put: anObject .
  ] ifNil:[
    (childs := referencesTo) ifNotNil:[
      idx := zeroBasedOffset + 1.
      childs size >= idx ifTrue:[ childs at: idx put: nil ].
    ]
  ]

]

{ #category : 'Formatting' }
CByteArray >> asString [
  | res |
  (res := super asString ) add: self _inspect .
  ^ res

]

{ #category : 'Accessing' }
CByteArray >> autoRelease [
  ^ (info bitAnd: AutoFree) ~~ 0  "fixed 48558"

]

{ #category : 'Updating' }
CByteArray >> autoRelease: aBoolean [

"change the auto-free behavior of the receiver's C to match aBoolean.
"

<primitive: 866>
self _primitiveFailed: #autoRelease: args: { aBoolean }

]

{ #category : 'Copying' }
CByteArray >> byteArrayFrom: zeroBasedStart to: zeroBasedEnd [

 "return a new ByteArray containing the specified bytes of the receiver."

^ self _copyFrom: zeroBasedStart to: zeroBasedEnd resKind: ByteArray

]

{ #category : 'Copying' }
CByteArray >> byteArrayFrom: zeroBasedStart numBytes: anInteger [

 "return a new ByteArray containing the specified bytes of the receiver."

  anInteger <= 0 ifTrue:[ ArgumentError signal:'number of bytes argument must be > 0'].

^ self _copyFrom: zeroBasedStart to: (zeroBasedStart + anInteger - 1) resKind: ByteArray

]

{ #category : 'Copying' }
CByteArray >> CByteArrayFrom: zeroBasedStart to: zeroBasedEnd [
 "return a new CByteArray , auto freed, containing specified bytes of receiver."

^ self _copyFrom: zeroBasedStart to: zeroBasedEnd resKind: CByteArray

]

{ #category : 'Updating' }
CByteArray >> ccallinAt: zeroBasedOffset put: aCCallin [
  "Store the C address for a bound CCallin. Returns receiver."
  self _signed: 10 at: zeroBasedOffset truncatePut: aCCallin .
  self _updateReferences: zeroBasedOffset value: aCCallin .

]

{ #category : 'Accessing elements' }
CByteArray >> ccalloutAt: zeroBasedStart name: aString result: resType args: argumentTypes [
  "Return a CCallout which encapsulates the pointer at zeroBasedStart in
   the receiver as a C function that can be be called.
   resType and argumentTypes must conform to documentation in
     CCallout class >> library:name:result:args:
   "
  | fAddr cco |
  fAddr := self pointerAt: zeroBasedStart resultClass: CPointer .
  cco := CCallout library: nil name: aString result: resType args: argumentTypes  .
  cco _bindFunction: fAddr .
  ^ cco

]

{ #category : 'Updating' }
CByteArray >> ccalloutAt: zeroBasedOffset put: aCCallout [
  "Store the C address for a CCallout. Returns receiver."

  (aCCallout isKindOf: CCallout) ifFalse:[ aCCallout _validateClass: CCallout].
  aCCallout bind .
  self _signed: 11 at: zeroBasedOffset truncatePut: aCCallout

]

{ #category : 'Copying' }
CByteArray >> shallowCopy [
  "Returns a copy of the receiver which shares the receiver's instance variables. 
   If  self autoRelease == true,
   the result's C memory will be a copy of the C memory of the receiver,
   and  will have   result autoRelease == true  . 
   Otherwise the result will contain a reference to the C memory of the receiver,
   and  will have   result autoRelease == false . "
  <primitive: 80>
  self _primitiveFailed: #shallowCopy .
]

{ #category : 'Copying' }
CByteArray >> postCopy [
  referencesTo ifNotNil:[:v| referencesTo := v copy ].
  ^ self
]

{ #category : 'Updating' }
CByteArray >> copyBytesFrom: anObject from: oneBasedStart to: oneBasedEnd into: zeroBasedDestOffset [
  ^ self copyBytesFrom: anObject from: oneBasedStart to: oneBasedEnd into: zeroBasedDestOffset
       allowCodePointZero: false .
]

{ #category : 'Updating' }
CByteArray >> copyBytesFrom: anObject from: oneBasedStart to: oneBasedEnd into: zeroBasedDestOffset allowCodePointZero: zeroBoolean [
  "Copy the specified bytes of anObject into the receiver at the given offset.
   anObject may be any byte format object or a CByteArray.

 An error is signalled of any of the following are true
    oneBasedStart < 1
    zeroBasedDestOffset < 0
    self size > 0 and:[ zeroBasedDestOffset >= self size ]

 If zeroBoolean==false and anObject is not a CByteArray 
 and anObject contains a byte, word, or codePoint with value zero,
 signals an error .

 Returns number of bytes copied (possibly zero) .

 Warning, ByteArray usually stores integers in big-endian order,
 not CPU native byte order. Most other byte format classes
 use CPU native byte order."

<primitive: 713>
zeroBasedDestOffset _validateClass: SmallInteger .
oneBasedStart _validateClass: SmallInteger .
oneBasedEnd _validateClass: SmallInteger .
zeroBoolean _validateClass: Boolean .

"if we get here, either receiver or a CByteArray passed as anObject
 is probably a committed instance that has been faulted in and has no Cdata."
self _primitiveFailed: #copyBytesFrom:from:to:into:
     args: { anObject . oneBasedStart .  oneBasedEnd . zeroBasedDestOffset . zeroBoolean }

]

{ #category : 'Copying' }
CByteArray >> copyFrom: anObj from: oneBasedStart to: oneBasedEnd into: zeroBasedDestOffset [

"anObject may be a String, ByteArray or CByteArray .
 copies specified bytes of anObject into receiver .

 An error is signalled of any of the following are true
    oneBasedStart < 1
    zeroBasedDestOffset < 0
    self size > 0 and:[ zeroBasedDestOffset >= self size ]

 The number of bytes copied may be zero.
 The method copyBytesFrom:from:to:into:   returns the number of bytes copied.

 Warning, ByteArray usually stores integers in big-endian order,
 not CPU native byte order."

 self copyBytesFrom: anObj from: oneBasedStart to: oneBasedEnd into: zeroBasedDestOffset allowCodePointZero: true .
 ^ self

]

{ #category : 'Accessing' }
CByteArray >> derivedFrom: aCByteArray [
  "used to ensure that aCByteArray is kept alive by GC as long
   as receiver is alive. "
  derivedFrom := aCByteArray

]

{ #category : 'Accessing elements' }
CByteArray >> doubleAt: zeroBasedOffset [
 "fetch 8 byte float and return a Float or SmallDouble"
 ^ self _unsigned: 0 at: zeroBasedOffset

]

{ #category : 'Updating' }
CByteArray >> doubleAt: zeroBasedOffset put: aFloat [
  "store a Float or SmallDouble into 64bits"
  self _signed: 9 at: zeroBasedOffset put: aFloat

]

{ #category : 'Updating' }
CByteArray >> doubleAt: zeroBasedOffset truncatePut: aFloat [
  "store a Float or SmallDouble into 64bits"
  self _signed: 0 at: zeroBasedOffset truncatePut: aFloat

]

{ #category : 'Accessing elements' }
CByteArray >> floatAt: zeroBasedOffset [
 "fetch 4 byte float and return a Float or SmallDouble"
 ^ self _unsigned: 9 at: zeroBasedOffset

]

{ #category : 'Updating' }
CByteArray >> floatAt: zeroBasedOffset put: aFloat [
  "store a Float or SmallDouble into 32bits.
  if aFloat would loose exponent bits
  or convert a non-zero to a zero when converted to
  a 4 byte C float, signals an error."
  self _signed: 10 at: zeroBasedOffset put: aFloat

]

{ #category : 'Updating' }
CByteArray >> floatAt: zeroBasedOffset truncatePut: aFloat [
  "store a Float or SmallDouble into 32bits"
  self _signed: -1 at: zeroBasedOffset truncatePut: aFloat

]

{ #category : 'Accessing' }
CByteArray >> gcFreeKind [
  ^ info bitAnd: GcFreeMask

]

{ #category : 'Private' }
CByteArray >> instVarAt: offset put: value [

 "disallowed"
 self shouldNotImplement: #instVarAt:put:

]

{ #category : 'Accessing elements' }
CByteArray >> int16At: zeroBasedOffset [
 "return specified signed 16 bit integer"
  ^ self _signed: 2 at: zeroBasedOffset  

]

{ #category : 'Updating' }
CByteArray >> int16At: zeroBasedOffset put: anInteger [
  "check that anInteger is representable in 16 bits and store"
  self _signed: -2 at: zeroBasedOffset put: anInteger

]

{ #category : 'Updating' }
CByteArray >> int16At: zeroBasedOffset truncatePut: anInteger [
  "truncate anInteger to 16 bits and store"
  self _signed: 2 at: zeroBasedOffset truncatePut: anInteger

]

{ #category : 'Accessing elements' }
CByteArray >> int32At: zeroBasedOffset [
 "return specified signed 32 bit integer"
  ^ self _signed: 4 at: zeroBasedOffset  

]

{ #category : 'Updating' }
CByteArray >> int32At: zeroBasedOffset put: anInteger [
  "check that anInteger is representable in 32 bits and store"
  self _signed: -4 at: zeroBasedOffset put: anInteger

]

{ #category : 'Updating' }
CByteArray >> int32At: zeroBasedOffset truncatePut: anInteger [
  "truncate anInteger to 32 bits and store"
  self _signed: 4 at: zeroBasedOffset truncatePut: anInteger

]

{ #category : 'Accessing elements' }
CByteArray >> int64At: zeroBasedOffset [

 "return specified signed 64 bit integer"
  ^ self _signed: 8 at: zeroBasedOffset  

]

{ #category : 'Updating' }
CByteArray >> int64At: zeroBasedOffset put: anInteger [
  "check that anInteger is representable in 64 bits and store"
  self _signed: -8 at: zeroBasedOffset put: anInteger

]

{ #category : 'Updating' }
CByteArray >> int64At: zeroBasedOffset truncatePut: anInteger [
  "truncate anInteger to 64 bits and store"
  self _signed: 8 at: zeroBasedOffset truncatePut: anInteger

]

{ #category : 'Accessing elements' }
CByteArray >> int8At: zeroBasedOffset [

 "return specified signed 8 bit integer"
  ^ self _signed: 1 at: zeroBasedOffset  

]

{ #category : 'Updating' }
CByteArray >> int8At: zeroBasedOffset put: anInteger [
  "check that anInteger is representable in 8 bits and store"
  self _signed: -1 at: zeroBasedOffset put: anInteger

]

{ #category : 'Updating' }
CByteArray >> int8At: zeroBasedOffset truncatePut: anInteger [
  "truncate anInteger to 8 bits and store"
  self _signed: 1 at: zeroBasedOffset truncatePut: anInteger

]

{ #category : 'Accessing' }
CByteArray >> isDead [
  ^ (info bitAnd: AppDeadMask) ~~ 0

]

{ #category : 'Accessing elements' }
CByteArray >> memoryAddress [
  "Returns the starting address of the C memory, as a signed 64bit Integer."
  ^ self _signed: 8 at: -2 

]

{ #category : 'Updating' }
CByteArray >> memset: anInt from: zeroBasedStart to: zeroBasedEnd [

 "set specified bytes of the receiver to the value anInt .
  end==-1 means to the end of the receiver.
  anInt must be >=0  and <= 255 . "

 <primitive: 712>
 | sz |
 anInt _validateClass: SmallInteger .
 (anInt < 0 or:[ anInt > 255]) ifTrue:[
   anInt _error: #rtErrArgOutOfRange args:{ 0 . 255 }
 ].
 zeroBasedStart _validateClass: SmallInteger .
 sz := self size .
 (zeroBasedStart < 0 or:[ zeroBasedStart >= sz ]) ifTrue:[
   zeroBasedStart _error: #rtErrArgOutOfRange args:{ 0 . sz - 1 }
 ].
 zeroBasedEnd _validateClass: SmallInteger .
 (zeroBasedEnd < -1 or:[ zeroBasedEnd >= sz ]) ifTrue:[
   zeroBasedEnd _error: #rtErrArgOutOfRange: args:{ -1 . sz - 1 }
 ].
 "if we get here the receiver is probably a committed instance that
  has been faulted in and has no Cdata."
 self _primitiveFailed: #memset:from:to:
      args: { anInt . zeroBasedStart . zeroBasedEnd }

]

{ #category : 'Instance Creation' }
CByteArray >> newFrom: anOffset numBytes: anInt [
 "Returns a new instance of CByteArray representing a portion
  of the receiver.  no fence, not auto freed,
  anInt == -1 means  numBytes == strlen(self.memory) - anOffset .
  anInt == -2 means  numBytes == self.memory_size - anOffset .
  anOffset is zero based"

^ CByteArray _newFrom: self offset: anOffset numBytes: anInt gcFree: 0

]

{ #category : 'Updating' }
CByteArray >> pointerAt: zeroBasedOffset put: aCByteArray [
  self _signed: 9 at: zeroBasedOffset truncatePut: aCByteArray .
  self _updateReferences: zeroBasedOffset value: aCByteArray .

]

{ #category : 'Accessing elements' }
CByteArray >> pointerAt: zeroBasedOffset resultClass: aClass [
  "Returns an instance of aClass encapsulating the C pointer
   fetched from 8 bytes of the receiver starting at zeroBasedOffset.
   zeroBasedOffset may need to be aligned on 8 bytes on some CPUs.

   If the C pointer at the specified offset is NULL, this method returns nil .

   aClass must be CByteArray, CPointer, a subclass of CByteArray
   or a subclass of CPointer.
  "
  ^ self _pointer: 0 at: zeroBasedOffset with: aClass with: 0
]

{ #category : 'Accessing elements' }
CByteArray >> pointerAt: zeroBasedOffset resultClass: aClass numBytes: aSize [
  "Returns an instance of aClass encapsulating the C pointer
   fetched from 8 bytes of the receiver starting at zeroBasedOffset.
   zeroBasedOffset may need to be aligned on 8 bytes on some CPUs.

   If the C pointer at the specified offset is NULL, this method returns nil .

   aClass must be CByteArray, CPointer, a subclass of CByteArray
   or a subclass of CPointer.
  "
  ^ self _pointer: 5 at: zeroBasedOffset with: aClass with: aSize
]


{ #category : 'Accessing' }
CByteArray >> setDead [
  "sets the isDead bit, without marking object dirty"
  <primitive: 847>
  self _primitiveFailed: #setDead

]

{ #category : 'Accessing' }
CByteArray >> size [
  "The result may be zero if the receiver encapsulates memory for
   which the size is not known, such as instances created by class methods
     fromCharStar:
     fromCharStarGcFree:
   Instances created by the class methods
     fromCPointer:numBytes:
     fromCPointerGcFree:numBytes:
     fromRegionOf:offset:numBytes:
   will haave the size known ."

  ^ info bitShift: -4
]

{ #category : 'Copying' }
CByteArray >> stringFrom: zeroBasedStart to: zeroBasedEnd [
 "return a new String containing the specified bytes of the receiver."

^ self _copyFrom: zeroBasedStart to: zeroBasedEnd resKind: String

]


{ #category : 'Copying' }
CByteArray >> stringFrom: zeroBasedStart [

 "return a new String containing the bytes of the receiver 
  from specified start byte to the first byte preceding a zero byte.
 
  Interprets the data starting at &body[zeroBasedStart] as 
  NUL terminated char*  data , and returns an instance of String containing
  that data."

^ self _copyFrom: zeroBasedStart to: -1 resKind: String
]

{ #category : 'Copying' }
CByteArray >> stringFrom: zeroBasedStart numBytes: anInteger [

 "return a new String containing the specified bytes of the receiver  .
 
  Interprets the data starting at &body[zeroBasedStart] as 
  char*  data , and returns an instance of String containing
  the specified number of bytes without regard to any NUL characters."

  anInteger <= 0 ifTrue:[ ArgumentError signal:'number of bytes argument must be > 0'].

^ self _copyFrom: zeroBasedStart to: zeroBasedStart + anInteger - 1 resKind: String
]

{ #category : 'Accessing elements' }
CByteArray >> stringFromCharStarAt: zeroBasedOffset [
  "Using the 8 bytes starting at zeroBasedOffset as a 'char *' type, 
   return a String with the contents of the referenced  NUL terminated C-string."

  ^ self _signed: 9 at: zeroBasedOffset 
]

{ #category : 'Accessing elements' }
CByteArray >> stringFromCharStarAt: zeroBasedOffset numBytes: anInteger [
  "Using the 8 bytes starting at zeroBasedOffset as a 'char *' type, 
   return a String containing specified number bytes without regard to NUL bytes"
  
  anInteger <= 0 ifTrue:[ ArgumentError signal:'number of bytes argument must be > 0'].

  ^ self _pointer: 1 at: zeroBasedOffset with: anInteger with: 0
]

{ #category : 'Accessing elements' }
CByteArray >> utf8FromCharStarAt: zeroBasedOffset numBytes: anInteger [
  "Using the 8 bytes starting at zeroBasedOffset as a 'char *' type, 
   return a Uf8 with specified number of bytes."
  
  anInteger <= 0 ifTrue:[ ArgumentError signal:'number of bytes argument must be > 0'].

  ^ self _pointer: 2 at: zeroBasedOffset with: anInteger with: 0
]

{ #category : 'Accessing elements' }
CByteArray >> byteArrayFromCharStarAt: zeroBasedOffset numBytes: anInteger [
  "Using the 8 bytes starting at zeroBasedOffset as a 'char *' type, 
   return a ByteArray with specified number of bytes."
  
  anInteger <= 0 ifTrue:[ ArgumentError signal:'number of bytes argument must be > 0'].

  ^ self _pointer: 4 at: zeroBasedOffset with: anInteger with: 0
]

{ #category : 'Accessing elements' }
CByteArray >> utf16FromPointerAt: zeroBasedOffset numBytes: anInteger [
  "Using the 8 bytes starting at zeroBasedOffset as a 'ushort *' type, 
   return a Utf16 with specified number of bytes."
  
  anInteger <= 0 ifTrue:[ ArgumentError signal:'number of bytes argument must be > 0'].
  (anInteger bitAnd: 1) ~~ 0 ifTrue:[ ArgumentError signal:'number of bytes must be even'].

  ^ self _pointer: 3 at: zeroBasedOffset with: anInteger with: 0
]

{ #category : 'Accessing elements' }
CByteArray >> uint16At: zeroBasedOffset [
 "return specified unsigned 16 bit integer"
 ^ self _unsigned: 2 at: zeroBasedOffset

]

{ #category : 'Updating' }
CByteArray >> uint16At: zeroBasedOffset put: anInteger [
  "check that anInteger is representable in 16 bits and store"
  self _signed: 2 at: zeroBasedOffset put: anInteger

]

{ #category : 'Accessing elements' }
CByteArray >> uint32At: zeroBasedOffset [
 "return specified unsigned 32 bit integer"
 ^ self _unsigned: 4 at: zeroBasedOffset

]

{ #category : 'Updating' }
CByteArray >> uint32At: zeroBasedOffset put: anInteger [
  "check that anInteger is representable in 32 bits and store"
  self _signed: 4 at: zeroBasedOffset put: anInteger

]

{ #category : 'Accessing elements' }
CByteArray >> uint64At: zeroBasedOffset [
 "return specified unsigned 64 bit integer"
 ^ self _unsigned: 8 at: zeroBasedOffset

]

{ #category : 'Updating' }
CByteArray >> uint64At: zeroBasedOffset put: anInteger [
  "check that anInteger is representable in 64 bits and store"
  self _signed: 8 at: zeroBasedOffset put: anInteger

]

{ #category : 'Accessing elements' }
CByteArray >> uint8At: zeroBasedOffset [
 "return specified unsigned 8 bit integer"
 ^ self _unsigned: 1 at: zeroBasedOffset

]

{ #category : 'Updating' }
CByteArray >> uint8At: zeroBasedOffset put: anInteger [
  "check that anInteger is representable in 8 bits and store"
  self _signed: 1 at: zeroBasedOffset put: anInteger

]

{ #category : 'Copying' }
CByteArray >> utf16From: zeroBasedStart to: zeroBasedEnd [

 "return a new Utf16 containing the specified bytes of the receiver."

^ self _copyFrom: zeroBasedStart to: zeroBasedEnd resKind: Utf16

]

{ #category : 'Copying' }
CByteArray >> utf8From: zeroBasedStart to: zeroBasedEnd [

 "return a new Utf8 containing the specified bytes of the receiver."

^ self _copyFrom: zeroBasedStart to: zeroBasedEnd resKind: Utf8

]


{ #category : 'Updating' }
CByteArray >> addArrayOfByteObjects: arrayOfByteObjs extraNullPointer: addExtraNull [
  "Write the elements of arrayOfByteObjs, which must be a kinds of String (single-byte),
   to the receiver, starting at offset 1 of the argument and overwriting existing contents.
   Elements of arrayOfByteObjs must be a kinds of String,
   DoubleByteString and QuadByteString are not allowed.
   Signals an error if any of the elements of arrayOfByteObjs contain codePoint zero."

|arySize stringAddress stringOffset ptrOffset |
stringOffset := (arySize := arrayOfByteObjs size) * 8 .
"Leave room for terminating NULL in list of pointers if requested"
addExtraNull ifTrue:[ stringOffset := stringOffset + 8].
stringAddress := self memoryAddress + stringOffset .
ptrOffset := 0 .
1 to: arySize do:[:n | | e sz |
  e := arrayOfByteObjs at: n .
	sz := e size.
	self copyBytesFrom: e from: 1 to: sz into: stringOffset allowCodePointZero: false .
	stringOffset := stringOffset + sz .
	self uint8At: stringOffset put: 0 .
	stringOffset := stringOffset + 1 .
	self int64At: ptrOffset put: stringAddress  .
	ptrOffset := ptrOffset + 8 .
	stringAddress := stringAddress + (sz + 1) .
  ].
"Add terminating NULL to list of pointers if requested"
addExtraNull ifTrue:[ self int64At: ptrOffset put: 0 ].  
^self
]

{ #category : 'Updating' }
CByteArray >> addArrayOfInt32: arrayOfInt32 [
  "Write the elements of arrayOfInt32 to the receiver, starting at offset 1 of the
   argument, and overwriting existing contents."

|offset|
offset := 0.
1 to: arrayOfInt32 size do:[:n| 
  self int32At: offset put: (arrayOfInt32 at: n).
  offset := offset + 4 
].
^self

]

{ #category : 'Updating' }
CByteArray >> addArrayOfInt64: arrayOfInt64 [
  "Write the elements of arrayOfInt64 to the receiver, starting at offset 1 of the
   argument and overwriting existing contents."

|offset|
offset := 0.
1 to: arrayOfInt64 size do:[:n | 
  self int64At: offset put: (arrayOfInt64 at: n).
  offset := offset + 8 
].
^self

]

{ #category : 'Instance Creation' }
CByteArray class >> fromArrayOfInt64: arrayOfInt64s [
  "Takes an array of 64-bit ints and creates a CByteArray."

| inst |
inst := self gcMalloc: (self computeSizeForArrayOfInt64: arrayOfInt64s).
^ inst addArrayOfInt64: arrayOfInt64s

]

{ #category : 'Instance Creation' }
CByteArray class >> fromArrayOfInt32: arrayOfInt32s [
  "Takes an array of 32-bit ints and creates a CByteArray."

| inst |
inst := self gcMalloc: (self computeSizeForArrayOfInt32: arrayOfInt32s).
^ inst addArrayOfInt32: arrayOfInt32s

]

{ #category : 'Instance Creation' }
CByteArray class >> computeSizeForArrayOfInt32: arrayOfInt32 [
  "Validate each element and compute the number of bytes needed to contain arrayOfInt32 in 
   an instance of the receiver which will represent an array of bytes in C."

| min max siz |
max := SmallInteger maximum32bitInteger .
min := SmallInteger minimum32bitInteger .
1 to: (siz := arrayOfInt32 size) do:[:n | | e |
    e := arrayOfInt32 at: n .
    e _isSmallInteger ifFalse:[  e _validateClass: SmallInteger ].
		(e < min or:[ e > max]) 
				ifTrue:[ e _error: #rtErrArgOutOfRange args: { min . max } ]
  ].
^ 4 * siz 
]

{ #category : 'Instance Creation' }
CByteArray class >> computeSizeForArrayOfInt64: arrayOfInt64 [
  "Validate each element and compute the number of bytes needed to contain arrayOfInt64 in 
   an instance of the receiver which will represent an array of bytes in C."

  | sz |
  1 to: (sz := arrayOfInt64 size) do:[:n| | e |
    (e := arrayOfInt64 at:n) _isSmallInteger ifFalse:[ 
       e _isInteger ifTrue:[ | min max |
         (e > (max:=16r7fffffffffffffff) or:[ e < (min:=-16r8000000000000000)]) ifTrue:[ 
            e _error: #rtErrArgOutOfRange args: { min . max }  
          ].
       ] ifFalse:[
         e _validateClass: Integer ].
       ].
  ].
  ^ 8 * sz
]

{ #category : 'Instance Creation' }
CByteArray class >> computeSizeForArrayOfByteObjects: arrayOfByteObjs extraNullPointer: addExtraNull [
  "Compute the number of bytes needed to contain arrayOfByteObjs in
   an instance of the receiver which will represent an array of bytes in C.
   Each element of arrayOfByteObjs must be a String . 
   DoubleByteString or QuadByteString are not allowed.
   Space for a NUL character for each element is included in the total.
   If addExtraNull is true, space for an extra NULL element in the array of C
   pointers will be included (some C functions require a terminating NULL)."

| totalBytes sz |
totalBytes := 0.

"Validate each element and add up sizes"
1 to: (sz := arrayOfByteObjs size) do:[:n | | e | 
  e := arrayOfByteObjs at: n .
  e charSize == 1 ifFalse:[ e _validateIsBytes ; _validateCharSizeIs: 1 ].
	totalBytes := totalBytes + e size .
 ].
totalBytes := totalBytes + sz . "one NUL terminating each string"
totalBytes := totalBytes +  (8 * sz)  .  "Add size for 8 byte pointers"
addExtraNull ifTrue:[ 
  totalBytes := totalBytes + 8  "Add space for terminating NULL pointer"
  ].
^ totalBytes
]

{ #category : 'Instance Creation' }
CByteArray class >> computeSizeForArrayOfUtf8Encoded: arrayOfStrings extraNullPointer: extraNullBoolean [
  "Compute the number of bytes needed to contain arrayOfStrings in an instance of
   the receiver which will represent an array of bytes in C. Elements in arrayOfStrings
   must be kinds of String or MultiByteString. Space for a NULL character for each
   element is included in the total. If extraNullBoolean is true, space for an additional
   NULL element in the array of C pointers will be included."

| totalBytes sz |
totalBytes := 0.
1 to: (sz :=arrayOfStrings size) do:[:n |
  totalBytes := totalBytes + (arrayOfStrings at: n) sizeForEncodeAsUTF8
 ].
totalBytes := totalBytes + sz . "one NUL terminating each encoded string."
totalBytes := totalBytes + (sz * 8) .  "one pointer for each string"
extraNullBoolean ifTrue:[ totalBytes := totalBytes + 8 ].
^ totalBytes
]


{ #category : 'Instance Creation' }
CByteArray class >> fromArrayOfByteObjects: arrayOfByteObjs extraNullPointer: addExtraNull [
  "Takes an array of byte objects and creates a C array of pointers to bytes.
   Each element of arrayOfByteObjs must be a String; 
   DoubleByteString or QuadByteString are not allowed.
   Space for a NUL character for each element is included in the total.
   If addExtraNull is true, an extra NULL pointer is appended to the list
   of pointers to terminate the list (some C functions require a terminating NULL)."

| inst |
inst := self gcMalloc:(self computeSizeForArrayOfByteObjects: arrayOfByteObjs 
                              extraNullPointer: addExtraNull).
^ inst addArrayOfByteObjects: arrayOfByteObjs extraNullPointer: addExtraNull

]

{ #category : 'Instance Creation' }
CByteArray class >> fromArrayEncodeUtf8: arrayOfStrings extraNullPointer: addExtraNull [
  "Takes an array of String objects and creates a C array of pointers to UTF8 encoded bytes.
   Each element of arrayOfStrings must be a String or MultiByteString . 
   Space for a NUL character for each element is included in the total.
   If addExtraNull is true, an extra NULL pointer is appended to the list
   of pointers to terminate the list (some C functions require a terminating NULL)."

| inst |
inst := self gcMalloc: (self computeSizeForArrayOfUtf8Encoded: arrayOfStrings 
                              extraNullPointer: addExtraNull).
^ inst addUtf8Encoded: arrayOfStrings extraNullPointer: addExtraNull
]

{ #category: 'Copying' }
CByteArray >> strlenFrom: zeroBasedStart [
  "Return result of strlen() starting from the byte of receiver specified by zeroBasedStart."

  ^ self _copyFrom: zeroBasedStart to: -1 resKind: nil .
]

{ #category: 'Copying' }
CByteArray >> _stringFromBytes: aByteArray [
  "senders of this method are generated by CHeader >> wrapperNamed:forStruct "
  | index |
  index := aByteArray indexOf: 0.
  ^aByteArray
    at: 1
    sizeBytes: 1
    stringSize: (0 == index ifTrue: [aByteArray size] ifFalse: [index - 1]).
]

{ #category : 'Updating' }
CByteArray >> encodeUTF8From: anObject into: zeroBasedDestOffset allowCodePointZero: zeroBoolean [
 "anObject may be a String or MultiByteString .
  The codepoints in anObject are encoded into UTF8 and the resulting
  UTF8 bytes are copied into the receiver starting at zeroBasedDestOffset . 

  If zeroBoolean==false, signals an Error if anObject contains codePoint zero.

  Returns a SmallInteger, possibly zero, the number of UTF8 bytes copied into
  the receiver.  There is no terminating NUL byte written to the destination"

 <primitive: 331>
 zeroBasedDestOffset _validateClass: SmallInteger .
 anObject _validateClasses:{ String . MultiByteString }.
 zeroBoolean _validateClass: Boolean .
 self _primitiveFailed: #encodeUTF8From:into:allowCodePointZero: 
     args: { anObject . zeroBasedDestOffset . zeroBoolean }
]

{ #category : 'Copying' }
CByteArray >> decodeUTF8from: zeroBasedStart to: zeroBasedEnd unicode: unicodeBoolean [
  "Decode the UTF8 encoded bytes of the receiver from zeroBasedStart to zeroBasedEnd ;
   if unicodeBoolean==false returns a String, DoubleByteString or QuadByteString ,
   if unicodeBoolean==true  returns a Unicode7 , Unicode16 or Unicode32 . 
  If zeroBasedEnd == -1,  strlen() is used starting from zeroBasedStart
  to determine the number of bytes to be decoded."
 
 <primitive: 357>
 zeroBasedStart _validateClass: SmallInteger .
 zeroBasedEnd _validateClass: SmallInteger .
 unicodeBoolean _validateClass: Boolean .
 self _primitiveFailed: #decodeUTF8from:to:unicode: args: { zeroBasedStart . zeroBasedEnd . unicodeBoolean }.
]

{ #category : 'Updating' }
CByteArray >> addUtf8Encoded: arrayOfStrings extraNullPointer: addExtraNull [
  "Write the elements of arrayOfStrings, which must be a kinds of String or
   MultiByteString, to the receiver, starting at offset 1 of the argument and 
   overwriting existing contents.
   Signals an Error if any of the elements of arrayOfStrings contain codePoint zero ."

| arySize stringAddress stringOffset ptrOffset |
stringOffset := (arySize := arrayOfStrings size) * 8 .
"Leave room for terminating NULL in list of pointers if requested"
addExtraNull ifTrue:[ stringOffset := stringOffset + 8].
stringAddress := self memoryAddress + stringOffset .
ptrOffset := 0 .
1 to: arySize do:[:n | |e sz |
  e := arrayOfStrings at: n .
	sz := self encodeUTF8From: e into: stringOffset allowCodePointZero: false .
	stringOffset := stringOffset + sz .
	self uint8At: stringOffset put: 0 .
	stringOffset := stringOffset + 1 .
	self int64At: ptrOffset put: stringAddress  .
	ptrOffset := ptrOffset + 8 .
	stringAddress := stringAddress + (sz + 1) .
].
"Add terminating NULL to list of pointers if requested"
addExtraNull ifTrue:[ self int64At: ptrOffset put: 0 ].  
^self

]

