"
This class is internal to the FFI implementation and 
represents a declaration from a C header file.
  instVars
     count -- nil or an Array of elements(or sizes?) used to define a constant size array .
  order of instVars chosen so instVars most needed for debugging are at the front.

    bitCount non-zero for field of a struct like     unsigned int c : 7; 
"
Class {
	#name : 'CDeclaration',
	#superclass : 'Object',
	#instVars : [
		'name',
		'type',
		'pointer',
		'fields',
		'parameters',
		'source',
		'storage',
		'file',
		'line',
		'count',
		'isVaryingArgCount',
		'enumTag',
		'isConstant',
		'includesCode',
		'bitCount',
		'header',
		'linkageSpec'
	],
	#category : 'FFI'
}

{ #category : 'other' }
CDeclaration class >> header: aHeader [
	"Return a CDeclaration (might not be self)"

	^(self readTypeFrom: aHeader) readDeclaration

]

{ #category : 'other' }
CDeclaration class >> readInitializer [
	"Since we don't have a real expression parser, we will go till we reach
	a comma, semicolon, or close parenthesis not in a nesting"

	| token parenthesisCount curlyBracketCount |
	(token := self peek) ifNil: [^self].
	token value = '__attribute__' ifTrue: [
		self next.
		self readAttributes.
		^self readInitializer
	].
	token isEqualsToken ifFalse: [^self].
	self next.
	parenthesisCount := 0.
	curlyBracketCount := 0.
	[true] whileTrue: [ | pkk |
		(pkk := self peek) ifNil: [ token error:'next token is empty' ].
    token := pkk .
		(parenthesisCount == 0 and: [curlyBracketCount == 0]) ifTrue: [
			token isCommaToken ifTrue: [^self].
			token isSemicolonToken ifTrue: [^self].
			token isCloseParenthesisToken ifTrue: [^self].
		].
		token isOpenParenthesisToken ifTrue: [parenthesisCount := parenthesisCount + 1] ifFalse: [
		token isCloseParenthesisToken ifTrue: [parenthesisCount := parenthesisCount - 1] ifFalse: [
		token isOpenCurlyBracketToken ifTrue: [curlyBracketCount := curlyBracketCount + 1] ifFalse: [
		token isCloseCurlyBracketToken ifTrue: [curlyBracketCount := curlyBracketCount - 1]]]].
		self next.
	].

]

{ #category : 'other' }
CDeclaration class >> readTypeFrom: aHeader [
	"Return a CDeclaration (might not be self)"

	^self basicNew readTypeFrom: aHeader

]

{ #category : 'Reporting' }
CDeclaration >> _addToReport: rpt [
  | brace |
  (name ~~ nil ) ifTrue:[ rpt add: (brace := '{') ].
  name ifNotNil:[:n | rpt add: n ; add: ' ' ; add: self byteSize asString ; add: ' '].
  fields ifNotNil:[:f | f do:[ :aField | aField _addToReport: rpt ]].
  (pointer = 0 and:[ type isKindOf: CDeclaration]) ifTrue:[ type _addToReport: rpt ].
  brace ifNotNil:[ rpt add:'}' . ]

]

{ #category : 'accessors' }
CDeclaration >> _fields [

	^fields.

]

{ #category : 'other' }
CDeclaration >> _next [
	| next |
	(next := header next) ifNil: [^nil].
	file ifNil: [
		file := next file.
		line := next line. 
    self trapLine .
	].
	^next.

]

{ #category : 'CCallout' }
CDeclaration >> addSourceTo: aStream [
  | fn |
  file ifNil:[ 
    fn := 'test'
  ] ifNotNil:[ 
    fn := file .
    fn size > 0 ifTrue:[ | gsIncl exp |
      gsIncl := GsFile _expandEnvVariable:'GEMSTONE' isClient: false .
      gsIncl := GsFile _expandFilename: gsIncl isClient: false . "follow symlinks"
      (gsIncl at: 1 equals: (exp := '/export')) ifTrue:[
         gsIncl := gsIncl copyFrom: exp size + 1  to: gsIncl size .
      ].
      gsIncl ifNotNil:[  | ofs |
        ofs := 1 .
        gsIncl last == $/ ifFalse:[ gsIncl add: $/ ].
        gsIncl add: 'include' .
        (fn at: 1 equals: exp) ifTrue:[ ofs := ofs + exp size ].
        (fn at: ofs  equals: gsIncl) ifTrue:[ 
          fn := '$GEMSTONE/include' , (fn copyFrom: gsIncl size + ofs  to: fn size)
        ].
        (System gemVersionReport at: 'gsBuildArchitecture') = 'Darwin (macOS)' ifTrue:[
          (fn at: 1 equals: '/Volumes') ifTrue:[ "Darwin"
            fn := fn copyFrom: '/Volumes' size + 1 to: fn size .
            (fn at: 1 equals: gsIncl) ifTrue:[ 
              fn := '$GEMSTONE/include' , (fn copyFrom: gsIncl size + 1 to: fn size)
            ].
          ].
        ]
      ].
    ].
  ].
	aStream 
		lf; tab; nextPut: $";
		nextPutAll: fn ;
		nextPutAll: ' line ';
		nextPutAll: line printString; lf .
  source ifNotNil:[:src | 
    src trimSeparators do: [:char | 
    aStream nextPut: char. char == $" ifTrue: [aStream nextPut: char]].
  ].
	aStream nextPut: $"; lf.
]

{ #category : 'CByteArray get' }
CDeclaration >> arrayAccessorForOffset: anInteger [

	^(AppendStream on: String new)
		nextPutAll: name; lf; tab;
		nextPutAll: '| array offset | "generated by CDeclaration>>arrayAccessorForOffset:"'; 
        lf; tab;
		nextPutAll: 'array := Array new: ';
		nextPutAll: count first printString;
		nextPut: $.; lf; tab;
		nextPutAll: 'offset := ';
		nextPutAll: anInteger printString;
		nextPut: $.; lf; tab;
		nextPutAll: '1 to: array size do: [:i | '; lf; tab; tab;
		nextPutAll: 'array at: i put: (self ';
		nextPutAll: self getSelector;
		nextPutAll: ' offset).'; lf; tab; tab;
		nextPutAll: 'offset := offset + ';
		nextPutAll: self baseByteSize printString;
		nextPut: $.; lf; tab;
		nextPutAll: '].'; lf; tab;
		nextPutAll: '^array.'; lf;
		contents.

]

{ #category : 'CByteArray get' }
CDeclaration >> arrayPointersAccessorForOffset: anInteger [

	^(AppendStream on: String new)
		nextPutAll: name; lf; tab;
		nextPutAll: '| array offset | "generated by CDeclaration>>arrayPointersAccessorForOffset:"'; 
        lf; tab;
		nextPutAll: 'array := Array new: ';
		nextPutAll: count first printString;
		nextPut: $.; lf; tab;
		nextPutAll: 'offset := ';
		nextPutAll: anInteger printString;
		nextPut: $.; lf; tab;
		nextPutAll: '1 to: array size do: [:i | '; lf; tab; tab;
		nextPutAll: 'array at: i put: (self pointerAt: offset resultClass: ';
		nextPutAll: header cByteArraySpecies name asString;
		nextPutAll: ').'; lf; tab; tab;
		nextPutAll: 'offset := offset + 8.'; lf; tab;
		nextPutAll: '].'; lf; tab;
		nextPutAll: '^array.'; lf;
		contents.

]

{ #category : 'CByteArray put' }
CDeclaration >> arrayPointersUpdatorForOffset: anInteger [

	^(AppendStream on: String new)
		nextPutAll: name;
		nextPutAll: ': anArrayOfCByteArrayInstances'; lf; tab;
		nextPutAll: '| offset | "generated by CDeclaration>>arrayPointersUpdatorForOffset:"'; 
      lf; tab;
		nextPutAll: 'offset := ';
		nextPutAll: anInteger printString;
		nextPut: $.; lf; tab;
		nextPutAll: '1 to: anArrayOfCByteArrayInstances size do: [:i | '; lf; tab; tab;
		nextPutAll: 'self'; lf; tab; tab; tab;
		nextPutAll: 'pointerAt: offset'; lf; tab; tab; tab; 
		nextPutAll: 'put: (anArrayOfCByteArrayInstances at: i).'; lf; tab; tab;
		nextPutAll: 'offset := offset + 8.'; lf; tab;
		nextPutAll: '].'; lf;
		contents.

]

{ #category : 'CByteArray put' }
CDeclaration >> arrayUpdatorForOffset: anInteger [

	^(AppendStream on: String new)
		nextPutAll: name;
		nextPutAll: ': aSequenceableCollection'; lf; tab;
		nextPutAll: '| offset | "generated by CDeclaration>>arrayUpdatorForOffset:"'; lf; tab;
		nextPutAll: 'offset := ';
		nextPutAll: anInteger printString;
		nextPut: $.; lf; tab;
		nextPutAll: '1 to: aSequenceableCollection size do: [:i | '; lf; tab; tab;
		nextPutAll: 'self '; lf; tab; tab; tab;
		nextPutAll: self getSelector; space;
		nextPutAll: 'offset'; lf; tab; tab; tab;
		nextPutAll: 'put: (aSequenceableCollection at: i).'; lf; tab; tab;
		nextPutAll: 'offset := offset + ';
		nextPutAll: self baseByteSize printString;
		nextPut: $.; lf; tab;
		nextPutAll: '].'; lf;
		contents.

]

{ #category : 'CCallout' }
CDeclaration >> asClassVarName [

	^'Function_' , name.

]

{ #category : 'other' }
CDeclaration >> atEnd [

	^header atEnd.

]

{ #category : 'size' }
CDeclaration >> baseByteSize [
  ^ self _baseByteSize at: 1 
]

{ #category : 'size' }
CDeclaration >> _baseByteSize [
  "Returns an Array { size . alignment } or
                    { size . alignment . fieldOffsets } "
	0 < pointer ifTrue: [^ #( 8 8 ) ].
	bitCount ifNotNil: [  | nBytes align |
    "Only  bit fields of base type  int   are supported."
	  (type == #'int32' or: [type == #'uint32']) ifTrue:[ nBytes := 4].
	  "(type == #'int8' or: [type == #'uint8']) ifTrue:[ nBytes := 1 ].
	   (type == #'int16' or: [type == #'uint16']) ifTrue:[ nBytes := 2].
	   (type == #'int64' or: [type == #'uint64']) ifTrue:[ nBytes := 8 ].
    "
    nBytes ifNil:[self error:'Invalid type for a bit field' ].
    bitCount > (align := nBytes*8) ifTrue:[self error:'bit field too big'].
    ^ { bitCount / 8 . nBytes }.
  ].
	(type == #'int8' or: [type == #'uint8']) ifTrue:[ ^ #( 1 1 )].
	(type == #'int16' or: [type == #'uint16']) ifTrue:[^ #( 2 2 )].
	(type == #'int32' or: [type == #'uint32']) ifTrue:[^ #( 4 4 )].
	(type == #'int64' or: [type == #'uint64']) ifTrue:[^ #( 8 8 )].
	(type == #void and: [0 < pointer]) ifTrue:[^ #( 8 8 )].
	type == #class ifTrue:[ ^self _baseStructByteSize].
	type == #struct ifTrue:[ ^self _baseStructByteSize].

	type == #union ifTrue:[ ^self _baseUnionByteSize].
	type == #'enum' ifTrue:[ ^ #( 4 4 )].
	type == #'function' ifTrue:[ ^ #( 8 8 )].
	type == #double ifTrue:[ ^ #( 8 8 )].
	(type isKindOf: CDeclaration) ifTrue:[ ^ type _baseByteSize].

  self error:'in _baseByteSize, unknown type ', type printString	.
  ^ nil .
]

{ #category : 'size' }
CDeclaration >> containsFloat  [ 
  type _isSymbol ifFalse:[ ^ type containsFloat ].
  (type == #double or:[ type == #float]) ifTrue:[ ^ true ].
  ^ false
]

{ #category : 'size' }
CDeclaration >> _baseStructByteSize [
  "Returns an Array { size . alignment . { fieldOffset ... fieldOffset} } 
   With alignment of fields and pads like a C compiler would.  "
	| total align rem offsets totalBits theFields bitsAlign |
  (storage == #typedef and:[ fields == nil and:[ self isStruct]]) ifTrue:[
    ^ type _baseStructByteSize .
  ].
  (type == #struct and:[ fields == nil  and:[ pointer == 0]]) ifTrue:[
     self error:'Incomplete struct definition (typedef with no fields defined)'.
  ].
	total := 0.
  align := 1 .
  offsets := { } .
  bitsAlign := 0 .
  totalBits := 0 .
  (theFields := fields copy ) add: nil .
	theFields do: [:each | | arr fieldSize fieldAlign pad fRem aSize aAlign |
     each ifNotNil:[
       arr := each _byteSize . 
       aSize := arr at: 1 .
       aAlign := arr at: 2 .
     ] ifNil:[
       aSize := 0 .
       aAlign := 1 .
     ].
     aSize < aAlign ifTrue:[  "a bitfield, or close out last bitfield"
       aSize > 0 ifTrue:[  "a bitfield" | nBits |
         each bitCount ifNil:[ self error:'Inconsistent bitfield'].
         nBits := aSize * 8  .
         totalBits == 0 ifTrue:[ "first bitfield after non-bitfield"
           bitsAlign := aAlign .
           totalBits := nBits.
           fieldSize := 0 .
         ] ifFalse:[    "accumulate another bitfield"
           (totalBits + nBits) <= (bitsAlign*8) ifTrue:[ "more bits in the word"
             totalBits := totalBits + nBits .
             fieldSize := 0 .
           ] ifFalse:[
             fieldSize := totalBits // 8 .  "word of bitfield is full"
             (totalBits \\ 8) ~~ 0 ifTrue:[ fieldSize := fieldSize + 1 ].
             fieldSize <= 0 ifTrue:[ self error:'inconsistent bitifield'].
             fieldAlign := bitsAlign .
             totalBits := nBits . "start accumulating next word of bits"
             bitsAlign := fieldAlign .
           ]
         ].
       ] ifFalse:[ "close out last bitfield if any"
         fieldSize := 0 .
         totalBits > 0 ifTrue:[
           fieldSize := totalBits // 8 . 
           (totalBits \\ 8) ~~ 0 ifTrue:[ fieldSize := fieldSize + 1 ].
           fieldAlign := bitsAlign .
         ].
       ]
     ] ifFalse:[
       fieldSize := aSize . "normal field"
       fieldAlign := aAlign .
     ] .
     fieldSize > 0 ifTrue:[ 
       pad := 0 .
       (fRem := total \\ fieldAlign) ~~ 0 ifTrue:[ 
          pad := fieldAlign - fRem . "pad in front of this element per its required alignment"
       ].
       total := total + pad .
       offsets add: total .
       total := total + fieldSize .
       fieldAlign > align ifTrue:[ 
         fieldAlign > 8 ifTrue:[ self error:'alignment exceeds 8'].
         align := fieldAlign .
       ].
     ].
	].
  (rem := total \\ align) ~~ 0 ifTrue:[ 
     total := total + (align - rem "pad at end, per largest align of any element").
  ].
	^ { total . align . offsets }
]

{ #category : 'size' }
CDeclaration >> _baseUnionByteSize [
  "Returns an Array { size . alignment } 
   With alignment of fields and end pad like a C compiler would.  "
  | total align |
  total := 0 . 
  align := 1 .
	fields do:[ : each | | arr |
    arr := each _baseByteSize .
    total := total max: (arr at: 1"size") .
    align := align max: (arr at: 2"alignment").
  ].
  align > 8 ifTrue:[ self error:'alignment exceeds 8'].
  ^ { total . align }
]

{ #category : 'other' }
CDeclaration >> beExternStorage [

	storage := #'extern'.

]

{ #category : 'accessors' }
CDeclaration >> bitCount [

	^bitCount.

]

{ #category : 'size' }
CDeclaration >> byteSizeForMalloc [
  "Return a size rounded up to 8 bytes, which may be a more strict
   rounding than a struct would have within another struct."
  | arr sz rem |
  arr := self _byteSize .
  sz := arr at: 1 .
  (rem := sz \\ 8) ~~ 0 ifTrue:[
     sz := sz + (8 - rem)
  ].
  ^ sz 
]

{ #category : 'size' }
CDeclaration >> _byteSize [
  "returns an Array { size . alignment } or
                    { size . alignment . fieldOffsets}   "
	| arr sz |
	arr := self _baseByteSize .
	count ifNil:[ ^ arr ].
  sz := arr at: 1 .
	count do: [:y | sz := sz * y].
  arr isInvariant ifTrue:[ arr := arr copy ].
	arr at: 1 put: sz .
  ^ arr .
]

{ #category : 'size' }
CDeclaration >> byteSize [
  ^ self _byteSize at: 1 .
]

{ #category : 'querying' }
CDeclaration >> canStorageBeMadeExternal [

	^(#(#auto #inline #static) includesIdentical: storage)
		or: [storage = #extern and: [linkageSpec == nil ]]

]

{ #category : 'CCallout' }
CDeclaration >> cCalloutForLibrary: aCLibrary [
  self usesStructs ifTrue:[ Error signal:'Not supported for a callout with structs'].
	^ CCallout
		library: aCLibrary 
		name: name 
		result: self resultType 
		args: (parameters collect: [:each | each argumentType])
		varArgsAfter: (isVaryingArgCount ifTrue: [parameters size] ifFalse: [-1]).

]

{ #category : 'Class Membership' }
CDeclaration >> cCalloutSpecies [
"Answer the class to be used for CCallout objects. Subclasses may
 overload this method as needed."
^ CCallout

]

{ #category : 'Class Membership' }
CDeclaration >> cDeclarationSpecies [
"Answer the class to be used for CDeclaration objects. Subclasses may
 overload this method as needed."
^ CDeclaration

]

{ #category : 'accessors' }
CDeclaration >> count [

	^count.

]

{ #category : 'Class Membership' }
CDeclaration >> cPreprocessorSpecies [
"Answer the class to be used for CPreprocessor objects. Subclasses may
 overload this method as needed."
^ CPreprocessor

]

{ #category : 'CCallout' }
CDeclaration >> createFunctionInitializerCode [
	| stream |
  self usesStructs ifTrue:[  ^ self _createFunctionInitializerCodeStructs ].
	stream := AppendStream on: String new.
	stream
		nextPut: self initializerFunctionName;
		nextPut: ' cLibrary'; lf; lf; tab;
		nextPut: self asClassVarName , ' := CCallout ' ; lf; tab; tab;
		nextPut: 'library: cLibrary'; lf; tab; tab;
		nextPut: 'name: ''' , name , $' ; lf; tab; tab;
		nextPut: 'result: #''' , self resultType , $' ; lf ; tab; tab;
		nextPut: 'args: #(' .
	parameters do: [:each | stream nextPut: ' ', each argumentType printString ].
	stream nextPut: ')' .
  isVaryingArgCount 
    ifFalse:[ stream nextPut: $. ; lf . ]
    ifTrue:[
       stream lf ; tab ; tab ;
         nextPut: 'varArgsAfter: ', self varArgsAfter asString , $. ; lf . 
    ].
  ^ stream contents.
]

{ #category : 'CCallout' }
CDeclaration >> _createFunctionInitializerCodeStructs [
  ^ self _functionSourcesStructs at: 2 .
]

{ #category : 'querying' }
CDeclaration >> derivedType [

	storage == #typedef ifFalse:[ ^self ].
	type == #class ifTrue: [^self].
	self isFunction ifTrue: [^self].
  type _isSymbol ifTrue:[ ^ self ].
  (type type == #struct and:[pointer == 0]) ifTrue:[ 
    "For typedef struct x_a { } x ;     return CDeclaration with name x "
    ^ self  
  ].
	^type
]

{ #category : 'accessors' }
CDeclaration >> enumList [

	^type == #'enum'
		ifTrue: [fields]
		ifFalse: [nil].

]

{ #category : 'accessors' }
CDeclaration >> enumTag [

	^enumTag.

]

{ #category : 'Errors' }
CDeclaration >> error: aString [

  Error signal: aString , ' near line ' , line asString , ' in file ' , file asString

]

{ #category : 'other' }
CDeclaration >> expectCloseParenthesisCount: anInteger [

	anInteger timesRepeat: [
		| token |
		(token := self next) isCloseParenthesisToken ifFalse: [token error: 'Expected '')'' '].
	].

]

{ #category : 'accessors' }
CDeclaration >> fields [

	^type == #struct
		ifTrue: [fields]
		ifFalse: [nil].

]

{ #category : 'accessors' }
CDeclaration >> file [

	^file.

]

{ #category : 'Private' }
CDeclaration >> _typeForSelector [
  "Returns an Array of the form  {  typeSymbol .  fieldSize }"
  type _isSymbol ifFalse:[ ^ type _typeForSelector ].
  type == #union ifTrue:[ ^ { type . self _byteSize at: 1 } ].
  ^ { self argumentType . self _byteSize at: 1 }
]

{ #category : 'CByteArray get' }
CDeclaration >> _getFieldSelector [
  "Returns a String if self is a scalar field of size <= 8, 
  otherwise returns an Array of the form  {  typeSymbol .  fieldSize }"
  | arr typ |
  arr := self _typeForSelector .
  typ := arr at: 1 .
	(#(
		#int8 #int16 #int32 #int64
		#uint8 #uint16 #uint32 #uint64
	) includesIdentical: typ) ifTrue:[ ^ typ , 'At:'].
	typ == #'enum' ifTrue: [ ^ #int32At: ].
	typ == #'function' ifTrue: [ ^ #uint64At: ].
  typ == #union ifTrue:[ | sz |
    sz := arr at: 2 .
    sz <= 8 ifTrue:[
      sz == 8 ifTrue:[ ^ #uint64At: ].
      ^ #uint32At: .
    ].
    ^ arr "a large union"
  ].
  (typ == #struct or:[ typ == #ptr ])
     ifFalse:[ self error:'in getSelector, unknown type ', type printString].

  ^ arr 
]

{ #category : 'Private' }
CDeclaration >> _arrayInfoForSelector [
  type isSymbol ifFalse:[ ^ type __arrayInfoForSelector ].
  ^ self __arrayInfoForSelector
]

{ #category : 'Private' }
CDeclaration >> __arrayInfoForSelector [
  | numElem siz baseSiz |
  (type isSymbol and:[ pointer == 0 and:[ count ~~ nil]]) ifTrue:[
    siz := self _byteSize . 
    baseSiz := self _baseByteSize .
    numElem := (siz at: 1) // (baseSiz at: 1) .
    ^ '/*Array*/ ', type asString,' x[', numElem asString,']  ' .
  ] .
  ^ nil 
]


{ #category : 'CByteArray get' }
CDeclaration >> getSelector [
  | sel |
  sel := self _getFieldSelector .
  sel _isOneByteString ifFalse:[ self error:'struct or large union not supported'].
  ^ sel
]

{ #category : 'accessors' }
CDeclaration >> header [

	^header.

]

{ #category : 'accessors' }
CDeclaration >> includesCode [

	^includesCode.

]

{ #category : 'type' }
CDeclaration >> initializeAsStructOrUnion [

	| token |
	storage := #auto .
	(token := self peek) isOpenCurlyBracketToken ifFalse: [^self].
	self next.
	fields ifNotNil: [self error: 'Fields already defined!'].
	fields := { }.
	[
		(token := self peek) isCloseCurlyBracketToken .
	] whileFalse: [
		| field |
		field := self class header: header.
		field storage == #auto ifFalse: [ token error: 'storage is not #auto'].
		[
			fields add: field.
			(token := self next) isCommaToken.
		] whileTrue: [
			field := field copy.
			field readDeclaration.
      field trapName .
		].
		token isSemicolonToken ifFalse: [ token error: 'expected '';'' '].
	].
	self next.

]

{ #category : 'CCallout' }
CDeclaration >> initializerFunctionName [
	^'initializeFunction_', name, '_inLibrary:'

]

{ #category : 'type' }
CDeclaration >> interpretTypeSpecifier [

	type class == Array ifFalse: [^self].
	(type size == 1 and:[
   #(#class #'enum' #'float' #'float128' #void) includesIdentical: type first]) 
     ifTrue: [type := type first. ^self].

	"https://en.cppreference.com/w/c/language/arithmetic_types#Complex_floating_types
	https://gcc.gnu.org/onlinedocs/gcc-4.8.0/gcc/Complex.html"
	(type includesIdentical: #'complex') ifTrue: [
		| string |
		type removeIdentical: #'complex'.
		self interpretTypeSpecifier.
		(type isKindOf: Symbol) ifFalse: [self error: 'Expected to be simplified to a single type'].
		string := String withAll: type.
		string at: 1 put: string first asUppercase.
		type := ('complex' , string) asSymbol.
		^self
	].

	(type includesIdentical: #'char') ifTrue: [
		type removeIdentical: #'char'.
		type = #(#'signed') ifTrue: [type := #'int8'. ^self].
		(type isEmpty or: [type = #(#'unsigned')]) ifTrue: [type := #'uint8'. ^self].
    self error:'for a char type , unknown type ', type printString	.
	].
	(type includesIdentical: #'bool') ifTrue: [
		type removeIdentical: #'bool'.
		type isEmpty ifTrue:[ type := #'uint8'. ^self ].
    self error:'for a bool type , unknown type ', type printString	.
	].
	(type includesIdentical: #double) ifTrue: [
		type removeIdentical: #double.
		type isEmpty ifTrue: [type := #double. ^self].
		type = #(#'long') ifTrue: [type := #'longDouble'. ^self].
    self error:'for a double type, unknown type ', type printString	.
	].
	(type isEmpty or: [#(#'int' #'unsigned' #'long' #'signed' #'short') 
                     anySatisfy: [:each | type includesIdentical: each]]) ifTrue: [
		| isUnsigned |
		(type includesIdentical: #'int') ifTrue: [type removeIdentical: #'int'].
		(isUnsigned := type includesIdentical: #'unsigned') ifTrue: [type removeIdentical: #'unsigned'].
		(type includesIdentical: #'signed') ifTrue: [type removeIdentical: #'signed'].
		type = #(#'long') ifTrue: [
			type := isUnsigned ifTrue: [#'uint64'] ifFalse: [#'int64'].
			^self.
		].
		type = #(#'long' #'long') ifTrue: [
			type := isUnsigned ifTrue: [#'uint64'] ifFalse: [#'int64'].
			^self.
		].
		type = #(#'short') ifTrue: [
			type := isUnsigned ifTrue: [#'uint16'] ifFalse: [#'int16'].
			^self.
		].
		type isEmpty ifTrue: [
			type := isUnsigned ifTrue: [#'uint32'] ifFalse: [#'int32'].
			^self.
		].
		self error:'unrecognized integer type: ', type printString .
	].
	(type size == 1 and:[
     #(#'uint64' #'int64' #'uint32' #'int32' #'uint16' #'int16' #'uint8' #'int8') 
         includesIdentical: type first]) ifTrue: [
		type := type first.
		^self.
	].
	(type size == 1 and: [type first isKindOf: CDeclaration]) ifTrue: [type := type first. ^self].
	self error:'unrecognized type: ', type printString .
]

{ #category : 'Private' }
CDeclaration >> usesStructs [
  self isFunction ifFalse:[ Error signal:'not a function' ].
  parameters do:[:x | x argumentType == #struct ifTrue:[ ^ true]].
  self resultType == #struct ifTrue:[ ^ true ] .
  ^ false
]

{ #category : 'CCallout' }
CDeclaration >> invokeFunctionCode [
	| args stream |
  self usesStructs ifTrue:[  ^ self _invokeFunctionCodeStructs ].
	args := { } .
  1 to: parameters size do: [:j | | each | 
    each := parameters at: j .
    args add:  (each _argName: j) .
  ].
	(stream := AppendStream on: String new)
		nextPutAll: name .
	args do: [:each | stream nextPut: '_: ', each, ' '].
  isVaryingArgCount ifTrue:[ 
   stream lf ; tab ; nextPut:'    varArgs: vaArray"pairs of type, arg" ' ].

	self addSourceTo: stream.
	stream
		tab; 
		nextPutAll: '"Interpreted as #';
		nextPutAll: self resultType;
		nextPutAll: ' from #(' .
	parameters do: [:each | stream space. each argumentType printOn: stream].
	stream 
		nextPutAll: ' )';
		nextPut: $"; lf; tab .
  isVaryingArgCount ifFalse:[
		stream nextPut: '^ ', self asClassVarName , ' callWith: { ' .
	  args do: [:each | stream nextPut: each , '. ' ].
    self _removeLastDot: stream collection .
    stream nextPut: ' } errno: nil ' ; lf .
	] ifTrue:[
    stream nextPut: ' | args | '; lf ; tab ;
      nextPut: ' args := { ' .
	  args do: [:each | stream nextPut: each , '. ' ].
    self _removeLastDot: stream collection .
    stream nextPut: ' }. ' ; lf ; tab;
      nextPut: ' vaArray ifNotNil:[ args addAll: vaArray ].'; lf; tab ;
      nextPut: ' ^ ', self asClassVarName , ' callWith: args errno: nil .'; lf .
  ].  
	^ stream contents .
]

{ #category : 'Private' }
CDeclaration >> _argName: idx [
  | n | 
  n :=  name ifNil:[ 'arg' , idx asString].
  n = 'self' ifTrue:[ n := 'selfArg' ].
  ^ n
]

{ #category : 'Private' }
CDeclaration >> _removeLastDot: aString [
  | idx |
  idx := aString indexOfLastByte: $. codePoint startingAt: aString size .
  idx ~~ 0 ifTrue:[ 
    aString removeFrom: idx to: idx  .
  ].
]

{ #category : 'Private' }
CDeclaration >> _cpuKind [
  | cpu |
  cpu := System gemVersionAt: #cpuArchitecture .
  cpu = 'x86-64' ifTrue:[ ^ #'x86-64' ].
  cpu = 'aarch64' ifTrue:[ ^ #arm64 ] . "Linux Arm"
  cpu = 'arm64' ifTrue:[ ^ #arm64 ] . "Darwin Arm"
  Error signal:'Unsupported CPU kind ', cpu asString .
]

{ #category : 'Private' }
CDeclaration >> _functionSourcesStructs [
  "returns an Array  {  ffiCallMethodSource  . calloutInitializerSource . resultStructSize }"
  | cpu strm resStructSize initSrc resTyp structSizes |
  cpu := self _cpuKind .
  (cpu == #'x86-64' or:[ cpu == #arm64]) ifFalse:[
     Error signal:'CCallout with structs not supported on CPU kind ', cpu asString.
  ].
  resStructSize := 0 .  
  structSizes := { nil } .
  (resTyp := self resultType) == #struct ifTrue:[
     resStructSize := type byteSize .
     resStructSize <= 0 ifTrue:[ self error:'inconsistent result struct size'].
     resStructSize > 8192"Capi_MAX_STRUCT_RESULT" ifTrue:[ 
       self error:'result struct size ', resStructSize asString,' exceeds max of 8192'.
     ].
     resStructSize := resStructSize max: 8 . "minimum C memory size of a result struct is 8"
     structSizes at: 1 put: resStructSize .
  ].
	(resTyp == #double or:[ resTyp == #float]) ifTrue:[
     self error:'structs not supported in combination with floats'.
  ].
	(strm := AppendStream on: String new)
		nextPut: name .  "start of selector for the method"
  resStructSize > 0 ifTrue:[
    strm nextPut: '_: result"struct',resStructSize asString, 'bytes" '.
  ].
	1 to: parameters size do: [:j| | each |  
    each := parameters at: j.
		strm nextPut: '_: ', (each _argName: j) .
    each isStruct ifTrue:[  | sz |
      sz := each byteSize .
      (sz <= 16 and:[ each containsFloat]) ifTrue:[
        self error:'float or double not supported within small struct argument'.
      ].
      strm nextPut:'"struct', sz asString, 'bytes"'  .
      structSizes add: (sz max: 8)  .
    ] ifFalse:[ 
      each containsFloat ifTrue:[ self error:'structs not supported in combination with floats'].
      structSizes add: nil.
    ].
	  strm nextPut:' ' .
	].                   
  isVaryingArgCount ifTrue:[ strm lf ; tab ; nextPut:' varArgs: vaArray "pairs of type, arg"'].
  strm lf .
  "end of selector"
	self addSourceTo: strm.  "add comments - the C function prototype"
	strm tab; 
		nextPut: '"Interpreted as #', self resultType, ' from #(' .
	parameters do: [:each | strm space. each argumentType printOn: strm ].
  isVaryingArgCount ifTrue:[ strm nextPut:'  ...varArgs '].
	strm nextPut: ' )" '; lf ; tab ;  "end of Interpretd as comment"
       nextPut: '| res args | '; lf ; tab .

  (initSrc := String new) 
    add: self initializerFunctionName , ' cLibrary' ; lf ;
    add: '  ', self asClassVarName, ' := CCalloutStructs library: cLibrary name: ''', name ,$'; lf;
    add: '  result: ' ;
    add:(resStructSize > 0 ifTrue:[ ' #struct ' ] ifFalse:[ self resultType printString]) ; lf .
  
  resStructSize > 0 ifTrue:[ strm nextPut:' res := result"resultStruct".' ; lf ].
  strm nextPut:'  args := { ' .         "begin building array of args"
  initSrc add:'  args: #( ' .

  parameters size > 0 ifTrue:[
	  1 to: parameters size do:[:j| | parm nam | 
      parm := parameters at: j .
      nam := parm _argName: j .
      strm nextPut: ' ', nam, ' .' .
      initSrc add: ' ', parm argumentType printString,'"',nam,'" ' .
	  ].
    self _removeLastDot: strm collection  .
  ].
	strm nextPut: ' } .'; lf;tab .  
  initSrc add: '   )' ; lf;tab .  "end args array"
	isVaryingArgCount ifTrue:[
    strm nextPut: 'vaArray ifNotNil:[ args addAll: vaArray ].'; lf; tab .
    initSrc add: ' varArgsAfter: ', parameters size asString ; lf; tab .
  ] ifFalse:[
    initSrc add: ' varArgsAfter: -1 ' ; lf ; tab.
  ].
  initSrc add:' structSizes: #( ' .
  structSizes do:[:x | initSrc add: x asString ,' ' ].
  initSrc add:' ). '; lf . 
  resStructSize == 0 ifTrue:[
    strm nextPut:'res := ' .  "nil is passed to structResult: keyword"
  ].
  strm nextPut: self asClassVarName, ' callWith: args structResult: res errno: nil.'; lf;tab;
    nextPut:' ^ res . ' ; lf .
  ^ { strm  contents . initSrc }
]

{ #category : 'CCallout' }
CDeclaration >> _invokeFunctionCodeStructs [
  ^ self _functionSourcesStructs at: 1 .
]

{ #category : 'accessors' }
CDeclaration >> isConstant [

	^isConstant.

]

{ #category : 'querying' }
CDeclaration >> isEmptyExternalLinkage [
	"Answer whether the receiver is the introduction of:
		extern ''C'' { ... }
	 Note that we also end up with this after reading the entire { ... } block."

	^self storage = #extern
		and: [linkageSpec ~~ nil 
		and: [self name == nil 
		and: [self type == nil ]]]

]

{ #category : 'querying' }
CDeclaration >> isEmptyVoid [

	^name == nil and: [
		storage == #auto and: [
		type == #void and: [
		count == nil and: [
		pointer == 0 and: [
		fields == nil and: [
		isConstant == false and: [
		self isFunction == false and: [
		bitCount == nil  ]]]]]]]].

]

{ #category : 'querying' }
CDeclaration >> isFunction [

	^parameters ~~ nil .

]

{ #category : 'CCallout' }
CDeclaration >> isSimple [

	^count == nil  and: [pointer = 0].

]

{ #category : 'accessors' }
CDeclaration >> isVaryingArgCount [

	^isVaryingArgCount.

]

{ #category : 'CCallout' }
CDeclaration >> knownArgumentTypes [

	^#( #int64 #uint64 #int32 #uint32 #int16 #uint16 #int8 #uint8 #bool #double #float #ptr #'char*' "#'&ptr'" #'const char*' ).

]

{ #category : 'CCallout' }
CDeclaration >> knownResultTypes [

	^#( #int64 #uint64 #int32 #uint32 #int16 #uint16 #int8 #uint8 #bool #double #float #ptr #'char*' #void ).

]

{ #category : 'accessors' }
CDeclaration >> line [

	^line.

]

{ #category : 'accessors' }
CDeclaration >> linkageSpec [

	^linkageSpec.

]

{ #category : 'declarator' }
CDeclaration >> mayHaveCode [

	storage == #'static' 	ifTrue: [^true].
	storage == #'inline' 	ifTrue: [^true].
	type == #class		ifTrue: [^true].
	((type isKindOf: CDeclaration) and: [type type == #class ]) ifTrue: [^true].
	^false

]

{ #category : 'accessors' }
CDeclaration >> members [

	^type == #union
		ifTrue: [fields]
		ifFalse: [nil].

]

{ #category : 'accessors' }
CDeclaration >> name [

	^name.

]

{ #category : 'other' }
CDeclaration >> next [
  | res |
	res := self peek ; _next .
  ^ res

]

{ #category : 'accessors' }
CDeclaration >> parameters [

	^parameters.

]

{ #category : 'other' }
CDeclaration >> peek [

	| next wasIdentifierInSystemNamespace |
	wasIdentifierInSystemNamespace := false.
	[
		| wasEmpty |
		next := header peek.
		next ifNil:[ ^ nil ].
		wasEmpty := next isEmptyToken.
		wasEmpty ifFalse: [
			| value |
			(wasIdentifierInSystemNamespace and: [next isOpenParenthesisToken]) ifTrue: [
				self skipFunction.
				^self peek.
			] ifFalse: [
				wasIdentifierInSystemNamespace := next isIdentifierToken and: [2 < (value := next value) size and: 
					[(value at: 1) == $_ and: [(value at: 2) == $~]]].
			].
		].
		wasEmpty or: [wasIdentifierInSystemNamespace].
	] whileTrue: [
		self _next.
	].
	^ next

]

{ #category : 'accessors' }
CDeclaration >> pointer [

	^pointer.

]

{ #category : 'other' }
CDeclaration >> printFunctionParametersOn: aStream [

	| comma |
	(name == nil  or: [name last ~~ $(]) ifTrue: [
		aStream nextPut: $(.
	].
	(parameters == nil  or: [parameters isEmpty]) ifTrue: [aStream nextPutAll: 'void)'. ^self].
	comma := ''.
	parameters do: [:each | 
		aStream nextPutAll: comma.
		each printOn: aStream.
		comma := ', '.
	].
	aStream nextPut: $).

]

{ #category : 'other' }
CDeclaration >> printOn: aStream [

	storage ifNotNil: [
		(storage == #auto or: [storage == #typedef and: [type == #class ]]) ifFalse: [aStream nextPutAll: storage; space].
		linkageSpec ifNotNil:[ aStream nextPut: $"; nextPutAll: linkageSpec; nextPut: $"; space].
	].
	isConstant ifTrue: [aStream nextPutAll: 'const '].
	((type isKindOf: CDeclaration) and: [type isFunction]) ifTrue: [
		type printOn: aStream from: self.
		^self.
	].
	type ~= #() ifTrue: [
		(type isKindOf: Symbol)
			ifTrue: [aStream nextPutAll: type]
			ifFalse: [type printOn: aStream]. 
		aStream space.
	].
	pointer timesRepeat: [aStream nextPut: $*].
	enumTag ifNotNil: [aStream nextPutAll: enumTag; space].
	(storage == #typedef and: [self isFunction and: [0 < pointer]]) ifTrue: [aStream nextPut: $(].
	name ifNotNil: [aStream nextPutAll: name].
	count ifNotNil: [
		count do: [:each | 
			aStream nextPut: $[.
			0 < each ifTrue: [each printOn: aStream].
			aStream nextPut: $].
		].
	].
	(storage == #typedef and: [self isFunction and: [0 < pointer]]) ifTrue: [aStream nextPut: $)].
	self isFunction ifTrue: [self printFunctionParametersOn: aStream].

]

{ #category : 'other' }
CDeclaration >> printOn: aStream from: aCDeclaration [

	| open close |
	(type isKindOf: Symbol) ifTrue: [
		aStream nextPutAll: type.
	] ifFalse: [
		aStream nextPutAll: '<???>'.
	].
	aStream space.
	0 == aCDeclaration pointer ifTrue: [
		open := ''.
		close := ''.
	] ifFalse: [
		open := '('. 
		close := ')'.
	].
	pointer timesRepeat: [aStream nextPut: $*].
	aStream nextPutAll: open.
	aCDeclaration pointer timesRepeat: [aStream nextPut: $*].
	(name ~~ nil  and: [storage ~~ #typedef ]) ifTrue: [aStream nextPutAll: name].
	aStream nextPutAll: close.
	self printFunctionParametersOn: aStream.

]

{ #category : 'accessors' }
CDeclaration >> properties [

	^type == #class
		ifTrue: [fields]
		ifFalse: [nil].

]

{ #category : 'attributes' }
CDeclaration >> readAttributes [
"
	http://tigcc.ticalc.org/doc/gnuexts.html
"
	| parenthesisCount token |
	(token := self next) isOpenParenthesisToken ifFalse: [token error: 'Expected ''('' '].
	(token := self next) isOpenParenthesisToken ifFalse: [token error: 'Expected ''('' '].
	parenthesisCount := 2.
	[
		0 < parenthesisCount.
	] whileTrue: [
		token := self next.
		token isOpenParenthesisToken ifTrue: [
			parenthesisCount := parenthesisCount + 1.
		] ifFalse: [
			token isCloseParenthesisToken ifTrue: [
				parenthesisCount := parenthesisCount - 1.
			] ifFalse: [
				(token isIdentifierToken and: [token value = '__mode__']) ifTrue: [^self readModeAttributes].
			].
		].
	].

]

{ #category : 'other' }
CDeclaration >> readBitFields [
	"We parse bit fields, but we don't handle them properly!"

	bitCount := self cPreprocessorSpecies evaluateHeader: header.
	(bitCount isKindOf: SmallInteger) ifFalse: [ self error: 'Expected a SmallInteger in readBitFields'].

]

{ #category : 'type' }
CDeclaration >> readComplexType [
"
>>--+--struct--+--+-----identifier---------------------------+--><
      |              |   |                        .----------------.     |
      '--union----'    |                       V                   |     |
                          '--+----------+--{----member--;--+--}-'
                             '-identifier-'
"
	self peek isIdentifierToken ifTrue: [
		| dictionary |
		name := self next value.
    self trapName .
		dictionary := type == #struct
			ifTrue: [header structs]
			ifFalse: [header unions].
		type := dictionary
			at: name
			ifAbsentPut:[ | x | x := self copy . x trapName . x].
	] ifFalse: [
		type := self copy.
    type trapName .
	].
	type fields ifNil: [type initializeAsStructOrUnion].
  self trapName .
	name := nil.

]

{ #category : 'declarator' }
CDeclaration >> readConstantExpression [

	| token x |
	[
		(x := self peek) ~~ nil and: [x isOpenSquareBracketToken].
	] whileTrue: [
		| value tok |
		self next.
		count ifNil: [count := { } ].
		value := (tok := self peek) isCloseSquareBracketToken 
			ifTrue: [0]
			ifFalse: [self cPreprocessorSpecies evaluateHeader: header].
		value _isSmallInteger  ifFalse: [ tok error: 'Expected a SmallInteger'].
		count add: value.  
		(token := self next) isCloseSquareBracketToken ifFalse:[ token error:'expected '']'' '].
	].

]

{ #category : 'declarator' }
CDeclaration >> readDeclaration [
	"Return a CDeclaration (might not be self)
	   .-,------------------------------.
	  V                                      |
	----declarator--+-------------+-+--;--><
						  '--initializer--'

	We handle only one declarator; the caller (in CHeader>>#'readOne') handles the comma.
"
	| declaration token1 token2 trace |
	trace ifNotNil:[ token1 := self peek].	"for debugging"
	(declaration := self readDeclarator) ifNil: [
		^self.
	].
	declaration includesCode ifTrue: [^declaration].
	trace ifNotNil:[ token2 := self peek].	"for debugging"
	^declaration 
		readInitializer;
		readPostlude;
		yourself

]

{ #category : 'declarator' }
CDeclaration >> readDeclarator [
	"Return a CDeclaration (might not be self)
	>>--+-------------------------+--directDeclarator--><
	      |  .---------------------.  |
	      |  V                        |  |
	      '----pointerOperator--+-'
"
	| declaration token1 token2 trace |
	trace ifNotNil:[ token1 := self peek].	"for debugging"
	(declaration := self readPointerOperator) ifNotNil: [
		trace ifNotNil:[ token2 := self peek ].	"for debugging"
		^declaration readDeclarator.
	].
	^self readDirectDeclarator.

]

{ #category : 'declarator' }
CDeclaration >> readDirectDeclarator [
"Return a CDeclaration (might not be self)
	>>--+------------------------------------------------------------------------------------------------------+-><
	    +--declaratorName----------------------------------------------------------------------------------------+
	      +--directDeclarator--(--parameterDeclarationList--)--+----------------+---+----------------------------+--+
	      |          '--cvQualifiers--'     '--exceptionSpecification--'    |
	      +--directDeclarator--[--+-----------------------+--]------------------------------------------------+
	      |     '-constantExpression-'                                                                           |
	      '--(--declarator--)----------------------------------------------------------------------------------'
        + ----  : ---  superclassName ----
"
	| token declarator superName |
	(token := self peek) ifNil: [^self].
	token isIdentifierToken ifTrue: [
    name := token value.
    self trapName .
		(token := self next; peek) ifNil: [^self].
	] ifFalse: [ 
		(((type isKindOf: CDeclaration) and: [type type == #class ]) == false  
          and: [token isOpenParenthesisToken]) ifTrue: [ 
			self next.	"skip the open parenthesis token"
			declarator := self cDeclarationSpecies header: header.
			(token := self next) isUpArrowToken ifTrue: [	"Block argument, not pointer to function"
				storage := #'block'. 
				(token := self next) isIdentifierToken ifTrue: [token := self next].
			].
			token isCloseParenthesisToken ifFalse: [self error: 'expected '')'''].
			(token := self peek) ifNil: [self error: 'next token is empty'].
		].
	].
  ((type == #class or:[ type == #struct]) and:[ token isColonToken ]) ifTrue:[ | nxt|
    self next .
    nxt := self peek . 
    nxt isPublicIdentifierToken ifTrue:[ self next ].   "TODO  accumulate nested fields to fix bug 51114 "
    superName := self next .
    token := self peek 
  ].
	token isOpenParenthesisToken ifTrue: [
		self readParameterDeclarationList.
		declarator ifNil: [^self].
		declarator 
			setStorage: storage
			type: self.
		storage := #auto .
		^declarator
	].
	token isOpenSquareBracketToken ifTrue: [
		declarator ifNotNil: [ token error:'unexpected declarator after ''['' ' ].
		self readConstantExpression.
		^self
	].
	declarator ifNil: [^self].
	"this is a specialized check for a function argument of the form ' struct dirent *(*[]) '."
	(declarator storage == #auto and: [
		declarator type == #'int32' and: [
		declarator name == nil  and: [
		declarator count = #(0)]]]) ifTrue: [
			pointer := pointer + declarator pointer + 1.
		].
	^self

]

{ #category : 'type' }
CDeclaration >> readEnumeration [
  "
  'enum' [ identifier ] '{' enumerator-list '}' [ variable-list ]
  or
  'enum' identifier variable-list 
  or
  'enum' identifier function-definition // #48859
  "

  | token integralConstant |
  token := self peek.
  token isColonToken ifTrue:[ self next . token := self peek ]. "seen on Darwin 23.2"
  token isIdentifierToken ifTrue: [
    enumTag := self next value.
    token := self peek.
	].
  token isOpenCurlyBracketToken ifTrue: [
    fields := Dictionary new.       "this allows us to tell if the enum included an enumerator-list"
    self next.
    integralConstant := -1.
    [
      (token := self next) isCloseCurlyBracketToken.
    ] whileFalse: [
      | identifier |
      token isIdentifierToken ifFalse: [token error: 'expected an identifier' ].
        identifier := token value.
        integralConstant := self peek isEqualsToken ifTrue: [ | tok |
          tok := self next.
          integralConstant := self cPreprocessorSpecies evaluateHeader: header.
          (integralConstant _isSmallInteger) ifFalse: [ tok error: 'Expected a SmallInteger'].
          integralConstant.
        ] ifFalse: [
          integralConstant + 1.
        ].
        (header enums includesKey: identifier) ifTrue: [ token error: 'Duplicate enum definition!'].
        header enums at: identifier put: integralConstant.
        fields at: identifier put: integralConstant.
        self peek isCommaToken ifTrue: [self next].
        ].
  	].
]

{ #category : 'declarator' }
CDeclaration >> readInitializer [
	"Since we don't have a real expression parser, we will go till we reach
	a comma, semicolon, or close parenthesis not in a nesting"

	| token parenthesisCount curlyBracketCount |
	(token := self peek) ifNil: [^self].
	token value = '__attribute__' ifTrue: [
		self next.
		self readAttributes.
		^self readInitializer
	].
	token isEqualsToken ifFalse: [^self].
	self next.
	parenthesisCount := 0.
	curlyBracketCount := 0.
	[true] whileTrue: [ | pkk |
		(pkk := self peek) ifNil: [ token error:'next token is empty' ].
    token := pkk .
		(parenthesisCount == 0 and: [curlyBracketCount == 0]) ifTrue: [
			token isCommaToken ifTrue: [^self].
			token isSemicolonToken ifTrue: [^self].
			token isCloseParenthesisToken ifTrue: [^self].
		].
		token isOpenParenthesisToken ifTrue: [parenthesisCount := parenthesisCount + 1] ifFalse: [
		token isCloseParenthesisToken ifTrue: [parenthesisCount := parenthesisCount - 1] ifFalse: [
		token isOpenCurlyBracketToken ifTrue: [curlyBracketCount := curlyBracketCount + 1] ifFalse: [
		token isCloseCurlyBracketToken ifTrue: [curlyBracketCount := curlyBracketCount - 1]]]].
		self next.
	].

]

{ #category : 'attributes' }
CDeclaration >> readModeAttributes [
"
	http://tigcc.ticalc.org/doc/gnuexts.html#SEC91_mode
	http://www.linuxtopia.org/online_books/programming_tool_guides/linux_using_gnu_compiler_collection/vector-extensions.html
		QI = 8-bit integer, HI = 16-bit integer, SI = 32-bit integer, DI = 64-bit integer, SF = 32-bit float, DF = 64-bit float, TC = 128-bit complex
		byte = 8-bit integer, word = 64-bit integer, pointer = 64-bit integer

	typedef int int8_t __attribute__ ( ( __mode__ ( __QI__ ) ) ) ;
"
	| token mode |
	(token := self peek) isOpenParenthesisToken ifFalse: [ token error: 'expected ''('' '].
	self next.
	(token := self next) key == #'identifier' ifFalse: [token error: 'Expected an identifier'].
	mode := token value.
	type := 
		mode = '__QI__' ifTrue: [type first == $u ifTrue: [#'uint8'] ifFalse: [#'int8']] ifFalse: [
		mode = '__HI__' ifTrue: [type first == $u ifTrue: [#'uint16'] ifFalse: [#'int16']] ifFalse: [
		mode = '__SI__' ifTrue: [type first == $u ifTrue: [#'uint32'] ifFalse: [#'int32']] ifFalse: [
		mode = '__DI__' ifTrue: [type first == $u ifTrue: [#'uint64'] ifFalse: [#'int64']] ifFalse: [
		mode = '__byte__' ifTrue: [type first == $u ifTrue: [#'uint8'] ifFalse: [#'int8']] ifFalse: [
		mode = '__word__' ifTrue: [type first == $u ifTrue: [#'uint64'] ifFalse: [#'int64']] ifFalse: [
		mode = '__pointer__' ifTrue: [type first == $u ifTrue: [#'uint64'] ifFalse: [#'int64']] 
      ifFalse: [
		mode = '__TC__' ifTrue: [#'complexDouble'] ifFalse: [
		self error: 'unrecognized __mode__: ' , mode printString .
  ]]]]]]]].
	self expectCloseParenthesisCount: 3.

]

{ #category : 'declarator' }
CDeclaration >> readParameterDeclarationList [
  | token |
  self next.  "open parenthesis token"
  parameters := { } .
  self trapName .
  [
    (token := self peek ) isCloseParenthesisToken 
  ] whileFalse: [
    self peek isDotToken ifTrue: [
      self next.
      2 timesRepeat: [(token := self next) isDotToken ifFalse: [ token error:'expected ''.'' ']].
      isVaryingArgCount := true.
      (token := self peek) isCloseParenthesisToken ifFalse: [ token error:'expected '')'' '].
    ] ifFalse: [
      parameters add: (self cDeclarationSpecies header: header).
      (token := self peek) isCommaToken ifTrue: [self next].
    ].
  ].
  self next.  "close parenthesis token"
  (parameters size = 1 and: [parameters first isEmptyVoid]) ifTrue: [
    parameters := #().
  ].
  self mayHaveCode ifFalse:[ ^self].
  [
    (token := self peek) ifNil: [^self].
    token isSemicolonToken ifTrue: [^self].
    token isOpenCurlyBracketToken
  ] whileFalse: [
    self next.
  ].
  self next.
  self skipToEndOfCodeBlock.

]

{ #category : 'declarator' }
CDeclaration >> readPointerOperator [
	"Return a CDeclaration (might not be self)
	>>--*--+----------------+--><
	          '-typeQualifiers-' 
"
	| token |
	(token := self peek) ifNil: [^nil].
	token isStarToken ifFalse: [^nil].
	pointer := pointer + 1.
	(token := self next; peek) ifNil: [^self].
	token isIdentifierToken ifFalse: [^self].
	(#('const' 'volatile') includes: token value) ifFalse: [^self].
	self next.
	"we don't keep track of const pointers, only the data"

]

{ #category : 'declarator' }
CDeclaration >> readPostlude [

	| token1 token2 |
	token1 := self peek.
  token1 ifNil: [^self].
	token1 isIdentifierToken ifTrue: [
		(2 < token1 value size and: [(token1 value at: 1) == $_ and: [(token1 value at: 2) == $_]]) ifTrue: [
			(token2 := self next; peek) ifNil: [self next. ^self].	"Ignore trailing tokens beginning with double-underscore"
			token2 isOpenParenthesisToken ifTrue: [
				token1 value = '__attribute__' ifTrue: [
					self readAttributes.
				] ifFalse: [
					self skipFunction.
				].
			].
			^self readPostlude.
		].
		self error: 'unexpected token: ', token1 printString.
	].
	token1 isColonToken ifTrue: [
		self next. 
		self readBitFields. 
		^self readPostlude.
	].
	token1 isOpenCurlyBracketToken ifTrue: [
		((type isKindOf: CDeclaration) and: [type type == #class ]) ifTrue: [
			self skipToEndOfCodeBlock.
			^self.
		].
		self isFunction ifTrue: [		"Functions might have code (bugs #43426 & 43782)"
			self next.
			self skipToEndOfCodeBlock.
			^self
		].
	].
	token1 isEqualsToken ifTrue: [	"Check for an initialization"
		[
			(token2 := self next; peek) ~~ nil  and: [token2 isSemicolonToken].
		] whileFalse.
		^self
	].

]

{ #category : 'type' }
CDeclaration >> readStorageClassSpecifier [
	"Return a CDeclaration (might not be self)
	( 'auto' | 'extern' | 'register' | 'static' | 'typedef' | 'inline' )
"
	| token value |
	(token := self peek) ifNil: [^nil].
	token isTildeToken ifTrue: [token := self next; peek].	"indicates a class destructor function, which we can ignore"
	[token isExtensionIdentifierToken] whileTrue: [token := self next; peek].	"safe to ignore"
	token  isIdentifierToken ifFalse: [^nil].
	token value = 'inline' ifTrue: [
		self next.
		storage := #'inline'.
		^self.
	].
	(#('auto' 'extern' 'register' 'static' 'typedef') includes: token value) ifTrue: [
		(storage ~~ nil  and: [storage ~= token value asSymbol])
			ifTrue: [ token error: 'We support only one storage specifier'].
		storage := self next value asSymbol.
		(token := self peek) ifNil: [^nil].
		value := token value.
		(storage = #'extern' and: [token type = #'string']) ifTrue: [
			linkageSpec := value.
			(token := self next; peek) isOpenCurlyBracketToken
				ifTrue: [type := nil].
		].
		^self.
	].
	^nil.

]

{ #category : 'type' }
CDeclaration >> readType [
	"Return a CDeclaration (might not be self)
"
	| declaration |
	declaration := self readTypeA.
	storage ifNil: [storage := #auto ].
	self interpretTypeSpecifier.
	^self.

]

{ #category : 'type' }
CDeclaration >> readTypeA [
	"Return a CDeclaration (might not be self)
	        .-----------------------------.  
	        V                                 |
	>>---+----------------------------+--><
           +--storageClassSpecifier--+			( 'auto' | 'extern' | 'register' | 'static' | 'typedef' | 'inline' )
           +--typeSpecifier-----------+			( 'void' | 'char' | 'short' | 'int' | 'long' | 'float' | 'double' | ... )
           '--typeQualifier-------------'				( 'const' | 'volatile' )
"
	| declaration token |
	(token := self peek) isAttributeIdentifierToken ifTrue: [self next; readAttributes. ^self readTypeA].
	(declaration := self readStorageClassSpecifier) ifNotNil: [^declaration readTypeA].
	(declaration := self readTypeSpecifier) ifNotNil: [^declaration readTypeA].
	(declaration := self readTypeQualifier) ifNotNil: [^declaration readTypeA].
	^self.

]

{ #category : 'type' }
CDeclaration >> readTypeFrom: aHeader [
	"Return a CDeclaration (might not be self)"

	header := aHeader.
	[self peek isEmptyToken] whileTrue: [self next].
	type := { } .
	pointer := 0.
	isConstant := false.
	includesCode := false.
	isVaryingArgCount := false.
	^self readType.

]

{ #category : 'type' }
CDeclaration >> readTypeQualifier [
 	"Return a CDeclaration (might not be self)
	( 'const' | 'volatile' )
"
	| token value |
	(token := self peek) 	ifNil: [^nil].
	token isIdentifierToken 	ifFalse: [^nil].
	value := token value.
	value = 'const' 	ifTrue: [self next. isConstant := true. ^self].
	value = 'volatile' 	ifTrue: [self next. ^self].
	^nil.

]

{ #category : 'type' }
CDeclaration >> readTypeSpecifier [
 	"Return a CDeclaration (might not be self)
		( 'void' | 'char' | 'bool' | 'short' | 'int' | 'long' | 'float' | 'double' | 'signed' | 'unsigned' 
       | '_Complex' | struct_or_union_spec | enum_spec | typedef_name )
	"
	| token value declaration |
	(token := self peek) ifNil: [^nil].
	token isIdentifierToken ifFalse: [^nil].
	value := token value.
	( #('void' 'char' 'bool' 'short' 'int' 'long' 'float' 'double' 'signed' 'unsigned') 
        includes: value) ifTrue: [
		type add: value asSymbol.
		self next.
		^ self.
	].
	value = '__float128'	ifTrue: [type := #'float128'.		self next. ^self].
	value = '_Complex'	ifTrue: [type add: #'complex'.	self next. ^self].		"this means that there are two of something else (e.g., double)"
	value = 'ptr'			ifTrue: [type := #ptr.				self next. ^self].
	value = 'enum'			ifTrue: [type := #'enum'.			self next. self readEnumeration. 	^self].
	value = 'struct'		ifTrue: [type := #struct .			self next. self readComplexType. ^self].
	value = 'union' 		ifTrue: [type := #union .			self next. self readComplexType. ^self].
	value = 'class' 		ifTrue: [type := #class .			self next. storage := #typedef .	^self].
	type = #() ifTrue: [
		declaration :=
			header types at: value ifAbsent: [
			header structs at: value ifAbsent: [
			header unions at: value ifAbsent: [
			header enums at: value ifAbsent: [nil]]]].
		declaration ifNotNil:[ 
      type := declaration derivedType. 
      type == declaration ifFalse:[ pointer := declaration pointer ].  "fix 48748"
      self next. 
      ^self
    ].
	].
	"These are a bit troubling. Sometimes the are expected to be built-in, and other times they are typedefs!"
	(value = '__builtin_va_list' 	and: [type = #()])	ifTrue: [type := #ptr . 		self next. ^self].
	(value = '__gnuc_va_list' 	and: [type = #()])	ifTrue: [type := #ptr . 		self next. ^self].
	^nil.

]

{ #category : 'CCallout' }
CDeclaration >> argumentType [
	(self isSimple and: [self knownArgumentTypes includesIdentical: type]) ifTrue: [^type].
	type == #'enum' ifTrue: [^#'int32'].
	(type == #'uint8' and: [isConstant]) ifTrue: [
		(count size == 1 and: [pointer == 0]) ifTrue: [^#'const char*'].		"const char foo[]"
		(count == nil and: [pointer == 1]) ifTrue: [^#'const char*'].			"const char * foo"
	].
  self isStruct ifTrue:[ ^ #struct ].
  (pointer > 0 or:[ count ~~ nil]) ifTrue:[ ^ #ptr ].
  type _isSymbol ifFalse:[ ^ type argumentType ] . "a typedef or similar"
	^ #ptr .
]

{ #category : 'CCallout' }
CDeclaration >> resultType [
	self isFunction ifFalse: [ self error: 'Not a function!'].
  ^ self _resultType .
]

{ #category : 'Private' }
CDeclaration >> _resultType [
	(self isSimple and: [self knownResultTypes includesIdentical: type]) ifTrue: [^type].
	type == #'enum' ifTrue: [^#'int32'].
	type == #'uint8' ifTrue: [
		(count size == 1 and: [pointer == 0]) ifTrue: [^#'char*'].		"const char foo[]"
		(count == nil and: [pointer == 1]) ifTrue: [^#'char*'].			"const char * foo"
	].
  self isStruct ifTrue:[ ^ #struct ].
  (pointer > 0 or:[ count ~~ nil]) ifTrue:[ ^ #ptr ].
  type _isSymbol ifFalse:[ ^ type _resultType ] . "a typedef or similar"
  ^ #ptr .
]

{ #category : 'other' }
CDeclaration >> isStruct [
  (type == #class or:[ type == #struct]) ifTrue:[ 
    pointer == 0 ifFalse:[ self error:'inconsistent pointer value'].
    ^ true 
  ].
  pointer > 0 ifTrue:[ ^ false ].
  (type isKindOf: CDeclaration) ifTrue:[ ^ type isStruct ].
  ^ false .
]

{ #category : 'other' }
CDeclaration >> setClassProperties: anArray [

	fields := anArray.

]

{ #category : 'Updating' }
CDeclaration >> setStorage: aSymbol type: aCDeclaration [

	storage := aSymbol.
	type := aCDeclaration.

]

{ #category : 'CByteArray get' }
CDeclaration >> simpleAccessorForOffset: anInteger [
  | str x ofsStr |
  ofsStr := anInteger asString .
  (str := String new)
		add: name; lf; tab;
    add: '   "generated by CDeclaration>>simpleAccessorForOffset: ', ofsStr , ' "';
      lf;tab.
  x  := self _getFieldSelector .
  x _isOneByteString ifTrue:[ 
		str add: '^ self ', x ,' ', ofsStr , ' . '; lf .
  ] ifFalse:[ | typ siz |
    "A field that is a struct, or a large union"
    typ := x at: 1 .  
    typ == #ptr ifTrue:[ 
       self _arrayInfoForSelector ifNotNil:[:ax | typ := ax ].
    ].
    siz := x at: 2 .
    str add: '" type  ', typ printString ,' ,  size ', siz asString, ' "'; lf; tab ;
       add: '^ CByteArray fromRegionOf: self offset: ', ofsStr ,
          ' numBytes: ', siz asString ,' . ' ; lf . 
  ].
  ^ str 
]

{ #category : 'CByteArray put' }
CDeclaration >> simpleUpdatorForOffset: anInteger [
 | str x ofsStr |
 ofsStr := anInteger asString .
  (str := String new)
    add: name , ': anObject'; lf; tab;
    add: '   "generated by CDeclaration>>simpleUpdatorForOffset: ', ofsStr , ' "'; 
      lf;tab.
  x  := self _getFieldSelector .
  x _isOneByteString ifTrue:[  
		str add:' self ',  x, ' ', ofsStr , ' put: anObject .'; lf  .
  ] ifFalse:[ | typ siz |
    "A field that is a struct, or a large union"
    typ := x at: 1 .
    siz := x at: 2 .
    str add: '" type  ', typ printString ,' size ', siz asString, ' "'; lf; tab ;
     add:' self copyBytesFrom: anObject"aCByteArray" from: 1 to: ', siz asString,
        '  into: ', ofsStr , ' allowCodePointZero: true .' ; lf .
  ].
  ^ str
]

{ #category : 'CByteArray get' }
CDeclaration >> singlePointerAccessorForOffset: anInteger [

	^(AppendStream on: String new)
		nextPutAll: name; lf; tab;
    nextPutAll: '   "generated by CDeclaration>>singlePointerAccessorForOffset:"'; lf;tab; 
		nextPutAll: '^self'; lf; tab; tab;
		nextPutAll: 'pointerAt: ';
		nextPutAll: anInteger printString; lf; tab; tab;
		nextPutAll: 'resultClass: ';
		nextPutAll: header cByteArraySpecies name asString;
		nextPut: $. ; lf;
		contents.

]

{ #category : 'CByteArray put' }
CDeclaration >> singlePointerUpdatorForOffset: anInteger [
  
	^(AppendStream on: String new)
		nextPut: name; 
		nextPut: ': aCByteArray'; lf; tab;
    nextPut: '   "generated by CDeclaration>>singlePointerUpdatorForOffset:"'; lf;tab; 
		nextPut: 'self'; lf; tab; tab;
		nextPut: 'pointerAt: ';
		nextPut: anInteger printString; lf; tab; tab;
		nextPut: 'put: aCByteArray.'; lf; tab; tab ;
		nextPut: 'self derivedFrom: aCByteArray.'; lf;
		contents.

]

{ #category : 'other' }
CDeclaration >> skipFunction [

	| parenthesisCount |
	parenthesisCount := 1.
	self next.
	[
		0 < parenthesisCount.
	] whileTrue: [
		| token |
		token := self next.
		token isCloseParenthesisToken ifTrue: [parenthesisCount := parenthesisCount - 1] ifFalse: [
		(token isIdentifierToken and: [token value last == $(]) ifTrue: [parenthesisCount := parenthesisCount + 1] ifFalse: [
		token isOpenParenthesisToken ifTrue: [parenthesisCount := parenthesisCount + 1]]].
	].

]

{ #category : 'declarator' }
CDeclaration >> skipToEndOfCodeBlock [

	| depth |
	includesCode := true.
	depth := 1.
	[
		0 < depth.
	] whileTrue: [
		| token |
		token := self next.
		token isOpenCurlyBracketToken ifTrue: [depth := depth + 1].
		token isCloseCurlyBracketToken ifTrue: [depth := depth - 1].
	].

]

{ #category : 'accessors' }
CDeclaration >> source [
	^ source

]

{ #category : 'Updating' }
CDeclaration >> source: aString [

	source ifNotNil: [^self].
	source := aString.
	(type isKindOf: CDeclaration) ifTrue: [type source: aString].
	fields ifNotNil: [
		type ~~ #'enum' ifTrue: [
			fields do: [:each | 
				each source ifNil: [
					each source: aString.
				].
			].
		].
	].
	parameters ifNotNil: [
		parameters do: [:each | 
			each source ifNil: [
				each source: aString.
			].
		].
	].

]

{ #category : 'CByteArray get' }
CDeclaration >> sourceStringsForAccessors [

	| list |
	list := { } .
  "cannot simply sum the sizes of the fields due to padding"
  (self _baseByteSize atOrNil: 3) ifNotNil:[ :offsets | 
    1 to: fields size do:[:j | | each |
      each := fields at: j .
		  list add: (each accessorForOffset: (offsets at: j)) .
	  ].
  ].
	^list.

]

{ #category : 'CByteArray put' }
CDeclaration >> sourceStringsForUpdators [
	| list |
	list := { } .
  "cannot simply sum the sizes of the fields due to padding"
  (self _baseByteSize atOrNil: 3) ifNotNil:[ :offsets | 
    1 to: fields size do:[:j | | each |
      each := fields at: j .
		  list add: (each updatorForOffset: (offsets at: j)) .
    ].
	].
	^list.

]

{ #category : 'accessors' }
CDeclaration >> storage [

	^storage.

]

{ #category : 'Updating' }
CDeclaration >> storage: aSymbol linkageSpec: optionalString [

	storage := aSymbol.
	linkageSpec := optionalString.

]

{ #category : 'other' }
CDeclaration >> stream [

	^header stream.

]

{ #category : 'CByteArray get' }
CDeclaration >> stringAccessorForOffset: anInteger [

	^(AppendStream on: String new)
		nextPutAll: name; lf; tab;
    nextPutAll: '   "generated by CDeclaration>>stringAccessorForOffset:"'; lf;tab; 
		nextPutAll: '^self _stringFromBytes: (self '; lf; tab; tab;
		nextPutAll: 'byteArrayFrom: ';
		nextPutAll: anInteger printString; lf; tab; tab;
		nextPutAll: 'to: ';
		nextPutAll: (anInteger + count first - 1) printString;
		nextPutAll: ').'; lf;
		contents.

]

{ #category : 'CByteArray get' }
CDeclaration >> stringPointerAccessorForOffset: anInteger [

	1 == count ifTrue: [
		^(AppendStream on: String new)
			nextPutAll: name; lf; tab;
      nextPutAll: '   "generated by CDeclaration>>stringPointerAccessorForOffset:"'; lf;tab; 
			nextPutAll: '^self stringFromCharStarAt: ';
			nextPutAll: anInteger printString; 
			nextPut: $.; lf; tab;
			nextPutAll: '"^self'; lf; tab; tab;
			nextPutAll: 'pointerAt: ';
			nextPutAll: anInteger printString; lf; tab; tab;
			nextPutAll: 'resultClass: ';
			nextPutAll: header cByteArraySpecies name asString;
			nextPutAll: '."'; lf;
			contents.
	] ifFalse: [
		^(AppendStream on: String new)
			nextPutAll: name; lf; tab;
			nextPutAll: '| array offset | "generated by CDeclaration>>stringPointerAccessorForOffset:"'; lf;tab; 
			nextPutAll: 'array := Array new: ';
			nextPutAll: count first printString;
			nextPut: $.; lf; tab;
			nextPutAll: 'offset := ';
			nextPutAll: anInteger printString;
			nextPut: $.; lf; tab;
			nextPutAll: '1 to: array size do: [:i | '; lf; tab; tab;
			nextPutAll: 'array at: i put: (self stringFromCharStarAt: offset).'; lf; tab; tab;
			nextPutAll: '"array at: i put: (self pointerAt: offset resultClass: ';
			nextPutAll: header cByteArraySpecies name asString;
			nextPutAll: ')."'; lf; tab; tab;
			nextPutAll: 'offset := offset + 8.'; lf; tab;
			nextPutAll: '].'; lf; tab;
			nextPutAll: '^array.'; lf;
			contents.
	].

]

{ #category : 'CByteArray put' }
CDeclaration >> stringPointerUpdatorForOffset: anInteger [

	1 == count ifTrue: [
		^(AppendStream on: String new)
			nextPut: name;
			nextPut: ': aCByteArray'; lf; tab;
      nextPutAll: '   "generated by CDeclaration>>stringPointerUpdatorForOffset:"'; lf;tab; 
			nextPut: '"self'; lf; tab; tab;
			nextPut: 'pointerAt: ';
			nextPut: anInteger printString; lf; tab; tab;
			nextPut: 'put: aCByteArray."'; lf; tab ; tab ;
		  nextPut: 'self derivedFrom: aCByteArray.'; lf;
			contents.
	] ifFalse: [
		^self arrayPointersUpdatorForOffset: anInteger.
	].

]

{ #category : 'CByteArray put' }
CDeclaration >> stringUpdatorForOffset: anInteger [
  | str |
  str := String new .
	str add: name, ': aCByteArray '; lf; tab;
    add: '   "generated by CDeclaration>>stringUpdatorForOffset:"'; lf; tab; 
		add: 'self copyBytesFrom: aCByteArray from: 1'; lf; tab;  
    add:'   to: (aCByteArray size min: ', count first asString,')'; lf; tab;
    add: '  into: ', anInteger asString, ' allowCodePointZero: true '; lf . 
  ^ str
]

{ #category : 'CByteArray put' }
CDeclaration >> structUpdatorForOffset: anInteger [
  | str |
  str := String new .
  str add: name, ': aCByteArray'; lf ; tab ;
    add: '   "generated by CDeclaration>>structUpdatorForOffset:"'; lf;tab; 
		add: 'self copyBytesFrom: aCByteArray from: 1 '; lf ; tab;
    add:'   to: (aCByteArray size min: ', self baseByteSize asString,')' ; lf; tab;
    add: '  into: ', anInteger asString, ' allowCodePointZero: true '; lf . 
  ^ str
]

{ #category : 'CByteArray get' }
CDeclaration >> structAccessorForOffset: anInteger [
	^(AppendStream on: String new)
		nextPutAll: name; lf; tab;
    nextPutAll: '   "generated by CDeclaration>>structAccessorForOffset:"'; lf;tab; 
		nextPutAll: '^self'; lf; tab; tab;
		nextPutAll: 'newFrom: ';
		nextPutAll: anInteger printString; lf; tab; tab;
		nextPutAll: 'numBytes: ';
		nextPutAll: self baseByteSize printString;
		nextPut: $.; lf;
		contents.

]

{ #category : 'Debugging' }
CDeclaration >> trapFile [
  "uncomment this and senders for debugging
    file ifNotNil:[:f |
      (f includesString:'t.hf') ifTrue:[ self  pause ].
    ].
  "

]

{ #category : 'Debugging' }
CDeclaration >> trapLine [
  "uncomment and edit here 
   and uncomment  refs to #TraceCpp in CHeader and CPreprocessorToken to enable tracing .
   Tracing works in conjunction with   CHeader >> addToSource: ,  CPreprocessorToken >> addToSource:
   TraceCpp values:  2 = pause at each addToSource:,  1 = just gciLogServer: of each addToSource:
   "
" file ifNotNil:[:f | | aLine |
    aLine := line .
    (f includesString:'gcits.hf') ifTrue:[ 
       line = 31 ifTrue:[ 
         SessionTemps current at: #TraceCpp put: 2 .
         self  pause .
       ].
    ]. 
  ]
 "
]

{ #category : 'Debugging' }
CDeclaration >> trapName [
  "uncomment this and senders for debugging  , example:
  name ifNotNil:[  
    (name at: 1 equals: 'GciErr') ifTrue:[ 
      GsFile gciLogServer: 'oop ', self asOop asString, ' name ' , name asString .
      self  pause 
    ]
  ].
 "
]

{ #category : 'accessors' }
CDeclaration >> type [
	"For many types, the value is a Symbol (e.g., #'uint64' or #struct).
	For other types, the value is a CDeclaration for a function or a class"

	^type.

]

{ #category : 'CByteArray get' }
CDeclaration >> accessorForOffset: anInteger [
  | typ |
	1 < count size ifTrue: [ self error: 'Multi-dimensional arrays not supported']. 
	0 == pointer ifTrue: [
    (typ := type) _isSymbol ifFalse:[ typ := type argumentType "handle typedef"].
		(typ == #class or:[ typ == #struct ]) ifTrue:[ ^ self structAccessorForOffset: anInteger ].
		count == nil ifTrue: [^self simpleAccessorForOffset: anInteger].
    1 < count size ifTrue:[ self error: 'Multi-dimensional arrays not supported!'].
		type == #'uint8' ifTrue: [^self stringAccessorForOffset: anInteger].
		^self arrayAccessorForOffset: anInteger.
	].
	1 < pointer ifTrue: [self error: 'Multi-level pointers not supported'].
	count == nil ifTrue: [^self singlePointerAccessorForOffset: anInteger].
	type == #'uint8' ifTrue: [^self stringPointerAccessorForOffset: anInteger].
	^self arrayPointersAccessorForOffset: anInteger.
]

{ #category : 'CByteArray put' }
CDeclaration >> updatorForOffset: anInteger [

	0 == pointer ifTrue: [ | typ |
    (typ := type) _isSymbol ifFalse:[ typ := type argumentType ].
		(typ == #class or:[ typ == #struct]) ifTrue:[ ^ self structUpdatorForOffset: anInteger].
		count == nil ifTrue:[ ^ self simpleUpdatorForOffset: anInteger].
		1 < count size ifTrue:[ self error: 'Multi-dimensional arrays not supported!'].
		type == #'uint8' ifTrue:[ ^self stringUpdatorForOffset: anInteger].
		^self arrayUpdatorForOffset: anInteger.
	].
	1 < pointer ifTrue: [self error: 'Multi-level pointers not supported!'].
	count == nil ifTrue: [^self singlePointerUpdatorForOffset: anInteger].
	type == #'uint8' ifTrue: [^self stringPointerUpdatorForOffset: anInteger].
	^self arrayPointersUpdatorForOffset: anInteger.
]

{ #category : 'CCallout' }
CDeclaration >> varArgsAfter [
	^ isVaryingArgCount ifTrue: [parameters size] ifFalse: [-1].
]

