Extension { #name : 'CanonicalObjectManager' }

{ #category : 'Instance Creation' }
CanonicalObjectManager class >> clearDefault [
"
	CanonicalObjectManager clearDefault.
"
	default := nil.

]

{ #category : 'Instance Creation' }
CanonicalObjectManager class >> default [

	default ifNil: [
		default := Array with: self new.
		default first objectSecurityPolicy: securityPolicy.
	].
	^default first.

]

{ #category : 'Instance Creation' }
CanonicalObjectManager class >> new [

	^super new
		initialize;
		yourself.

]

{ #category : 'Security' }
CanonicalObjectManager class >> securityPolicy [

	^securityPolicy

]

{ #category : 'Security' }
CanonicalObjectManager class >> securityPolicy: anObjectSecurityPolicy [

	securityPolicy := anObjectSecurityPolicy.
	default ifNotNil: [self default objectSecurityPolicy: anObjectSecurityPolicy].

]

{ #category : 'Accessors' }
CanonicalObjectManager >> _registries [

	^registries.

]

{ #category : 'Canonical' }
CanonicalObjectManager >> _registryFor: anObject ifAbsent: aBlock [

	^registries
		at: anObject class classHistory
		ifAbsent: aBlock.

]

{ #category : 'Cleanup' }
CanonicalObjectManager >> _writeSet [

	| writeSet queue |
	queue := OrderedCollection withAll: System _writtenObjects.
	writeSet := IdentitySet new.
	[
		queue notEmpty.
	] whileTrue: [
		| next |
		next := queue removeLast.
		next class isBytes ifFalse: [
			writeSet add: next.
			1 to: next _primitiveSize do: [:i |
				| x |
				x := next _primitiveAtNoFault: i otherwise: nil.
				(x isSpecial or: [x class isBytes or: [x isCommitted or: [writeSet includes: x]]]) not ifTrue: [
					queue add: x.
				].
			].
		].
	].
	^writeSet.

]

{ #category : 'Cleanup' }
CanonicalObjectManager >> _writeSetAdding: newObjects ignoring: oldObjects [

	| writeSet queue |
	queue := OrderedCollection withAll: System _writtenObjects.
	writeSet := IdentitySet new.
	[
		queue notEmpty.
	] whileTrue: [
		| next |
		next := queue removeLast.
		next class isBytes ifFalse: [
			writeSet add: next.
			1 to: next _primitiveSize do: [:i |
				| x |
				x := next _primitiveAt: i.
				(x isSpecial or: [x class isBytes or: [x isCommitted or: [writeSet includes: x]]]) not ifTrue: [
					queue add: x.
				].
			].
		].
	].
	^writeSet.

]

{ #category : 'Accessors' }
CanonicalObjectManager >> canonicalClasses [

	^registries keys
		inject: IdentitySet new
		into: [:sum :each | sum addAll: each. sum].

]

{ #category : 'Registry Management' }
CanonicalObjectManager >> canonicalizePotentialCanonicalObjects [

	registries do: [:each | each canonicalizePotentialCanonicalObjects].

]

{ #category : 'Canonical' }
CanonicalObjectManager >> canonicalObjectFor: anObject [
	"returns a possible canonical object"

	| domain |
	domain := self
		_registryFor: anObject
		ifAbsent: [^anObject].
	^domain canonicalObjectFor: anObject.

]

{ #category : 'Cleanup' }
CanonicalObjectManager >> cleanupAll: aCollection [

	| beginTime endTime |
	beginTime := System timeNs.
	aCollection do: [:each |
		referencingObjectPolicy canonicalizeReferencesIn: each.
	].
	endTime := System timeNs.
	self
		sessionCacheStatAt: 3 		incrementBy: 1;  						"count of calls to #'cleanupAll:'"
		sessionCacheStatAt: 4 		incrementBy: aCollection size;  	"count of objects scanned during cleanup"
		sessionCacheStatAt: 10	incrementBy: ((endTime - beginTime) // 1000 max: 1);	"microseconds in #'cleanupAll:'"
		yourself.

]

{ #category : 'Cleanup' }
CanonicalObjectManager >> cleanupWriteSet [

	| beginTime writeSet endTime |
	beginTime := System timeNs.
	writeSet := self _writeSet.
	endTime := System timeNs.
	self
		sessionCacheStatAt: 9 incrementBy: ((endTime - beginTime) // 1000 max: 1);	"microseconds in building writeSet"
		cleanupAll: writeSet;
		yourself.

]

{ #category : 'Registry Management' }
CanonicalObjectManager >> createRegistryFor: aClass [

	registries
		at: aClass classHistory
		ifAbsentPut: [CanonicalObjectRegistry newForManager: self].

]

{ #category : 'Canonical' }
CanonicalObjectManager >> hasCanonicalObjectFor: anObject [

	| domain |
	domain := self
		_registryFor: anObject
		ifAbsent: [^false].
	^domain hasCanonicalObjectFor: anObject.

]

{ #category : 'Canonical' }
CanonicalObjectManager >> hasUltimateCanonicalObjectFor: anObject [

	| domain |
	domain := self
		_registryFor: anObject
		ifAbsent: [^false].
	^domain hasUltimateCanonicalObjectFor: anObject.

]

{ #category : 'Initialize' }
CanonicalObjectManager >> initialize [

	registries :=  IdentityKeyValueDictionary new.
	referencingObjectPolicy := ReferencingObjectPolicy newForManager: self.

]

{ #category : 'Registry Management' }
CanonicalObjectManager >> isCanonicalClass: aClass [

	^registries includesKey: aClass classHistory.

]

{ #category : 'Canonical' }
CanonicalObjectManager >> isCanonicalObject: anObject [
	"Could answer true for two equivalent objects if both are in potentialCanonicalObjects"

	| domain |
	domain := self
		_registryFor: anObject
		ifAbsent: [^false].
	^domain isCanonicalObject: anObject.

]

{ #category : 'Canonical' }
CanonicalObjectManager >> isUltimateCanonicalObject: anObject [

	| domain |
	domain := self
		_registryFor: anObject
		ifAbsent: [^false].
	^domain isUltimateCanonicalObject: anObject.

]

{ #category : 'Security' }
CanonicalObjectManager >> objectSecurityPolicy: anObjectSecurityPolicy [

	super objectSecurityPolicy: anObjectSecurityPolicy.
	referencingObjectPolicy objectSecurityPolicy: anObjectSecurityPolicy.
	registries objectSecurityPolicy: anObjectSecurityPolicy.

]

{ #category : 'Accessors' }
CanonicalObjectManager >> referencingObjectPolicy [

	^referencingObjectPolicy

]

{ #category : 'Statistics' }
CanonicalObjectManager >> resetSessionCacheStats [

	| endOffset |
	sessionCacheStatOffset ifNil: [^self].
	endOffset := sessionCacheStatOffset + self sessionCacheStats size - 1.
	sessionCacheStatOffset to: endOffset do: [:i |
		System sessionCacheStatAt: i put: 0.
	].

]

{ #category : 'Statistics' }
CanonicalObjectManager >> sessionCacheStatAt: anIndex [

	sessionCacheStatOffset ifNil: [^0].
	^System
		sessionCacheStatAt: sessionCacheStatOffset + anIndex - 1.

]

{ #category : 'Statistics' }
CanonicalObjectManager >> sessionCacheStatAt: anIndex incrementBy: aSmallInt [

	sessionCacheStatOffset ifNil: [^self].
	System
		sessionCacheStatAt: sessionCacheStatOffset + anIndex - 1
		incrementBy: aSmallInt.

]

{ #category : 'Statistics' }
CanonicalObjectManager >> sessionCacheStatAt: anIndex put: aSmallInt [

	sessionCacheStatOffset ifNil: [^self].
	System
		sessionCacheStatAt: sessionCacheStatOffset + anIndex - 1
		put: aSmallInt.

]

{ #category : 'Statistics' }
CanonicalObjectManager >> sessionCacheStatDescriptions [

	^#(
	"1"	'count of objects added to global registry'
	"2"	'count of objects added to potentialCanonicalObjects'

	"3"	'count of calls to #''cleanupAll:'''
	"4"    'count of objects scanned during cleanup'
	"5"    'count of referencing objects scanned during cleanup'
	"6"    'count of referencing objects changed during cleanup'
	"7"    'count of references changed during cleanup'
	"8"	'count of invariant referencing objects scanned during cleanup'

	"9"    'microseconds in building writeSet'
	"10"	'microseconds in #''cleanupAll:'''
	)

]

{ #category : 'Statistics' }
CanonicalObjectManager >> sessionCacheStatOffset [
	"nil means do not collect any statistics; aSmallInteger means start statistics here"

	^sessionCacheStatOffset

]

{ #category : 'Statistics' }
CanonicalObjectManager >> sessionCacheStatOffset: anIntegerOrNil [
	"nil means do not collect any statistics; aSmallInteger means start statistics here"

	sessionCacheStatOffset := anIntegerOrNil.

]

{ #category : 'Statistics' }
CanonicalObjectManager >> sessionCacheStats [
"
	CanonicalObjectManager soleInstance sessionCacheStats.
"
	^self sessionCacheStatsForSession: System session.

]

{ #category : 'Statistics' }
CanonicalObjectManager >> sessionCacheStatsForSession: anInteger [

	sessionCacheStatOffset ifNil: [^#()].
	^(System sessionCacheStatsForSessionId: anInteger)
		copyFrom: sessionCacheStatOffset + 1
		to: sessionCacheStatOffset + self sessionCacheStatDescriptions size.

]

{ #category : 'Statistics' }
CanonicalObjectManager >> sessionCacheStatsSize [

	^self sessionCacheStatDescriptions size

]

{ #category : 'Canonical' }
CanonicalObjectManager >> ultimateCanonicalObjectFor: anObject [
	"returns an actual canonical object (with possible conflicts)"

	| domain |
	domain := self
		_registryFor: anObject
		ifAbsent: [^anObject].
	^domain ultimateCanonicalObjectFor: anObject.

]
