Extension { #name : 'CanonicalObjectRegistry' }

{ #category : 'Instance Creation' }
CanonicalObjectRegistry class >> new [

	self shouldNotImplement: #'new'.

]

{ #category : 'Instance Creation' }
CanonicalObjectRegistry class >> newForManager: aManager [

	^super new
		initializeForManager: aManager;
		yourself.

]

{ #category : 'Accessors' }
CanonicalObjectRegistry >> _canonicalObjectPolicy [

	^canonicalObjectPolicy.

]

{ #category : 'Accessors' }
CanonicalObjectRegistry >> _potentialCanonicalObjects [

	^potentialCanonicalObjects.

]

{ #category : 'Accessors' }
CanonicalObjectRegistry >> _registry [

	^registry.

]

{ #category : 'Registry Management' }
CanonicalObjectRegistry >> canonicalizePotentialCanonicalObjects [

	| list |
	list := potentialCanonicalObjects asArray.
	potentialCanonicalObjects removeAll: list.
	list do: [:each |
		self ultimateCanonicalObjectFor: each.
	].

]

{ #category : 'Lookup' }
CanonicalObjectRegistry >> canonicalObjectFor: anObject [
	"returns an actual or potential canonical object (without
	conflicts, but might need to canonicalize potentials later).
	Does not yet deal with local registry."

	| canonical |
	canonical := self ultimateCanonicalObjectOrNilFor: anObject.
	canonical ifNotNil: [^canonical].
	potentialCanonicalObjects do: [:each | anObject = each ifTrue: [^each]].
	(canonicalObjectPolicy wantsToCanonicalize: anObject) ifFalse: [^anObject].
	canonical := canonicalObjectPolicy canonicalRepresentationOf: anObject.
	potentialCanonicalObjects add: canonical.
	manager sessionCacheStatAt: 2 incrementBy: 1.
	^canonical.

]

{ #category : 'Lookup' }
CanonicalObjectRegistry >> hasCanonicalObjectFor: anObject [

	| canonical |
	canonical := self ultimateCanonicalObjectOrNilFor: anObject.
	canonical ifNotNil: [^true].
	^potentialCanonicalObjects includesValue: anObject.

]

{ #category : 'Lookup' }
CanonicalObjectRegistry >> hasUltimateCanonicalObjectFor: anObject [

	"^(registry at: anObject otherwise: nil) notNil."
	^registry includesKey: anObject

]

{ #category : 'Initialize' }
CanonicalObjectRegistry >> initializeForManager: aManager [

	canonicalObjectPolicy := CanonicalObjectPolicy new.
	manager := aManager.
	potentialCanonicalObjects := RcIdentityBag new.
	registry := KeyValueDictionary new.
	self objectSecurityPolicy: aManager objectSecurityPolicy.

]

{ #category : 'Lookup' }
CanonicalObjectRegistry >> isCanonicalObject: anObject [
	"Could answer true for two equivalent objects if both are in potentialCanonicalObjects"

	| canonical |
	canonical := self ultimateCanonicalObjectOrNilFor: anObject.
	canonical == anObject ifTrue: [^true].
	^potentialCanonicalObjects includesIdentical: anObject.

]

{ #category : 'Lookup' }
CanonicalObjectRegistry >> isUltimateCanonicalObject: anObject [

	^(registry at: anObject otherwise: nil) == anObject.

]

{ #category : 'Security' }
CanonicalObjectRegistry >> objectSecurityPolicy: anObjectSecurityPolicy [

	super objectSecurityPolicy: anObjectSecurityPolicy.
	canonicalObjectPolicy objectSecurityPolicy: anObjectSecurityPolicy.
	potentialCanonicalObjects objectSecurityPolicy: anObjectSecurityPolicy.
	registry objectSecurityPolicy: anObjectSecurityPolicy.

]

{ #category : 'Lookup' }
CanonicalObjectRegistry >> ultimateCanonicalObjectFor: anObject [
	"returns an actual canonical object (with possible conflicts)"

	^registry
		at: anObject
		ifAbsentPut: [
			manager sessionCacheStatAt: 1 incrementBy: 1.
			canonicalObjectPolicy canonicalRepresentationOf: anObject].

]

{ #category : 'Lookup' }
CanonicalObjectRegistry >> ultimateCanonicalObjectOrNilFor: anObject [

	^registry
		at: anObject
		ifAbsent: [nil].

]
