Extension { #name : 'Class' }

{ #category : 'Private' }
Class >> _logServer: aString [
  (SessionTemps current at: #ROWAN_TRACE otherwise: nil) == false ifFalse:[
     GsFile gciLogServer: aString .
  ]
]

{ #category : 'Private' }
Class >> _clearCachedOrganizer [

   ClassOrganizer clearCachedOrganizer

]

{ #category : 'Subclass Creation' }
Class >> _makeClassVarDict: anArrayOfClassVars [

"Turns Array of class variable names into a SymbolDictionary.
 Returns nil if anArrayOfClassVars is empty or nil. "

| newDict numCvs |
self _validatePrivilege ifTrue:[
  anArrayOfClassVars ifNotNil:[
    anArrayOfClassVars _isArray ifFalse:[
      (anArrayOfClassVars _validateClass: Array) ifFalse:[ ^ nil ].
    ].
    (numCvs := anArrayOfClassVars size) ~~ 0 ifTrue:[ | assocClass  |
      newDict := SymbolDictionary new.
      assocClass := SessionTemps current at:#ClassVariablesAssociationClass otherwise: SymbolAssociation.
      1 to: numCvs do: [:index|
        | aVarName aSym definingClass |
	aVarName := anArrayOfClassVars at: index .
        (aVarName _isOneByteString or:[ aVarName _validateClass: CharacterCollection]) ifTrue:[
	  aSym := aVarName asSymbol .
	  aSym validateIsIdentifier "fix bug 9666" .
	  definingClass := self _classDefiningClassVar: aSym .  "fix bug 10480"
	  definingClass ifNotNil:[
            LookupError new object: definingClass; key: aSym ; reason: #classErrClassVarNameExists ;
		details: 'class variable already exists'; signal
	  ] ifNil:[
	    newDict addAssociation:( assocClass newWithKey: aSym value: nil )
	  ]
        ].
      ].
    ].
  ].
].
^ newDict .

]

{ #category : 'Private' }
Class >> _sortedClassVarNames [

"Returns a sorted list of class variable names. "

 | res |
 res := SortedCollection new .
 classVars ifNotNil:[ res addAll: classVars keys ].
 ^ res

]

{ #category : 'Private' }
Class >> _validateInstancesPersistent [

"check that it is legal to make instances of the receiver
 persistent."

| list |
list := {
  IdentityKeySoftValueDictionary . IdentitySoftCollisionBucket .
  KeySoftValueDictionary . SoftCollisionBucket . SoftReference . Semaphore
 }.
1 to: list size do:[ :n |  | aCls |
  aCls := list at: n .
  aCls == self ifTrue:[
    ImproperOperation new reason: #rtErrSuperclassIsNP;
	details: 'instances of this class are always non-persistent';
	object: self ; signal .
    ^ false
  ]
].
^ true
]

{ #category : 'Private' }
Class >> unsafeRenameInstVarNamed: oldName to: newName [
"Renames an instance variable of a class. Once completed, all instances of
 the receiver and its subclasses will have the new instance variable name.
 Recompiles all methods in the receiver and its subclasses which directly reference
 the instance variable.

 Returns true on success. Raises an exception and prevents the transaction
 from being committed on error. Raises an error if the current transaction contains
 uncommitted changes. Raises an error if oldName is an inherited instance 
 variable. Renaming inherited instance variables is disallowed.

 *** WARNINGS ***
1) Do not use if any instances are members of any indexed collection. Doing so will likely corrupt the indexes and make them unusable.
2) This is a private method. Use at your own peril.
 *** WARNINGS ****
"
System needsCommit 
  ifTrue: [ self _error: #rtErrAbortWouldLoseData ] .
(self instVarNames includesIdentical: oldName asSymbol)
	ifFalse:[ Error signal: ('class ', self name asString, ' does not have an instance variable named #', oldName asString) ].
[
  |oldMethsAndOffsets oldIvName newIvName newMeths org |
  org := ClassOrganizer new .
  oldIvName := oldName asSymbol .
  newIvName := newName asSymbol .
  oldMethsAndOffsets := Array new .
  (org accessorsOf: oldIvName inClass: self ) do:[:meth | 
    meth isMethodForBlock ifTrue:[ Error signal: 'problem in class organizer' ].
    (meth sourceOffsetsOfInstVar: oldIvName) ifNotNil:[ :offsets|
      oldMethsAndOffsets add: { meth . offsets }.
    ] ifNil:[ Error signal: 'instVar not found in method' ].
  ].

  self _unsafeRenameInstVarNamed: oldIvName to: newIvName.
  self allSubclasses do:[:e| e _unsafeRenameInstVarNamed: oldIvName to: newIvName.].
  newMeths := Array new .
  oldMethsAndOffsets do:[:ary | | meth offsets src |
    meth := ary at: 1 .
    offsets := ary at: 2 .
    src := meth sourceString copy .
    (SortedCollection withAll: offsets) reverseDo:[:ofs |
      src removeFrom: ofs to: (ofs + oldIvName size - 1) .
      src insertAll: newIvName at: ofs .
    ].
  newMeths add:( meth inClass compileMethod: src ).
 ].
  self _refreshClassCache: true .
] on: Error do:[:ex|   System disableCommitsUntilAbortWithReason: 'error in #unsafeRenameInstVarNamedto:' . ex resume ].
^ true
]

{ #category : 'Private' }
Class >> _unsafeRenameInstVarNamed: oldName to: newName [

"Do not call this method directly. Call Class>>#unsafeRenameInstVarNamed:to:
 instead, which calls this method."
| idx |
idx := self _instVarNames indexOfIdentical: oldName  .
idx == 0 ifTrue:[ self error: ('instvar with name', oldName asString, ' not found')].
self _instVarNames _unsafeAt: idx put: newName.
self _refreshClassCache: true .
^ true
]
