Extension { #name : 'EnumeratedPathTerm' }

{ #category : 'Updating' }
EnumeratedPathTerm >> _canonicalizeTerm: aSymbol [
  "Canonicalize the pathTerm name, so that terms are in alphabetical order"

  | terms sz canon |
  terms := (aSymbol subStringsDelimitedBy: $|) sortAscending.
  canon := String new.
  sz := terms size.
  1 to: sz do: [ :index |
    | term |
    term := terms at: index.
    canon add: term.
    index < sz
      ifTrue: [ canon add: $| ] ].
  ^ canon asSymbol

]

{ #category : 'Accessing' }
EnumeratedPathTerm >> _ivOffsetFor: anObject pathName: pathName [
  "Returns the instance variable offset of anObject for the pathName which is one of the enumarated instance variables."

  ^ (self _classOf: anObject) _idxIvOffsetOf: pathName asSymbol

]

{ #category : 'Accessing' }
EnumeratedPathTerm >> _ivOffsetsFor: anObject [
  ^ self pathTermElements
    collect: [ :pathTermElement | self _ivOffsetFor: anObject pathName: pathTermElement ]

]

{ #category : 'Accessing' }
EnumeratedPathTerm >> _nextObjectFor: anObject pathName: aPathName [
  "Returns the object at the instance variable that corresponds to the receiver
 path term."

  | ivOffset |
  ivOffset := self _ivOffsetFor: anObject pathName: aPathName.
  ivOffset ifNil: [ ^ nil ].
  ^ self _nextObjectFor: anObject atInstVar: ivOffset

]

{ #category : 'Updating Indexes' }
EnumeratedPathTerm >> addMappingsForObject: anObject logging: aBoolean [
  "Add index dictionary and dependency list entries for anObject."

  | ivOffset indexDict sz |
  (nil == anObject or: [ self size == 0 ])
    ifTrue: [ ^ self ].
  self pathTermElements
    do: [ :pathTermElement |
      (self _ivOffsetFor: anObject pathName: pathTermElement)
        ifNil: [ self _invalidIvOffset: anObject ]
        ifNotNil: [ :off | ivOffset := off ] ].
  anObject
    getDepListAndAddPathTerm: self
    withIVOffset: ivOffset
    logging: aBoolean.	" add an entry to the objects dependency list ... note that we ar only using the last ivOffset and that is okay as we compensate in the places that the ivOffset is used..."
  sz := children size.
  indexDict := self getIndexDictionary.
  self
    nextObj: anObject
    do: [ :nextObj |
      " insert an entry into the index dictionary "
      indexDict
        _at: nextObj
        put: anObject
        term: self
        logging: aBoolean.
      updateBtree ~~ nil
        ifTrue: [
          " insert an entry into the B-tree "
          updateBtree btreeAt: nextObj put: anObject.
          needsDepList
            ifTrue: [
              " add an entry to the very last object's dependency list "
              nextObj getDepListAndAddLastElementPathTerm: self logging: aBoolean ] ].
      nil ~~ nextObj
        ifTrue: [
          1 to: sz do: [ :i |
            " make recursive call to add mappings "
            (children at: i) addMappingsForObject: nextObj logging: aBoolean ] ] ]

]

{ #category : 'Updating Indexes' }
EnumeratedPathTerm >> addMappingsUsing: mapInfo logging: aBoolean [
  "Adds index dictionary and dependency list entries for the object described by
 the map info object.  If aBoolean is true, operations will be recorded in the
 system redo log."

  | pathTerm object ivOffset indexDict dictUpdated |
  pathTerm := mapInfo pathTerm.
  pathTerm size == 0
    ifTrue: [ ^ self ].
  object := mapInfo object.
  nil == object
    ifTrue: [ ^ self ].
  self pathTermElements
    do: [ :pathTermElement |
      (self _ivOffsetFor: object pathName: pathTermElement)
        ifNil: [ self _invalidIvOffset: object ]
        ifNotNil: [ :off | ivOffset := off ] ].
  object
    getDepListAndAddPathTerm: pathTerm
    withIVOffset: ivOffset
    logging: aBoolean.	" add an entry to the objects dependency list "
  dictUpdated := false.
  indexDict := self getIndexDictionary.
  1 to: mapInfo size do: [ :k |
    | nextMaps |
    " for each child mapping "
    nextMaps := mapInfo at: k.
    1 to: nextMaps size do: [ :i |
      | nextMap nextObj nextPathTerm |
      " there is a collection of mappings for each element in the set "
      nextMap := nextMaps at: i.
      nextObj := nextMap object.
      updateBtree ~~ nil
        ifTrue: [
          " insert an entry into the B-tree "
          updateBtree btreeAt: nextObj put: object.
          (needsDepList and: [ nil ~~ nextObj ])
            ifTrue: [
              " add an entry to the very last object's dependency list "
              nextObj
                getDepListAndAddLastElementPathTerm: pathTerm
                logging: aBoolean ] ].
      (updateDict notNil and: [ dictUpdated not ])
        ifTrue: [
          " insert an entry into the index dictionary "
          indexDict
            _at: nextObj
            put: object
            term: self
            logging: aBoolean ].
      (nextPathTerm := nextMap pathTerm) ~~ nil
        ifTrue: [ nextPathTerm addMappingsUsing: nextMap logging: aBoolean ] ].
    dictUpdated := true ]

]

{ #category : 'Deprecated' }
EnumeratedPathTerm >> allDependencyListsFor: anObject into: all [
  "Put any dependency lists that can be found from the receiver
into the given set."

  | nextObj depList |
  self deprecated: 'EnumeratedPathTerm>>allDependencyListsFor:into: deprecated v3.3.
     Use DependencyList class>>depMapValues to collect dependency lists.'.  "bug 43886"

  (nil == anObject or: [ self size == 0 ])
    ifTrue: [ ^ self ].
  (depList := DependencyList for: anObject) == nil
    ifTrue: [ ^ self ].
  all add: depList.
  self indicatesIndexOnRootNsc
    ifTrue: [ ^ self ].
  self pathTermElements
    do: [ :pathName |
      nextObj := self _nextObjectFor: anObject pathName: pathName.
      nextObj
        ifNotNil: [
          needsDepList
            ifTrue: [
              (depList := DependencyList for: nextObj) ~~ nil
                ifTrue: [ all add: depList ] ].
          1 to: children size do: [ :i | (children at: i) allDependencyListsFor: nextObj into: all ] ] ]

]

{ #category : 'Audit' }
EnumeratedPathTerm >> auditDepListFor: obj index: indexObj on: aString optionalSentinel: optionalSentinel [
  "Private."

  | depList j ivOffset |
  (nil == obj or: [ self size == 0 ])
    ifTrue: [ ^ nil ].
  depList := DependencyList for: obj.
  depList == nil
    ifTrue: [
      obj isInvariant
        ifFalse: [
          self
            auditInfo:
              {obj.
              indexObj.
              depList}
            on: aString.
          aString
            add: ' Object has no dependency list';
            add: Character lf . self checkDebug: aString ] ]
    ifFalse: [
      (DepListTable _hasDependencyBitFor: obj)
        ifFalse: [
          self
            auditInfo:
              {obj.
              indexObj.
              depList}
            on: aString.
          aString
            add: ' Object has DependencyList but no dependency bit is set';
            add: Character lf. self checkDebug: aString  ].
      j := depList _findOffsetForPathTerm: self.
      j == nil
        ifTrue: [
          self
            auditInfo:
              {obj.
              indexObj.
              depList}
            on: aString.
          aString
            add: 'DependencyList has no entry for path term';
            add: Character lf . self checkDebug: aString ]
        ifFalse: [
          "found entry for self in depList"
          ivOffset := depList at: j + 1.	"may be negative for last element reference count"
          ivOffset < 0
            ifTrue: [
              "reference count for last element, but we don't expect reference
               count here ... see needsDepList clause below where depList
               existence is sufficient"
              self
                auditInfo:
                  {obj.
                  indexObj.
                  depList}
                on: aString.
              aString
                add: 'DependencyList on last element in unexepected location ';
                add: ivOffset asString;
                add: Character lf. self checkDebug: aString .
              ^ self ] ] ].
  needsDepList
    ifTrue: [
      self
        nextObj: obj
        do: [ :nextObj |
          (DependencyList needsDepList: nextObj)
            ifTrue: [
              depList := DependencyList for: nextObj.
              depList == nil
                ifTrue: [
                  self
                    auditInfo:
                      {nextObj.
                      indexObj.
                      depList}
                    on: aString.
                  aString
                    add: ' Object has no dependency list';
                    add: Character lf . self checkDebug: aString ] ] ] ].
  ^ self

]

{ #category : 'Audit' }
EnumeratedPathTerm >> auditIndexDictionaryForObject: obj occurrences: num on: aString [
  "Private.  Unsupported method for GemStone Technical Support."

  "Verifies that the given object has the correct index dictionary entries for the receiver."

  | sz indexObj newNum dictChecked nextObjBag |
  sz := self size.
  (nil == obj _or: [ sz == 0 ])
    ifTrue: [ ^ aString ].
  nextObjBag := IdentityBag new.
  self nextObj: obj do: [ :nextObj | nextObjBag add: nextObj ].
  nextObjBag
    do: [ :nextObj |
      dictChecked := false.
      1 to: sz do: [ :i |
        " for each index that utilizes the path term "
        indexObj := self at: i.
        (indexObj isRangeEqualityIndex and: [ indexObj size == offset ])
          ifFalse: [
            " verify index dictionary has correct entries "
            dictChecked
              ifFalse: [
                | occurs |
                occurs := nextObjBag occurrencesOf: nextObj.
                newNum := self
                  auditDictionaryEntryFor: obj
                  and: nextObj
                  occurrences: num * occurs
                  on: aString.
                dictChecked := true ] ] ].
      nextObj ~~ nil
        ifTrue: [
          1 to: children size do: [ :i |
            (children at: i)
              auditIndexDictionaryForObject: nextObj
              occurrences: newNum
              on: aString ] ] ]

]

{ #category : 'Audit' }
EnumeratedPathTerm >> auditNscCountsFor: obj on: aString count: btreeCounts [
  "Private. "

  | sz |
  sz := self size.
  (nil == obj _or: [ sz == 0 ])
    ifTrue: [ ^ aString ].
  nil == obj
    ifTrue: [ ^ self ].
  self
    auditDepListFor: obj
    index: nil
    on: aString
    optionalSentinel: Object new.
  self
    nextObj: obj
    do: [ :nextObj |
      | index |
      index := 1.
      children isEmpty
        ifTrue: [ self auditDirectNscCountsFor: nextObj on: aString count: btreeCounts ]
        ifFalse: [
          1 to: sz do: [ :i |
            | count indexObj |
            indexObj := self at: i.
            (indexObj isRangeEqualityIndex and: [ indexObj size == offset ])
              ifTrue: [
                count := btreeCounts at: index.
                count == 0
                  ifTrue: [
                    aString
                      add: Character lf;
                      add: ' -- The number of entries in Btree [';
                      add: (self at: i) btreeRoot asOop asString;
                      add: '] for pathTerm [';
                      add: self asOop asString;
                      add: ' offset ';
                      add: offset asString;
                      add:
                          '] does not match the number of entries in the base collection (extra elements in either the base collection or btree).';
                      add: Character lf ].
                btreeCounts at: index put: count - 1.
                index := index + 1 ] ].
          nextObj ~~ nil
            ifTrue: [
              1 to: children size do: [ :i |
                (children at: i)
                  auditNscCountsFor: nextObj
                  on: aString
                  count: (btreeCounts at: index).
                index := index + 1 ] ] ] ]

]

{ #category : 'Audit' }
EnumeratedPathTerm >> checkDebug: aString [
  BtreePlusNodeAuditor checkDebug: aString 

]

{ #category : 'Updating Indexes' }
EnumeratedPathTerm >> cleanupDependencyListFor: anObject [
  ""

  | deplistOffset nextObj depList |
  (nil == anObject or: [ self size == 0 ])
    ifTrue: [ ^ self ].
  (depList := DependencyList for: anObject) == nil
    ifTrue: [ ^ self ].
  deplistOffset := depList removeCompletelyPathTerm: self for: anObject.	" remove the receiver from the dependency list "
  (self indicatesIndexOnRootNsc
    or: [
      "last element dependency list"
      deplistOffset < 0 ])
    ifTrue: [ ^ self ].
  self pathTermElements
    do: [ :pathName |
      (self _ivOffsetFor: anObject pathName: pathName)
        ifNotNil: [ :ivOffset |
          nextObj := self _nextObjectFor: anObject atInstVar: ivOffset.
          nextObj
            ifNotNil: [
              needsDepList
                ifTrue: [
                  (depList := DependencyList for: nextObj) ~~ nil
                    ifTrue: [ depList removeCompletelyPathTerm: self for: nextObj ] ].
              1 to: children size do: [ :i | (children at: i) cleanupDependencyListFor: nextObj ] ] ] ]

]

{ #category : 'Testing' }
EnumeratedPathTerm >> coversIvOffset: anIvOffset for: anObject [
  "Answer true if the receiver covers the given iv offset (see bug 46705)"

  self pathTermElements
    do: [ :pathTermElement |
      (self _ivOffsetFor: anObject pathName: pathTermElement)
        ifNil: [ ^false ]
        ifNotNil: [ :off | anIvOffset = off ifTrue: [ ^true ] ] ].
  ^ false

]

{ #category : 'Updating Indexes' }
EnumeratedPathTerm >> getMappingInfoFor: anObject ifObject: origObject atOffset: origOffset replaceWith: otherObject [
  "Gathers all the objects, their dependency lists, and the instance variable
 offset for each path term in the tree of path terms.  Puts them in a
 MappingInfo object.  Builds a tree of MappingInfo objects for each child path
 term."

  | ivOffset nextObj maps mapInfo |
  mapInfo := self _mappingInfoClass new pathTerm: self.
  (nil == anObject or: [ self size == 0 ])
    ifTrue: [ ^ mapInfo ].
  mapInfo object: anObject.
  (updateBtree ~~ nil or: [ children isEmpty ])
    ifTrue: [
      maps := Array new.
      self pathTermElements
        do: [ :pathName |
          ivOffset := self _ivOffsetFor: anObject pathName: pathName.
          ivOffset
            ifNotNil: [
              nextObj := self _nextObjectFor: anObject atInstVar: ivOffset.
              updateBtree ~~ nil
                ifTrue: [
                  (self _checkBtreeComparisonWith: nextObj)
                  ifFalse: [
                    | exa |
                    (exa := ImproperOperation new)
                      _number:
                          (ErrorSymbols at: #'rtErrRangeEqualityIndexInvalidClassKindForBtree');
                      args: { nextObj class. self name. updateBtree lastElementClassDescription }.
                    ^ {false.	" indicating no indexing objects were modified "
                    exa} ] ].
              maps add: (self _mappingInfoClass new object: nextObj) ] ].
      mapInfo addLast: maps.
      ^ mapInfo ].
  1 to: children size do: [ :i |
    " for each following path term "
    maps := Array new.
    self pathTermElements
      do: [ :pathName |
        | aMapInfo |
        ivOffset := self _ivOffsetFor: anObject pathName: pathName.
        ivOffset
          ifNotNil: [
            " get the next object along the path "
            nextObj := self _nextObjectFor: anObject atInstVar: ivOffset.
            aMapInfo := (children at: i)
              getMappingInfoFor: nextObj
              ifObject: origObject
              atOffset: origOffset
              replaceWith: otherObject.
            aMapInfo class == self _mappingInfoClass
              ifFalse: [ ^ aMapInfo	"an Array of error info" ].
            maps add: aMapInfo ] ].
    mapInfo addLast: maps ].
  ^ mapInfo

]

{ #category : 'Testing' }
EnumeratedPathTerm >> hasMatchingTerm: aSymbol [
  "Answer true if the receiver's term matches <aSymbol>"

  ^ self name == (self _canonicalizeTerm: aSymbol)

]

{ #category : 'Testing' }
EnumeratedPathTerm >> indicatesIndexOnNscElements [
  "Returns false. The ides is not on elements of an nsc."

  ^ false

]

{ #category : 'Testing' }
EnumeratedPathTerm >> isEnumeratedTerm [
  "Returns true if the receiver indicates an enumerated path term."

  ^ true

]

{ #category : 'Updating' }
EnumeratedPathTerm >> name: aSymbol [
  name := self _canonicalizeTerm: aSymbol

]

{ #category : 'Accessing' }
EnumeratedPathTerm >> nextObj: anObj do: aBlock [
  | found |
  (nil == anObj or: [ self size == 0 ])
    ifTrue: [ ^ self ].
  found := false.
  self pathTermElements
    do: [ :pathName |
      | ivOffset |
      (ivOffset := self _ivOffsetFor: anObj pathName: pathName) ~~ nil
        ifTrue: [
          | nextObj |
          found := true.
          nextObj := self _nextObjectFor: anObj atInstVar: ivOffset.
           aBlock value: nextObj ] ].
  found
    ifFalse: [ ^ self _invalidIvOffset: anObj ]

]

{ #category : 'Accessing' }
EnumeratedPathTerm >> pathTermElements [
  "return the individual pathTerm elements"

  ^ self name subStringsDelimitedBy: $|

]

{ #category : 'Updating Indexes' }
EnumeratedPathTerm >> removeMappingsFor: anObject lastOne: aBoolean logging: doLogging [
  "Remove entries in the index dictionary and dependency lists for anObject."

  | nextObj depList more isLast |
  self size == 0
    ifTrue: [ ^ self ].
  self pathTermElements
    do: [ :pathName |
      nextObj := self _nextObjectFor: anObject pathName: pathName.
      updateBtree ~~ nil
        ifTrue: [
          " remove an entry from the B-tree "
          updateBtree btreeRemoveKey: nextObj value: anObject.
          DependencyList removePathTerm: self for: nextObj logging: doLogging	" remove dependency list entry for next object " ].
      more := false.
      updateDict ~~ nil
        ifTrue: [
          " remove the index dictionary entry "
          more := updateDict
            removeKey: nextObj
            value: anObject
            term: self
            logging: doLogging ].
      (nil ~~ nextObj and: [ children isEmpty not ])
        ifTrue: [
          " make recursive call to remove mappings "
          isLast := aBoolean and: [ more not ].
          1 to: children size do: [ :i | (children at: i) removeMappingsFor: nextObj lastOne: isLast logging: doLogging ].
          isLast
            ifTrue: [
              " remove dependency list entries for next object "
              (depList := DependencyList for: nextObj) ~~ nil
                ifTrue: [ depList removePathTerms: children for: nextObj logging: doLogging ] ] ] ]

]

{ #category : 'Updating Indexes' }
EnumeratedPathTerm >> removeMappingsSkipBtreeFor: anObject lastOne: aBoolean logging: doLogging [
  "Remove entries in the index dictionary and dependency lists for anObject.
 This version for index removal, it does not update the btree since the btree
 is going away (#40858)."

  | nextObj depList more isLast |
  self size == 0
    ifTrue: [ ^ self ].
  self pathTermElements
    do: [ :pathName |
      nextObj := self _nextObjectFor: anObject pathName: pathName.
      updateBtree ~~ nil
        ifTrue: [
          " skip update of btree, index is going away"
          " remove dependency list entry for next object "
          DependencyList removePathTerm: self for: nextObj logging: doLogging ].
      more := false.
      updateDict ~~ nil
        ifTrue: [
          " remove the index dictionary entry "
          more := updateDict
            removeKey: nextObj
            value: anObject
            term: self
            logging: doLogging ].
      (nil ~~ nextObj and: [ children isEmpty not ])
        ifTrue: [
          " make recursive call to remove mappings "
          isLast := aBoolean and: [ more not ].
          1 to: children size do: [ :i | (children at: i) removeMappingsFor: nextObj lastOne: isLast logging: doLogging ].
          isLast
            ifTrue: [
              " remove dependency list entries for next object "
              (depList := DependencyList for: nextObj) ~~ nil
                ifTrue: [ depList removePathTerms: children for: nextObj logging: doLogging ] ] ] ]

]
