"
This class is part of the External Sessions implementation and represents
an execution Error from the GCI interface to the legacy external session.

Constraints:
	gsResumable: Boolean
	gsTrappable: Object
	gsNumber: SmallInteger
	currGsHandler: GsExceptionHandler
	gsStack: Object
	gsReason: String
	gsDetails: Object
	tag: Object
	messageText: Object
	gsArgs: Object
	errorDescription: Object
	externalSession: Object
	originalNumber: Object

"
Class {
	#name : 'GciLegacyError',
	#superclass : 'Error',
	#instVars : [
		'errorDescription',
		'externalSession',
		'originalNumber'
	],
	#category : 'External Sessions'
}

{ #category : 'signalling' }
GciLegacyError class >> signal: anErrorDescription in: aLegacyExternalSession [

	self new
		signal: anErrorDescription
		in: aLegacyExternalSession.

]

{ #category : 'accessing' }
GciLegacyError >> actualExceptionClassOr: aBlock [
	"Answer the class of the actual exception if there is one,
	 or the result of evaluating aBlock if not."

	| actualCls |
	actualCls := (LegacyErrNumMap atOrNil: self number ) ifNotNil: [ :a | a atOrNil: 1].
	^actualCls ifNil: aBlock

]

{ #category : 'accessing' }
GciLegacyError >> category [

	^errorDescription category.

]

{ #category : 'other' }
GciLegacyError >> clearStack [

	externalSession clearStackFor: errorDescription.

]

{ #category : 'accessing' }
GciLegacyError >> context [

	^errorDescription context.

]

{ #category : 'other' }
GciLegacyError >> continue [
	"Continue code execution in GemStone after an error.
	See GciContinue() in the GemBuilder for C manual for details."

	^externalSession
		continue: errorDescription context
		replacingTopOfStackWithOop: nil.

]

{ #category : 'other' }
GciLegacyError >> continueWith: anObject [
	"This function is a variant of the continue method, except
	that it allows you to modify the call stack before attempting
	to continue the suspended Smalltalk execution.
	See GciContinueWith() in the GemBuilder for C manual for details."

	^externalSession
		continue: errorDescription context
		replacingTopOfStackWithOop: anObject asOop.

]

{ #category : 'accessing' }
GciLegacyError >> externalErrorNumber [

	^errorDescription number.

]

{ #category : 'accessing' }
GciLegacyError >> externalSession [

	^externalSession.

]

{ #category : 'testing' }
GciLegacyError >> matchesClasses: expClass [
	"expClass is either a Class or an Array of Classes"

	| actualCls |
	actualCls := self actualExceptionClassOr: [^false].
	expClass _isArray
		ifTrue: [^expClass anySatisfy: [:aCls | actualCls isSubclassOf: aCls]]
		ifFalse: [^actualCls isSubclassOf: expClass]

]

{ #category : 'accessing' }
GciLegacyError >> originalNumber [
  ^ originalNumber

]

{ #category : 'signalling' }
GciLegacyError >> signal: anErrorDescription in: aLegacyExternalSession [

	errorDescription := anErrorDescription.
	externalSession := aLegacyExternalSession.
	(originalNumber := errorDescription number) == 1001 ifTrue: [^self signalCompileError].
	messageText := errorDescription message.
	"avoid killing current session with fatal errors"
	(originalNumber between: 4000 and: 4999) ifTrue: [
		messageText add: ' original number: ' , originalNumber asString.
		"and self number is inherited value 2710"
	] ifFalse: [
		self _number: originalNumber.
	].
	"self stack ifNotNil: [:stack | messageText lf ; add: stack]."
	self details: messageText.
	self signal.

]

{ #category : 'signalling' }
GciLegacyError >> signalCompileError [

	gsArgs := errorDescription asArray.
	"gsArgs size: gciErrSType argCount.
	gsArgs := gsArgs collect: [:each |
		externalSession valueOfOop: each toLevel: 4.
	]."
	messageText := errorDescription message.
	self details: messageText.
	self signal.

]

{ #category : 'accessing' }
GciLegacyError >> stack [
  errorDescription ifNil:[ ^ nil ].
  errorDescription contextOop == nil asOop ifTrue: [^nil].
  originalNumber >= 4000 ifTrue:[ ^ nil ].
  ^ [ externalSession _getStackForOop: errorDescription contextOop 
    ] on: Error do:[ 'stack not available' ].
]
