Extension { #name : 'GsClassicConjunctiveClauseOptimizer' }

{ #category : 'optimizing' }
GsClassicConjunctiveClauseOptimizer >> consolidatePredicates: primaries with: primaryPredicate [
  "See if the primaryPredicates can be consolidated with any of the primiaries."

  "see QueryExecuter>>_consolidatePredicatesAt:and:"

  | consolidatedPredicate |
  primaries
    do: [ :secondaryPredicate |
      (secondaryPredicate ~~ nil  and: [ primaryPredicate ~~ secondaryPredicate ])
        ifTrue: [
          (primaryPredicate canConsolidateWith: secondaryPredicate)
            ifTrue: [
              primaries at: (primaries indexOf: secondaryPredicate) put: nil.
              consolidatedPredicate := primaryPredicate
                consolidateWith: secondaryPredicate.
              consolidatedPredicate evaluator: primaryPredicate evaluator1.
              primaries
                at: (primaries indexOf: primaryPredicate)
                put: consolidatedPredicate.
              ^ self ] ] ]

]

{ #category : 'optimizing' }
GsClassicConjunctiveClauseOptimizer >> consolidateRangePredicates [
  "See if any predicates can be consolidated (i.e. can be satisfied
	with a single lookup using a range predicate)."

  "see QueryExecuter>>_consolidatePredicatesAt:and:"

  | primaries |
  primaries := self predicates copy.
  1 to: primaries size do: [ :index |
    (primaries at: index)
      ifNotNil: [ :primaryPredicate | self consolidatePredicates: primaries with: primaryPredicate ] ].
  predicates := primaries select: [ :each | each ~~ nil  ]

]

{ #category : 'optimizing' }
GsClassicConjunctiveClauseOptimizer >> consolidateUnaryConstantPredicates [
  "Simplify formulas involving predicates that are unary-constant (true or false) or
   constant-constant. The predicate is removed, or the expression is consolidated."

  | otherPredicates truePredicate falsePredicate |
  self predicates size <= 1
    ifTrue: [
      "a single predicate ... no more optimization called for"
      ^ self ].
  truePredicate := falsePredicate := nil.
  otherPredicates := OrderedCollection new.
  self predicates
    do: [ :predicate |
      (predicate isConstantConstant and: [ predicate isConstantBound ])
        ifTrue: [
          predicate executePredicate
            ifTrue: [ truePredicate := predicate ]
            ifFalse: [ falsePredicate := predicate ] ]
        ifFalse: [ otherPredicates add: predicate ] ].
  falsePredicate
    ifNotNil: [ predicates := {falsePredicate} ]
    ifNil: [ truePredicate ifNotNil: [ predicates := otherPredicates ] ]

]

{ #category : 'optimizing' }
GsClassicConjunctiveClauseOptimizer >> normalizePredicates [
  predicates := self predicates
    collect: [ :each |
      "normalize predicates to make simplify analysis"
      each normalize ]

]

{ #category : 'optimizing' }
GsClassicConjunctiveClauseOptimizer >> removeRedundantPredicates [
  "see QueryExecuter>>_removeRedundantOperatorsAt:and:"

  | primaries |
  [
  primaries := self predicates copy.
  1 to: primaries size do: [ :index |
    (primaries at: index)
      ifNotNil: [ :primaryPredicate | self removeRedundantPredicates: primaries with: primaryPredicate ] ].
  predicates := primaries select: [ :each | each ~~ nil  ] ]
    on: GsUnsatisfiableQueryNotification
    do: [ :note |
      "short circuit analysis and replace whole enchilada with a `false` predicate"
      predicates := OrderedCollection with: (GsQueryPredicate constant: false).
      ^ self ]

]

{ #category : 'optimizing' }
GsClassicConjunctiveClauseOptimizer >> removeRedundantPredicates: primaries with: primaryPredicate [
  "see QueryExecuter>>_removeRedundantOperatorsAt:and:"

  primaries
    do: [ :secondaryPredicate |
      (secondaryPredicate ~~ nil  and: [ primaryPredicate ~~ secondaryPredicate ])
        ifTrue: [
          (primaryPredicate redundantPredicateBetween: secondaryPredicate)
            ifNotNil: [ :redundantPredicate | primaries at: (primaries indexOf: redundantPredicate) put: nil ] ] ]

]

{ #category : 'optimizing' }
GsClassicConjunctiveClauseOptimizer >> reorderPredicates [
  | constList identList equalList otherList indexList |
  self predicates size < 2
    ifTrue: [ ^ self ].
  constList := {}.
  identList := {}.
  equalList := {}.
  otherList := {}.
  indexList := {}.
  self predicates
    do: [ :each |
      " if it is constant-constant "
      each isConstantConstant
        ifTrue: [ constList addLast: each ]
        ifFalse: [
          " if an index exists on the path and it is not path-path "
          (each isPathPath not and: [ each usesPathEvaluator not ])
            ifTrue: [ indexList addLast: each ]
            ifFalse: [
              " if it is an identity operation (== or ~~) "
              each usesIdentityOperation
                ifTrue: [ identList addLast: each ]
                ifFalse: [
                  " if operation is = or ~= "
                  each usesEqualityOperation
                    ifTrue: [ equalList addLast: each ]
                    ifFalse: [ otherList addLast: each ] ] ] ] ].
  predicates := OrderedCollection new.
  1 to: constList size do: [ :i | predicates addLast: (constList at: i) ].
  1 to: indexList size do: [ :i | predicates addLast: (indexList at: i) ].
  1 to: identList size do: [ :i | predicates addLast: (identList at: i) ].
  1 to: equalList size do: [ :i | predicates addLast: (equalList at: i) ].
  1 to: otherList size do: [ :i | predicates addLast: (otherList at: i) ]

]

{ #category : 'optimizing' }
GsClassicConjunctiveClauseOptimizer >> runOptimizations [
  self queryOptions transformCommonPaths
    ifTrue: [ self transformCommonPaths ].
  self queryOptions normalizePredicates
    ifTrue: [ self normalizePredicates ].
  self queryOptions removeRedundantPredicates
    ifTrue: [ self removeRedundantPredicates ].
  self queryOptions consolidateRangePredicates
    ifTrue: [ self consolidateRangePredicates ].
  self queryOptions consolidateUnaryConstantPredicates
    ifTrue: [ self consolidateUnaryConstantPredicates ].
  self queryOptions reorderPredicates
    ifTrue: [ self reorderPredicates ]

]

{ #category : 'optimizing' }
GsClassicConjunctiveClauseOptimizer >> transformCommonPaths [
  "see QueryExecuter>>_checkPathPathAt:"

  predicates := self predicates collect: [ :each | each transformCommonPaths ]

]
