Extension { #name : 'GsEnumeratedPathTerm' }

{ #category : 'Updating' }
GsEnumeratedPathTerm >> _canonicalizeTerm: aSymbol [
  "Canonicalize the pathTerm name, so that terms are in alphabetical order"

  | terms sz canon |
  terms := (aSymbol subStringsDelimitedBy: $|) sortAscending.
  canon := String new.
  sz := terms size.
  1 to: sz do: [ :index |
    | term |
    term := terms at: index.
    canon add: term.
    index < sz
      ifTrue: [ canon add: $| ] ].
  ^ canon asSymbol

]

{ #category : 'Testing' }
GsEnumeratedPathTerm >> _isLastOccurrenceInIndexFor: anObject [

  ^ (self findReachableRootsFor: anObject) size <= 1

]

{ #category : 'Accessing' }
GsEnumeratedPathTerm >> _ivOffsetFor: anObject pathName: pathName [
  "Returns the instance variable offset of anObject for the pathName which is one of the enumarated instance variables."

  ^ (self _classOf: anObject) _idxIvOffsetOf: pathName asSymbol

]

{ #category : 'Accessing' }
GsEnumeratedPathTerm >> _ivOffsetsFor: anObject [
  ^ self pathTermElements
    collect: [ :pathTermElement | self _ivOffsetFor: anObject pathName: pathTermElement ]

]

{ #category : 'Accessing' }
GsEnumeratedPathTerm >> _nextObjectFor: anObject pathName: aPathName [
  "Returns the object at the instance variable that corresponds to the receiver
 path term."

  | ivOffset |
  ivOffset := self _ivOffsetFor: anObject pathName: aPathName.
  ivOffset ifNil: [ ^ nil ].
  ^ self _nextObjectFor: anObject atInstVar: ivOffset

]

{ #category : 'Updating Indexes' }
GsEnumeratedPathTerm >> addMappingsForObject: anObject root: rootObject logging: aBoolean [
  "Add dependency list entries for anObject."

  | ivOffset sz |
  (nil == anObject or: [ self size == 0 ])
    ifTrue: [ ^ self ].
  self pathTermElements
    do: [ :pathTermElement | (self _ivOffsetFor: anObject pathName: pathTermElement)
        ifNil: [ self _invalidIvOffset: anObject ]
        ifNotNil: [ :off | ivOffset := off ] ].
  anObject
    getDepListAndAddPathTerm: self
    withIVOffset: ivOffset
    logging: aBoolean.
	" add an entry to the objects dependency list ... note that we ar only using the last ivOffset and that is okay as we compensate in the places that the ivOffset is used..."
  sz := children size.
  self nextObj: anObject do: [ :nextObj |
      updateBtree ~~ nil
        ifTrue: [
          " insert an entry into the btree "
          (self _checkBtreeComparisonWith: nextObj)
            ifFalse: [
              ImproperOperation new
                _number:
                    (ErrorSymbols at: #'rtErrRangeEqualityIndexInvalidClassKindForBtree');
                args: { nextObj class. self name. updateBtree lastElementClassDescription };
                signal ].
          updateBtree btreeAt: nextObj put: anObject root: rootObject.
          needsDepList
            ifTrue: [
              " add an entry in the dependency list "
              nextObj getDepListAndAddLastElementPathTerm: self logging: false ] ].
      nil == nextObj
        ifTrue: [ self recordNilOnPathForRoot: rootObject ]
        ifFalse: [
          1 to: sz do: [ :i | " make recursive call to add mappings "
            (children at: i)
              addMappingsForObject: nextObj
              root: rootObject
              logging: aBoolean ] ] ]

]

{ #category : 'Audit' }
GsEnumeratedPathTerm >> auditDepListFor: obj index: indexObj using: auditor optionalSentinel: optionalSentinel [
  "Private."

  | depList j ivOffset |
  (nil == obj or: [ self size == 0 ])
    ifTrue: [ ^ nil ].
  depList := DependencyList for: obj.
  depList == nil
    ifTrue: [
      obj isInvariant
        ifFalse: [ auditor pathTermObjectHasNoDependencyList: self object: obj indexObj: indexObj depList: depList ] ]
    ifFalse: [
      (DepListTable _hasDependencyBitFor: obj)
        ifFalse: [ auditor pathTermObjectHasNoDependencyBitSet: self object: obj indexObj: indexObj depList: depList ].
      j := depList _findOffsetForPathTerm: self.
      j == nil
        ifTrue: [ auditor dependencyListHasNoEntryForPathTerm: self object: obj indexObj: indexObj depList: depList ]
        ifFalse: [
          "found entry for self in depList"
          ivOffset := depList at: j + 1.	"may be negative for last element reference count"
          ivOffset < 0
            ifTrue: [
              "reference count for last element, but we don't expect reference
               count here ... see needsDepList clause below where depList
               existence is sufficient"
              auditor dependencyListUnexpectedLastElementLocation: self object: obj indexObj: indexObj depList: depList ivOffset: ivOffset.
              ^ self ] ] ].
  needsDepList
    ifTrue: [
      self
        nextObj: obj
        do: [ :nextObj |
          nextObj ifNotNil: [
            (DependencyList needsDepList: nextObj)
              ifTrue: [
                depList := DependencyList for: nextObj.
                depList == nil
                  ifTrue: [
                    nextObj isInvariant
                      ifFalse: [ auditor pathTermObjectHasNoDependencyList: self object: nextObj indexObj: indexObj depList: depList ] ] ] ] ] ].
  ^ self

]

{ #category : 'Audit' }
GsEnumeratedPathTerm >> auditNscCountsFor: obj using: auditor count: btreeCounts [
  "Private. "

  | sz |
  sz := self size.
  (nil == obj _or: [ sz == 0 ])
    ifTrue: [ ^ self ].
  self
    auditDepListFor: obj
    index: nil
    using: auditor
    optionalSentinel: Object new.
  self nextObj: obj do: [ :nextObj | | index |
      index := 1.
      children isEmpty
        ifTrue: [ self auditDirectNscCountsFor: nextObj using: auditor count: btreeCounts ]
        ifFalse: [ 1 to: sz do: [ :i | | count indexObj |
            indexObj := self at: i.
            indexObj size == offset
              ifTrue: [ count := btreeCounts at: index.
                count == 0
                  ifTrue: [ auditor pathTermIncorrectNumberOfBtreeEntries: self index: i offset: offset ].
                btreeCounts at: index put: count - 1.
                index := index + 1 ] ].
          nextObj ~~ nil
            ifTrue: [ 1 to: children size do: [ :i | (children at: i)
                  auditNscCountsFor: nextObj
                  using: auditor
                  count: (btreeCounts at: index).
                index := index + 1 ] ] ] ]

]

{ #category : 'Audit' }
GsEnumeratedPathTerm >> auditUpdateCacheFor: anObject root: root nilOnPathRoots: nilOnPathRoots occurrences: num using: auditor [
  "verify proper accounting for objects with nil on path"

  self nextObj: anObject do: [ :nextObj |
     (nil == nextObj and: [ updateBtree isNil ])
      ifTrue: [
        | nilOnPathNum |
        nilOnPathNum := nilOnPath asIdentitySet occurrencesOf: root.
        nilOnPathNum = num
          ifFalse: [ auditor pathTermIncorrectNilOnPathCount: self root: root got: nilOnPathNum  expected: num ] ]
      ifFalse: [
        1 to: children size do: [ :i |
          (children at: i)
            auditUpdateCacheFor: nextObj
            root: root
            nilOnPathRoots: nilOnPathRoots
            occurrences: num
            using: auditor ] ] ]

]

{ #category : 'Updating Indexes' }
GsEnumeratedPathTerm >> cleanupDependencyListFor: anObject [
  ""

  | deplistOffset nextObj depList |
  (nil == anObject or: [ self size == 0 ])
    ifTrue: [ ^ self ].
  (depList := DependencyList for: anObject) == nil
    ifTrue: [ ^ self ].
  deplistOffset := depList removeCompletelyPathTerm: self for: anObject.	" remove the receiver from the dependency list "
  (self indicatesIndexOnRootNsc
    or: [
      "last element dependency list"
      deplistOffset < 0 ])
    ifTrue: [ ^ self ].
  self pathTermElements
    do: [ :pathName |
      (self _ivOffsetFor: anObject pathName: pathName)
        ifNotNil: [ :ivOffset |
          nextObj := self _nextObjectFor: anObject atInstVar: ivOffset.
          nextObj
            ifNotNil: [
              needsDepList
                ifTrue: [
                  (depList := DependencyList for: nextObj) ~~ nil
                    ifTrue: [ depList removeCompletelyPathTerm: self for: nextObj ] ].
              1 to: children size do: [ :i | (children at: i) cleanupDependencyListFor: nextObj ] ] ] ]

]

{ #category : 'Testing' }
GsEnumeratedPathTerm >> coversIvOffset: anIvOffset for: anObject [
  "Answer true if the receiver covers the given iv offset (see bug 46705)"

  self pathTermElements
    do: [ :pathTermElement |
      (self _ivOffsetFor: anObject pathName: pathTermElement)
        ifNil: [ self _invalidIvOffset: anObject ]
        ifNotNil: [ :off | anIvOffset = off ifTrue: [ ^true ] ] ].
  ^ false

]

{ #category : 'Modification' }
GsEnumeratedPathTerm >> findReachableRootObjectMapsFor: anObject ivOffset: ivOffset oldValue: oldValue [

  | indexObj factor theFactor rootObjectMap traversalRootObjects traversalReachableRoots |
  rootObjectMap := BtreePlusRootObjectMap new
    pivotObject: anObject;
    pivotPathTerm: self;
    yourself.
  self findReachableRootObjectMaps: rootObjectMap for: anObject ivOffset: ivOffset oldValue: oldValue sanity: false.
  traversalRootObjects := rootObjectMap resolveTraversalRootObjectsFor: oldValue.
  indexObj := self at: self size.
factor := rootObjectMap cumulativeFactor.
  (traversalRootObjects includes: anObject)
    ifTrue: [ theFactor := (traversalRootObjects occurrencesOf: anObject) ]
    ifFalse: [
      | enumeratedValues |
      enumeratedValues := IdentityBag new.
      self nextObj: anObject do: [:nextObj |
        enumeratedValues add: nextObj].
      theFactor := (enumeratedValues occurrencesOf: oldValue) ].
  theFactor ~~ 1
    ifTrue: [ factor := factor + theFactor ].
factor == 0 ifTrue: [ ^ traversalRootObjects ].
  traversalReachableRoots := IdentityBag new.
  traversalRootObjects asIdentitySet do: [:rootObject |
    | num |
    theFactor := factor + (indexObj cumulativeFactorFor: rootObject upTo: self startingAt: 1).
    num := traversalRootObjects occurrencesOf: rootObject.
    traversalReachableRoots add: rootObject withOccurrences: (num / theFactor) ceiling ].
  ^ traversalReachableRoots

]

{ #category : 'Modification' }
GsEnumeratedPathTerm >> findReachableRootsFor: anObject ivOffset: ivOffset oldValue: oldValue [

  | rootObjects indexObj reachableRoots factor |
  rootObjects := super findReachableRootsFor: anObject ivOffset: ivOffset oldValue: oldValue.
  indexObj := self at: self size.
  (rootObjects includes: anObject)
    ifTrue: [ factor := rootObjects occurrencesOf: anObject ]
    ifFalse: [
      | enumeratedValues |
      enumeratedValues := IdentityBag new.
      self nextObj: anObject do: [:nextObj |
        enumeratedValues add: nextObj].
      factor := (enumeratedValues occurrencesOf: oldValue) ].
  factor == 1
    ifTrue: [ ^ rootObjects ].
  reachableRoots := IdentityBag new.
  rootObjects asIdentitySet do: [:rootObject |
    | num theFactor |
    theFactor := factor + (indexObj cumulativeFactorFor: rootObject upTo: self startingAt: 1).
    num := rootObjects occurrencesOf: rootObject.
    reachableRoots add: rootObject withOccurrences: (num / theFactor) ceiling ].
  ^ reachableRoots

]

{ #category : 'Collection Based Modification' }
GsEnumeratedPathTerm >> findRootObjectMaps: rootObjectMap for: anObject [
  "Traverse btree nodes from receiver to leaf term. Collect rootObjects for all btree elements
   for which anObject is a key or parent of a key."

  (rootObjectMap traversalMap includesKey: anObject)
    ifTrue: [
      "only way for the key to be present, is that to have come through this pathTerm/object before.
       And that can happen if one or more of the enumerated terms has identical values"
      ^ self  ].
  self pathTermElements
    do: [ :pathName |
      "Make a full pass for each of the enumerated instance variables"
      | ivOffset |
      ivOffset := self _ivOffsetFor: anObject pathName: pathName.
      ivOffset ifNil: [ self _invalidIvOffset: anObject ].
      self findRootObjectMaps: rootObjectMap for: anObject ivOffset: ivOffset]

]

{ #category : 'Collection Based Modification' }
GsEnumeratedPathTerm >> findRootObjectMaps: rootObjectMap for: anObject ivOffset: ivOffset [

  (rootObjectMap traversalMap includesKey: anObject)
    ifTrue: [
      "only way for the key to be present, is that to have come through this pathTerm/object before.
       And that can happen if one or more of the enumerated terms has identical values"
      ^ self  ].
  super findRootObjectMaps: rootObjectMap for: anObject ivOffset: ivOffset

]

{ #category : 'Collection Based Modification' }
GsEnumeratedPathTerm >> findRootObjectMaps: rootObjectMap for: anObject ivOffset: ivOffset oldValue: oldValue [

  (rootObjectMap traversalMap includesKey: anObject)
    ifTrue: [
      "only way for the key to be present, is that to have come through this pathTerm/object before.
       And that can happen if one or more of the enumerated terms has identical values"
      ^ self  ].
  super findRootObjectMaps: rootObjectMap for: anObject ivOffset: ivOffset oldValue: oldValue

]

{ #category : 'Collection Based Modification' }
GsEnumeratedPathTerm >> findRootObjectMaps: rootObjectMap for: anObject oldValue: oldValue [
  "Traverse btree nodes from receiver to leaf term. Collect rootObjects for all btree elements
   for which anObject is a key or parent of a key."
| enumeratedValues factor |
  enumeratedValues := IdentityBag new.
  self nextObj: anObject do: [:nextObj |
    enumeratedValues add: nextObj].
factor := 0.
  self pathTermElements
    do: [ :pathName |
      "Make a full pass for each of the enumerated instance variables"
      | ivOffset nextOldValue |
      ivOffset := self _ivOffsetFor: anObject pathName: pathName.
      ivOffset ifNil: [ self _invalidIvOffset: anObject ].
      nextOldValue := self _nextObjectFor: oldValue atInstVar: ivOffset.
      factor := factor + (enumeratedValues occurrencesOf: nextOldValue)].
factor > 1 ifTrue: [
rootObjectMap cumulativeFactor: (factor + (rootObjectMap cumulativeFactor))].
  (rootObjectMap traversalMap includesKey: anObject)
    ifTrue: [
      "only way for the key to be present, is that to have come through this pathTerm/object before.
       And that can happen if one or more of the enumerated terms has identical values"
      ^ self  ].
  self pathTermElements
    do: [ :pathName |
      "Make a full pass for each of the enumerated instance variables"
      | ivOffset nextOldValue |
      ivOffset := self _ivOffsetFor: anObject pathName: pathName.
      ivOffset ifNil: [ self _invalidIvOffset: anObject ].
      nextOldValue := self _nextObjectFor: oldValue atInstVar: ivOffset.
      self
         findRootObjectMaps: rootObjectMap
         for: anObject
         ivOffset: ivOffset
         oldValue: nextOldValue ].

]

{ #category : 'Collection Based Modification' }
GsEnumeratedPathTerm >> findRootObjectsFor: anObject [
  "Traverse btree nodes from receiver to leaf term. Collect rootObjects for all btree elements
   for which anObject is a key or parent of a key."

  | rootObjects |
  rootObjects := IdentityBag new.
  self pathTermElements
    do: [ :pathName |
      "Make a full pass for each of the enumerated instance variables"
      | roots ivOffset |
      ivOffset := self _ivOffsetFor: anObject pathName: pathName.
      ivOffset ifNil: [ self _invalidIvOffset: anObject ].
      roots := self findRootObjectsFor: anObject ivOffset: ivOffset.
      rootObjects := rootObjects _union: roots ].
  ^ rootObjects

]

{ #category : 'Collection Based Modification' }
GsEnumeratedPathTerm >> findRootObjectsFor: anObject oldValue: oldValue [
  "Traverse btree nodes from receiver to leaf term. Collect rootObjects for all btree elements
   for which anObject is a key or parent of a key."

  | rootObjects factor enumeratedValues reachableRoots |
  rootObjects := IdentityBag new.
  enumeratedValues := IdentityBag new.
  self nextObj: anObject do: [:nextObj |
    enumeratedValues add: nextObj].
  factor := 0.
  self pathTermElements
    do: [ :pathName |
      "Make a full pass for each of the enumerated instance variables"
      | roots ivOffset nextOldValue |
      ivOffset := self _ivOffsetFor: anObject pathName: pathName.
      ivOffset ifNil: [ self _invalidIvOffset: anObject ].
      nextOldValue := self _nextObjectFor: oldValue atInstVar: ivOffset.
      factor := factor + (enumeratedValues occurrencesOf: nextOldValue).
      roots := self
        findRootObjectsFor: anObject
        ivOffset: ivOffset
        oldValue: nextOldValue.
      rootObjects := rootObjects _union: roots ].
  factor == 1 ifTrue: [ ^ rootObjects ].
  reachableRoots := IdentityBag new.
  rootObjects asIdentitySet do: [:rootObject |
    | num |
    num := rootObjects occurrencesOf: rootObject.
    reachableRoots add: rootObject withOccurrences: (num / factor) ceiling ].
  ^ reachableRoots

]

{ #category : 'Testing' }
GsEnumeratedPathTerm >> hasMatchingTerm: aSymbol [
  "Answer true if the receiver's term matches <aSymbol>"

  ^ self name == (self _canonicalizeTerm: aSymbol)

]

{ #category : 'Testing' }
GsEnumeratedPathTerm >> indicatesIndexOnNscElements [
  "Returns false. The ides is not on elements of an nsc."

  ^ false

]

{ #category : 'Testing' }
GsEnumeratedPathTerm >> isEnumeratedTerm [
  "Returns true if the receiver indicates an enumerated path term."

  ^ true

]

{ #category : 'Accessing' }
GsEnumeratedPathTerm >> nextObj: anObj do: aBlock [
  | found |
  (nil == anObj or: [ self size == 0 ])
    ifTrue: [ ^ self ].
  found := false.
  self pathTermElements
    do: [ :pathName |
      | ivOffset |
      (ivOffset := self _ivOffsetFor: anObj pathName: pathName) ~~ nil
        ifTrue: [
          | nextObj |
          found := true.
          nextObj := self _nextObjectFor: anObj atInstVar: ivOffset.
          aBlock cull: nextObj cull: ivOffset ] ].
  found
    ifFalse: [ ^ self _invalidIvOffset: anObj ]

]

{ #category : 'Accessing' }
GsEnumeratedPathTerm >> pathTermElements [
  "return the individual pathTerm elements"

  ^ self name subStringsDelimitedBy: $|

]

{ #category : 'Removing' }
GsEnumeratedPathTerm >> removeMappingsFor: anObject root: rootObject lastOne: aBoolean logging: doLogging [
  "Remove entries in the btree and dependency lists for anObject."

  | nextObj depList isLast |
  self size == 0
    ifTrue: [ ^ self ].
  self pathTermElements
    do: [ :pathName |
      nextObj := self _nextObjectFor: anObject pathName: pathName.
      updateBtree ~~ nil
        ifTrue: [
          " remove an entry from the B-tree "
          (updateBtree btreeRemoveKey: nextObj value: anObject root: rootObject)
            ifFalse: [ self _errorObjectNotInBtree: nextObj value: anObject root: rootObject ].
          DependencyList removePathTerm: self for: nextObj logging: doLogging	" remove dependency list entry for next object " ].
      nextObj ifNil: [ self removeNilOnPathForRoot: rootObject ].
      (nil ~~ nextObj and: [ children isEmpty not ])
        ifTrue: [
          aBoolean
            ifTrue: [
              | last |
              last := (children at: 1) _isLastOccurrenceInIndexFor: nextObj.
              isLast := aBoolean and: [ last ] ]
            ifFalse: [ isLast := false ].
          " make recursive call to remove mappings "
          1 to: children size do: [ :i | (children at: i) removeMappingsFor: nextObj root: rootObject lastOne: isLast logging: doLogging ].
          isLast
            ifFalse: [
              "check again in case there are no more references left in btree"
              isLast := ((children at: 1) findReachableRootsFor: nextObj) isEmpty ].
          isLast
            ifTrue: [
              " remove dependency list entries for next object "
              (depList := DependencyList for: nextObj) ~~ nil
                ifTrue: [ depList removePathTerms: children for: nextObj logging: doLogging ] ] ] ]

]

{ #category : 'Modification' }
GsEnumeratedPathTerm >> traverseAllReachableFor: root indexObj: indexObj [

  ^ (indexObj traverseAllWithParents: root upTo: self startingAt: 1) collect: [:each | each at: 2]

]
