Extension { #name : 'GsIndexSpec' }

{ #category : 'Private' }
GsIndexSpec >> _optimizedCharacterCollectionIndex: path constraintType: constraintType collator: collatorOrNil options: aGsIndexOptions [
  "Specify that an equality index on the given path using the given index options be created.
     The btreePlusIndex and optimizedComparison options will be unconditionally set in the
     index options.

   If the collatorOrNil is nil and unicode comparison mode is in effect (see
     CharacterCollection class>>isInUnicodeComparisonMode), the default collator will be used
     for comparison operations in the index.

   If the collatorOrNil is nil and unicode comparison mode is NOT in effect (see
     CharacterCollection class>>isInUnicodeComparisonMode), then the native String
     comparison operators (#<, #<=, #=, #~, #>, #>=, #==, #~~) will be used for
     comparison operations in the index.

   If the collatorOrNil is not nil, the given collator will be used for comparison operations
     in the index, whether or not unicode comparison mode is in effect (see CharacterCollection
     class>>isInUnicodeComparisonMode).
"

  "constraintType may be: #unicodeString, #symbol, or #string:

     If #unicodeString, then the elements of the collection will be constrained to any type of
     String (String, Unicode7, DoubleByteString, Unicode16, QuadByteString, Unicode32) and a
     collator will be used for comparison.

     If #symbol, then the elements of the collection will be constrained to be a type of
     Symbol (Symbol, DoubleByteSymbol, or QuadByteSymbol).

     If #string, then the elements of the collection will be constrained based on whether
       or not unicode comparison mode is in effect or not (see CharacterCollection
       class>>isInUnicodeComparisonMode).

       When unicode comparison mode is in effect the elements of the collection will be
         constrained to be any type of String (String, Unicode7, DoubleByteString, Unicode16,
         QuadByteString,  Unicode32).

       When unicode comparison mode is NOT in effect the elements of the collection will be
         constrained to be a type of non-unicode String (String, DoubleByteString,
         QuadByteString).
"

  | options spec |
  options := aGsIndexOptions + GsIndexOptions btreePlusIndex + GsIndexOptions optimizedComparison.
  spec := constraintType == #unicodeString
    ifTrue: [ | collator |
      collator := collatorOrNil ifNil: [ self defaultCollator ].
      (UnicodeIndexSpecification
        path: (self preparedPath: path)
        collator: collator)
          constraintType: constraintType; "#unicodeString"
          yourself]
    ifFalse: [ | theCollator |
      theCollator := collatorOrNil
        ifNil: [
          CharacterCollection isInUnicodeComparisonMode
            ifTrue: [
              "in unicode comparison mode, we always have a collator"
              self defaultCollator]
            ifFalse: [ nil ] ]
        ifNotNil: [ collatorOrNil ].
      (ConstrainedCharacterCollectionIndexSpecification
        path: (self preparedPath: path)
        collator: theCollator)
          constraintType: constraintType;
          yourself ].
  spec options: options.
  self specs add: spec

]

{ #category : 'Index Creation' }
GsIndexSpec >> createIndexesOn: anNsc [
  self specs do: [ :indexSpec | indexSpec createIndexOn: anNsc ]

]

{ #category : 'Private' }
GsIndexSpec >> defaultCollator [
  ^ IcuCollator default

]

{ #category : 'Private' }
GsIndexSpec >> defaultOptions [
  defaultOptions ifNil: [ defaultOptions := GsIndexOptions default ].
  ^ defaultOptions

]

{ #category : 'Specification' }
GsIndexSpec >> equalityIndex: path lastElementClass: aClass [
  "Specify that  an equality index on the given path and last element class, using default
   index options be created."

  self equalityIndex: path lastElementClass: aClass options: self defaultOptions

]

{ #category : 'Specification' }
GsIndexSpec >> equalityIndex: path lastElementClass: aClass options: aGsIndexOptions [
  "Specify that an equality index with the given path, last element class and index options
   be created.

   If the last element class is a unicode class (one of the Unicode* classes or any
     CharacterCollection when using unicode comparison mode), then create a unicode
     equality index using the default collator.

   If the reducedConflict option is specified, create a reduced conflict equality index.

   For btree plus indexes, optimized comparisons will not be used. If optimized comparisons is
     desired, use #stringOptimizedIndex:.
"

  | specClass options |
  (RangeEqualityIndex _isUnicodeLastElementClass: aClass)
    ifTrue: [ ^ self unicodeIndex: path collator: self defaultCollator options: aGsIndexOptions ].
  options := (aClass == CharacterCollection or: [aClass isSubclassOf: CharacterCollection])
    ifTrue: [ aGsIndexOptions - GsIndexOptions optimizedComparison ]
    ifFalse: [ aGsIndexOptions ].
  specClass := options reducedConflict
    ifTrue: [ RcEqualityIndexSpecification ]
    ifFalse: [ EqualityIndexSpecification ].
  self specs
    add: ((specClass path: (self preparedPath: path) lastElementClass: aClass)
        options: options;
        yourself)

]

{ #category : 'Specification' }
GsIndexSpec >> identityIndex: path [
  "Specify that  an identiy index on the given path, using default index options be created."

  self identityIndex: path options: self defaultOptions

]

{ #category : 'Specification' }
GsIndexSpec >> identityIndex: path options: aGsIndexOptions [
  "Specify that an identity index with the given pathand index options be created."

  | options |
  options := aGsIndexOptions.
  (options legacyIndex )
    ifTrue: [
      options reducedConflict
        ifTrue: [ Warning signal: 'The reducedConflict option has no effect for an identity index' ] ]
    ifFalse: [
      "new indexes need to have reducedConflict option explicitly set"
      options := options + GsIndexOptions reducedConflict].
  self specs add: ((IdentityIndexSpecification path: (self preparedPath: path))
        options: options;
        yourself)

]

{ #category : 'Private' }
GsIndexSpec >> preparedPath: aPath [
  "strip each from <aPath>"

  | pathTerms stream |
  pathTerms := aPath asArrayOf32PathTerms.
  pathTerms first = #'each'
    ifFalse: [ ^ aPath ].
  stream := String new.
  2 to: pathTerms size do: [ :index |
    stream addAll: (pathTerms at: index).
    index < pathTerms size
      ifTrue: [ stream add: $. ] ].
  ^ stream

]

{ #category : 'Printing' }
GsIndexSpec >> printOn: aStream [
  aStream
    nextPutAll: self class name asString , ' new';
    lf.
  1 to: self specs size do: [ :index |
    | spec |
    spec := self specs at: index.
    aStream tab.
    spec printSelectorOn: aStream.
    aStream
      nextPut: $;;
      lf ].
  aStream
    tab;
    nextPutAll: 'yourself.'

]

{ #category : 'Index Creation' }
GsIndexSpec >> removeIndexesFrom: anNsc [
  self specs do: [ :indexSpec | indexSpec removeIndexFrom: anNsc ]

]

{ #category : 'Private' }
GsIndexSpec >> specs [
  specs ifNil: [ specs := Array new ].
  ^ specs

]

{ #category : 'Specification' }
GsIndexSpec >> stringOptimizedIndex: path [
  "Specify that an equality index on the given path be created. The btreePlusIndex and
     optimizedComparison options will be unconditionally set in the index options.

   When unicode comparison mode is in effect (see CharacterCollection
     class>>isInUnicodeComparisonMode), the elements of the collection will be constrained to
     be any type of String (String, Unicode7, DoubleByteString, Unicode16, QuadByteString,
     Unicode32). The current default collator will be used for comparison operations in the
     index.

   When unicode comparison mode is NOT in effect (see CharacterCollection
     class>>isInUnicodeComparisonMode), the elements of the collection will be constrained to
     be a type of non-unicode String (String, DoubleByteString, QuadByteString). The
     native String comparison operators (#<, #<=, #=, #~, #>, #>=, #==, #~~) will be used for
     comparison operations in the index.
"

  self
    stringOptimizedIndex: path
    options: self defaultOptions

]

{ #category : 'Specification' }
GsIndexSpec >> stringOptimizedIndex: path options: aGsIndexOptions [
  "Specify that an equality index on the given path using the given index options be created.
     The btreePlusIndex and optimizedComparison options will be unconditionally set in the
     index options.

   When unicode comparison mode is in effect (see CharacterCollection
     class>>isInUnicodeComparisonMode), the elements of the collection will be constrained to
     be any type of String (String, Unicode7, DoubleByteString, Unicode16, QuadByteString,
     Unicode32). The current default collator will be used for comparison operations in the
     index.

   When unicode comparison mode is NOT in effect (see CharacterCollection
     class>>isInUnicodeComparisonMode), the elements of the collection will be constrained to
     be a type of non-unicode String (String, DoubleByteString, QuadByteString). The
     native String comparison operators (#<, #<=, #=, #~, #>, #>=, #==, #~~) will be used
     for comparison operations in the index.
"

  self
    _optimizedCharacterCollectionIndex: path
    constraintType: #string
    collator: nil
    options: aGsIndexOptions

]

{ #category : 'Specification' }
GsIndexSpec >> symbolOptimizedIndex: path [
  "Specify that an equality index on the given path be created. The btreePlusIndex and
     optimizedComparison options will be unconditionally set in the index options.

   When unicode comparison mode is in effect (see CharacterCollection
     class>>isInUnicodeComparisonMode), the elements of the collection will be constrained to
     be any type of Symbol (Symbol, DoubleByteSymbol, QuadByteSymbol). The current default
     collator will be used for comparison operations in the index.

   When unicode comparison mode is NOT in effect (see CharacterCollection
     class>>isInUnicodeComparisonMode), the elements of the collection will be constrained to
     be a type of Symbol (Symbol, DoubleByteSymbol, QuadByteSymbol). The native String
     comparison operators (#<, #<=, #=, #~, #>, #>=, #==, #~~) will be used for
     comparison operations in the index.
"

  self
    symbolOptimizedIndex: path
    options: self defaultOptions

]

{ #category : 'Specification' }
GsIndexSpec >> symbolOptimizedIndex: path collator: icuCollator [
  "Specify that an equality index on the given path be created. The btreePlusIndex and
     optimizedComparison options will be unconditionally set in the index options.

   Whether or not unicode comparison mode is in effect (see CharacterCollection
     class>>isInUnicodeComparisonMode), the elements of the collection will be constrained to
     be any type of Symbol (Symbol, DoubleByteSymbol, QuadByteSymbol). The given
     collator will be used for comparison operations in the index.
"

  self
    symbolOptimizedIndex: path
    collator: icuCollator
    options: self defaultOptions

]

{ #category : 'Specification' }
GsIndexSpec >> symbolOptimizedIndex: path collator: icuCollator options: aGsIndexOptions [
  "Specify that an equality index on the given path using the given index options be created.
     The btreePlusIndex and optimizedComparison options will be unconditionally set in the
     index options.

   Whether or not unicode comparison mode is in effect (see CharacterCollection
     class>>isInUnicodeComparisonMode), the elements of the collection will be constrained to
     be any type of Symbol (Symbol, DoubleByteSymbol, QuadByteSymbol). The given
     collator will be used for comparison operations in the index.
"

  self
    _optimizedCharacterCollectionIndex: path
    constraintType: #symbol
    collator: icuCollator
    options: aGsIndexOptions

]

{ #category : 'Specification' }
GsIndexSpec >> symbolOptimizedIndex: path options: aGsIndexOptions [
  "Specify that an equality index on the given path using the given index options be created.
     The btreePlusIndex and optimizedComparison options will be unconditionally set in the
     index options.

   When unicode comparison mode is in effect (see CharacterCollection
     class>>isInUnicodeComparisonMode), the elements of the collection will be constrained to
     be any type of Symbol (Symbol, DoubleByteSymbol, QuadByteSymbol). The current default
     collator will be used for comparison operations in the index.

   When unicode comparison mode is NOT in effect (see CharacterCollection
     class>>isInUnicodeComparisonMode), the elements of the collection will be constrained to
     be a type of Symbol (Symbol, DoubleByteSymbol, QuadByteSymbol). The native String
     comparison operators (#<, #<=, #=, #~, #>, #>=, #==, #~~) for comparison
     operations in the index.
"

  self
    _optimizedCharacterCollectionIndex: path
    constraintType: #symbol
    collator: nil
    options: aGsIndexOptions

]

{ #category : 'Specification' }
GsIndexSpec >> unicodeIndex: path [
  "Specify that a unicode equality index on the given path using the default collator and default
    index options be created.

   When using string comparison mode, a unicode equality index is constrained to the unicode
    string classes Unicode7, Unicode16, and Unicode32.

   When using unicode comparison mode, a unicode equality index is constrained to be a type
     of CharacterCollection.

   For btree plus indexes, optimized comparisons will not be used. If optimized comparisons is
     desired, use either #unicodeStringOptimizedIndex: or #symbolOptimizedIndex:collator:.
"

  self
    unicodeIndex: path
    collator: self defaultCollator
    options: self defaultOptions

]

{ #category : 'Specification' }
GsIndexSpec >> unicodeIndex: path collator: collator [
  "Specify that a unicode equality index on the given path using the given collator and default
    index options be created.

   When using string comparison mode, a unicode equality index is constrained to the unicode
    string classes Unicode7, Unicode16, and Unicode32.

   When using unicode comparison mode, a unicode equality index is constrained to be a type
     of CharacterCollection.

   For btree plus indexes, optimized comparisons will not be used. If optimized comparisons is
     desired, use either #unicodeStringOptimizedIndex: or #symbolOptimizedIndex:collator:.
"

  self unicodeIndex: path collator: collator options: self defaultOptions

]

{ #category : 'Specification' }
GsIndexSpec >> unicodeIndex: path collator: collator options: aGsIndexOptions [
  "Specify that a unicode equality index on the given path using the given collator and given
    index options be created.

   When using string comparison mode, a unicode equality index is constrained to the unicode
    string classes Unicode7, Unicode16, and Unicode32.

   When using unicode comparison mode, a unicode equality index is constrained to be a type
     of CharacterCollection.

   For btree plus indexes, optimized comparisons will not be used. If optimized comparisons is
     desired, use either #unicodeStringOptimizedIndex: or #symbolOptimizedIndex:collator:.
"

  | options specClass spec |
  options := aGsIndexOptions.
  specClass := UnicodeIndexSpecification.
  options := options - GsIndexOptions optimizedComparison.
  options legacyIndex
    ifTrue: [
      options reducedConflict
        ifTrue: [ specClass := RcUnicodeIndexSpecification ] ].
 spec := (specClass path: (self preparedPath: path) collator: collator)
    options: options;
    yourself.
  self specs add: spec

]

{ #category : 'Specification' }
GsIndexSpec >> unicodeIndex: path options: aGsIndexOptions [
  "Specify that a unicode equality index on the given path using the default collator and given
    index options be created.

   When using string comparison mode, a unicode equality index is constrained to the unicode
    string classes Unicode7, Unicode16, and Unicode32.

   When using unicode comparison mode, a unicode equality index is constrained to be a type
     of CharacterCollection.

   For btree plus indexes, optimized comparisons will not be used. If optimized comparisons is
     desired, use either #unicodeStringOptimizedIndex: or #symbolOptimizedIndex:collator:.
"

  self unicodeIndex: path collator: self defaultCollator options: aGsIndexOptions

]

{ #category : 'Specification' }
GsIndexSpec >> unicodeStringOptimizedIndex: path [
  "Specify that an equality index on the given path be created using the default collator and
     default index options. The btreePlusIndex and optimizedComparison options will be
     unconditionally set in the index options.

   Whether or not unicode comparison mode is in effect (see CharacterCollection
     class>>isInUnicodeComparisonMode), the elements of the collection will be constrained to
     be any type of String (String, Unicode7, DoubleByteString, Unicode16, QuadByteString,
     Unicode32). The current default collator will be used for comparison operations in the
     index.
"

  self
    unicodeStringOptimizedIndex: path
    collator: self defaultCollator
    options: self defaultOptions

]

{ #category : 'Specification' }
GsIndexSpec >> unicodeStringOptimizedIndex: path collator: collator [
  "Specify that an equality index on the given path be created using the given collator and
     the default index options. The btreePlusIndex and optimizedComparison options will be
     unconditionally set in the index options.

   Whether or not unicode comparison mode is in effect (see CharacterCollection
     class>>isInUnicodeComparisonMode), the elements of the collection will be constrained to
     be any type of String (String, Unicode7, DoubleByteString, Unicode16, QuadByteString,
     Unicode32). The current default collator will be used for comparison operations in the
     index.
"

  self
    unicodeStringOptimizedIndex: path
    collator: collator
    options: self defaultOptions

]

{ #category : 'Specification' }
GsIndexSpec >> unicodeStringOptimizedIndex: path collator: collator options: aGsIndexOptions [
  "Specify that an equality index on the given path be created using the given collator and
     the given index options. The btreePlusIndex and optimizedComparison options will be
     unconditionally set in the index options.

   Whether or not unicode comparison mode is in effect (see CharacterCollection
     class>>isInUnicodeComparisonMode), the elements of the collection will be constrained to
     be any type of String (String, Unicode7, DoubleByteString, Unicode16, QuadByteString,
     Unicode32). The current default collator will be used for comparison operations in the
     index.
"

  self
    _optimizedCharacterCollectionIndex: path
    constraintType: #'unicodeString'
    collator: collator
    options: aGsIndexOptions

]

{ #category : 'Specification' }
GsIndexSpec >> unicodeStringOptimizedIndex: path options: aGsIndexOptions [
  "Specify that an equality index on the given path be created using the default collator and
     the given index options. The btreePlusIndex and optimizedComparison options will be
     unconditionally set in the index options.

   Whether or not unicode comparison mode is in effect (see CharacterCollection
     class>>isInUnicodeComparisonMode), the elements of the collection will be constrained to
     be any type of String (String, Unicode7, DoubleByteString, Unicode16, QuadByteString,
     Unicode32). The current default collator will be used for comparison operations in the
     index.
"

  self
    unicodeStringOptimizedIndex: path
    collator: self defaultCollator
    options: aGsIndexOptions

]
