Class {
	#name : 'GsPackageLibrary',
	#superclass : 'Object',
	#category : nil
}

{ #category : 'Operations' }
GsPackageLibrary class >> createPackageNamed: aSymbol [

    | package packageLibrary |
    package := self packageNamed: aSymbol.
    package ~~ nil ifTrue: [ ^package ].
    packageLibrary := self packageLibrary.
    packageLibrary createDictionaryNamed: aSymbol at: 1.
    ^GsPackage installIn: (packageLibrary objectNamed: aSymbol).

]

{ #category : 'Private - Installation Support' }
GsPackageLibrary class >> getMonticelloRepositoryDirectory [
| basePath |
basePath := self getSeasideDirectory .
basePath ==  nil
  ifTrue:[ ^nil ] .
basePath addAll: 'monticello/repository/' .
^ basePath

]

{ #category : 'Private - Installation Support' }
GsPackageLibrary class >> getSeasideDirectory [
| basePath |
basePath := System gemEnvironmentVariable: 'SEASIDE'.
basePath == nil
 ifTrue:[
   basePath := System gemEnvironmentVariable: 'GEMSTONE'.
   basePath == nil
     ifTrue:[^nil]
     ifFalse:[basePath addAll: '/seaside/'].
  ]
  ifFalse:[
    basePath last == $/
      ifFalse: [ basePath add: $/ ].
  ].
^basePath

]

{ #category : 'private' }
GsPackageLibrary class >> installMonticelloPackagesDuring: aBlock [
        "install the libraries #'SessionMethods' and #'Monticell_Methods'.
         Enable #'SessionMethods' and configure for new classes to be installed into UserGlobals.
         Evaluate <aBlock> with the SymbolDictionary named #'Monticello_Globals' as the argument.
         Uninstall the library #'Monticell_Methods', but leave #'SessionMethods' enabled"

        ^self
            installMonticelloPackagesHome: (GsCurrentSession currentSession objectNamed: #UserGlobals)
            during: aBlock

]

{ #category : 'private' }
GsPackageLibrary class >> installMonticelloPackagesHome: homeSymbolDict during: aBlock [
        "install the libraries #'SessionMethods' and #'Monticell_Methods'.
         Enable #'SessionMethods' and configure for new classes to be installed into <homeSymbolDict>.
         Evaluate <aBlock> with the SymbolDictionary named #'Monticello_Globals' as the argument.
         Uninstall the library #'Monticell_Methods', but leave #'SessionMethods' enabled"

	| policy methodsPackage sessionMethodsPackage preInstalled enabled |
	methodsPackage := self packageNamed: #'Monticello_Methods'.
	sessionMethodsPackage := GsPackageLibrary createPackageNamed: #SessionMethods.
	preInstalled := (self userSymbolList objectNamed: methodsPackage symbolDict name) ~~ nil.
	enabled := GsPackagePolicy current enabled.

	[
	preInstalled
		ifTrue: [ enabled ifFalse: [ GsPackagePolicy current enable ] ]
		ifFalse: [
			self installPackage: sessionMethodsPackage.
			self installPackage: methodsPackage ].

	policy := GsPackagePolicy current.
	policy homeSymbolDict:  homeSymbolDict.
	policy externalSymbolList: Array new.

	aBlock value: (self packageNamed: #'Monticello_Globals') ]
		ensure: [
			preInstalled ifFalse: [ self uninstallPackage: methodsPackage ].
			enabled ifFalse: [ GsPackagePolicy current disable ] ].


]

{ #category : 'Operations' }
GsPackageLibrary class >> installPackage: aPackage [

    	self installPackage: aPackage enable: true

]

{ #category : 'Operations' }
GsPackageLibrary class >> installPackage: aPackage enable: aBool [

    	| symDict pkgDict |
    	symDict := aPackage symbolDict.
	pkgDict := Globals.
    	aPackage prereqs do: [:symbolDict |  | p |
     		p := symbolDict at: aPackage class globalName otherwise: nil.
        	p ~~ nil
			ifTrue: [ self installPackage: p  enable: false ].
		(self userSymbolList indexOf: symbolDict) < (self userSymbolList indexOf: pkgDict)
			ifTrue: [ pkgDict := p symbolDict ].
    	].
	(self userSymbolList objectNamed: symDict name) == nil
		ifTrue: [ self userSymbolList add: symDict before: pkgDict ].
	(self sessionSymbolList objectNamed: symDict name) == nil
		ifTrue: [ self  sessionSymbolList add: symDict before: pkgDict ].
	aBool
		ifTrue: [ | policy |
			policy := GsPackagePolicy current.
			policy enable.
			policy homeSymbolDict:  aPackage symbolDict.
			policy externalSymbolList: (self sessionSymbolList asArray copyWithout: aPackage symbolDict).
		].

]

{ #category : 'Operations' }
GsPackageLibrary class >> installPackageNamed: aSymbol [
  | pkg |
  (pkg := self packageNamed: aSymbol) ifNil:[
    Error signal:'Package ', aSymbol asString,' not found'.
  ].
  ^ self installPackage: pkg

]

{ #category : 'Testing' }
GsPackageLibrary class >> isPackageInstalled: aPackage [
	"Ansser true if <aPackage> has already been installed"

	^(self userSymbolList objectNamed: aPackage symbolDict name) ~~ nil

]

{ #category : 'Monticello' }
GsPackageLibrary class >> lastFileNameOf: packageName inDirectory: directory [
	"returns a String"

	| versionNumberBlock mczFilename |
	versionNumberBlock := self versionNumberBlock.
	mczFilename := (((directory childFiles
				collect: [:reference | reference basename])
					select: [:mczName | (mczName indexOfSubCollection: packageName) = 1])
					sortBy: [:a :b | (versionNumberBlock value: a) <= (versionNumberBlock value: b)])
					last.
	^mczFilename

]

{ #category : 'Monticello' }
GsPackageLibrary class >> lastFileNameOf: packageName inPath: aPath [
  "returns a String"
  self installMonticelloPackagesDuring: [:globalsPackage |
   | directory |
    directory := ((globalsPackage symbolDict at: #FSDiskFilesystem) current
                     pathFromString: aPath ) asReference.
   ^ self lastFileNameOf: packageName inDirectory: directory
  ].

]

{ #category : 'Monticello' }
GsPackageLibrary class >> loadLastVersionOf: packageName fromRepositoryPath: repositoryPath [
	"load the latest version of the package <packageName> into the current user's UserGlobals symbol dictionary from
	the repository located in the directory <repositoryPath> on the server."

	self installMonticelloPackagesDuring: [:globalsPackage |
    | directory repository mczFilename |
		directory := ((globalsPackage symbolDict at: #FSDiskFilesystem) current
										pathFromString: repositoryPath) asReference.
		mczFilename := self lastFileNameOf: packageName inDirectory: directory.

		repository := (globalsPackage symbolDict at: #MCFilesystemRepository) new directory: directory.
		self loadMCZFile: mczFilename fromRepository: repository
  ]

]

{ #category : 'private' }
GsPackageLibrary class >> loadMCZFile: mczFilename fromRepository: repository [
	"load the given <mczFilename> from the given GsMonticello repository <repository>"

        GsFile gciLogServer: 'loading ', mczFilename printString.
	(repository loadVersionFromFileNamed: mczFilename) load.

]

{ #category : 'Monticello' }
GsPackageLibrary class >> loadMCZFile: mczFilename fromRepositoryPath: repositoryPath [
	"load the given mcz file <mczFilename> into the current user's UserGlobals symbol dictionary from
	the repository located in the directory <repositoryPath> on the server."

	self
            loadMCZFile: mczFilename
            home: #UserGlobals
            fromRepositoryPath: repositoryPath

]

{ #category : 'Monticello' }
GsPackageLibrary class >> loadMCZFile: mczFilename home: homeSymbolDictName fromRepositoryPath: repositoryPath [
    "load the given mcz file <mczFilename> into the current user'ssymbol
     dictionary named <homeSymbolDictName> from
     the repository located in the directory <repositoryPath> on the server."

    self
      installMonticelloPackagesHome: (GsCurrentSession currentSession objectNamed: homeSymbolDictName asSymbol)
      during: [:globalsPackage | |repository |
		repository := (globalsPackage symbolDict at: #MCFilesystemRepository) new
			directory: ((globalsPackage symbolDict at: #FSDiskFilesystem) current pathFromString: repositoryPath) asReference.
		self loadMCZFile: mczFilename fromRepository: repository ].

]

{ #category : 'Accessing' }
GsPackageLibrary class >> packageLibrary [

    | packageLibrary |
    packageLibrary := (AllUsers userWithId: self packageLibraryUserName ifAbsent: []) symbolList objectNamed: self packageLibraryName.
    packageLibrary == nil
        ifTrue: [
            packageLibrary := SymbolList new.
            self userGlobals at: self packageLibraryName put: packageLibrary.
        ].
    ^packageLibrary

]

{ #category : 'Accessing' }
GsPackageLibrary class >> packageLibraryName [

    ^#PackageLibrary

]

{ #category : 'Accessing' }
GsPackageLibrary class >> packageLibraryUserName [

    ^'DataCurator'

]

{ #category : 'Searching' }
GsPackageLibrary class >> packageNamed: aSymbol [

    	| symDict |
	symDict := self packageLibrary objectNamed: aSymbol.
	symDict ifNil:[ ^ nil ].
	^symDict at: GsPackage globalName otherwise: nil

]

{ #category : 'Reporting' }
GsPackageLibrary class >> packagesReport [
  | aSymList str |
  str := String new .
  aSymList := self packageLibrary .
  aSymList do:[:aDict |
    (aDict at: GsPackage globalName otherwise: nil) ifNotNil:[:pack |
       str add: 'GsPackage oop ' , pack asOop asString .
       pack name ifNotNil:[:nam|  str add: ' for ' , nam ].
       str lf .
    ].
  ].
  ^ str

]

{ #category : 'Monticello' }
GsPackageLibrary class >> saveNewVersionOf: packageName author: authorInitials message: commitComment
  toPath: repositoryPath [
  "write a new version of the package <packageName>
   to the repository located in the directory <repositoryPath> on the server."
	self installMonticelloPackagesDuring:
		[:globalsPackage |
		| directory repository pkg wc version verName |
		directory := ((globalsPackage symbolDict at: #FSDiskFilesystem) current
					pathFromString: repositoryPath) asReference.
		repository := (globalsPackage symbolDict at: #MCFilesystemRepository) new
					directory: directory.
		(globalsPackage symbolDict at: #MCPlatform) current authorInitials: authorInitials.
		pkg := (globalsPackage symbolDict at: #MCPackage) named: packageName.
		wc := pkg workingCopy.
		version := wc newVersionWithName: (verName := wc uniqueVersionName)
					message: commitComment.
		repository storeVersion: version.
		System commitTransaction ifFalse: [Error signal: 'commit failed'].
		^'wrote ' , verName asString , ' , ' , version summary]

]

{ #category : 'Accessing' }
GsPackageLibrary class >> sessionSymbolList [

  ^ GsSession currentSession symbolList      "fix 49328"

]

{ #category : 'Operations' }
GsPackageLibrary class >> uninstallPackage: aPackage [

    self userSymbolList remove: aPackage symbolDict ifAbsent: [].
    self sessionSymbolList remove: aPackage symbolDict ifAbsent: [].
    GsPackagePolicy current refreshSessionMethodDictionary.

]

{ #category : 'Operations' }
GsPackageLibrary class >> uninstallPackageAndPrereqs: aPackage [

    "Recursively uninstall the package and its prereqs"

    | symDict pkg |
    symDict := aPackage symbolDict.
    aPackage prereqs do: [:symbolDict |
        pkg := symbolDict at: GsPackage globalName otherwise: nil.
        pkg ~~ nil ifTrue: [ GsPackageLibrary uninstallPackageAndPrereqs: pkg ].
    ].
    self userSymbolList remove: symDict ifAbsent: [].
    self sessionSymbolList remove: symDict ifAbsent: [].

]

{ #category : 'Accessing' }
GsPackageLibrary class >> userGlobals [

    ^self userSymbolList objectNamed: #UserGlobals

]

{ #category : 'Accessing' }
GsPackageLibrary class >> userSymbolList [

    ^System myUserProfile symbolList

]

{ #category : 'private' }
GsPackageLibrary class >> versionNumberBlock [
	"Private. Version number is apparently the portion of the name between the last two periods.
	If no periods, the entire name. If one period, everything before that period."

	^
	[:aName |
	| lastIndex numberPortion foundPeriod index |
	foundPeriod := false.
	index := aName size.
	[foundPeriod or: [index = 0]] whileFalse:
			[(aName at: index = $.) ifTrue: [foundPeriod := true].
			index := index - 1].
	numberPortion := foundPeriod
				ifTrue:
					[lastIndex := index.
					foundPeriod := false.
					[foundPeriod or: [index = 0]] whileFalse:
							[(aName at: index = $.) ifTrue: [foundPeriod := true].
							index := index - 1].
					aName copyFrom: index + 1 to: lastIndex]
				ifFalse: [aName].
	numberPortion asNumber]

]
