Extension { #name : 'GsPathTerm' }

{ #category : 'Instance Creation' }
GsPathTerm class >> new [

"Create an initialized instance of the receiver."

^ self basicNew initialize

]

{ #category : 'Updating' }
GsPathTerm >> _addChildTerm: aPathTerm [
  children addLast: aPathTerm

]

{ #category : 'Testing' }
GsPathTerm >> _checkBtreeComparisonWith: anObject [

"Returns whether the receiver's B-tree can hold an entry with anObject as the
 key."

^ updateBtree _canCompareWith: anObject

]

{ #category : 'Accessing' }
GsPathTerm >> _classOf: anObject [
  "Reflection classOf:"

  <primitive: 166>
  self _primitiveFailed: #'_classOf:' args: {anObject}.
  self _uncontinuableError

]

{ #category : 'Updating Indexes' }
GsPathTerm >> _clear [


"Assigns nil to important instance variables and sets the receiver's size
 to zero."

children := nil.
updateBtree := nil.
self size: 0.

]

{ #category : 'Updating' }
GsPathTerm >> _determineChildren [
  "Determine the children of the receiver by getting the next path term for each
 index object that utilizes the receiver path term."

  | indexObj |
  children size: 0.
  1 to: self size do: [ :i |
    indexObj := self at: i.
    (indexObj size > offset
      and: [ (children includesIdentical: (indexObj at: offset + 1)) not ])
      ifTrue: [ self _addChildTerm: (indexObj at: offset + 1) ] ]

]

{ #category : 'Index Maintenance' }
GsPathTerm >> _doAddMappingsFor: anNSC on: indexObj indexManager: indexManager hasSet: hasSet [
  "Adds the mappings in the index dictionary, dependency lists, and so on for all
 elements in the receiver."

  | bag |
  bag := anNSC _asIdentityBag.
  hasSet
    ifTrue: [
      | updateIndexList objNsc incomplete |
      " if there is a set-valued instance variable, collect objects in a
      Bag on which to update the index dictionary "
      updateIndexList := false.
      (self == indexObj lastPathTerm
        _and: [ self size > 1 _and: [ (self at: 1) size ~= (self at: 2) size ] ])
        ifTrue: [
          " see if index is subsumed by an existing index "
          " if last path term is #*, will need to update the
          last objects' index list "
          self indicatesMultiValue
            ifTrue: [
              updateIndexList := true.
              objNsc := NscBuilder
                for: IdentityBag new
                max: indexObj sizeForNscBuilder ]
            ifFalse: [ objNsc := nil ] ]
        ifFalse: [ objNsc := NscBuilder for: IdentityBag new max: indexObj sizeForNscBuilder ].
      incomplete := #'_incompletePathTraversal'.
       1 to: bag size do: [ :i |
        " for each element in the NSC ... "
        " traverse up to the point of the unshared path term "
        indexObj
          traverse: (bag _at: i)
          upTo: self
          startingAt: 1
          addingIndex: true
          do: [:endObj :rootObj |
            objNsc ~~ nil
              ifTrue: [
                updateIndexList
                  ifTrue: [ self addIndexListEntriesFor: objNsc completeBag index: indexObj ]
                  ifFalse: [
                    endObj ~~ incomplete
                      ifTrue: [
                        " only remove mappings once for a given object "
                        self
                          addMappingsForObject: endObj
                          root: rootObj
                          logging: false ] ].
                i \\ 100 == 0
                  ifTrue: [ indexManager commitIndexMaintenance: indexObj at: i ] ] ] ] ]
    ifFalse: [
      | object |
      "no set-valued terms"
      indexManager autoCommit
        ifTrue: [ anNSC _lockForIndexCreation ].
      indexObj firstPathTerm == self
        ifTrue: [
          " see if need to traverse to the first unshared path term "
          1 to: bag size do: [ :i |
            object := bag _at: i.
            self addMappingsForObject: object logging: false.
            i \\ 100 == 0
              ifTrue: [ indexManager commitIndexMaintenance: indexObj at: i ] ] ]
        ifFalse: [
          1 to: bag size do: [ :i |
            " traverse up to the point of the unshared path term "
            object := bag _at: i.
            self addMappingsForObject: object traverseUpTo: self for: indexObj logging: false.
            i \\ 100 == 0
              ifTrue: [ indexManager commitIndexMaintenance: indexObj at: i ] ] ] ].
  ^ anNSC

]

{ #category : 'Testing' }
GsPathTerm >> _doNotPerformPretraversalFor: anObject [
  "Returns whether to do a preparatory traversal (using MappingInfo objects)
 when anObject is being added to an indexed Bag or IdentityBag."

  ^ true

]

{ #category : 'Index Maintenance' }
GsPathTerm >> _doRemoveIndex: indexObj nsc: nsc [
      "Remove the index associated with the receiver from the nsc."

      | bag incomplete indexManager allTerms |
      indexManager := IndexManager current.
      bag := nsc _asIdentityBag.
      incomplete := #'_incompletePathTraversal'.
      indexObj _findSetValuedPathTermOffset ~~ 0
        ifTrue: [
          | updateIndexList objNsc |
          "See if there is a set-valued instance variable before the unshared path
     term."
          updateIndexList := false.
          (self == indexObj lastPathTerm
            _and: [ self size > 1 _and: [ (self at: 1) size ~= (self at: 2) size ] ])
            ifTrue: [
              " see if index is subsumed by an existing index "
              " if last path term is #*, will need to update the
              last objects' index list "
              self indicatesMultiValue
                ifTrue: [
                  updateIndexList := true.
                  objNsc := NscBuilder
                    for: IdentityBag new
                    max: indexObj sizeForNscBuilder ]
                ifFalse: [ objNsc := nil ] ]
            ifFalse: [ objNsc := NscBuilder for: IdentityBag new max: indexObj sizeForNscBuilder ].
          1 to: bag size do: [ :i |
            " for each element in the NSC ... "
            " traverse up to the point of the unshared path term "
            indexObj
              traverse: (bag _at: i)
              upTo: self
              startingAt: 1
              addingIndex: false
              do: [:endObj :rootObj |
                objNsc ~~ nil
                  ifTrue: [
                    updateIndexList
                      ifTrue: [ self removeIndexListEntriesFor: objNsc completeBag index: indexObj ]
                      ifFalse: [
                        endObj ~~ incomplete
                          ifTrue: [
                            " only remove mappings once for a given object "
                            self
                              removeMappingsFor: endObj
                              root: rootObj
                              lastOne: true
                              logging: false.
                            DependencyList removePathTerm: self for: endObj ] ] ] ] ] ]
        ifFalse: [ | obj |
          self indicatesIndexOnRootNsc
            ifTrue: [
              " see if the index is on elements of the NSC "
              1 to: bag size do: [ :j |
                obj := bag _at: j.
                self removeDirectMappingFor: obj logging: false.
                DependencyList removePathTerm: self for: obj.
                indexManager commitIndexMaintenance: indexObj at: j ] ]
            ifFalse: [
              [ | endObj |
              1 to: bag size do: [ :j |
                obj := bag _at: j.
                endObj := indexObj traverse: obj upTo: self.
                (nil ~~ endObj and: [ incomplete ~~ endObj ])
                  ifTrue: [
                    self removeMappingsSkipBtreeFor: endObj root: obj lastOne: true logging: false.
                    DependencyList removePathTerm: self for: endObj.
                    indexManager commitIndexMaintenance: indexObj at: j ] ] ]
                onException: Error
                do: [ :ex |
                  ex number == (ErrorSymbols at: #'rtErrObjectPathTermNotInDependencyList')
                    ifTrue: [
                      "continue execution"
                      ex resume ]
                    ifFalse: [ ex outer ] ] ].
          allTerms := self _thisAndAllChildTerms ].
      ^ allTerms

]

{ #category : 'Private' }
GsPathTerm >> _errorInvalidOffset: anObject [
  "If missing path slots are tolerated, return an empty MappingInfo. Otherwise
   return an error array"

  (LookupError new
    _number:(ErrorSymbols at: #'rtErrObjectInvalidOffset');
    args: {anObject. name printString}) signal

]

{ #category : 'Error Handling' }
GsPathTerm >> _errorObjectNotInBtree: key value: value root: root [
  "An entry for the key/value/root was not present in the B-tree."
  
  System gemConfigurationAt:#GemExceptionSignalCapturesStack put: true .
  ImproperOperation new
    args:
        {key.
          value.
          root};
    _number: (ErrorSymbols at: #'rtErrRangeEqualityIndexObjectNotInBtree');
    signal:
        'An entry for the key/value/root tuple (' , key asOop printString , '/'
            , value asOop printString , '/' , root asOop printString
            , ') was not present in the index [', updateBtree asOop printString, ']'

]

{ #category : 'Traversing' }
GsPathTerm >> _getNextObjectForTraversal: anObject [
  " get the next object along the path "

  | ivOffset nextObj |
  ivOffset := anObject _findIVOffsetForPathTerm: self.
  ivOffset == nil
    ifTrue: [
      anObject _errorPathTermNotInDependencyList: self.
      nextObj := nil ]
    ifFalse: [ nextObj := anObject instVarAt: ivOffset ].
  ^ nextObj

]

{ #category : 'Become Support' }
GsPathTerm >> _getPathTermIndexReferencesInto: refsToRcvr for: anObject [
  "Place information about references to the receiver due to the receiver's
 participation in an index into the given Array.  The Array consists of pairs:

 1) An object that references the receiver.  If the object is directly
    contained in an indexed NSC, then this object is the NSC.  If the object
    in the dependency list is a tracking object (for object modification
    tracking), then this object is the tracking object.
 2) The offset of the instance variable in that object that references the
    receiver.
 3) If the object is the last path term, this number is < 0 and is a reference
    count for the number of times .
 3) If the object in the dependency list is a tracking
    object, this number is 0.

 This is used only in support of the 'become:' method."

  | indexObj rootObjects parents prevPathTerm |
  self size = 0
    ifTrue: [ ^ self ].
  indexObj := self at: 1.
  self offset == 1
    ifTrue: [
      "anObject is the first path term in an index (which means anObject
       is directly contained in an indexed NSC) "
      anObject
        _addToReferences: indexObj nscRoot
        offset: 0
        occurrences: (indexObj nscRoot occurrencesOf: anObject)
        into: refsToRcvr.
      ^ self ].
  rootObjects := self findRootObjectsFor: anObject.
  parents := IdentityBag new.
  rootObjects asIdentitySet do: [:root |
    (indexObj traverseAllWithParents: root upTo: self startingAt: 1)
      do: [ :ar | | intermediateObjectReachableFromRoot parentObject |
        parentObject := ar at: 1.
        intermediateObjectReachableFromRoot := ar at: 2.
        intermediateObjectReachableFromRoot == anObject
          ifTrue: [ parents add: parentObject withOccurrences: (rootObjects occurrencesOf: root) ] ] ].
  " get the previous path term in the index "
  prevPathTerm := indexObj at: self offset - 1.
  parents do: [:parentObj |
    prevPathTerm indicatesMultiValue
      ifTrue: [
        anObject
          _addToReferences: parentObj
          offset: 0
          occurrences: (parentObj occurrencesOf: anObject)
          into: refsToRcvr ]
      ifFalse: [
        anObject
          _addToReferences: parentObj
          offset: (prevPathTerm _ivOffsetFor: parentObj)
          occurrences: 1
          into: refsToRcvr ] ]

]

{ #category : 'Private' }
GsPathTerm >> _invalidIvOffset: anObject [
  "If missing path slots are tolerated, return. Otherwise signal an error"

  self termsRequired
    ifFalse: [ ^ self ].
  anObject _errorInvalidOffset: name.
  self _uncontinuableError

]

{ #category : 'Testing' }
GsPathTerm >> _isLastOccurrenceInIndexFor: anObject [

  ^ (self findReachableRootsFor: anObject) size <= 1

]

{ #category : 'Testing' }
GsPathTerm >> _isObsoletePathTerm [

"Returns whether the receiver is obsolete."

^ self size == 0

]

{ #category : 'Accessing' }
GsPathTerm >> _ivOffsetFor: anObject [

"Returns the instance variable offset of anObject for the instance variable
 corresponding to the receiver path term."

^ (self _classOf: anObject) _idxIvOffsetOf: name

]

{ #category : 'Private' }
GsPathTerm >> _makeObsolete [

"Makes the receiver and all its children obsolete."



1 to: children size do: [ :i | (children at: i) _makeObsolete ].
self _clear.

]

{ #category : 'Accessing' }
GsPathTerm >> _nextObjectFor: anObject [

"Returns the object at the instance variable that corresponds to the receiver
 path term."


^ self _nextObjectFor: anObject atInstVar: (self _ivOffsetFor: anObject)

]

{ #category : 'Accessing' }
GsPathTerm >> _nextObjectFor: anObject atInstVar: ivOffset [

"Returns the object at the instance variable that corresponds to the receiver
 path term."

<primitive: 167>
self _primitiveFailed: #_nextObjectFor:atInstVar:
     args: { anObject . ivOffset } .
self _uncontinuableError

]

{ #category : 'Reduced Conflict Support' }
GsPathTerm >> _selectiveAbort [

 "Do nothing"

]

{ #category : 'Accessing' }
GsPathTerm >> _thisAndAllChildTerms [

"Returns an Array containing the receiver and all child path terms."


^ self _thisAndAllChildTermsInto: { }

]

{ #category : 'Accessing' }
GsPathTerm >> _thisAndAllChildTermsInto: array [

"Returns an Array containing the receiver and all child path terms."


array add: self.

1 to: children size do: [ :i |
  (children at: i) _thisAndAllChildTermsInto: array
].

^ array

]

{ #category : 'Testing' }
GsPathTerm >> _totalNumSetValuedTerms [
  ""

  | total |
  self indicatesNsc
    ifTrue: [ total := 1 ]
    ifFalse: [ total := 0 ].
  1 to: children size do: [ :i | total := total + (children at: i) _totalNumSetValuedTerms ].
  ^ total

]

{ #category : 'Adding' }
GsPathTerm >> addDirectMappingFor: anObject root: rootObject nsc: anNsc logging: aBoolean [
  " Update the B-tree and the dependency list for anObject."

  updateBtree ~~ nil
    ifTrue: [ updateBtree btreeAt: anObject put: anObject root: rootObject ].
  needsDepList
    ifTrue: [ " add an entry in the dependency list "
      anObject getDepListAndAddLastElementPathTerm: self logging: aBoolean ]

]

{ #category : 'Adding' }
GsPathTerm >> addMappingsForObject: anObject at: ivOffset root: rootObject logging: aBoolean [
  ^self addMappingsForObject: anObject root: rootObject logging: aBoolean

]

{ #category : 'Adding' }
GsPathTerm >> addMappingsForObject: anObject logging: aBoolean [
  self addMappingsForObject: anObject root: anObject logging: aBoolean

]

{ #category : 'Adding' }
GsPathTerm >> addMappingsForObject: anObject root: rootObject logging: aBoolean [
  "Add dependency list entries for anObject."

  | ivOffset nextObj |
  (nil == anObject or: [ self size == 0 ])
    ifTrue: [ ^ self ].
  (ivOffset := self _ivOffsetFor: anObject) == nil
    ifTrue: [ ^ self _errorInvalidOffset: anObject ].
  anObject
    getDepListAndAddPathTerm: self
    withIVOffset: ivOffset
    logging: aBoolean.	" add an entry to the objects dependency list "
  nextObj := self _nextObjectFor: anObject atInstVar: ivOffset.	" get the next object along the path "
  updateBtree ~~ nil
    ifTrue: [ " insert an entry into the B-tree "
      (self _checkBtreeComparisonWith: nextObj)
        ifFalse: [
          ImproperOperation new
            _number:
                (ErrorSymbols at: #'rtErrRangeEqualityIndexInvalidClassKindForBtree');
            args: { nextObj class. self name. updateBtree lastElementClassDescription };
            signal ].
      updateBtree btreeAt: nextObj put: anObject root: rootObject.
      needsDepList
        ifTrue: [ " add an entry to the very last object's dependency list "
          nextObj getDepListAndAddLastElementPathTerm: self logging: aBoolean ] ].
  nil == nextObj
    ifTrue: [ ^ self recordNilOnPathForRoot: rootObject].
  1 to: children size do: [ :i | " make recursive call to add mappings "
    (children at: i)
      addMappingsForObject: nextObj
      root: rootObject
      logging: aBoolean ]

]

{ #category : 'Adding' }
GsPathTerm >> addMappingsForObject: rootObject traverseUpTo: aPathTerm for: anIndexObject logging: aBoolean [
  | object |
  object := anIndexObject traverse: rootObject upTo: aPathTerm.
  self addMappingsForObject: object root: rootObject logging: aBoolean

]

{ #category : 'Formatting' }
GsPathTerm >> asString [
  ^ super asString , ' on ' , self name asString printString

]

{ #category : 'Audit' }
GsPathTerm >> auditDepListFor: obj index: indexObj using: auditor optionalSentinel: optionalSentinel [
  "Private."

  | depList j nextObj ivOffset |
  (nil == obj or: [ self size == 0 ])
    ifTrue: [ ^ self ].
  depList := DependencyList for: obj.
  depList == nil
    ifTrue: [
      (obj isInvariant not and: [ self isOptionalTerm not ])
        ifTrue: [ auditor pathTermObjectHasNoDependencyList: self object: obj indexObj: indexObj depList: depList ].
      nextObj := self _nextObjectFor: obj ]
    ifFalse: [
      (DepListTable _hasDependencyBitFor: obj)
        ifFalse: [ auditor pathTermObjectHasNoDependencyBitSet: self object: obj indexObj: indexObj depList: depList ].
      j := depList _findOffsetForPathTerm: self.
      j == nil
        ifTrue: [
          | ivOff |
          "no entry for self in depList"
          ivOff := self _ivOffsetFor: obj.
          (self isOptionalTerm and: [ ivOff isNil ])
            ifTrue: [
              "no dependencyList if there's no slot"
              nextObj := optionalSentinel ]
            ifFalse: [ auditor dependencyListHasNoEntryForPathTerm: self object: obj indexObj: indexObj depList: depList ] ]
        ifFalse: [
          "found entry for self in depList"
          ivOffset := depList at: j + 1.	"may be negative for last element reference count"
          ivOffset < 0
            ifTrue: [
              "reference count for last element, but we don't expect reference
               count here ... see needsDepList clause below where depList
               existence is sufficient"
              auditor dependencyListUnexpectedLastElementLocation: self object: obj indexObj: indexObj depList: depList ivOffset: ivOffset.
              ^ obj ]
            ifFalse: [ | expectedIvOffset |
              ivOffset ~~ (expectedIvOffset := self _ivOffsetFor: obj)
                ifTrue: [
                  auditor dependencyListHasIncorrectIvOffset: self object: obj indexObj: indexObj depList: depList ivOffset: ivOffset expectedIvOffset: expectedIvOffset.
                  ivOffset := expectedIvOffset "allow audit to continue, despite fatal nature of this situation" ].
              nextObj := obj instVarAt: ivOffset ] ] ].
  ((needsDepList and: [ nextObj ~~ nil ])
    and: [ DependencyList needsDepList: nextObj ])
    ifTrue: [
      depList := DependencyList for: nextObj.
      (depList == nil and: [ self isOptionalTerm not ])
        ifTrue: [ auditor pathTermObjectHasNoDependencyList: self object: obj indexObj: indexObj depList: depList ] ].
  ^ nextObj

]

{ #category : 'Audit' }
GsPathTerm >> auditDepListForLastElementObject: obj occurrences: num index: indexObj using: auditor [
  | depList j |
  (self size == 0 or: [ obj isInvariant ])
    ifTrue: [ ^ self ].
  depList := DependencyList for: obj.
  depList == nil
    ifTrue: [ auditor
        pathTermObjectHasNoDependencyList: self
        object: obj
        indexObj: indexObj
        depList: depList ]
    ifFalse: [ (DepListTable _hasDependencyBitFor: obj)
        ifFalse: [ auditor
            pathTermObjectHasNoDependencyBitSet: self
            object: obj
            indexObj: indexObj
            depList: depList ].
      j := depList _findOffsetForPathTerm: self.
      j == nil
        ifTrue: [ "hack to avoid false positives for index paths ending in *"
          self name = #'*'
            ifFalse: [ auditor
                dependencyListHasNoEntryForPathTerm: self
                object: obj
                indexObj: indexObj
                depList: depList ] ]
        ifFalse: [ | referenceCount nonReferenceCountTerm |
          referenceCount := depList at: j + 1.
          nonReferenceCountTerm := false.
          referenceCount > 0
            ifTrue: [ "upgraded indexes from pre-3.2 may have positive rerence count, but only for Byte indexable ojbects"
              obj class isBytes
                ifFalse: [ "ivOffset and not a reference count"
                  nonReferenceCountTerm := true ] ]
            ifFalse: [ referenceCount := referenceCount negated ].
          nonReferenceCountTerm
            ifTrue: [ "not a reference count and will be audited by PathTerm>>auditDepListFor:index:on:optionalSentinel: "
              "noop"
               ]
            ifFalse: [ referenceCount == num
                ifFalse: [ auditor
                    dependencyListHasIncorrectRefCount: self
                    object: obj
                    indexObj: indexObj
                    depList: depList
                    referenceCount: referenceCount
                    numOccurrences: num ] ] ] ]

]

{ #category : 'Audit' }
GsPathTerm >> auditDirectNscCountsFor: obj using: auditor count: btreeCounts [
  "Private. "

  | index |
  self size == 0
    ifTrue: [ ^ self ].
  (needsDepList and: [ nil ~~ obj ])
    ifTrue: [ | depList |
      depList := DependencyList for: obj.
      (depList == nil and: [ obj isInvariant not ])
        ifTrue: [ auditor pathTermObjectHasNoDependencyList: self object: obj indexObj: nil depList: depList ]
        ifFalse: [ obj isInvariant not
            ifTrue: [ (DepListTable _hasDependencyBitFor: obj)
                ifFalse: [ auditor pathTermObjectHasNoDependencyBitSet: self object: obj indexObj: nil depList: depList ] ] ] ].
  index := 1.
  1 to: self size do: [ :i | | count indexObj |
    indexObj := self at: i.
    indexObj size == offset
      ifTrue: [ count := btreeCounts at: index.
        count == 0
          ifTrue: [ auditor pathTermIncorrectNumberOfBtreeEntries: self index: i offset: offset ].
        btreeCounts at: index put: count - 1.
        index := index + 1 ] ]

]

{ #category : 'Audit' }
GsPathTerm >> auditInfo: array using: auditor [
  auditor auditInfoFor: self info: array

]

{ #category : 'Audit' }
GsPathTerm >> auditNsc: nsc on: aString level: level [
  "Verifies that the equality index objects are consistent.
 Returns a string that describes any inconsistencies found."

  | indexObj btreeCounts count |
  "aString add: 'entering obj, num is ', num printString; lf."
  self size == 0
    ifTrue: [ ^ aString ].	" for each index that utilizes the path term "
  level == 1 ifTrue: [ self auditUpdateCache: nsc on: aString ].
  btreeCounts := {}.
  1 to: self size do: [ :i | indexObj := self at: i.
    indexObj size == offset
      ifTrue: [ " verify B-tree has correct entries "
        count := indexObj btreeRoot
          auditNsc: nsc
          for: self
          offset: i
          on: aString.
        btreeCounts add: count ] ].
  1 to: children size do: [ :i | | resultArray |
    resultArray := (children at: i) auditNsc: nsc on: aString level: level + 1.
    btreeCounts add: resultArray ].
  ^ btreeCounts

]

{ #category : 'Audit' }
GsPathTerm >> auditNscCountsFor: obj on: aString count: btreeCounts [
  "Private. "

  | auditor |
  auditor := BtreePlusNodeAuditor new auditResultString: aString; yourself.
  true
    ifTrue:[ self auditNscCountsFor: obj using: auditor count: btreeCounts ]
    ifFalse: [
      "use this audit to provide more detailed information about btree element count audit failures"
      "not fully tested, but it is know to provide better information audit errors with the
       BtreePlusNodeAuditor>>pathTermIncorrectNumberOfBtreeEntries:index:offset: message."

      "bypass call to _auditBtreeCounts:on: in UnorderedCollection>>_fastAuditEqualityIndexes as well"
      self auditNscForRootObj: obj rootObj: obj using: auditor ].
  ^ auditor auditResultString

]

{ #category : 'Audit' }
GsPathTerm >> auditNscCountsFor: obj using: auditor count: btreeCounts [
  "Private. "

  | nextObj index optionalSentinel |
  self size == 0
    ifTrue: [ ^ self ].
  self indicatesIndexOnNscElements
    ifTrue: [ ^ self auditDirectNscCountsFor: obj using: auditor count: btreeCounts ].
  nil == obj
    ifTrue: [ ^ self ].
  optionalSentinel := Object new.
  nextObj := self
    auditDepListFor: obj
    index: nil
    using: auditor
    optionalSentinel: optionalSentinel.
  nextObj == optionalSentinel
    ifTrue: [ ^ self ].
  index := 1.
  1 to: self size do: [ :i | | count indexObj |
    indexObj := self at: i.
    indexObj size == offset
      ifTrue: [ count := btreeCounts at: index.
        (count == 0 and: [ self isOptionalTerm not ])
          ifTrue: [ "an index with optional path terms may have 0 entries"
             auditor pathTermIncorrectNumberOfBtreeEntries: self index: i offset: offset ].
        (count == 0 and: [ self isOptionalTerm ])
          ifFalse: [ btreeCounts at: index put: count - 1 ].
        index := index + 1 ] ].
  1 to: children size do: [ :i | (children at: i)
      auditNscCountsFor: nextObj
      using: auditor
      count: (btreeCounts at: index).
    index := index + 1 ]

]

{ #category : 'Audit' }
GsPathTerm >> auditNscForRootObj: obj rootObj: rootObj using: auditor [
  "Private. "

  | nextObj optionalSentinel |
  self size == 0
    ifTrue: [ ^ self ].
  self indicatesIndexOnNscElements
    ifTrue: [ "^ self auditDirectNscRootObj: obj using: auditor" self error: 'not yet implemented' ].
  nil == obj
    ifTrue: [ ^ self ].
  optionalSentinel := Object new.
  nextObj := self
    auditDepListFor: obj
    index: nil
    using: auditor
    optionalSentinel: optionalSentinel.
  (nextObj == optionalSentinel or: [ nextObj == nil ])
    ifTrue: [ ^ self ].
  1 to: self size do: [ :i | | indexObj |
    indexObj := self at: i.
    indexObj size == offset
      ifTrue: [ | col found expected |
        col := IdentityBag new.
        indexObj btreeRoot btreeRootNode _findAndCountKey: nextObj value: obj root: rootObj into: col.
        (found := col size) == (expected := indexObj nscRoot occurrencesOf: rootObj)
          ifFalse: [
            auditor
              btree: indexObj btreeRoot
              incorrectCountForKeyValueRootPairs: self
              key: nextObj
              value: obj
              root: rootObj
              found: found
              expected: expected ] ] ].
  1 to: children size do: [ :i | (children at: i)
      auditNscForRootObj: nextObj
      rootObj: rootObj
      using: auditor ]

]

{ #category : 'Audit' }
GsPathTerm >> auditUpdateCache: nsc on: aString [
  "Verifies that the update caches (nilOnPath) are consistent.
 Returns a string that describes any inconsistencies found."

  | auditor prevObj |
  auditor := BtreePlusNodeAuditor new auditResultString: aString; yourself.
  prevObj := #_incompletePathTraversal.
  nsc do: [ :obj |
    obj ~~ prevObj
      ifTrue: [
        self auditUpdateCacheFor: obj
          root: obj
          nilOnPathRoots: nil
          occurrences: (nsc occurrencesOf: obj)
          using: auditor ].
    prevObj := obj ].

]

{ #category : 'Audit' }
GsPathTerm >> auditUpdateCacheFor: anObject root: root nilOnPathRoots: nilOnPathRoots occurrences: num using: auditor [
  "verify proper accounting for objects with nil on path"

  | ivOffset nextObj |
  self indicatesIndexOnRootNsc
    ifTrue: [ ^ self ].
  (ivOffset := self _ivOffsetFor: anObject) == nil
    ifTrue: [
      self isOptionalTerm ifTrue: [ ^ self ].
      auditor pathTermInvalidIvOffset: self ivOffset: ivOffset object: anObject ].
  nextObj := self _nextObjectFor: anObject atInstVar: ivOffset.
  (nil == nextObj and: [ updateBtree isNil ])
    ifTrue: [
      | nilOnPathNum |
      nilOnPathNum := nilOnPath occurrencesOf: root.
      nilOnPathRoots
        ifNil: [
          nilOnPathNum = num
           ifFalse: [ auditor pathTermIncorrectNilOnPathCount: self root: root got: nilOnPathNum  expected: num ] ]
        ifNotNil: [
          | expected count |
          expected := nilOnPathRoots occurrencesOf: root.
          count := 0.
          expected
            timesRepeat: [
              nilOnPathRoots
                remove: root
                ifAbsent: [
                  auditor pathTermIncorrectNilOnPathCount: self root: root got: count expected: expected.
                  ^ self ].
              count := count + 1 ] ].
      ^ self ].
  1 to: children size do: [ :i |
    (children at: i)
      auditUpdateCacheFor: nextObj
      root: root
      nilOnPathRoots: nilOnPathRoots
      occurrences: num
      using: auditor ]

]

{ #category : 'Accessing' }
GsPathTerm >> children [

"Returns the value of the instance variable 'children'."

^children

]

{ #category : 'Updating' }
GsPathTerm >> children: newValue [

"Modify the value of the instance variable 'children'."

children := newValue

]

{ #category : 'Updating Indexes' }
GsPathTerm >> cleanupDependencyListFor: anObject [
  ""

  | ivOffset nextObj depList |
  (nil == anObject or: [ self size == 0 ])
    ifTrue: [ ^ self ].
  (depList := DependencyList for: anObject) == nil
    ifTrue: [ ^ self ].
  ivOffset := depList removeCompletelyPathTerm: self for: anObject.	" remove the receiver from the dependency list "
  (self indicatesIndexOnRootNsc
    or: [
      "last element dependency list"
      ivOffset < 0 ])
    ifTrue: [ ^ self ].
  ivOffset == 0
    ifTrue: [ nextObj := self _nextObjectFor: anObject ]
    ifFalse: [
      [ nextObj := self _nextObjectFor: anObject atInstVar: ivOffset ]
        onException: Error
        do: [ :ex |
          "Handle any error thrown during execution of _nextObjectFor:
     Cannot resolve the path for this guy --- can't have been added to index,
     in fact this object could have caused the failure in the first place"
          ^ self ] ].
  nil == nextObj
    ifTrue: [ ^ self ].
  needsDepList
    ifTrue: [
      (depList := DependencyList for: nextObj) ~~ nil
        ifTrue: [ depList removeCompletelyPathTerm: self for: nextObj ].
      ^ self ].
  1 to: children size do: [ :i | (children at: i) cleanupDependencyListFor: nextObj ]

]

{ #category : 'Clustering' }
GsPathTerm >> clusterDepthFirst [

"Cluster the receiver."

name cluster.
securityPolicies cluster.
children cluster.
1 to: children size do: [ :i | (children at: i) cluster ]

]

{ #category : 'Testing' }
GsPathTerm >> coversIvOffset: anIvOffset for: anObject [
  "Answer true if the receiver covers the given iv offset (see bug 46705)"

  ^false

]

{ #category : 'Modification' }
GsPathTerm >> findAllReachableNilOnPathRoots [

  | reachableRoots |
  reachableRoots := IdentityBag new.
  reachableRoots addAll: nilOnPath.
  1 to: children size do: [:i |
    reachableRoots addAll: (children at: i) findAllReachableNilOnPathRoots ].
  ^ reachableRoots

]

{ #category : 'Modification' }
GsPathTerm >> findReachableNilOnPathRoots [

  | reachableRoots |
  reachableRoots := IdentityBag new.
  reachableRoots addAll: nilOnPath.
  1 to: children size do: [:i |
    reachableRoots addAll: (children at: i) nilOnPath ].
  ^ reachableRoots

]

{ #category : 'Modification' }
GsPathTerm >> findReachableRootObjectMaps: rootObjectMap for: anObject ivOffset: ivOffset oldValue: oldValue sanity: sanity [
  | traversalRootObjects |
  self findRootObjectMaps: rootObjectMap for: anObject ivOffset: ivOffset oldValue: oldValue.
  traversalRootObjects := rootObjectMap resolveTraversalRootObjectsFor: anObject.
  ^ traversalRootObjects

]

{ #category : 'Collection Based Modification' }
GsPathTerm >> findReachableRootObjectMapsFor: anObject [
  | rootObjectMap indexObj reachableRoots rootObjects |
  rootObjectMap := BtreePlusRootObjectMap new
    pivotObject: anObject;
    pivotPathTerm: self;
    yourself.
  self findRootObjectMaps: rootObjectMap for: anObject.
  rootObjects := rootObjectMap resolveTraversalRootObjectsFor: anObject.
  indexObj := self at: self size.
  reachableRoots := IdentityBag new.
  rootObjects asIdentitySet do: [:root |
    (indexObj traverseAllWithParents: root upTo: self startingAt: 1)
      do: [ :ar | | intermediateObjectReachableFromRoot |
         intermediateObjectReachableFromRoot := ar at: 2.
        intermediateObjectReachableFromRoot == anObject
          ifTrue: [ reachableRoots add: root withOccurrences: (rootObjects occurrencesOf: root) ] ] ].
  ^ reachableRoots

]

{ #category : 'Modification' }
GsPathTerm >> findReachableRootObjectMapsFor: anObject ivOffset: ivOffset oldValue: oldValue [
  | rootObjectMap |
  rootObjectMap := BtreePlusRootObjectMap new
    pivotObject: anObject;
    pivotPathTerm: self;
    yourself.
  ^ self
    findReachableRootObjectMaps: rootObjectMap
    for: anObject
    ivOffset: ivOffset
    oldValue: oldValue
    sanity: false

]

{ #category : 'Collection Based Modification' }
GsPathTerm >> findReachableRootsFor: anObject [
  | rootObjects indexObj reachableRoots |
  rootObjects := self findRootObjectsFor: anObject.
  indexObj := self at: self size.
  "select rootObjects for which anNsc is reachable"
  reachableRoots := IdentityBag new.
  rootObjects asIdentitySet do: [:root |
    (indexObj traverseAllWithParents: root upTo: self startingAt: 1)
      do: [ :ar | | intermediateObjectReachableFromRoot |
         intermediateObjectReachableFromRoot := ar at: 2.
        intermediateObjectReachableFromRoot == anObject
          ifTrue: [ reachableRoots add: root withOccurrences: (rootObjects occurrencesOf: root) ] ] ].
  ^ reachableRoots

]

{ #category : 'Modification' }
GsPathTerm >> findReachableRootsFor: anObject ivOffset: ivOffset oldValue: oldValue [
  | rootObjects indexObj reachableRoots reachableNilOnPathRoots |
  rootObjects := self findRootObjectsFor: anObject ivOffset: ivOffset oldValue: oldValue.
  reachableNilOnPathRoots := self findReachableNilOnPathRoots.
  rootObjects addAll: reachableNilOnPathRoots.
  indexObj := self at: self size.
  "select rootObjects for which anObject is reachable"
  reachableRoots := IdentityBag new.
  rootObjects asIdentitySet do: [:root |
    (self traverseAllReachableFor: root indexObj: indexObj)
      do: [ :intermediateObjectReachableFromRoot |
        intermediateObjectReachableFromRoot == anObject
          ifTrue: [ reachableRoots add: root withOccurrences: (rootObjects occurrencesOf: root) ] ] ].
  ^ reachableRoots

]

{ #category : 'Collection Based Modification' }
GsPathTerm >> findRootObjectMaps: rootObjectMap for: anObject [
  "Traverse btree nodes from receiver to leaf term. Collect rootObjects for all btree elements
   for which anObject is a key or parent of a key."

  | ivOffset |
  ivOffset := self _ivOffsetFor: anObject.
  ivOffset ifNil: [ self _invalidIvOffset: anObject ].
  self findRootObjectMaps: rootObjectMap for: anObject ivOffset: ivOffset

]

{ #category : 'Collection Based Modification' }
GsPathTerm >> findRootObjectMaps: rootObjectMap for: anObject ivOffset: ivOffset [
  "Traverse btree nodes from receiver to leaf term. Collect rootObjects for all btree elements
   for which anObject at is a key or parent of a key."

  | nextObj |
  nextObj := self _nextObjectFor: anObject atInstVar: ivOffset.
  nextObj
    ifNil: [
      | indexObj |
      indexObj := self at: self size.
      nilOnPath do: [:nilOnPathRoot |
        indexObj
          traverse: nilOnPathRoot
          parent: nil
          thru: self
          startingAt: 1
          do: [ :term :parent :child |
            (rootObjectMap traversalMap at: child ifAbsentPut: [ IdentityBag new ]) add: parent  ] ] ].
  updateBtree ~~ nil
    ifTrue: [
      "btree exists for receiver, collect the root objects, where anObject is the key and nextObj
       is the value."
      self findRootObjectMaps: rootObjectMap forKey: nextObj value: anObject ].
  (nil == nextObj or: [ self children isEmpty ])
    ifTrue: [ ^ self ].
  1 to: children size do: [:i |
    (children at: i) findRootObjectMaps: rootObjectMap for: nextObj ]

]

{ #category : 'Collection Based Modification' }
GsPathTerm >> findRootObjectMaps: rootObjectMap for: anObject ivOffset: ivOffset oldValue: oldValue [
  "Traverse btree nodes from receiver to leaf term. Collect rootObjects for all btree elements
   for which anObject at is a key or parent of a key."

  | nextObj |
  nextObj := self _nextObjectFor: anObject atInstVar: ivOffset.
  oldValue
    ifNil: [
      (rootObjectMap traversalMap includesKey: anObject)
        ifFalse: [
          | indexObj pivotObject  nscRoot |
          indexObj := self at: self size.
          nscRoot := indexObj nscRoot.
          pivotObject := rootObjectMap pivotObject.
          nilOnPath asIdentitySet do: [:nilOnPathRoot |
            "nilOnPath collection may have more entries than exist in the nscRoot - set valued
               path terms, but we are only interested in the root object counts for nscRoot."
            indexObj
              traverse: nilOnPathRoot
              parent: nil
              thru: self
              startingAt: 1
              do: [ :term :parent :child |
                (nscRoot occurrencesOf: nilOnPathRoot) timesRepeat: [ (rootObjectMap traversalMap at: child ifAbsentPut: [ IdentityBag new ]) add: parent ] ].
            pivotObject == nilOnPathRoot
              ifTrue: [ (nscRoot occurrencesOf: nilOnPathRoot) timesRepeat: [ (rootObjectMap traversalMap at: nilOnPathRoot ifAbsentPut: [ IdentityBag new ]) add: nilOnPathRoot ] ] ] ] ].
  nextObj == oldValue ifFalse: [ self error: 'unexpected error: oldValue and nextObj should be identical by definition.' ].
  updateBtree ~~ nil
    ifTrue: [
      "btree exists for receiver, collect the root objects, where anObject is the key and nextObj
       is the value."
      self findRootObjectMaps: rootObjectMap forKey: nextObj value: anObject ].
  (nil == nextObj or: [ self children isEmpty ])
    ifTrue: [ ^ self ].
  1 to: children size do: [:i |
    (children at: i) findRootObjectMaps: rootObjectMap for: nextObj oldValue: oldValue ]

]

{ #category : 'Collection Based Modification' }
GsPathTerm >> findRootObjectMaps: rootObjectMap for: anObject oldValue: oldValue [
  "Traverse btree nodes from receiver to leaf term. Collect rootObjects for all btree elements
   for which anObject is a key or parent of a key."

  | ivOffset |
  ivOffset := self _ivOffsetFor: anObject.
  ivOffset ifNil: [ self _invalidIvOffset: anObject ].
  self
      findRootObjectMaps: rootObjectMap
      for: anObject
      ivOffset: ivOffset
      oldValue: (self _nextObjectFor: oldValue atInstVar: ivOffset).

]

{ #category : 'Modification' }
GsPathTerm >> findRootObjectMaps: rootObjectMap forKey: aKey value: aValue [
  | indexRootObjectMap  |
  indexRootObjectMap := rootObjectMap copy.
  updateBtree
    findRootObjectMaps: indexRootObjectMap
    pathTerm: self
    key: aKey
    value: aValue.
  rootObjectMap mergeFromCopy: indexRootObjectMap.

]

{ #category : 'Collection Based Modification' }
GsPathTerm >> findRootObjectsFor: anObject [
  "Traverse btree nodes from receiver to leaf term. Collect rootObjects for all btree elements
   for which anObject is a key or parent of a key."

  | ivOffset |
  ivOffset := self _ivOffsetFor: anObject.
  ivOffset ifNil: [ self _invalidIvOffset: anObject ].
  ^ self findRootObjectsFor: anObject ivOffset: ivOffset

]

{ #category : 'Collection Based Modification' }
GsPathTerm >> findRootObjectsFor: anObject ivOffset: ivOffset [
  "Traverse btree nodes from receiver to leaf term. Collect rootObjects for all btree elements
   for which anObject at is a key or parent of a key."

  | rootObjects nextObj roots |
  nextObj := self _nextObjectFor: anObject atInstVar: ivOffset.
  rootObjects := updateBtree ~~ nil
    ifTrue: [
      "btree exists for receiver, collect the root objects, where anObject is the key and nextObj
       is the value."
      self findRootObjectsForKey: nextObj value: anObject ]
    ifFalse: [ IdentityBag new ].
  (nil == nextObj or: [ self children isEmpty ])
    ifTrue: [ ^ rootObjects ].
  1 to: children size do: [:i |
    roots := (children at: i) findRootObjectsFor: nextObj.
    rootObjects := rootObjects _union: roots  ].
  ^ rootObjects

]

{ #category : 'Collection Based Modification' }
GsPathTerm >> findRootObjectsFor: anObject ivOffset: ivOffset oldValue: oldValue [
  "Traverse btree nodes from receiver to leaf term. Collect rootObjects for all btree elements
   for which anObject at is a key or parent of a key."

  | rootObjects nextObj roots |
  rootObjects := IdentityBag new.
  nextObj := self _nextObjectFor: anObject atInstVar: ivOffset.
  nextObj == oldValue ifFalse: [ self error: 'unexpected error: oldValue and nextObj should be identical by definition.' ].
  updateBtree ~~ nil
    ifTrue: [
      "btree exists for receiver, collect the root objects, where anObject is the key and nextObj
       is the value."
      rootObjects := self findRootObjectsForKey: nextObj value: anObject ].
  (nil == nextObj or: [ self children isEmpty ])
    ifTrue: [ ^ rootObjects ].
  1 to: children size do: [:i |
    roots := (children at: i) findRootObjectsFor: nextObj oldValue: oldValue.
    rootObjects := rootObjects _union: roots  ].
  ^ rootObjects

]

{ #category : 'Collection Based Modification' }
GsPathTerm >> findRootObjectsFor: anObject oldValue: oldValue [
  "Traverse btree nodes from receiver to leaf term. Collect rootObjects for all btree elements
   for which anObject is a key or parent of a key."

  | ivOffset |
  ivOffset := self _ivOffsetFor: anObject.
  ivOffset ifNil: [ self _invalidIvOffset: anObject ].
  ^ self
      findRootObjectsFor: anObject
      ivOffset: ivOffset
      oldValue: (self _nextObjectFor: oldValue atInstVar: ivOffset).

]

{ #category : 'Modification' }
GsPathTerm >> findRootObjectsForKey: aKey value: aValue [
  ^ updateBtree findRootObjectsForKey: aKey value: aValue

]

{ #category : 'Accessing' }
GsPathTerm >> getIndexDictionary [

"Returns the index dictionary associated with the receiver."

"no idexDictionary associated with the receiver"

^ nil

]

{ #category : 'Accessing' }
GsPathTerm >> getParentTerm [

"Returns the parent path term of the receiver.  This is determined by looking
 at the previous path term of the first index object that utilizes this path
 term.  If there is no parent, returns nil."

offset > 1
    ifTrue: [ ^ (self at: 1) at: offset - 1 ]
    ifFalse: [ ^ nil ]

]

{ #category : 'Accessing' }
GsPathTerm >> getRootNsc [

"Returns the root NSC for the index in which the receiver is a term."

^ (self at: 1) nscRoot

]

{ #category : 'Testing' }
GsPathTerm >> hasIndexDictionary [
  "Answer if receiver's index may have an indexDictionary."

  ^ false

]

{ #category : 'Testing' }
GsPathTerm >> hasMatchingTerm: aSymbol [
  "Answer true if the receiver's term matches <aSymbol>"

  ^ self name == aSymbol

]

{ #category : 'Testing' }
GsPathTerm >> indicatesIndexOnNscElements [

"Returns true if the path term indicates that the index is on elements of the
 NSC itself.  This is the case if the path is '' (two single quotes).  This
 method is overridden by SetValuedPathTerm to return true."

^ name ==  #''

]

{ #category : 'Testing' }
GsPathTerm >> indicatesIndexOnRootNsc [
  "Returns true if the path term indicates that the index is on elements of the
 NSC itself.  This is the case if the path is '' (two single quotes)."

  ^ name == #''

]

{ #category : 'Testing' }
GsPathTerm >> indicatesMultiValue [
  "Returns false, the receiver does not indicate that the index is on an
 instance variable that may have multiple values (i.e., collection based path term)"

  ^ false

]

{ #category : 'Testing' }
GsPathTerm >> indicatesNsc [
  "Returns false, the receiver does not indicate that the index is on an
 instance variable that is expected to be a collection (i.e., set valued pathterm."

  ^ false

]

{ #category : 'Initialization' }
GsPathTerm >> initialize [

"Initialize the receiver with default values."

securityPolicies := GsObjectSecurityPolicySet new.
children := { } .
needsDepList := false.
nilOnPath := RcLowMaintenanceIdentityBag new

]

{ #category : 'Testing' }
GsPathTerm >> isEnumeratedTerm [
  "Returns true if the receiver indicates an enumerated path term."

  ^ false

]

{ #category : 'Testing' }
GsPathTerm >> isOptionalTerm [
  ^ false

]

{ #category : 'Testing' }
GsPathTerm >> isRangeEqualityIndexLastPathTerm [

"Returns true if the receiver is the last path term of a RangeEqualityIndex."

1 to: self size do: [ :i |
    ( (self at: i) isRangeEqualityIndex and: [ (self at: i) size == offset ] )
        ifTrue: [ ^ true ]
].
^ false

]

{ #category : 'Testing' }
GsPathTerm >> isSelectorTerm [
  ^ false

]

{ #category : 'Testing' }
GsPathTerm >> isSetValuedTerm [
  "Returns true if the receiver indicates a set-valued instance variable."

  ^ false

]

{ #category : 'Accessing' }
GsPathTerm >> name [

"Returns the value of the instance variable 'name'."

^name

]

{ #category : 'Updating' }
GsPathTerm >> name: newValue [

"Modify the value of the instance variable 'name'."

name := newValue

]

{ #category : 'Accessing' }
GsPathTerm >> needsDepList [

"Returns the value of the instance variable 'needsDepList'."

^needsDepList

]

{ #category : 'Updating' }
GsPathTerm >> needsDepList: newValue [

"Modify the value of the instance variable 'needsDepList'."

needsDepList := newValue

]

{ #category : 'Accessing' }
GsPathTerm >> nilOnPath [
  "return nil or identity bag that contains root objects with nil along the path"

  ^ nilOnPath

]

{ #category : 'Accessing' }
GsPathTerm >> objectSecurityPolicy: anObjectSecurityPolicy [

"Assign the receiver and its children to the given security policy."

super objectSecurityPolicy: anObjectSecurityPolicy.
children objectSecurityPolicy: anObjectSecurityPolicy.
1 to: children size do: [ :i |
    (children at: i) objectSecurityPolicy: anObjectSecurityPolicy
].
securityPolicies objectSecurityPolicy: anObjectSecurityPolicy.

]

{ #category : 'Accessing' }
GsPathTerm >> offset [

"Returns the value of the instance variable 'offset'."

^offset

]

{ #category : 'Updating' }
GsPathTerm >> offset: newValue [

"Modify the value of the instance variable 'offset'."

offset := newValue

]

{ #category : 'Private' }
GsPathTerm >> recordNilOnPathForRoot: rootObject [

  "record fact that rootObject has a nil on the path for this path term"

  updateBtree ifNotNil: [ ^ self ].
  nilOnPath add: rootObject

]

{ #category : 'Removing' }
GsPathTerm >> removeDirectMappingFor: anObject logging: doLogging [
  "There is a range index directly on the elements of the NSC.  Update the
 B-tree."

  updateBtree ~~ nil
    ifTrue: [ " remove it from the B-tree "
      updateBtree btreeRemoveKey: anObject value: anObject root: anObject ]

]

{ #category : 'Removing' }
GsPathTerm >> removeDirectMappingFor: anObject root: root nsc: anNsc logging: doLogging [
  "There is a range index directly on the elements of the NSC.  Update the
 B-tree."

  updateBtree ~~ nil
    ifTrue: [ " remove it from the B-tree "
      (updateBtree btreeRemoveKey: anObject value: anObject root: root)
         ifFalse: [ self _errorObjectNotInBtree: anObject value: anObject root: root ] ]

]

{ #category : 'Index Maintenance' }
GsPathTerm >> removeIndex: indexObj [

"The given index is being removed from this path term."

1 to: self size do: [ :i |
    indexObj == (self at: i)
        ifTrue: [ ^ self removeAtIndex: i ]
]


]

{ #category : 'Removing' }
GsPathTerm >> removeMappingsFor: anObject at: ivOffset root: rootObject lastOne: lastOne logging: doLogging [

  ^self removeMappingsFor: anObject root: rootObject lastOne: lastOne logging: doLogging

]

{ #category : 'Removing' }
GsPathTerm >> removeMappingsFor: anObject lastOne: lastOne logging: doLogging [
  ^self removeMappingsFor: anObject root: anObject lastOne: lastOne logging: doLogging

]

{ #category : 'Removing' }
GsPathTerm >> removeMappingsFor: anObject root: rootObject lastOne: aBoolean logging: doLogging [
  "Remove entries in the btree and dependency lists for anObject."

  | nextObj depList more isLast |
  self size == 0
    ifTrue: [ ^ self ].
  nextObj := self _nextObjectFor: anObject.
  updateBtree ~~ nil
    ifTrue: [
      " remove an entry from the B-tree "
      (updateBtree btreeRemoveKey: nextObj value: anObject root: rootObject)
        ifFalse: [ self _errorObjectNotInBtree: nextObj value: anObject root: rootObject ].
      DependencyList removePathTerm: self for: nextObj logging: doLogging	" remove dependency list entry for next object " ].
  nextObj ifNil: [ self removeNilOnPathForRoot: rootObject ].
  more := false.
  (nil ~~ nextObj and: [ children isEmpty not ])
    ifTrue: [
      aBoolean
        ifTrue: [
          | last |
          last := (children at: 1) _isLastOccurrenceInIndexFor: nextObj.
          isLast := aBoolean and: [ last ] ]
        ifFalse: [ isLast := false ].
          " make recursive call to remove mappings "
      1 to: children size do: [ :i |
        (children at: i) removeMappingsFor: nextObj root: rootObject lastOne: isLast logging: doLogging ].
      isLast
        ifFalse: [
          "check again in case there are no more references left in btree"
          isLast := ((children at: 1) findReachableRootsFor: nextObj) isEmpty ].
      isLast
        ifTrue: [
          " remove dependency list entries for next object "
          (depList := DependencyList for: nextObj) ~~ nil
            ifTrue: [ depList removePathTerms: children for: nextObj logging: doLogging ] ] ]

]

{ #category : 'Updating Indexes' }
GsPathTerm >> removeMappingsSkipBtreeFor: anObject root: rootObj lastOne: aBoolean logging: doLogging [

"Remove entries in the dependency lists for anObject.
 This version for index removal, it does not update the btree since the btree
 is going away (#40858)."

| nextObj depList isLast |
self size == 0
  ifTrue: [ ^ self ].

nextObj := self _nextObjectFor: anObject.

updateBtree ~~ nil
  ifTrue: [
    " skip update of btree, index is going away"

    " remove dependency list entry for next object "
    DependencyList removePathTerm: self for: nextObj logging: doLogging.
  ].

( nil ~~ nextObj and: [ children isEmpty not ] )
  ifTrue: [
    aBoolean
      ifTrue: [
        | last |
        last := (children at: 1) _isLastOccurrenceInIndexFor: nextObj.
        isLast := aBoolean and: [last ] ]
      ifFalse: [ isLast := false ].
    " make recursive call to remove mappings "
    1 to: children size do: [ :i |
      (children at: i) removeMappingsSkipBtreeFor: nextObj
        root: rootObj
        lastOne: isLast
        logging: doLogging
    ].
        " remove dependency list entries for next object "
        (depList := DependencyList for: nextObj) ~~ nil
          ifTrue: [ depList removePathTerms: children for: nextObj logging: doLogging ]
  ]

]

{ #category : 'Private' }
GsPathTerm >> removeNilOnPathForRoot: rootObject [

  "rootObject with nil on path has been removed"

  updateBtree ifNotNil: [ ^ self ].
  nilOnPath remove: rootObject

]

{ #category : 'Accessing' }
GsPathTerm >> requirePathTerms: aBool [
  "compat with PathTerm implementation"

  ^ self termsRequired: aBool

]

{ #category : 'Accessing' }
GsPathTerm >> termsRequired [
  ^ termsRequired ifNil: [ termsRequired := true ]

]

{ #category : 'Accessing' }
GsPathTerm >> termsRequired: aBool [
  termsRequired := aBool

]

{ #category : 'Modification' }
GsPathTerm >> traverseAllReachableFor: root indexObj: indexObj [

  | res |
  res := IdentitySet new.
  (indexObj traverseAllWithParents: root upTo: self startingAt: 1) do: [:each | res add: (each at: 2) ].
  ^ res

]

{ #category : 'Updating Indexes' }
GsPathTerm >> undoDirectMappingFor: anObject logging: doLogging [
  "There is a range index directly on the elements of the NSC."
  " no indexDictionary in the receiver ... noop"

]

{ #category : 'Updating Indexes' }
GsPathTerm >> undoMappingsFor: anObject lastOne: aBoolean logging: doLogging [

"Remove entries in the index dictionary and dependency lists for anObject."


| nextObj depList more isLast |
self size == 0
  ifTrue: [ ^ self ].

[
    nextObj := self _nextObjectFor: anObject.
] onException: Error do:[ :ex |
    "Handle any error thrown during execution of _nextObjectFor:
     Cannot resolve the path for this guy --- can't have been added to index,
     in fact this object could have caused the failure in the first place"
    ^ self
].

updateBtree ~~ nil
  ifTrue: [
    " remove dependency list entry for next object "
    DependencyList removePathTerm: self for: nextObj logging: doLogging.
  ].

more := false.
( nil ~~ nextObj and: [ children isEmpty not ] )
  ifTrue: [ " make recursive call to remove mappings "
    isLast := aBoolean and: [ more not ].
    1 to: children size do: [ :i |
      (children at: i) undoMappingsFor: nextObj
        lastOne: isLast
        logging: doLogging
    ].
    isLast
      ifTrue: [
        " remove dependency list entries for next object "
        (depList := DependencyList for: nextObj) ~~ nil
          ifTrue: [ depList removePathTerms: children for: nextObj logging: doLogging ]
      ]
  ]

]

{ #category : 'Modification' }
GsPathTerm >> update: anObject at: ivOffset to: newValue [
  "The instance variable for anObject is being changed to newValue.
 Update the btrees and dependency lists for the old value and the
 newValue.  Returns true if the index objects were modified correctly;
 otherwise returns an Array containing error information."

  | oldValue reachableRoots |
  self size == 0
    ifTrue: [ ^ true ].
  oldValue := anObject instVarAt: ivOffset.
  reachableRoots := self findReachableRootObjectMapsFor: anObject ivOffset: ivOffset oldValue: oldValue.
  updateBtree ifNotNil: [
    self
      updateBtreeDirectFor: anObject
      at: ivOffset
      oldValue: oldValue
      to: newValue
      roots: reachableRoots ].
  self
    updateBtreeFor: anObject
    at: ivOffset
    oldValue: oldValue
    to: newValue
    roots: reachableRoots.
  ^ true

]

{ #category : 'Updating' }
GsPathTerm >> updateBtree [
  "Answer the value of the instance variable 'updateBtree'."

  ^ updateBtree

]

{ #category : 'Updating' }
GsPathTerm >> updateBtree: aGsAbstractIndex [
  "Modify the value of the instance variable 'updateBtree'."

  updateBtree := aGsAbstractIndex

]

{ #category : 'Modification' }
GsPathTerm >> updateBtreeDirectFor: anObject at: ivOffset oldValue: oldValue to: newValue roots: reachableRoots [
  "The instance variable for anObject is being changed from olValue to newValue.
 Update the btrees associated with the receiver for the old value and the newValue."

  "BtreePlusEqualityTests>>#testEnumeratedIndexUpdateLeaf fails wrong result set"
  "BtreePlusEqualityTests>>#testMultiEnumeratedIndexUpdateB passes (modulo deplist count)"

  (self _checkBtreeComparisonWith: newValue)
    ifFalse: [
      ImproperOperation new
        args: {newValue class. self name. updateBtree lastElementClassDescription};
        _number:
            (ErrorSymbols at: #'rtErrRangeEqualityIndexInvalidClassKindForBtree');
        signal].
  reachableRoots do: [ :root |
    (updateBtree
      btreeRemoveKey: oldValue
      value: anObject
      root: root)
        ifFalse: [ self _errorObjectNotInBtree: oldValue value: anObject root: root ] ].
  (needsDepList and: [ nil ~~ oldValue ])
    ifTrue: [
      reachableRoots size
        timesRepeat: [ DependencyList removePathTerm: self for: oldValue logging: false ] ].
  reachableRoots do: [ :root |
    " insert an entry into the B-tree "
    updateBtree btreeAt: newValue put: anObject root: root.
    needsDepList
      ifTrue: [
        " add an entry in the dependency list "
        newValue getDepListAndAddLastElementPathTerm: self logging: false ] ].

]

{ #category : 'Modification' }
GsPathTerm >> updateBtreeFor: anObject at: ivOffset oldValue: oldValue to: newValue roots: reachableRoots [
  "The instance variable for anObject is being changed from olValue to newValue.
 There are no btrees associated directly with the receiver. Traverse the receiver's
 children and update the btrees and dependency lists for the old value and the newValue."

  | doLogging |
  doLogging := UnorderedCollection _isRcIndexLoggingEnabled.
  reachableRoots do: [ :root |
    self updateEntriesFor: oldValue
      to: newValue
      root: root
      lastOne: true
      logging: doLogging ].
  (oldValue ~~ nil and: [ children size > 0 ])
    ifTrue: [
      | childRoots |
      childRoots := (children at: 1) findReachableRootObjectMapsFor: oldValue.
      (childRoots - reachableRoots) isEmpty
        ifTrue: [
          | depList |
          " remove dependency list entries for next object "
          (depList := DependencyList for: oldValue) ~~ nil
            ifTrue: [ depList removePathTerms: children for: oldValue logging: doLogging ] ] ]

]

{ #category : 'Modification' }
GsPathTerm >> updateEntriesFor: oldValue to: newValue root: rootObject lastOne: aBoolean logging: doLogging [
  "Remove entries in the index dictionary and dependency lists for anObject."

  self size == 0
    ifTrue: [ ^ self ].
  oldValue == nil
    ifTrue: [ self removeNilOnPathForRoot: rootObject ]
    ifFalse: [
      1 to: children size do: [ :i |
        (children at: i)
          removeMappingsFor: oldValue
          root: rootObject
          lastOne: aBoolean
          logging: doLogging ] ].
  newValue == nil
    ifTrue: [ self recordNilOnPathForRoot: rootObject ]
    ifFalse: [
      1 to: children size do: [ :i |
        (children at: i)
          addMappingsForObject: newValue root: rootObject logging: doLogging ] ]

]
