Extension { #name : 'GsProcess' }

{ #category : 'Private' }
GsProcess >> _convertToPortableStack: forTrimToLevel [
  "convert receiver's stack from native code to portable code format.
   Receiver must be suspended and stack in object memory.
  forTrimToLevel should be -1 if not being called for stack trim,
  otherwise a positive stack level. "

  self _nativeStack ifTrue:[
    [
      | newIps ipOfs fpOfs ar_size lev |
      newIps := { }  .
      ipOfs := 1 .
      fpOfs := topFpOffset .
      ar_size := arStack size .
      lev := 1 .
      [ fpOfs < 0 ] whileTrue:[ | fpIdx codePtr meth natIp portIp |

        fpIdx := ar_size  + (self _toWordOfs: fpOfs) + 1 .
        codePtr := arStack at:( fpIdx + FP_codePtr_OFS ).
        meth := self _fetchMethod:  codePtr  .
        natIp := arStack  at: ipOfs .
        portIp := meth _nativeIpOffsetToPortable: natIp asReturn: true .
        "uncomment for tracing" 
        "GsFile gciLogServer:'lev ', lev asString, ' meth ', meth asOop asString, ' natIp 16r',
          natIp asHexString, ' portIp ', portIp asString ."
        portIp <= 0 ifTrue:[
          (forTrimToLevel == -1
           or:[ forTrimToLevel >= 1 and:[ lev <= forTrimToLevel ]]) ifTrue:[
            portIp := GsNMethod _firstPortableIpoffset .
          ] ifFalse:[
            "examples: error 2085 from an ifTrue bytecode..."
            self error:'native to portable IP not exact in frame ', lev asString .
          ]
        ].
        arStack  at: ipOfs put: portIp .
        ipOfs := fpIdx + FP_rtnAddr_OFS .

        fpOfs := arStack at: fpIdx"+ FP_savedFP_OFS==0" . "get callerFpOfs"
        lev := lev + 1 .
      ].
      self _setInterpretedStack .
    ] onSynchronous: Error do:[ :ex |
      InternalError new details: ex description ;
           reason: 'nativeStackNotConvertableToInterpreted' ; signal
    ].
  ].

]

{ #category : 'Private' }
GsProcess >> _executeUnwindBlk: aBlock numTries: nTries timeout: timeoutMs stack: arStk topFp: topFpIdx resumeCnt: resumeCnt [
  "Returns decremented numTries"
  | numTries saveDbg sema hitDebug newStk saveTopFp signaled "trace" |
  "trace := true."
  numTries := nTries .
  saveDbg := self debugActionBlock .
  sema := Semaphore new .
  self debugActionBlock: [ :ex | 
     "trace ifNotNil:[ GsFile gciLogServer:self asOop asString, ' _executeUnwindBlk debugActionBlock got ' , ex asString]."
     self debugActionBlock: nil .
     hitDebug := true .
     sema signal .
  ].
  "push frame for ExecBlock>>_terminationBlockValue: on top of stack"
  (newStk := GsStackBuffer basicNew) 
     addAll: { OC_GSNMETHOD_FIRST_INSTR_OFFSET"savedIp" .  
              ((ExecBlock persistentMethodDictForEnv: 0) at: #_terminationBlockValue: ) "codePtr" .
              arStk at: topFpIdx .  "saved FP"
              OC_GSNMETHOD_FIRST_INSTR_OFFSET"rtnAddr".
              sema  "first arg" .
              aBlock "rcvr" } ;
     addAll: arStk .
  arStack := newStk .
  saveTopFp := topFpOffset .
  topFpOffset := (newStk size - 2) * -8 .
  dbgFpsCache := nil .
  self _resumeForTermination: resumeCnt .
  timeoutMs ifNotNil:[
    signaled := sema waitForMilliseconds: timeoutMs .
  ] ifNil: [
    sema wait . signaled := true .
  ].
  "trace ifNotNil:[ GsFile gciLogServer: self asOop asString, ' _executeUnwindBlk signaled=', 
      signaled asString, ' timeoutMs=', timeoutMs asString]."
  (signaled == false or:[ hitDebug == true]) ifTrue:[
     numTries := numTries - 1 .
  ].
  self debugActionBlock: saveDbg .

  self convertToPortableStack: -1 .
  arStack := arStk . "restore stack "
  topFpOffset := saveTopFp .
  dbgFpsCache := nil .
  ^ numTries
]

{ #category : 'Private' }
GsProcess >> _finishTermination: terminationException [
  "returns receiver"
  | wasInCrit sched "trace" |
  "trace := true. "
  "trace ifNotNil:[ GsFile gciLogServer: self asOop asString, ' _finishTermination:']."
  wasInCrit := (sched := self _scheduler) _enterCritical .
  block ifNil: [
    uaCount_depth := 0.
    arStack := nil.
    interruptFlag_stEntryCount := 0 .
  ] ifNotNil:[
    block := nil
  ].
  dbgFpsCache := nil .
  breakpointLevel := 0 .
  lastBreakpt := nil .
  self _setModeinfoBit: ModeInfo_terminated value: 1 .
  self _signalJoiners .
  wasInCrit == 0 ifTrue:[ sched _exitCritical ].
  (modeInfo bitAnd: ModeInfo_forked) == 0 ifTrue:[
    self _unscheduleProcess .
    self _terminatedReturnToGci: terminationException
  ].
  sched _terminateScheduledProcess: self. "switch to another process"
]

{ #category : 'Private' }
GsProcess >> _resumeForTermination: resumeCntArray [
  resumeCntArray at: 1 put:(resumeCntArray at:1) + 1 .
  self _scheduler _resumeForTermination: self resumeCnt: resumeCntArray
]

{ #category : 'Private' }
GsProcess >> _terminateCurrentWithResult: aResult [

"Sets terminated and terminationStarted bits in the currently running
  process (which may be different from the receiver),
  stores the result, and clears its waitingOn, and block.
  For the current in-memory stack, unlocks it and clears its use count.
  Stores nil in instVars arStack,clientData.
  Stores nil in  environment instVar.

  Returns currently running process.
"

  <primitive: 720>
  self _primitiveFailed: #_terminateCurrentWithResult: args: { aResult }

]

{ #category : 'Private' }
GsProcess >> _terminatedReturnToGci: anException [

"If receiver was not the most recently started main GsProcess,
 i.e. not from most recent IntSendMsg or IntExecute ,
 return the receiver.

 Otherwise return control to the GCI with a GciErrorSType encapsulating
 anException .  The receiver will be the GsProcess associated with that
 GciErrorSType , and will have no stack."

<primitive: 543>
"return control to the GCI"
anException _resumable: false ; signalToGci

]

{ #category : 'Private' }
GsProcess >> _terminationStarted [

  "returns true if receiver has started executing ensure: blocks"
  ^ (modeInfo bitAnd: ModeInfo_terminationStarted ) ~~ 0
]

{ #category : 'Deprecated' }
GsProcess >> _trimStackToLevel: aLevel [

"Legacy implementation, Deprecated.

 No execution of ensure blocks of trimmed frames is performed.

 Deletes stack entries from 1 to (aLevel - 1) inclusive, thus making aLevel the
 new top-of-stack(TOS).  At new TOS, looks in the implementation class of the
 method at TOS, using the selector of the method at TOS.  If that class's method
 dictionary contains a different GsNMethod, the GsNMethod currently in the method
 dictionary is installed as the method at the TOS.  The saved instruction
 pointer for TOS is set to the entry of the method at TOS.

 Limitations:
   If the receiver was executing in native code, and the stack
   is trimmed, any resumption of execution will be in interpreted mode.

   If the new top-of-stack is an anonymous method, it is not possible to
   determine whether that method has been recompiled, and the method at new top
   of stack will not be changed.  Debuggers should use the
   GsNMethod | _recompileWithSource method to implement recompilation from a
   debugger pane.  _recompileWithSource: raises an error on attempts to
   recompile an anonymous method.

   Raises an error if the new top-of-stack would represent the entry to an
   ExecBlock and the home method of the block has been recompiled.

   Raises an error if the new top-of-stack would represent a return
   across an FFI , Ruby C extension, or UserAction call.

 If aLevel = 1, reset instruction pointer to beginning of the method.

 Debuggers must not cache or directly manipulate VariableContexts
 when examining or altering stacks.

 Provides the ability for a debugger to restart execution after recompiling
 a method that is active on the stack of the receiver."

 | fpIdx aMeth oldHome oldmClass newTosIdx oldDepth newDepth newTosFpOfs
   prevFpIdx envId |
  aLevel <= 1 ifTrue:[ ^ self ].
  self convertToPortableStack: aLevel .

  "check for argument aLevel out of range"
  oldDepth := self localStackDepth .
  aLevel > oldDepth  ifTrue:[ ^ self ].
  newDepth := oldDepth - (aLevel - 1) .

  1 to: aLevel do:[:n|
    prevFpIdx := fpIdx .
    fpIdx := self _frameOffsetAt: n  .
    aMeth := self _fetchMethod:( arStack at:( fpIdx + FP_codePtr_OFS)) .
    aMeth ifNil:[ Error signal:'Cannot trim stack across a reenter marker'].
  ].

  "check to see whether new TOS method has been recompiled, and if so,
   install the new method."
  oldHome := aMeth homeMethod .
  oldmClass := oldHome inClass .
  envId := oldHome environmentId .
  oldmClass ifNotNil:[
    "not an anonymous method"
    | newHome oldCptr |
    newHome := oldmClass compiledMethodAt: oldHome selector environmentId: envId
			otherwise: aMeth .
    newHome == oldHome ifFalse:[
      aMeth == oldHome ifFalse:[
        Error signal:'Cannot trim stack to a block in recompiled method'.
      ].
      "install recompiled method"
      oldCptr := arStack at: (fpIdx + FP_codePtr_OFS) .
      oldCptr class == GsNMethod ifTrue:[
        arStack at: (fpIdx + FP_codePtr_OFS) put: newHome .
      ] ifFalse:[
        self _halt:'should not be here'
      ].
    ].
  ].
  newTosIdx := fpIdx + FP_codePtr_OFS .
  "push savedIP which is byte offset to first instr in the method"
  newTosIdx := newTosIdx - 1 .
  arStack at: newTosIdx put: OC_GSNMETHOD_FIRST_INSTR_OFFSET .
  aLevel > 1 ifTrue:[
    "don't need  _tosIsIpNil in arStack object"
    newTosFpOfs := arStack at: prevFpIdx .
    topFpOffset := newTosFpOfs .  "adjust offset to Top FP"
    dbgFpsCache := nil .
    "adjust the depth of the receiver."
     uaCount_depth := ((uaCount_depth bitShift: -32)  bitShift:32)
         bitOr: (newDepth bitAnd: 16rFFFFFFFF) ].
  newTosIdx > 1 ifTrue:[
    "delete the unwanted stack frames.
     Since FP linkages in arStack are relative to   arStack size
       there is no other fixup needed. "
    arStack removeFrom:1 to: newTosIdx - 1 .
  ].
  dbgFpsCache := nil . "clear frame offsets cache"
  self clearLastBreakpointHistory .  
]

{ #category : 'Private' }
GsProcess >> _trimStackToLevel: levelArg abandonActiveUnwind: abandonBool numTries: numTries timeout: timeoutMs [
  "levelArg == nil means terminate , level ~~ nil means trim stack .
   Has no effect if aLevel <= 1 or if the receiver has finished termination.

   abandonBool == true means a currently executing ensure block will be abandoned,
   false means it will be allowed to continue up to the timeout.

   Attempts to execute ensure blocks in stack frames that will be trimmed, per the timeout.

   numTries determines how many expirations of the timeout are allowed before any further
   execution of ensure blocks will be skipped.

   If the receiver was executing in native code, and the stack
   is trimmed, any resumption of execution will be at the start of the new TOS method.

   Raises an error if the new top-of-stack would represent a return
   across an FFI or UserAction call.

   Debuggers must not cache or directly manipulate VariableContexts
   when examining or altering stacks."

  | aLevel activeEnsBlkFp activeEnsLevel activeEnsureBlks arStk ensurBlks fpIdx 
    nTries newDepth oldDepth prevFpIdx resumeCnt rtnAddr saveBpLev theEx topFp aMeth
    "trace" "fp_codeptr_ofs fp_lastarg_ofs fp_marker_ofs fp_rtnaddr_ofs"  |
  "trace := true."
  (modeInfo bitAnd: ModeInfo_continuationMask) ~~ 0 ifTrue:[
     ^ ImproperOperation new object: self;
           signal:'cannot terminate an instance that is a continuation' .
  ].
  nTries := numTries .
  aLevel := levelArg .
  aLevel ifNotNil:[ aLevel <= 1 ifTrue:[ ^ self ]].

  (modeInfo bitAnd: ModeInfo_terminationStarted) ~~ 0 ifTrue:[ "termination already started"
     self _isTerminated ifTrue:[ ^ self "termination finished, do nothing" ].
     ^ Error signal: (levelArg ifNil:['termination'] ifNotNil:['stack trim']) , ' already started'.
  ].
  saveBpLev := breakpointLevel .
  breakpointLevel := 1 .
  self _setModeinfoBit: ModeInfo_terminationStarted value: 1 .

  aLevel ifNil:[
   [  Notification signal ] onSynchronous: AbstractException do:[ :ex |
     "capture stack in case GemExceptionSignalCapturesStack is true"
     (theEx := TerminateProcess new) _gsStack: ex _gsStack .
   ].
  ].
  self convertToPortableStack: (aLevel ifNil:[-1]) .
  "trace ifNotNil:[
    fp_rtnaddr_ofs := FP_rtnAddr_OFS . 
    fp_codeptr_ofs := FP_codePtr_OFS .
    fp_lastarg_ofs := FP_lastArg_OFS .
    fp_marker_ofs := FP_markerNil_OFS .
  ]."

  "check for argument aLevel out of range"
  oldDepth := self localStackDepth .
  arStk := arStack shallowCopy .
  
  aLevel ifNil:[ 
    aLevel := oldDepth   . "termination"
    timeoutMs ifNotNil:[ self debugActionBlock:[:ex| ^ self _finishTermination: theEx ]].
  ] ifNotNil:[   "trim stack"
    newDepth := oldDepth - (aLevel - 1) 
  ].
  ensurBlks := { } .
  activeEnsureBlks := IdentitySet new .

  1 to: aLevel do:[:n| | markOfs |
    fpIdx ifNotNil:[ rtnAddr := arStk at: (fpIdx + FP_rtnAddr_OFS) ].
    prevFpIdx := fpIdx .
    fpIdx := self _frameOffsetAt: n  .
    topFp ifNil:[ topFp := fpIdx ].
    aMeth := self _fetchMethod:( arStk at:( fpIdx + FP_codePtr_OFS)) .
    aMeth ifNil:[ ArgumentError signal:'Cannot trim stack across a reenter marker'].
    (markOfs := fpIdx + FP_markerNil_OFS) >= 1 ifTrue:[ | aBlk marker blkOfs |
      marker := arStk at: markOfs .
      "trace ifNotNil:[ GsFile gciLogServer:'marker ', marker asOop asString ]."
      marker == OOP_ENSURE_Mark_NIL ifTrue:[
        aBlk := arStk at:(blkOfs := fpIdx + FP_lastArg_OFS).
        aBlk _isExecBlock ifTrue:[
          ensurBlks add: blkOfs ; add: aBlk .
        ] ifFalse:[
          aBlk ifNotNil:[ UncontinuableError signal:'corrupt stack, expected an ExecBlock, ',
                          aBlk class name ].
        ]. 
      ].
      marker == OOP_terminationBlockValue_Mark_NIL ifTrue:[
        "trace ifNotNil:[
          GsFile gciLogServer: self asOop asString, ' active unwind block at fpIdx ', fpIdx asString.]."
        activeEnsBlkFp ifNil:[ activeEnsBlkFp := fpIdx . activeEnsLevel := n ].
        activeEnsureBlks  add: (arStk at: fpIdx + FP_lastArg_OFS"zero args rcvr").
      ]. 
    ].
  ].
  levelArg ifNotNil:[ | oldHome newHome oldCptr |
    oldHome := aMeth homeMethod .
    newHome := oldHome inClass compiledMethodAt: oldHome selector environmentId: oldHome environmentId
                          otherwise: nil .
    newHome ifNil:[ ^ ArgumentError signal:'cannot trim stack to a method that has been removed'].
    newHome == oldHome ifFalse:[
      aMeth == oldHome ifFalse:[
        ArgumentError signal:'Cannot trim stack to a block in recompiled method'.
      ].
    ].
    "install recompiled method"
    oldCptr := arStack at: (fpIdx + FP_codePtr_OFS) .
    oldCptr class == GsNMethod ifTrue:[
      arStack at: (fpIdx + FP_codePtr_OFS) put: newHome .
    ] ifFalse:[
      UncontinuableError signal:'corrupt stack, expected a GsNMethod'. 
    ].
    self  _setStackBreakAt: levelArg .
  ].
  resumeCnt := { 0 } .
  abandonBool ifFalse:[
    activeEnsBlkFp ifNotNil:[:fp| "continue execution topmost active unwind block" | sema |
      "trace ifNotNil:[ GsFile gciLogServer: self asOop asString, ' continue active unwind block '.]."
      sema := Semaphore new .  
      "install sema in first method temp of ExecBlock>>_terminationBlockValue"
      arStack at: fp + FP_MarkerValue_OFS - 1 put: sema . 
      self _setStackBreakAt: activeEnsLevel .
      timeoutMs ifNil:[ 
        self _resumeForTermination: resumeCnt .
        sema wait 
      ] ifNotNil:[
        nTries := self _waitForActiveUnwind: sema timeoutMs: timeoutMs numTries: nTries 
                   resumeCnt: resumeCnt
      ]. 
      self convertToPortableStack: -1 .
    ].
  ].
  "execute not-started ensure blocks that will be trimmed"
  ensurBlks size > 0 ifTrue:[ 
    1 to: ensurBlks size by: 2 do:[:n | |blk |
      nTries > 0 ifTrue:[
        "trace ifNotNil:[ GsFile gciLogServer:self asOop asString, ' execute unwind block n=', n asString. ]."
        arStk at: (ensurBlks at: n "blkOfs") put: nil .  "_removeEnsureAtFP "
        blk := (ensurBlks at: n + 1 ) .
        (activeEnsureBlks includes: blk) ifTrue:[ 
          UncontinuableError signal:'Active unwind block still referenced'
        ].
        nTries := self _executeUnwindBlk: blk numTries: nTries timeout: timeoutMs stack: arStk 
                      topFp: topFp resumeCnt: resumeCnt
      ].
    ].
    arStack := arStk shallowCopy .
  ].
  breakpointLevel := saveBpLev .
  levelArg ifNotNil:[  "finish trimStack" | newTosIdx newTosFpOfs |
    newTosIdx := fpIdx + FP_codePtr_OFS .
    "make savedIP be start of the method"
    newTosIdx := newTosIdx - 1 .
    arStack at: newTosIdx put: OC_GSNMETHOD_FIRST_INSTR_OFFSET . 
	  newTosFpOfs := arStack at: prevFpIdx .
	  topFpOffset := newTosFpOfs .  "adjust offset to Top FP"
    dbgFpsCache := nil .
	  "adjust the depth of the receiver."
	   uaCount_depth := ((uaCount_depth bitShift: -32)  bitShift:32)
			   bitOr: (newDepth bitAnd: 16rFFFFFFFF) .
   
    newTosIdx > 1 ifTrue:[
      "delete the unwanted stack frames.
       Since FP linkages in arStack are relative to   arStack size
         there is no other fixup needed. "
      arStack removeFrom:1 to: newTosIdx - 1 .
    ].
    dbgFpsCache := nil . "clear frame offsets cache"
    self clearLastBreakpointHistory .  
    self _setModeinfoBit: ModeInfo_terminationStarted value: 0 .
  ] ifNil:[
    self _finishTermination: theEx
  ]
]

{ #category : 'Private' }
GsProcess >> _waitForActiveUnwind: aSemaphore timeoutMs: timeoutMs numTries: numTries resumeCnt: resumeCnt [
  "returns decremented numTries"
  | signaled saveDbg hitDebug nTries "trace" |
   "trace := true."
   nTries := numTries .
   saveDbg := self debugActionBlock .
   self debugActionBlock: [ :ex | 
     "trace ifNotNil:[ GsFile gciLogServer: self asOop asString, 
          ' _waitForActiveUnwind debugActionBlock got ' , ex asString]."
     self debugActionBlock: nil .
     hitDebug := true .
     aSemaphore signal .
   ].
   self _resumeForTermination: resumeCnt .
   signaled := aSemaphore waitForMilliseconds: timeoutMs .
   "trace ifNotNil:[ GsFile gciLogServer: self asOop asString, 
        ' _waitForActiveUnwind signaled=', signaled asString ]."
   (signaled == false or:[ hitDebug == true]) ifTrue:[
     nTries := nTries - 1 .
   ].
   self debugActionBlock: saveDbg .
   ^ nTries
]

{ #category : 'Debugging Support' }
GsProcess >> abandonUnwindAndTrimStackToLevel: aLevel [
  "Trim stack down to the first active unwind block at or above aLevel .
   Trim that unwind block activation without attempting to complete it .
   Evalute any remaining unwind blocks from that point to aLevel ,
   while trimming stack to aLevel.
   Errors will go to the debugger or the GCI.  The timeout for attempting to  
   continue or evaluate unwind blocks is infinite."

  GsProcess current == self ifTrue:[ 
    ^ Error signal:'cannot send abandonUnwindAndTrimStackToLevel: to self'
 ].
  ^ self _trimStackToLevel: aLevel abandonActiveUnwind: true 
          numTries: SmallInteger maximumValue timeout: nil .
]

{ #category : 'Changing Process State' }
GsProcess >> terminate [
 "Terminates the receiver. If receiver is not the current process,
  it will be raised to normal priority and sent a termination interrupt.

  If receiver is a forked process, i.e. not the GsProcess of the
  current execute or perform, returns the receiver after receiver
  has been terminated.

  If receiver is a non-forked process , then
  after completion of the termination,  a not-trappable
  an error is returned to the GCI, containing
  a not-trappable instance of TerminateProcess,
  The   instance of TerminateProcess may have a stack captured,
  but   the GsProcess associated with the error has no stack
  and is terminated."

  ^ self terminateTries: 5 eachTimeoutMs: 10000.
]

{ #category : 'Debugging Support' }
GsProcess >> terminate9 [
  "Used by implementation of GciClearStack. "
  ^ self terminate
]

{ #category : 'Private' }
GsProcess >> _terminate9 [
  " for testing"
  ^ self terminateTries: 1 eachTimeoutMs: 2000 .
]

{ #category : 'Deprecated' }
GsProcess >> terminateTimeoutMs: timeoutMs [
  "Deprecated"
   
   self deprecated:  'GsProcess >> terminateTimeoutMs: Deprecated as of GS/64 3.7, use terminateTries:eachTimeoutMs: ' .
   ^ self terminateTries: 1 eachTimeoutMs: timeoutMs
]

{ #category : 'Private' }
GsProcess >> _incrementTerminationsInProgress: delta [
  | cnt |
  cnt := ioSelectResult ifNil:[ 0 ]. 
  ioSelectResult := cnt + delta .
]

{ #category : 'Private' }
GsProcess >> _waitForTerminator: timeoutMs [
  "wait for the receiver to finish executing terminate of some other GsProcess"
  | tStart tNow |
  tStart := System _timeMs .
  [ true ] whileTrue:[ 
    Delay waitForMilliseconds: 10 .
    ioSelectResult <= 0 ifTrue:[ ^ self "done"].
    tNow := System _timeMs .
    (tNow - tStart) > timeoutMs ifTrue:[
      Error signal: 'timeout waiting for termination of another process'.
    ]
  ] 
]

{ #category : 'Debugging Support' }
GsProcess >> terminateTries: numTries eachTimeoutMs: timeoutMs [
	"To interrupt the terminatee if unwind blocks do not complete within the given time limit, 
	there must be a monitor process running at a higher priority than the terminatee.
	If the terminator and terminatee are the same process, a new monitor process will 
	be forked at a priority one higher than the terminatee.
	Otherwise, the terminator acts as the monitor. 
	If the terminator's priority is <= the terminatee's, its priority will be temporarily raised
	to one higher than the terminatee's, then restored to its previous priority when termination is complete."

	| terminator terminatorPriority terminateePriority |
  self _isTerminated ifTrue:[ ^ self  "fix 50702" ].
  ioSelectResult ifNotNil:[ :cnt |
    "this process is executing terminate of some other process"
    cnt > 0 ifTrue:[ self _waitForTerminator: numTries * timeoutMs ].
  ]. 
	terminator := GsProcess current.
	terminatorPriority := terminator priority.
	terminateePriority := self priority.
	terminateePriority > Processor highestPriority ifTrue: [ 
			"Can happen if you send terminate to a terminator"
			terminateePriority := Processor highestPriority.
			self priority: terminateePriority 
  ].
	terminator == self ifTrue: [ 
			[ self terminateTries: numTries eachTimeoutMs: timeoutMs 
      ] newProcess
				_setPriority: terminateePriority + 1;
				resume 
  ] ifFalse: [ 
			| mustRestorePriority |
      terminator _incrementTerminationsInProgress: 1 .
      "increment counter of terminations being executed"
			terminatorPriority <= terminateePriority ifTrue: [ 
					mustRestorePriority := true.
					terminator _setPriority: terminateePriority + 1 
      ] ifFalse: [ mustRestorePriority := false ].
			[ self _trimStackToLevel: nil abandonActiveUnwind: false numTries: numTries
				  timeout: timeoutMs 
      ] ensure: [ 
        mustRestorePriority ifTrue: [ terminator priority: terminatorPriority ] .
        terminator _incrementTerminationsInProgress: -1 .
      ] 
  ]
]

{ #category : 'Debugging Support' }
GsProcess >> trimStackToLevel: aLevel [
  "Continue execution of the first first active unwind block found above aLevel .
   Evalute any remaining unwind blocks from that point to aLevel ,
   while trimming stack to aLevel.  
   Errors will go to the debugger or the GCI.  The timeout for attempting to  
   continue or evaluate unwind blocks is infinite."

  GsProcess current == self ifTrue:[ ^ Error signal:'cannot send trimStackToLevel: to self'].
  ^ self _trimStackToLevel: aLevel abandonActiveUnwind: false 
          numTries: SmallInteger maximumValue timeout: nil .
]

{ #category : 'Changing Process State' }
GsProcess >> unsafeTerminate [
  "agressive termination that executes in the sender's green thread
   without yielding and without using the receiver's stack to execute
   the ensure blocks ."
  | minfo theEx |
  ((minfo := modeInfo) bitAnd: ModeInfo_continuationMask) ~~ 0 ifTrue:[
    Error signal:'cannot terminate an instance that is a continuation'
  ].
  self _isTerminated ifTrue:[ ^ self "fix 50702" ].
  [  Notification signal ] onSynchronous: AbstractException do:[ :ex |
	 "capture stack in case GemExceptionSignalCapturesStack is true"
	 (theEx := TerminateProcess new) _gsStack: ex _gsStack .
  ].
  self _finishTermination: theEx .
]

{ #category : 'Debugging Support' }
GsProcess >> unsafeTrimStackToLevel: aLevel [

"Deletes stack entries from 1 to (aLevel - 1) inclusive, thus making aLevel the
 new top-of-stack(TOS).  At new TOS, there is no change to the installed method,
 and the saved instruction pointer is left unchanged.

 Unwind blocks above aLevel are discarded, without attempting to complete execution
 or to evalute not yet started unwind blocks."

 | fpIdx aMeth newTosIdx oldDepth newDepth newTosFpOfs
   prevFpIdx rtnAddr |
  aLevel <= 1 ifTrue:[ ^ self ].

  self convertToPortableStack: -1 .

  "check for argument aLevel out of range"
  oldDepth := self localStackDepth .
  aLevel > oldDepth  ifTrue:[ ^ self ].
  newDepth := oldDepth - (aLevel - 1) .

  1 to: aLevel do:[:n|
    fpIdx ifNotNil:[ rtnAddr := arStack at: (fpIdx + FP_rtnAddr_OFS) ].
    prevFpIdx := fpIdx .
    fpIdx := self _frameOffsetAt: n  .
    aMeth := self _fetchMethod:( arStack at:( fpIdx + FP_codePtr_OFS)) .
    aMeth ifNil:[ Error signal:'Cannot trim stack across a reenter marker'].
  ].

  newTosIdx := fpIdx + FP_codePtr_OFS .
  "push savedIP which the callee's return address "
  newTosIdx := newTosIdx - 1 .
  arStack at: newTosIdx put: rtnAddr . "no _tosIsIpNil in arStack object"

	newTosFpOfs := arStack at: prevFpIdx .
	topFpOffset := newTosFpOfs .  "adjust offset to Top FP"
  dbgFpsCache := nil .
	"adjust the depth of the receiver."
	 uaCount_depth := ((uaCount_depth bitShift: -32)  bitShift:32)
			 bitOr: (newDepth bitAnd: 16rFFFFFFFF) .
 
  newTosIdx > 1 ifTrue:[
    "delete the unwanted stack frames.
     Since FP linkages in arStack are relative to   arStack size
       there is no other fixup needed. "
    arStack removeFrom:1 to: newTosIdx - 1 .
  ].
  dbgFpsCache := nil . "clear frame offsets cache"
  self clearLastBreakpointHistory .  
]
