Extension { #name : 'GsQuery' }

{ #category : 'instance creation' }
GsQuery class >> _initialize [
  self
    defaultQueryOptions;
    defaultOptimizerClass

]

{ #category : 'defaults' }
GsQuery class >> defaultOptimizerClass [
  DefaultOptimizerClass ifNil: [ DefaultOptimizerClass := Gs32Optimizer ].
  ^ DefaultOptimizerClass

]

{ #category : 'defaults' }
GsQuery class >> defaultOptimizerClass: anOptimizerClass [
  DefaultOptimizerClass := anOptimizerClass

]

{ #category : 'defaults' }
GsQuery class >> defaultQueryOptions [
  DefaultQueryOptions ifNil: [ DefaultQueryOptions := GsQueryOptions default ].
  ^ DefaultQueryOptions

]

{ #category : 'defaults' }
GsQuery class >> defaultQueryOptions: aGsQueryOptions [
  DefaultQueryOptions := aGsQueryOptions

]

{ #category : 'instance creation' }
GsQuery class >> fromFormula: aGsQueryFormula [
  ^ self fromFormula: aGsQueryFormula on: nil options: self defaultQueryOptions

]

{ #category : 'instance creation' }
GsQuery class >> fromFormula: aGsQueryFormula on: anNsc [
  ^ self
    fromFormula: aGsQueryFormula
    on: anNsc
    options: self defaultQueryOptions

]

{ #category : 'instance creation' }
GsQuery class >> fromFormula: aGsQueryFormula on: anNsc options: aGsQueryOptions [
  ^ self new
    formula: aGsQueryFormula immediateInvariant;
    on: anNsc;
    queryOptions: aGsQueryOptions;
    yourself

]

{ #category : 'instance creation' }
GsQuery class >> fromFormula: aGsQueryFormula options: aGsQueryOptions [
  ^ self fromFormula: aGsQueryFormula on: nil options: aGsQueryOptions

]

{ #category : 'instance creation' }
GsQuery class >> fromSelectBlock: aSelectBlock [
  | formula |
  formula := GsQueryPredicate fromSelectBlock: aSelectBlock.
  ^ self fromFormula: formula immediateInvariant

]

{ #category : 'instance creation' }
GsQuery class >> fromSelectBlock: aSelectBlock on: anNsc [
  "formula returned by
	GsQueryPredicate class>>fromSelectBlock:usingQueryExecuterWith: is already
	bound to nsc, so we need to directly set nsc"

  | formula |
  formula := GsQueryPredicate
    fromSelectBlock: aSelectBlock
    usingQueryExecuterWith: anNsc.
  ^ self new
    formula: formula immediateInvariant;
    _nsc: anNsc;
    yourself

]

{ #category : 'instance creation' }
GsQuery class >> fromString: aString [
  ^ self fromFormula: (GsQueryPredicate fromQueryStatement: aString)

]

{ #category : 'instance creation' }
GsQuery class >> fromString: aString on: anNsc [
  ^ self fromFormula: (GsQueryPredicate fromQueryStatement: aString) on: anNsc

]

{ #category : 'instance creation' }
GsQuery class >> fromString: aString on: anNsc options: aGsQueryOptions [
  ^ self
    fromFormula: (GsQueryPredicate fromQueryStatement: aString)
    on: anNsc
    options: aGsQueryOptions

]

{ #category : 'instance creation' }
GsQuery class >> fromString: aString options: aGsQueryOptions [
  ^ self
    fromFormula: (GsQueryPredicate fromQueryStatement: aString)
    options: aGsQueryOptions

]

{ #category : 'private' }
GsQuery >> _collator [
  "Returns IcuCollator to be used when comparing Unicode strings"

  ^ collator

]

{ #category : 'private' }
GsQuery >> _collator: anIcuCollator [
  "Set the receiver's collator. Use a copy of anIcuCollator to disallow
   changes to strength, etc, that might affect the sort ordering. "

  collator := anIcuCollator copy immediateInvariant

]

{ #category : 'private' }
GsQuery >> _formula [
  ^ formula

]

{ #category : 'private' }
GsQuery >> _formula: aQueryFormula [
  self resetCache.
  formula := aQueryFormula.
  self queryOptions optionalPathTerms
    ifTrue: [ formula := aQueryFormula optionalPathTerms ]

]

{ #category : 'private' }
GsQuery >> _nsc [
  ^ nsc

]

{ #category : 'private' }
GsQuery >> _nsc: anNscOrSequenceableCollection [
  self resetCache.
  nsc := anNscOrSequenceableCollection

]

{ #category : 'private' }
GsQuery >> _queryResult [
  ^ queryResult

]

{ #category : 'testing' }
GsQuery >> _queryUsesIdentityIndex [
  "answer true if the one or more of the predicates has an evaluator that is an identity index"

  self _formula ifNil: [ ^ false ].
  ^ self _formula _queryUsesIdentityIndex

]

{ #category : 'testing' }
GsQuery >> _queryUsesIndex [
  "answer true if the one or more of the predicates has an evaluator that is a range or identity index"

  self _formula ifNil: [ ^ false ].
  ^ self _formula _queryUsesIndex

]

{ #category : 'testing' }
GsQuery >> _queryUsesRangeEqualityIndex [
  "answer true if the one or more of the predicates has an evaluator that is a range equality index"

  self _formula ifNil: [ ^ false ].
  ^ self _formula _queryUsesIndex and: [ self _formula _queryUsesIdentityIndex not ]

]

{ #category : 'private' }
GsQuery >> _validateQueryableCollection: anNscOrSequenceableCollection [
  "throw an error if the receiver does not support queries on anNscOrSequenceableCollection"

  anNscOrSequenceableCollection ifNil: [ ^ self ].
  anNscOrSequenceableCollection class _canCreateQueryOnInstances
    ifFalse: [
      ^ GsMalformedQueryExpressionError
        collectionIsNotQueryable:
          anNscOrSequenceableCollection class name asString
        senderLocationID: #'_validateQueryableCollection:' ]

]

{ #category : 'collection' }
GsQuery >> any [
  ^ self anyIfNone: [ ^ self error: 'this collection is empty' ]

]

{ #category : 'collection' }
GsQuery >> anyIfNone: noneBlock [
  ^ self detect: [ :each | ^ each ] ifNone: noneBlock

]

{ #category : 'converting' }
GsQuery >> asArray [
  ^ self queryResult asArray

]

{ #category : 'converting' }
GsQuery >> asIdentityBag [
  ^ self queryResult asIdentityBag

]

{ #category : 'querying' }
GsQuery >> bind: variableName to: value [
  "bind the given <value> to the <variableName> reference in the query predicate"

  self _formula: (self _formula bind: variableName to: value)

]

{ #category : 'private' }
GsQuery >> cacheQueryResult [
  ^ self queryOptions cacheQueryResult

]

{ #category : 'collection' }
GsQuery >> collect: collectBlock [
  | resultSet cachedResult |
  self formula isInvariant
    ifFalse: [ self error: 'formula should be invariant' ].
  self cacheQueryResult
    ifTrue: [
      queryResult ifNotNil: [ ^ queryResult collect: collectBlock ].
      cachedResult := nsc speciesForCollect new ].
  resultSet := nsc speciesForCollect new.
  self formula _checkForBug43764: self.
  self formula
    executeAndDo: [ :obj |
      | res |
      res := collectBlock value: obj.
      resultSet add: res.
      self cacheQueryResult
        ifTrue: [ cachedResult add: obj ] ].
  queryResult := cachedResult.
  ^ resultSet

]

{ #category : 'collection' }
GsQuery >> detect: detectBlock [
  ^ self
    detect: detectBlock
    ifNone: [ ^ self _error: #'assocErrNoElementsDetected' args: {detectBlock} ]

]

{ #category : 'collection' }
GsQuery >> detect: detectBlock ifNone: noneBlock [
  | cachedResult |
  self formula isInvariant
    ifFalse: [ self error: 'formula should be invariant' ].
  self cacheQueryResult
    ifTrue: [
      queryResult ifNotNil: [ ^ queryResult detect: detectBlock ].
      cachedResult := nsc species new ].
  self formula _checkForBug43764: self.
  self formula
    executeAndDo: [ :obj |
      (detectBlock value: obj)
        ifTrue: [ ^ obj ].
      self cacheQueryResult
        ifTrue: [ cachedResult add: obj ] ].
  queryResult := cachedResult.
  ^ noneBlock value

]

{ #category : 'collection' }
GsQuery >> do: doBlock [
  | cachedResult |
  self formula isInvariant
    ifFalse: [ self error: 'formula should be invariant' ].
  self cacheQueryResult
    ifTrue: [
      queryResult ifNotNil: [ ^ queryResult do: doBlock ].
      cachedResult := nsc species new ].
  self formula _checkForBug43764: self.
  self cacheQueryResult
    ifTrue: [
      self formula
        executeAndDo: [ :obj |
          doBlock value: obj.
          cachedResult add: obj ] ]
    ifFalse: [ self formula executeAndDo: [ :obj | doBlock value: obj ] ].
  queryResult := cachedResult.
  ^ self

]

{ #category : 'querying' }
GsQuery >> elementValue: each [
  "the pathTerms in the query will use anObject as the starting point ... analagous to an object in the nsc that is bound to a query"

  self formula _checkForBug43764: self.
  ^ self formula elementValue: each

]

{ #category : 'accessing' }
GsQuery >> formula [
  ^ self queryOptions autoOptimize
    ifTrue: [ self optimizedFormula ]
    ifFalse: [ formula ]

]

{ #category : 'private' }
GsQuery >> formula: aQueryFormula [
  | theFormula |
  originalFormula := aQueryFormula unbind.
  theFormula := self _nsc
    ifNil: [ aQueryFormula ]
    ifNotNil: [ aQueryFormula bindEvaluatorsFor: self _nsc collator: self _collator ].
  self _formula: theFormula

]

{ #category : 'private' }
GsQuery >> immediateInvariant [
  super immediateInvariant.
  formula immediateInvariant

]

{ #category : 'collection' }
GsQuery >> includes: anObject [
  ^ self queryResult includes: anObject

]

{ #category : 'collection' }
GsQuery >> isEmpty [
  ^ self queryResult isEmpty

]

{ #category : 'querying' }
GsQuery >> on: anNsc [
  "Bind the query to the given <anNsc>"

  self on: anNsc collator: nil

]

{ #category : 'querying' }
GsQuery >> on: anNscOrSequenceableCollection collator: anIcuCollator [
  "Bind the query to the given <anNsc> and <anIcuCollator>"

  self _validateQueryableCollection: anNscOrSequenceableCollection.
  self _nsc: anNscOrSequenceableCollection.
  self _collator: anIcuCollator.
  self _formula
    ifNotNil: [
      self
        _formula:
          (self _formula bindEvaluatorsFor: self _nsc collator: self _collator) ]

]

{ #category : 'querying' }
GsQuery >> optimize [
  self _formula: self optimizedFormula

]

{ #category : 'accessing' }
GsQuery >> optimizedFormula [
  ^ self optimizeFormulaUsing: self class defaultOptimizerClass

]

{ #category : 'private' }
GsQuery >> optimizeFormulaUsing: queryOptimizerClass [
  formula ifNil: [ ^ nil ].
  ^ queryOptimizerClass
    optimize: formula
    on: self _nsc
    options: self queryOptions

]

{ #category : 'querying' }
GsQuery >> optionalPathTerms [
  self _formula ifNotNil: [ self _formula: self _formula optionalPathTerms ]

]

{ #category : 'accessing' }
GsQuery >> originalFormula [
  ^ originalFormula

]

{ #category : 'printing' }
GsQuery >> printOn: aStream [
  self formula printOn: aStream

]

{ #category : 'accessing' }
GsQuery >> queryOptions [
  queryOptions ifNil: [ queryOptions := self class defaultQueryOptions ].
  ^ queryOptions

]

{ #category : 'private' }
GsQuery >> queryOptions: aGsQueryOptions [
  queryOptions := aGsQueryOptions.
  queryOptions optionalPathTerms
    ifTrue: [ formula := formula optionalPathTerms ]

]

{ #category : 'accessing' }
GsQuery >> queryResult [
  self formula isInvariant
    ifFalse: [ self error: 'formula should be invariant' ].
  ^ self cacheQueryResult
    ifTrue: [
      queryResult
        ifNil: [
          self formula _checkForBug43764: self.
          queryResult := self formula execute ].
      queryResult ]
    ifFalse: [
      self formula _checkForBug43764: self.
      self formula execute ]

]

{ #category : 'collection' }
GsQuery >> readStream [
  "Return a BtreeReadStream or RangeIndexReadStream on the query result"

  self formula isInvariant
    ifFalse: [ self error: 'formula should be invariant' ].
  self formula _checkForBug43764: self.
  self formula _checkForStreamableQuery: self.
  ^ self formula readStream

]

{ #category : 'collection' }
GsQuery >> reject: rejectBlock [
  | resultSet cachedResult |
  self formula isInvariant
    ifFalse: [ self error: 'formula should be invariant' ].
  self cacheQueryResult
    ifTrue: [
      queryResult ifNotNil: [ ^ queryResult reject: rejectBlock ].
      cachedResult := nsc species new ].
  resultSet := nsc species new.
  self formula _checkForBug43764: self.
  self formula
    executeAndDo: [ :obj |
      (rejectBlock value: obj)
        ifFalse: [ resultSet add: obj ].
      self cacheQueryResult
        ifTrue: [ cachedResult add: obj ] ].
  queryResult := cachedResult.
  ^ resultSet

]

{ #category : 'querying' }
GsQuery >> resetCache [
  queryResult := nil

]

{ #category : 'collection' }
GsQuery >> reversedReadStream [
  "Return a BtreeReadStream or RangeIndexReadStream on the query result"

  self formula isInvariant
    ifFalse: [ self error: 'formula should be invariant' ].
  self formula _checkForBug43764: self.
  self formula _checkForStreamableQuery: self.
  ^ self formula reversedReadStream

]

{ #category : 'collection' }
GsQuery >> select: selectBlock [
  | resultSet cachedResult |
  self formula isInvariant
    ifFalse: [ self error: 'formula should be invariant' ].
  self cacheQueryResult
    ifTrue: [
      queryResult ifNotNil: [ ^ queryResult select: selectBlock ].
      cachedResult := nsc species new ].
  resultSet := nsc species new.
  self formula _checkForBug43764: self.
  self formula
    executeAndDo: [ :obj |
      (selectBlock value: obj)
        ifTrue: [ resultSet add: obj ].
      self cacheQueryResult
        ifTrue: [ cachedResult add: obj ] ].
  queryResult := cachedResult.
  ^ resultSet

]

{ #category : 'collection' }
GsQuery >> size [
  ^ self queryResult size

]

{ #category : 'converting' }
GsQuery >> withOptimizedFormula [
  ^ self class new
    formula: self optimizedFormula immediateInvariant;
    on: self _nsc;
    queryOptions: self queryOptions copy;
    yourself

]

{ #category : 'converting' }
GsQuery >> withOriginalFormula [
  ^ self class new
    formula: self originalFormula immediateInvariant;
    on: self _nsc;
    queryOptions: self queryOptions copy;
    yourself

]
