Extension { #name : 'GsQueryFormula' }

{ #category : 'initialization' }
GsQueryFormula class >> _initialize [
  self _addClassVar: #ComparisonSelectorMap
     value: (
      SymbolDictionary new
        at: #'<' put: #'_idxForCompareLessThan:';
        at: #'>' put: #'_idxForCompareGreaterThan:';
        at: #'=' put: #'_idxForCompareEqualTo:';
        at: #'==' put: #'==';
        at: #'<=' put: #'_idxForCompareLessThanOrEqualTo:';
        at: #'>=' put: #'_idxForCompareGreaterThanOrEqualTo:';
        at: #'~=' put: #'_idxForCompareNotEqualTo:';
        at: #'~~' put: #'~~';
        yourself ).
  self _addClassVar: #InverseOperatorMap
      value: (SymbolDictionary new
        at: #'<' put: #'>';
        at: #'>' put: #'<';
        at: #'=' put: #'=';
        at: #'==' put: #'==';
        at: #'<=' put: #'>=';
        at: #'>=' put: #'<=';
        at: #'~=' put: #'~=';
        at: #'~~' put: #'~~';
        yourself ).
   self _addClassVar: #NegatedOperatorMap
      value: ( SymbolDictionary new
        at: #'<' put: #'>=';
        at: #'>' put: #'<=';
        at: #'=' put: #'~=';
        at: #'==' put: #'~~';
        at: #'<=' put: #'>';
        at: #'>=' put: #'<';
        at: #'~=' put: #'=';
        at: #'~~' put: #'==';
        at: #'unary' put: #'unaryNot';
        at: #'unaryNot' put: #'unary';
        yourself ).

]

{ #category : 'initialization' }
GsQueryFormula class >> comparisonSelectorFor: operator [
  "Returns the #_idxForCompare"

  ^ ComparisonSelectorMap at: operator

]

{ #category : 'accessing' }
GsQueryFormula class >> inverseOperatorFor: operator [
  "Returns the operator that is the inverse of the given operator.
 This is used to reverse the comparison order of operands in a predicate.
 For example, the inverse of <= is >=; the inverse of = is = (no change)."

  ^ InverseOperatorMap at: operator

]

{ #category : 'accessing' }
GsQueryFormula class >> negatedOperatorFor: operator [
  "Returns the operator that is the negation of the given operator.
 This is used to implement executeClauseNegated.
 For example, the negation of <= is >; the inverse of = is ~= ."

  ^ NegatedOperatorMap at: operator

]

{ #category : 'private' }
GsQueryFormula >> _bindEvaluator: anNsc for: path isRangeEqualityOperation: isRangeOperation [
  | pathEvaluator iList theIndexObj legacyIndexesOnNsc |
  pathEvaluator := self pathEvaluatorFor: path.
  iList := anNsc _indexedPaths.
  iList
    ifNil: [
      "no indexes defined on nsc ... do it the hard way"
      ^ pathEvaluator nsc: anNsc ].
  legacyIndexesOnNsc := false.
  1 to: iList size by: 2 do: [ :i |
    | indexObj |
    indexObj := iList at: i.
    legacyIndexesOnNsc := indexObj isLegacyIndex. "all indexes on an nsc are either legacy or not"
    (iList at: i + 1) == 1
      ifTrue: [
        " see if the index matches the path "
        (indexObj hasIndexOnPath: pathEvaluator)
          ifTrue: [
            " if range index found for range operation or
          identity index found for identity operation "
            ((isRangeOperation and: [ indexObj isRangeEqualityIndex ])
              or: [ isRangeOperation not and: [ indexObj isIdentityIndex ] ])
              ifTrue: [
                (self _canIndexSatifyPathTermsRequired: indexObj)
                  ifTrue: [ ^ indexObj ] ]
              ifFalse: [ theIndexObj := indexObj ] ] ] ].
  isRangeOperation
    ifTrue: [
      "no range index found ... do it the hard way"
      ^ pathEvaluator nsc: anNsc ]
    ifFalse: [
      theIndexObj ~~ nil
        ifTrue: [
          (self _canIndexSatifyPathTermsRequired: theIndexObj)
            ifTrue: [
              "index found for non-range query, go ahead and use it"
              ^ theIndexObj ] ] ].
  theIndexObj := IdentityIndex new.
  theIndexObj
    indexDictionary: nsc _indexDictionary;
    nscRoot: anNsc.
  iList _putAllCommonPathTermsForPathArray: pathEvaluator into: theIndexObj.
  theIndexObj size == pathEvaluator size
    ifTrue: [
      (self _canIndexSatifyPathTermsRequired: theIndexObj)
        ifTrue: [
          legacyIndexesOnNsc
            ifTrue: [ ^ theIndexObj ]
          ifFalse: [
            "eventually this should be conditional on a GsIndexOption ... for now it's naked"
            GsQueryExpectedImplicitIdentityIndexError signal: self ] ] ].
  ^ pathEvaluator nsc: anNsc	"no qualifying identity indexes found"

]

{ #category : 'testing' }
GsQueryFormula >> _canIndexSatifyPathTermsRequired: indexObj [
  pathTermsRequired
    ifNil: [
      "not set ... use index if at all possible"
      ^ true ].
  pathTermsRequired
    ifTrue: [
      indexObj termsRequired
        ifTrue: [ ^ true ] ]
    ifFalse: [
      indexObj termsRequired
        ifFalse: [ ^ true ] ].
  ^ false

]

{ #category : 'private' }
GsQueryFormula >> _checkForBug43764: aGsQuery [
  "Bug 43764 describes a query pattern involving collection-valued path terms that produce
   incorrect query results. See the 3.2 Release Notes for more detailed information."

  | queryOptions |
  self _exposedToBug43764
    ifFalse: [ ^ self	"not exposed" ].
  queryOptions := aGsQuery queryOptions.
  [ GsQueryConjoinSetChecker check: self on: aGsQuery _nsc ]
    on: GsMalformedQueryExpressionError
    do: [ :ex |
      "checker will throw an error if the query is exposed to bug 43764"
      (queryOptions autoOptimize
        and: [ queryOptions applyDeMorgansLaws and: [ queryOptions consolidateRangePredicates ] ])
        ifTrue: [
          "exposed to bug ... pass the exception"
          ex pass ]
        ifFalse: [
          [
          | newQuery newQueryOptions |
          "determine if query is evaluable with proper options set"
          newQueryOptions := queryOptions + GsQueryOptions applyDeMorgansLaws
            + GsQueryOptions consolidateRangePredicates
            + GsQueryOptions autoOptimize.
          newQuery := aGsQuery copy.
          newQuery queryOptions: newQueryOptions.
          GsQueryConjoinSetChecker check: newQuery formula on: aGsQuery _nsc ]
            on: GsMalformedQueryExpressionError
            do: [ :ex2 |
              "exposed to bug ... pass the exception"
              ex2 pass ] ].
      GsMalformedQueryExpressionError
        bug43764QueryMustBeOptimized: #'_checkForBug43764:' ].
  ^ self	"not exposed"

]

{ #category : 'private' }
GsQueryFormula >> _checkForStreamableQuery: aGsQuery [
  "Signal an error if the formula is not suited for streaming over result set"

  "a streamable query is not exposed to bug 43764, so no need to check"

  | queryOptions |
  queryOptions := aGsQuery queryOptions.
  [ GsStreamableQueryChecker check: self on: aGsQuery _nsc ]
    on: GsMalformedQueryExpressionError
    do: [ :ex |
      "checker will throw an error if the query cannot be streamed"
      (queryOptions autoOptimize
        and: [ queryOptions applyDeMorgansLaws and: [ queryOptions consolidateRangePredicates ] ])
        ifTrue: [
          "cannot be streamed"
          ex pass ]
        ifFalse: [
          [
          | newQuery newQueryOptions |
          "determine if query is evaluable with proper options set"
          newQueryOptions := queryOptions + GsQueryOptions applyDeMorgansLaws
            + GsQueryOptions consolidateRangePredicates
            + GsQueryOptions autoOptimize.
          newQuery := aGsQuery copy.
          newQuery queryOptions: newQueryOptions.
          GsStreamableQueryChecker check: newQuery formula on: aGsQuery _nsc ]
            on: GsMalformedQueryExpressionError
            do: [ :ex2 |
              "cannot stream ... pass the exception"
              ex2 pass ] ].
      GsMalformedQueryExpressionError
        streamableQueryMustBeOptimized: #'_checkForStreamableQuery:' ].
  ^ self	"streamable query"

]

{ #category : 'testing' }
GsQueryFormula >> _evaluatorCanStreamQueries [
  "Answer true if the evaluator(s) support streaming operations"

  self _evaluators
    do: [ :each |
      each isStreamable
        ifFalse: [ ^ false ] ].
  ^ true

]

{ #category : 'private' }
GsQueryFormula >> _evaluators [
  "return list of evaluators associated with predicate"

  ^ #()

]

{ #category : 'private' }
GsQueryFormula >> _exposedToBug43764 [
  "Bug 43764 describes a query pattern involving collection-valued path terms that produce
   incorrect query results. See the 3.2 Release Notes for more detailed information."

  | exposed |
  exposed := false.
  self
    _predicateAndEvaluatorsDo: [ :predicate :evaluators |
      (predicate usesComparisonOperation and: [ predicate usesRangeOperation not ])
        ifTrue: [
          "potentially exposed if predicate uses comparison operators, and is not a
           range predicate"
          evaluators
            do: [ :evaluator |
              evaluator hasCollectionBasedTerm
                ifTrue: [
                  exposed
                    ifTrue: [
                      "exposed if receiver has two or more collection based path terms"
                      ^ true ].
                  exposed := true ] ] ] ].
  ^ false

]

{ #category : 'accessing' }
GsQueryFormula >> _nsc [
  ^ nsc

]

{ #category : 'accessing' }
GsQueryFormula >> _nsc: anNsc collator: anIcuCollatorOrNil [
  nsc := anNsc.
  collator := anIcuCollatorOrNil

]

{ #category : 'private' }
GsQueryFormula >> _parsePath: path [
  "strip each off terms"

  | terms |
  terms := path asArrayOf32PathTerms.
  terms := terms size = 1
    ifTrue: [ #(#'') ]
    ifFalse: [ terms copyFrom: 2 to: terms size ].
  ^ terms

]

{ #category : 'accessing' }
GsQueryFormula >> _pathTerms [
  ^ self _parsePath: self path

]

{ #category : 'accessing' }
GsQueryFormula >> _pathTermsRequired [
  ^ pathTermsRequired

]

{ #category : 'accessing' }
GsQueryFormula >> _pathTermsRequired: aBoolOrNil [
  "set to true, if you want to quarantee that path terms are required during query
   evaluation. If an index is available, but the index uses optional path terms,
   then the query will use brute force evaluation."

  "set to false if you want to guaranatee that path terms are optional during query
   evaluation. If an index is available, but the index uses required path terms,
   then the query will use brute force evaluation."

  "set to nil (default) if you do not care. Indexes will be used if available
   otherwise brute force evaluation will be used with optional path terms."

  pathTermsRequired := aBoolOrNil

]

{ #category : 'private' }
GsQueryFormula >> _predicateAndEvaluatorsDo: aBlock [
  "visit all predicates in receiver and provide access to predicate and list of evaluators
   for the predicate"

  aBlock value: self value: self _evaluators

]

{ #category : 'testing' }
GsQueryFormula >> _queryUsesIdentityIndex [
  "answer true if the one or more of the predicates has an evaluator that is an identity index"

  self _evaluators
    do: [ :each |
      (each isPathEvaluator not
        and: [ each isIdentityIndex ])
        ifTrue: [ ^ true ] ].
  ^ false

]

{ #category : 'testing' }
GsQueryFormula >> _queryUsesIndex [
  "answer true if the one or more of the predicates has an evaluator that is a range or identity index"

  self _evaluators
    do: [ :each |
      (each isPathEvaluator not
        and: [ each isIdentityIndex or: [ each isRangeEqualityIndex ] ])
        ifTrue: [ ^ true ] ].
  ^ false

]

{ #category : 'operators' }
GsQueryFormula >> & aClause [
  ^ GsCompoundClause clause: self operator: #'&' clause: aClause

]

{ #category : 'operators' }
GsQueryFormula >> | aClause [
  ^ GsCompoundClause clause: self operator: #'|' clause: aClause

]

{ #category : 'visitors' }
GsQueryFormula >> acceptVisitor: aFormulaVisitor [
  aFormulaVisitor acceptFormula: self

]

{ #category : 'optimizing' }
GsQueryFormula >> applyDeMorgansNegationTransform [
  "actively apply De Morgan's laws"

  self subclassResponsibility: #'applyDeMorgansNegationTransform'

]

{ #category : 'optimizing' }
GsQueryFormula >> applyDeMorgansTransform [
  "do not transform, but propagate the transform"

  self subclassResponsibility: #'applyDeMorgansTransform'

]

{ #category : 'converting' }
GsQueryFormula >> asFormula [
  ^ self

]

{ #category : 'converting' }
GsQueryFormula >> asFormulaWithSelectorParts: selectorParts [
  ^ GsUnaryClause clause: self operator: selectorParts first inputValue asSymbol

]

{ #category : 'converting' }
GsQueryFormula >> asFormulaWithSelectorParts: selectorParts arguments: arguments [
  ^ self subclassResponsibility: #asFormulaWithSelectorParts:arguments:

]

{ #category : 'converting' }
GsQueryFormula >> asQuery [
  ^ GsQuery fromFormula: self

]

{ #category : 'transforming' }
GsQueryFormula >> bind: variableName to: value [
  self subclassResponsibility: #'bind:to:'

]

{ #category : 'private' }
GsQueryFormula >> bindEvaluators [
  self immediateInvariant

]

{ #category : 'transforming' }
GsQueryFormula >> bindEvaluatorsFor: anNsc collator: anIcuCollatorOrNil [
  | bound |
  bound := self copy.
  anNsc ifNil: [ ^ bound ].
  bound _nsc: anNsc collator: anIcuCollatorOrNil.
  bound bindEvaluators.
  ^ bound

]

{ #category : 'private' }
GsQueryFormula >> buildGsQueryParserNodeFor: aGsQueryParser [
  aGsQueryParser build: self messages: nil

]

{ #category : 'optimizing' }
GsQueryFormula >> canConsolidateEnumeratedWith: secondaryPredicate [
  ^ false

]

{ #category : 'optimizing' }
GsQueryFormula >> canConsolidateEnumeratedWithPathConstantPredicate: primaryPredicate [
  ^ false

]

{ #category : 'optimizing' }
GsQueryFormula >> canConsolidateWith: secondaryPredicate [
  ^ false

]

{ #category : 'optimizing' }
GsQueryFormula >> canConsolidateWithPathConstantPredicate: primaryPredicate [
  ^ false

]

{ #category : 'transforming' }
GsQueryFormula >> collapseToRangePredicateForConstantPathPredicate: aPredicate [
  "default is to and the two predicates together, last half of double dispatch so reverse order"

  ^ aPredicate & self

]

{ #category : 'transforming' }
GsQueryFormula >> collapseToRangePredicateForPathConstantPredicate: aPredicate [
  "default is to and the two predicates together, last half of double dispatch so reverse order"

  ^ aPredicate & self

]

{ #category : 'transforming' }
GsQueryFormula >> collapseToRangePredicateIfPossible: aPathXPredicate [
  "default is to and the two predicates together"

  ^ self & aPathXPredicate

]

{ #category : 'accessing' }
GsQueryFormula >> collator [
  "Returns IcuCollator to be used when comparing Unicode strings"

  ^ collator

]

{ #category : 'accessing' }
GsQueryFormula >> comparisonSelectorFor: operator [
  ^ self class comparisonSelectorFor: operator

]

{ #category : 'accessing' }
GsQueryFormula >> constantReferenceFor: aName [
  ^ GsConstantReferenceAssociation newWithKey: aName value: nil

]

{ #category : 'querying-private' }
GsQueryFormula >> elementValue: anObject [
  "the pathTerms in the query will use anObject as the starting point ... analagous to an object in the nsc that is bound to a query"

  self subclassResponsibility: #'elementValue:'

]

{ #category : 'querying' }
GsQueryFormula >> execute [
  ^ self executeClause

]

{ #category : 'querying' }
GsQueryFormula >> executeAndDo: aBlock [
  self subclassResponsibility: #'executeAndDo:'

]

{ #category : 'querying-private' }
GsQueryFormula >> executeClause [
  self subclassResponsibility: #'executeClause'

]

{ #category : 'querying-private' }
GsQueryFormula >> executeClauseNegated [
  "used when not encountered"

  self subclassResponsibility: #'executeClauseNegated'

]

{ #category : 'querying-private' }
GsQueryFormula >> executeClauseNegatedOn: anNsc [
  | bound |
  bound := self bindEvaluatorsFor: anNsc collator: self collator.
  ^ bound executeClauseNegated

]

{ #category : 'querying-private' }
GsQueryFormula >> executeClauseOn: anNsc [
  | bound |
  bound := self bindEvaluatorsFor: anNsc collator: self collator.
  ^ bound executeClause

]

{ #category : 'querying-private' }
GsQueryFormula >> executeNegatedAndDo: aBlock [
  self subclassResponsibility: #'executeNegatedAndDo:'

]

{ #category : 'accessing' }
GsQueryFormula >> inverseOperatorFor: operator [
  ^ self class inverseOperatorFor: operator

]

{ #category : 'testing' }
GsQueryFormula >> isCompoundPredicate [
  ^ false

]

{ #category : 'testing' }
GsQueryFormula >> isConjunctiveNormalForm [
  ^ true

]

{ #category : 'testing' }
GsQueryFormula >> isConstantConstant [
  ^ false

]

{ #category : 'testing' }
GsQueryFormula >> isDisjunctiveClause [
  ^ false

]

{ #category : 'testing' }
GsQueryFormula >> isDisjunctiveNormalForm [
  ^ true

]

{ #category : 'testing' }
GsQueryFormula >> isNegationClause [
  ^ false

]

{ #category : 'testing' }
GsQueryFormula >> isPathPath [
  ^ false

]

{ #category : 'testing' }
GsQueryFormula >> isPredicate [
  ^ false

]

{ #category : 'testing' }
GsQueryFormula >> isRangeEqualityOperation: queryOp [
  "Returns true if the given search operation is one of < > = <= >= or ~=."

  ^ queryOp ~~ #'==' and: [ queryOp ~~ #'~~' ]

]

{ #category : 'accessing' }
GsQueryFormula >> negatedOperatorFor: operator [
  ^ self class negatedOperatorFor: operator

]

{ #category : 'operators' }
GsQueryFormula >> normalize [
  ^ self copy

]

{ #category : 'operators' }
GsQueryFormula >> not [
  ^ GsUnaryClause clause: self operator: #'not'

]

{ #category : 'accessing' }
GsQueryFormula >> operationSelectors [
  "Returns an Array of comparison operation selectors whose ordering matches the order used by the QueryExecutor."

  ^ #(#'<' #'>' #'=' #'==' #'<=' #'>=' #'~=' #'~~' #'unary' #'unaryNot')

]

{ #category : 'accessing' }
GsQueryFormula >> optionalPathTerms [
  | bound |
  bound := self copy _pathTermsRequired: false.
  nsc ifNil: [ ^ bound immediateInvariant ].
  ^ bound bindEvaluators

]

{ #category : 'accessing' }
GsQueryFormula >> pathEvaluatorClass [
  pathTermsRequired ifNil: [ ^ PathEvaluator ].
  ^ pathTermsRequired
    ifTrue: [ PathEvaluator ]
    ifFalse: [ OptionalTermPathEvaluator ]

]

{ #category : 'accessing' }
GsQueryFormula >> pathEvaluatorFor: path [
  | pathEvaluator terms holder |
  pathEvaluator := self pathEvaluatorClass basicNew.
  terms := self _parsePath: path.
  terms do: [ :term | pathEvaluator add: term ].
  pathEvaluator collator: self collator.
  pathEvaluator initialize .
  holder := { pathEvaluator }.
  pathEvaluator := nil .
  ^ PathEvaluator asMostSpecificType: holder .

]

{ #category : 'accessing' }
GsQueryFormula >> pathReferenceFor: aName [
  ^ GsQueryPathReferenceAssociation newWithKey: aName value: nil

]

{ #category : 'querying-private' }
GsQueryFormula >> prepareForExecution [
  "Return an object equivalent to the receiver that is prepared for execution within
   the context of a block."

  ^ self

]

{ #category : 'querying' }
GsQueryFormula >> query: anNsc [
  ^ GsQuery new
    on: anNsc;
    formula: self;
    yourself

]

{ #category : 'querying' }
GsQueryFormula >> readStream [
  self subclassResponsibility: #'readStream'

]

{ #category : 'optimizing' }
GsQueryFormula >> redundantDisjunctivePredicateBetween: secondaryPredicate [
  ^ nil

]

{ #category : 'optimizing' }
GsQueryFormula >> redundantDisjunctivePredicateBetweenPathConstant: primaryPredicate [
  ^ nil

]

{ #category : 'optimizing' }
GsQueryFormula >> redundantPredicateBetween: secondaryPredicate [
  ^ nil

]

{ #category : 'optimizing' }
GsQueryFormula >> redundantPredicateBetweenPathConstant: primaryPredicate [
  ^ nil

]

{ #category : 'querying' }
GsQueryFormula >> reversedReadStream [
  ^ self readStream reversed

]

{ #category : 'optimizing' }
GsQueryFormula >> transformCommonPaths [
  "If the predicate is a path-path and path paths are the same:
	- if operator is =, ==, <=, >= predicate will always be true.
	  Transform  to unary true query"

  ^ self

]

{ #category : 'transforming' }
GsQueryFormula >> unbind [
  "remove all bindings"

  | unbound |
  unbound := self copy.
  unbound _nsc: nil collator: nil.
  ^ unbound

]

{ #category : 'testing' }
GsQueryFormula >> usesComparisonOperation [
  ^ false

]

{ #category : 'testing' }
GsQueryFormula >> usesEqualityOperation [
  ^ false

]

{ #category : 'testing' }
GsQueryFormula >> usesEqualOperation [
  ^ false

]

{ #category : 'testing' }
GsQueryFormula >> usesIdenticalToOperation [
  ^ false

]

{ #category : 'testing' }
GsQueryFormula >> usesIdentityOperation [
  ^ false

]

{ #category : 'testing' }
GsQueryFormula >> usesPathEvaluator [
  ^ false

]

{ #category : 'testing' }
GsQueryFormula >> usesPathEvaluatorForFirstClause [
  "only real distinction from usesPathEvaluator is for GsCompoundClause"

  ^ self usesPathEvaluator

]

{ #category : 'testing' }
GsQueryFormula >> usesRangeOperation [
  ^ false

]

{ #category : 'accessing' }
GsQueryFormula >> variableReferenceFor: aName [
  ^ GsVariableReferenceAssociation newWithKey: aName value: nil

]
