Extension { #name : 'GsQueryOptions' }

{ #category : 'options' }
GsQueryOptions class >> applyDeMorgansLaws [
  "Apply De Morgan's Laws to eliminate `not` in queries:

    The negation of a conjunction is the disjunction of the negations.
    The negation of a disjunction is the conjunction of the negations.

    Where:

          (each.firstName = 'Dale') not

        becomes:

          (each.firstName ~= 'Dale')

    and:

          ((1 <= each.numberOfChildren) & (each.numberOfChildren <= 4)) not

        becomes:

          (each.numberOfChildren < 1) | (each.numberOfChildren > 4)
    and:

          ((1 > each.numberOfChildren) | (each.numberOfChildren > 4)) not

        becomes:

          (1 <= each.numberOfChildren) & (each.numberOfChildren <= 4)
"

  ^ self new
    setApplyDeMorgansLaws;
    yourself

]

{ #category : 'options' }
GsQueryOptions class >> autoOptimize [
  "Query formula is automatically optimized"

  ^ self new
    setAutoOptimize;
    yourself

]

{ #category : 'options' }
GsQueryOptions class >> cacheQueryResult [
  "Cache query result"

  ^ self new
    setCacheQueryResult;
    yourself

]

{ #category : 'options' }
GsQueryOptions class >> consolidateEnumerablePredicates [
  "Consolidate path-constant predicates to single enumerated predicate
      if an index exists that has an enumerated path term and predicates
      using the enumerated path  term can be combined into a single
      predicate

          (each.firstName = 'Martin') | (each.lastName = 'Martin')

        becomes:

          (each.firstName|lastName = 'Martin')"

  ^ self new
    setConsolidateEnumerablePredicates;
    yourself

]

{ #category : 'options' }
GsQueryOptions class >> consolidateRangePredicates [
  "Convert 2 path-constant predicates into a single range predicate when possible

          (each.age > 4) & (each.age < 19)

        becomes:

          (4 < each.age < 19)"

  ^ self new
    setConsolidateRangePredicates;
    yourself

]

{ #category : 'options' }
GsQueryOptions class >> consolidateUnaryConstantPredicates [
  "Simplify formulas involving predicates that are unary-constant (true or false) or
   constant-constant. The predicate is removed, or the expression is consolidated.

  (true) & <other predicates>
    becomes:
  <other predicates>

  (true) | <otherpredicates>
    becomes:
  (true)

  (false) & <other predicates>
    becomes:
  (false)

  (false) | <other predicates>
    becomes:
  <other predicates>

  (true) & (each.name = 'Dale') & (each.gender == #male)
    becomes:
  (each.name = 'Dale') & (each.gender == #male)"

  ^ self new
    setConsolidateUnaryConstantPredicates;
    yourself

]

{ #category : 'options' }
GsQueryOptions class >> default [
  "all options set except cacheQueryResult"

  ^ self optimizerOptions + self autoOptimize

]

{ #category : 'options' }
GsQueryOptions class >> normalizePredicates [
  "Replace constant-path predicates with equivalent path-constant predicates.

          (19 > each.age)

        becomes:

          (each.age < 19)"

  ^ self new
    setNormalizePredicates;
    yourself

]

{ #category : 'options' }
GsQueryOptions class >> optimizerOptions [
  "all options that control operation of optimizer"

  ^ self new
    setApplyDeMorgansLaws;
    setConsolidateEnumerablePredicates;
    setConsolidateRangePredicates;
    setConsolidateUnaryConstantPredicates;
    setNormalizePredicates;
    setRemoveRedundantPredicates;
    setReorderPredicates;
    setTransformCommonPaths

]

{ #category : 'Instance Creation' }
GsQueryOptions class >> optionalPathTerms [
  "Do not signal an error if instance variable missing along index path."

  ^ self new
    setOptionalPathTerms;
    yourself

]

{ #category : 'options' }
GsQueryOptions class >> removeRedundantPredicates [
  "Eliminate redundant predicates. i.e., predicates that fall within range
	of other predicates

          (each.age < 19) & (each.age < 4)

        becomes:

          (each.age < 4)"

  ^ self new
    setRemoveRedundantPredicates;
    yourself

]

{ #category : 'options' }
GsQueryOptions class >> reorderPredicates [
  "Reorder predicates based on following ordering rules:
        constant predicates
        indexed predicates
        identity comparison predicates
        equality comparison predicates
        all others

          (each.age <= 21) & (each.gender == #male) &
          (each.name = 'Dale') & (each.father = each.father)

        assuming each.name is indexed and each.age is not indexed, becomes:

          (true) & (each.name = 'Dale') &
          (each.gender == #male) & (each.age <= 21)"

  ^ self new
    setReorderPredicates;
    yourself

]

{ #category : 'options' }
GsQueryOptions class >> transformCommonPaths [
  "Convert predicates with common path-path operands to an equivalent
	constant predicate.

          (each.firstName = each.firstName)

        becomes:

          (true)"

  ^ self new
    setTransformCommonPaths;
    yourself

]

{ #category : 'Comparison' }
GsQueryOptions >> = aGsQueryOptions [
  (aGsQueryOptions isKindOf: self class)
    ifFalse: [ ^ false ].
  ^ self applyDeMorgansLaws == aGsQueryOptions applyDeMorgansLaws
    and: [
      self autoOptimize == aGsQueryOptions autoOptimize
        and: [
          self cacheQueryResult == aGsQueryOptions cacheQueryResult
            and: [
              self consolidateEnumerablePredicates
                == aGsQueryOptions consolidateEnumerablePredicates
                and: [
                  self consolidateRangePredicates == aGsQueryOptions consolidateRangePredicates
                    and: [
                      self normalizePredicates == aGsQueryOptions normalizePredicates
                        and: [
                          self removeRedundantPredicates == aGsQueryOptions removeRedundantPredicates
                            and: [
                              self reorderPredicates == aGsQueryOptions reorderPredicates
                                and: [
                                  self transformCommonPaths == aGsQueryOptions transformCommonPaths
                                    and: [
                                      self consolidateUnaryConstantPredicates
                                        ==
                                          aGsQueryOptions consolidateUnaryConstantPredicates
                                        and: [ self optionalPathTerms == aGsQueryOptions optionalPathTerms ] ] ] ] ] ] ] ] ] ]

]

{ #category : 'Accessing' }
GsQueryOptions >> applyDeMorgansLaws [
  | val |
  val := self dynamicInstVarAt: #'applyDeMorgansLaws'.
  val ifNil: [ ^ false ].
  ^ val

]

{ #category : 'Accessing' }
GsQueryOptions >> applyDeMorgansLaws: aBoolean [
  self dynamicInstVarAt: #'applyDeMorgansLaws' put: aBoolean

]

{ #category : 'Accessing' }
GsQueryOptions >> autoOptimize [
  | val |
  val := self dynamicInstVarAt: #'autoOptimize'.
  val ifNil: [ ^ false ].
  ^ val

]

{ #category : 'Accessing' }
GsQueryOptions >> autoOptimize: aBoolean [
  self dynamicInstVarAt: #'autoOptimize' put: aBoolean

]

{ #category : 'Accessing' }
GsQueryOptions >> cacheQueryResult [
  | val |
  val := self dynamicInstVarAt: #'cacheQueryResult'.
  val ifNil: [ ^ false ].
  ^ val

]

{ #category : 'Accessing' }
GsQueryOptions >> cacheQueryResult: aBoolean [
  self dynamicInstVarAt: #'cacheQueryResult' put: aBoolean

]

{ #category : 'Operators' }
GsQueryOptions >> clearApplyDeMorgansLaws [
  self applyDeMorgansLaws: false

]

{ #category : 'Operators' }
GsQueryOptions >> clearAutoOptimize [
  self autoOptimize: false

]

{ #category : 'Operators' }
GsQueryOptions >> clearCacheQueryResult [
  self cacheQueryResult: false

]

{ #category : 'Operators' }
GsQueryOptions >> clearConsolidateEnumerablePredicates [
  self consolidateEnumerablePredicates: false

]

{ #category : 'Operators' }
GsQueryOptions >> clearConsolidateRangePredicates [
  self consolidateRangePredicates: false

]

{ #category : 'Operators' }
GsQueryOptions >> clearConsolidateUnaryConstantPredicates [
  self consolidateUnaryConstantPredicates: false

]

{ #category : 'Operators' }
GsQueryOptions >> clearNormalizePredicates [
  self normalizePredicates: false

]

{ #category : 'Operators' }
GsQueryOptions >> clearOptionalPathTerms [
  self optionalPathTerms: false

]

{ #category : 'Operators' }
GsQueryOptions >> clearRemoveRedundantPredicates [
  self removeRedundantPredicates: false

]

{ #category : 'Operators' }
GsQueryOptions >> clearReorderPredicates [
  self reorderPredicates: false

]

{ #category : 'Operators' }
GsQueryOptions >> clearTransformCommonPaths [
  self transformCommonPaths: false

]

{ #category : 'Operators' }
GsQueryOptions >> combineWith: aGsQueryOptions [
  self applyDeMorgansLaws
    ifTrue: [ aGsQueryOptions setApplyDeMorgansLaws ].
  self autoOptimize
    ifTrue: [ aGsQueryOptions setAutoOptimize ].
  self cacheQueryResult
    ifTrue: [ aGsQueryOptions setCacheQueryResult ].
  self consolidateEnumerablePredicates
    ifTrue: [ aGsQueryOptions setConsolidateEnumerablePredicates ].
  self consolidateRangePredicates
    ifTrue: [ aGsQueryOptions setConsolidateRangePredicates ].
  self normalizePredicates
    ifTrue: [ aGsQueryOptions setNormalizePredicates ].
  self removeRedundantPredicates
    ifTrue: [ aGsQueryOptions setRemoveRedundantPredicates ].
  self consolidateUnaryConstantPredicates
    ifTrue: [ aGsQueryOptions setConsolidateUnaryConstantPredicates ].
  self reorderPredicates
    ifTrue: [ aGsQueryOptions setReorderPredicates ].
  self transformCommonPaths
    ifTrue: [ aGsQueryOptions setTransformCommonPaths ].
  self optionalPathTerms
    ifTrue: [ aGsQueryOptions setOptionalPathTerms ].
  ^ aGsQueryOptions

]

{ #category : 'Accessing' }
GsQueryOptions >> consolidateEnumerablePredicates [
  | val |
  val := self dynamicInstVarAt: #'consolidateEnumerablePredicates'.
  val ifNil: [ ^ false ].
  ^ val

]

{ #category : 'Accessing' }
GsQueryOptions >> consolidateEnumerablePredicates: aBoolean [
  self dynamicInstVarAt: #'consolidateEnumerablePredicates' put: aBoolean

]

{ #category : 'Accessing' }
GsQueryOptions >> consolidateRangePredicates [
  | val |
  val := self dynamicInstVarAt: #'consolidateRangePredicates'.
  val ifNil: [ ^ false ].
  ^ val

]

{ #category : 'Accessing' }
GsQueryOptions >> consolidateRangePredicates: aBoolean [
  self dynamicInstVarAt: #'consolidateRangePredicates' put: aBoolean

]

{ #category : 'Accessing' }
GsQueryOptions >> consolidateUnaryConstantPredicates [
  | val |
  val := self dynamicInstVarAt: #'consolidateUnaryConstantPredicates'.
  val ifNil: [ ^ false ].
  ^ val

]

{ #category : 'Accessing' }
GsQueryOptions >> consolidateUnaryConstantPredicates: aBoolean [
  self dynamicInstVarAt: #'consolidateUnaryConstantPredicates' put: aBoolean

]

{ #category : 'Comparison' }
GsQueryOptions >> hash [
  ^ self applyDeMorgansLaws hash
    bitXor:
      ((self autoOptimize hash bitShift: 1)
        bitXor:
          ((self cacheQueryResult hash bitShift: 2)
            bitXor:
              ((self consolidateEnumerablePredicates hash bitShift: 3)
                bitXor:
                  ((self consolidateRangePredicates hash bitShift: 4)
                    bitXor:
                      ((self normalizePredicates hash bitShift: 5)
                        bitXor:
                          ((self removeRedundantPredicates hash bitShift: 6)
                            bitXor:
                              ((self reorderPredicates hash bitShift: 7)
                                bitXor:
                                  ((self transformCommonPaths hash bitShift: 8)
                                    bitXor:
                                      (self consolidateUnaryConstantPredicates hash bitShift: 9)))))))))

]

{ #category : 'Initialize-Release' }
GsQueryOptions >> initialize [
  super initialize.
  self applyDeMorgansLaws: false.
  self autoOptimize: false.
  self cacheQueryResult: false.
  self consolidateEnumerablePredicates: false.
  self consolidateRangePredicates: false.
  self normalizePredicates: false.
  self removeRedundantPredicates: false.
  self consolidateUnaryConstantPredicates: false.
  self reorderPredicates: false.
  self transformCommonPaths: false.
  self optionalPathTerms: false

]

{ #category : 'Operators' }
GsQueryOptions >> negateWith: aGsQueryOptions [
  self applyDeMorgansLaws
    ifTrue: [ aGsQueryOptions clearApplyDeMorgansLaws ].
  self autoOptimize
    ifTrue: [ aGsQueryOptions clearAutoOptimize ].
  self cacheQueryResult
    ifTrue: [ aGsQueryOptions clearCacheQueryResult ].
  self consolidateEnumerablePredicates
    ifTrue: [ aGsQueryOptions clearConsolidateEnumerablePredicates ].
  self consolidateRangePredicates
    ifTrue: [ aGsQueryOptions clearConsolidateRangePredicates ].
  self normalizePredicates
    ifTrue: [ aGsQueryOptions clearNormalizePredicates ].
  self removeRedundantPredicates
    ifTrue: [ aGsQueryOptions clearRemoveRedundantPredicates ].
  self consolidateUnaryConstantPredicates
    ifTrue: [ aGsQueryOptions clearConsolidateUnaryConstantPredicates ].
  self reorderPredicates
    ifTrue: [ aGsQueryOptions clearReorderPredicates ].
  self transformCommonPaths
    ifTrue: [ aGsQueryOptions clearTransformCommonPaths ].
  self optionalPathTerms
    ifTrue: [ aGsQueryOptions clearOptionalPathTerms ].
  ^ aGsQueryOptions

]

{ #category : 'Accessing' }
GsQueryOptions >> normalizePredicates [
  | val |
  val := self dynamicInstVarAt: #'normalizePredicates'.
  val ifNil: [ ^ false ].
  ^ val

]

{ #category : 'Accessing' }
GsQueryOptions >> normalizePredicates: aBoolean [
  self dynamicInstVarAt: #'normalizePredicates' put: aBoolean

]

{ #category : 'Accessing' }
GsQueryOptions >> optionalPathTerms [
  | val |
  val := self dynamicInstVarAt: #'optionalPathTerms'.
  val ifNil: [ ^ false ].
  ^ val

]

{ #category : 'Accessing' }
GsQueryOptions >> optionalPathTerms: aBool [
  self dynamicInstVarAt: #'optionalPathTerms' put: aBool

]

{ #category : 'Accessing' }
GsQueryOptions >> removeRedundantPredicates [
  | val |
  val := self dynamicInstVarAt: #'removeRedundantPredicates'.
  val ifNil: [ ^ false ].
  ^ val

]

{ #category : 'Accessing' }
GsQueryOptions >> removeRedundantPredicates: aBoolean [
  self dynamicInstVarAt: #'removeRedundantPredicates' put: aBoolean

]

{ #category : 'Accessing' }
GsQueryOptions >> reorderPredicates [
  | val |
  val := self dynamicInstVarAt: #'reorderPredicates'.
  val ifNil: [ ^ false ].
  ^ val

]

{ #category : 'Accessing' }
GsQueryOptions >> reorderPredicates: aBoolean [
  self dynamicInstVarAt: #'reorderPredicates' put: aBoolean

]

{ #category : 'Operators' }
GsQueryOptions >> setApplyDeMorgansLaws [
  self applyDeMorgansLaws: true

]

{ #category : 'Operators' }
GsQueryOptions >> setAutoOptimize [
  self autoOptimize: true

]

{ #category : 'Operators' }
GsQueryOptions >> setCacheQueryResult [
  self cacheQueryResult: true

]

{ #category : 'Operators' }
GsQueryOptions >> setConsolidateEnumerablePredicates [
  self consolidateEnumerablePredicates: true

]

{ #category : 'Operators' }
GsQueryOptions >> setConsolidateRangePredicates [
  self consolidateRangePredicates: true

]

{ #category : 'Operators' }
GsQueryOptions >> setConsolidateUnaryConstantPredicates [
  self consolidateUnaryConstantPredicates: true

]

{ #category : 'Operators' }
GsQueryOptions >> setNormalizePredicates [
  self normalizePredicates: true

]

{ #category : 'Operators' }
GsQueryOptions >> setOptionalPathTerms [
  self optionalPathTerms: true

]

{ #category : 'Operators' }
GsQueryOptions >> setRemoveRedundantPredicates [
  self removeRedundantPredicates: true

]

{ #category : 'Operators' }
GsQueryOptions >> setReorderPredicates [
  self reorderPredicates: true

]

{ #category : 'Operators' }
GsQueryOptions >> setTransformCommonPaths [
  self transformCommonPaths: true

]

{ #category : 'Accessing' }
GsQueryOptions >> transformCommonPaths [
  | val |
  val := self dynamicInstVarAt: #'transformCommonPaths'.
  val ifNil: [ ^ false ].
  ^ val

]

{ #category : 'Accessing' }
GsQueryOptions >> transformCommonPaths: aBoolean [
  self dynamicInstVarAt: #'transformCommonPaths' put: aBoolean

]
