Extension { #name : 'GsRangeQueryPredicate' }

{ #category : 'instance creation' }
GsRangeQueryPredicate class >> constant: constant1 operator: operator1 path: path operator: operator2 constant: constant2 [
  "e.g.: 4 < a.b.c < 6 or 6 > a.b.c >= 4 "

  ^ (self new
    constant1: constant1;
    operator1: operator1;
    path: path;
    operator2: operator2;
    constant2: constant2;
    yourself) validate

]

{ #category : 'instance creation' }
GsRangeQueryPredicate class >> constant: constant1 operator: operator1 path: path operator: operator2 variable: variable2 [
  "e.g.: 4 < a.b.c < 6 or 6 > a.b.c >= 4 "

  ^ (self new
    constant1: constant1;
    operator1: operator1;
    path: path;
    operator2: operator2;
    variable2: variable2;
    yourself) validate

]

{ #category : 'instance creation' }
GsRangeQueryPredicate class >> operand: operand1 operator: operator1 path: path operator: operator2 operand: operand2 [
  "e.g.: 4 < a.b.c < 6 or 6 > a.b.c >= 4 "

  ^ (self new
    operand1: operand1;
    operator1: operator1;
    path: path;
    operator2: operator2;
    operand2: operand2;
    yourself) validate

]

{ #category : 'instance creation' }
GsRangeQueryPredicate class >> variable: variable1 operator: operator1 path: path operator: operator2 constant: constant2 [
  "e.g.: 4 < a.b.c < 6 or 6 > a.b.c >= 4 "

  ^ (self new
    variable1: variable1;
    operator1: operator1;
    path: path;
    operator2: operator2;
    constant2: constant2;
    yourself) validate

]

{ #category : 'instance creation' }
GsRangeQueryPredicate class >> variable: variable1 operator: operator1 path: path operator: operator2 variable: variable2 [
  "e.g.: 4 < a.b.c < 6 or 6 > a.b.c >= 4 "

  ^ (self new
    variable1: variable1;
    operator1: operator1;
    path: path;
    operator2: operator2;
    variable2: variable2;
    yourself) validate

]

{ #category : 'private' }
GsRangeQueryPredicate >> _evaluators [
  "return list of evaluators associated with predicate"

  ^ {(self evaluator)}

]

{ #category : 'visiting' }
GsRangeQueryPredicate >> acceptVisitor: aFormulaVisitor [
  super acceptVisitor: aFormulaVisitor.
  aFormulaVisitor acceptRangePredicate: self

]

{ #category : 'optimizing' }
GsRangeQueryPredicate >> applyDeMorgansNegationTransform [
  "actively apply De Morgan's laws ... convert to | of two negated predicates"

  | theConstant1 theConstant2 theOperator1 theOperator2 predicate1 predicate2 |
  self isNormal
    ifTrue: [
      theConstant1 := self constant1.
      theOperator1 := self operator1.
      theConstant2 := self constant2.
      theOperator2 := self operator2 ]
    ifFalse: [
      theConstant1 := self constant2.
      theOperator1 := self inverseOperatorFor: self operator2.
      theConstant2 := self constant1.
      theOperator2 := self inverseOperatorFor: self operator1 ].
  predicate1 := GsQueryPredicate
    constant: theConstant1
    operator: (self negatedOperatorFor: theOperator1)
    path: self path.
  predicate2 := GsQueryPredicate
    path: self path
    operator: (self negatedOperatorFor: theOperator2)
    constant: theConstant2.
  ^ predicate1 | predicate2

]

{ #category : 'converting' }
GsRangeQueryPredicate >> asFormulaWithSelectorParts: selectorParts arguments: arguments [
  | selector |
  selector := selectorParts first inputValue asSymbol.
  ^ self perform: selector with: arguments first

]

{ #category : 'transforming' }
GsRangeQueryPredicate >> bind: variableName to: value [
  | bound |
  bound := self copy.
  bound operand1: (bound operand1 bind: variableName to: value).
  bound operand2: (bound operand2 bind: variableName to: value).
  ^ bound immediateInvariant

]

{ #category : 'private' }
GsRangeQueryPredicate >> bindEvaluators [
  evaluator := self
    _bindEvaluator: nsc
    for: self path
    isRangeEqualityOperation: true.
  super bindEvaluators

]

{ #category : 'accessing' }
GsRangeQueryPredicate >> constant1 [
  ^ operand1 _idxValue

]

{ #category : 'accessing' }
GsRangeQueryPredicate >> constant1: anObject [
  operand1 := GsConstantReferenceAssociation
    newWithKey: anObject asString
    value: anObject

]

{ #category : 'accessing' }
GsRangeQueryPredicate >> constant2 [
  ^ operand2 _idxValue

]

{ #category : 'accessing' }
GsRangeQueryPredicate >> constant2: anObject [
  operand2 := GsConstantReferenceAssociation
    newWithKey: anObject asString
    value: anObject

]

{ #category : 'querying' }
GsRangeQueryPredicate >> elementValueEvaluator [
  "companion method to #elementValue:"

  "anObject is the value obtained by traversing path terms to the last value ... analgous to the last element class"

  "not all subclasses are required to implement this selector, if they do not implement this selector, then they must implement #elementValue:"

  ^ (evaluator == nil  or: [ evaluator isPathEvaluator not ])
    ifTrue: [ self pathEvaluatorFor: self path ]
    ifFalse: [ evaluator ]

]

{ #category : 'accessing' }
GsRangeQueryPredicate >> evaluator [
  ^ evaluator

]

{ #category : 'accessing' }
GsRangeQueryPredicate >> evaluator: anEvaluator [
  evaluator := anEvaluator

]

{ #category : 'querying' }
GsRangeQueryPredicate >> executeAndDo: aBlock [
  | queryEvaluator |
  queryEvaluator := self evaluator asQueryEvaluator.
  queryEvaluator doBlock: aBlock.
  ^ self executeClause: queryEvaluator

]

{ #category : 'querying-private' }
GsRangeQueryPredicate >> executeClause [
  | queryEvaluator |
  queryEvaluator := self evaluator asQueryEvaluator.
  ^ self executeClause: queryEvaluator

]

{ #category : 'querying-private' }
GsRangeQueryPredicate >> executeClause: queryEvaluator [
  | theConstant1 theConstant2 theOperator1 theOperator2 validNilTransformOperators |
  self isNormal
    ifTrue: [
      theConstant1 := self constant1.
      theOperator1 := self operator1.
      theConstant2 := self constant2.
      theOperator2 := self operator2 ]
    ifFalse: [
      theConstant1 := self constant2.
      theOperator1 := self inverseOperatorFor: self operator2.
      theConstant2 := self constant1.
      theOperator2 := self inverseOperatorFor: self operator1 ].
  validNilTransformOperators := #(#'<=' #'>=').
  ((theConstant1 == nil and: [ theConstant2 == nil ])
    and: [
      (validNilTransformOperators includes: self operator1)
        and: [ validNilTransformOperators includes: self operator2 ] ])
    ifTrue: [ ^ queryEvaluator findAllValuesEqualTo: nil ].
  ((theConstant1 == nil and: [ theConstant2 ~~ nil ])
    and: [ validNilTransformOperators includes: self operator1 ])
    ifTrue: [ theOperator1 := #'=' ].
  ((theConstant1 ~~ nil and: [ theConstant2 == nil ])
    and: [ validNilTransformOperators includes: self operator2 ])
    ifTrue: [ theOperator2 := #'=' ].
  ^ queryEvaluator
    findAllValuesGreaterThan: theConstant1
    andEquals: theOperator1 == #'<='
    andLessThan: theConstant2
    andEquals: theOperator2 == #'<='

]

{ #category : 'querying-private' }
GsRangeQueryPredicate >> executeClauseNegated [
  "Deoptimize the range predicate and propogate executeClauseNegated"

  | theConstant1 theConstant2 theOperator1 theOperator2 predicate1 predicate2 |
  self isNormal
    ifTrue: [
      theConstant1 := self constant1.
      theOperator1 := self operator1.
      theConstant2 := self constant2.
      theOperator2 := self operator2 ]
    ifFalse: [
      theConstant1 := self constant2.
      theOperator1 := self inverseOperatorFor: self operator2.
      theConstant2 := self constant1.
      theOperator2 := self inverseOperatorFor: self operator1 ].
  predicate1 := (GsQueryPredicate
    constant: theConstant1
    operator: theOperator1
    path: self path) bindEvaluatorsFor: nsc collator: self collator.
  predicate2 := (GsQueryPredicate
    path: self path
    operator: theOperator2
    constant: theConstant2) bindEvaluatorsFor: nsc collator: self collator.
  ^ (predicate1 & predicate2) executeClauseNegated

]

{ #category : 'querying-private' }
GsRangeQueryPredicate >> executeNegatedAndDo: aBlock [
  "Deoptimize the range predicate and propogate executeClauseNegated"

  | theConstant1 theConstant2 theOperator1 theOperator2 predicate1 predicate2 |
  self isNormal
    ifTrue: [
      theConstant1 := self constant1.
      theOperator1 := self operator1.
      theConstant2 := self constant2.
      theOperator2 := self operator2 ]
    ifFalse: [
      theConstant1 := self constant2.
      theOperator1 := self inverseOperatorFor: self operator2.
      theConstant2 := self constant1.
      theOperator2 := self inverseOperatorFor: self operator1 ].
  predicate1 := (GsQueryPredicate
    constant: theConstant1
    operator: theOperator1
    path: self path) bindEvaluatorsFor: nsc collator: self collator.
  predicate2 := (GsQueryPredicate
    path: self path
    operator: theOperator2
    constant: theConstant2) bindEvaluatorsFor: nsc collator: self collator.
  ^ predicate1 & predicate2 executeNegatedAndDo: aBlock

]

{ #category : 'private' }
GsRangeQueryPredicate >> immediateInvariant [
  super immediateInvariant.
  operand1 immediateInvariant.
  operand2 immediateInvariant

]

{ #category : 'testing' }
GsRangeQueryPredicate >> isNormal [
  " operator1  and operator2 must be < or <= "

  ^ (self operator1 == #'<' or: [ self operator1 == #'<=' ])
    and: [ ^ self operator2 == #'<' or: [ self operator2 == #'<=' ] ]

]

{ #category : 'querying-private' }
GsRangeQueryPredicate >> lastElementValue: anObject [
  "anObject is the value obtained by traversing path terms to the last value ... analgous to the last element class"

  | selector1 selector2 |
  self isNormal
    ifTrue: [
      selector1 := self comparisonSelectorFor: self operator1.
      selector2 := self comparisonSelectorFor: self operator2 ]
    ifFalse: [
      selector1 := self
        comparisonSelectorFor: (self inverseOperatorFor: self operator1).
      selector2 := self
        comparisonSelectorFor: (self inverseOperatorFor: self operator2) ].
  ^ (self constant1 perform: selector1 with: anObject)
    & (anObject perform: selector2 with: self constant2)

]

{ #category : 'accessing' }
GsRangeQueryPredicate >> operand1 [
  ^ operand1

]

{ #category : 'accessing' }
GsRangeQueryPredicate >> operand1: anObject [
  operand1 := anObject

]

{ #category : 'accessing' }
GsRangeQueryPredicate >> operand2 [
  ^ operand2

]

{ #category : 'accessing' }
GsRangeQueryPredicate >> operand2: anObject [
  operand2 := anObject

]

{ #category : 'accessing' }
GsRangeQueryPredicate >> operator1 [

   "Return the value of the instance variable 'operator1'."
   ^operator1

]

{ #category : 'accessing' }
GsRangeQueryPredicate >> operator1: newValue [

   "Modify the value of the instance variable 'operator1'."
   operator1 := newValue

]

{ #category : 'accessing' }
GsRangeQueryPredicate >> operator2 [

   "Return the value of the instance variable 'operator2'."
   ^operator2

]

{ #category : 'accessing' }
GsRangeQueryPredicate >> operator2: newValue [

   "Modify the value of the instance variable 'operator2'."
   operator2 := newValue

]

{ #category : 'accessing' }
GsRangeQueryPredicate >> path [
  "don't like that we're obscuring instance variable reference this way ... rename instance variable"

  ^ path key

]

{ #category : 'accessing' }
GsRangeQueryPredicate >> path: aPathString [
  path := self pathReferenceFor: aPathString

]

{ #category : 'private' }
GsRangeQueryPredicate >> postCopy [
  operand1 := operand1 copy.
  path := path copy.
  operand2 := operand2 copy.
  super postCopy

]

{ #category : 'querying-private' }
GsRangeQueryPredicate >> prepareForExecution [
  "Return an object equivalent to the receiver that is prepared for execution within
   the context of a block."

  | ans |
  ans := GsPreparedRangeQueryPredicate new
    constant1: self constant1;
    operator1: self operator1;
    path: self path;
    operator2: self operator2;
    constant2: self constant2;
    _nsc: nsc collator: collator;
    evaluator: self evaluator;
    prepare.
  ^ans

]

{ #category : 'printing' }
GsRangeQueryPredicate >> printOn: aStream [
  aStream
    nextPutAll:
      '(' , self operand1 printString , ' ' , self operator1 asString , ' '
        , self rawPath printString , ' ' , self operator2 asString , ' '
        , self operand2 printString , ')'

]

{ #category : 'accessing' }
GsRangeQueryPredicate >> rawPath [
  "don't like that we're obscuring instance variable reference this way ... rename instance variable"

  ^ path

]

{ #category : 'querying' }
GsRangeQueryPredicate >> readStream [
  | queryEvaluator theConstant1 theConstant2 theOperator1 theOperator2 validNilTransformOperators stream |
  queryEvaluator := self evaluator asQueryEvaluator.
  self isNormal
    ifTrue: [
      theConstant1 := self constant1.
      theOperator1 := self operator1.
      theConstant2 := self constant2.
      theOperator2 := self operator2 ]
    ifFalse: [
      theConstant1 := self constant2.
      theOperator1 := self inverseOperatorFor: self operator2.
      theConstant2 := self constant1.
      theOperator2 := self inverseOperatorFor: self operator1 ].
  validNilTransformOperators := #(#'<=' #'>=').
  stream := ((theConstant1 == nil and: [ theConstant2 == nil ])
    and: [
      (validNilTransformOperators includes: self operator1)
        and: [ validNilTransformOperators includes: self operator2 ] ])
    ifTrue: [
      queryEvaluator
        _findAllValuesGreaterThan: theConstant1
        andEquals: true
        andLessThan: theConstant2
        andEquals: true
        using: (BtreeComparisonForCompare newForComparison: nil) ]
    ifFalse: [
      ((theConstant1 == nil and: [ theConstant2 ~~ nil ])
        and: [ validNilTransformOperators includes: self operator1 ])
        ifTrue: [ theOperator1 := #'=' ].
      ((theConstant1 ~~ nil and: [ theConstant2 == nil ])
        and: [ validNilTransformOperators includes: self operator2 ])
        ifTrue: [ theOperator2 := #'=' ].
      queryEvaluator
        _findAllValuesGreaterThan: theConstant1
        andEquals: theOperator1 == #'<='
        andLessThan: theConstant2
        andEquals: theOperator2 == #'<='
        using: (BtreeComparisonForCompare newForComparison: nil) ].
  stream
    streamQuerySpec:
      (stream btreeRangeComparisonQuerySpec
         key: theConstant1
         selector:
           (theOperator1 == #<=
             ifTrue: [ #'>=' ]
             ifFalse: [ #'>' ]); "denormalize the selector"
         key2: theConstant2 selector2: theOperator2;
         yourself).
  ^ stream

]

{ #category : 'transforming' }
GsRangeQueryPredicate >> unbind [
  "remove all bindings"

  | unbound |
  unbound := super unbind.
  unbound operand1: unbound operand1 unbind.
  unbound operand2: unbound operand2 unbind.
  unbound evaluator: nil.
  ^ unbound immediateInvariant

]

{ #category : 'testing' }
GsRangeQueryPredicate >> usesComparisonOperation [
  ^ true

]

{ #category : 'testing' }
GsRangeQueryPredicate >> usesPathEvaluator [
  self evaluator ifNil: [ ^ true ].
  ^ self evaluator isPathEvaluator

]

{ #category : 'testing' }
GsRangeQueryPredicate >> usesRangeOperation [
  ^ true

]

{ #category : 'private' }
GsRangeQueryPredicate >> validate [
  " operator1 and operator2 must be < or <=
      OR
	 operator1 and operator2 must be > or >= "

  self isNormal
    ifTrue: [ ^ self ].
  ((self operator1 == #'<' or: [ self operator1 == #'<=' ])
    and: [ self operator2 == #'<' or: [ self operator2 == #'<=' ] ])
    ifTrue: [ ^ self ].
  ((self operator1 == #'>' or: [ self operator1 == #'>=' ])
    and: [ self operator2 == #'>' or: [ self operator2 == #'>=' ] ])
    ifTrue: [ ^ self ].
  ^ (GsQueryPredicate
    operand: self operand1
    operator: self operator1
    operand: self rawPath)
    &
      (GsQueryPredicate
        operand: self rawPath
        operator: self operator2
        operand: self operand2)

]

{ #category : 'accessing' }
GsRangeQueryPredicate >> variable1: aVariableName [
  operand1 := self variableReferenceFor: aVariableName

]

{ #category : 'accessing' }
GsRangeQueryPredicate >> variable2: aVariableName [
  operand2 := self variableReferenceFor: aVariableName

]
