Extension { #name : 'GsSignalingSocket' }

{ #category : 'Async Notifications' }
GsSignalingSocket class >> _readNotificationExample [
 "Start a GsProcess to service asynchronous  Notifications"
 [
   | sock classes block1 block2 block3 block4 |
   classes := { InterSessionSignal . ObjectsCommittedNotification . TransactionBacklog .
                GcFinalizeNotification } .
   block1 := [:ex | "replace with application specific code" GsFile gciLogServer: ex asString ].
   block2 := [:ex | "replace with application specific code" GsFile gciLogServer: ex asString ].
   block3 := [:ex | "replace with application specific code" GsFile gciLogServer: ex asString ].
   block4 := [:ex | "required code" ex _finalizeEphemerons ]  .
     
   sock := GsSignalingSocket newForAsyncExceptions: classes .
   [ true ] whileTrue:[
     [ sock readNotification
     ] onException: classes
       do: { block1 . block2 . block3 . block4 } .
   ].
 ] fork
]

{ #category : 'Async Notifications' }
GsSignalingSocket class >> disableAsyncExceptions [
  ^ self _zeroArgClassPrim: 37 
]

{ #category : 'Instance Creation' }
GsSignalingSocket class >> fromFileHandle: aSmallInteger [
 "Read carefully the documentation in GsSocket >> fromFileHandle: "

 | r |
 r := super fromFileHandle: aSmallInteger .
 ^ r

]

{ #category : 'Queries' }
GsSignalingSocket class >> getHostAddressByName: aHostName [

"aHostName must be a String .
 Returns the ip address for aHostName .

 Signals an Error if aHostName cannot be resolved.

 The resulting ip address is a String in a form acceptable to
 inet_pton(AF_INET6, aString, ...)

 If getaddrinfo() returns more than one address for aHostName,
 the first element of the list from getaddrinfo() is returned.
"

 (super getHostAddressByName: aHostName) ifNotNil:[:r | ^ r ].
 self signalError .

]

{ #category : 'Queries' }
GsSignalingSocket class >> getHostNameByAddress: address [

"Returns the host name for the given ip address.
 Signals an Error if the host is undefined.

 The address needs to be a String in a form acceptable to
 either inet_pton(AF_INET6, address, ...)
 or    inet_pton(AF_INET, address, ...) in the host operating
 system."

 (super getHostNameByAddress: address) ifNotNil:[:r | ^ r ].
 self signalError .

]

{ #category : 'Queries' }
GsSignalingSocket class >> getLocalHostName [

"Returns the name of the local host or signals an Error."

 (super getLocalHostName) ifNotNil:[:r | ^ r ].
 self signalError .

]

{ #category : 'Queries' }
GsSignalingSocket class >> getServiceNameByPort: servicePort withProtocol: protocol [

"Returns the service name for the given port number and protocol.
 Protocol may be nil (interpreted as tcp), 'tcp', or 'udp' .
 Signals an Error if the service is undefined."

 (super getServiceNameByPort: servicePort withProtocol: protocol)
   ifNotNil:[:r | ^ r ].
 self signalError .

]

{ #category : 'Queries' }
GsSignalingSocket class >> getServicePortByName: serviceName [

"Returns the port number for the given service or signals an Error."

 ^ self getServicePortByName: serviceName withProtocol: nil

]

{ #category : 'Queries' }
GsSignalingSocket class >> getServicePortByName: serviceName withProtocol: protocol [

"Returns the port number for the given service and protocol.
 Protocol may be nil (interpreted as tcp), 'tcp', or 'udp' .
 Signals an Error if the service is undefined."

 (super getServicePortByName: serviceName withProtocol: protocol)
   ifNotNil:[:r | ^ r ].
 self signalError .

]

{ #category : 'Instance Creation' }
GsSignalingSocket class >> new [
"Returns a new non-blocking TCP socket
 or signals an Error if unable to create a new socket.

 On Solaris this creates a socket with family AF_INET6.

 On other platforms this creates a socket with family AF_INET.
 which may be bound or connected to IPv4 addresses or to IPv4-mapped IPv6 addresses."

 | s |
 s := self _basicNew .
 (s _zeroArgPrim: 1) ifNil:[ self signalError ].
 ^ s

]

{ #category : 'Async Notifications' }
GsSignalingSocket class >> newForAsyncExceptions: anArray [

" Returns a GsSignalingSocket which has been registered to receive
  data representing asynchronous Notifications . 

  anArray must be an Array containing  one or more of the classes
    InterSessionSignal , ObjectsCommittedNotification, TransactionBacklog ,
    GcFinalizeNotification .

  Only one socket can be registered in a session to receive asynchronous Notifications.
  Signals an error if some other socket is registered to receive Notifications.
  GsSignalingSocket class >> disableAsyncExceptions must be used to cancel a previous 
  registration before the next execution of newForAsyncExceptions:.

  The socket returned represents one end of an underlying AF_UNIX socket pair,
  the other end of that pair is written to by the C thread that receives the
  out of band bytes from stone representing the Notifications .
 
  For use in conjunction with code that forks a GsProcess which will
  service the registered notifications, for an example see 
  GsSignalingSocket class >> _readNotificationExample  .

  Use GsSignalingSocket >> readNotification to receive the Notifications.
  Do not use any of 
    InterSessionSignal class >>enableSignalling
    ObjectsCommittedNotification class >> enableSignalling
    TransactionBacklog class >> enableSignalling
  in conjunction with this method.

  
  "

  | oobs |
   anArray _validateInstanceOf: Array .
   oobs := { } .
   anArray do:[:elem | | val |
	elem == InterSessionSignal ifTrue:[ val := OOB_SIGNAL_FROM_GEM ] .
	elem == ObjectsCommittedNotification ifTrue:[ val := OOB_SIGNAL_COMMITTED_OBJS ] .
	elem == TransactionBacklog ifTrue:[ val := OOB_SIGNAL_ABORT ].
	elem == GcFinalizeNotification ifTrue:[ val := OOB_SIGNAL_GC_FINALIZE ].
	val ifNil:[ Error signal:
		'not one of InterSessionSignal , ObjectsCommittedNotification, TransactionBacklog, GcFinalizeNotification'.
		].
	oobs add: val .
	].
   "For each class to be handled, disable normal signalling"
   anArray do:[:cls | cls disableSignalling ].
  ^ self _basicNew _twoArgPrim: 14 with: oobs with: nil .
]

{ #category : 'Instance Creation' }
GsSignalingSocket class >> newIpv6 [

"Returns a new non-blocking TCP socket
 or signals an Error if unable to create a new socket.

 This creates a socket with family AF_INET6
 which may be bound or connected to IPv6 addresses.
 On Linux and solaris the result socket may also
 be bound or connected to IPv4-mapped IPv6 addresses."

 | s |
 s := self _basicNew .
 (s _zeroArgPrim: 34) ifNil:[ self signalError ].
 ^ s

]

{ #category : 'Instance Creation' }
GsSignalingSocket class >> newUdp [

"Returns a new non-blocking UDP socket
 or signals an Error if unable to create a new socket.

 On Solaris this creates a socket with family AF_INET6.

 On other platforms this creates a socket with family AF_INET
 which may be bound or connected to IPv4 addresses or to IPv4-mapped IPv6 addresses."

 | s |
 s := self _basicNew .
 (s _zeroArgPrim: 26) ifNil:[ self signalError ].
 ^ s

]

{ #category : 'Instance Creation' }
GsSignalingSocket class >> newUdpIpv6 [

"Returns a new non-blocking UDP socket
 or signals an Error if unable to create a new socket.

 This creates a socket with family AF_INET6
 which may be bound or connected to IPv6 addresses.
 On Linux and solaris the result socket may also
 be bound or connected to IPv4-mapped IPv6 addresses."

 | s |
 s := self _basicNew .
 (s _zeroArgPrim: 35) ifNil:[ self signalError ].
 ^ s

]

{ #category : 'Error Reporting' }
GsSignalingSocket class >> signalError [
  ^ SocketError signal: (self lastErrorString ifNil:[ 'no details' ]).

]

{ #category : 'Error Reporting' }
GsSignalingSocket class >> signalError: aString [
  ^ SocketError signal:
      (self lastErrorString ifNotNil:[:s | aString , ', ' , s]  ifNil:[ aString ]).

]

{ #category : 'Private' }
GsSignalingSocket >> _rawReadWillNotBlockWithin: timeoutMs [

  ^ self readWillNotBlockWithin: timeoutMs

]

{ #category : 'Server Operations' }
GsSignalingSocket >> accept [

"Accept a client request for a connection on the receiver.
 Returns a socket created for a new connection,
 or signals an Error if there was some problem.
 The result is an instance of speciesForAccept , and has non-blocking state
 equal to non-blocking state of the receiver."

 (super accept) ifNotNil:[:r |
   ^ r
 ].
 self signalError .

]

{ #category : 'Server Operations' }
GsSignalingSocket >> acceptTimeoutMs: timeoutMs [
 "Returns a socket created for a new connection,
  or signals an Error if there was an error or a timeout.
  The result is an instance of speciesForAccept , and has non-blocking state
  equal to non-blocking state of the receiver."
^ self acceptTimeoutMs: timeoutMs errorOnTimeout: true

]

{ #category : 'Server Operations' }
GsSignalingSocket >> acceptTimeoutMs: timeoutMs errorOnTimeout: errOnTimeoutBool [
 "Returns a socket created for a new connection.
  or signals an Error if there was an error.
  If errOnTimeoutBool == true  timeout signals an Error ,
  otherwise returns false on timeout.
  The result is an instance of speciesForAccept , and has non-blocking state
  equal to non-blocking state of the receiver."
| res aSocket cnt |
aSocket := self speciesForAccept _basicNew .
res := true.
cnt := 1 .
[ res == true or:[ res == false ]] whileTrue: [
  "primitive will wait until data available if C socket is blocking."
  res := self _twoArgPrim: 4 with: aSocket with: nil .
  (res == false) ifTrue: [
    "socket is non-blocking and would have blocked, process scheduler
     will suspend the current GsProcess"
    res := self _rawReadWillNotBlockWithin: timeoutMs .
    res ifNil:[ ^ self signalError ].
    res == false ifTrue:[
      errOnTimeoutBool ifTrue:[ ^ self signalError:'accept timed out'].
      ^ false .
    ].
  ].
  cnt := cnt + 1 "for debugging"
].
res ifNil:[ ^ self signalError ].
res := aSocket.
^res

]

{ #category : 'Client Operations' }
GsSignalingSocket >> bindTo: portNumber [

"Binds the receiver to the specified port number.  Use the makeServer
 methods to do the bind when creating a server socket.
 If portNumber is nil then a random port is selected.
 This method is provided to bind a client socket to a specific port
 before it is connected.  Returns the port number actually
 bound to (should be the same as the argument unless argument is nil),
 or signals an Error if not successful."

 ^ self bindTo: portNumber toAddress: nil

]

{ #category : 'Client Operations' }
GsSignalingSocket >> bindTo: portNumber toAddress: address [

"Binds the receiver to the specified port number and address.
 Use the makeServer methods to do the bind when creating a server socket.
 This method is provided to bind a client socket to a specific port
 and/or address before doing a listen or connect on the socket.
 If portNumber is nil then a random port is selected.
 If address is nil then any network interface(IN6ADDR_ANY_INIT) is used.
 Returns the port number actually bound to (should be the same as
 the argument unless argument is nil),
 or signals an Error if not successful.

 address is a String containing a numeric IP address in one of these forms
   IPv4  dotted-decimal format  d.d.d.d
   IPv6  hex format   x:x:x:x:x:x:x:x ,
   IPv4-mapped IPv6  ::FFFF:d.d.d.d,
   where d is an 8 bit decimal number and x is a 16 bit hexadecimal number.
   An IPv6 format may contain at most one :: which is a contigous
   group of zeros.

 Gs64 v3.1:  address = -1, denoting IPv4 INADDR_BROADCAST, no longer supported.

"
 (super bindTo: portNumber toAddress: address) ifNotNil:[:r | ^ r ].
 self signalError .

]

{ #category : 'Client Operations' }
GsSignalingSocket >> connectTo: portNumber on: aHost timeoutMs: timeoutMs [

"Connect the receiver to the server socket identified by portNumber and aHost.
aHost may be the name of the host or its numeric address,
or aHost == -1 for <broadcase> , or aHost == nil for IN6ADDR_ANY_INIT .
portNumber maybe either a SmallInteger, or the String name of a service.
timeoutMs is a SmallInteger specifying maximum time to wait for the
connection to complete, or -1 to wait indefinitely.
If aHost is the name of a host, connect is attempted on IPv4 first
if getaddrinfo returns any IPv4 addresses for aHost, then attempted
on IPv6.
Returns true if the connection succeeded otherwise signals an Error."

(super connectTo: portNumber on: aHost timeoutMs: timeoutMs) ~~ true
  ifTrue:[ self signalError:'connect failed' ].
^ true

]

{ #category : 'Socket Operations' }
GsSignalingSocket >> linger: bool length: timeOut [

"Sets up the receiver so that if unsent data is waiting to be transmitted
 at the time the receiver is closed, the current GsProcess will
 be suspended until either the data is transmitted, or the given timeOut expires.
 timeOut is in units of seconds.

 Returns the receiver or signals an Error."

 (super linger: bool length: timeOut) ifNotNil:[:r | ^ r ].
 self signalError .

]

{ #category : 'Server Operations' }
GsSignalingSocket >> makeListener: queueLength [

"Turns the receiver into a listening socket.
 The queueLength argument specifies the size of the listen backlog queue for
 incoming connections.
 Returns the receiver or signals an Error."

 (super makeListener: queueLength) ifNotNil:[:r | ^ r ].
 self signalError .

]

{ #category : 'Writing' }
GsSignalingSocket >> nbwrite: amount from: byteObj startingAt: index [

"If receiver is a non-blocking socket,
 write at most amount bytes from byteObj to the receiver without blocking.

 If receiver is a blocking socket,
 write at most amount bytes from byteObj to the receiver blocking until
 the socket accepts data, or until the gem process receives SIGTERM.

 The first byte written will be from the position indicated by index.
 The maximum number of bytes to write is specified by amount.
 It is an error if amount bytes are not available.
 Returns the number of bytes written, or signals an Error .
 Returns 0 if non-blocking socket would have blocked."

 | res |
 [
   res := self _write: byteObj startingAt: index ofSize: amount.
   (res == false) ifTrue: [
     "would have blocked"
     ^ 0
   ].
   res == true  "if res==true, socket got EINTR and we retry"
 ] whileTrue .
 res ifNil:[ self signalError ].
 ^ res

]

{ #category : 'Accessing' }
GsSignalingSocket >> option: optionName [

"Returns the current value of the specified option.
 Signals an Error if an error occurred."

 (super option: optionName) ifNotNil:[:r | ^ r ].
 self signalError .

]

{ #category : 'Accessing' }
GsSignalingSocket >> port [
 "Returns a SmallInteger port number or signals an error if socket is not bound"
 | p |
 p := super port .
 (p == nil or:[ p == 0]) ifTrue:[ ^ SocketError signal:'not bound'].
 ^ p
]

{ #category : 'Socket Operations' }
GsSignalingSocket >> option: optionName put: value [

"Sets the value of the specified option.
 Returns the receiver or signals an Error."

 (super option: optionName put: value) ifNotNil:[:r | ^ r ].
 self signalError .

]

{ #category : 'Configuration' }
GsSignalingSocket >> raiseExceptionOnError: aBoolean [

"Sets the value of raiseExceptionOnError to aBoolean for the receiver.
 Returns the receiver.  
 raiseExceptionOnError: true is done automatically by C state initialization
 for instance creation of kinds of GsSignalingSocket"

 aBoolean ~~ true ifTrue:[ ArgumentError signal:'should not shutoff errors in GsSignalingSocket'].
 super raiseExceptionOnError: aBoolean .

]

{ #category : 'Reading' }
GsSignalingSocket >> read: maxBytes into: byteObj startingAt: index [

"Reads up to the given number of bytes into the given byte object (for
 example, a String).  The first byte read will go into the position
 indicated by index.
 Returns the number of bytes read, or 0 if EOF on the receiver,
 or signals an Error .

 maxBytes must be a SmallInteger > 0 and <= 500000000 .

 An error will be raised if index is greater than the size of byteObj.
 Note that index can be equal to the size of byteObj plus 1 in which
 case any bytes read will be appended to byteObj.

 If no data is available for reading
 the current GsProcess is suspended until data arrives.
 The readWillNotBlock or readWillNotBlockWithin: methods may be used to
 determine whether or not data is ready for reading before calling this method."

 (super read: maxBytes into: byteObj startingAt: index) ifNotNil:[:r | ^ r ].
 self signalError .

]

{ #category : 'Reading' }
GsSignalingSocket >> read: maxBytes into: byteObj startingAt: index maxWait: timeMs [
 "Reads up to the given number of bytes into the given byte object (for
 example, a String).  The first byte read will go into the position
 indicated by index.
 Returns the number of bytes read, or 0 for EOF, otherwise signals an error.

 If no data is available for reading
 the current GsProcess is suspended for up to timeMs until data arrives."

 | status |
 status := super read: maxBytes into: byteObj startingAt: index maxWait: timeMs .
 status ifNotNil:[
   status == false ifTrue:[ SocketError signal:'timeout waiting to read'].
   ^ status "number of bytes read or 0 for EOF"
 ].
 ^ self signalError .

]

{ #category : 'Async Notifications' }
GsSignalingSocket >> readNotification [
   "Receiver must be the result of  GsSignalingSocket class >> newForAsyncExceptions: .
 
    See GsSignalingSocket class >> _readNotificationExample
   "
   | string byte |
   self _isAsyncExceptionReader ifFalse:[ 
     ^ Error signal:'receiver was not created by GsSignalingSocket class >> newForAsyncExceptions:'.
   ].
   string := self read: 1 .   "read from the socket pair, other end written by oob reader thread"
   byte := string codePointAt: 1 . 
   byte == OOB_SIGNAL_FROM_GEM ifTrue:[ ^ InterSessionSignal pollAndSignal ].
   byte == OOB_SIGNAL_COMMITTED_OBJS ifTrue:[ ^ ObjectsCommittedNotification signal ].
   byte == OOB_SIGNAL_ABORT ifTrue:[ 
     System transactionLevel > 0 ifTrue:[  ^ self "ignore if in a transaction" ].
     ^ TransactionBacklog signal   
   ].
   byte == OOB_SIGNAL_GC_FINALIZE ifTrue:[
     ^ GcFinalizeNotification signal .
   ].
   Error signal:'invalid byte ', byte asString,' in readNotifications'. 
]

{ #category : 'Reading' }
GsSignalingSocket >> readString: maxBytes [

"Reads up to the given number of bytes, returning them in a String whose size is
 between 0 and maxBytes inclusive.  If an error occurs, an Error is signaled.
 EOF is not an error, if EOF occurs on the receiver before any bytes are read,
 a String of size zero is returned.

 maxBytes must be a SmallInteger > 0 .
 If maxBytes is greater than the size of the operating system's buffer for
 the socket, the size of the result string may be a function of this
 buffer size, even if more data is available from the sender.  Repeated
 invocation of this method may be necessary to obtain all of the data.

 If no data is available for reading
 the current GsProcess is suspended until data arrives.
 The readWillNotBlock or readWillNotBlockWithin: methods may be used to
 determine whether or not data is ready for reading before calling this method."

  | bytes |
  bytes := String new.
  self read: maxBytes into: bytes startingAt: 1.
  ^ bytes

]

{ #category : 'Reading' }
GsSignalingSocket >> readString: amount untilFalse: aBlock [

"Reads bytes from the receiver into a String which it creates and returns.
 Will keep reading until amount bytes are read or
 the one argument Block  aBlock  returns false.
 Each time aBlock is evaluated it is given the number of bytes read so far.
 aBlock is only evaluated after a partial read.
 Returns the String or signals an Error."

| bytes |
bytes := String new.
self read: amount into: bytes untilFalse: aBlock .
^ bytes.

]

{ #category : 'Error Reporting' }
GsSignalingSocket >> signalError [
  ^ SocketError signal: (self lastErrorString ifNil:[ 'no details' ]).

]

{ #category : 'Error Reporting' }
GsSignalingSocket >> signalError: aString [
  ^ SocketError signal:
     (self lastErrorString ifNotNil:[:s | aString , ', ' , s]  ifNil:[ aString ]).

]

{ #category : 'Writing' }
GsSignalingSocket >> write: amount from: byteObj startingAt: index [

"Write bytes from byteObj to the receiver.
 The first byte written will be from the position indicated by index.
 The number of bytes written is specified by amount.
 It is in an error if amount bytes are not available.
 Returns the number of bytes written, or signals an Error if an error occurs.

 If the receiver is not ready for writing
 the current GsProcess is suspended until data arrives.
 The writeWillNotBlock or writeWillNotBlockWithin: methods may be used to
 determine whether or not data is ready for writing before calling this method."

 | offset numToWrite waitCnt |
 offset := index.
 numToWrite := amount.
 [ true ] whileTrue:[ | result |
   [
     result := self _write: byteObj startingAt: offset ofSize: numToWrite .
     result == true
   ] whileTrue . "loop to handle EINTR "

   result _isSmallInteger ifTrue:[
     numToWrite  := numToWrite - result .
     (numToWrite <= 0) ifTrue: [ ^ amount ]. "All done"
     waitCnt := nil .
     offset := offset + result .
   ] ifFalse: [
     result ifNil:[ self signalError ].
     waitCnt ifNotNil:[
       waitCnt < 3 ifTrue:[
         waitCnt := waitCnt + 1 .
         Delay waitForMilliseconds: 1.
       ] ifFalse:[
         GsFile gciLogServer:'socket EAGAIN fd ', fileDescriptor asString .
         System _printSocketTrace .
         SocketError signal:'EAGAIN from socket write after poll said write-ready'.
       ].
     ] ifNil:[
       waitCnt := 0 .
     ].
     self _waitForWriteReady .
   ].
 ].

]

{ #category : 'Examples' }
GsSignalingSocket class >> clientExample [
^ self clientExample: true usingPort: 57785 address: nil
]

{ #category : 'Examples' }
GsSignalingSocket class >> clientExample: logToGciClient usingPort: portNum address: serverAddress [

"logToGciClient is a Boolean, true if GsFile-based logging is to be done to
  the RPC Gci client, false if to the server process's log file .
  Use false if executing via topaz 'nbrun'  .

 This client will connect to a server created with serverExample, and read
 the string object that the server sends.

 Creates a socket, connect it to port portNum, read a string, close the
 socket, and check the resulting object.  Returns true if successful.

 The server should already be listening for connections when this method is
 invoked."

| socket dataString dataObj chunk firstChunk sleepTime lastStr |

[
socket := GsSignalingSocket new.
[socket connectTo: portNum on: serverAddress] on: SocketError do:[:ex|
  Error signal: 'Unable to connect to port ' , portNum asString , '. Wait 30 seconds or so, then ensure serverExample is started first.'
  ].

dataString := String new .

firstChunk := true .
sleepTime := 0 .
[ socket readWillNotBlockWithin: 5000] whileFalse: [
  GsFile gciLog: 'Waiting for server to write...' onClient: logToGciClient .
  sleepTime := sleepTime + 5 .
  sleepTime > 20 ifTrue:[
    GsFile gciLog: 'Not ready to read after 20sec' onClient: logToGciClient.
    Error signal: 'Not ready to read after 20sec' .
  ].
].
GsFile gciLog: 'Got something from the server to read.' onClient: logToGciClient .
[
  [chunk := socket readString: 4000 "max bytes" ] on: SocketError do:[:ex| ex signal: 'Error reading from socket' ].
  firstChunk ifTrue:[
    firstChunk := false .
    chunk size <= 0 ifTrue:[ Error signal: 'readWillNotBlock disagrees with read'].
    ] .
  dataString addAll: chunk .
  (chunk at: chunk size) == (Character withValue: 0)  "until null terminator"
] untilTrue .
lastStr := socket readString: 100 .
lastStr size == 0 ifFalse:[ Error signal:'did not get EOF as expected'].
socket close.
dataString size: (dataString size - 1) "strip null terminator" .
dataObj := (PassiveObject newWithContents: dataString) activate.
dataObj size == 5000 ifFalse:[ Error signal: 'bad size of dataObj' ].
1 to: 5000 do:[:j |
  (dataObj at:j) == j ifFalse:[ Error signal: 'invalid contents in dataObj' ].
  ] .
] ensure:[ socket close ].
^ true.
]

{ #category : 'Examples' }
GsSignalingSocket class >> serverExample [
^ self serverExample: true usingPort: 57785 address: nil
]

{ #category : 'Examples' }
GsSignalingSocket class >> serverExample: logToGciClient usingPort: portNum address: listeningAddress [

"logToGciClient is a Boolean, true if GsFile-based logging is to be done to
  the RPC Gci client, false if to the server process's log file .
  Use false if executing via topaz 'nbrun'  .

 Creates a socket, binds it to port portNum, and waits for a connection.
 When a connection is established, sends some data to the client, and
 closes the connection.  Returns true if successful.

 You will need two GemStone sessions running from two independent
 interface processes to run both this and the clientExample.  The Gem
 processes for the two sessions must be on the same machine. (For
 example two Topaz sessions.)

 Warning: This method will cause your current session to hang until a
 connection is established."

| server client dataObj dataString numWritten dataStream errStr
  sleepTime |

[
server := GsSignalingSocket new.

(server makeServer: 5 atPort: portNum atAddress: listeningAddress) ifNil: [
  errStr := server lastErrorString.
  Error signal: errStr.
  ].

server port == portNum ifFalse:[ Error signal: 'bad port number' ] .

GsFile gciLog: 'Waiting for GsSignalingSocket clientExample to connect...'
	onClient: logToGciClient .
[server readWillNotBlockWithin: 5000] whileFalse: [
  GsFile gciLog: 'waiting...'
	onClient: logToGciClient
  ].
GsFile gciLog: 'Client connected, starting accept.' onClient: logToGciClient .

[client := server accept ] on: SocketError do:[:ex| ex signal: 'Error accepting client connection' ].
GsFile gciLog: 'accept finished. client = ', client asString  onClient: logToGciClient .
[client linger: true length: 10] on: SocketError do:[:ex| 
	ex signal: 'Error setting linger time '
].  "wait after closing until data is processed"

dataObj := Array new:5000 .
1 to: 5000 do:[:j | dataObj at:j put: j ].
dataString := String new .
dataStream := WriteStream on: dataString .
PassiveObject passivate: dataObj toStream: dataStream .
dataString add: (Character withValue: 0). "null terminate so client knows
                                           when to stop reading"
sleepTime := 0 .
[ client writeWillNotBlock ] whileFalse:[
  sleepTime == 0 ifTrue:[
     GsFile gciLog: 'waiting because write to client would block' onClient: logToGciClient.
  ].
  sleepTime > 5000 ifTrue:[
    GsFile gciLog: 'ERROR, client would block' onClient: logToGciClient .
    Error signal: 'socket write will block'
  ].
  System _sleepMs: 20 .
  sleepTime := sleepTime + 20 .
].
GsFile gciLog: 'waited ' , sleepTime asString, 'ms ' onClient: logToGciClient.
GsFile gciLog: 'about to write ', dataString size asString, ' bytes'  onClient: logToGciClient .
[numWritten := client write: dataString ] on: SocketError do:[:ex| 
	ex signal: 'Error writing data to client'
].
GsFile gciLog: 'wrote ', numWritten asString, ' bytes'  onClient: logToGciClient .
numWritten == dataString size ifFalse:[ Error signal: 'error writing to socket'].
] ensure:[ 
	client ifNotNil:[:s| s close ].
	server ifNotNil:[:s| s close ].
].

^ true
]
