"
GsSysLog is a class used to write entries to the system log via syslogd.
 It has class-side behavior only.
"
Class {
	#name : 'GsSysLog',
	#superclass : 'Object',
	#category : nil
}

{ #category : 'Private' }
GsSysLog class >> _allFacilitySymbols [

^ { #LOG_USER   . #LOG_LOCAL0  . #LOG_LOCAL1 . #LOG_LOCAL2 . #LOG_LOCAL3 .
    #LOG_LOCAL4 . #LOG_LOCAL5  . #LOG_LOCAL6 . #LOG_LOCAL7 }

]

{ #category : 'Private' }
GsSysLog class >> _allPrioritySymbols [

^ { #LOG_EMERG  . #LOG_ALERT . #LOG_CRIT . #LOG_ERR . #LOG_WARNING .
    #LOG_NOTICE . #LOG_INFO  . #LOG_DEBUG }

]

{ #category : 'Private' }
GsSysLog class >> _validateString: anObj withMaxSize: anInt [
| sz |
(anObj isKindOfClass: Utf8) ifFalse:[
  anObj  _validateKindOfClass: String .  "single byte strings or Utf8 only"
].
sz := anObj _basicSize . "Use #_basicSize in case #size has been overloaded."
((sz == 0) or:[ sz > anInt ])
  ifTrue:[ anObj _error: #rtErrBadSize args:{ 1 . anInt } ] .
^ true

]

{ #category : 'Setting Flags' }
GsSysLog class >> allFlagBits [
^ 63

]

{ #category : 'Setting Flags' }
GsSysLog class >> defaultFlags [
"Answer a SmallInteger containing the default flags used for writing messages
 to the system log."
^ self setFlagIncludePid: (self setFlagLogToConsoleOnError: 0)

]

{ #category : 'Setting Flags' }
GsSysLog class >> setFlagForStderr: anInt [
^ anInt bitOr: 1

]

{ #category : 'Setting Flags' }
GsSysLog class >> setFlagIncludePid: anInt [
^ anInt bitOr: 8

]

{ #category : 'Setting Flags' }
GsSysLog class >> setFlagLogOnGemHost: anInt [
^ anInt bitOr: 32

]

{ #category : 'Setting Flags' }
GsSysLog class >> setFlagLogOnStoneHost: anInt [
^ anInt bitOr: 16

]

{ #category : 'Setting Flags' }
GsSysLog class >> setFlagLogToConsoleOnError: anInt [
^ anInt bitOr: 4

]

{ #category : 'Setting Flags' }
GsSysLog class >> setFlagNoCloselog: anInt [
^ anInt bitOr: 2

]

{ #category : 'Logging' }
GsSysLog class >> writeErrorMessage: messageString [
"Writes an error message to the syslog using default settings."

^ self writeLogMessage: messageString
       prefix: nil
       priority: #LOG_ERR
       facility: #LOG_USER
       flags: self defaultFlags

]

{ #category : 'Logging' }
GsSysLog class >> writeInfoMessage: messageString [
"Writes an informational message to the syslog using default settings."
^ self writeLogMessage: messageString
       prefix: nil
       priority: #LOG_INFO
       facility: #LOG_USER
       flags: self defaultFlags

]

{ #category : 'Logging' }
GsSysLog class >> writeLogMessage: messageString prefix: prefixString priority: priSym facility: facSym flags: flagsInt [
"Writes messageString to the system log by calling the UNIX fuctions openlog(),
 syslog() and closelog().

 The messageString argument must be an instance of String or a instance of a
 subclass of String, or an instance of Utf8.
 It also must have a size between 1 and 4905 characters.
 The message written vi syslog() will be terminated at any codepoint of zero
 within messageString .

 The prefixString argument must be an instance of String, an instance of a
 subclass of String, or nil.  If it is not nil, then it also must have a size
 between 1 and 1023 characters.  A value of nil causes the default prefix
 of 'GemStone' to be used for the prefix.  Characters beyond a codepoint of zero
 in prefixString are ignored.

 The priSym argument indicates the priority of the message and must be one
 of the following symbols:

   #LOG_EMERG      system is unusable
   #LOG_ALERT      action must be taken immediately
   #LOG_CRIT       critical conditions
   #LOG_ERR        error conditions
   #LOG_WARNING    warning conditions
   #LOG_NOTICE     normal, but significant, condition
   #LOG_INFO       informational message
   #LOG_DEBUG      debug-level message

 The facSym argument indicates what type of program is logging the message
 (called the facility) and must be one of the following symbols:

   #LOG_USER       generic user-level messages
   #LOG_LOCAL0     reserved for local use
   #LOG_LOCAL1     reserved for local use
   #LOG_LOCAL2     reserved for local use
   #LOG_LOCAL3     reserved for local use
   #LOG_LOCAL4     reserved for local use
   #LOG_LOCAL5     reserved for local use
   #LOG_LOCAL6     reserved for local use
   #LOG_LOCAL7     reserved for local use

The flagsInt argument is a SmallInteger which is either 0 or the bitwise-OR
of any of the following values:

  1 - Write the message to stderr of this process as well as the system
      logger.
  2 - Do not call closelog() after writing the message.
  4 - Write directly to system console if there is an error while sending
      to system logger (LOG_CONS option).
  8 - Include the PID with each message (LOG_PID option).
 16 - Write the message to the system logger on the stone's host.  Has no
      effect unless the gem is running on a host remote from the stone.
 32 - Write the message on the system logger gem's host.  Has no
      effect unless the gem is running on a host remote from the stone.

 For sessions that are not on same host as the stone, at least one of the bits
 corresponding to the values 16 and 32 must be set.

 Note: The host system may impose message and prefix size limits smaller than
       the limits specified above.  Consult the system documentation for the
       syslog() function to determine what the actual limits enforced by the
       system are.

 Signals an error if the privilege NoPerformOnServer is true and the
 special entry /GsSysLog/writeLogMessage is not present in the allowlist of
 allowed commands for the session's UserProfile. See the methods in
 UserProfile under category PerformOnServer for more information.

 Returns the receiver."

<primitive: 1006>
self _validateString: messageString withMaxSize: 4095 .
prefixString == nil
  ifFalse:[ self _validateString: prefixString withMaxSize: 1023 ] .
priSym  _validateClass: Symbol .
facSym  _validateClass: Symbol .
flagsInt _validateClass: SmallInteger .
((flagsInt < 0) or:[ flagsInt > self allFlagBits ])
  ifTrue:[ flagsInt _error: #rtErrArgOutOfRange args:{ 0 . self allFlagBits } ] .
(self _allPrioritySymbols includesIdentical: priSym)
  ifFalse:[ ArgumentError new object: priSym ;
                          signal: 'Invalid priority'].
(self _allFacilitySymbols includesIdentical: facSym)
  ifFalse:[ ArgumentError new object: facSym ;
                          signal: 'Invalid facility'].
((flagsInt bitAnd: (16 bitOr:32)) == 0 and:[ System sessionIsOnStoneHost not])
  ifTrue:[ ArgumentError new object: flagsInt ;
		signal:'sessionIsOnStoneHost==true, flags must specify stone or gem host'].
^ self _primitiveFailed: #_primSysLogMessage:prefix:priority:facility:flags:
       args: { messageString . prefixString . priSym . facSym . flagsInt  }

]

{ #category : 'Logging' }
GsSysLog class >> writeWarningMessage: messageString [
"Writes a warning message to the syslog using default settings."

^ self writeLogMessage: messageString
       prefix: nil
       priority: #LOG_WARNING
       facility: #LOG_USER
       flags: self defaultFlags

]
