"
GsTestCase is the GemStone-specific refinement of the SUnit TestCase class.
"
Class {
	#name : 'GsTestCase',
	#superclass : 'TestCase',
	#category : nil
}

{ #category : 'Converting' }
GsTestCase class >> asPrintable: aString [
| res |
GsFile stdoutServer isTerminal ifFalse:[ ^ aString ]
"avoid freezing certain Xterm windows with control chars".
res := String new .
1 to: aString size do:[:n | | ch |
  ch := aString codePointAt: n .
  ch < 16r20 ifTrue:[
    (ch == 10"\n" or:[ ch == 9"\t" ]) ifTrue:[ res addCodePoint: ch ]
             ifFalse:[ res add: $^ ; addCodePoint: 16r40 + ch ]
     ] ifFalse:[
  ch < 16r7F ifTrue:[ res addCodePoint: ch ] ifFalse:[ 
  ch == 16r7F ifTrue:[ res addAll:'^?' ] ifFalse:[ | cpStr sz |
    res addAll:'\u' .
    cpStr := ch asHexString . 
    sz := cpStr size .
    [ sz < 4 ] whileTrue:[ res add: $0 . sz := sz + 1 ].
    res addAll: cpStr .
 ]]]
].   
^ res

]

{ #category : 'Debugging' }
GsTestCase class >> confirm: aSelector [

	^[
		self debug: aSelector.
		true.
	] on: Error , TestFailure do: [:ex | 
		false.
	].

]

{ #category : 'Debugging' }
GsTestCase class >> debug: aSelector [
  SessionTemps current at:#GsTestCase_debug put: true . "supress stack report in exception message"
  self openSunitLog: #stdout .
  ^ super debug: aSelector

]

{ #category : 'Debugging' }
GsTestCase class >> debugEx [

"receiver should be a concrete test case class .
 run and get stack at first error."
^ self debugSuite: self suite

]

{ #category : 'Debugging' }
GsTestCase class >> debugEx: aTestCaseClass [

"Run suite for specified class and get stack at first error."

^ self debugSuite: aTestCaseClass suite

]

{ #category : 'Debugging' }
GsTestCase class >> debugSuite: aTestSuite [

  AlmostOutOfMemoryError enable ; threshold: 90 .
  GsTestCase openSunitLog: #stdout .
  [
    SessionTemps current at:#GsTestCase_debug put: true . "supress stack report in exception message"
    aTestSuite tests do: [:each | 
      (each isKindOf: TestSuite)
        ifTrue:  [self debugSuite: each] 
        ifFalse: [each debugInSuite].
    ].
  ] ensure: [
    TestResource resetResources: self resources.
  ].
  GsTestCase flushLog .
  ^ 'ok'  

]

{ #category : 'Debugging' }
GsTestCase class >> dummyHostCallDebugger [
GsFile stdoutServer isTerminal ifFalse:[ | f |
  f := GsFile openWriteOnServer:'HostCallDebugger.log' .
  f nextPutAll:'HostCallDebugger', ': process ' , System gemProcessId asString,
     ' waiting for topaz -r to attach with DEBUGGEM' ; lf .
  f close .
].

]

{ #category : 'Logging' }
GsTestCase class >> flushLog [

| sunitLog |
sunitLog := System sessionStateAt: 1 .
sunitLog ifNotNil:[ sunitLog flush ].

]

{ #category : 'Testing' }
GsTestCase class >> isAbstract [
	"Override to true if a TestCase subclass is Abstract and should not have
	TestCase instances built from it"

	^self sunitName = #GsTestCase

]

{ #category : 'Testing' }
GsTestCase class >> isAix [
  ^ false "AIX not supported"

]

{ #category : 'Testing' }
GsTestCase class >> isArm64 [
  | cpu |
  cpu := System gemVersionAt: #cpuArchitecture .
  ^ cpu = 'aarch64' or:[ cpu = 'arm64']
]

{ #category : 'Testing' }
GsTestCase class >> isArm64Darwin [
^ (System gemVersionAt: #cpuArchitecture) = 'arm64'

]

{ #category : 'Testing' }
GsTestCase class >> isArm64Linux [
^ (System gemVersionAt: #cpuArchitecture) = 'aarch64'
]

{ #category : 'Testing' }
GsTestCase class >> isDarwin [
  ^ (System gemVersionAt: #osName) = 'Darwin'
]

{ #category : 'Testing' }
GsTestCase class >> isDebug [
  ^ self isDebug: nil

]

{ #category : 'Testing' }
GsTestCase class >> isDebug: anException [
  | tmps |
  ((tmps := SessionTemps current) at:#GsTestCase_debug otherwise: false) ifTrue:[ 
    ^ true
    "running a test case via debug: or debugEx , let topaz handle the error" 
  ].
  (System gemEnvironmentVariable:'GsTestCaseWaitForDebug') ifNotNil:[
    self pauseForDebug.
    ^ false
  ].
  ^ false "log failure to SUnit.log and keep running"

]

{ #category : 'Testing' }
GsTestCase class >> isLinux [
   ^ (System gemVersionAt: 'osName') = 'Linux'

]

{ #category : 'Testing' }
GsTestCase class >> isLinuxOrDarwin [
  
  ^self isLinux or:[ self isDarwin].
]

{ #category : 'Testing' }
GsTestCase class >> isLinuxOrDarwinOrAix [
  ^self isLinuxOrDarwin or: [self isAix]
]

{ #category : 'Testing' }
GsTestCase class >> isLinuxOrSolaris [
| os |
os := System gemVersionAt: 'osName'.
^ os = 'Linux' or:[ os = 'SunOS' ]

]

{ #category : 'Testing' }
GsTestCase class >> isSolaris [
  "Needed on class side by GsSocketTestCase"
  ^(System gemVersionAt: 'osName') = 'SunOS'

]

{ #category : 'Testing' }
GsTestCase class >> isSparcSolaris [
^ (System gemVersionAt: #cpuArchitecture) = 'SPARC'

]

{ #category : 'Testing' }
GsTestCase class >> isUbuntu20Linux [
  ^ self isUbuntuLinux and:[ (System performOnServer: 'cat /etc/os-release') includesString: 'Ubuntu 20' ]

]

{ #category : 'Testing' }
GsTestCase class >> isUbuntuLinux [
  ^ self isLinux and:[ (System gemVersionAt: 'osVersion') includesString:'Ubuntu' ] 

]

{ #category : 'Testing' }
GsTestCase class >> isWindows [

  ^ GsFile gciClientIsWindows .

]

{ #category : 'Testing' }
GsTestCase class >> isX86_64 [
  | cpu |
  cpu := System gemVersionAt: #cpuArchitecture .
  ^ cpu = 'x86-64'
]

{ #category : 'Logging' }
GsTestCase class >> log: aString [
| sunitLog |
sunitLog := System sessionStateAt: 1 .
sunitLog ifNil:[
  "if not already open, assume in an external session's gem, log to filesystem"
  sunitLog := self openSunitLog: #filesystem
].
sunitLog nextPutAll: (self asPrintable: aString) ; flush

]

{ #category : 'Logging' }
GsTestCase class >> logCr: aString [

| str |
str := aString , Character lf .
GsTestCase log: str 

]

{ #category : 'Logging' }
GsTestCase class >> openSunitLog: aKind [
"aKind == #stdout means to GsFile stdout, 
  otherwise to SUnit.log file if GCI client is topaz -l "
| aLog |
aLog := System sessionStateAt: 1.
aLog ifNil:[
  (aKind ~~ #stdout and:[ System clientIsRemote not ]) ifTrue:[
    "in a topaz -l, only open SUnit.log"  
    aLog := GsFile openAppendOnServer: GsTestSuite logFilePath.
  ].
  aLog ifNil:[ 
    "use stdout if file system open failed. Example: remote topaz -l
     and logFilePath generates file system path 
     based on extent locations without /export ."
    aLog := GsFile stdoutServer . "assume topaz -l, if interactive"
  ].
  System sessionStateAt: 1 put: aLog.
].
^ aLog
]

{ #category : 'Logging' }
GsTestCase class >> openSunitLog [
  "Open ./SUnit.log  on filesystem without regard to state of  System clientIsRemote"
  | aLog |
  aLog := System sessionStateAt: 1.
  aLog ifNil:[
    aLog := GsFile openAppendOnServer: GsTestSuite logFilePath.
    System sessionStateAt: 1 put: aLog.
  ].
  ^ aLog 
]

{ #category : 'Debugging' }
GsTestCase class >> pauseForDebug [
  | now |
  now := DateAndTime now asStringMs .
  GsFile stdoutServer isTerminal ifFalse:[ 
     self dummyHostCallDebugger .
     GsFile gciLogServer:'--------'.
     GsFile gciLogServer:( GsProcess stackReportToLevel:30).
     GsFile gciLogServer:'--------'.
     GsFile gciLogServer:'Waiting for topaz -r debugger to attach'.
     GsFile stdoutServer flush .
     System waitForDebug .
  ] ifTrue:[
    self pause "not trappable error ; back to topaz "
  ]
]

{ #category : 'Miscellaneous' }
GsTestCase class >> random [
  ^ SessionTemps current at:#GsTestCase_Random ifAbsentPut: [Random new].
]

{ #category : 'Test Suite' }
GsTestCase class >> suite [

	| suite tests |
	suite := super suite.
	tests := suite tests.
	tests copy do: [:each | 
		(each isKindOf: GsTestSuite) ifTrue: [
			| assoc |
			assoc := System myUserProfile resolveSymbol: each name asSymbol.
			(assoc ~~ nil and: [assoc value wantsToBeRunByItself]) ifTrue: [
				tests remove: each.
			].
		].
	].
	^suite.

]

{ #category : 'Test Suite' }
GsTestCase class >> suiteAsExecutionSource [

  " return an execution string of all test cases that
    don't need to be run by themselves, for use
    in debugging a new VM from topaz -l  ."
  | str LF |
  str := String new .
  LF := Character lf .
  str addAll:'run'; add: LF .
  GsTestCase suite tests do:[:aTst | 
    str addAll:'GsTestCase debugEx: '; addAll: aTst name ; add: $. ; add: LF
  ].
  str add: $% ; add: LF .
  ^ str

]

{ #category : 'Test Suite' }
GsTestCase class >> suiteClass [
	^GsTestSuite

]

{ #category : 'Testing' }
GsTestCase class >> wantsToBeRunByItself [

	^false.

]

{ #category : 'Transaction Management' }
GsTestCase >> abort [
	System abortTransaction.

]

{ #category : 'Miscellaneous' }
GsTestCase >> addAllTestsTo: aList [

	aList add: self.

]

{ #category : 'Miscellaneous' }
GsTestCase >> addSession: aSession [
"return a session for similarity with collection methods"
	^self sessions add: aSession

]

{ #category : 'Asserting' }
GsTestCase >> assert: aBoolean [ 
  
  self assert: aBoolean description:'' .

]

{ #category : 'Asserting' }
GsTestCase >> assert: aBoolean description: aString [
  | str | 
  aBoolean == true  ifFalse: [
    str := aString .
    str _isExecBlock ifTrue:[ str := aString value ].
    self logFailure: str.
    GsTestResult resumableFailure sunitSignalWith: str
  ]

]

{ #category : 'Asserting' }
GsTestCase >> assert: aBoolean description: aString resumable: resumableBoolean [
  | exception |
  aBoolean == true  ifFalse: [
    self logFailure: aString.
    exception := resumableBoolean ifTrue: [
      GsTestResult resumableFailure
    ] ifFalse: [
      GsTestResult failure
    ].
    exception sunitSignalWith: aString
  ]

]

{ #category : 'Asserting' }
GsTestCase >> assert: a equals: b description: c [
  self assert: a = b description: 
    a asString, ' not equal to ' , b asString , ' for ', c

]

{ #category : 'Asserting' }
GsTestCase >> assert: a greaterEqual: b [ 
  self assert: a >= b description: 
    a asString, ' not >= ' , b asString  

]

{ #category : 'Asserting' }
GsTestCase >> assert: a identical: b [
  a == b ifFalse:[
    (a _isSmallInteger and:[ b _isSmallInteger]) ifTrue:[
      ^ self assert: a equals: b .  "for better printing"
    ].
  ].
  self assert: a == b description: 'OOP ' , a asOop asString, ' not identical to OOP ' , b asOop asString

]

{ #category : 'Asserting' }
GsTestCase >> assert: a identical: b description: c [
  self assert: a == b description: 
   'OOP ' , a asOop asString, ' not identical to OOP ' , b asOop asString , ' for ', c

]

{ #category : 'Asserting' }
GsTestCase >> assert: objA includes: objB [
  (objA includes: objB) ifFalse:[ | a b |
    a := objA printString .
    b := objB printString .
    a size > 100 ifTrue:[ a size: 100; addAll:'... ' ].
    b size > 100 ifTrue:[ b size: 100; addAll:'... ' ].
	  self
		 assert: false
		 description: a, ' does not include ' , b .
  ].

]

{ #category : 'Asserting' }
GsTestCase >> assert: objA includesIdentical: objB [
  (objA includesIdentical: objB) ifFalse:[ | a b |
    a := objA printString .
    b := objB printString .
    a size > 100 ifTrue:[ a size: 100; addAll:'... ' ].
    b size > 100 ifTrue:[ b size: 100; addAll:'... ' ].
	  self
		 assert: false
		 description: a, ' does not includeIdentical: ' , b .
  ].

]

{ #category : 'Asserting' }
GsTestCase >> assert: objA isKindOfClass: objB [
self assert: (objA isKindOfClass: objB)
     description: (objA printString, ' is not a kind of class ', objB printString)

]

{ #category : 'Asserting' }
GsTestCase >> assert: objA isMemberOfClass: objB [
self assert: (objA isMemberOfClass: objB)
     description: (objA printString, ' is not a member of class ', objB printString)

]

{ #category : 'Asserting' }
GsTestCase >> assert: a lessThan: b [ 
  self assert: a < b description: 
    a asString, ' not < ' , b asString  

]

{ #category : 'Asserting' }
GsTestCase >> assert: aString matchPattern: aPatternArray [

	self assert: (aString matchPattern: aPatternArray) description: aString printString, ' does not match ', aPatternArray printString.

]

{ #category : 'Asserting' }
GsTestCase >> assert: a notIdentical: b [
  self assert: a ~~ b description: ' expected a ', a printString, ' to not be identical to b '

]

{ #category : 'Asserting' }
GsTestCase >> assert: a notIdentical: b description: str [ 
  self assert: a ~~ b description: str .

]

{ #category : 'Asserting' }
GsTestCase >> assertQuietNan: f1 [
  self assert: (f1 _isNaN _and: [ f1 kind == #quietNaN ])

]

{ #category : 'Transaction Management' }
GsTestCase >> begin [
	
	System beginTransaction.

]

{ #category : 'Transaction Management' }
GsTestCase >> commit [
   ^GsSession isSolo or: [System commitTransaction or: [self error: 'commit failed']]

]

{ #category : 'Miscellaneous' }
GsTestCase >> compileMethod: aStr class: anObject [

	self assert: (anObject compileMethod: aStr dictionaries: GsSession currentSession symbolList category: 'other') == nil.
]

{ #category : 'Transaction Management' }
GsTestCase >> continue [

	^System continueTransaction

]

{ #category : 'Debugging' }
GsTestCase >> debug [
   [ (self class selector: testSelector) runCase
   ] ensure: [
     [ TestResource resetResources: self resources
     ] on: Notification do:[:ex | 
        ex class defaultHandlers size > 0 ifTrue:[ ex pass ].
        Error signal: ex asString 
     ]
   ].

]

{ #category : 'Debugging' }
GsTestCase >> debugInSuite [
   "used from GsTestCase class >> debugSuite: that does its own cleanup" 
  (self class selector: testSelector) runCase
]

{ #category : 'Asserting' }
GsTestCase >> deny: a equals: b [
  self assert: (a = b) not description: 'OOP ' , a asOop asString, ' is equal to OOP ' , b asOop asString

]

{ #category : 'Asserting' }
GsTestCase >> deny: a identical: b [
  self assert: a ~~ b description: 'OOP ' , a asOop asString, ' is identical to OOP ' , b asOop asString

]

{ #category : 'Asserting' }
GsTestCase >> deny: objA includes: objB [
  (objA includes: objB) ifTrue:[ | a b |
    a := objA printString .
    b := objB printString .
    a size > 100 ifTrue:[ a size: 100; addAll:'... ' ].
    b size > 100 ifTrue:[ b size: 100; addAll:'... ' ].
	  self
		 assert: false
		 description: a, ' should not include ' , b .
  ].

]

{ #category : 'Asserting' }
GsTestCase >> deny: objA includesIdentical: objB [
  (objA includesIdentical: objB) ifTrue:[ | a b |
    a := objA printString .
    b := objB printString .
    a size > 100 ifTrue:[ a size: 100; addAll:'... ' ].
    b size > 100 ifTrue:[ b size: 100; addAll:'... ' ].
	  self
		 assert: false
		 description: a, ' should not includeIdentical: ' , b .
  ].

]

{ #category : 'Asserting' }
GsTestCase >> ensureExtentsExist: anInteger [

	| existingName i lastNumber extOffset |
	extOffset := System repositoryIsEncrypted ifTrue:[ 4 ] ifFalse:[ 3 ] .
	SystemRepository numberOfExtents >= anInteger ifTrue: [^self].
	existingName := SystemRepository fileNames last.
	i := existingName size - extOffset.
	(existingName copyFrom: i to: i + extOffset) = SystemRepository defaultFileExtension
	  ifFalse: [ self error: 'unexpected extent name' ].
	[
		i > 1 and: [(existingName at: i - 1) isDigit].
	] whileTrue: [
		i := i - 1.
	].
	i = (existingName size - extOffset) ifTrue: [
		self error: 'extent name does not end in digit'.
	].
	lastNumber := (existingName copyFrom: i to: existingName size - (extOffset + 1)) asNumber.
	[
		SystemRepository numberOfExtents < anInteger.
	] whileTrue: [
		| newName |
		lastNumber := lastNumber + 1.
		newName := (existingName copyFrom: 1 to: i - 1) , 
			lastNumber printString , SystemRepository defaultFileExtension .
		SystemRepository createExtent: newName.
	].
	System hasMissingGcGems ifTrue: [
		System stopAllGcGems.
		System startAllGcGems.
	].

]

{ #category : 'Asserting' }
GsTestCase >> ensureGcRunning [
	System ensureGcRunning.
	self deny: System hasMissingGcGems description: 'System hasMissingGcGems in ensureGcRunning'.

]

{ #category : 'Miscellaneous' }
GsTestCase >> forceLogoutSessions [
  self sessions ifNotNil:[:list | 
    list do: [:each | each ifNotNil:[ each forceLogout ] ].
    list size: 0 .
  ].

]

{ #category : 'Miscellaneous' }
GsTestCase >> gciErrorClass [
  ^ GciError
]

{ #category : 'Running' }
GsTestCase >> ignoringDeprecatedDo: aBlock [
  ^ aBlock on: Deprecated do:[:ex | ex resume  ]

]

{ #category : 'Running' }
GsTestCase >> ignoringDeprecatedShould: aBlock raise: anObject [
  ^ self should: [  aBlock on: Deprecated do:[:ex | ex resume ]] 
      raise: anObject  description:''

]

{ #category : 'Testing' }
GsTestCase >> isAix [
 ^ self class isAix

]

{ #category : 'Testing' }
GsTestCase >> isAix6 [ 
^ self isAix and:[ (System gemVersionReport at: #osVersion) = '6' ].

]

{ #category : 'Testing' }
GsTestCase >> isAix7 [
^ self isAix and:[ (System gemVersionReport at: #osVersion) = '7' ].

]

{ #category : 'Testing' }
GsTestCase >> isArm64Darwin [
	^self class isArm64Darwin

]

{ #category : 'Testing' }
GsTestCase >> isArm64Linux [
	^self class isArm64Linux

]

{ #category : 'Testing' }
GsTestCase >> isAwsSupported [
  
  ^self isLinuxOrDarwin

]

{ #category : 'Testing' }
GsTestCase >> isAzureSupported [
  
  ^self isLinux and:[ self isX86_64 ]

]

{ #category : 'Testing' }
GsTestCase >> isDarwin [
  ^self class isDarwin
]

{ #category : 'Testing' }
GsTestCase >> isLinux [
	^self class isLinux

]

{ #category : 'Testing' }
GsTestCase >> isLinuxOrDarwin [
  
  ^self class isLinuxOrDarwin
]

{ #category : 'Testing' }
GsTestCase >> isLinuxOrDarwinOrAix [

  ^ self class isLinuxOrDarwinOrAix
]

{ #category : 'Testing' }
GsTestCase >> isLinuxOrSolaris [
  ^self class isLinuxOrSolaris

]

{ #category : 'Testing' }
GsTestCase >> isSolaris [
  ^ self class isSolaris

]

{ #category : 'Testing' }
GsTestCase >> isSolo [
  ^GsSession isSolo

]

{ #category : 'Testing' }
GsTestCase >> isSparcSolaris [
  ^ self class isSparcSolaris

]

{ #category : 'Testing' }
GsTestCase >> isUbuntuLinux [
  ^ self class isUbuntuLinux

]

{ #category : 'Testing' }
GsTestCase >> isValidException: expClass [

	expClass _isArray 
		ifTrue: [ 
			(expClass includesIdentical: AbstractException) 
				ifTrue: ["error in test code"
					self assert: false 
						description:'expected classes must not include AbstractException']] 
		ifFalse: [
			self assert: expClass ~~ AbstractException 
				description:'expected class must not == AbstractException'
	].
	^true

]

{ #category : 'Testing' }
GsTestCase >> isWindows [

	^self class isWindows

]

{ #category : 'Testing' }
GsTestCase >> isX86_64 [

  ^self class isX86_64
]

{ #category : 'Miscellaneous' }
GsTestCase >> loadAverageString [
  | str idx |
  str := System performOnServer:'uptime' .
  idx := str findString: 'load average' startingAt: 1 .
  idx ~~ 0 ifTrue:[
    idx > 1 ifTrue:[ 
      str at: idx-1 put: $[ ; at: str size put: $] .
      idx := idx-1
    ].          
    ^ str copyFrom: idx to: str size .
  ].
  ^ 'load average not available'

]

{ #category : 'Logging' }
GsTestCase >> log: aString [

GsTestCase log: aString

]

{ #category : 'Logging' }
GsTestCase >> logCr: aString [

GsTestCase logCr: aString

]

{ #category : 'Logging' }
GsTestCase >> logLoadAverage [
  self logCr:' ' , self loadAverageString , ' ' .

]

{ #category : 'Running' }
GsTestCase >> nilTestCaseInstanceVariables [
  "nil out instance variables of the test case AFTER tearDown has been called. Since all test cases
   are kept around by the testSuite, we need to clear out any objects that may be referenced by the
   test cases themselves to avoid blowing out temp obj memory while running ernie0."

  "skip #testSelector in slot 1"
  2 to: self class allInstVarNames size do: [:ivIndex |
    self instVarAt: ivIndex put: nil ]

]

{ #category : 'Debugging' }
GsTestCase >> pauseForDebug [
  self class pauseForDebug

]

{ #category : 'Printing' }
GsTestCase >> printOn: aStream [

	aStream
		nextPutAll: self class printString;
		nextPutAll: ' debug: #';
		nextPutAll: (testSelector == nil ifTrue: ['??'] ifFalse: [testSelector]).


]

{ #category : 'Miscellaneous' }
GsTestCase >> random [
    ^ self class random
]

{ #category : 'Miscellaneous' }
GsTestCase >> removeUserId: userId [
	| user |

	(user := AllUsers userWithId: userId ifAbsent: [nil]) == nil
		ifFalse: [AllUsers removeAndCleanup: user].

]

{ #category : 'Miscellaneous' }
GsTestCase >> removeUserIds: userIds [

	userIds do: [:userId | self removeUserId: userId].

]

{ #category : 'Miscellaneous' }
GsTestCase >> resultDirectory [
  | dir |
  [ dir := System clientEnvironmentVariable: 'resultdir'
  ] on: Error do:[:ex| "ignore user action error if client is a GsTsExternalSession"].
  dir ifNil:[
    dir := (System gemEnvironmentVariable: 'resultdir') ifNil:[
         "assume interactive debugging"
         GsFile serverCurrentDirectory
    ] 
  ].
  ^ dir
]

{ #category : 'Running' }
GsTestCase >> run [
        | result |
        result := GsTestResult new.
        self run: result.
        ^result

]

{ #category : 'Running' }
GsTestCase >> runCase [
	| msg tStart tEnd cnt cntStr isRO |
	msg := (tStart := Time _now) asStringMs .
	msg add: $   ;
		addAll: self printString;
		addAll:' - start...' .
	GsTestCase logCr: msg.
  [
	  [ super runCase ] ensure: [ self nilTestCaseInstanceVariables ].
  ] on: Notification do:[:ex | 
    ex class defaultHandlers size > 0 ifTrue:[ ex pass ].
    Error signal: ex asString  .
    Error signalNotTrappable: 'logic error in GsTestCase >> runCase'.
  ].
  tEnd := Time _now .
  cntStr := String new .
  (isRO := GsSession isSolo) ifFalse:[
	  cnt := System persistentCounterAt: 25 incrementBy: 1 .
	  cntStr add: $Z ; addAll: cnt asString .
  ].
	(msg := String new)
		add: 'pass; ';
		add: (tEnd subtractTime: tStart) asMilliseconds asString;
		add: ' ms. ' .
  isRO ifFalse:[ msg add: ' tranlogMark ' ; add: cntStr  ].
	GsTestCase logCr: msg.
  isRO ifFalse:[ System _markTranlogWith: cntStr ].

]

{ #category : 'Miscellaneous' }
GsTestCase >> sessions [
  | tmps | 
  ^ (tmps := SessionTemps current) at:#GsTestCase_sessions ifAbsent:[
    ^ tmps at: #GsTestCase_sessions put: { } .
  ]

]

{ #category : 'Miscellaneous' }
GsTestCase >> setAbortTimeout: sigAbTime [
  "Note this method will only work if SystemUser's password is swordfish"

  sigAbTime ~~ 60 ifTrue:[ | systemUser timeStr |
    timeStr := sigAbTime asString .
    (System stoneConfigurationAt: #StnGemAbortTimeoutSeconds) = sigAbTime ifFalse:[
      (systemUser := GsExternalSession newDefault) username: 'SystemUser'; login .
       systemUser executeString:
  'System stoneConfigurationAt: #StnGemAbortTimeoutSeconds put: ', timeStr, ' . 
   System stoneConfigurationAt: #StnGemLostOtTimeout put: ', timeStr, '.' .
       systemUser logout .
      self assert: (System stoneConfigurationAt: #StnGemAbortTimeoutSeconds) equals: sigAbTime .
      self assert: (System stoneConfigurationAt: #StnGemLostOtTimeout) equals: sigAbTime .
    ].
  ]

]

{ #category : 'Setup and Tear Down' }
GsTestCase >> setUp [
  super setUp.
  SessionTemps current at:#GsTestCase_sessions put: { } 

]

{ #category : 'Asserting' }
GsTestCase >> should: aBlock [
  "reimplement to catch any unexpected exceptions"
  [ 
    self assert: aBlock value 
  ] onException: AbstractException do:[:ex |
    self assert: false description: 'unexpected exception'
  ]

]

{ #category : 'Asserting' }
GsTestCase >> should: aBlock raise: anObject [ 

 ^ self should: aBlock raise: anObject description:'' details: nil

]

{ #category : 'Asserting' }
GsTestCase >> should: aBlock raise: anObject description: descString [
  ^ self should: aBlock raise: anObject description: descString details: nil
]

{ #category : 'Asserting' }
GsTestCase >> should: aBlock raise: anObject description: descString details: detailsString [
	| expectedErrorNumber expClass |
	anObject _isInteger 
		ifTrue: [	expectedErrorNumber := anObject] 
		ifFalse: [
			anObject _isSymbol  
				ifTrue: [expectedErrorNumber := ErrorSymbols at: anObject] 
				ifFalse: [
					^(anObject isBehavior and: [anObject isSubclassOf: AbstractException])
						ifTrue: [self should: aBlock raiseClasses: anObject numbers: #() description: descString details: detailsString ] 
						ifFalse: [super should: aBlock raise: anObject]
			]		
	].	
	
	(expClass := Exception legacyNumberToClasses: expectedErrorNumber) ifNil:[
		^self assert: false description: 'invalid error number' 
	].
	self 
		should: aBlock 
		raiseClasses: expClass 
		numbers: (expectedErrorNumber ifNil:[ #() ] ifNotNil:[ { expectedErrorNumber } ])
		description: descString details: detailsString

]

{ #category : 'Asserting' }
GsTestCase >> should: aBlock raise: anObject details: aString [ 

 ^ self should: aBlock raise: anObject description:'' details: aString

]

{ #category : 'Asserting' }
GsTestCase >> should: aBlock raiseClasses: expClass numbers: expectedErrorNumbers description: descString [
  "expClass is either a Class or an Array of Classes."
  ^ self should: aBlock raiseClasses: expClass numbers: expectedErrorNumbers description: descString details: nil
]

{ #category : 'Asserting' }
GsTestCase >> should: aBlock raiseClasses: expClass numbers: expectedErrorNumbers description: descString 
   details: detailsString [
  "expClass is either a Class or an Array of Classes."
  
	| saveEx success exString actualErrorNumber |
	saveEx := nil . "force method temps here for debugging"
	actualErrorNumber := nil .
	exString := ' no exception raised ' .
	success := false .
	self isValidException: expClass.
	[
	  aBlock onException: expClass do: [:ex | 
		  saveEx := ex . "for debugging"
		  actualErrorNumber := ex number.
		  exString := ex description , ' (' , actualErrorNumber printString , ')'.
		  success := true  .
    ].
	] onException: AbstractException do: [:ex |
    "(ex isKindOf: AlmostOutOfMemoryError) ifTrue:[ ex pass ]." "uncomment to debug ernie10"
		saveEx := ex .
		actualErrorNumber := ex number.
		exString := ex description , ' (' , actualErrorNumber printString , ')'.
		[
			(GsCurrentSession currentSession resolveSymbol:#GciError)  ifNotNil: [ :assoc |
				ex class == assoc _value ifTrue:[
					success ifFalse: [success := ex matchesClasses: expClass]. 
				]
			]
		] onSynchronous: AbstractException do: [:exb |  exb return "ignore"].
     
		"do the assert inside handler, so original signal  point still on the stack!"
		self 
			assert: success 
			description: 'Expected ', expClass printString , 
				' (' , expectedErrorNumbers printString , ') ' , 
				descString , ', GOT ' , exString.
		^self "if execution continued..."
	].
	self 
		assert: success 
		description: 'Expected ', expClass printString , 
			' (' , expectedErrorNumbers printString , ') ' , 
			descString , ', GOT ' , exString.
  detailsString ifNotNil:[ | eStr |
    ((eStr := saveEx asString) includesString: detailsString) ifFalse:[
      self assert: false description:'Error message details do not match'.
    ].
  ].

]

{ #category : 'Asserting' }
GsTestCase >> should: aBlock raiseOneOf: anArray [ 

  "anArray should be an Array of Integers or error symbols or exception classes "
  | expClasses expNumbers |
  expClasses := { } .
  expNumbers := { } .
  anArray do:[:elem | | errNum aClass |
     elem _isSymbol ifTrue:[ 
       errNum := ErrorSymbols at: elem 
     ] ifFalse:[ 
       elem _isInteger ifTrue:[ 
         errNum := elem .
         expNumbers add: errNum .
       ] ifFalse:[
         (elem isBehavior and:[ elem isSubclassOf: Exception]) ifTrue:[ 
            aClass := elem
         ] ifFalse:[
            ^ self assert: false description: 'invalid element in arg raiseOneOf:'
     ]]].
     aClass ifNil:[ (aClass := Exception legacyNumberToClasses: errNum) ifNil:[
       ^ self assert: false description: 'invalid error number' 
     ]].
     aClass _isArray ifTrue:[ expClasses addAll: aClass ]
       ifFalse:[ expClasses add: aClass ]. 
  ].
  self should: aBlock raiseClasses: expClasses numbers: expNumbers description: ''

]

{ #category : 'Asserting' }
GsTestCase >> signalFailure: aString [
        GsTestResult failure sunitSignalWith: aString

]

{ #category : 'Miscellaneous' }
GsTestCase >> smFloatLiteralClass [ 
^ 1.0 class .


]

{ #category : 'Miscellaneous' }
GsTestCase >> smFloatResultClass [ 
^ (1.0 + 1.0) class

]

{ #category : 'Setup and Tear Down' }
GsTestCase >> tearDown [
  | sessions tm nSess cnt |
  sessions := SessionTemps current removeKey: #GsTestCase_sessions ifAbsent:[ #() ].
  (nSess := sessions size) > 0 ifTrue:[ 
    1 to: nSess do:[:j | | sess |
      (sess := sessions at: j ) ifNotNil:[
        sess isLoggedIn ifFalse:[ sess := nil ].
        sess ifNotNil:[ 
          [
            sess isCallInProgress ifTrue:[ sess forceLogout . 
            ] ifFalse:[ 
             sess nbLogout 
            ].
          ] on: self gciErrorClass do:[:ex | 
             ex originalNumber == 4100 ifTrue:["ignore invalid session"] 
                                       ifFalse:[ ex pass ]
          ].
        ].
      ].
      sessions at: j put: sess.
    ].
    sessions do:[:aSess | aSess ifNotNil:[ aSess _waitForLogout ]].
    tm := Time now .
    "allow for one other session owned by a TestResource."
    [ (cnt := System currentUserSessionCount) > 2 ] whileTrue:[
      Delay waitForMilliseconds:50 .
      (Time now subtractTime: tm ) asMilliseconds > 8000 ifTrue:[
         self log: '(--', System currentSessionsReport , '--)'.
         Error signal:'sessions not logged out'
      ].
    ]
  ].
  super tearDown.

]

{ #category : 'Miscellaneous' }
GsTestCase >> untrappableError [

" Send this message for debugging the test framework,
  so you can get a stack with no trapping of the error by the test framework.
  Error 6004 (HardBreak) is not trappable by Smalltalk code ."

System signal:6004 args:#() signalDictionary:GemStoneError

]
