Extension { #name : 'GsUnaryClause' }

{ #category : 'instance creation' }
GsUnaryClause class >> clause: clause operator: anOperator [
  ^ self new
    clause: clause;
    operator: anOperator;
    yourself

]

{ #category : 'private' }
GsUnaryClause >> _predicateAndEvaluatorsDo: aBlock [
  "visit all predicates in receiver and provide access to predicate and list of evaluators
   for the predicate"

  super _predicateAndEvaluatorsDo: aBlock.
  self clause _predicateAndEvaluatorsDo: aBlock

]

{ #category : 'testing' }
GsUnaryClause >> _queryUsesIdentityIndex [
  "answer true if the one or more of the clause has an evaluator that is an identity index"

  ^ clause _queryUsesIdentityIndex

]

{ #category : 'testing' }
GsUnaryClause >> _queryUsesIndex [
  "answer true if the one or more of the clause has an evaluator that is a range or identity index"

  ^ clause _queryUsesIndex

]

{ #category : 'visiting' }
GsUnaryClause >> acceptVisitor: aFormulaVisitor [
  super acceptVisitor: aFormulaVisitor.
  aFormulaVisitor acceptUnaryClause: self

]

{ #category : 'optimizing' }
GsUnaryClause >> applyDeMorgansNegationTransform [
  "actively apply De Morgan's laws ... operating on copy"

  | transformedClause |
  transformedClause := self operator == #'not'
    ifTrue: [ self clause applyDeMorgansTransform ]
    ifFalse: [ self clause applyDeMorgansNegationTransform ].
  self
    operator: #'null';
    clause: transformedClause.
  ^ self

]

{ #category : 'optimizing' }
GsUnaryClause >> applyDeMorgansTransform [
  "do not transform, but propagate the transform ... operating on copy"

  | transformedClause |
  transformedClause := self operator == #'not'
    ifTrue: [ self clause applyDeMorgansNegationTransform ]
    ifFalse: [ self clause applyDeMorgansTransform ].
  self
    operator: #'null';
    clause: transformedClause.
  ^ self

]

{ #category : 'converting' }
GsUnaryClause >> asFormulaWithSelectorParts: selectorParts arguments: arguments [
  | selector |
  selector := selectorParts first inputValue asSymbol.
  ^ self perform: selector with: arguments first

]

{ #category : 'transforming' }
GsUnaryClause >> bind: variableName to: value [
  | bound |
  bound := self copy.
  bound clause: (bound clause bind: variableName to: value).
  ^ bound immediateInvariant

]

{ #category : 'private' }
GsUnaryClause >> bindEvaluators [
  clause := clause bindEvaluatorsFor: nsc collator: self collator.
  super bindEvaluators

]

{ #category : 'accessing' }
GsUnaryClause >> clause [
  ^ clause

]

{ #category : 'accessing' }
GsUnaryClause >> clause: newValue [
  clause := newValue

]

{ #category : 'querying-private' }
GsUnaryClause >> elementValue: anObject [
  "the pathTerms in the query will use anObject as the starting point ... analagous to an object in the nsc that is bound to a query"

  | result |
  result := self clause elementValue: anObject.
  self operator == #'null'
    ifTrue: [ ^ result ].
  ^ result not

]

{ #category : 'querying' }
GsUnaryClause >> executeAndDo: aBlock [
  ^ self operator == #'not'
    ifTrue: [ self clause executeNegatedAndDo: aBlock ]
    ifFalse: [ self clause executeAndDo: aBlock ]

]

{ #category : 'querying-private' }
GsUnaryClause >> executeClause [
  ^ self operator == #'not'
    ifTrue: [ self clause executeClauseNegated ]
    ifFalse: [ self clause executeClause ]

]

{ #category : 'querying-private' }
GsUnaryClause >> executeClauseNegated [
  ^ self operator == #'not'
    ifTrue: [ self clause executeClause ]
    ifFalse: [ self clause executeClauseNegated ]

]

{ #category : 'querying-private' }
GsUnaryClause >> executeNegatedAndDo: aBlock [
  ^ self operator == #'not'
    ifTrue: [ self clause executeAndDo: aBlock ]
    ifFalse: [ self clause executeNegatedAndDo: aBlock ]

]

{ #category : 'private' }
GsUnaryClause >> immediateInvariant [
  super immediateInvariant.
  clause immediateInvariant

]

{ #category : 'testing' }
GsUnaryClause >> isConjunctiveNormalForm [
  ^ self operator == #'null' and: [ self clause isConjunctiveNormalForm ]

]

{ #category : 'testing' }
GsUnaryClause >> isDisjunctiveNormalForm [
  ^ self operator == #'null' and: [ self clause isDisjunctiveNormalForm ]

]

{ #category : 'testing' }
GsUnaryClause >> isNegationClause [
  ^ self operator == #'not'

]

{ #category : 'operators' }
GsUnaryClause >> normalize [
  | normalized |
  normalized := super normalize.
  normalized clause: clause normalize

]

{ #category : 'accessing' }
GsUnaryClause >> operator [
  ^operator

]

{ #category : 'accessing' }
GsUnaryClause >> operator: aSymbol [
  operator := aSymbol

]

{ #category : 'private' }
GsUnaryClause >> postCopy [
  clause := clause copy.
  super postCopy

]

{ #category : 'printing' }
GsUnaryClause >> printOn: aStream [
  | parenthesizeClause |
  parenthesizeClause := self clause isCompoundPredicate
    and: [ self operator ~~ #'null' ].
  parenthesizeClause
    ifTrue: [ aStream nextPut: $( ].
  aStream nextPutAll: self clause printString.
  parenthesizeClause
    ifTrue: [ aStream nextPut: $) ].
  self operator ~~ #'null'
    ifTrue: [ aStream nextPutAll: ' ' , self operator asString ]

]

{ #category : 'transforming' }
GsUnaryClause >> unbind [
  "remove all bindings"

  | unbound |
  unbound := super unbind.
  unbound clause: unbound clause unbind.
  ^ unbound immediateInvariant

]
