Extension { #name : 'IcuLocale' }

{ #category : 'Private' }
IcuLocale class >> _locales: opcode value: aValue [
  "opcode   libicu function
       0    Locale::getISOCountries
       1    Locale::getISOLanguages
       2    Locale::getDefault
       3    Locale::setDefault
       5    Locale::getAvailableLocales
       6    Collator::getAvailableLocales
  "
<primitive: 914>
opcode == 3 ifTrue:[ aValue _validateClass: IcuLocale ]
            ifFalse:[ aValue _validateClass: UndefinedObject ].
self _primitiveFailed: #_locales:value: args: { opcode . aValue }

]

{ #category : 'Accessing' }
IcuLocale class >> availableLocales [
  "calls Locale::getAvailableLocales() in locid.h. Returns an Array of IcuLocales"
  ^ self _locales: 5 value: nil

]

{ #category : 'Instance Creation' }
IcuLocale class >> default [
  "Return an IcuLocale equivalent to the default Locale.
   If  IcuLocale(C)>>default: has not been executed,
   the result is dependent on the locale of the operating system.

   If the locale of the operating system would imply a locale
   en_US_POSIX, then the locale en_US is returned instead,
   since en_US_POSIX is intended for binary collation and does
   not support case-insensitive searches.
"
  ^ self _locales: 2 value: nil

]

{ #category : 'Instance Creation' }
IcuLocale class >> default: anIcuLocale [
  "Sets the default Locale for ICU library to the argument.
   Also installs a default IcuCollator for the argument.
   Returns the receiver."
  | coll |
  coll := IcuCollator forLocale: anIcuLocale .
  self _locales: 3 value: anIcuLocale .
  IcuCollator default: coll .

]

{ #category : 'Instance Creation' }
IcuLocale class >> getUS [
  ^ self language:'EN' country:'US' variant:nil

]

{ #category : 'Accessing' }
IcuLocale class >> isoCountries [
  "calls Locale::getISOCountries() in locid.h Returns an Array of Strings"
  ^ self _locales: 0 value: nil

]

{ #category : 'Accessing' }
IcuLocale class >> isoLanguages [
  "calls Locale::getISOLanguages() in locid.h  Returns an Array of Strings"
  ^ self _locales: 1 value: nil

]

{ #category : 'Instance Creation' }
IcuLocale class >> language: langString country: countryString variant: variantString [

" Return a new Locale for the specified language, country and variant.
  Calls  Locale::Locale(const char*  language,
                        const   char*  country,
                        const   char*  variant);
  See source/icu/common/unicode/locid.h
  or  icu-project.org/apiref/icu4c/classLocale.html  for more details

  Arguments must be nil or Strings of size 127 or less.
  langString must be nonNil"
<primitive: 912>

self _primitiveFailed: #language:country:variant:
	args:{ langString . countryString. variantString}

]

{ #category : 'Instance Creation' }
IcuLocale class >> new [
  "disallowed"
  self shouldNotImplement: #new

]

{ #category : 'Private' }
IcuLocale >> _getAttribute: opcode [
  "opcodes: 0 getLanguage, 1 getScript, 2 getCountry, 3 getVariant, 4 getName
    5 getDisplayName "
<primitive: 913>
self _primitiveFailed: #_getAttribute: args: { opcode }

]

{ #category : 'Accessing' }
IcuLocale >> _name [
    "Returns a String, the name of the entire locale,
   with the language, country and variant separated by underbars.
   calls Locale::getName() in locid.h"
  ^ self _getAttribute: 4

]

{ #category : 'Comparing' }
IcuLocale >> = anIcuLocale [

anIcuLocale == self ifTrue:[ ^ true ].
(anIcuLocale isKindOf: IcuLocale) ifFalse:[ ^ false ] .
^ name = anIcuLocale name

]

{ #category : 'Formatting' }
IcuLocale >> asString [
  ^ 'IcuLocale ' , name asString

]

{ #category : 'Accessing' }
IcuLocale >> country [
  "Returns a String, the locale's ISO-3166 country code.
   See Locale::getCountry() in locid.h "
  ^ self _getAttribute: 2

]

{ #category : 'Accessing' }
IcuLocale >> displayName [
  "Returns displayable name of the receiver using the en_US locale.
   Equivalent to Locale::getDisplayName() in locid.h "
  ^ self _getAttribute: 5

]

{ #category : 'Comparing' }
IcuLocale >> hash [
^ name hash

]

{ #category : 'Accessing' }
IcuLocale >> language [
  "Returns a String, the locale's ISO-639 language code.
   See Locale::getLanguage() in locid.h "
  ^ self _getAttribute: 0 .

]

{ #category : 'Accessing' }
IcuLocale >> name [
  "Returns a String, the name of the entire locale,
   with the language, country and variant separated by underbars.
   Equivalent to Locale::getName() in locid.h "
  ^ name

]

{ #category : 'Formatting' }
IcuLocale >> printOn: aStream [

  aStream nextPutAll: self asString .

]

{ #category : 'Accessing' }
IcuLocale >> script [
  "Returns a String, the locale's ISO-15924 abbreviation script code.
   See Locale::getScript() in locid.h "
  ^ self _getAttribute: 1

]

{ #category : 'Accessing' }
IcuLocale >> variant [
  "Returns a String, the locale's variant code.
   See Locale::getVariant() in locid.h "
  ^ self _getAttribute: 3

]
