Extension { #name : 'IcuSortedCollection' }

{ #category : 'Instance Creation' }
IcuSortedCollection class >> new [
"Returns a new instance of the receiver which will use
 the current default IcuLocale"

| inst |
(inst := self _basicNew) _initialize: IcuCollator default .
^inst

]

{ #category : 'Instance Creation' }
IcuSortedCollection class >> newUsingCollator: aCollator [
"Returns a new instance of the receiver which will use
 the specified collator."

| inst |
(inst := self _basicNew) _initialize: aCollator .
^inst

]

{ #category : 'Initialize' }
IcuSortedCollection >> _defaultBlock [

 ^ [ :a :b :aCollator | (a compareTo: b collator: aCollator useMinSize: false) <= 0 ]

]

{ #category : 'Private' }
IcuSortedCollection >> _findIndex: anObject [

"Finds and returns the index for placing the given object into the receiver.
 A simple binary probe is used."

| obj lower upper half probe |

self size == 0 ifTrue: [ ^1 ].

lower := 1.
upper := self size.
[ half := upper - lower + 1 // 2.
  probe := lower + half.
  -1 < half and: [ lower <= upper ] ] whileTrue: [
  obj := self at: probe.
  (sortBlock value: obj value: anObject value: collator) ifTrue: [
    "after this index - go to the next higher cell"
    lower := probe + 1.
  ]
  ifFalse: [
    "before this index - go to the next lower cell"
    upper := probe - 1
  ]
].
^probe

]

{ #category : 'Initialize' }
IcuSortedCollection >> _initialize: aCollator [
  "Use a copy of aCollator to disallow changes to strength, etc,
   that would require a resort of the receiver."
  collator := aCollator copy immediateInvariant .
  sortBlock := self _defaultBlock .

]

{ #category : 'Copying' }
IcuSortedCollection >> _mergeSortAddAll: aCollection [

"The receiver must be empty. Adds aCollection to the receiver
 using the merge-sort implementation provided by BlockSorter.
 Returns the receiver."

| arr |
self size == 0 ifFalse:[ self error:'not empty' ].
arr := aCollection sortWithBlock: sortBlock collator: collator .  "merge sort"
super _insertAll: arr at: 1 .
^ self

]

{ #category : 'Accessing' }
IcuSortedCollection >> collator [
  ^ collator

]

{ #category : 'Updating' }
IcuSortedCollection >> collator: aCollator [
  "Change the receiver's collator and resort the receiver.
   Use a copy of aCollator to disallow changes to strength, etc,
   that would require a resort of the receiver."

  collator := aCollator copy immediateInvariant .
  self size > 1 ifTrue: [
    self resort
  ]

]

{ #category : 'Searching' }
IcuSortedCollection >> indexOf: anObject [

"Returns the index of the first occurrence of an object equal to anObject
 in the receiver.  If the receiver does not contain such an object, this
 returns zero."

| idx |
" _findIndex: returns where anObject would be inserted, so look at
surrounding slots to see if the object is present "
idx := self _findIndex: anObject.

(sortBlock value: anObject value: anObject value: collator) ifTrue: [
  "Need to look at the slots less than idx"
  | foundIdx | "We need to find the first one that is equal"
  foundIdx := 0.
  (idx-1) _downTo: 1 do: [ :i |
    (anObject = (self at: i)) ifTrue: [foundIdx := i].
    (i > 1) ifTrue: [
      "Check to see if the next element might be equal to the current element
       according to the sort block."
      (sortBlock value: (self at: i) value: (self at: i-1) value: collator) ifFalse: [
        "We don't need to look at any more because i-1 can't be equal
         to i."
        ^foundIdx
      ].
    ].
  ].
  ^foundIdx
] ifFalse: [
  "Need to look at the slots greater than idx"
  | lastIdx |
  lastIdx := self size.
  idx to: lastIdx do: [ :i |
    (anObject = (self at: i)) ifTrue: [^i].
    (i < lastIdx) ifTrue: [
      "Check to see if the next element might be equal to the current element
       according to the sort block."
      (sortBlock value: (self at: i) value: (self at: i+1) value: collator) ifTrue: [
        "We don't need to look at any more because i+1 can't be equal
         to i."
        ^0
      ].
    ].
  ].
  ^0.
].

]

{ #category : 'Searching' }
IcuSortedCollection >> indexOfIdentical: anObject [

"Private.  Returns the index of the first element in the receiver that is
 identical to the argument.  If the receiver does not have any elements that are
 identical to the argument, returns zero."

| idx |

(self size <= 2000) ifTrue: [
  "OrderedCollection's indexOfIdentical: uses a primitive and is
   faster for smaller SortedCollections"
  ^super indexOfIdentical: anObject.
].

" _findIndex: returns where anObject would be inserted, so look at
surrounding slots to see if the object is present "
idx := self _findIndex: anObject.

(sortBlock value: anObject value: anObject value: collator) ifTrue: [
  "Need to look at the slots less than idx"
  | foundIdx | "We need to find the first one that is equal"
  foundIdx := 0.
  (idx-1) _downTo: 1 do: [ :i |
    (anObject == (self at: i)) ifTrue: [foundIdx := i].
    (i > 1) ifTrue: [
      "Check to see if the next element might be equal to the current element
       according to the sort block."
      (sortBlock value: (self at: i) value: (self at: i-1) value: collator) ifFalse: [
        "We don't need to look at any more because i-1 can't be equal
         to i."
        ^foundIdx
      ].
    ].
  ].
  ^foundIdx
] ifFalse: [
  "Need to look at the slots greater than or equal to idx"
  | lastIdx |
  lastIdx := self size.
  idx to: lastIdx do: [ :i |
    (anObject == (self at: i)) ifTrue: [
      ^i
    ].
    (i < lastIdx) ifTrue: [
      "Check to see if the next element might be equal to the current element
       according to the sort block."
      (sortBlock value: (self at: i) value: (self at: i+1) value: collator) ifTrue: [
        "We don't need to look at any more because i+1 can't be equal
         to i."
        ^0
      ].
    ].
  ].
  ^0.
].

]
