Extension { #name : 'IdentityIndex' }

{ #category : 'Support' }
IdentityIndex >> _addAllFor: aKey
into: anNscBuilderOrArray
offset: offset
pathTerm: pathTerm [

"Traverse the reverse mappings for the given key, adding the object at the end
 of the traversal to the given NscBuilder or Array."

<primitive: 328>
self _primitiveFailed: #_addAllFor:into:offset:pathTerm: .
self _uncontinuableError

]

{ #category : 'Testing' }
IdentityIndex >> _checkSameIndexOptionsAs: anIndexSpec [
  ^ self termsRequired == anIndexSpec requirePathTerms

]

{ #category : 'Updating' }
IdentityIndex >> _clear [

"Assigns nil to important instance variables and sets the receiver's size
 to zero."

indexDictionary := nil.
nscRoot := nil.
rcRangeBucket := nil.
self size: 0.

]

{ #category : 'Testing' }
IdentityIndex >> _commonPathTermsWith: anEvaluatorOrIndex [
  ^ anEvaluatorOrIndex _commonPathTermsWithIndex: self

]

{ #category : 'Testing' }
IdentityIndex >> _commonPathTermsWithIndex: anIndex [
  | commonSize commonTerms |
  commonSize := self size min: anIndex size.
  commonTerms := {}.
  1 to: commonSize do: [ :index |
    | term |
    (term := (self at: index) name) = (anIndex at: index) name
      ifFalse: [ ^ commonTerms ].
    commonTerms add: term ].
  ^ commonTerms

]

{ #category : 'Testing' }
IdentityIndex >> _commonPathTermsWithPathEvaluator: anPathEvaluator [
  | commonSize commonTerms |
  commonSize := self size min: anPathEvaluator size.
  commonTerms := {}.
  1 to: commonSize do: [ :index |
    | term |
    (term := (self at: index) name) = (anPathEvaluator at: index)
      ifFalse: [ ^ commonTerms ].
    commonTerms add: term ].
  ^ commonTerms

]

{ #category : 'Indexing Support' }
IdentityIndex >> _doPutCommonPathTermsForPathArray: pathArray for: anIndexList [

"Find the common path terms that already exist for indexes whose path
 components match pathArray (an Array of Strings).  Adds those path terms to
 the end of receiver.  Do not put a common path term into receiver that
 is the last path term for an existing index unless the existing index
 is an identity index."

| firstUnique somePathTerms j minSize sz someSz someIsIdentityIndex isRangeIndex |
firstUnique := 0.
isRangeIndex := self isRangeEqualityIndex.
sz := pathArray size.
1 to: anIndexList size by: 2 do: [ :i |
    j := anIndexList at: i + 1.
    " only consider index objects for which the NSC is the root "
    j == 1
        ifTrue: [
            somePathTerms := anIndexList at: i.
            someIsIdentityIndex := somePathTerms isIdentityIndex.
            someSz := somePathTerms size.
            minSize := someSz min: sz.

            [ j <= minSize and:
            [ (somePathTerms at: j) name = (pathArray at: j) ] ] whileTrue: [
                ( j > firstUnique and:
                [ ( j < someSz or: [ someIsIdentityIndex and: [ j == someSz ] ] )_and:
                [ j ~~ sz or: [ isRangeIndex not ] ] ] )
                   ifTrue: [
                       firstUnique := j.
                       self addLast: (somePathTerms at: j)
                   ].
                j := j + 1
            ]
        ]
].

]

{ #category : 'Error Handling' }
IdentityIndex >> _errorCannotInvokeRangeOperationOnIdentityIndex [

"An attempt was made to use a range operation (< > = <= >=) on a path
 expression only supported by an identity index."

^ self _error: #rtErrIdentityIndexCannotInvokeRangeOperation

]

{ #category : 'Searching' }
IdentityIndex >> _findFirstUnsharedPathTerm [
  "Returns the first path term that is not shared by any other indexes.  This is
 the first path term whose only index is the receiver.  If one is not found
 (the index shares the same path with another index), then returns nil."

  | pathTerm sz indexObj |
  sz := self size.
  1 to: sz do: [ :i |
    pathTerm := self at: i.
    (pathTerm size == 1 and: [ self == (pathTerm at: 1) ])
      ifTrue: [ ^ pathTerm ] ].
  pathTerm := self at: self size.	" check if there is an equality and identity index on the same path "
  1 to: pathTerm size do: [ :i |
    " for each index that utilizes the last path term "
    indexObj := pathTerm at: i.
    (indexObj ~~ self and: [ indexObj size == sz ])
      ifTrue: [
        " if some other index has the same number of path terms "
        ^ pathTerm ] ].
  ^ nil

]

{ #category : 'Support' }
IdentityIndex >> _findSetValuedPathTermOffset [
  "Returns the offset of the set-valued path term in the receiver.  If none
 exists, returns 0."

  1 to: self size do: [ :i |
    (self at: i) indicatesMultiValue
      ifTrue: [ ^ i ] ].
  ^ 0

]

{ #category : 'Testing' }
IdentityIndex >> _isIndexObject [

"Returns true if the receiver is an index object; returns false otherwise."

^ true

]

{ #category : 'Indexing Support' }
IdentityIndex >> _partialPathComponentsStringUpTo: offset [

"Returns the path components string up to the given offset."

| str |
str := String new.
1 to: (offset min: self size) do: [ :i |
    str add: (self at: i) name.
    i == offset
        ifFalse: [ str add: $. ]
].
^ str

]

{ #category : 'Indexing Support' }
IdentityIndex >> _preIndexCreationBuildIndexDictionaryFor: pathArray for: anNsc [
  "afford an opportunity to resize dictionary before adding new elements during index creation"

  (pathArray size == 1 and: [ self isRangeEqualityIndex ])
    ifFalse: [ | indexDict |
      anNsc _calculateIndexDictionarySize: pathArray size.
      indexDict := anNsc _getIndexDictionary.
      self indexDictionary: indexDict.
      indexDict rebuildTable: anNsc _getIndexDictionaryCreationSize for: self ]

]

{ #category : 'Updating' }
IdentityIndex >> _setPathTermState [

"For each path term, indicate the need to update the index dictionary."

1 to: self size do: [ :i |
    (self at: i) updateDict: indexDictionary.
]

]

{ #category : 'Statistics' }
IdentityIndex >> _statisticsInto: dict [

"Puts statistical information into the given dictionary."


]

{ #category : 'Traversing' }
IdentityIndex >> _traverseObject: anObject [
  "Traverse the sub-objects of the given object using the path terms of the
 receiver.  This method assumes that the index is not over a set-valued
 instance variable.  For that kind of index, use the traverse:startingAt:into:
 method.  If a nil value is reached before the end of the path, returns the
 incompletePathTraversal object."

  | nextObj sz pathTerm |
  nextObj := anObject.
  sz := self size.
  1 to: sz do: [ :i |
    pathTerm := self at: i.
    nextObj := pathTerm _getNextObjectForTraversal: nextObj.
    (nil == nextObj and: [ i ~~ sz ])
      ifTrue: [ ^ #'_incompletePathTraversal' ] ].
  ^ nextObj

]

{ #category : 'Traversing' }
IdentityIndex >> _traverseObject: anObject incompletesInto: incompleteArray incomplete: incomplete [
  "Traverse the sub-objects of the given object using the path terms of the
 receiver.  This method assumes that the index is not over a set-valued
 instance variable.  For that kind of index, use the traverse:startingAt:into:
 method.  If a nil value is reached before the end of the path, place the object
 into the appropriate Array in the incompleteArray of Arrays."

  | nextObj sz pathTerm |
  nextObj := anObject.
  sz := self size.
  1 to: sz do: [ :i |
    pathTerm := self at: i.
    nextObj := pathTerm _getNextObjectForTraversal: nextObj.
    (nil == nextObj and: [ i ~~ sz ])
      ifTrue: [
        incompleteArray == nil
          ifTrue: [ ^ nil ].
        (incompleteArray at: i + 1) add: anObject.
        ^ incomplete ] ].
  ^ nextObj

]

{ #category : 'Updating Indexes' }
IdentityIndex >> addDirectMappingsFor: aBag
indexList: iList [

"Add an entry for an index on the elements of the NSC."

| anObject firstPathTerm indexMgr |

indexMgr := self indexManager.

firstPathTerm := self firstPathTerm.
indexMgr autoCommit
  ifTrue: [ nscRoot _lockForIndexCreation ].

1 to: aBag size do: [ :i |
  anObject := aBag _at: i.
  " insert an entry into the index dictionary "
  indexDictionary
	_at: anObject
	put: anObject
	term: firstPathTerm
	logging: false.
  (i \\ 100) == 0 ifTrue:[ indexMgr commitIndexMaintenance: self at: i ].
]

]

{ #category : 'Converting' }
IdentityIndex >> asIndexSpecification [
  ^ (IdentityIndexSpecification path: self pathComponentsString)
    requirePathTerms: self termsRequired;
    legacyIndex: true;
    yourself

]

{ #category : 'Support' }
IdentityIndex >> asPathEvaluator [
  "Returns a PathEvaluator over the same path as the receiver."

  | pathEval holder |
  pathEval := PathEvaluator new.
  1 to: self size do: [ :i | pathEval add: (self at: i) name ].
  pathEval nsc: nscRoot.
  holder := { pathEval } .
  pathEval := nil .
  ^ PathEvaluator asMostSpecificType: holder .

]

{ #category : 'Converting' }
IdentityIndex >> asQueryEvaluator [

  ^IdentityIndexQueryEvaluator on: self

]

{ #category : 'Converting' }
IdentityIndex >> asString [
  | str |
  str := '('.
  1 to: self size do: [ :index |
    | pt |
    pt := self at: index.
    str := str , pt name asString.
    index < self size
      ifTrue: [ str := str , '.' ] ].
  ^ super asString , str , ')'

]

{ #category : 'Indexing Support' }
IdentityIndex >> btreeComparisonQuerySpec [
  ^ BtreeComparisonQuerySpec new

]

{ #category : 'Clustering' }
IdentityIndex >> clusterDepthFirst [

"Cluster the receiver.  Only need to cluster the path terms."

1 to: self size do: [ :i |
  (self at: i) clusterDepthFirst
]

]

{ #category : 'Accessing' }
IdentityIndex >> collator [
  "Returns IcuCollator to be used when comparing Unicode strings ... nil indicates that default collator should be used."

  ^ nil

]

{ #category : 'Indexing Support' }
IdentityIndex >> comparisonForCompare [
  ^ BtreeComparisonForCompare newForComparison: self collator

]

{ #category : 'Accessing' }
IdentityIndex >> firstPathTerm [

"Returns the first path term in the path components list."

^ self at: 1

]

{ #category : 'Testing' }
IdentityIndex >> hasCollectionBasedTerm [
  "Returns true if the path has an enumerated path term or a term that indicates a
   set-valued instance variable."

  ^ self hasEnumeratedTerm or: [ self hasSetValuedTerm ]

]

{ #category : 'Testing' }
IdentityIndex >> hasEnumeratedTerm [
  "Returns true if the path has a term that indicates an enumerated path term."

  1 to: self size do: [ :i |
    (self at: i) isEnumeratedTerm
      ifTrue: [ ^ true ] ].
  ^ false

]

{ #category : 'Testing' }
IdentityIndex >> hasIndexDictionary [
  "Answer if receiver has an indexDictionary."

  ^ self indexDictionary notNil

]

{ #category : 'Testing' }
IdentityIndex >> hasIndexOnPath: pathArray [
  "Returns whether the receiver's path components match the path represented by
 pathArray."

  self size == pathArray size
    ifFalse: [ ^ false ].
  1 to: self size do: [ :i |
    ((self at: i) hasMatchingTerm: (pathArray at: i))
      ifFalse: [ ^ false ] ].
  ^ isComplete

]

{ #category : 'Testing' }
IdentityIndex >> hasIndexOnPath: pathArray startingAt: offset [

"Returns whether the receiver's path components match the path represented by
 pathArray."

self size - offset + 1 == pathArray size
  ifFalse: [ ^ false ].

offset to: self size do: [ :i |
  (self at: i) name == (pathArray at: i + 1 - offset)
    ifFalse: [ ^ false ].
].
^ isComplete

]

{ #category : 'Support' }
IdentityIndex >> hasSamePathAs: anIndex [

"Returns whether the given index is on the same path as the receiver.
 This can only be true if the index is identical."

^ self == anIndex

]

{ #category : 'Testing' }
IdentityIndex >> hasSetValuedTerm [
  "Returns true if the receiver has any path term that indicates a set-valued
 instance variable."

  1 to: self size do: [ :i |
    (self at: i) isSetValuedTerm
      ifTrue: [ ^ true ] ].
  ^ false

]

{ #category : 'Accessing' }
IdentityIndex >> indexDictionary [

"Returns the value of the instance variable 'indexDictionary'."

^indexDictionary

]

{ #category : 'Updating' }
IdentityIndex >> indexDictionary: newValue [

"Modifies the value of the instance variable 'indexDictionary'."

indexDictionary := newValue

]

{ #category : 'Accessing' }
IdentityIndex >> indexManager [

^IndexManager current

]

{ #category : 'Accessing' }
IdentityIndex >> indexType [
  ^ #'identity'

]

{ #category : 'Testing' }
IdentityIndex >> isComplete [

"Returns the value of the instance variable 'isComplete'."

isComplete == nil
  ifTrue: [ ^ true ].
^ isComplete

]

{ #category : 'Updating' }
IdentityIndex >> isComplete: newValue [

"Modifies the value of the instance variable 'isComplete'."

isComplete := newValue

]

{ #category : 'Testing' }
IdentityIndex >> isIdentityIndex [

"Returns true."

^ true

]

{ #category : 'Testing' }
IdentityIndex >> isIndexOnNscElements [

"Returns whether the receiver is an index directly on the elements of an NSC
 (either the root NSC or a set-valued instance variable on the path)."

^ self lastPathTerm indicatesIndexOnNscElements

]

{ #category : 'Testing' }
IdentityIndex >> isIndexOnRootNsc [

"Returns whether the receiver is an index directly on the elements of
 the root NSC."

^ self lastPathTerm indicatesIndexOnRootNsc

]

{ #category : 'Testing' }
IdentityIndex >> isLegacyIndex [

"Legacy indexes are kinds of IdentityIndex, so return true."

^ true

]

{ #category : 'Testing' }
IdentityIndex >> isPathEvaluator [

"Returns false."

^ false

]

{ #category : 'Testing' }
IdentityIndex >> isRangeEqualityIndex [

"Returns false."

^ false

]

{ #category : 'Testing' }
IdentityIndex >> isStreamable [

"Returns false."

^ false

]

{ #category : 'Accessing' }
IdentityIndex >> lastElementClass [

^ self error: 'not valid for identity index on ' , self pathComponentsString

]

{ #category : 'Accessing' }
IdentityIndex >> lastElementClassDescription [
  "answer a description of the lastElementClass of the receiver,
   suitable for use in an error message"

  ^ 'No lastElement class for Identity Indexes'

]

{ #category : 'Accessing' }
IdentityIndex >> lastPathComponentsDictionaryOffset [

"Returns the last index into the path components list that is mapped in the
 index dictionary (as opposed to the B-tree)."

^ self size

]

{ #category : 'Accessing' }
IdentityIndex >> lastPathTerm [

"Returns the last path term in the path components list."

^ self at: self size

]

{ #category : 'Accessing' }
IdentityIndex >> nscRoot [

"Returns the value of the instance variable 'nscRoot'."

^nscRoot

]

{ #category : 'Updating' }
IdentityIndex >> nscRoot: newValue [

"Modify the value of the instance variable 'nscRoot'."

nscRoot := newValue

]

{ #category : 'Accessing' }
IdentityIndex >> nwayMergeProgress [

"Returns nil (identity indexes do not utilize n-way merge)."

^ nil

]

{ #category : 'Testing' }
IdentityIndex >> optimizingComparison [
  "Answer true if comparison operations can be optimized in Btree operations and elsewhere.
   Controlled by optimizedComparison index option."

  "optimizedComparison index option not supported by legacy indexes."

  ^ false

]

{ #category : 'Converting' }
IdentityIndex >> options [

  ^self asIndexSpecification options

]

{ #category : 'Accessing' }
IdentityIndex >> pathComponentsString [

"Returns a string of the path components."

^ self pathComponentsStringStartingAt: 1

]

{ #category : 'Accessing' }
IdentityIndex >> pathComponentsStringStartingAt: offset [

"Returns a string of the path components."

| str sz |

sz := self size.
str := String new.
offset to: sz do: [ :i |
    str addAll: (self at: i) name.
    i == sz
        ifFalse: [ str add: $.  ]
].
^ str

]

{ #category : 'Accessing' }
IdentityIndex >> pathSize [

"Returns the number of path terms."

^ self size

]

{ #category : 'Updating Indexes' }
IdentityIndex >> postIndexCreation: original [

"Iterates through all ID entry holders, updating the index dictionary."

" indicate that it is ok to update the dictionary "
indexDictionary ~~ nil
  ifTrue: [ | array |
    " get the Array of IndexDictionaryEntryHolder's "
    array := indexDictionary entryHolders.
    1 to: array size do: [ :i | | holder |
      holder := array at: i.
      holder update.
      holder size: 0.
    ].
    " nil out the Array of ID entry holders for garbage collection "
    indexDictionary entryHolders: nil.
    indexDictionary okToUpdate: true.
    "now ensure there's room for growth"
    indexDictionary rebuildTable: 0 for: self.
  ].

isComplete := true.

self indexManager autoCommit
  ifTrue: [ |systm |
    systm := System .
    systm commitTransaction
      ifFalse: [ nscRoot _errorCouldNotCommitDuringIndexCreation ].
    systm transactionMode == #manualBegin ifTrue:
        [systm beginTransaction]
  ]

]

{ #category : 'Updating Indexes' }
IdentityIndex >> preIndexCreation [

"Initializes the working set of collision buckets in the index dictionary."

isComplete := false.
progress := 0.

indexDictionary ~~ nil
  ifTrue: [ indexDictionary initializeWorkingSetInterval: self ].
^ nil

]

{ #category : 'Updating Indexes' }
IdentityIndex >> preIndexRemoval [

isComplete := false.
progress := 0.

^ nil

]

{ #category : 'Formatting' }
IdentityIndex >> printOn: aStream [

"Puts a displayable representation of the receiver on the given stream."

"Copy the implementation from Object so we don't inherit it from Collection."

aStream nextPutAll: self asString

]

{ #category : 'Accessing' }
IdentityIndex >> progress [

"Returns the value of the instance variable 'progress'."

^progress

]

{ #category : 'Updating' }
IdentityIndex >> progress: newValue [

"Modifies the value of the instance variable 'progress'."

progress := newValue

]

{ #category : 'Accessing' }
IdentityIndex >> sizeForNscBuilder [

"Returns the size to be used when an NscBuilder is collecting objects."

^ nscRoot size

]

{ #category : 'Accessing' }
IdentityIndex >> sortNodeClass [
  "Returns the class of SortNode to use."

  ^ self lastElementClass sortNodeClass

]

{ #category : 'Testing' }
IdentityIndex >> termsRequired [
  "Returns true if the receiver has any path term that has termsRequired set."

  1 to: self size do: [ :i |
    (self at: i) termsRequired
      ifTrue: [ ^ true ] ].
  ^ false

]

{ #category : 'Traversing' }
IdentityIndex >> traverse: anObject upTo: aPathTerm [
  "Traverse the sub-objects of the given object using the path terms of the
 receiver up to the given path term.  This method assumes that the index is not
 over a set-valued instance variable.  For that kind of index, use the
 traverse:upTo:startingAt:into: method."

  | nextObj sz i |
  nil == anObject
    ifTrue: [ ^ anObject ].
  nextObj := anObject.
  sz := self size.
  i := 1.
  [ (self at: i) ~~ aPathTerm and: [ i <= sz ] ]
    whileTrue: [
      nextObj := (self at: i) _getNextObjectForTraversal: nextObj.
      nil == nextObj
        ifTrue: [ ^ nextObj ].
      i := i + 1 ].
  ^ nextObj

]

{ #category : 'Traversing' }
IdentityIndex >> traverse: anObject upTo: aPathTerm startingAt: anOffset into: resultSet [
  "Traverse the sub-objects of anObject, using the path terms of the receiver up
 to the given path term.  Add each object at the end of the traversal to the
 result set.  Returns the result set."

  | nextObj nscOffset sz i |
  nextObj := anObject.
  sz := self size.
  i := anOffset.
  [ (self at: i) ~~ aPathTerm _and: [ i <= sz _and: [ nil ~~ nextObj ] ] ]
    whileTrue: [
      | thePathTerm |
      thePathTerm := self at: i.
      thePathTerm indicatesMultiValue
        ifTrue: [
          " if indexing over a collection-based instance variable "
          nscOffset := i + 1.
          thePathTerm
            nextObj: nextObj
            do: [ :obj |
              " for each object in the collection-based instance variable
				make recursive call "
              self
                traverse: obj
                upTo: aPathTerm
                startingAt: nscOffset
                into: resultSet ].
          ^ resultSet ]
        ifFalse: [ nextObj := (self at: i) _getNextObjectForTraversal: nextObj ].
      i := i + 1 ].
  (nil ~~ nextObj _and: [ resultSet ~~ nil ])
    ifTrue: [ resultSet add: nextObj ].
  ^ resultSet

]

{ #category : 'Traversing' }
IdentityIndex >> traverse: anObject upTo: aPathTerm startingAt: anOffset into: resultSet addingIndex: addingIndex [
  "Traverse the sub-objects of anObject, using the path terms of the receiver up
 to the given path term.  Add each object at the end of the traversal to the
 result set.  If addingIndex is true, add an index list entry for the
 receiver to any NSCs encountered along the path; if false, remove an index list entry for the receiver.  Returns the result set."

  | nextObj sz i thePathTerm nscOffset |
  nextObj := anObject.
  sz := self size.
  i := anOffset.
  [ (self at: i) ~~ aPathTerm _and: [ i <= sz _and: [ nil ~~ nextObj ] ] ]
    whileTrue: [
      thePathTerm := self at: i.
      thePathTerm indicatesMultiValue
        ifTrue: [
          " if indexing over a collection-based instance variable "
          thePathTerm
            _updateCollectionBasedIndexFor: self
            on: nextObj
            offset: i
            addingIndex: addingIndex.
          nscOffset := i + 1.
          thePathTerm
            nextObj: nextObj
            do: [ :obj |
              " for each object in the collection-based instance variable make recursive call "
              self
                traverse: obj
                upTo: aPathTerm
                startingAt: nscOffset
                into: resultSet
                addingIndex: addingIndex ].
          ^ resultSet ]
        ifFalse: [ nextObj := (self at: i) _getNextObjectForTraversal: nextObj ].
      i := i + 1 ].
  (nil ~~ nextObj _and: [ resultSet ~~ nil ])
    ifTrue: [ resultSet add: nextObj ].
  ^ resultSet

]

{ #category : 'Traversing' }
IdentityIndex >> traverseObject: anObject [

"Traverse the sub-objects of the given object using the path terms of the
 receiver.  This method assumes that the index is not over a set-valued
 instance variable.  For that kind of index, use the traverse:startingAt:into:
 method.  If a nil value is reached before the end of the path, returns the
 incompletePathTraversal object."

self isIndexOnNscElements
    ifTrue: [ ^ anObject ].

nil == anObject
    ifTrue: [ ^ #_incompletePathTraversal ].

^ self _traverseObject: anObject

]

{ #category : 'Traversing' }
IdentityIndex >> traverseObject: anObject
incompletesInto: incompleteArray
incomplete: incomplete [

"Traverse the sub-objects of the given object using the path terms of the
 receiver.  This method assumes that the index is not over a set-valued
 instance variable.  For that kind of index, use the traverse:startingAt:into:
 method.  If a nil value is reached before the end of the path, place the object
 into the appropriate Array in the incompleteArray of Arrays."

self isIndexOnNscElements
    ifTrue: [ ^ anObject ].

nil == anObject
    ifTrue: [ ^ incomplete ].

^ self _traverseObject: anObject
  incompletesInto: incompleteArray
  incomplete: incomplete

]

{ #category : 'Updating Indexes' }
IdentityIndex >> updatingByteObject: aByteObject startingAt: startPt withNewValue: newValue [

"The byte object is being updated.  For identity indexes, nothing is required
 to keep the index up to date."


]
