Extension { #name : 'IndexList' }

{ #category : 'Instance Creation' }
IndexList class >> new [

"Returns a new, initialized instance of the receiver."

^ self basicNew initialize

]

{ #category : 'Updating' }
IndexList >> _addIndex: anIndexObj withOffset: anInteger rootTerm: rootTerm [

"Add an entry (index object, path offset) to the end of the index list
 and update the root terms if necessary."

self addLast: anIndexObj.
self addLast: anInteger.

rootTerm ~~ nil
    ifTrue: [ rootTerms addLast: rootTerm ].

]

{ #category : 'Testing' }
IndexList >> _allIndexesAreRc [

"Returns true if all indexes are Rc."

self indexObjectsDo: [:indexObj | indexObj isReducedConflictIndex ifFalse: [ ^ false ] ].
^ true

]

{ #category : 'Testing' }
IndexList >> _anyIndexIsLegacy [

"Returns true if any one of the indexes is a legacy index."

self indexObjectsDo: [:indexObj | indexObj isLegacyIndex ifTrue: [ ^ true ] ].
^ false

]

{ #category : 'Updating' }
IndexList >> _calculateRootPathTerms [
  "Determines the list of root path terms for the receiver."

  rootTerms size: 0.
  self
    indexObjectsAndOffsetsDo: [ :indexObj :offset |
      indexObj size > 0
        ifTrue: [
          | pathTerm |
          pathTerm := indexObj size < offset
            ifTrue: [ indexObj lastPathTerm ]
            ifFalse: [ indexObj at: offset ].
          (rootTerms includesIdentical: pathTerm)
            ifFalse: [ rootTerms addLast: pathTerm ] ] ]

]

{ #category : 'Searching' }
IndexList >> _findOffsetFor: indexObj [

"Returns the integer entry number (position in the index list) at which
 indexObj is located in the receiver; or nil if it is not found."

"Performs a linear search."

1 to: self size by: 2 do: [ :i |
  indexObj == (self at: i)
        ifTrue: [ ^ i ]
].
^ nil

]

{ #category : 'Searching' }
IndexList >> _findOffsetFor: indexObj withOffset: anInteger [

"Returns the integer entry number (position in the index list) at which
 indexObj is located in the receiver; or nil if it is not found."

"Performs a linear search."

1 to: self size by: 2 do: [ :i |
  ( indexObj == (self at: i) and: [ (self at: i + 1) == anInteger ] )
        ifTrue: [ ^ i ]
].
^ nil

]

{ #category : 'Testing' }
IndexList >> _hasASetValuedTerm [
  "Answers true if the receiver contains a set-valued term."

  self
    indexObjectsDo: [ :iObj |
      iObj hasSetValuedTerm
        ifTrue: [ ^ true ] ].
  ^ false

]

{ #category : 'Testing' }
IndexList >> _hasExplicitIndex [

"Returns whether the receiver has an explicitly created index for it."

2 to: self size by: 2 do: [ :i |
  (self at: i) == 1
    ifTrue: [ ^ true ]
].
^ false

]

{ #category : 'Searching' }
IndexList >> _numberOfCommonPathTermsForPathArray: pathArray [

"Returns the number of path terms that already exist for indexes whose path
 components match pathArray (an Array of Strings)."

| firstUnique somePathTerms j continue count k |
firstUnique := 0.
count := 0.
k := 1.
1 to: self size by: 2 do: [ :i |
  somePathTerms := self at: i.
  j := self at: i + 1.
  j > 0
    ifTrue: [
      continue := true.

      [ k <= pathArray size and: [ j <= somePathTerms size and:
      [ (somePathTerms at: j) name = (pathArray at: k) ] ] ] whileTrue: [
        j > firstUnique
         ifTrue: [
           firstUnique := j.
           count := count + 1
         ].
        j := j + 1.
        k := k + 1.
      ]
    ]
].
^ count

]

{ #category : 'Updating Indexes' }
IndexList >> _putAllCommonPathTermsForPathArray: pathArray into: indexObj [

"Finds the common path terms that already exist for indexes whose path
 components match pathArray (an Array of Strings).  Add those path terms to
 the end of given indexObj."

| firstUnique somePathTerms j minSize sz |
firstUnique := 0.
sz := pathArray size.
1 to: self size by: 2 do: [ :i |
    somePathTerms := self at: i.
    j := self at: i + 1.
    " only consider index objects for which the NSC is the root "
    j == 1
        ifTrue: [
            minSize := somePathTerms size min: sz.

            [ j <= minSize and:
            [ (somePathTerms at: j) name = (pathArray at: j) ] ] whileTrue: [
                j > firstUnique
                   ifTrue: [
                       firstUnique := j.
                       indexObj addLast: (somePathTerms at: j)
                   ].
                j := j + 1
            ]
        ]
].

]

{ #category : 'Reduced Conflict Support' }
IndexList >> _selectiveAbort [

  "Do nothing"

]

{ #category : 'Statistics' }
IndexList >> _statisticsInto: dict [

"Put statistical information into the given dictionary."

| pages indexObj pathTerm |
pages := IdentitySet new.

pages add: self page.
pages add: rootTerms page.

1 to: self size by: 2 do: [ :i |
  indexObj := self at: i.
  indexObj _isIndexObject
    ifTrue: [
      indexObj _statisticsInto: dict.
      pages add: indexObj page.

      1 to: indexObj size do: [ :j |
        pathTerm := indexObj at: j.
        pages add: pathTerm page.
        pages add: pathTerm children page.
      ]
    ]
].
dict at: #IndexObjectPages put: pages

]

{ #category : 'Updating' }
IndexList >> addIndex: anIndexObj withOffset: anInteger nsc: anNsc [

"Add an entry (index object, path offset) to the end of the index list."

| rootTerm |
" do not allow duplicate entries "
1 to: self size by: 2 do: [ :i |
    " if entry already exists, just returns "
  anIndexObj == (self at: i)
    ifTrue: [
      (self at: i + 1) == anInteger
        ifTrue: [ ^ self ]
        ifFalse: [
          anNsc _error: #rtErrNscParticipatesInMultipleTerms "InternalError"
            args: { anIndexObj pathComponentsString }
        ]
    ]
].

anIndexObj size < anInteger
    ifTrue: [ rootTerm := anIndexObj lastPathTerm ]
    ifFalse: [ rootTerm := anIndexObj at: anInteger ].

(rootTerms includesIdentical: rootTerm)
    ifTrue: [ rootTerm := nil ].

" add the entry to the end of the index list "
self _addIndex: anIndexObj withOffset: anInteger rootTerm: rootTerm

]

{ #category : 'Updating' }
IndexList >> addTracker: tracker [

"Adds the tracking object to the receiver."

" do not allow duplicate entries "
1 to: self size by: 2 do: [ :i |
    " if entry already exists, just returns "
  tracker == (self at: i)
    ifTrue: [ ^ self ]
].

" add the entry to the end of the index list "
self addLast: tracker.
self addLast: 0.

]

{ #category : 'Updating Indexes' }
IndexList >> buildPathTermsFor: indexObj with: pathArray [
  "should only be used with legacy indexes"

  ^ self buildPathTermsFor: indexObj with: pathArray fromSpec: nil

]

{ #category : 'Updating Indexes' }
IndexList >> buildPathTermsFor: indexObj with: pathArray fromSpec: anIndexSpec [
  "Add path terms to the given indexObj, using the pathArray (Array of Strings).
 First find all common path terms (from existing indexes), then create any new
 path terms."

  | pathTerm startOffset pathTermClass optionalPathTerms |
  " set the common path terms "
  indexObj _doPutCommonPathTermsForPathArray: pathArray for: self.
  startOffset := indexObj size + 1.	" keep the starting point so we don't add duplicate entries "
  1 to: indexObj size do: [ :i | " add the index to shared path terms "
    (indexObj at: i) addLast: indexObj ].
  pathTermClass := PathTerm.
  optionalPathTerms := false.
  anIndexSpec ifNotNil: [ pathTermClass := anIndexSpec defaultPathTermClass.
      optionalPathTerms := anIndexSpec optionalPathTerms ].
  1 to: startOffset - 1 do: [ :i | | term |
    term := indexObj at: i.
    optionalPathTerms
      ifTrue: [ term termsRequired
          ifTrue: [ ImproperOperation signal:
                'Attempting to create an index with optionalPathTerms true, but an existing shared path term requires terms: '
                  , term name printString ] ]
      ifFalse: [ term termsRequired not
          ifTrue: [ ImproperOperation signal:
                'Attempting to create an index with required path terms, but a shared path term is optional: '
                  , term name printString ] ] ].
  startOffset to: pathArray size do: [ :i | | term |
    " create any new path terms that are needed "
    term := pathArray at: i.
    term = #'*'
      ifTrue: [ pathTerm := anIndexSpec defaultSetValuedPathTermClass new
          requirePathTerms: optionalPathTerms not;
          yourself ]
      ifFalse: [ (term includes: $|)
          ifTrue: [ pathTerm := anIndexSpec defaultEnumeratedPathTermClass new
              requirePathTerms: optionalPathTerms not;
              yourself ]
          ifFalse: [ (term size > 0 and: [ (term at: 1) == $# ])
              ifTrue: [ pathTerm := anIndexSpec defaultSelectorPathTermClass new
                  requirePathTerms: optionalPathTerms not;
                  yourself ]
              ifFalse: [ pathTerm := pathTermClass new ] ] ].
    pathTerm
      name: (pathArray at: i);
      offset: i;
      addLast: indexObj.
    indexObj addLast: pathTerm.
    pathTerm objectSecurityPolicy: GsIndexingObjectSecurityPolicy ].
  (1 max: startOffset - 1) to: indexObj size do: [ :i | " now calculate the children for each path term of the index object "
    (indexObj at: i) _determineChildren ].
  indexObj _setPathTermState.	" initialize path term state (whether to update B-trees, dictionary, etc) "
  self addIndex: indexObj withOffset: 1 nsc: indexObj nscRoot	" add an entry to the NSC's indexed paths list "

]

{ #category : 'Updating' }
IndexList >> hasTrackingObjects [

"Returns whether any tracking objects are contained in the receiver."

2 to: self size by: 2 do: [ :i |
  (self at: i) > 0
    ifFalse: [ ^ true ]
].
^ false

]

{ #category : 'Testing' }
IndexList >> includesIndex: indexObj [

"Returns whether the given index object is in the receiver."

1 to: self size by: 2 do: [ :i |
    indexObj == (self at: i)
        ifTrue: [ ^ true ]
].
^ false

]

{ #category : 'Enumerating' }
IndexList >> indexObjectsAndOffsetsDo: aBlock [

"Iterates over each (index, offset) pair in the index list, executing the
 two-argument block with the index object and offset as the arguments."

1 to: self size by: 2 do: [ :i |
  (self at: i) _isIndexObject
    ifTrue: [ aBlock value: (self at: i) value: (self at: i + 1) ]
]

]

{ #category : 'Enumerating' }
IndexList >> indexObjectsDo: aBlock [

"Iterates over each (index, offset) pair in the index list, executing the
 one-argument block with the index object as the argument."

1 to: self size by: 2 do: [ :i |
  (self at: i) _isIndexObject
    ifTrue: [ aBlock value: (self at: i) ]
]

]

{ #category : 'Initialization' }
IndexList >> initialize [

"Initializes the instance with default values."

rootTerms := { } .
isConstrained := false. "not used as ov v2.0"

]

{ #category : 'Accessing' }
IndexList >> objectSecurityPolicy: anObjectSecurityPolicy [

"Assigns the receiver, its tree of path terms, and the index objects to the
 given security policy."

| pathTerm |
anObjectSecurityPolicy == GsIndexingObjectSecurityPolicy
  ifFalse: [ self _error: #objectSecurityPolicyNotSharedDepListObjectSecurityPolicy "InternalError"].

super objectSecurityPolicy: anObjectSecurityPolicy.
securityPolicies ifNotNil:[ securityPolicies objectSecurityPolicy: anObjectSecurityPolicy ].
rootTerms objectSecurityPolicy: anObjectSecurityPolicy.
1 to: rootTerms size do: [ :i |
    pathTerm := rootTerms at: i.
    pathTerm offset == 1
        ifTrue: [ pathTerm objectSecurityPolicy: anObjectSecurityPolicy ]
].
self indexObjectsAndOffsetsDo: [ :indexObj :offset |
    offset == 1
        ifTrue: [ indexObj objectSecurityPolicy: anObjectSecurityPolicy ]
]

]

{ #category : 'Removing' }
IndexList >> removeAllIndexesFor: anNsc [

"Removes all indexes from the receiver (it may contain tracking objects)."

| indexes |

indexes := { } .
1 to: self size by: 2 do: [ :i |
  (self at: i + 1) > 0
    ifTrue: [ indexes add: (self at: i); add: (self at: i + 1) ]
].
1 to: indexes size by: 2 do: [ :i |
  self removeIndex: (indexes at: i) withOffset: (indexes at: i + 1) for: anNsc
]

]

{ #category : 'Removing' }
IndexList >> removeIndex: indexObj withOffset: anInteger for: anNsc [

"Removes the entry (an entry has 2 slots) containing the given index.  If there
 are no remaining entries, then set the NSC's index list to nil.  Remaining
 entries are moved forward in the list."

| i sz |
sz := self size.

i := self _findOffsetFor: indexObj withOffset: anInteger.
i == nil
    ifTrue: [ ^ self ].

" if only one entry, clear the index list "
sz == 2
    ifTrue: [ ^ anNsc _clearIndexList ].

self removeFrom: i to: i + 1 .

self _calculateRootPathTerms.

]

{ #category : 'Removing' }
IndexList >> removeIndexesInPathTerm: pathTerm for: anNsc [

"Removes an entry in the receiver for each index that uses the given path term."

" for each index that utilizes the path term "
1 to: pathTerm size do: [ :i |
    " remove the entry in the index list for the NSC "
    self removeIndex: (pathTerm at: i)
        withOffset: pathTerm offset + 1
        for: anNsc
]

]

{ #category : 'Removing' }
IndexList >> removeTracker: tracker for: anNsc [

"Removes the tracking object from the receiver."

| i |

i := self _findOffsetFor: tracker.
i == nil
  ifTrue: [ ^ self ].

" if only one entry, clear the index list "
self size == 2
  ifTrue: [ ^ anNsc _clearIndexList ].

self removeFrom: i to: i + 1 .

]

{ #category : 'Accessing' }
IndexList >> rootTerms [

"Returns the value of the instance variable 'rootTerms'."

^rootTerms

]

{ #category : 'Updating' }
IndexList >> rootTerms: newValue [

"Modifies the value of the instance variable 'rootTerms'."

rootTerms := newValue

]
