Extension { #name : 'IndexManager' }

{ #category : 'Repository Conversion' }
IndexManager class >> _loadBitMap: bm [
  | files |
  bm removeAll.
  files := GsFile contentsOfDirectory: self getNscIndexDir onClient: false.
  files := files
    select: [ :e | (e findPattern: (Array with: '.bm') startingAt: 1) ~~ 0 ].
  1 to: files size do: [ :n |
    | fn |
    fn := files at: n.
    bm readFromFile: fn ].
  ^ bm size

]

{ #category : 'Repository Conversion' }
IndexManager class >> _loadBitmap: bm fromFile: fn [
  ^ SortedCollection _loadBitmap: bm fromFile: fn

]

{ #category : 'Accessing' }
IndexManager class >> autoCommit [

^self current autoCommit

]

{ #category : 'Updating' }
IndexManager class >> autoCommit: aBool [
"Set autoCommit status; if true, this enabled commits to be performed without
specific commands, including due to TransactionBacklog"

self current autoCommit: aBool

]

{ #category : 'Repository Conversion' }
IndexManager class >> buildFileNameForGem: anInt [
  | result |
  result := String withAll: 'failedCharCollNscs_'.
  result
    addAll: anInt asString;
    addAll: '.bm'.
  ^ result

]

{ #category : 'Repository Conversion' }
IndexManager class >> createConversionResultFileForTotalGems: totalGems [
  "Called by the master gem to consolidate all lists of Indexes which failed
   conversion into a single result file.
   This method requires $upgradeLogDir to be set to a directory that contains
   conversion result files for the given gems."

  | uld bm outFile totalObjs |
  bm := GsBitmap new.
  uld := SortedCollection getUpgradeLogDir.
  1 to: totalGems do: [ :n |
    | fn |
    fn := uld copy.
    fn addAll: (self buildFileNameForGem: n).
    (GsFile existsOnServer: fn)
      ifTrue: [ bm readFromFile: fn ]
  ].
  totalObjs := bm size.
  totalObjs == 0
    ifTrue: [ ^ true ].
  outFile := uld copy.
  outFile addAll: 'AllFailedCharCollNscs.bm'.
  (GsFile existsOnServer: outFile)
    ifTrue: [ GsFile removeServerFile: outFile ].
  bm writeToFile: outFile.
  GsFile
    gciLogServer:
      'A total of ' , totalObjs asString
        , ' CharacterCollection indexes failed to convert'.
  ^ true

]

{ #category : 'Accessing' }
IndexManager class >> current [

  ^ Current ifNil:[
     Current := IndexManager _basicNew initialize .
     Current .
  ].

]

{ #category : 'Updating' }
IndexManager class >> dirtyObjectCommitThreshold: anInteger [

self current dirtyObjectCommitThreshold: anInteger

]

{ #category : 'Repository Conversion' }
IndexManager class >> findAllCharacterCollectionIndexesForUser: aUserId [
  "Find all indexes for which the last element class is a kind of CharacterCollection,
   and write these indexRoots to bitmap file under  $upgradeLogDir"
  | bm dirPath pid |
  bm := GsBitmap new.
  IndexManager current getAllIndexes
    do: [ :index |
      index isIdentityIndex
        ifFalse: [
          | lec |
          lec := index lastElementClass.
          (lec == CharacterCollection or: [ lec isBehavior and: [ lec inheritsFrom: CharacterCollection ] ])
            ifTrue: [ bm add: (index nscRoot) ] ] ].
  (bm size) = 0
    ifTrue: [ ^ self ].
  dirPath := self getNscIndexDir.
  (GsFile exists: dirPath)
    ifTrue: [ GsFile removeServerDirectory: dirPath ].
  GsFile createServerDirectory: dirPath.
  pid := (System gemVersionAt: 'processId') printString.
  bm writeToFile: ( dirPath , '/NscsWithCharacterCollectionIndexes-' , pid , '.bm')

]

{ #category : 'Repository Conversion' }
IndexManager class >> getNscIndexDir [
  ^ SortedCollection getUpgradeLogDir , '/ccnscbm'

]

{ #category : 'Accessing' }
IndexManager class >> new [
   "use
      IndexManager current
    to get the current instance"

   self shouldNotImplement: #new

]

{ #category : 'Updating' }
IndexManager class >> percentTempObjSpaceCommitThreshold: anInteger [

self current percentTempObjSpaceCommitThreshold: anInteger

]

{ #category : 'Repository Conversion' }
IndexManager class >> rebuildAllCharCollIndexes: anArray [
  "Rebuild indexes for an Array of nscs with CharacterColletion indexes
     by recording the indexes aassociated with the nsc, removing all indexes
     on the nsc, and then rebuilding all indexes.

   Caller must handle and catch AlmostOutOfMemory errors and handle commits
 and aborts."

  | sys commitBlock oldAutoCommit |
  sys := System.
  commitBlock := [
  sys commitTransaction
    ifFalse: [
      sys abortTransaction.
      TransactionError signal: 'Unexpected commit failure in index rebuild' ] ].
  oldAutoCommit := self sessionAutoCommit.
  [
  self sessionAutoCommit: true.
  1 to: anArray size do: [ :n |
    | nsc indexSpec |
    nsc := anArray at: n.
    indexSpec := nsc indexSpec.
    GsFile gciLogServer: 'Rebuilding indexes for nsc: ', nsc asOop printString.
    GsFile gciLogServer: indexSpec printString.
    nsc removeAllIndexes.
    indexSpec createIndexesOn: nsc.
    indexSpec := nsc indexSpec.
    GsFile gciLogServer: 'Rebuilt indexes for nsc: ', nsc asOop printString.
    GsFile gciLogServer: indexSpec printString ] ]
    ensure: [
      self sessionAutoCommit: oldAutoCommit.
      commitBlock value ]

]

{ #category : 'Repository Conversion' }
IndexManager class >> rebuildCharacterCollectionIndexesFromFilesForGem: gemNum of: totalGems [
  "Rebuild indexes on CharacterCollections.  Must previously have run
   findAllCharacterCollectionIndexesForUser:.
   This method requires $upgradeLogDir to be set to a directory that contains a
   bitmap of all indexes requiring rebuild."

  | bm errBm ar myShare failed ok unique |
  bm := GsBitmap new.
  errBm := GsBitmap new.
  ar := SortedCollection
    convertInstancesFromFilesForGem: gemNum
    of: totalGems
    bm: bm
    errBm: errBm
    bmDirectoryPath: self getNscIndexDir
    conversion: [ :array :ignored :_errBm | self rebuildAllCharCollIndexes: array ].
  myShare := ar at: 1.
  unique := ar at: 2.
  SortedCollection
    writeFailedConversionObjsToFileForGem: gemNum
    bmForErrors: errBm.	"Now write out the ones that failed conversion to a file, if any."
  failed := errBm size.
  ok := myShare - failed.
  System incrementGlobalSessionStatAt: 0 by: ok.
  System incrementGlobalSessionStatAt: 1 by: failed.
  System incrementGlobalSessionStatAt: 2 by: 1.
  ^ String new
    addAll: 'CharacterCollection index rebuilding done. Of ';
    addAll: myShare asString;
    addAll: ' total: ';
    addAll: failed asString;
    addAll: ' failed, ';
    addAll: ok asString;
    addAll: ' successful (';
    addAll: unique asString;
    addAll: ' unique).';
    yourself

]

{ #category : 'Index Maintenance' }
IndexManager class >> removeAllIncompleteIndexes [

^ self current removeAllIncompleteIndexes.

]

{ #category : 'Index Maintenance' }
IndexManager class >> removeAllIndexes [

^ self current removeAllIndexes.

]

{ #category : 'Accessing' }
IndexManager class >> sessionAutoCommit [

^self current sessionAutoCommit

]

{ #category : 'Updating' }
IndexManager class >> sessionAutoCommit: aBool [

self current sessionAutoCommit: aBool

]

{ #category : 'Repository Conversion' }
IndexManager class >> writeTotalsFiles [
  SortedCollection writeTotalsFile: 'ccidx_ok.txt' counterIndex: 0.
  SortedCollection writeTotalsFile: 'ccidx_failed.txt' counterIndex: 1.
  SortedCollection writeTotalsFile: 'ccidx_done.txt' counterIndex: 2.
  ^ true

]

{ #category : 'Accessing' }
IndexManager >> _addIndex: anIndex [
  "  Add the given index into the collection of all indexes."

  anIndex objectSecurityPolicy: GsIndexingObjectSecurityPolicy.
  self getAllIndexes add: anIndex

]

{ #category : 'Updating Indexes - Private' }
IndexManager >> _addMappingsFor: anNSC on: indexObj [
  "Adds the mappings in the index dictionary, dependency lists, and so on for all
 elements in the receiver."

  | pathTerm setOffset hasSet |
  " since this index may use path terms from other indexes,
find the first one that is not shared by another index (only
need to add mappings from that point onward) "
  setOffset := indexObj _findSetValuedPathTermOffset.
  pathTerm := indexObj _findFirstUnsharedPathTerm.
  pathTerm
    ifNil: [
      " index is subsumed by an existing index "
      pathTerm := indexObj lastPathTerm.
      ^ pathTerm _doAddSubsumedIndex: indexObj nsc: anNSC ].
  setOffset == 0
    ifTrue: [ hasSet := false ]
    ifFalse: [
      "Determine if there is a set-valued instance variable before the unshared path
 term."
      hasSet := setOffset < pathTerm offset ].
  ^ pathTerm _doAddMappingsFor: anNSC on: indexObj indexManager: self hasSet: hasSet

]

{ #category : 'Testing' }
IndexManager >> _autoCommitPolicy [

^ (System __sessionStateAt: 11)

]

{ #category : 'Index Updating - Private' }
IndexManager >> _buildIndex: index for: anNsc fromSpec: anIndexSpec [
  [ | pathArray original |
    pathArray := anIndexSpec path asArrayOfPathTerms.
    index _preIndexCreationBuildIndexDictionaryFor: pathArray for: anNsc.
    index nscRoot: anNsc.
    self _addIndex: index.
    anNsc _getIndexList
      buildPathTermsFor: index
      with: pathArray
      fromSpec: anIndexSpec.
    anNsc _putInWriteSet.
    original := index preIndexCreation.
    index isIndexOnRootNsc
      ifTrue: [ " optimize creation of indexes on elements of the NSC itself "
        index
          addDirectMappingsFor: anNsc _asIdentityBag
          indexList: anNsc _indexedPaths ]
      ifFalse: [ self _addMappingsFor: anNsc on: index ].
    index postIndexCreation: original 
  ] onSynchronous: Error do: [ :ex | 
     self _handleError: ex msg:' during IndexManager>>_buildIndex:for:fromSpec:' .
     ex pass .
  ]
]

{ #category : 'Index Updating - Private' }
IndexManager >> _handleError: ex msg: aString [ 
  ex stackReport ifNotNil:[:stk | 
    GsFile gciLogServer:'---( in IndexManager >> _handleError:msg: 
aString = ', aString,'
disabling commits
', stk , ' )---'. 
  ].
  (ex isKindOf: ImproperOperation) ifTrue:[
    System disableCommitsUntilAbortWithReason: ex class name , aString
  ] ifFalse:[
    System disableCommitsWithReason: ex class name , aString
  ]
]

{ #category : 'Index Updating - Private' }
IndexManager >> _checkExistingIndexesFor: anNsc against: index withSpec: anIndexSpec ifPresent: presentBlock [
  "evaluate presentBlock if an existing index matches the specified index"

  | pathArray indexes pathString indexOptions |
  indexOptions := index options.
  indexOptions legacyIndex
    ifTrue: [
      "legacy indexes are not supported on RcIndentitySet or RcLowMaintenanceIdentityBag (bug47179)"
      anNsc _validateLegacyRcIndexSupport ].
  (anNsc _indexedPaths )
    ifNotNil: [ :ilist |
      ilist indexObjectsDo: [:idxObj |
        "All indexes on anNsc must all be either legacy or btree plus ... no mixing"
        | idxObjOptions  |
        idxObjOptions := idxObj options.
        ((idxObjOptions legacyIndex == indexOptions legacyIndex)
          and: [idxObjOptions btreePlusIndex == indexOptions btreePlusIndex])
          ifFalse: [
            self error: 'A ',
              (idxObjOptions legacyIndex ifTrue: ['legacy'] ifFalse: ['btree plus']),
              ' index exists for this collection and it conflicts with the requested ',
              (idxObjOptions btreePlusIndex ifTrue: ['legacy'] ifFalse: ['btree plus']),
              ' index.'] ] ].
  pathString := anIndexSpec path.
  pathArray := pathString asArrayOfPathTerms.
  indexes := anNsc _findIndexesWithPath: pathArray.
  indexes
    do: [ :idxObj |
      idxObj indexType == index indexType
        ifTrue: [
          "index already exists on the given path for this type"
          (idxObj _checkSameIndexOptionsAs: anIndexSpec)
            ifFalse: [ "Bug44038: If index options are different, then signal an error"
              ^ self error: 'An ' , index indexType asString printString
                    , ' index already exists on the path '
                    , pathString printString , ' for this collection with '
                    , (idxObj termsRequired
                        ifTrue: [ 'REQUIRED' ]
                        ifFalse: [ 'OPTIONAL' ])
                    , ' path terms, which conflicts with the requested '
                    , (index termsRequired
                        ifTrue: [ 'REQUIRED' ]
                        ifFalse: [ 'OPTIONAL' ]) , ' path terms.' ].
          index indexType == #'identity'
            ifTrue: [ "no chance for conflicting index types for identity index"
              ^ presentBlock value ].
          (index _checkSameLastElementClassAs: idxObj)
            ifTrue: [ "If path and lastElementClass are the same, we'll let it slide"
              ^ presentBlock value ].	"Bug 43035: signal error if attempt was made to create a different
           index with the same path"
          ^ self error: 'An ' , index indexType asString printString
                , ' index already exists on the path ' , pathString printString
                , ' for this collection with lastElementClass of '
                , idxObj lastElementClassDescription printString
                , ' which conflicts with the requested lastElementClass of '
                , index lastElementClassDescription printString , '.' ] ]

]

{ #category : 'Testing' }
IndexManager >> _clearAutoCommitPolicy [

System __sessionStateAt: 11 put: nil

]

{ #category : 'Private' }
IndexManager >> _doCommit [
  | systm |
  systm := System.
  systm _commitPrintingDiagnostics
    ifFalse: [ self _errorCouldNotCommitDuringIndexCreation ].
  systm transactionMode == #'manualBegin'
    ifTrue: [ systm beginTransaction ]

]

{ #category : 'Private' }
IndexManager >> _equalityIndex [

  ^ #equality

]

{ #category : 'Private' }
IndexManager >> _identityIndex [

  ^ #identity

]

{ #category : 'Testing' }
IndexManager >> _initializeAutoCommitPolicy [

System __sessionStateAt: 11 put: (IndexManagerAutoCommitPolicy on: self)

]

{ #category : 'Private' }
IndexManager >> _nextWordFrom: aReadStream [
  "Assume that the aReadStream's collection is a kind of String.  Returns the
 next word in the string or nil if there is no next word."

  | result |
  result := String new.
  aReadStream skipSeparators.
  [ aReadStream atEnd or: [ aReadStream peek isSeparator ] ]
    whileFalse: [ result add: aReadStream next ].
  result size == 0
    ifTrue: [ ^ nil ]
    ifFalse: [ ^ result ]

]

{ #category : 'Index Maintenance' }
IndexManager >> _oldRemoveAllIndexes [
  "The _oldRemoveAllIndexes algorithm is actually pretty spiffy as long as there is
   a single index per nsc and each object participates in only one index. In fact
   this algorithm will (slightly) out-perform the new removeAllIndexes algorithm.
   When an object has more than one entry in the dependency list, this algorithm
   quickly blows up as the shared dependency list is recalculated for each element
   "

  | sys |
  sys := System.
  ^ [
  | roots sz |
  sys
    setIndexProgressCountTo: UnorderedCollection statValueForRemovingAllIndexes.
  roots := self getAllNSCRoots.
  sz := roots size.
  sys setProgressCountTo: sz.
  1 to: sz do: [ :n |
    | nsc result |
    nsc := roots at: n.
    result := self removeAllIndexesOn: nsc.
    nsc == result
      ifFalse: [
        self
          _error: #'rtErrRemoveAllIndexesFailed'
          args:
            {nsc.
            result} ].
    sys decrementProgressCountBy: 1 ] ]
    ensure: [
      sys
        setIndexProgressCountTo: 0;
        setProgressCountTo: 0 ]

]

{ #category : 'Accessing' }
IndexManager >> _removeIndex: anIndex [

  "Remove the given index from the collection of all indexes.
   Make sure that IVs are niled in anIndex, since it will
   survive (in rc redo log) until abort or commit."

  self getAllIndexes remove: anIndex otherwise: nil .
  anIndex _clear.

]

{ #category : 'Index Updating - Private' }
IndexManager >> _removeIndex: indexType for: anNSC on: aPathString [
  | iList indexes index pathArray |
  anNSC _checkIndexPathExpression: aPathString.
  iList := anNSC _indexedPaths.
  iList ifNil: [ self _error: #'rtErrNoIndexForPath' args: {aPathString} ].
  pathArray := aPathString asArrayOfPathTerms.	" check if an index exists for the path string "
  indexes := anNSC _findIndexesWithPath: pathArray.
  indexes isEmpty
    ifFalse: [
      " make sure it is an identity index or equality index"
      indexes
        do: [ :iObj |
          indexType == self _identityIndex
            ifTrue: [
              iObj isIdentityIndex
                ifTrue: [ index := iObj ] ]
            ifFalse: [
              iObj isRangeEqualityIndex
                ifTrue: [ index := iObj ] ] ] ].
  index
    ifNil: [
      " no index was found with the given path "
      ^ self _error: #'rtErrNoIndexForPath' args: {aPathString} ].
  [
  IndexManager current
    executeStartingIndexMaintenance: [
      index preIndexRemoval.
      anNSC _removeIndex: index.
      self _removeIndex: index ] ]
    onSynchronous: Error 
    do:[ :ex | 
      self _handleError: ex msg: ' during IndexManager>>_removeIndex:for:on:' .
      ex pass .
    ]
]

{ #category : 'Querying' }
IndexManager >> _reportDataForIndexes [
  "private"

  | list |
  list := OrderedCollection new.
  self getAllIndexes do: [ :each | | firstStr string |
      string := each class name copyFrom: 1 to: each class name size - 5.
      (5 < string size and: [ (string copyFrom: 1 to: 5) = 'Range' ])
        ifTrue: [ string := string copyFrom: 6 to: string size ].
      firstStr := string.
      string := each
        inject: ''
        into: [ :str :pathTerm | str , '.' , pathTerm name ].
      string := string copyFrom: 2 to: string size.
      list add: {firstStr.
          string.
          (each isIdentityIndex
            ifTrue: [ '' ]
            ifFalse: [ each lastElementClass name ]).
          (each nscRoot).
          (each isComplete)} ].
  list := list collect: [ :each | | nsc |
      nsc := each at: 4.
      Association new
        key: (each at: 1) , '-' , (each at: 2) , ((each at: 3) notEmpty
                ifTrue: [ '-' , (each at: 3) ]
                ifFalse: [ '' ])
        value:
          nsc size printString , ' (' , nsc asOop printString , ')' , ((each at: 5)
                ifTrue: [ '' ]
                ifFalse: [ ' - INCOMPLETE' ]) ].
  ^ list

]

{ #category : 'Querying' }
IndexManager >> _reportDataForMethods [
		"private"

	| indexCreationSelectors classOrganizer smallList bigList |
	classOrganizer := ClassOrganizer new.
	smallList := OrderedCollection new.
	indexCreationSelectors := #(
		#'createEqualityIndexOn:'
		#'createEqualityIndexOn:withLastElementClass:'
		#'createIdentityIndexOn:'
		#'createRcEqualityIndexOn:'
		#'createRcEqualityIndexOn:withLastElementClass:'
	).
	indexCreationSelectors do: [:each | | arrays |
		arrays := classOrganizer sendersOf: each.
		1 to: arrays first size do: [:i |
			smallList add: (Array
				with: (arrays first at: i)
				with: each
				with: (arrays last at: i)).
		].
	].
	smallList := smallList reject: [:each | each first inClass == UnorderedCollection].
	smallList := smallList reject: [:each | each first inClass name == #'IndexedQueryExtensionsTestCase'].
	bigList := OrderedCollection new.
	smallList do: [:each |
		| selector pieces firstPiece lastPiece source index |
		selector := each at: 2.
		pieces := selector subStrings: $:.
		pieces := pieces copyFrom: 1 to: pieces size - 1.
		firstPiece := pieces first.
		lastPiece := pieces last.
		source := each first sourceString.
		index := (each at: 3) - 1.
		[
			0 < (index := source indexOfSubCollection: firstPiece startingAt: index + 1).
		] whileTrue: [
			| readStream array word |
			readStream := ReadStreamPortable on: (source copyFrom: index to: source size).
			array := { ((word := self _nextWordFrom: readStream) copyFrom: 7 to: word size - 8) .
						nil . nil . nil . nil } .
			word := self _nextWordFrom: readStream.
			word := word copyFrom: 2 to: word size - 1.
			[''';.' includes: word last] whileTrue: [word := word copyFrom: 1 to: word size - 1].
			array at: 2 put: word.
			2 <= pieces size ifTrue: [
				self _nextWordFrom: readStream.
                                word := self _nextWordFrom: readStream.
				word := word reject: [:char | ''';.' includes: char].
			] ifFalse: [
				word := ''.
			].
			array
				at: 3 put: word;
				at: 4 put: each first inClass name;
				at: 5 put: each first selector  .
			bigList add: array.
		].
	].
	bigList := bigList collect: [:each |
		Association new
			key: ((each at: 1) , '-' , (each at: 2) ,
					((each at: 3) notEmpty ifTrue: ['-' , (each at: 3)] ifFalse: ['']))
			value: ((each at: 4) , '>>#' , (each at: 5)).
	].
	^ bigList

]

{ #category : 'Accessing' }
IndexManager >> autoCommit [
  "Use session-specific value if sessionAutoCommit has been set (bug40602)"
  | val |
  self sessionAutoCommit ifNotNil:[:v | ^ v ].
  val := autoCommit ifNil:[ self autoCommit: false . val := false ].
  self sessionAutoCommit: val .
  ^ autoCommit

]

{ #category : 'Updating' }
IndexManager >> autoCommit: aBool [
"Set autoCommit status; if true, this enabled commits to be performed without
specific commands, including due to TransactionBacklog"

autoCommit := aBool

]

{ #category : 'Accessing' }
IndexManager >> btreeBasicLeafNodeClass [

  self useKeyEncryption ifTrue: [ ^ BtreeBasicLeafNode ].

  ^ BtreeLeafNode

]

{ #category : 'Accessing' }
IndexManager >> btreeLeafNodeClass [

  ^ BtreeLeafNode

]

{ #category : 'Index Maintenance' }
IndexManager >> commitIndexMaintenance: indexObjOrNil at: progressCount [
  self shouldCommit
    ifTrue: [
      indexObjOrNil ifNotNil: [ indexObjOrNil progress: progressCount ].
      self _doCommit ]

]

{ #category : 'Index Creation' }
IndexManager >> createEqualityIndexFor: anNSC on: aPathString withLastElementClass: aClass [
  ^ (EqualityIndexSpecification path: aPathString lastElementClass: aClass)
    createIndexOn: anNSC

]

{ #category : 'Index Creation' }
IndexManager >> createIdentityIndexFor: anNSC on: aPathString [
  ^ (IdentityIndexSpecification path: aPathString) createIndexOn: anNSC

]

{ #category : 'Index Creation' }
IndexManager >> createIndexFor: anNsc fromSpec: anIndexSpec [
  | index |
  anIndexSpec _validateSpecificationOn: anNsc.
  " see if incomplete indexes "
  anNsc _hasIncompleteIndexes
    ifTrue: [ ^ self _error: #'rtErrCollectionWithIncompleteIndex' ].
  index := anIndexSpec _createIndex.
  self
    _checkExistingIndexesFor: anNsc
    against: index
    withSpec: anIndexSpec
    ifPresent: [ ^ self ].
  self
    executeStartingIndexMaintenance: [ self _buildIndex: index for: anNsc fromSpec: anIndexSpec ]

]

{ #category : 'Index Creation' }
IndexManager >> createRcEqualityIndexFor: anNSC on: aPathString withLastElementClass: aClass [
  "Replaces UnorderedCollection>>_createEqualityIndexOn:withLastElementClass:commitInterval:, modulo the error handler"

  ^ (RcEqualityIndexSpecification path: aPathString lastElementClass: aClass)
    createIndexOn: anNSC

]

{ #category : 'Accessing' }
IndexManager >> dirtyObjectCommitThreshold [

^ dirtyObjectCommitThreshold

]

{ #category : 'Updating' }
IndexManager >> dirtyObjectCommitThreshold: anInteger [

dirtyObjectCommitThreshold := anInteger

]

{ #category : 'Private' }
IndexManager >> execStartingIndexMaintenance: aBlock [
  "Behavior of index maintenance with respect to transaction state:

When in manual mode:
    When IndexManager current autoCommit is true:
        When originally outside of a transaction:
            Begin a transaction.
            Commit the final transaction of index creation.
            End outside of a transaction.
        When originally inside of a transaction:
            Commit the final transaction of index creation.
            End outside of a transaction.
When in auto mode:
    When IndexManager current autoCommit is true:
        Commit the final transaction of index creation.
"

  | inAutoCommit autoCommitPolicy systm |
  systm := System.
  inAutoCommit := self autoCommit  .
  inAutoCommit ifTrue: [
      systm transactionMode == #'manualBegin' ifTrue: [
          systm inTransaction ifFalse: [
              systm needsCommit ifTrue: [ self _error: #'rtErrAbortWouldLoseData' ].
              systm beginTransaction 
          ] 
      ].
      self _initializeAutoCommitPolicy 
  ].
  (autoCommitPolicy := self _autoCommitPolicy)
    ifNil: [ aBlock value ]
    ifNotNil: [
      [ autoCommitPolicy evaluate: aBlock for: self ]
        ensure: [
          inAutoCommit
            ifTrue: [ self _clearAutoCommitPolicy ] ] ].
  inAutoCommit
    ifTrue: [
      systm inTransaction
        ifTrue: [
          systm _commitPrintingDiagnostics
            ifFalse: [ self _errorCouldNotCommitDuringIndexCreation ] ] ]

]

{ #category : 'Index Maintenance' }
IndexManager >> executeStartingIndexMaintenance: aBlock [
"Disable STN_GEM_TIMEOUT, then perform index maintenance.

Behavior of index maintenance with respect to transaction state:

When in manual mode:
    When IndexManager current autoCommit is true:
        When originally outside of a transaction:
            Begin a transaction.
            Commit the final transaction of index creation.
            End outside of a transaction.
        When originally inside of a transaction:
            Commit the final transaction of index creation.
            End outside of a transaction.
When in auto mode:
    When IndexManager current autoCommit is true:
        Commit the final transaction of index creation."
| sys |
sys := System .
^ [
    sys disableStoneGemTimeout .
    self execStartingIndexMaintenance: aBlock
  ] ensure:[ sys enableStoneGemTimeout ] .

]

{ #category : 'Accessing' }
IndexManager >> getAllIndexes [
  "Return a collection of all indexes in the system."

  allIndexes == nil
    ifTrue: [
      allIndexes := RcIdentitySet new.
      allIndexes objectSecurityPolicy: GsIndexingObjectSecurityPolicy ].
  ^ allIndexes

]

{ #category : 'Accessing' }
IndexManager >> getAllNSCRoots [

  "Return a collection of all nsc's with indexes in the system."

  | set |
  set := IdentitySet new.
  self getAllIndexes do: [:index |
    set add: index nscRoot
  ].
  ^set

]

{ #category : 'Initializing' }
IndexManager >> initialize [

  "When creating initial instance, make sure all IVs are set, since we don't
   want to dirty the object during normal operation."

  self autoCommit: false .
  self getAllIndexes.
  self useKeyEncryption: true.
  self dirtyObjectCommitThreshold: SmallInteger maximumValue . "disabled by default"
  "was 75% ,  Gs64 v3.0 change to 60 for new AlmostOutOfMemory handling. "
  self percentTempObjSpaceCommitThreshold: 60.
  self objectSecurityPolicy: GsIndexingObjectSecurityPolicy.

]

{ #category : 'Querying' }
IndexManager >> nscsWithBadIndexes [
  "IndexManager current nscsWithBadIndexes"

  ^ self getAllNSCRoots
    reject: [ :each |
      | audit |
      audit := each auditIndexes.
      audit = 'Indexes are OK'
        or: [
          audit
            =
              'Indexes are OK and the receiver participates in one or more indexes with collection-valued path terms' ] ]

]

{ #category : 'Querying' }
IndexManager >> nscsWithIncompleteIndexes [
        "IndexManager current nscsWithIncompleteIndexes"

	^ self getAllNSCRoots select: [:each | each _hasIncompleteIndexes].

]

{ #category : 'Updating' }
IndexManager >> objectSecurityPolicy: anObjectSecurityPolicy [

"Assigns the receiver and the allIndexes collection to anObjectSecurityPolicy."

<primitive: 2001>
| prot |
prot := System _protectedMode .
[
  anObjectSecurityPolicy == GsIndexingObjectSecurityPolicy
    ifFalse: [ self _error: #objectSecurityPolicyNotSharedDepListObjectSecurityPolicy ].

  self _objectSecurityPolicy: anObjectSecurityPolicy.
  self getAllIndexes objectSecurityPolicy: anObjectSecurityPolicy.
] ensure:[
  prot _leaveProtectedMode
]

]

{ #category : 'Accessing' }
IndexManager >> percentTempObjSpaceCommitThreshold [

^ percentTempObjSpaceCommitThreshold

]

{ #category : 'Updating' }
IndexManager >> percentTempObjSpaceCommitThreshold: anInteger [

  anInteger _isSmallInteger ifFalse:[ anInteger _validateClass: SmallInteger ].
  (anInteger < 10 or:[ anInteger > 100]) ifTrue:[
     OutOfRange new name:'commit threshold' min: 10 max: 100 actual: anInteger;
       details: 'invalid commit threshold' ;  signal
  ].
  percentTempObjSpaceCommitThreshold := anInteger

]

{ #category : 'Accessing' }
IndexManager >> rcBtreeBasicLeafNodeClass [

  self useKeyEncryption ifTrue: [ ^ RcBtreeBasicLeafNode ].

  ^ RcBtreeLeafNode

]

{ #category : 'Accessing' }
IndexManager >> rcBtreeLeafNodeClass [

  ^ RcBtreeLeafNode

]

{ #category : 'Index Updating' }
IndexManager >> removeAllCompleteIndexesOn: anNSC [
  "Remove all complete indexes for anNSC. If all of the anNSC's indexes can be removed,
   this method returns the receiver.
   If an error occurs during index removal, it may not be possible to
   commit the current transaction later.
  "
  | iList rootIndexes hasNonRootIndex |
  (iList := anNSC _indexedPaths) ifNil: [ ^ anNSC ].
  hasNonRootIndex := false.
  rootIndexes := { } .
  " scan each entry to see if a non-root index is present "
  1 to: iList size by: 2 do: [ :i |
    " if it is a root index ... "
    (((iList at: i + 1) == 1) and: [ (iList at: i) isComplete])
        " add it to a list "
        ifTrue: [ rootIndexes add: (iList at: i) ]
        " otherwise flag that we're really not removing ALL indexes "
        ifFalse: [ hasNonRootIndex := true ]
  ].
  ^ [ | result |
      IndexManager current executeStartingIndexMaintenance: [
        result := anNSC _removeAllRootIndexes: rootIndexes hasNonRootIndex: hasNonRootIndex.
        result == anNSC ifTrue: [
          rootIndexes do: [:each | self _removeIndex: each.  ].
        ].
      ].
      result
    ] onSynchronous: Error do:[ :ex |
      self _handleError: ex msg:' during IndexManager>>removeAllCompleteIndexesOn:'.
      ex pass .
    ]
]

{ #category : 'Index Maintenance' }
IndexManager >> removeAllIncompleteIndexes [

  self getAllNSCRoots do: [:nsc |
    self removeAllIncompleteIndexesOn: nsc
  ].

]

{ #category : 'Index Updating' }
IndexManager >> removeAllIncompleteIndexesOn: anNSC [
  "Remove all incomplete indexes for anNSC.
   Return the number of incomplete indexes found."

  | iList incompleRootIndexes |
  (iList := anNSC _indexedPaths) == nil
    ifTrue: [ ^ 0 ].
  incompleRootIndexes := Array new.
  1 to: iList size by: 2 do: [ :i |
    " scan each entry to see if a non-root index is present "
    ((iList at: i + 1) == 1 _and: [ (iList at: i) isComplete not ])
      ifTrue: [
        " if it is an incomplete root index ... add it to a list "
        incompleRootIndexes add: (iList at: i) ] ].
  IndexManager current
    executeStartingIndexMaintenance: [
      incompleRootIndexes
        do: [ :indexObj |
          indexObj size > 0
            ifTrue: [
              | pathTerm |
              pathTerm := indexObj _findFirstUnsharedPathTerm.
              indexObj nscRoot _undoIndexCreation: indexObj pathTerm: pathTerm ].
          self _removeIndex: indexObj ] ].
  ^ incompleRootIndexes size

]

{ #category : 'Index Maintenance' }
IndexManager >> removeAllIndexes [
  "This algorithm ignores the collections and traverses the dependent objects
   directly removing the dependent for each object that does not have a
   modification tracking dependency. The objects with modification tracking
   dependencies are tracked and the SharedDependencyLists table is rebuilt
   for only those objects ... Without any modification tracking going on, the
   SharedDependencyLists table is simply reinitialized. The bookkeeping required
   for handling modification tracking dependencies is the extra cost that makes
   this one a bit more expensive than _oldRemoveAllIndexes in the simple 'flat'
   case.

   IndexManagerAutoCommitPolicy is used to periodically commit if
   temp object space gets low and to perform a final commit at end
   of operation.

   The removeAllIndexes progress count is used to track progress, but
   now tracks the number of indexed objects left that have path terms
   to remove."

  | sys |
  sys := System.
  sys
    setIndexProgressCountTo: UnorderedCollection statValueForRemovingAllIndexes.	"Setup progress counter"
  self
    executeStartingIndexMaintenance: [
      [
      | sharedDepLists trackedObjects |
      "Perform using index manager auto commit.."
      sharedDepLists := SharedDependencyLists allEntries.
      trackedObjects := IdentitySet new.
      DependencyList depMapKeysToHiddenSet: 2.
      sys setProgressCountTo: (sys hiddenSetSize: 2).
      System
        hiddenSet: 2
        do: [ :dependentObject |
          | depList hasTracker terms |
          depList := DependencyList for: dependentObject.
          hasTracker := false.
          terms := {}.
          1 to: depList size by: 2 do: [ :i |
            (depList at: i + 1) = 0
              ifTrue: [ hasTracker := true ]
              ifFalse: [
                | pathTerm |
                pathTerm := depList at: i.
                terms add: pathTerm ] ].
          hasTracker
            ifTrue: [
              "modification trackers present, so remove the individual pathTerms
               from dependencyList"
              terms
                do: [ :pathTerm | DependencyList removePathTerm: pathTerm for: dependentObject ].
              trackedObjects add: dependentObject ]
            ifFalse: [
              sharedDepLists remove: depList otherwise: nil .
              DependencyList set: nil for: dependentObject ].
          sys decrementProgressCountBy: 1 ].
      self getAllNSCRoots
        do: [ :nsc |
          | iList |
          "Clear _indexedPaths field from all indexed collections"
          iList := nsc _indexedPaths.
          iList
            ifNotNil: [
              iList hasTrackingObjects
                ifTrue: [ iList removeAllIndexesFor: nsc ]
                ifFalse: [ nsc _indexedPaths: nil ] ] ].
      sharedDepLists isEmpty
        ifTrue: [
          "no modification trackers present, so simply initialize SharedDependencyLists"
          SharedDependencyLists initialize ]
        ifFalse: [
          | trackedDepLists |
          "rebuild SharedDependencyLists for modification trackers"
          trackedDepLists := IdentitySet new.
          trackedObjects
            do: [ :dependentObject |
              | depList |
              depList := DependencyList for: dependentObject.
              trackedDepLists add: depList ].
          SharedDependencyLists initialize.
          trackedDepLists
            do: [ :depList | SharedDependencyLists at: depList logging: false ] ].
      self resetAllIndexes	"Initialize IndexManager allIndexes reference" ]
        ensure: [ sys setIndexProgressCountTo: 0; setProgressCountTo: 0 ] ]

]

{ #category : 'Index Updating' }
IndexManager >> removeAllIndexesOn: anNSC [

  | result |
result := self removeAllCompleteIndexesOn: anNSC.
result == anNSC
  ifTrue: [
    self removeAllIncompleteIndexesOn: anNSC.
  ].
^result

]

{ #category : 'Index Maintenance' }
IndexManager >> removeAllTracking [
  "This method removes all indexes and all modification tracking meta data. This is
   the fastest algorithm of all. It unconditionally removes the dependencs for each
   object.

   IndexManagerAutoCommitPolicy is used to periodically commit if
   temp object space gets low and to perform a final commit at end
   of operation.

   The removeAllIndexes progress count is used to track progress, but
   now tracks the number of indexed objects left that have path terms
   to remove."

  | sys |
  "Setup progress counter"
  sys := System.
  sys
    setIndexProgressCountTo: UnorderedCollection statValueForRemovingAllIndexes.
  "Perform using index manager auto commit.."
  self
    executeStartingIndexMaintenance: [
      [
      | nscs |
      nscs := self getAllNSCRoots.
      sys setProgressCountTo: nscs size.
      DependencyList depMapKeysToHiddenSet: 2.
      DependencyList clearForHiddenSet: 2.	"Clear dependencyList for the dependent objects"
      1 to: nscs size do: [ :k |
        | uc |
        uc := nscs _at: k.
        uc _indexedPaths: nil.	"Clear the indexed structures from UnorderedCollections"
        sys decrementProgressCountBy: 1 ].
      SharedDependencyLists initialize.	"Initialize the SharedDependencyLists structure"
      self resetAllIndexes	"Initialize IndexManager allIndexes reference" ]
        ensure: [ sys setIndexProgressCountTo: 0 ; setProgressCountTo: 0 ] ]

]

{ #category : 'Index Updating' }
IndexManager >> removeEqualityIndexFor: anNSC on: aPathString [

  self _removeIndex: self _equalityIndex for: anNSC on: aPathString

]

{ #category : 'Index Updating' }
IndexManager >> removeIdentityIndexFor: anNSC on: aPathString [

  self _removeIndex: self _identityIndex for: anNSC on: aPathString

]

{ #category : 'Index Updating' }
IndexManager >> resetAllIndexes [
  allIndexes
    ifNil: [
      "force lazy init to fresh RcIdentityBag to avoid conflicts (bug43804)"
      self getAllIndexes ]
    ifNotNil: [
      allIndexes _isRcIdentityBag
        ifTrue: [
          "switch to using RcIdentitySet"
          allIndexes := nil.
          self getAllIndexes ]
        ifFalse: [
          "allow for concurrent index creation (bug43803)"
          allIndexes removeAll: allIndexes ] ]

]

{ #category : 'Accessing' }
IndexManager >> sessionAutoCommit [

^ SessionTemps current at: #INDEX_MANAGER_AUTO_COMMIT otherwise: nil

]

{ #category : 'Updating' }
IndexManager >> sessionAutoCommit: aBoolOrNil [
"Override persistent autoCommit on a session by session basis.
 If <aBoolOrNil> is nil, remove override.
 If true, this enabled commits to be performed without specific commands, including
 due to TransactionBacklog"

aBoolOrNil 
  ifNil:[ SessionTemps current removeKey: #INDEX_MANAGER_AUTO_COMMIT otherwise: nil ]
  ifNotNil: [ SessionTemps current at: #INDEX_MANAGER_AUTO_COMMIT put: aBoolOrNil ]

]

{ #category : 'Testing' }
IndexManager >> shouldCommit [
"Answer true if a commit should be performed during index maintenance.
   Note - that this method should only be executed from within a
   executeStartingIndexMaintenance: block, since _autoCommitPolicy
   may not have been initialized correctly otherwise."

| policy |
self sessionAutoCommit ~~ true ifTrue: [ ^ false ].

(policy := self _autoCommitPolicy ) ifNil:[ ^ false ].
^ policy shouldCommit

]

{ #category : 'Accessing' }
IndexManager >> sortNodeClass [

  self useKeyEncryption ifTrue: [ ^ BasicSortNode ].

  ^ SortNode

]

{ #category : 'Querying' }
IndexManager >> usageReport [
"
	Returns a string describing all indexes, the collection sizes, and
	the (visible) methods that could create these indexes.

	IndexManager current usageReport.
"

	| indexes methods keys stream |
	(stream := (AppendStream on: String new))
		lf; nextPutAll: 'Index Type - path - lastElementClass';
		lf; tab; nextPutAll: 'Collection size (collection OOP)';
		lf; tab; tab; nextPutAll: 'Index creation methods';
		lf.
	indexes := self _reportDataForIndexes.
	methods := self _reportDataForMethods.
	keys := (indexes collect: [:each | each key]) asSet asSortedCollection.
	keys do: [:eachKey |
		stream lf; nextPutAll: eachKey.
		(indexes select: [:each | each key = eachKey]) do: [:each |
			stream lf; tab; nextPutAll: each value.
		].
		(methods select: [:each | each key = eachKey]) do: [:each |
			stream lf; tab; tab; nextPutAll: each value.
		].
		stream lf.
	].
	^stream contents.

]

{ #category : 'Accessing' }
IndexManager >> useKeyEncryption [

  ^ useKeyEncryption

]

{ #category : 'Updating' }
IndexManager >> useKeyEncryption: aBool [

  useKeyEncryption := aBool

]

{ #category : 'Querying' }
IndexManager >> auditIndexes [
  | nscs errs |
  nscs := IdentitySet new .
  allIndexes ifNotNil:[:a | 
    a do:[:idx | nscs add: idx nscRoot ].
  ].
  errs := { } .
  nscs do:[:aColl | | status |
    status := aColl _fastAuditIndexes .
    (status at: 1) ifFalse:[ errs add: (status at: 2) ].
  ].
  errs size > 0 ifTrue:[
    GsFile gciLogServer:'ERROR ', errs printString .
    ^ errs
  ].
  ^ 'Indexes are OK'
]

{ #category : 'Querying' }
IndexManager >> allIndexesReport [
  | rpt coll |
  rpt := String new .
  coll := Array withAll: self getAllIndexes .
  1 to: coll size do:[:n | | anIndex nsc |
    anIndex := coll at: n .
    rpt add: n asString; add: ': ';
      add: 'a', anIndex class name, '(oop ', anIndex asOop asString, ')' .
    nsc := anIndex nscRoot .
    rpt add: ' on a', nsc class name, '(oop ', nsc asOop asString,')';
        add: ' size ', nsc size asString ; lf .
    rpt add: '     path ', anIndex pathComponentsString printString .
    ([ anIndex lastElementClass ] on: Error do:[:ex| ex return: nil ])
      ifNotNil:[:lastCls| rpt add: ' lastElementClass ', lastCls name, ' '  ].
    rpt lf ; add:'    ', anIndex options printString ; lf .
    rpt add: '  pathTerms'.
    anIndex do:[:aPathTerm |
      rpt add: ' ', aPathTerm class name .
    ].
    rpt lf .
  ].
  ^ rpt
]

