Extension { #name : 'IndexedQueryEvaluator' }

{ #category : 'Instance Creation' }
IndexedQueryEvaluator class >> on: anIndex [

  ^ self new index: anIndex

]

{ #category : 'Support' }
IndexedQueryEvaluator >> _allIdenticalTo: anObject in: nsc [

"Returns a new NSC containing the given object with the number of occurrences
 as in the root NSC."

| result tmp |
result := self _primAllIdenticalTo: anObject in: nsc _asIdentityBag.
result class = nsc species
  ifFalse: [
    tmp := nsc species new.
    tmp addAll: result.
    result := tmp
  ].
^ result

]

{ #category : 'Support' }
IndexedQueryEvaluator >> _allIdenticalTo: anObject in: nsc do: aBlock [
  "Returns a new NSC containing the given object with the number of occurrences
 as in the root NSC."

  | result |
  result := self _primAllIdenticalTo: anObject in: nsc _asIdentityBag.
  result do: aBlock

]

{ #category : 'Error Handling' }
IndexedQueryEvaluator >> _errorCannotInvokeRangeOperationOnIdentityIndex [

"An attempt was made to use a range operation (< > = <= >=) on a path
 expression only supported by an identity index."

^ self index _errorCannotInvokeRangeOperationOnIdentityIndex

]

{ #category : 'Searching' }
IndexedQueryEvaluator >> _findAllValuesThroughSetsNotIdenticalTo: aKey [
  "Returns a set of all values that satisfy the query.  This method works
 when the path is through a set-valued instance variable."

  | tmpVals tmpHolder startPt pathTerm idx |
  tmpHolder := NscBuilder
    for: self nscRoot species new
    max: self sizeForNscBuilder.
  idx := self index.
  tmpVals := idx indexDictionary
    _findAllValuesNotIdenticalTo: aKey
    atTerm: (idx at: idx lastPathComponentsDictionaryOffset)
    into: Array new.
  startPt := (idx lastPathComponentsDictionaryOffset - 1) max: 1.
  pathTerm := idx at: startPt.
  1 to: tmpVals size do: [ :i |
    idx
      _addAllFor: (tmpVals _at: i)
      into: tmpHolder
      offset: startPt
      pathTerm: pathTerm ].
  ^ tmpHolder completeBag

]

{ #category : 'Searching' }
IndexedQueryEvaluator >> _findFirstKeyNotIdenticalTo: aValue atTerm: pathTerm [

  ^IndexedQueryEvaluator subclassResponsibility: #_findFirstKeyNotIdenticalTo:atTerm:

]

{ #category : 'Searching' }
IndexedQueryEvaluator >> _findFirstValueForKey: aKey [

"Returns any root value for the given key by following the path terms in reverse
 order.  If none are found, returns aValue."

| startPt tmpVal incomplete |

self index hasIndexDictionary
  ifFalse: [ ^ aKey ].

tmpVal := aKey.
self index isIndexOnNscElements
    ifTrue: [ startPt := self index size ]
    ifFalse: [ startPt := self index lastPathComponentsDictionaryOffset ].

incomplete := #_incompletePathTraversal.
startPt _downTo: 1 do: [ :i |
  tmpVal := self index indexDictionary
    firstAt: tmpVal
    term: (self index at: i)
    otherwise: incomplete.
  (tmpVal == incomplete)
    ifTrue: [ ^ incomplete ].
].
^ tmpVal

]

{ #category : 'Support' }
IndexedQueryEvaluator >> _primAllIdenticalTo: anObject in: nsc [

"Returns a new NSC containing the given object with the number of occurrences
 as in the root NSC."

<primitive: 340>

self _primitiveFailed: #_primAllIdenticalTo:in: args: { anObject . nsc } .
self _uncontinuableError

]

{ #category : 'Accessing' }
IndexedQueryEvaluator >> btreeComparisonQuerySpec [

  ^self index btreeComparisonQuerySpec

]

{ #category : 'Accessing' }
IndexedQueryEvaluator >> collator [
  "Return the IcuCollator to be used in comparing Unicode strings"

  ^ self index collator

]

{ #category : 'Accessing' }
IndexedQueryEvaluator >> doBlock [
  ^ doBlock

]

{ #category : 'Accessing' }
IndexedQueryEvaluator >> doBlock: aBlock [
  doBlock := aBlock

]

{ #category : 'Query Select' }
IndexedQueryEvaluator >> findAllValuesIdenticalTo: aValue [

  ^IndexedQueryEvaluator subclassResponsibility: #findAllValuesIdenticalTo:

]

{ #category : 'Query Select' }
IndexedQueryEvaluator >> findAllValuesNotIdenticalTo: aValue [
  "Returns a set of all values that satisfy the query."

  | tmpHolder res |
  self index isIndexOnRootNsc
    ifTrue: [
      tmpHolder := self nscRoot species new.
      tmpHolder addAll: self nscRoot.
      tmpHolder removeAll: (self _allIdenticalTo: aValue in: self nscRoot).
      doBlock ifNotNil: [ tmpHolder do: doBlock ].
      ^ tmpHolder ].
  self hasCollectionBasedTerm
    ifTrue: [
      res := self _findAllValuesThroughSetsNotIdenticalTo: aValue.
      doBlock ifNotNil: [ res do: doBlock ].
      ^ res ].
  tmpHolder := NscBuilder
    for: self nscRoot species new
    max: self sizeForNscBuilder.
  tmpHolder _addAllNsc: (self _findAllValuesIdenticalToAsNsc: aValue).
  " get all objects identical to aValue and identical to nil "
  tmpHolder _addAllNsc: (self _allIdenticalTo: nil in: self nscRoot).
  1 to: self index size - 1 do: [ :i |
    " get all objects with nil along the path "
    self index
      _addAllFor: nil
      into: tmpHolder
      offset: i
      pathTerm: (self index at: i) ].
  res := self nscRoot _asIdentityBag - tmpHolder completeBag _asIdentityBag.
  doBlock ifNotNil: [ res do: doBlock ].
  res class == self nscRoot species
    ifFalse: [
      | finalRes |
      finalRes := self nscRoot species new.
      finalRes addAll: res.
      ^ finalRes ].
  ^ res

]

{ #category : 'Query Select' }
IndexedQueryEvaluator >> findAllValuesNotIdenticalTo: aValue do: aBlock [
  "Iterates over all values that satisfy the query."

  | result |
  result := self findAllValuesNotIdenticalTo: aValue.
  result do: aBlock.
  ^ result

]

{ #category : 'Query Detect' }
IndexedQueryEvaluator >> findFirstValueIdenticalTo: aValue [

  ^IndexedQueryEvaluator subclassResponsibility: #findFirstValueIdenticalTo:

]

{ #category : 'Query Detect' }
IndexedQueryEvaluator >> findFirstValueNotIdenticalTo: aValue [

"Returns the first value that satisfies the query."

| key lastPathTerm |

lastPathTerm := self index lastPathTerm.
" optimize query for objects in the NSC itself "
self index isIndexOnRootNsc
  ifTrue: [
    index nscRoot do: [ :element |
      element == aValue
        ifFalse: [ ^ element ] ].
    ^ #_incompletePathTraversal ].

key := self _findFirstKeyNotIdenticalTo: aValue atTerm: lastPathTerm.
(key == #_incompletePathTraversal)
  ifTrue: [ ^ #_incompletePathTraversal ].

^ self findFirstValueIdenticalTo: key

]

{ #category : 'Testing' }
IndexedQueryEvaluator >> hasCollectionBasedTerm [
  ^ self index hasCollectionBasedTerm

]

{ #category : 'Testing' }
IndexedQueryEvaluator >> hasSetValuedTerm [
  ^ self index hasSetValuedTerm

]

{ #category : 'Accessing' }
IndexedQueryEvaluator >> index [

  ^index

]

{ #category : 'Updating' }
IndexedQueryEvaluator >> index: anIndex [

  index := anIndex

]

{ #category : 'Accessing' }
IndexedQueryEvaluator >> nscRoot [

^self index nscRoot

]

{ #category : 'Accessing' }
IndexedQueryEvaluator >> sizeForNscBuilder [

"Returns the size to be used when an NscBuilder is collecting objects."

^ self nscRoot size

]
