Extension { #name : 'UnorderedCollection' }

{ #category : 'Object Canonicalization' }
UnorderedCollection >> _canonicalizeReferencesUsingPolicy: aReferencingObjectPolicy [

	^aReferencingObjectPolicy canonicalizeReferencesInCollection: self

]

{ #category : 'Indexing Support' }
UnorderedCollection >> _getIndexDictionary [

"Returns the index dictionary that is shared by all indexes.  If there are no
 indexes, create an index dictionary."

| iList iDict |
iList := self _indexedPaths.
iList == nil
  ifTrue: [
    iDict := RcIndexDictionary new: self _getIndexDictionaryCreationSize.
  ]
  ifFalse: [
    1 to: iList size by: 2 do: [ :i |
      " only use the index dictionary if the receiver is the root
        NSC of the index "
      ( (iList at: i + 1) == 1 and:
      [ (iList at: i) indexDictionary ~~ nil ] )
        ifTrue: [ ^ (iList at: i) indexDictionary ]
    ].
    iDict := RcIndexDictionary new: self _getIndexDictionaryCreationSize.
  ].
iDict objectSecurityPolicy: GsIndexingObjectSecurityPolicy.

^ iDict

]

{ #category : 'Indexing Support' }
UnorderedCollection >> _getIndexList [

"Returns the index list for the receiver.  If one does not exist, create it."

| iList |
iList := self _indexedPaths.
iList == nil
  ifTrue: [
    iList := IndexList new.
    self _indexedPaths: iList.
                                                        "deleted moveToDisk"
    iList objectSecurityPolicy: GsIndexingObjectSecurityPolicy.
  ].

^ iList

]

{ #category : 'Updating Indexes - Private' }
UnorderedCollection >> _removeAllIncompleteIndexes [

" Remove index list entries for any incomplete indexes found in the receiver.
Returns the number of incomplete indexes found."

^IndexManager current removeAllIncompleteIndexesOn: self

]

{ #category : 'Updating Indexes - Private' }
UnorderedCollection >> _removeIndex: indexObj [
  "Removes the index from the receiver.  Returns the receiver."

  | pathTerm parentTerm iList allTerms |
  " since indexes may share path terms, find the first one that is not shared "
  pathTerm := indexObj _findFirstUnsharedPathTerm.
  pathTerm
    ifNil: [
      indexObj _findSetValuedPathTermOffset == 0
        ifFalse: [
          pathTerm := indexObj lastPathTerm.
          pathTerm _doRemoveSubsumedIndex: indexObj nsc: self ] ]
    ifNotNil: [
      allTerms := pathTerm _doRemoveIndex: indexObj nsc: self.
      parentTerm := pathTerm getParentTerm ].
  1 to: indexObj size do: [ :i |
    " remove index from each path term "
    (indexObj at: i) removeIndex: indexObj ].
  allTerms
    ifNotNil: [
      " remove dependency lists from global table that are no longer needed "
      SharedDependencyLists removeEntriesContaining: allTerms ].
  iList := self _indexedPaths.
  iList removeIndex: indexObj withOffset: 1 for: self.	" remove the entry in the receiver's index list "
  parentTerm ifNotNil: [ parentTerm _determineChildren ]

]

{ #category : 'Updating Indexes - Private' }
UnorderedCollection >> _undoCreationOfSubsumedIndex: indexObj [
  "The given index is subsumed by an existing index and is incomplete."

  "nothing special needs to be done over and above that done when the index itself is removed ... keep this method as place holder"



]

{ #category : 'Updating Indexes - Private' }
UnorderedCollection >> _undoIndexCreation: indexObj pathTerm: firstUnsharedPathTerm [
  "Removes any vestiges of the incomplete index, and attempts to commit this
 transaction until successful."

  | parentTerm iList bag incomplete indexManager obj endObj |
  firstUnsharedPathTerm
    ifNil: [ self _undoCreationOfSubsumedIndex: indexObj ]
    ifNotNil: [
      | setOffset |
      indexManager := IndexManager current.
      parentTerm := firstUnsharedPathTerm getParentTerm.
      bag := self _asIdentityBag.
      incomplete := #'_incompletePathTraversal'.
      ((setOffset := indexObj _findSetValuedPathTermOffset) ~~ 0
        _and: [ setOffset < firstUnsharedPathTerm offset ])
        ifTrue: [
          | updateIndexList objNsc |
          "See if there is a set-valued instance variable before the unshared path term."
          updateIndexList := false.
          (firstUnsharedPathTerm == indexObj lastPathTerm
            _and: [
              firstUnsharedPathTerm size > 1
                _and: [ (firstUnsharedPathTerm at: 1) size ~= (firstUnsharedPathTerm at: 2) size ] ])
            ifTrue: [
              " see if index is subsumed by an existing index "
              firstUnsharedPathTerm indicatesMultiValue
                ifTrue: [
                  " if last path term is #*, will need to update the last objects' index list "
                  updateIndexList := true.
                  objNsc := NscBuilder
                    for: IdentityBag new
                    max: indexObj sizeForNscBuilder ]
                ifFalse: [ objNsc := nil ] ]
            ifFalse: [ objNsc := NscBuilder for: IdentityBag new max: indexObj sizeForNscBuilder ].
          updateIndexList
            ifFalse: [
              1 to: bag size do: [ :i |
                " for each element in the NSC ... "
                " traverse up to the point of the unshared path term "
                indexObj
                  traverse: (bag _at: i)
                  upTo: firstUnsharedPathTerm
                  startingAt: 1
                  into: objNsc ].
              objNsc ~~ nil
                ifTrue: [
                  objNsc := objNsc completeBag.
                  1 to: objNsc size do: [ :j |
                    endObj := objNsc _at: j.
                    endObj ~~ incomplete
                      ifTrue: [
                        " only remove mappings once for a given object "
                        firstUnsharedPathTerm
                          undoMappingsFor: endObj
                          lastOne: true
                          logging: false.
                        DependencyList
                          removePathTerm: firstUnsharedPathTerm
                          for: endObj ] ] ] ] ]
        ifFalse: [
          firstUnsharedPathTerm indicatesIndexOnRootNsc
            ifTrue: [
              1 to: bag size do: [ :j |
                obj := bag _at: j.
                firstUnsharedPathTerm undoDirectMappingFor: obj logging: false.
                DependencyList removePathTerm: firstUnsharedPathTerm for: obj ] ]
            ifFalse: [
              1 to: bag size do: [ :j |
                endObj := indexObj traverse: (bag _at: j) upTo: firstUnsharedPathTerm.
                (nil ~~ endObj and: [ incomplete ~~ endObj ])
                  ifTrue: [
                    firstUnsharedPathTerm undoMappingsFor: endObj lastOne: true logging: false.
                    DependencyList
                      removePathTerm: firstUnsharedPathTerm
                      for: endObj ] ] ] ] ].
  1 to: indexObj size do: [ :i |
    " remove index from each path term "
    (indexObj at: i) removeIndex: indexObj ].
  firstUnsharedPathTerm
    ifNotNil: [
      " make path term (and its children) unusable "
      firstUnsharedPathTerm _makeObsolete ].
  (iList := self _indexedPaths) ~~ nil
    ifTrue: [
      " remove the entry in the receiver's index list "
      iList removeIndex: indexObj withOffset: 1 for: self ].
  parentTerm ~~ nil
    ifTrue: [ parentTerm _determineChildren ]

]

{ #category : 'Updating Indexes' }
UnorderedCollection >> createEqualityIndexOn: aPathString withLastElementClass: aClass [
  "Creates an equality index on the path specified by aPathString.  The equality
 index is ordered according to the sort-provided comparison operators provided
 by aClass."

  "Note: An error occurring during index creation may cause the
 current transaction to be unable to commit."

  self createEqualityIndexOn: aPathString withLastElementClass: aClass options: GsIndexOptions default

]

{ #category : 'Updating Indexes' }
UnorderedCollection >> createEqualityIndexOn: aPathString withLastElementClass: aClass options: indexOptions [
  "Creates an equality index on the path specified by aPathString using the given indexOptions.
   The equality index is ordered according to the sort-provided comparison operators provided
   by aClass."

  "Note: An error occurring during index creation may cause the
 current transaction to be unable to commit."

  GsIndexSpec new
    equalityIndex: aPathString lastElementClass: aClass options: indexOptions;
    createIndexesOn: self

]

{ #category : 'Updating Indexes' }
UnorderedCollection >> createIdentityIndexOn: aPathString [
  "Creates an identity index on aPathString.  Generates an error if aPathString
 is not a path for the element kind of the receiver or if any term of the path
 except the last term is not constrained."

  "If an error occurs during index creation, it may not be possible to commit the
 current transaction later."

  GsIndexSpec new
    identityIndex: aPathString;
    createIndexesOn: self

]

{ #category : 'Updating Indexes' }
UnorderedCollection >> createRcEqualityIndexOn: aPathString withLastElementClass: aClass [
  "Creates an equality index on the path specified by aPathString.  The equality
 index is ordered according to the sort-provided comparison operators provided
 by aClass."

  "Note: An error occurring during index creation may cause the
 current transaction to be unable to commit."

  GsIndexSpec new
    equalityIndex: aPathString
      lastElementClass: aClass
      options: GsIndexOptions reducedConflict;
    createIndexesOn: self

]

{ #category : 'Repository Conversion' }
UnorderedCollection >> fixRefsAfterConversion [

"Default UnorderedCollection method for fixing references 
 ObsLargePositiveInteger and ObsLargeNegativeInteger instances that can
 now be represented as a SmallInteger and Floats and SmallFloats 
 which can now be represented as a SmallDouble."

| aBagOrSet convBm |
convBm := (GsBitmap newForHiddenSet: #Conversion).
(convBm includes: self) 
        ifTrue:[^false]. "already fixed this one"

"Fix inst var refs first"
self fixInstVarRefsAfterConversion.

"now handle the rest of it"
aBagOrSet := self select:[:e| e needsFixingAfterConversion].
1 to: aBagOrSet _basicSize do:[:n| |obj|
        obj := aBagOrSet _at: n.
        self remove: obj.
        obj := obj + 0.
        self add: obj.
].
convBm add: self.
^true

]

{ #category : 'Accessing Indexes' }
UnorderedCollection >> indexSpec [
  | indexSpec |
  indexSpec := GsIndexSpec new.
  self _indexedPaths ifNil: [ ^ indexSpec ].
  self _indexedPaths
    indexObjectsDo: [ :indexObj | indexSpec specs add: indexObj asIndexSpecification ].
  ^ indexSpec

]

{ #category : 'Updating Indexes' }
UnorderedCollection >> removeAllIndexes [

"Remove all indexes for the receiver. If all of the receiver's indexes can be
 removed, this method returns the receiver."

"If an error occurs during index removal, it may not be possible to commit the
 current transaction later."

| mgr |
(mgr := IndexManager current) executeStartingIndexMaintenance: [
      mgr removeAllIndexesOn: self ] .
^ self

]

{ #category : 'Updating Indexes' }
UnorderedCollection >> removeEqualityIndexOn: aPathString [

"If an equality index exists on aPathString, remove that index.  If the
 path string is invalid or no index exists on the given path, an error
 is raised.  If aPathString is an implicit index (due to the receiver's
 participation as a set-valued instance variable in some other NSC's
 index), then this method returns the path string."

"If an error occurs during index removal, it may not be possible to commit the
 current transaction later."

| mgr |
self _checkIndexPathExpression: aPathString.

(mgr := IndexManager current) executeStartingIndexMaintenance:[
    mgr removeEqualityIndexFor: self on: aPathString
] .

]

{ #category : 'Updating Indexes' }
UnorderedCollection >> removeIdentityIndexOn: aPathString [

"If an identity index exists on aPathString, and aPathString is not a
 proper prefix of some indexed path, the index is removed.  If the
 path string is invalid or no index exists on the given path, an error
 is raised.  If aPathString is an implicit index (due to the receiver's
 participation as a set-valued instance variable in some other NSC's
 index), then this method returns the path string."

"If an error occurs during index removal, it may not be possible to commit the
 current transaction later."

| mgr |
self _checkIndexPathExpression: aPathString.

(mgr := IndexManager current) executeStartingIndexMaintenance: [
  mgr removeIdentityIndexFor: self on: aPathString ] .

]

{ #category : 'Indexing Audit' }
UnorderedCollection >> _auditBtreeCounts: btreeCounts on: aString [

btreeCounts do: [:each |
  (each class == Array) ifTrue: [
      self _auditBtreeCounts: each on: aString.
    ] ifFalse: [
      "must be a 0, or there is a problem with index structure"
      each ~~ 0 ifTrue: [
          aString lf ;
             add: ' -- Number of entries in Btree does not match number of entries in base collection'; lf .
          BtreePlusNodeAuditor checkDebug:'bad count'. 
        ].
    ].
].

]

