Extension { #name : 'ObsoleteSymbolDictionary' }

{ #category : 'Repository Conversion' }
ObsoleteSymbolDictionary class >> _correspondingNewClass [

"The class all instances of receiver are converted to during conversion."

(self == ObsoleteSymbolDictionary)
  ifTrue: [ ^ SymbolDictionary ].

^ self.

]

{ #category : 'Storing and Loading' }
ObsoleteSymbolDictionary class >> loadFrom: passiveObj [

"Reads from passiveObj the passive form of an object with named instance
 variable format.  Converts the object to its active form by loading the
 information into a new instance of the receiver.  Returns the new instance."

^ self loadFrom: passiveObj mappingToClass: SymbolDictionary

]

{ #category : 'Accessing' }
ObsoleteSymbolDictionary >> _behaviorKeys [

"Returns a SymbolSet containing keys in the receiver whose values are
 Behaviors."

"This is used as an optimization by the GemBuilder for Smalltalk browser."

| result |
result := SymbolSet new.
1 to: self size do:
  [ :i |
    (self _at: i) value isBehavior
      ifTrue: [ result _addSymbol: (self _at: i) key ].
  ].
^result

]

{ #category : 'Browser Methods' }
ObsoleteSymbolDictionary >> _classAndVersionStrings [

"For all Behaviors in the receiver, returns an OrderedCollection of Strings
 showing the class name and version.  This method is used as an optimization by
 the GemBuilder for Smalltalk browser."

| result |
result := OrderedCollection new .
self do: [ :anAssoc | | each |
  each := anAssoc value.
  each isBehavior
  ifTrue: [ result add:
              ( each classHistory size = 1
                 ifTrue: [ each name asString ]
                 ifFalse: [ each name , ' [ ' ,
                      ( each classHistory indexOf: each ) printString , ' ]' ]
               )
    ]
  ].
^result

]

{ #category : 'Adding' }
ObsoleteSymbolDictionary >> addAll: aCollection [

"Adds all of the elements of aCollection to the receiver and returns the
 receiver.  If aCollection is a kind of KeyValueDictionary, then adds new
 Associations that reference the key/value pairs found in aCollection."

(self == aCollection) ifTrue: [ ^ self addAll: (aCollection copy) ].
aCollection accompaniedBy: self do: [:me :each | me add: each ].
^ self

]

{ #category : 'Repository Conversion' }
ObsoleteSymbolDictionary >> asGsMethodDictionary [

"Convert an instance of ObsoleteSymbolDictionary to an instance of
 GsMethodDictionary."

| newGsDict |

newGsDict := GsMethodDictionary new.
self keysAndValuesDo: [ :aKey :aValue |
  newGsDict at: (aKey convertTo5) put: aValue.
  ].
newGsDict objectSecurityPolicy: self objectSecurityPolicy .
^ newGsDict.

]

{ #category : 'Accessing' }
ObsoleteSymbolDictionary >> associationAt: aKey ifAbsent: aBlock [

"Returns the SymbolAssociation with key aKey.  If no such SymbolAssociation
 exists, returns the result of evaluating the zero-argument block aBlock."

<primitive: 48>
| aSym |
aKey _isSymbol ifFalse:[
  aSym := Symbol _existingWithAll: aKey .
  aSym ~~ nil ifTrue:[
    ^ self associationAt: aSym ifAbsent: aBlock .
    ].
  ].
aBlock == nil ifTrue:[ ^ nil ] .
^ aBlock value

]

{ #category : 'Accessing' }
ObsoleteSymbolDictionary >> at: aKey [

"Returns the value of the SymbolAssociation with key aKey.  Generates an error
 if no such SymbolAssociation exists."

| anAssoc |
anAssoc := self associationAt: aKey
                ifAbsent: nil "avoid creating a ComplexBlock" .
anAssoc == nil ifTrue:[ anAssoc := self _errorKeyNotFound: aKey ].
^ anAssoc value

]

{ #category : 'Accessing' }
ObsoleteSymbolDictionary >> at: aKey ifAbsent: aBlock [

"Returns the value of the Association with key aKey.  If no such
 Association exists, returns the result of evaluating the zero-argument
 block aBlock."

| assoc |
assoc := self associationAt: aKey otherwise: nil .
assoc == nil ifTrue:[
  aBlock == nil ifTrue:[ ^ nil ].
  ^ aBlock value
  ].
^ assoc value

]

{ #category : 'Updating' }
ObsoleteSymbolDictionary >> at: aKey put: aValue [

"Creates a new SymbolAssociation with the given key and
 value and adds it to the receiver.  If the receiver already contains an
 Association with the given key, this makes aValue the value of that
 Association.  Returns aValue."

<primitive: 304>
aKey _isSymbol ifFalse:[
  ^ self at: (aKey asSymbol) put: aValue
  ].
^ self _primitiveFailed: #at:put: args: { aKey . aValue }

]

{ #category : 'Repository Conversion' }
ObsoleteSymbolDictionary >> convertTo5 [

"Returns an instance of SymbolDictionary containing the converted contents
 of the receiver."

<primitive: 2001> "enter protected mode"
| prot |
prot := System _protectedMode .
[
  | newSymDict tempArray |

  (self class == ObsoleteSymbolDictionary) ifFalse: [
    ^ self
  ].
  "User defined subclasses cannot be converted here."

  newSymDict := SymbolDictionary new.
  tempArray := Array new.

  self doAssociations: [ :anObsSymAssoc |
    tempArray add: (anObsSymAssoc convertTo5)
    ].

  newSymDict objectSecurityPolicy: self objectSecurityPolicy .
  self _becomeDictionary: newSymDict .

  tempArray do: [ :anAssoc |
    self addAssociation: anAssoc
    ].
] ensure:[
  prot _leaveProtectedMode
].
^ self

]

{ #category : 'Repository Conversion' }
ObsoleteSymbolDictionary >> convertToCategories [

"Convert an instance of ObsoleteSymbolDictionary to an instance of
 GsMethodDictionary."

| newGsDict |

newGsDict := GsMethodDictionary new.
self keysAndValuesDo: [ :aKey :aValue |
  newGsDict at: (aKey asSymbol) put: (aValue convertToSymbolSet).
  ].
newGsDict objectSecurityPolicy: self objectSecurityPolicy .
^ newGsDict.

]

{ #category : 'Searching' }
ObsoleteSymbolDictionary >> includesKey: aKey [

"Returns true if the receiver contains a SymbolAssociation whose key is equal
 to aKey.  Returns false otherwise."

self at: aKey ifAbsent:[ ^ false ].
^ true

]

{ #category : 'Accessing' }
ObsoleteSymbolDictionary >> keys [

"Reimplemented.  Returns a SymbolSet containing the receiver's keys."

| result |
result := SymbolSet new.
1 to: self size do:
  [ :i | result _addSymbol: (self _at: i) key ].
^result

]

{ #category : 'Accessing' }
ObsoleteSymbolDictionary >> name [

"Returns the key of an Association whose value is the receiver.  If the
 receiver contains no such Association, returns nil."

^ (self detectAssociations:{:i| i.value == self } ifNone:[^ nil ] ) key .

]

{ #category : 'Repository Conversion' }
ObsoleteSymbolDictionary >> rehashForConversion [

"Private. Converts the receiver to an instance of Dictionary. Only done if
 the receiver is an instance of ObsoleteDictionary. Instances of subclasses
 of ObsoleteDictionary are not converted. Returns the receiver."

<primitive: 2001> "enter protected mode"
| prot |
prot := System _protectedMode .
[ | newDict anArray onlySymbols |

  self class == ObsoleteSymbolDictionary ifFalse: [
    ^ self
  ].

  anArray := Array new.
  onlySymbols := true.
  self doAssociations: [ :anAssoc |
    (anAssoc class == SymbolAssociation) ifFalse: [ onlySymbols := false ].
    anArray add: anAssoc.
    ].

  (onlySymbols)
    ifTrue: [ newDict := SymbolDictionary new: (self size) ]
    ifFalse: [ newDict := Dictionary new: (self size) ].

  newDict objectSecurityPolicy: self objectSecurityPolicy.
  newDict _becomeDictionary: self.

  anArray accompaniedBy: self do: [:me :anAssoc | me add: anAssoc.  ].
] ensure:[
  prot _leaveProtectedMode
].
^ self

]
