"
Object
   ObjectFilteringPolicyMap( maps )

An ObjectFilteringPolicyMap specifies, for each possible IPv4 address,
which ObjectFilteringPolicy should be applied to x509 sessions on a
machine at that address.
 
An ObjectFilteringPolicy has no operational effect until it is installed 
by sending it #installObjectFilter.

  maps is a KeyValueDictionary , 
    keys are instances of IPv4Subnet , 
    values are instances of ObjectFilteringPolicy .

  Instances, the maps and the keys are all assigned to
  ObjectFiltersSecurityPolicy so by default they will
  not be transmitted to remote X509 caches.

  There is no API for removing entries. 
  Recommended practice is to build a new instance when 
  removals of subnets is needed.
"
Class {
	#name : 'ObjectFilteringPolicyMap',
	#superclass : 'Object',
	#instVars : Array [
		'maps'
	],
	#category : 'X509-Core'
}

{ #category : 'accessing' }
ObjectFilteringPolicyMap class >> installedObjectFilter [
  "Return the instance currently installed in HostAgentUser's  UserGlobals"
  ^ ((AllUsers userWithId:'HostAgentUser') resolveSymbol: #ObjectFilter) value
]

{ #category : 'instance creation' }
ObjectFilteringPolicyMap class >> new [
  |obj|
  (obj := super new) initialize ;
     objectSecurityPolicy: ObjectFiltersSecurityPolicy .
  ^ obj 
]

{ #category : 'updating' }
ObjectFilteringPolicyMap >> atSubnet: anIPv4Subnet putPolicy: anObjectFilteringPolicy [
  | mySecPolicy |
  mySecPolicy := self objectSecurityPolicy .
  anObjectFilteringPolicy objectSecurityPolicy == mySecPolicy ifFalse:[
    anObjectFilteringPolicy objectSecurityPolicy: mySecPolicy .
  ].
  (anIPv4Subnet isKindOf: IPv4Subnet) ifFalse:[
     ArgumentError new name: 'anIPv4Subnet' expectedClass: IPv4Subnet
                actualArg: anIPv4Subnet ; signal
  ].
  (anObjectFilteringPolicy isKindOf: ObjectFilteringPolicy) ifFalse:[
     ArgumentError new name: 'anObjectFilteringPolicy' expectedClass: ObjectFilteringPolicy
                actualArg: anObjectFilteringPolicy ; signal
  ].
  maps at: anIPv4Subnet copyForFilterMap put: anObjectFilteringPolicy .
]

{ #category : 'accessing' }
ObjectFilteringPolicyMap >> byteArrayForIP: ipAddrString [
  ^ (self policyForIP: ipAddrString) asByteArray 
]

{ #category : 'initialization' }
ObjectFilteringPolicyMap >> initialize [
  maps := KeyValueDictionary new .
  maps objectSecurityPolicy: ObjectFiltersSecurityPolicy .
  self atSubnet: (IPv4Subnet named: 'All' forSubnet: '0.0.0.0/0')
       putPolicy:  ObjectFilteringPolicy new .  
]

{ #category : 'updating' }
ObjectFilteringPolicyMap >> installObjectFilter [
  "Install the receiver as the value of UserGlobals at:#ObjectFilter
   In the HostAgentUser's  UserGlobals, so that any subsequent
   hostagent sessions will use the receiver.  "
  | assoc |
  assoc := (AllUsers userWithId:'HostAgentUser') resolveSymbol: #ObjectFilter .
  assoc value: self .
]

{ #category : 'printing' }
ObjectFilteringPolicyMap >> mappingReport [
  "Answers a string detailing the policies for all subnets."
  | fps str prevAssoc |
  fps := SortedCollection sortBlock:[:a :b | a key <= b key ] .
  maps keysAndValuesDo:[:k :v| fps add:(Association newWithKey: k value: v) ].
  str := String new .
  fps do:[:assoc |
    prevAssoc ifNotNil:[ str lf ].
    str addAll: assoc key asString ; addAll:' --> ' ; addAll: assoc value mappingReport .
    prevAssoc := assoc .
  ].
  ^ str .
]

{ #category : 'accessing' }
ObjectFilteringPolicyMap >> policiesForSubnet: arg [
  "arg may be either an IPv4Subnet or a Cidr string. "
  | res snet |
  (arg isKindOf: IPv4Subnet) ifTrue:[ snet := arg ]
          ifFalse:[ snet := IPv4Subnet named:'tmp' forSubnet: arg].
  res := SortedCollection sortBlock:[:a :b | a key <= b key ].
  maps keysAndValuesDo:[:k :v|
   (k containsSubnet: snet) ifTrue:[ res add:(Association newWithKey: k value: v)]
  ].
  ^ Array withAll: res .
]

{ #category : 'accessing' }
ObjectFilteringPolicyMap >> policyForIP: ipAddrString [
  "Returns an ObjectFilteringPolicy, the policy defined for
   the smallest subnet in the receiver which contains 
   the address specified by ipAddrString  "
  ^ (self _entryForIp: ipAddrString) at: 2 .
]

{ #category : 'printing' }
ObjectFilteringPolicyMap >> reportForIP: ipAddrString [
  "Returns an String, the mapping report for
   the smallest subnet in the receiver which contains 
   the address specified by ipAddrString  "
  | arr str |
  arr := self _entryForIp: ipAddrString . 
  (str := String new ) addAll: (arr at: 1) asString ; addAll: ' --> ';
     addAll: (arr at: 2) mappingReport .
  ^ str
]

{ #category : 'printing' }
ObjectFilteringPolicyMap >> reportForSubnet: arg [
  | arr str prev |
  arr := self policiesForSubnet: arg .
  str := String new .
  arr do:[:assoc |
     prev ifNotNil:[ str lf ].
     str addAll: assoc key asString ; addAll:' --> ';
        addAll: assoc value mappingReport .
     prev := assoc 
  ].
  ^ str
]

{ #category : 'accessing' }
ObjectFilteringPolicyMap >> size [
  ^ maps size
]

{ #category : 'accessing' }
ObjectFilteringPolicyMap >> subnetForIP: ipAddrString [
  "Returns an IPv4Subnet"
  ^ (self _entryForIp: ipAddrString) at: 1 .
]

{ #category : 'private' }
ObjectFilteringPolicyMap >> _entryForIp: ipAddrString [
  "Returns { anIPv4Subnet . anObjectFilteringPolicy }"
  | argCidr resNet resMask resPolicy |
  argCidr := Cidr fromString: ipAddrString , '/32' . 
  resMask := -1.
  maps keysAndValuesDo:[:k :v| | kCidr |
    kCidr := k cidr .
    (kCidr containsCidr: argCidr) ifTrue:[ | nm |
      (nm := kCidr netmask) > resMask ifTrue:[
        resMask := nm .
        resNet :=  k . 
        resPolicy := v .
      ].
    ].
  ].
  resPolicy ifNil:[ Error signal:'BUG: Failed to find any policy for ' , argCidr asString].
  ^ { resNet . resPolicy }
]
