"
No class-specific documentation for HostAgentTlsActor, hierarchy is: 
Object
  TlsActor( trustAnchor certificate privateKey libSsl libCrypto sslCtx)
    HostAgentTlsActor

"
Class {
	#name : 'HostAgentTlsActor',
	#superclass : 'TlsActor',
	#instVars : Array [
		'logger'
	],
	#category : 'X509-HostAgent'
}

{ #category : 'instance creation' }
HostAgentTlsActor class >> forSsl: anSsl [
	"Return an instance initialized with the context from which anSsl was created.
	anSsl must be a CPointer to an SSL struct."

	^self new initializeFromSsl: anSsl
]

{ #category : 'private-login' }
HostAgentTlsActor >> doTlsAccept: timeoutMs [
	| errStr |
  ssl := self class makeSslWithCtx: sslCtx fd: tcpSocket id .
  ssl ifNil:[ self error: 'malloc failure in makeSslWithCtx' ].

  "establish auto-free of ssl on GC of tcpSocket, if we don't succeed in
   passing the ssl and file descriptor to a new pgsvr thread."
  tcpSocket _installSsl: ssl .  

  "use the primitive in GsSecureSocket."
  errStr := tcpSocket _secureAcceptTimeoutMs: timeoutMs .
  errStr == false ifTrue:[ errStr := 'SSL accept timeout' ].
  errStr _isOneByteString ifTrue:[ 
    LoginDeniedError signal: 'SSL accept failed, ', errStr
  ].
]

{ #category : 'private-login' }
HostAgentTlsActor >> doTlsConnect: timeoutMs [
	| errStr |
  ssl := self class makeSslWithCtx: sslCtx fd: tcpSocket id .
  ssl ifNil:[ self error: 'malloc failure in makeSslWithCtx' ].

  "establish auto-free of ssl on GC of tcpSocket, if we don't succeed in
   passing the ssl and file descriptor to a new pgsvr thread."
  tcpSocket _installSsl: ssl .  

  "use the primitive in GsSecureSocket."
  errStr := tcpSocket _secureConnectTimeoutMs: timeoutMs . 
  errStr == false ifTrue:[ errStr := 'SSL connect timeout' ].
  errStr _isOneByteString ifTrue:[ 
    LoginDeniedError signal: 'SSL connect failed, ', errStr
  ].
]

{ #category : 'private-login' }
HostAgentTlsActor class >> makeSslWithCtx: aSslCtx fd: aFileDescriptor [
  "Builds an ssl by calling SSL_new ; then calls SSL_set_fd.
   Returns a CPointer encapsulating an SSL* , or nil if malloc failed."

  <primitive: 1071>
  aSslCtx _validateInstanceOf: CPointer .
  aFileDescriptor _validateInstanceOf: SmallInteger .
  self _primitiveFailed: #makeSsl: args: { aSslCtx . aFileDescriptor }.
]

{ #category : 'error handling' }
HostAgentTlsActor >> error: errorDescription [
  | msg |
  msg := errorDescription asString copy .
  msg lf ; addAll: (GsProcess stackReportToLevel: 100) ; lf .
  logger log: msg level: #error .
	LoginDeniedError signal: errorDescription
]

{ #category : 'initialization' }
HostAgentTlsActor >> initializeFromSsl: anSsl [

	sslCtx := libSsl SSL_get_SSL_CTX_: anSsl.
	self checkForNullReturn: sslCtx from: 'SSL_get_SSL_CTX'
]

{ #category : 'private' }
HostAgentTlsActor >> ipv4FromIpv6: aByteArray [
	"Given a 16-byte array representing an IPv6 IPv4-mapped address, answer the 4-byte equivalent IPv4 address."

	(aByteArray copyFrom: 1 to: 12) = #[0 0 0 0 0 0 0 0 0 0 255 255]
		ifFalse: 
			[self error: 'Byte array ' , aByteArray printString
						, ' is not a valid IPv4-mapped address.'].
	^aByteArray copyFrom: 13 to: 16
]

{ #category : 'accessing state' }
HostAgentTlsActor >> logger [
	^logger

]

{ #category : 'accessing state' }
HostAgentTlsActor >> logger: newValue [
	logger := newValue
]

{ #category : 'accessing state' }
HostAgentTlsActor >> peerCertificateTypes [
	"Get the peer certificate and get the certificate types out of the cert."

	| subject oid |
	subject := self peerSubject.
	oid := libCrypto OBJ_txt2obj_: '1.3.6.1.4.1.47749.1.1' _: 1.
	self checkForNullReturn: oid from: 'OBJ_txt2obj'.
	^self getAllValuesOf: oid from: subject
]

{ #category : 'accessing state' }
HostAgentTlsActor >> peerIpString [

	^tcpSocket peerAddress
]

{ #category : 'accessing state' }
HostAgentTlsActor >> peerStoneNames [
	"Get the peer certificate and get the stone names out of the cert."

	| subject oid |
	subject := self peerSubject.
	oid := libCrypto OBJ_txt2obj_: '1.3.6.1.4.1.47749.1.3' _: 1.
	self checkForNullReturn: oid from: 'OBJ_txt2obj'.
	^self getAllValuesOf: oid from: subject
]

{ #category : 'accessing state' }
HostAgentTlsActor >> peerSubnets [
	"Get the peer certificate and get the subnet restrictions out of the cert."

	| subject oid |
	subject := self peerSubject.
	oid := libCrypto OBJ_txt2obj_: '1.3.6.1.4.1.47749.1.2' _: 1.
	self checkForNullReturn: oid from: 'OBJ_txt2obj'.
	^self getAllValuesOf: oid from: subject
]

{ #category : 'accessing state' }
HostAgentTlsActor >> peerUserNames [
	"Get the peer certificate and get the usernames out of the cert."

	| subject oid |
	subject := self peerSubject.
	oid := libCrypto OBJ_txt2obj_: '1.3.6.1.4.1.47749.1.4' _: 1.
	self checkForNullReturn: oid from: 'OBJ_txt2obj'.
	^self getAllValuesOf: oid from: subject
]

{ #category : 'certificate info' }
HostAgentTlsActor >> remoteIpOfFd: aFileDescriptor [
	"Answer an IPv4 address as a ByteArray.
	If the file descriptor is for an IPv6 connection, and has an IPv4-mapped address,
	answer the IPv4 equivalent. True IPv6 addresses are not yet supported."

	| addrClass libcSocket addrInfo socklen retval family |
	addrClass := StructSockaddrIn6.
	libcSocket := LibcSocket new.
	addrInfo := addrClass new.
	socklen := CByteArray gcMalloc: 4.
	socklen uint32At: 0 put: addrClass byteSize.
	retval := libcSocket getpeername_: aFileDescriptor _: addrInfo _: socklen.
	retval = 0
		ifFalse: 
			[self error: 'Could not get peername for file descriptor '
						, aFileDescriptor printString].
	family := addrInfo sin_family.
	family = LibcSocket af_inet
		ifTrue: 
			[addrInfo changeClassTo: StructSockaddrIn.
			^addrInfo address].
	family = LibcSocket af_inet6
		ifTrue: [^self ipv4FromIpv6: addrInfo address].
	self
		error: 'Socket is neither IPv4 nor IPv6. Family is ' , family printString
]

{ #category : 'accessing state' }
HostAgentTlsActor >> ssl [

	^ssl
]

{ #category : 'accessing state' }
HostAgentTlsActor >> tcpSocket [

	^ tcpSocket 

]

{ #category : 'accessing state' }
HostAgentTlsActor >> tcpSocket: aGsSocket [

	tcpSocket := aGsSocket
]

{ #category : 'initialization' }
HostAgentTlsActor >> tlsMethod [

	^libSsl TLS_server_method
]
