#set -x
#=========================================================================
# Copyright (C) GemTalk Systems 1986-2024.  All Rights Reserved.
#
# Name - nrspec.sh
#
# Purpose - Parse a network resource string.  This script is "."'d by
#           other shell scripts.  It exits if a parse error is found.
#
# Inputs:  $nrspec -- the string to parse
#          $comid  -- for error messages
#
# Outputs:
#     nrprotocol -- network protocol, i.e. "TCP"
#     nrnodename -- node name of resource
#     nrbody -- "body" of resource, i.e., after second "!"
#     nrfirstpart -- all of nrs except nrbody
#     nrtaskname -- taskname, i.e, first work of body
#     nrexplicit -- either "no" or "yes"
#
#=========================================================================

########### BEGIN protocol, node, and filespec from NRSpec netLdiName ######
# nrspec and values from it...  
nrprotocol=""		# "tcp" or ""
nrnodename=""           # "node1" or "" 
nrbody=""               # "!decnet@node2!file" or ""
nrfirstpart=""          # "!tcp@node1#dbf!" or ""
nrtaskname=""           # "dbf" or ""
nrexplicit=no           # is this explicit NRSpec syntax

# if 1st char is NOT a ! ... i.e. "foo" or "/export/home/gemsys.dbf" 
if [ `expr "//$nrspec" : '\(...\)'` != //! ]; then
  nrbody="$nrspec"
else
  # else we have a NRSpec...  i.e. "!tcp@www.gemstone.com#auth..!!..!d.dbf"
  nrexplicit=y          # toggle explicit NRSpec true

  nrtmpstr=`expr "$nrspec" : '!\(.*\)$'` # drop initial !
  nrbang=`expr "$nrtmpstr" : '.*!.*'`	    # find the closing ! 
  if [ $nrbang -eq 0 ]; then
    # closing ! is missing
    # say nrspec.endbang $comid "$nrspec"
    echo "${comid}[Error]:    closing "!" is missing in argument   $nrspec "
    exit 1
  fi
  nrtmpstr=`expr "$nrtmpstr" : '\(.*\)!.*'` # drop final !

  # find nrfirstpart, tmpstr, and nrbody i.e. "!...!", "...", and the body 
  nrfirstpart=`expr "$nrspec" : '.*\(![^!]*!\).*'`
  nrbody=`expr "$nrspec" : '.*![^!]*!\(.*\)$'`

  # figure task name from nrfirstpart
  nrtaskletter=`expr "$nrfirstpart" : '.*\#\([d|f|g|s|t]\)'`
  case $nrtaskletter in
    d) nrtaskname="dbf";;
    f) nrtaskname="file";;
    g) nrtaskname="gateway";;
    s) nrtaskname="server";; 
    t) nrtaskname="task";;
  esac

  # if #, then substring upto # i.e. "tcp@www.gemstone.com#dbf.." 
  #   OR "tcp@www.gemstone.com#auth..."  OR "tcp@www.gemstone.com"
  #   OR "tcp#dbf" OR "tcp"
  nrpoundindex=`expr "$nrtmpstr" : ".*#.*"`
  if [ $nrpoundindex -ne 0 ]; then
    nrtmpstr=`expr "$nrtmpstr" : '\([^#]*\)#.*'`
  fi  

  #   NOW, if no "@", then get protocol i.e. "tcp"
  nratindex=`expr "$nrtmpstr" : ".*@.*"`
  if [ $nratindex -eq 0 ]; then
    nrprotocol="$nrtmpstr" 
  else
    #   else if "@" then protocol@nodename i.e. "tcp@www.gemstone.com"
# echo "nrtmpstr is \"$nrtmpstr\""
    nrprotocol=`expr "$nrtmpstr" : '\([^@]*\)@.*'`
    nrnodename=`expr "$nrtmpstr" : '[^@]*@\([^#@]*\)'`
  fi
fi

# for debugging
# echo "nrspec:      \"$nrspec\""
# echo "nrprotocol:  \"$nrprotocol\""
# echo "nrnodename:  \"$nrnodename\""
# echo "nrbody:      \"$nrbody\""
# echo "nrfirstpart: \"$nrfirstpart\""
# echo "nrtaskname:  \"$nrtaskname\""
# echo "nrexplicit:  \"$nrexplicit\""
# exit 0

############### END protocol, node, and filespec from NRSpec ########## 
